#!/usr/bin/env Rscript

hlp = "Plot figure with clusters and associations with cytokine responses."

# Paths
in.data = "./data/data.Robj"
out.dir = file.path("./output/associations/")
dir.create(out.dir, recursive = TRUE, showWarnings = FALSE)
img.dir = file.path(out.dir, "img")
dir.create(img.dir, recursive = TRUE, showWarnings = FALSE)

# Parameters 
alpha = 0.05  # P-value

# Libs 
require(ggplot2)
require(ggpubr)

# Load data object
load(in.data)
obj = .GlobalEnv$obj
df = obj$metadata

# Combine responses
C2 = obj$assays$CytokineFold2$data
cm2 = obj$assays$CytokineFold2$metadata
C3 = obj$assays$CytokineFold3$data
cm3 = obj$assays$CytokineFold3$metadata
cm2$Time = 2
cm3$Time = 3
C = as.matrix(cbind(C2, C3))
cm = rbind(cm2, cm3)
df[,colnames(C)] = C[row.names(df),]

# Features
rf = data.frame(row.names = colnames(C), Feature=colnames(C), stringsAsFactors = F)
rf[,colnames(cm)] = cm[rf$Feature,]
df$Cluster = as.factor(df$Cluster)

# Try all selected variables
for(v in colnames(C)){
  keep = !is.na(df[,v]) & !is.na(df$Cluster)
  cs = kruskal.test(df[,v], g = df$Cluster)
  p = rf[v, "pvalue"] = cs$p.value
  mark = ""
  if(p < alpha) mark = " *"
  label = sprintf("%s + %s (t=%d)  %s", rf[v, "Cytokine"], rf[v, "Stimulus"], rf[v, "Time"], mark)
  
  # Plot positioninng
  fname = file.path(img.dir, sprintf("mds_clusters_%s.pdf", v))
  rf[v, "pathMDS"] = fname
  pl = ggplot(data=df[keep,]) + 
    geom_point(aes(x=MDS1, y=MDS2, color=df[keep,v]), show.legend = F) + 
    xlab("MDS 1") + ylab("MDS 2") +
    theme(legend.title = element_blank(),
          legend.margin=margin(),
          text = element_text(size=7)) + ggtitle(label)
  ggsave(fname, width = 2.3, height = 2.2, plot=pl)
  message(sprintf("Written %s", fname))
  
  # Distribution plots
  fname = file.path(img.dir, sprintf("box_s_%s.pdf", v))
  rf[v, "pathDist"] = fname
  ggplot(data=df[keep,], aes(x=Cluster, y=df[keep, v], fill=df[keep, v])) + geom_boxplot() + 
    theme(legend.title = element_blank(), legend.margin=margin(),
          text = element_text(size=7)) + ylab("Fold change") + ggtitle(" ")
  ggsave(fname, width = 2.7, height = 2.2)
  message(sprintf("Written %s", fname))
}

# Write results
rf$padj = p.adjust(rf$pvalue)
rf = rf[order(rf$pvalue),]
fname = file.path(out.dir, "summary_cytokines.csv")
write.csv(rf, fname)
message(sprintf("Written %s", fname))
