clc
clear all 
close all
clearvars

%% Relative crater density computation on Mercury, the Moon and Mars
% 
% @author Anthony Lagain (anthony.lagain@curtin.edu.au) 2018
%  
% Resources:
%     (1) Binh Pham (2021. ciplot(lower,upper,x,colour,alpha) 
%         (https://www.mathworks.com/matlabcentral/fileexchange/63314-ciplot-lower-upper-x-colour-alpha), 
%         MATLAB Central File Exchange. Retrieved July 27, 2021.
%     (2) Robbins, S.J. (2019). A New Global Database of Lunar Impact Craters >12 km: 1. Crater Locations 
%         and Sizes, Comparisons With Published Databases, and Global Analysis. Journal of Geophysical Research: 
%         Planets 124, 871-892.
%     (3) Herrick, R.R., et al. (2018). Observations From a Global Database of Impact Craters on Mercury With 
%         Diameters Greater than 5 km. Journal of Geophysical Research: Planets 123, 2089-2109.
%     (4) Lagain, A. et al. (2021). Mars Crater Database: A participative project for the classification 
%         of the morphological characteristics of large Martian craters. Large Meteorite Impacts and 
%         Planetary Evolution VI, Wolf Uwe Reimold, Christian Koeberl.
%     (5) Fortezzo, C., Spudis, P., Harrel, S., 2020. Release of the digital unified global geologic map of 
%         the Moon at 1:5,000,000-scale., in: 51st Lunar and Planetary Science Conference.
%     (6) Kinczyk, M. et al. (2019). The First Global Geological Map of Mercury, in: EPSC-DPS Joint Meeting.
%     (7) Tanaka, K.L. et al. (2014). Geologic map of Mars.



%% set param
inter = 5;
%%% Mercury
inter_lat_mer = inter;
%%% Moon
moon_diam_threshold = [3 10 20 50]; % set crater diameter / used to set the equivalent crater diameter on other bodies
inter_lat_moon = inter;
%%% Mars
inter_lat_mars = inter;

%%% Compute on hemispheric (abs(lat)) or N/S distinguished 
%%% --> change cvalue
c = 1; % 1=N/S 2=hemispheric

min_lat = [-89.9999 0];                   
col = [2 3];

%% scaling laws

theta = 45;
rho_i = 1900;
rho_t = 3000;
L = 10:1:10000;
v_i_merc = 42500;   %v are from Lefeuvre & Wieczorek 2008
v_i_moon = 19400;
v_i_mars = 10600;
g_merc = 3.61;
g_moon = 1.625;
g_mars = 3.75; 
D_c_merc = 11700; %Susorney et al., 2016
D_c_moon = 15000; %Kruger et al., 2018, on highlands
D_c_mars = 6000; %Robbins & Hynek 2012b

D_t_moon=1.161*((rho_i/rho_t).^(1/3))*(L.^0.78)*(v_i_moon.^0.44)*(g_moon.^-0.22)*(sin(theta).^(1/3));
D_f_comp_moon = 1.17*((D_t_moon.^1.13)./(D_c_moon.^0.13));
for i = length(D_f_comp_moon);
    if D_f_comp_moon(i)<= D_c_moon;
       D_f_comp_moon(i) = 1.25*D_t_moon(i);
    else
       D_f_comp_moon = D_f_comp_moon;
    end
end
diam_imp_m_moon = moon_diam_threshold*1000;
D_imp_moon = interp1(D_f_comp_moon, L,diam_imp_m_moon)./1000;

L = D_imp_moon*1000;

D_t_merc=1.161*((rho_i/rho_t).^(1/3))*(L.^0.78)*(v_i_merc.^0.44)*(g_merc.^-0.22)*(sin(theta).^(1/3));
D_f_comp_merc = (1.17*((D_t_merc.^1.13)./(D_c_merc.^0.13)));
for i = 1:length(moon_diam_threshold)
    if  (D_f_comp_merc(i) <= D_c_merc);
        D_f_comp_merc(i) = 1.25*D_t_merc(i);
    else
        D_f_comp_merc = D_f_comp_merc;
    end
end
D_f_comp_merc = D_f_comp_merc./1000;


D_t_mars=1.161*((rho_i/rho_t).^(1/3))*(L.^0.78)*(v_i_mars.^0.44)*(g_mars.^-0.22)*(sin(theta).^(1/3));
D_f_comp_mars = (1.17*((D_t_mars.^1.13)./(D_c_mars.^0.13)))./1000;
for i = 1:length(moon_diam_threshold)
    if  (D_f_comp_mars(i) <= D_c_mars);
        D_f_comp_merc(i) = 1.25*D_t_mars(i);
    else
        D_f_comp_mars = D_f_comp_mars;
    end
end
D_f_comp_mars = D_f_comp_mars./1000;


merc_diam_threshold = D_f_comp_merc./1000;
mars_diam_threshold = D_f_comp_mars*1000;

D_comp_body = [L./1000; merc_diam_threshold; moon_diam_threshold; mars_diam_threshold]

%% area
tic
mercuryellipsoid = referenceEllipsoid('mercury','km');
moonellipsoid = referenceEllipsoid('moon','km');
marsellipsoid = referenceEllipsoid('mars','km');

% range longitude
lon1 = -179.9999;
lon2 = 179.9999;
% bodies total area 
A_T_merc = 7.4754e+07;
A_T_moon = 3.7959e+07;
A_T_mars = 1.4441e+08;

% bin lat
bin_mer = min_lat(c):inter_lat_mer:89.9999;
bin_moon = min_lat(c):inter_lat_moon:89.9999;
bin_mars = min_lat(c):inter_lat_mars:89.9999;

bin_mars_shape = bin_mars;
bin_mer_shape = bin_mer;

cd shp\mercury;
selec_mer = shaperead('mercury_area_selec.shp');
area_selec_mer = sum(areaint(selec_mer.Y,selec_mer.X,mercuryellipsoid));
shape_area_selec_mer = polyshape(selec_mer.X,selec_mer.Y,'simplify', false);
figure(1)
plot(shape_area_selec_mer)
axis equal
area_selec_mer = sum(areaint(shape_area_selec_mer.Vertices(:,2),shape_area_selec_mer.Vertices(:,1),mercuryellipsoid));
[quad.geometry] = deal('polygon');
n_mer = length(bin_mer)-1;

for i = 1:n_mer;
    quad.Lat = [bin_mer_shape(i) bin_mer_shape(i+1)];
    quad.Long = [lon1 lon2];
    [latq,lonq] = outlinegeoquad(quad.Lat,quad.Long,1,1);
    shape_quad = polyshape(lonq, latq,'simplify', false);
    polyin = intersect(shape_quad,shape_area_selec_mer,'KeepCollinearPoints',true);
    hold on
    plot(polyin)
    if isempty(polyin.Vertices) >0
    area_mer(i) = 0;
    else
       area_mer(i) = sum(areaint(polyin.Vertices(:,2),polyin.Vertices(:,1),mercuryellipsoid));
    end
end

bin_moon_shape = bin_moon;
cd ../..
cd shp\moon;
selec_moon = shaperead('moon_area_selec.shp');
area_selec_moon = sum(areaint(selec_moon.Y,selec_moon.X,moonellipsoid));
shape_area_selec = polyshape(selec_moon.X,selec_moon.Y,'simplify', false);
figure(2)
plot(shape_area_selec)
axis equal
area_selec_moon = sum(areaint(shape_area_selec.Vertices(:,2),shape_area_selec.Vertices(:,1),moonellipsoid));
[quad.geometry] = deal('polygon');
n_moon = length(bin_moon)-1;

for i = 1:n_moon;
    quad.Lat = [bin_moon_shape(i) bin_moon_shape(i+1)];
    quad.Long = [lon1 lon2];
    [latq,lonq] = outlinegeoquad(quad.Lat,quad.Long,1,1);
    shape_quad = polyshape(lonq, latq,'simplify', false);
    polyin = intersect(shape_quad,shape_area_selec,'KeepCollinearPoints',true);
    hold on
    plot(polyin)
    if isempty(polyin.Vertices) >0
    area_moon(i) = 0;
    else
       area_moon(i) = sum(areaint(polyin.Vertices(:,2),polyin.Vertices(:,1),moonellipsoid));
    end
end

bin_mars_shape = bin_mars;

cd ../..
cd shp\mars;
noc = shaperead('mars_area_selec.shp');
area_noc_mars = sum(areaint(noc.Y,noc.X,marsellipsoid));
shape_noc = polyshape(noc.X,noc.Y,'simplify', false);
figure(3)
plot(shape_noc)
axis equal
area_noc = sum(areaint(shape_noc.Vertices(:,2),shape_noc.Vertices(:,1),marsellipsoid));
[quad.geometry] = deal('polygon');
n_mars = length(bin_mars)-1;

for i = 1:n_mars;
    quad.Lat = [bin_mars_shape(i) bin_mars_shape(i+1)];
    quad.Long = [lon1 lon2];
    [latq,lonq] = outlinegeoquad(quad.Lat,quad.Long,1,1);
    shape_quad = polyshape(lonq, latq,'simplify', false);
    polyin = intersect(shape_quad,shape_noc,'KeepCollinearPoints',true);
    hold on
    plot(polyin)
    if isempty(polyin.Vertices) >0
    area_polyin(i) = 0;
    else
       area_mars(i) = sum(areaint(polyin.Vertices(:,2),polyin.Vertices(:,1),marsellipsoid));
    end
end

sprintf('Surface loaded and area computed')
toc

%% Load DB
cd ../..
tic
%%% DB Mercury
mercury = readtable('mercury_herrick_2018.csv');
lat_mer = table2array(mercury(:,2));
lat_mer_abs=abs(lat_mer);
long_mer = table2array(mercury(:,3));
diam_mer = table2array(mercury(:,4));
db_mer = [long_mer lat_mer lat_mer_abs diam_mer];
% shapefile
inarea = inpolygon(db_mer(:,1),db_mer(:,2),shape_area_selec_mer.Vertices(:,1),shape_area_selec_mer.Vertices(:,2));
db_mer = db_mer(inarea,:);
sprintf('DB Mercury loaded')
toc

%%% DB Moon
tic
moon = readtable('moon_robbins_2019.csv');
lat_moon = table2array(moon(:,2));
lat_moon_abs=abs(lat_moon);
long_moon  = table2array(moon(:,3));
diam_moon  = table2array(moon(:,4));
% convert 0-360 --> -180 - 180
for i = 1:length(long_moon);
    long_moon(i) = long_moon(i)-180;
end
db_moon = [long_moon lat_moon lat_moon_abs diam_moon];
sub_diam_moon=db_moon(:,4)>=moon_diam_threshold(1); 
db_moon=db_moon(sub_diam_moon,:);
% shapefile
inarea = inpolygon(db_moon(:,1),db_moon(:,2),shape_area_selec.Vertices(:,1),shape_area_selec.Vertices(:,2));
db_moon = db_moon(inarea,:);
sprintf('DB Moon loaded')
toc

%%% DB mars
tic
mars = readtable('mars_lagain_2021.csv');
lat_mars = table2array(mars(:,4));
lat_mars_abs=abs(lat_mars);
long_mars  = table2array(mars(:,3));
diam_mars  = table2array(mars(:,2))./1000;
type = table2array(mars(:,5));
db_mars = [long_mars lat_mars  lat_mars_abs diam_mars  type];
% south
lat_type_mars_south=db_mars(:,2)<0 & db_mars(:,5)<=3; 
db_mars=db_mars(lat_type_mars_south,:);
sub_diam_mars=db_mars(:,4)>=mars_diam_threshold(1); 
db_mars=db_mars(sub_diam_mars,:);
% shapefile
innoc = inpolygon(db_mars(:,1),db_mars(:,2),noc.X,noc.Y);
db_mars = db_mars(innoc,:);
sprintf('DB Mars loaded')
toc

%% Threshold diam DB
% Mercury
sub_diam_mer_5=db_mer(:,4)>=merc_diam_threshold(1); 
db_mer_5km=db_mer(sub_diam_mer_5,:);
sub_diam_mer_10=db_mer(:,4)>=merc_diam_threshold(2); 
db_mer_10km=db_mer(sub_diam_mer_10,:);
sub_diam_mer_20=db_mer(:,4)>=merc_diam_threshold(3); 
db_mer_20km=db_mer(sub_diam_mer_20,:);
sub_diam_mer_50=db_mer(:,4)>=merc_diam_threshold(4); 
db_mer_50km=db_mer(sub_diam_mer_50,:);

% Moon
sub_diam_moon_1=db_moon(:,4)>=moon_diam_threshold(1); 
db_moon_1km=db_moon(sub_diam_moon_1,:);
sub_diam_moon_5=db_moon(:,4)>=moon_diam_threshold(2); 
db_moon_5km=db_moon(sub_diam_moon_5,:);
sub_diam_moon_10=db_moon(:,4)>=moon_diam_threshold(3); 
db_moon_10km=db_moon(sub_diam_moon_10,:);
sub_diam_moon_20=db_moon(:,4)>=moon_diam_threshold(4); 
db_moon_20km=db_moon(sub_diam_moon_20,:);
% Mars
sub_diam_mars_1=db_mars(:,4)>=mars_diam_threshold(1); 
db_mars_1km=db_mars(sub_diam_mars_1,:);
sub_diam_mars_5=db_mars(:,4)>=mars_diam_threshold(2); 
db_mars_5km=db_mars(sub_diam_mars_5,:);
sub_diam_mars_10=db_mars(:,4)>=mars_diam_threshold(3); 
db_mars_10km=db_mars(sub_diam_mars_10,:);
sub_diam_mars_20=db_mars(:,4)>=mars_diam_threshold(4); 
db_mars_20km=db_mars(sub_diam_mars_20,:);

%% histo
figure(5)
% mercury
h1 = histogram(db_mer_5km(:,col(c)),bin_mer);
counts_mer_5km = h1.Values;
BinEdges_mer = h1.BinEdges + h1.BinWidth./2;
BinEdges_mer_plot = BinEdges_mer(1:end-1);
ratio_mer_5km = (counts_mer_5km./area_mer)./(sum(counts_mer_5km)./sum(area_mer));
hold off
h2 = histogram(db_mer_10km(:,col(c)),bin_mer);
counts_mer_10km = h2.Values;
BinEdges_mer = h2.BinEdges + h2.BinWidth./2;
ratio_mer_10km = (counts_mer_10km./area_mer)./(sum(counts_mer_10km)./sum(area_mer));
hold off
h3 = histogram(db_mer_20km(:,col(c)),bin_mer);
counts_mer_20km = h3.Values;
BinEdges_mer = h3.BinEdges + h3.BinWidth./2;
ratio_mer_20km = (counts_mer_20km./area_mer)./(sum(counts_mer_20km)./sum(area_mer));
hold off
h4 = histogram(db_mer_50km(:,col(c)),bin_mer);
counts_mer_50km = h4.Values;
BinEdges_mer = h4.BinEdges + h4.BinWidth./2;
ratio_mer_50km = (counts_mer_50km./area_mer)./(sum(counts_mer_50km)./sum(area_mer));

% Moon
hold off
h9 = histogram(db_moon_1km(:,col(c)),bin_moon);
counts_moon_1km = h9.Values;
BinEdges_moon = h9.BinEdges + h9.BinWidth./2;
BinEdges_moon_plot = BinEdges_moon(1:end-1);
ratio_moon_1km = (counts_moon_1km./area_moon)./(sum(counts_moon_1km)./sum(area_moon));
hold off
h10 = histogram(db_moon_5km(:,col(c)),bin_moon);
counts_moon_5km = h10.Values;
BinEdges_moon = h10.BinEdges + h10.BinWidth./2;
BinEdges_moon_plot = BinEdges_moon(1:end-1);
ratio_moon_5km = (counts_moon_5km./area_moon)./(sum(counts_moon_5km)./sum(area_moon));
hold off
h11 = histogram(db_moon_10km(:,col(c)),bin_moon);
counts_moon_10km = h11.Values;
BinEdges_moon = h11.BinEdges + h11.BinWidth./2;
BinEdges_moon_plot = BinEdges_moon(1:end-1);
ratio_moon_10km = (counts_moon_10km./area_moon)./(sum(counts_moon_10km)./sum(area_moon));
hold off
h12 = histogram(db_moon_20km(:,col(c)),bin_moon);
counts_moon_20km = h12.Values;
BinEdges_moon = h12.BinEdges + h12.BinWidth./2;
BinEdges_moon_plot = BinEdges_moon(1:end-1);
ratio_moon_20km = (counts_moon_20km./area_moon)./(sum(counts_moon_20km)./sum(area_moon));

hold off

% mars
hold off
h13 = histogram(db_mars_1km(:,col(c)),bin_mars);
counts_mars_1km = h13.Values;
BinEdges_mars = h13.BinEdges + h13.BinWidth./2;
BinEdges_mars_plot = BinEdges_mars(1:end-1);
ratio_mars_1km = (counts_mars_1km(1:length(area_mars))./area_mars)./(sum(counts_mars_1km(1:length(area_mars)))./sum(area_mars));
hold off
h14 = histogram(db_mars_5km(:,col(c)),bin_mars);
counts_mars_5km = h14.Values;
BinEdges_mars = h14.BinEdges + h14.BinWidth./2;
BinEdges_mars_plot = BinEdges_mars(1:end-1);
ratio_mars_5km = (counts_mars_5km(1:length(area_mars))./area_mars)./(sum(counts_mars_5km(1:length(area_mars)))./sum(area_mars));
hold off
h15 = histogram(db_mars_10km(:,col(c)),bin_mars);
counts_mars_10km = h15.Values;
BinEdges_mars = h15.BinEdges + h15.BinWidth./2;
BinEdges_mars_plot = BinEdges_mars(1:end-1);
ratio_mars_10km = (counts_mars_10km(1:length(area_mars))./area_mars)./(sum(counts_mars_10km(1:length(area_mars)))./sum(area_mars));
hold off
h16 = histogram(db_mars_20km(:,col(c)),bin_mars);
counts_mars_20km = h16.Values;
BinEdges_mars = h16.BinEdges + h16.BinWidth./2;
BinEdges_mars_plot = BinEdges_mars(1:end-1);
ratio_mars_20km = (counts_mars_20km(1:length(area_mars))./area_mars)./(sum(counts_mars_20km(1:length(area_mars)))./sum(area_mars));
hold off

%% errors
% Mercury
err_mer_5km = (((sqrt(counts_mer_5km))./area_mer)./(((sum(counts_mer_5km))./sum(area_mer))));
min_err_mer_5km = ratio_mer_5km- err_mer_5km;
max_err_mer_5km = ratio_mer_5km + err_mer_5km ;
err_mer_10km = (((sqrt(counts_mer_10km))./area_mer)./(((sum(counts_mer_10km))./sum(area_mer))));
min_err_mer_10km = ratio_mer_10km- err_mer_10km;
max_err_mer_10km = ratio_mer_10km + err_mer_10km ;
err_mer_20km = (((sqrt(counts_mer_20km))./area_mer)./(((sum(counts_mer_20km))./sum(area_mer))));
min_err_mer_20km = ratio_mer_20km- err_mer_20km;
max_err_mer_20km = ratio_mer_20km + err_mer_20km ;
err_mer_50km = (((sqrt(counts_mer_50km))./area_mer)./(((sum(counts_mer_50km))./sum(area_mer))));
min_err_mer_50km = ratio_mer_50km- err_mer_50km;
max_err_mer_50km = ratio_mer_50km + err_mer_50km ;

% Moon
err_moon_1km = (((sqrt(counts_moon_1km))./area_moon)./(((sum(counts_moon_1km))./sum(area_moon))));
min_err_moon_1km = ratio_moon_1km- err_moon_1km;
max_err_moon_1km = ratio_moon_1km + err_moon_1km ;
err_moon_5km = (((sqrt(counts_moon_5km))./area_moon)./(((sum(counts_moon_5km))./sum(area_moon))));
min_err_moon_5km = ratio_moon_5km- err_moon_5km;
max_err_moon_5km = ratio_moon_5km + err_moon_5km ;
err_moon_10km = (((sqrt(counts_moon_10km))./area_moon)./(((sum(counts_moon_10km))./sum(area_moon))));
min_err_moon_10km = ratio_moon_10km- err_moon_10km;
max_err_moon_10km = ratio_moon_10km + err_moon_10km ;
err_moon_20km = (((sqrt(counts_moon_20km))./area_moon)./(((sum(counts_moon_20km))./sum(area_moon))));
min_err_moon_20km = ratio_moon_20km- err_moon_20km;
max_err_moon_20km = ratio_moon_20km + err_moon_20km ;

% Mars
err_mars_1km = (((sqrt(counts_mars_1km(1:length(area_mars))))./area_mars)./(((sum(counts_mars_1km(1:length(area_mars))))./sum(area_mars))));
min_err_mars_1km = ratio_mars_1km- err_mars_1km;
max_err_mars_1km = ratio_mars_1km + err_mars_1km ;
err_mars_5km = (((sqrt(counts_mars_5km(1:length(area_mars))))./area_mars)./(((sum(counts_mars_5km(1:length(area_mars))))./sum(area_mars))));
min_err_mars_5km = ratio_mars_5km- err_mars_5km;
max_err_mars_5km = ratio_mars_5km + err_mars_5km ;
err_mars_10km = (((sqrt(counts_mars_10km(1:length(area_mars))))./area_mars)./(((sum(counts_mars_10km(1:length(area_mars))))./sum(area_mars))));
min_err_mars_10km = ratio_mars_10km- err_mars_10km;
max_err_mars_10km = ratio_mars_10km + err_mars_10km ;
err_mars_20km = (((sqrt(counts_mars_20km(1:length(area_mars))))./area_mars)./(((sum(counts_mars_20km(1:length(area_mars))))./sum(area_mars))));
min_err_mars_20km = ratio_mars_20km- err_mars_20km;
max_err_mars_20km = ratio_mars_20km + err_mars_20km ;

%% Figure crater density 
% close all
f = figure (7)
subplot(131)
ciplot (min_err_mer_5km,max_err_mer_5km,BinEdges_mer(1:end-1),'c');
hold on
ciplot (min_err_mer_10km,max_err_mer_10km,BinEdges_mer(1:end-1),'b');
hold on
ciplot (min_err_mer_20km,max_err_mer_20km,BinEdges_mer(1:end-1),'g');
hold on
ciplot (min_err_mer_50km,max_err_mer_50km,BinEdges_mer(1:end-1),'r');
hold on
p(1) = yline(1, 'LineWidth', 2);
hold on
p(2) = plot(BinEdges_mer(1:end-1), ratio_mer_5km,'c','LineWidth', 2,'DisplayName',['>' num2str(round(merc_diam_threshold(1),2)) 'km, N = ' num2str(length(db_mer_5km))]);
hold on
p(3) = plot(BinEdges_mer(1:end-1), ratio_mer_10km,'b','LineWidth', 2,'DisplayName',['>' num2str(round(merc_diam_threshold(2),2)) 'km, N = ' num2str(length(db_mer_10km))]);
hold on
p(4) = plot(BinEdges_mer(1:end-1), ratio_mer_20km,'g','LineWidth', 2,'DisplayName',['>' num2str(round(merc_diam_threshold(3),2)) 'km, N = ' num2str(length(db_mer_20km))]);
hold on
p(5) = plot(BinEdges_mer(1:end-1), ratio_mer_50km,'r','LineWidth', 2,'DisplayName',['>' num2str(round(merc_diam_threshold(4),2)) 'km, N = ' num2str(length(db_mer_50km))]);
hold on
for i=1:length(ratio_mer_50km)
    avg_mer(i)=(ratio_mer_5km(i)+ratio_mer_10km(i)+ratio_mer_20km(i)+ratio_mer_50km(i))./4;
end

hold on 
p(6) = plot(BinEdges_mer(1:end-1), avg_mer,'LineStyle','--','Color','k','LineWidth', 2,'DisplayName','Average');
legend(p(2:6),'Location', 'Best')
xlabel ('Latitude ({\circ})')
ylabel (['Relative crater density per ',num2str(round(inter_lat_mer),0),'{\circ} latitude bin']);
title ({'Mercury, Herrick (2018)'});
xlim([min_lat(c) 90])
ylim([0.25 2.25])

box on
grid on


subplot(132)
ciplot (min_err_moon_1km,max_err_moon_1km,BinEdges_moon(1:end-1),'c');
hold on
ciplot (min_err_moon_5km,max_err_moon_5km,BinEdges_moon(1:end-1),'b');
hold on
ciplot (min_err_moon_10km,max_err_moon_10km,BinEdges_moon(1:end-1),'g');
hold on
ciplot (min_err_moon_20km,max_err_moon_20km,BinEdges_moon(1:end-1),'r');
hold on
p(1) = yline(1, 'LineWidth', 2);
hold on
p(2) = plot(BinEdges_moon(1:end-1), ratio_moon_1km,'c','LineWidth', 2,'DisplayName',['>' num2str(round(moon_diam_threshold(1),2)) 'km, N = ' num2str(length(db_moon_1km))]);
hold on
p(3) = plot(BinEdges_moon(1:end-1), ratio_moon_5km,'b','LineWidth', 2,'DisplayName',['>' num2str(round(moon_diam_threshold(2),2)) 'km, N = ' num2str(length(db_moon_5km))]);
hold on
p(4) = plot(BinEdges_moon(1:end-1), ratio_moon_10km,'g','LineWidth', 2,'DisplayName',['>' num2str(round(moon_diam_threshold(3),2)) 'km, N = ' num2str(length(db_moon_10km))]);
hold on
p(5) = plot(BinEdges_moon(1:end-1), ratio_moon_20km,'r','LineWidth', 2,'DisplayName',['>' num2str(round(moon_diam_threshold(4),2)) 'km, N = ' num2str(length(db_moon_20km))]);
for i=1:length(ratio_moon_20km)
    avg_moon(i)=(ratio_moon_1km(i)+ratio_moon_5km(i)+ratio_moon_10km(i)+ratio_moon_20km(i))./4;
end

hold on 
p(6) = plot(BinEdges_moon(1:end-1), avg_moon,'LineStyle','--','Color','k','LineWidth', 2,'DisplayName','Average');

legend(p(2:6),'Location', 'Best');
xlabel ('Latitude ({\circ})');
ylabel (['Relative crater density per ',num2str(round(inter_lat_moon),0),'{\circ} latitude bin']);
title ({'Moon, Robbins (2019)'});
ylim([0 4])
xlim([min_lat(c) 90]);
box on
grid on

subplot(133)
limit = size(BinEdges_mars);
limit = limit(2)/2;
ciplot (min_err_mars_1km(2:length(min_err_mars_1km)),max_err_mars_1km(2:length(min_err_mars_1km)),BinEdges_mars(2:length(min_err_mars_1km)),'c');
hold on
ciplot (min_err_mars_5km(2:length(min_err_mars_1km)),max_err_mars_5km(2:length(min_err_mars_1km)),BinEdges_mars(2:length(min_err_mars_1km)),'b');
hold on
ciplot (min_err_mars_10km(2:length(min_err_mars_1km)),max_err_mars_10km(2:length(min_err_mars_1km)),BinEdges_mars(2:length(min_err_mars_1km)),'g');
hold on
ciplot (min_err_mars_20km(2:length(min_err_mars_1km)),max_err_mars_20km(2:length(min_err_mars_1km)),BinEdges_mars(2:length(min_err_mars_1km)),'r');
hold on
p(1) = yline(1, 'LineWidth', 2);
hold on
p(2) = plot(BinEdges_mars(2:limit), ratio_mars_1km(2:limit),'c','LineWidth', 2,'DisplayName',['>' num2str(round(mars_diam_threshold(1),2)) 'km, N = ' num2str(length(db_mars_1km))]);
hold on
% plot(BinEdges_mars(1:end-1), ratio_mars_1km_s,'c','DisplayName',['smooth >1km, N = ' num2str(length(db_mars_1km))])
% hold on
p(3) = plot(BinEdges_mars(2:limit), ratio_mars_5km(2:limit),'b','LineWidth', 2,'DisplayName',['>' num2str(round(mars_diam_threshold(2),2)) 'km, N = ' num2str(length(db_mars_5km))]);
hold on
p(4) = plot(BinEdges_mars(2:limit), ratio_mars_10km(2:limit),'g','LineWidth', 2,'DisplayName',['>' num2str(round(mars_diam_threshold(3),2)) 'km, N = ' num2str(length(db_mars_10km))]);
hold on
p(5) = plot(BinEdges_mars(2:limit), ratio_mars_20km(2:limit),'r','LineWidth', 2,'DisplayName',['>' num2str(round(mars_diam_threshold(4),2)) 'km, N = ' num2str(length(db_mars_20km))]);

for i=1:length(ratio_mars_20km)
    avg_mars(i)=(ratio_mars_1km(i)+ratio_mars_5km(i)+ratio_mars_10km(i)+ratio_mars_20km(i))./4;
end

hold on 
p(6) = plot(BinEdges_mars(1:limit), avg_mars(1:limit),'LineStyle','--','Color','k','LineWidth', 2,'DisplayName','Average');

legend(p(2:6),'Location', 'Best');
xlabel ('Latitude ({\circ})');
ylabel (['Relative crater density per ',num2str(round(inter_lat_mars),0),'{\circ} latitude bin']);
title ({'Mars, Lagain et al. (2021)'});
ylim([0.7 1.5]);
xlim([min_lat(c) 90]);
box on
grid on

h = gcf;



%% CIPLOT FUNCTION
function ciplot(lower,upper,x,colour);
     
% ciplot(lower,upper)       
% ciplot(lower,upper,x)
% ciplot(lower,upper,x,colour)
%
% Plots a shaded region on a graph between specified lower and upper confidence intervals (L and U).
% l and u must be vectors of the same length.
% Uses the 'fill' function, not 'area'. Therefore multiple shaded plots
% can be overlayed without a problem. Make them transparent for total visibility.
% x data can be specified, otherwise plots against index values.
% colour can be specified (eg 'k'). Defaults to blue.
% Raymond Reynolds 24/11/06
if length(lower)~=length(upper)
    error('lower and upper vectors must be same length')
end
if nargin<4
    colour='b';
end
if nargin<3
    x=1:length(lower);
end
% convert to row vectors so fliplr can work
if find(size(x)==(max(size(x))))<2
x=x'; end
if find(size(lower)==(max(size(lower))))<2
lower=lower'; end
if find(size(upper)==(max(size(upper))))<2
upper=upper'; end
fill([x fliplr(x)],[upper fliplr(lower)],colour,'facealpha',0.2)
end
