/*!
  @file vector_monitor_test.cc
  @brief Tests for VectorMonitor
  @author Masaki Waga <masakiwaga@gmail.com>
  @date 2019/12/12
  @copyright GPL3+
  @sa VectorMonitor
*/
#include <boost/test/unit_test.hpp>
#include <boost/mpl/list.hpp>
#include "../src/vector_monitor.hh"
#include "../test/fixture/acc_automaton_fixture.hh"
#include "../test/fixture/invalid_transit_automaton_fixture.hh"

using namespace Parma_Polyhedra_Library;

#define Coefficient mpq_class

template<typename Zone>
struct DummyTimedWordSubject : public SingleSubject<TimedWordEvent<Zone, Coefficient>> {
  DummyTimedWordSubject(std::vector<TimedWordEvent<Zone, Coefficient>> &&vec) :vec(std::move(vec)) {}
  virtual ~DummyTimedWordSubject(){}
  void notifyAll() {
    for (const auto &event: vec) {
      this->notifyObservers(event);
    }
    vec.clear();
  }
  std::vector<TimedWordEvent<Zone, Coefficient>> vec;
};

template<typename Zone>
struct DummyVectorMonitorObserver : public Observer<VectorMonitorResult<Zone>> {
  DummyVectorMonitorObserver() {}
  virtual ~DummyVectorMonitorObserver() {}
  void notify(const VectorMonitorResult<Zone>& result) {
    resultVec.push_back(result);
  }
  std::vector<VectorMonitorResult<Zone>> resultVec;
};

template<typename Zone>
struct ACCVectorMonitorFixture : public ACCFixture {
  void feed(std::vector<TimedWordEvent<Zone, Coefficient>> &&vec) {
    auto monitor = std::make_shared<VectorMonitor<Zone>>(automaton);
    std::shared_ptr<DummyVectorMonitorObserver<Zone>> observer = std::make_shared<DummyVectorMonitorObserver<Zone>>();
    monitor->addObserver(observer);
    DummyTimedWordSubject<Zone> subject{std::move(vec)};
    subject.addObserver(monitor);
    subject.notifyAll();
    resultVec = std::move(observer->resultVec);
  }
  std::vector<VectorMonitorResult<Zone>> resultVec;
};

template<typename Zone>
struct InvalidTransitonVectorMonitorFixture : public InvalidTransitionFixture {
  void feed(std::vector<TimedWordEvent<Zone, Coefficient>> &&vec) {
    auto monitor = std::make_shared<VectorMonitor<Zone>>(automaton);
    std::shared_ptr<DummyVectorMonitorObserver<Zone>> observer = std::make_shared<DummyVectorMonitorObserver<Zone>>();
    monitor->addObserver(observer);
    DummyTimedWordSubject<Zone> subject{std::move(vec)};
    subject.addObserver(monitor);
    subject.notifyAll();
    resultVec = std::move(observer->resultVec);
  }
  std::vector<VectorMonitorResult<Zone>> resultVec;
};

BOOST_AUTO_TEST_SUITE(VectorMonitorTest)

BOOST_FIXTURE_TEST_CASE(ACCtest1, ACCVectorMonitorFixture<NNC_Polyhedron>)
{
  std::array<Parma_Polyhedra_Library::Variable, 5> x = 
    {Parma_Polyhedra_Library::Variable(0), 
     Parma_Polyhedra_Library::Variable(1),
     Parma_Polyhedra_Library::Variable(2),
     Parma_Polyhedra_Library::Variable(3),
     Parma_Polyhedra_Library::Variable(4)};
  std::vector<TimedWordEvent<NNC_Polyhedron, Coefficient>> dummyTimedWord;  
  Constraint_System cs;
  // staying at state[0] for 2 time units
  cs.insert(x[0] == 40 + 16);
  cs.insert(x[1] == 35 + 17);
  cs.insert(x[2] == 30 + 18);
  cs.insert(x[3] == 25 + 19);
  cs.insert(x[4] == 20 + 20);
  
  TimedWordEvent<NNC_Polyhedron, Coefficient> event = {NNC_Polyhedron(cs), 2};
  dummyTimedWord.push_back(event);

  feed(std::move(dummyTimedWord));
  BOOST_CHECK_EQUAL(resultVec.size(), 0);
}

BOOST_FIXTURE_TEST_CASE(ACCtestMatch1, ACCVectorMonitorFixture<NNC_Polyhedron>)
{
  automaton.states[0]->isMatch = true;
  automaton.states[1]->isMatch = false;
  automaton.states[2]->isMatch = false;
  automaton.states[3]->isMatch = false;
  automaton.states[4]->isMatch = false;
  automaton.states[5]->isMatch = false;

  std::array<Parma_Polyhedra_Library::Variable, 5> x = 
    {Parma_Polyhedra_Library::Variable(0), 
     Parma_Polyhedra_Library::Variable(1),
     Parma_Polyhedra_Library::Variable(2),
     Parma_Polyhedra_Library::Variable(3),
     Parma_Polyhedra_Library::Variable(4)};
  auto clock = Parma_Polyhedra_Library::Variable(5);
  std::vector<TimedWordEvent<NNC_Polyhedron, Coefficient>> dummyTimedWord;  
  Constraint_System cs;
  // staying at state[0] for 2 time units
  cs.insert(x[0] == 40 + 16);
  cs.insert(x[1] == 35 + 17);
  cs.insert(x[2] == 30 + 18);
  cs.insert(x[3] == 25 + 19);
  cs.insert(x[4] == 20 + 20);
  
  TimedWordEvent<NNC_Polyhedron, Coefficient> event = {NNC_Polyhedron(cs), 2};
  dummyTimedWord.push_back(event);

  feed(std::move(dummyTimedWord));
  BOOST_CHECK_EQUAL(resultVec.size(), 1);

  cs.insert(clock == 0);
  auto expectedZone = NNC_Polyhedron(cs);

  using namespace Parma_Polyhedra_Library::IO_Operators;
  std::cout << resultVec.front().zone << std::endl;
  BOOST_TEST((resultVec.front().zone == expectedZone));
  BOOST_CHECK_EQUAL(resultVec.front().timestamp, 2);
}

  BOOST_FIXTURE_TEST_CASE(InvalidTransitTest1, InvalidTransitonVectorMonitorFixture<NNC_Polyhedron>)
  {
    Parma_Polyhedra_Library::Variable x = Parma_Polyhedra_Library::Variable(0);
    std::vector<TimedWordEvent<NNC_Polyhedron, Coefficient>> dummyTimedWord;
    Constraint_System cs;
    // We cannot go to the accepting state because of the invalid invariant
    cs.insert(x == 5);

    TimedWordEvent<NNC_Polyhedron, Coefficient> event = {NNC_Polyhedron(cs), 5};
    dummyTimedWord.push_back(event);

    feed(std::move(dummyTimedWord));
    BOOST_CHECK_EQUAL(resultVec.size(), 0);
  }

BOOST_AUTO_TEST_SUITE_END()
