/*!
  @file timed_word_parser_test.cc
  @brief Test for TimedWordParser
  @author Masaki Waga <masakiwaga@gmail.com>
  @date 2019/12/10
  @copyright GPL3+
  @sa TimedWordParser
 */
#include <boost/test/unit_test.hpp>
#include <boost/mpl/list.hpp>
#include <sstream>
#include "../src/timed_word_parser.hh"

BOOST_AUTO_TEST_SUITE(TimedWordParserTest)

using namespace Parma_Polyhedra_Library;

template<typename Zone>
struct WithdrawWordFixture {
  std::stringstream sigStream;
  std::stringstream wordStream;
  static_assert(hasUnconstrain<Zone>::value, "Unsupported Zone");
  static_assert(hasIntersectionAssign<Zone, Zone>::value, "Unsupported Zone");
  static_assert(std::conjunction<hasUnconstrain<Zone>, hasIntersectionAssign<Zone, Zone>>::value, "Unsupported Zone");
  void execute() {
    // Construct signature for withdraw
    TimedWordParser<Zone, int> parser {wordStream, 3};

    Variable x(0), y(1), z(2);
    std::array<Constraint_System, 2> cs;
    cs[0] = Constraint_System(x == 10);
    cs[0].insert(y == 20);
    cs[0].insert(z ==15);

    cs[1] = Constraint_System(x == 20);
    cs[1].insert(y == -10);
    cs[1].insert(z == 0);

    const std::vector<TimedWordEvent<Zone, int>> expectedEvents =
      {{Zone(cs[0]), 10},
       {Zone(cs[1]), 20}
      };

    TimedWordEvent<Zone, int> event;

    for (const auto &expectedEvent: expectedEvents) {
      BOOST_TEST(parser.parse(event));
      BOOST_TEST((event.valuation == expectedEvent.valuation));
      BOOST_CHECK_EQUAL(event.timestamp, expectedEvent.timestamp);
    }
    BOOST_TEST(!parser.parse(event));
  
  }
};

template<typename Zone>
struct RationalWordFixture {
  std::stringstream sigStream;
  std::stringstream wordStream;
  static_assert(hasUnconstrain<Zone>::value, "Unsupported Zone");
  static_assert(hasIntersectionAssign<Zone, Zone>::value, "Unsupported Zone");
  static_assert(std::conjunction<hasUnconstrain<Zone>, hasIntersectionAssign<Zone, Zone>>::value, "Unsupported Zone");
  void execute() {
    // Construct signature for withdraw
    TimedWordParser<Zone, int> parser {wordStream, 3};

    Variable x(0), y(1), z(2);
    std::array<Constraint_System, 2> cs;
    cs[0] = Constraint_System(x == 1);
    cs[0].insert(2 * y == 5);
    cs[0].insert(-10 * z == 2);

    cs[1] = Constraint_System(100 * x == 20);
    cs[1].insert(10 * y == -1);
    cs[1].insert(z == 0);

    const std::vector<TimedWordEvent<Zone, int>> expectedEvents =
      {{Zone(cs[0]), 10},
       {Zone(cs[1]), 20}
      };

    TimedWordEvent<Zone, int> event;

    for (const auto &expectedEvent: expectedEvents) {
      BOOST_TEST(parser.parse(event));
      BOOST_TEST((event.valuation == expectedEvent.valuation));
      BOOST_CHECK_EQUAL(event.timestamp, expectedEvent.timestamp);
    }
    BOOST_TEST(!parser.parse(event));
  
  }
};

/*!
  @class RationalWordRationalTimestampsFixture
  @brief Test of a timed quantitative word with rational timestapms.

  Here, the signal is as follows.

  1.0	2.5	-.2	1.0
  .20	-0.10	0.00	1.5
 */
template<typename Zone>
struct RationalWordRationalTimestampsFixture {
  std::stringstream sigStream;
  std::stringstream wordStream;
  static_assert(hasUnconstrain<Zone>::value, "Unsupported Zone");
  static_assert(hasIntersectionAssign<Zone, Zone>::value, "Unsupported Zone");
  static_assert(std::conjunction<hasUnconstrain<Zone>, hasIntersectionAssign<Zone, Zone>>::value, "Unsupported Zone");
  void execute() {
    // Construct signature for withdraw
    TimedWordParser<Zone, mpq_class> parser {wordStream, 3};

    Variable x(0), y(1), z(2);
    std::array<Constraint_System, 2> cs;
    cs[0] = Constraint_System(x == 1);
    cs[0].insert(2 * y == 5);
    cs[0].insert(-10 * z == 2);

    cs[1] = Constraint_System(100 * x == 20);
    cs[1].insert(10 * y == -1);
    cs[1].insert(z == 0);

    const std::vector<TimedWordEvent<Zone, mpq_class>> expectedEvents =
      {{Zone(cs[0]), mpq_class(1)},
       {Zone(cs[1]), mpq_class(3,2)}
      };

    TimedWordEvent<Zone, mpq_class> event;

    for (const auto &expectedEvent: expectedEvents) {
      BOOST_TEST(parser.parse(event));
      BOOST_TEST((event.valuation == expectedEvent.valuation));
      BOOST_CHECK_EQUAL(event.timestamp, expectedEvent.timestamp);
    }
    BOOST_TEST(!parser.parse(event));
  
  }
};

/*!
  @class PartialWordFixture
  @brief Test of a partial word

  Here, the signal is as follows.

  1   N/A  -3.5 @ 10
  N/A   5  0 @ 20
  10 2  0 @ 25
  N/A   N/A  N/A @ 50
 */
template<typename Zone>
struct PartialWordFixture {
  std::stringstream sigStream;
  std::stringstream wordStream;
  static_assert(hasUnconstrain<Zone>::value, "Unsupported Zone");
  static_assert(hasIntersectionAssign<Zone, Zone>::value, "Unsupported Zone");
  static_assert(std::conjunction<hasUnconstrain<Zone>, hasIntersectionAssign<Zone, Zone>>::value, "Unsupported Zone");
  void execute() {
    // Construct signature for withdraw
    TimedWordParser<Zone, int> parser {wordStream, 3};

    Variable x(0), y(1), z(2);
    std::array<Zone, 4> zones;
    zones[0] = Zone(3);
    zones[0].add_constraint(x == 1);
    zones[0].add_constraint(10 * z == -35);

    zones[1] = Zone(3);
    zones[1].add_constraint(y == 5);
    zones[1].add_constraint(z == 0);

    zones[2] = Zone(3);
    zones[2].add_constraint(x == 10);
    zones[2].add_constraint(y == 2);
    zones[2].add_constraint(z == 0);

    zones[3] = Zone(3);

    const std::vector<TimedWordEvent<Zone, int>> expectedEvents =
      {{zones[0], 10},
       {zones[1], 20},
       {zones[2], 25},
       {zones[3], 50}
      };

    TimedWordEvent<Zone, int> event;

    for (const auto &expectedEvent: expectedEvents) {
      BOOST_TEST(parser.parse(event));
      BOOST_TEST((event.valuation == expectedEvent.valuation));
      BOOST_CHECK_EQUAL(event.timestamp, expectedEvent.timestamp);
    }
    BOOST_TEST(!parser.parse(event));
  
  }
};

BOOST_FIXTURE_TEST_CASE(NNC_Polyhedron_word, WithdrawWordFixture<NNC_Polyhedron>)
{
  wordStream << "10\t20\t15\t10" << "\n"
             << "20\t-10\t0\t20" << "\n";
  execute();
}

BOOST_FIXTURE_TEST_CASE(NNC_Polyhedron_rational_word, RationalWordFixture<NNC_Polyhedron>)
{
  wordStream << "1.0\t2.5\t-.2\t10" << "\n"
             << ".20\t-0.10\t0.00\t20" << "\n";
  execute();
}

BOOST_FIXTURE_TEST_CASE(NNC_Polyhedron_partial_word, PartialWordFixture<NNC_Polyhedron>)
{
  wordStream << "1.0\tN/A\t-3.5\t10" << "\n"
             << "N/A\t5.0\t0\t20" << "\n"
             << "10\t2.0\t0\t25" << "\n"
             << "N/A\tN/A\tN/A\t50" << "\n";
  execute();
}

BOOST_FIXTURE_TEST_CASE(NNC_Polyhedron_rational_word_rational_timestamps, RationalWordRationalTimestampsFixture<NNC_Polyhedron>)
{
  wordStream << "1.0\t2.5\t-.2\t1.0" << "\n"
             << ".20\t-0.10\t0.00\t1.5" << "\n";
  execute();
}

BOOST_FIXTURE_TEST_CASE(mpz_Box_word, WithdrawWordFixture<Box<Integer_Interval>>)
{
  wordStream << "10\t20\t15\t10" << "\n"
             << "20\t-10\t0\t20" << "\n";
  execute();
}

BOOST_FIXTURE_TEST_CASE(mpz_Box_rational_word, RationalWordFixture<Box<Integer_Interval>>)
{
  // This will be an overapproximation.
  wordStream << "1.0\t2.5\t-.2\t10" << "\n"
             << ".20\t-0.10\t0.00\t20" << "\n";
  execute();
}

BOOST_FIXTURE_TEST_CASE(mpz_Box_partial_word, PartialWordFixture<Box<Integer_Interval>>)
{
  // This will be an overapproximation
  wordStream << "1.0\tN/A\t-3.5\t10" << "\n"
             << "N/A\t5.0\t0\t20" << "\n"
             << "10\t2.0\t0\t25" << "\n"
             << "N/A\tN/A\tN/A\t50" << "\n";
  execute();
}

BOOST_FIXTURE_TEST_CASE(mpq_Box_word, WithdrawWordFixture<Box<Rational_Interval>>)
{
  wordStream << "10\t20\t15\t10" << "\n"
             << "20\t-10\t0\t20" << "\n";
  execute();
}

BOOST_FIXTURE_TEST_CASE(mpq_Box_rational_word, RationalWordFixture<Box<Rational_Interval>>)
{
  wordStream << "1.0\t2.5\t-.2\t10" << "\n"
             << ".20\t-0.10\t0.00\t20" << "\n";
  execute();
}

BOOST_FIXTURE_TEST_CASE(mpq_Box_partial_word, PartialWordFixture<Box<Rational_Interval>>)
{
  wordStream << "1.0\tN/A\t-3.5\t10" << "\n"
             << "N/A\t5.0\t0\t20" << "\n"
             << "10\t2.0\t0\t25" << "\n"
             << "N/A\tN/A\tN/A\t50" << "\n";
  execute();
}

typedef Interval<int, Integer_Interval_Info> Int_Interval;

BOOST_FIXTURE_TEST_CASE(int_Box_word, WithdrawWordFixture<Box<Int_Interval>> )
{
  wordStream << "10\t20\t15\t10" << "\n"
             << "20\t-10\t0\t20" << "\n";
  execute();
}

BOOST_FIXTURE_TEST_CASE(C_Polyhedron_word, WithdrawWordFixture<C_Polyhedron>)
{
  wordStream << "10\t20\t15\t10" << "\n"
             << "20\t-10\t0\t20" << "\n";
  execute();
}

BOOST_FIXTURE_TEST_CASE(C_Polyhedron_rational_word, RationalWordFixture<C_Polyhedron>)
{
  wordStream << "1.0\t2.5\t-.2\t10" << "\n"
             << ".20\t-0.10\t0.00\t20" << "\n";
  execute();
}

BOOST_FIXTURE_TEST_CASE(C_Polyhedron_partial_word, PartialWordFixture<C_Polyhedron>)
{
  wordStream << "1.0\tN/A\t-3.5\t10" << "\n"
             << "N/A\t5.0\t0\t20" << "\n"
             << "10\t2.0\t0\t25" << "\n"
             << "N/A\tN/A\tN/A\t50" << "\n";
  execute();
}

BOOST_FIXTURE_TEST_CASE(BDS_word, WithdrawWordFixture<BD_Shape<mpq_class>>)
{
  wordStream << "10\t20\t15\t10" << "\n"
             << "20\t-10\t0\t20" << "\n";
  execute();
}

BOOST_FIXTURE_TEST_CASE(BDS_rational_word, RationalWordFixture<BD_Shape<mpq_class>>)
{
  wordStream << "1.0\t2.5\t-.2\t10" << "\n"
             << ".20\t-0.10\t0.00\t20" << "\n";
  execute();
}

BOOST_FIXTURE_TEST_CASE(BDS_partial_word, PartialWordFixture<BD_Shape<mpq_class>>)
{
  wordStream << "1.0\tN/A\t-3.5\t10" << "\n"
             << "N/A\t5.0\t0\t20" << "\n"
             << "10\t2.0\t0\t25" << "\n"
             << "N/A\tN/A\tN/A\t50" << "\n";
  execute();
}

BOOST_FIXTURE_TEST_CASE(BDS_int64_word, WithdrawWordFixture<BD_Shape<int64_t>>)
{
  wordStream << "10\t20\t15\t10" << "\n"
             << "20\t-10\t0\t20" << "\n";
  execute();
}

BOOST_FIXTURE_TEST_CASE(BDS_int64_rational_word, RationalWordFixture<BD_Shape<int64_t>>)
{
  // This will be an overapproximation
  wordStream << "1.0\t2.5\t-.2\t10" << "\n"
             << ".20\t-0.10\t0.00\t20" << "\n";
  execute();
}

BOOST_FIXTURE_TEST_CASE(BDS_int64_partial_word, PartialWordFixture<BD_Shape<int64_t>>)
{
  // This will be an overapproximation
  wordStream << "1.0\tN/A\t-3.5\t10" << "\n"
             << "N/A\t5.0\t0\t20" << "\n"
             << "10\t2.0\t0\t25" << "\n"
             << "N/A\tN/A\tN/A\t50" << "\n";
  execute();
}

// BOOST_FIXTURE_TEST_CASE(BDS_float_word, WithdrawWordFixture<BD_Shape<double>>)
// {
//   wordStream << "10\t20\t15\t10" << "\n"
//              << "20\t-10\t0\t20" << "\n";
//   execute();
// }

// BOOST_FIXTURE_TEST_CASE(BDS_float_rational_word, RationalWordFixture<BD_Shape<double>>)
// {
//   wordStream << "1.0\t2.5\t-.2\t10" << "\n"
//              << ".20\t-0.10\t0.00\t20" << "\n";
//   execute();
// }

BOOST_AUTO_TEST_SUITE_END()
