#include <boost/test/unit_test.hpp>
#include <boost/mpl/list.hpp>
#include "../src/powerset_monitor.hh"
#include "../test/fixture/acc_automaton_fixture.hh"

using namespace Parma_Polyhedra_Library;

#define Coefficient mpq_class

struct DummyTimedWordSubject : public SingleSubject<TimedWordEvent<NNC_Polyhedron, Coefficient>> {
  DummyTimedWordSubject(std::vector<TimedWordEvent<NNC_Polyhedron, Coefficient>> &&vec) :vec(std::move(vec)) {}
  virtual ~DummyTimedWordSubject(){}
  void notifyAll() {
    for (const auto &event: vec) {
      notifyObservers(event);
    }
    vec.clear();
  }
  std::vector<TimedWordEvent<NNC_Polyhedron, Coefficient>> vec;
};

template<typename Zone>
struct DummyPowersetMonitorObserver : public Observer<PowersetMonitorResult<Zone>> {
  DummyPowersetMonitorObserver() {}
  virtual ~DummyPowersetMonitorObserver() {}
  void notify(const PowersetMonitorResult<Zone>& result) {
    resultVec.push_back(result);
  }
  std::vector<PowersetMonitorResult<Zone>> resultVec;
};

struct ACCPowersetMonitorFixture : public ACCFixture {
  void feed(std::vector<TimedWordEvent<NNC_Polyhedron, Coefficient>> &&vec) {
    auto monitor = std::make_shared<PowersetMonitor<NNC_Polyhedron>>(automaton);
    std::shared_ptr<DummyPowersetMonitorObserver<NNC_Polyhedron>> observer = std::make_shared<DummyPowersetMonitorObserver<NNC_Polyhedron>>();
    monitor->addObserver(observer);
    DummyTimedWordSubject subject{std::move(vec)};
    subject.addObserver(monitor);
    subject.notifyAll();
    resultVec = std::move(observer->resultVec);
  }
  std::vector<PowersetMonitorResult<NNC_Polyhedron>> resultVec;
};

BOOST_AUTO_TEST_SUITE(PowersetMonitorTest)

BOOST_FIXTURE_TEST_CASE(ACCtest1, ACCPowersetMonitorFixture)
{
  std::array<Parma_Polyhedra_Library::Variable, 5> x = 
    {Parma_Polyhedra_Library::Variable(0), 
     Parma_Polyhedra_Library::Variable(1),
     Parma_Polyhedra_Library::Variable(2),
     Parma_Polyhedra_Library::Variable(3),
     Parma_Polyhedra_Library::Variable(4)};
  std::vector<TimedWordEvent<NNC_Polyhedron, Coefficient>> dummyTimedWord;  
  Constraint_System cs;
  // staying at state[0] for 2 time units
  cs.insert(x[0] == 40 + 16);
  cs.insert(x[1] == 35 + 17);
  cs.insert(x[2] == 30 + 18);
  cs.insert(x[3] == 25 + 19);
  cs.insert(x[4] == 20 + 20);
  
  TimedWordEvent<NNC_Polyhedron, Coefficient> event = {NNC_Polyhedron(cs), 2};
  dummyTimedWord.push_back(event);

  feed(std::move(dummyTimedWord));
  BOOST_CHECK_EQUAL(resultVec.size(), 0);
}

BOOST_FIXTURE_TEST_CASE(ACCtestMatch1, ACCPowersetMonitorFixture)
{
  automaton.states[0]->isMatch = true;
  automaton.states[1]->isMatch = false;
  automaton.states[2]->isMatch = false;
  automaton.states[3]->isMatch = false;
  automaton.states[4]->isMatch = false;
  automaton.states[5]->isMatch = false;

  std::array<Parma_Polyhedra_Library::Variable, 5> x = 
    {Parma_Polyhedra_Library::Variable(0), 
     Parma_Polyhedra_Library::Variable(1),
     Parma_Polyhedra_Library::Variable(2),
     Parma_Polyhedra_Library::Variable(3),
     Parma_Polyhedra_Library::Variable(4)};
  auto clock = Parma_Polyhedra_Library::Variable(5);
  std::vector<TimedWordEvent<NNC_Polyhedron, Coefficient>> dummyTimedWord;  
  Constraint_System cs;
  // staying at state[0] for 2 time units
  cs.insert(x[0] == 40 + 16);
  cs.insert(x[1] == 35 + 17);
  cs.insert(x[2] == 30 + 18);
  cs.insert(x[3] == 25 + 19);
  cs.insert(x[4] == 20 + 20);
  
  TimedWordEvent<NNC_Polyhedron, Coefficient> event = {NNC_Polyhedron(cs), 2};
  dummyTimedWord.push_back(event);

  feed(std::move(dummyTimedWord));
  BOOST_CHECK_EQUAL(resultVec.size(), 1);

  cs.insert(clock == 0);
  auto expectedZone = Pointset_Powerset<NNC_Polyhedron>(NNC_Polyhedron(cs));

  using namespace Parma_Polyhedra_Library::IO_Operators;
  std::cout << resultVec.front().zone << std::endl;
  BOOST_TEST((resultVec.front().zone == expectedZone));
  BOOST_CHECK_EQUAL(resultVec.front().timestamp, 2);
}

BOOST_AUTO_TEST_SUITE_END()
