#pragma once

#include <sstream>

#include "../../src/automaton.hh"

/*
  @brief This automaton is the Adaptive Cruise Controller model in https://easychair.org/publications/open/q6RV.
*/
struct ACCFixture {
  ACCFixture() {
    std::array<Parma_Polyhedra_Library::Variable, 5> x = 
      {Parma_Polyhedra_Library::Variable(0), 
       Parma_Polyhedra_Library::Variable(1),
       Parma_Polyhedra_Library::Variable(2),
       Parma_Polyhedra_Library::Variable(3),
       Parma_Polyhedra_Library::Variable(4)};
    auto clock = Parma_Polyhedra_Library::Variable(5);

    // Construct automaton
    automaton.states.resize(6);
    for (auto &state: automaton.states) {
      state = std::make_shared<PolyhedraHAState>();
    }
    /// set accept/reject
    automaton.states[0]->isMatch = false;
    automaton.states[1]->isMatch = false;
    automaton.states[2]->isMatch = false;
    automaton.states[3]->isMatch = false;
    automaton.states[4]->isMatch = false;
    automaton.states[5]->isMatch = true;

    /// set the invariant
    Parma_Polyhedra_Library::Constraint_System invariantCS;
    for (int i = 0; i < 4; ++i) {
      invariantCS.insert(x[i] - x[i + 1] <= 10);
      invariantCS.insert(x[i] - x[i + 1] >= 2);
    }
    invariantCS.insert(clock >= 0);

    for (auto &state: automaton.states) {
      state->invariant = Parma_Polyhedra_Library::NNC_Polyhedron(invariantCS);
    }

    /// set the flow
    std::array<std::array<std::pair<int, int>, 5>, 6> flowCoeff = 
      {{{std::make_pair(1, 8), {2, 17}, {1, 9}, {2, 19}, {1, 10}},
        {std::make_pair(1, 12), {1, 10}, {1, 8}, {1, 9}, {1, 10}},
        {std::make_pair(1, 12), {1, 12}, {1, 10}, {2, 17}, {2, 19}},
        {std::make_pair(1, 12), {1, 12}, {1, 12}, {1, 10}, {1, 9}},
        {std::make_pair(1, 12), {1, 12}, {1, 12}, {1, 12}, {1, 10}},
        {std::make_pair(1, 0), {1, 0}, {1, 0}, {1, 0}, {1, 0}},
        }};
    for (int i = 0; i < 6; ++i) {
      Parma_Polyhedra_Library::Constraint_System flowCS;
      for (int j = 0; j < 5; ++j) {
        flowCS.insert(flowCoeff[i][j].first * x[j] == flowCoeff[i][j].second);
      }
      flowCS.insert(clock == 1);
      automaton.states[i]->flow = Parma_Polyhedra_Library::NNC_Polyhedron(flowCS);
    }

    // set the initial zone
    Parma_Polyhedra_Library::Constraint_System initCS;
    initCS.insert(x[0] == 40);
    initCS.insert(x[1] == 35);
    initCS.insert(x[2] == 30);
    initCS.insert(x[3] == 25);
    initCS.insert(x[4] == 20);
    initCS.insert(clock == 0);
    automaton.states[0]->initZone = Parma_Polyhedra_Library::NNC_Polyhedron(initCS);
    for (int i = 1; i < 6; ++i) {
      // We put another dimension for clock variable
      automaton.states[i]->initZone = Parma_Polyhedra_Library::NNC_Polyhedron(Dimension + 1, Parma_Polyhedra_Library::EMPTY);
    }

    // Set the transitions
    const auto emptyVariables = Parma_Polyhedra_Library::Variables_Set();
    const auto universalZone = Parma_Polyhedra_Library::NNC_Polyhedron(Dimension + 1);

    // #### FROM STATE 0 ####
    automaton.states[0]->next.resize(4);
    {
      auto tmpZone = universalZone;
      tmpZone.add_constraint(x[0] - x[1] <= 4);
      automaton.states[0]->next[0] = {tmpZone, emptyVariables, universalZone, automaton.states[1]};
    }
    {
      auto tmpZone = universalZone;
      tmpZone.add_constraint(x[1] - x[2] <= 4);
      automaton.states[0]->next[1] = {tmpZone, emptyVariables, universalZone, automaton.states[2]};
    }
    {
      auto tmpZone = universalZone;
      tmpZone.add_constraint(x[2] - x[3] <= 4);
      automaton.states[0]->next[2] = {tmpZone, emptyVariables, universalZone, automaton.states[3]};
    }
    {
      auto tmpZone = universalZone;
      tmpZone.add_constraint(x[3] - x[4] <= 4);
      automaton.states[0]->next[3] = {tmpZone, emptyVariables, universalZone, automaton.states[4]};
    }

    // #### FROM STATE 1 ####
    automaton.states[1]->next.resize(2);
    {
      auto tmpZone = universalZone;
      tmpZone.add_constraint(x[0] - x[1] >= 4);
      automaton.states[1]->next[0] = {tmpZone, emptyVariables, universalZone, automaton.states[0]};
    }
    {
      auto tmpZone = universalZone;
      tmpZone.add_constraint(x[0] - x[1] <= 1);
      automaton.states[1]->next[1] = {tmpZone, emptyVariables, universalZone, automaton.states[5]};
    }

    // #### FROM STATE 2 ####
    automaton.states[2]->next.resize(2);
    {
      auto tmpZone = universalZone;
      tmpZone.add_constraint(x[1] - x[2] >= 4);
      automaton.states[2]->next[0] = {tmpZone, emptyVariables, universalZone, automaton.states[0]};
    }
    {
      auto tmpZone = universalZone;
      tmpZone.add_constraint(x[1] - x[2] <= 1);
      automaton.states[2]->next[1] = {tmpZone, emptyVariables, universalZone, automaton.states[5]};
    }

    // #### FROM STATE 3 ####
    automaton.states[3]->next.resize(2);
    {
      auto tmpZone = universalZone;
      tmpZone.add_constraint(x[2] - x[3] >= 4);
      automaton.states[3]->next[0] = {tmpZone, emptyVariables, universalZone, automaton.states[0]};
    }
    {
      auto tmpZone = universalZone;
      tmpZone.add_constraint(x[2] - x[3] <= 1);
      automaton.states[3]->next[1] = {tmpZone, emptyVariables, universalZone, automaton.states[5]};
    }

    // #### FROM STATE 4 ####
    automaton.states[4]->next.resize(2);
    {
      auto tmpZone = universalZone;
      tmpZone.add_constraint(x[3] - x[4] >= 4);
      automaton.states[4]->next[0] = {tmpZone, emptyVariables, universalZone, automaton.states[0]};
    }
    {
      auto tmpZone = universalZone;
      tmpZone.add_constraint(x[3] - x[4] <= 1);
      automaton.states[4]->next[1] = {tmpZone, emptyVariables, universalZone, automaton.states[5]};
    }

    // #### FROM STATE 5 ####
    automaton.states[5]->next.clear();
  }

  /*!
    @brief The hybrid automaton
    
    Since there are diagonal constraints and invariants, we use NNC_Polyhedron.
   */
  const std::size_t Dimension = 5;
  PolyhedraHA automaton {Dimension};
};
