#pragma once

#include "subject.hh"
#include "observer.hh"
#include "timed_word_subject.hh"
#include "automaton.hh"

#include <boost/unordered_set.hpp>

// #define DEBUG_VERBOSE

template<typename Zone>
struct VectorMonitorResult {
  std::size_t index;
  mpq_class timestamp;
  Zone zone;
};

template<typename Zone>
class VectorMonitor : public SingleSubject<VectorMonitorResult<Zone>>,
                      public Observer<TimedWordEvent<Zone, mpq_class>> {
public:
  explicit VectorMonitor(const HybridAutomaton<Zone, Parma_Polyhedra_Library::Variables_Set> &automaton) :timeVariable (Parma_Polyhedra_Library::Variable(automaton.dimension)), 
                                                                                                          dimension(automaton.dimension), automaton(automaton) {
    absTime = 0;
    configurations.clear();
    for (const State state: automaton.states) {
      configurations[state] = {state->initZone};
    }

  }

  virtual ~VectorMonitor() = default;

  void notify(const TimedWordEvent<Zone, mpq_class> &event) override {
    if (configurations.empty()) {
      std::cerr << "Warning: no reachable configurations\n";
      return;
    }
    Zone valuation = event.valuation;
    const mpq_class timestamp = event.timestamp;

    const mpq_class maxTime = timestamp - absTime;
    Zone currentZone = Zone(dimension + 1);
    currentZone.add_constraint(maxTime.get_den() * timeVariable <= maxTime.get_num());
    for (auto &conf: configurations) {
      const State state = conf.first;
      Vector tmpZones;
      for (Zone &zone: conf.second) {
        flow(state, zone);
        zone.intersection_assign(currentZone);
        assert(state->flow.space_dimension() == dimension + 1);
        assert(zone.space_dimension() == dimension + 1);
        assert(currentZone.space_dimension() == dimension + 1);
        if (!zone.is_empty()) {
          tmpZones.push_back(std::move(zone));
        }
      }
      conf.second = std::move(tmpZones);
    }

#ifdef DEBUG_VERBOSE
    std::cout << "Before Exploration\n";
    for (std::size_t i = 0; i < automaton.states.size(); ++i) {
      auto it = configurations.find(automaton.states[i]);
      if (it == configurations.end()) {
        std::cout << "not found\n";
      } else {
        using namespace Parma_Polyhedra_Library::IO_Operators;
        for (const Zone& zone: it->second) {
          std::cout << zone << "\n";
        }
      }
    }
#endif

    for (std::size_t i = 0; i < unrollBound; i++) {
#ifdef DEBUG_VERBOSE
      std::cout << i << std::endl;
#endif
      boost::unordered_map<State, Vector>  newConfigurations;
      for (const auto &conf: configurations) {
        const State state = conf.first;
        const Vector &zones = conf.second;
        for (const Transition& transition: state->next) {
          State target = transition.target.lock();
          Vector tmpZones;
          for (Zone zone: zones) {
            assert(zone.space_dimension() == dimension + 1);
            if (transit(transition, zone)) {
              flow(target, zone);
              zone.intersection_assign(currentZone);
            }
            if (!zone.is_empty()) {
              tmpZones.push_back(std::move(zone));
            }
          }
          if (!tmpZones.empty()) {
            auto it = newConfigurations.find(target);
            if (it != newConfigurations.end()) {
              // if target is not new
              //              it->second.reserve(it->second.size() + tmpZones.size());
              it->second.insert(it->second.end(), tmpZones.begin(), tmpZones.end());
            } else {
              // if target is new
              newConfigurations[target] = std::move(tmpZones);
            }
          }
        }
      }
      if (confAddIfNotContains(configurations, newConfigurations)) {
        break;
      }
#ifdef DEBUG_VERBOSE
      std::cout << "Exploration " << i << "\n";
      for (std::size_t i = 0; i < automaton.states.size(); ++i) {
        auto it = configurations.find(automaton.states[i]);
        if (it == configurations.end()) {
          std::cout << "not found\n";
        } else {
          using namespace Parma_Polyhedra_Library::IO_Operators;
          for (const Zone& zone: it->second) {
            std::cout << zone << "\n";
          }
        }
      }
#endif
    }

#ifdef DEBUG_VERBOSE
    std::cout << "After Exploration\n";
    for (std::size_t i = 0; i < automaton.states.size(); ++i) {
      auto it = configurations.find(automaton.states[i]);
      if (it == configurations.end()) {
        std::cout << "not found\n";
      } else {
        using namespace Parma_Polyhedra_Library::IO_Operators;
        for (const Zone& zone: it->second) {
          std::cout << zone << "\n";
        }
      }
    }
#endif

    // Here, we concatenate the polyhedra for the valuation (dimension is this->dimension) and another polyhedra for time (dimension is 1) to make a valuation with timestamp.
    valuation.concatenate_assign(Zone(Parma_Polyhedra_Library::Constraint_System(maxTime.get_den() * Parma_Polyhedra_Library::Variable(0) == maxTime.get_num())));

    for (auto it = configurations.begin(); it != configurations.end(); ) {
      const State state = it->first;
      for (auto itZone = it->second.begin(); itZone != it->second.end();) {
        assert(itZone->space_dimension() == dimension + 1);
#ifdef DEBUG_VERBOSE
        {
        auto it = std::find(automaton.states.begin(), automaton.states.end(), state);
        std::cout << "state " << it - automaton.states.begin() << " is reachable." << std::endl;
        //        if (automaton.states[3] == state) {
          using namespace Parma_Polyhedra_Library::IO_Operators;
          std::cout << "just before intersection: " << *itZone << std::endl;
          //        }
        }
#endif
        itZone->intersection_assign(valuation);
        if (itZone->is_empty()) {
          itZone = it->second.erase(itZone);
        } else {
#ifdef DEBUG_VERBOSE
          auto it = std::find(automaton.states.begin(), automaton.states.end(), state);
          std::cout << "state " << it - automaton.states.begin() << " is reachable." << std::endl;
#endif
          itZone->unconstrain(timeVariable);
          itZone->add_constraint(timeVariable == 0);
          itZone++;
        }
      }
      if (it->second.empty()) {
        it = configurations.erase(it);
        continue;
      }
      Powerset powerZones;
      toPowerset(std::move(it->second), powerZones);
      // powerZones.unconstrain(timeVariable);
      // powerZones.add_constraint(timeVariable == 0);
      powerZones.omega_reduce();
      powerZones.pairwise_reduce();
      toVector(std::move(powerZones), it->second);
      if (state->isMatch) {
        for (const Zone &zone: it->second) {
          this->notifyObservers({index, timestamp, zone});
        }
      }
      it++;
    }
    absTime = timestamp;
    index++;

#ifdef DEBUG_VERBOSE
    std::cout << "After Intersecting with the Observation\n";
    for (std::size_t i = 0; i < automaton.states.size(); ++i) {
      auto it = configurations.find(automaton.states[i]);
      if (it == configurations.end()) {
        std::cout << "not found\n";
      } else {
        using namespace Parma_Polyhedra_Library::IO_Operators;
        for (const Zone& zone: it->second) {
          std::cout << zone << "\n";
        }
      }
    }
#endif
  }

private:
  using Powerset = Parma_Polyhedra_Library::Pointset_Powerset<Zone>;
  using Vector = std::list<Zone>;
  using State = std::shared_ptr<AutomatonState<Zone, Parma_Polyhedra_Library::Variables_Set>>;
  using Transition = AutomatonTransition<Zone, Parma_Polyhedra_Library::Variables_Set>;
  boost::unordered_map<State, Vector> configurations;
  Parma_Polyhedra_Library::Variable timeVariable;
  mpq_class absTime;
  std::size_t index = 0;
  const std::size_t unrollBound = -1;
  const std::size_t dimension;
  HybridAutomaton<Zone, Parma_Polyhedra_Library::Variables_Set> automaton;

  void toVector(Powerset &&powerset, Vector &vector) {
    vector.clear();
    for (const auto &determinate: powerset) {
      vector.emplace_back(determinate.pointset());
    }
  }
  void toPowerset(const Vector &vector, Powerset &powerset) {
    powerset = Powerset(dimension + 1, Parma_Polyhedra_Library::EMPTY);
    for (const Zone& zone: vector) {
      powerset.add_disjunct(zone);
    }
  }
  void toPowerset(Vector &&vector, Powerset &powerset) {
    powerset = Powerset(dimension + 1, Parma_Polyhedra_Library::EMPTY);
    for (Zone& zone: vector) {
      powerset.add_disjunct(std::move(zone));
    }
  }

  void reduceVector(Vector &vector) {
    Powerset powerset;
    toPowerset(std::move(vector), powerset);
    powerset.omega_reduce();
    powerset.pairwise_reduce();
    toVector(std::move(powerset), vector);
  }

  bool vectorAddIfNotContains(Vector& a, const Vector &b) {
    Powerset power_a, power_b;
    toPowerset(std::move(a), power_a);
    toPowerset(b, power_b);
    bool result = power_a.contains(power_b);
    power_a.upper_bound_assign(power_b);
    power_a.omega_reduce();
    power_a.pairwise_reduce();
    toVector(std::move(power_a), a);
    return result;
  }
  
  bool confAddIfNotContains(boost::unordered_map<State, Vector> &a, const boost::unordered_map<State, Vector> &b) {
    bool result = true;
    for (const auto &b_pair: b) {
      auto it = a.find(b_pair.first);
      if (it == a.end()) {
        configurations[b_pair.first] = std::move(b_pair.second);
        result = false;
      } else {
        if (!vectorAddIfNotContains(it->second, b_pair.second)) {
          result = false;
        }
      }
    }
    return result;
  }
};
