/*!
  @file timed_word_parser.hh
  @brief Parser of a quantitative timed word
  @author Masaki Waga <masakiwaga@gmail.com>
  @date 2019/12/10
  @copyright GPL3+
 */
#pragma once

#include <istream>
#include <string>
#include <vector>
#include <iostream>
#include <sstream>
#include <regex>
#include "common_types.hh"

template<typename Zone, typename TimeStamp>
struct TimedWordEvent {
  Zone valuation;
  TimeStamp timestamp;
};

template<class T>
static inline bool  parseRational(std::string& str, T &numerator, T &denominator) {
  denominator = 1;

  const std::regex decimalsRegex {R"(^-?\d*.?\d+$)"};
  if (! std::regex_match(str, decimalsRegex)) {
    return false;
  }

  // Parse decimals
  auto pos = str.find('.');
  if (pos != std::string::npos) {
    for (int i = str.size() - pos; i > 1; --i) {
      denominator *= 10;
    }
    str.erase(pos, 1);
    // Remove the initial 0s
    if (str.front() == '0' || (str.at(0) == '-' && str.at(1) == '0') || (str.at(0) == '-' && str.at(1) == '.' && str.at(2) == '0')) {
      auto itBegin = str.begin();
      for (;itBegin != str.end() && *itBegin != '0'; itBegin++);
      auto itEnd = itBegin;
      for (;itEnd != str.end() && *itEnd == '0'; itEnd++);
      itEnd--;
      str.erase(itBegin, itEnd);
    }
  }
  std::stringstream ss;
  if (str.empty()) {
    numerator = 0;
  } else {
    ss << str;
    ss >> numerator;
  }
  return true;
}

template<class T>
std::istream& readNumber(std::istream &is, T& num) {
  is >> num;
  return is;
}

template<>
inline std::istream& readNumber<mpq_class>(std::istream &is, mpq_class& num) {
  std::string str;
  is >> str;
  mpz_class numerator, denominator;
  parseRational(str, numerator, denominator);
  num.get_num() = numerator;
  num.get_den() = denominator;
  num.canonicalize();
  return is;
}

/*!
  @class TimedWordParser
  @brief Parser of a timed word
 */
template<typename T, typename U, typename = void>
class TimedWordParser;

template<typename Zone, typename TimeStamp>
class TimedWordParser<Zone, TimeStamp, typename std::enable_if_t<std::conjunction<hasUnconstrain<Zone>, hasIntersectionAssign<Zone, Zone>>::value>> {
public:
  TimedWordParser(std::istream &is, const std::size_t valuationSize) : is(is), valuationSize(valuationSize) {
  }
  /*!
    @brief Parse and return an event with data
    @retval true If the parse succeeded
    @retval false If the parse failed
   */
  bool parse(TimedWordEvent<Zone, TimeStamp> &event) {
    std::string line;
    if (is.eof()) {
      return false;
    }
    Parma_Polyhedra_Library::Constraint_System cs;
    for (std::size_t i = 0; i < valuationSize; i++) {
      std::string str;
      if (is >> str) {
        // skip if the input is not decimals
        if (! std::regex_match(str, decimalsRegex)) {
          continue;
        }
        Parma_Polyhedra_Library::Coefficient numerator, denominator;
        parseRational(str, numerator, denominator);
        cs.insert(denominator * Parma_Polyhedra_Library::Variable(i) == numerator);
      } else {
        return false;
      }
    }
    event.valuation = Zone(valuationSize);
    event.valuation.add_constraints(std::move(cs));
    readNumber(is, event.timestamp);
    return true;
  }
private:
  std::istream &is;
  const std::size_t valuationSize;

  const std::regex decimalsRegex {R"(^-?\d*.?\d+$)"};
};
