/*!
  @file simulator.cc
  @brief Implement the main function of the simulator
  @author Masaki Waga <masakiwaga@gmail.com>
  @data 2020/01/16
*/
// (setq irony-additional-clang-options '("-std=c++17"))

#include "automaton_parser.hh"

#include <boost/program_options.hpp>
#include <random>
#include <valarray>

#ifndef VERSION
#define VERSION "HEAD"
#endif

using namespace boost::program_options;
using namespace boost;
using namespace std;

using boost::operator>>;
using std::operator>>;
using ::operator>>;

using Zone = Parma_Polyhedra_Library::NNC_Polyhedron;

/*
  @brief randomly sample one point from the given zone
 */
static mpq_class samplePointFromZone(const Zone &zone, const std::size_t index, std::mt19937 &mt) {
  using Box = Parma_Polyhedra_Library::Box<Parma_Polyhedra_Library::Rational_Interval>;
  using Interval = Parma_Polyhedra_Library::Rational_Interval;
  Box boxZone = Box(zone);
  if (Zone(boxZone) != zone) {
    std::cerr << "Warning: The initial configuration, flow, or updated_vars is not a box. This makes the result an overapproximation.\n";
  }
  Interval interval = boxZone.get_interval(Parma_Polyhedra_Library::Variable(index));
  double lower = interval.lower().get_d();
  double upper = interval.upper().get_d();
  if (lower == -std::numeric_limits<double>::infinity()) {
    lower = 0;
  }
  if (upper == std::numeric_limits<double>::infinity()) {
    upper = lower + 10;
  }
  if (ceil(lower) <= floor (upper)) {
    std::uniform_int_distribution<> valuaton_rand(ceil(lower), floor(upper));
    return valuaton_rand(mt);
  } else {
    std::uniform_real_distribution<double> valuaton_rand(lower, upper);
    return valuaton_rand(mt);
  }
}

/*
  @brief randomly sample one point from the given zone
 */
static void samplePointFromZone(Zone zone, std::valarray<mpq_class> &valuation, std::mt19937 &mt) {
  valuation = std::valarray(mpq_class(0), zone.space_dimension());
  for (int i = 0; i < zone.space_dimension(); i++) {
    valuation[i] = samplePointFromZone(zone, i, mt);
    zone.add_constraint(Variable(i) * valuation[i].get_den() == valuation[i].get_num());
  }
}
                                              
static bool satisfy(Zone zone, const std::valarray<mpq_class> valuation) {
  for (int i = 0; i < valuation.size(); i++) {
    zone.add_constraint(Variable(i) * valuation[i].get_den() == valuation[i].get_num());
  }
  return !zone.is_empty();
}

static int simulate(const std::string &hybridAutomatonFileName, std::size_t length) {
  using Variables = Parma_Polyhedra_Library::Variables_Set;
  using State = std::shared_ptr<AutomatonState<Zone, Parma_Polyhedra_Library::Variables_Set>>;
  using Transition = AutomatonTransition<Zone, Parma_Polyhedra_Library::Variables_Set>;
  // using Powerset = Parma_Polyhedra_Library::Pointset_Powerset<Zone>;
  using HAType = HybridAutomaton<Zone, Variables>; 
  using BoostHAType = BoostHybridAutomaton<Parma_Polyhedra_Library::Constraint_System, Variables>;

  std::random_device rnd;
  std::mt19937 mt(rnd());

  HAType HA;

  // parse HA
  std::ifstream in(hybridAutomatonFileName);
  if (in.fail()) {
    std::cerr << "Error: " << strerror(errno) << " " << hybridAutomatonFileName.c_str() << std::endl;
    return 1;
  }
  BoostHAType BoostHA;
  parseBoostHA(in, BoostHA);
  convBoostHA(BoostHA, HA);

  std::vector<State> states;
  std::vector<std::valarray<mpq_class>> valuations;
  std::vector<unsigned long> timePoints;
  std::vector<unsigned long> usedCount;
  states.reserve(length);
  valuations.reserve(length);
  timePoints.reserve(length);
  usedCount.reserve(length);

  std::uniform_int_distribution<> duration_rand(1, 5);
  std::uniform_int_distribution<> transition_rand(0, 1);
  while (valuations.size() < length) {
    //    printf("%lu\n", valuations.size());
    assert(valuations.size() == timePoints.size());
    assert(valuations.size() == usedCount.size());
    assert(valuations.size() == states.size());
    if (valuations.empty()) {
      // pick an initial configuration randomly.
      std::vector<std::pair<State, Zone>> initialConfigurations;
      initialConfigurations.clear();
      for (const State state: HA.states) {
        if (!state->initZone.is_empty()) {
          initialConfigurations.emplace_back(state, state->initZone);
        }
      }
      std::uniform_int_distribution<> ranged_rand(0, initialConfigurations.size() - 1);
      const auto initialConfiguration = initialConfigurations.at(ranged_rand(mt));
      states.push_back(initialConfiguration.first);
      valuations.emplace_back(0.0, HA.dimension);
      samplePointFromZone(initialConfiguration.second, valuations.back(), mt);
      timePoints.push_back(0);
      usedCount.push_back(0);
    } else {
      bool success = false;
      for (int i = 0; i < 10; i++) {
        usedCount.back()++;
        unsigned long duration = duration_rand(mt);
        std::valarray flow(mpq_class(0), HA.dimension);
        samplePointFromZone(states.back()->flow, flow, mt);
        std::valarray candidateValuation = valuations.back();
        //        std::cout << "flow: " << flow[0].get_d() << "\t" << flow[1].get_d() << "\n";
        candidateValuation += flow * duration;
        //        std::cout << "candidate valuation: " << candidateValuation[0].get_d() << "\t" << candidateValuation[1].get_d() << "\n";
        if (satisfy(states.back()->invariant, candidateValuation)) {
          //          puts("satisfied");
          valuations.emplace_back(std::move(candidateValuation));
          timePoints.push_back(timePoints.back() + duration);
          usedCount.push_back(0);
          states.push_back(states.back());
          success = true;
          break;
        }
        //        puts("NOT satisfied");
      }
      if (!success) {
        while (!usedCount.empty() && usedCount.back() > 5) {
          //          std::cout << valuations.back()[0] << "\t" << valuations.back()[1] << "\n";
          states.pop_back();
          valuations.pop_back();
          timePoints.pop_back();
          usedCount.pop_back();
        }
        continue;
      }
      std::vector<Transition> availableTransitions;
      std::copy_if(states.back()->next.begin(), 
                   states.back()->next.end(), 
                   std::back_inserter(availableTransitions),
                   [&valuations](Transition transition) {
                     return satisfy(transition.guard, valuations.back());
                   });
      if (!availableTransitions.empty() && !transition_rand(mt)) {
        std::uniform_int_distribution<> transition_rand(0, availableTransitions.size() - 1);
        Transition transition = availableTransitions.at(transition_rand(mt));
        states.back() = transition.target.lock();
        for (const auto variable: transition.updatedVars) {
          valuations.back()[variable] = samplePointFromZone(transition.updatedZone, variable, mt);
        }
      }
    }
  }

  for (int i = 0; i < length; i++) {
    for  (int j = 0; j < valuations.at(i).size() - 1; j++) {
      printf("%lf\t", valuations.at(i)[j].get_d());
    }
    printf("%lu\n", timePoints.at(i));
  }

  return 0;
}

int main(int argc, char *argv[]) {
  const auto programName = "HAMoniSim";

  const auto die =
    [&programName](const char *message, int status) {
      std::cerr << programName << ": " << message << std::endl;
      exit(status);
    };

  // visible options
  options_description visible("description of options");
  std::string hybridAutomatonFileName;
  std::size_t length;
  visible.add_options()
    ("help,h", "help")
    ("version,V", "version")
    ("length,l", value<std::size_t>(&length)->default_value(0), "lenght of the simulation")
    ("automaton,f", value<std::string>(&hybridAutomatonFileName)->default_value("stdin"),
     "input file of the hybrid automaton");

  command_line_parser parser(argc, argv);
  parser.options(visible);
  variables_map vm;
  const auto parseResult = parser.run();
  store(parseResult, vm);
  notify(vm);

  if (vm.count("version")) {
    std::cout << programName << " " << VERSION << "\n"
              << visible << std::endl;
    return 0;
  }
  if (hybridAutomatonFileName.empty() || vm.count("help")) {
    std::cout << programName << " [OPTIONS] (-f <automaton_file>)\n"
              << visible << std::endl;
    return 0;
  }
  if (length == 0) {
    die("positive length must be given", 1);
  }

  if (hybridAutomatonFileName == "stdin") {
    die("hybrid automton file name must be given", 1);
  }

  return simulate(hybridAutomatonFileName, length);
 
  return 0;
}
