#pragma once

#include "subject.hh"
#include "observer.hh"
#include "timed_word_subject.hh"
#include "automaton.hh"

#include <boost/unordered_set.hpp>

template<typename Zone>
struct PowersetMonitorResult {
  std::size_t index;
  mpq_class timestamp;
  Parma_Polyhedra_Library::Pointset_Powerset<Zone> zone;
};

template<typename Zone>
class PowersetMonitor : public SingleSubject<PowersetMonitorResult<Zone>>,
                        public Observer<TimedWordEvent<Zone, mpq_class>> {
public:
  explicit PowersetMonitor(const HybridAutomaton<Zone, Parma_Polyhedra_Library::Variables_Set> &automaton) :timeVariable (Parma_Polyhedra_Library::Variable(automaton.dimension)), dimension(automaton.dimension), automaton(automaton) {
    absTime = 0;
    configurations.clear();
    for (const State state: automaton.states) {
      configurations[state] = Powerset(state->initZone);
      assert(configurations[state].space_dimension() == dimension + 1);
    }

  }

  virtual ~PowersetMonitor() = default;

  void notify(const TimedWordEvent<Zone, mpq_class> &event) override {
    if (configurations.empty()) {
      std::cerr << "Warning: no reachable configurations\n";
      return;
    }
    Zone valuation = event.valuation;
    const mpq_class timestamp = event.timestamp;
    boost::unordered_map<State, Powerset>  nextConfigurations;
    mpq_class maxTime = timestamp - absTime;

    Zone currentZone = Zone(dimension + 1);
    currentZone.add_constraint(maxTime.get_den() * timeVariable <= maxTime.get_num());
    for (auto &conf: configurations) {
      const State state = conf.first;
      Powerset tmpZones = Powerset(dimension + 1, Parma_Polyhedra_Library::EMPTY);
      for (const auto &determinate: conf.second) {
        Zone zone = determinate.pointset();
        flow(state, zone);
        zone.intersection_assign(currentZone);
        assert(state->flow.space_dimension() == dimension + 1);
        assert(zone.space_dimension() == dimension + 1);
        assert(currentZone.space_dimension() == dimension + 1);
        tmpZones.add_disjunct(zone);
      }
      conf.second = std::move(tmpZones);
    }

#ifdef DEBUG_VERBOSE
    std::cout << "Before Exploration\n";
    for (std::size_t i = 0; i < 6; ++i) {
      auto it = configurations.find(automaton.states[i]);
      if (it == configurations.end()) {
        std::cout << "not found\n";
      } else {
        using namespace Parma_Polyhedra_Library::IO_Operators;
        std::cout << it->second << "\n";
      }
    }
#endif

    for (std::size_t i = 0; i < unrollBound; i++) {
#ifdef DEBUG_VERBOSE
      std::cout << i << std::endl;
#endif
      boost::unordered_map<State, Powerset>  tmpConfigurations = configurations;
      for (const auto &conf: configurations) {
        const State state = conf.first;
        const Powerset zones = conf.second;
        for (const Transition& transition: state->next) {
          State target = transition.target.lock();
          Powerset tmpZones = Powerset(dimension + 1, Parma_Polyhedra_Library::EMPTY);
          for (const auto &determinate: zones) {
            assert(determinate.pointset().space_dimension() == dimension + 1);
            Zone zone = determinate.pointset();
            using namespace Parma_Polyhedra_Library::IO_Operators;
            if (transit(transition, zone)) {
              flow(target, zone);
              zone.intersection_assign(currentZone);
            }
            tmpZones.add_disjunct(zone);
          }
          if (!tmpZones.is_empty()) {
            if (tmpConfigurations.find(target) != tmpConfigurations.end()) {
              assert(tmpConfigurations[target].space_dimension() == dimension + 1);
              assert(tmpZones.space_dimension() == dimension + 1);
              // if target is not new
              tmpConfigurations[target].least_upper_bound_assign(std::move(tmpZones));
            } else {
              // if target is new
              tmpConfigurations[target] = std::move(tmpZones);
            }
            tmpConfigurations[target].omega_reduce();
            tmpConfigurations[target].pairwise_reduce();
          }
        }
      }
      if (tmpConfigurations == configurations) {
        break;
      }
      configurations = std::move(tmpConfigurations);
#ifdef DEBUG_VERBOSE
      std::cout << "Exploration " << i << "\n";
      for (std::size_t i = 0; i < 6; ++i) {
        auto it = configurations.find(automaton.states[i]);
        if (it == configurations.end()) {
          std::cout << "not found\n";
        } else {
          using namespace Parma_Polyhedra_Library::IO_Operators;
          std::cout << it->second << "\n";
        }
      }
#endif
    }

#ifdef DEBUG_VERBOSE
    std::cout << "After Exploration\n";
    for (std::size_t i = 0; i < 6; ++i) {
      auto it = configurations.find(automaton.states[i]);
      if (it == configurations.end()) {
        std::cout << "not found\n";
      } else {
        using namespace Parma_Polyhedra_Library::IO_Operators;
        std::cout << it->second << "\n";
      }
    }
#endif

    // Here, we concatenate the polyhedra for the valuation (dimension is this->dimension) and another polyhedra for time (dimension is 1) to make a valuation with timestamp.
    valuation.concatenate_assign(Zone(Parma_Polyhedra_Library::Constraint_System(maxTime.get_den() * Parma_Polyhedra_Library::Variable(0) == maxTime.get_num())));

    for (auto it = configurations.begin(); it != configurations.end(); ) {
      const State state = it->first;
      Powerset &zones = it->second;
      zones.omega_reduce();
      zones.pairwise_reduce();
      zones.intersection_assign(Powerset(valuation));
      zones.unconstrain(timeVariable);
      zones.add_constraint(timeVariable == 0);
      if (zones.is_empty()) {
        it = configurations.erase(it);
        continue;
      } else {
        if (state->isMatch) {
          this->notifyObservers({index, timestamp, zones});          
        }
      }
      it++;
    }
    absTime = timestamp;
    index++;

#ifdef DEBUG_VERBOSE
    std::cout << "After Intersecting with the Observation\n";
    for (std::size_t i = 0; i < 6; ++i) {
      auto it = configurations.find(automaton.states[i]);
      if (it == configurations.end()) {
        std::cout << "not found\n";
      } else {
        using namespace Parma_Polyhedra_Library::IO_Operators;
        std::cout << it->second << "\n";
      }
    }
#endif
  }

private:
  using Powerset = Parma_Polyhedra_Library::Pointset_Powerset<Zone>;
  using State = std::shared_ptr<AutomatonState<Zone, Parma_Polyhedra_Library::Variables_Set>>;
  using Transition = AutomatonTransition<Zone, Parma_Polyhedra_Library::Variables_Set>;
  boost::unordered_map<State, Powerset> configurations;
  Parma_Polyhedra_Library::Variable timeVariable;
  mpq_class absTime;
  std::size_t index = 0;
  const std::size_t unrollBound = -1;
  const std::size_t dimension;
  HybridAutomaton<Zone, Parma_Polyhedra_Library::Variables_Set> automaton;
};
