#define BOOST_GRAPH_USE_SPIRIT_PARSER // for header only

#include "automaton_parser.hh"

#include <boost/program_options.hpp>

#include "powerset_monitor.hh"
#include "vector_monitor.hh"
#include "printer.hh"

#ifndef VERSION
#define VERSION "HEAD"
#endif

using namespace boost::program_options;
using namespace boost;
using namespace std;

using boost::operator>>;
using std::operator>>;
using ::operator>>;

/*!
 * @brief Execute the monitoring procedure
 *
 * @param [in] hybridAutomatonFileName filename of the timed automaton
 * @param [in] timedWordFileName filename of the timed word. When it is "stdin", the monitor reads from standard input.
 */
template<typename HAType, typename BoostHAType, typename Zone, typename Timestamp, typename Monitor, typename Printer>
int execute(const std::string &hybridAutomatonFileName,
            const std::string &timedWordFileName) {
  HAType HA;

  // parse HA
  std::ifstream haStream(hybridAutomatonFileName);
  if (haStream.fail()) {
    std::cerr << "Error: " << strerror(errno) << " " << hybridAutomatonFileName.c_str() << std::endl;
    return 1;
  }
  BoostHAType BoostHA;
  parseBoostHA(haStream, BoostHA);
  convBoostHA(BoostHA, HA);

  // construct BooleanPrinter
  const auto printer = std::make_shared<Printer>();

  // construct Monitor
  const auto monitor = std::make_shared<Monitor>(HA);
  monitor->addObserver(printer);

  // construct TimedWordParser
  std::unique_ptr<TimedWordParser<Zone, Timestamp>> timedWordParser;
  std::fstream timedWordFileStream;
  if (timedWordFileName == "stdin") {
    timedWordParser = std::make_unique<TimedWordParser<Zone, Timestamp>>(std::cin, HA.dimension);
  } else {
    timedWordFileStream.open(timedWordFileName);
    if (timedWordFileStream.fail()) {
      std::cerr << "Error: " << strerror(errno) << " " << timedWordFileName.c_str() << std::endl;
      return 1;
    }
    timedWordParser = std::make_unique<TimedWordParser<Zone, Timestamp>>(timedWordFileStream, HA.dimension);
  }

  // construct TimedWordSubject
  TimedWordSubject<Zone, Timestamp> timedWordSubject(std::move(timedWordParser));
  timedWordSubject.addObserver(monitor);

  // monitor all
  timedWordSubject.parseAndSubjectAll();
  return 0;
}

int main(int argc, char *argv[]) {
  const auto programName = "HAMoni";
  std::cin.tie(0);
  std::ios::sync_with_stdio(false);

  const auto die =
    [&programName](const char *message, int status) {
      std::cerr << programName << ": " << message << std::endl;
      exit(status);
    };

  // visible options
  options_description visible("description of options");
  std::string timedWordFileName;
  std::string hybridAutomatonFileName;
  visible.add_options()
    ("help,h", "help")
    ("powerset", "Monitor only using Pointset_Powerset of PPL")
    ("vector", "Monitor mainly using std::vector")
    ("polyhedron", "Monitor using NNC_Polyhedron of PPL")
    ("c_polyhedron", "Monitor using C_Polyhedron of PPL")
    // ("octagonal", "Monitor using Octagonal_Shape of PPL")
    ("bds", "Monitor using BD_Shape of PPL")
    ("box", "Monitor using Box<RationalInterval> of PPL")
    ("version,V", "version")
    ("input,i", value<std::string>(&timedWordFileName)->default_value("stdin"), "input file of quantitative timed words")
    ("automaton,f", value<std::string>(&hybridAutomatonFileName)->default_value(""),
     "input file of the hybrid automaton");

  command_line_parser parser(argc, argv);
  parser.options(visible);
  variables_map vm;
  const auto parseResult = parser.run();
  store(parseResult, vm);
  notify(vm);

  if (vm.count("version")) {
    std::cout << programName << " " << VERSION << "\n"
              << visible << std::endl;
    return 0;
  }
  if (hybridAutomatonFileName.empty() || vm.count("help")) {
    std::cout << programName << " [OPTIONS] -f <automaton_file> (-i <timedword_file>)\n"
              << visible << std::endl;
    return 0;
  }

  if (vm.count("powerset")) {
    if (vm.count("polyhedron")) {
      using Zone = Parma_Polyhedra_Library::NNC_Polyhedron;
      using Variables = Parma_Polyhedra_Library::Variables_Set;
      return execute<HybridAutomaton<Zone, Variables>,
                     BoostHybridAutomaton<Parma_Polyhedra_Library::Constraint_System, Variables>,
                     Zone,
                     mpq_class,
                     PowersetMonitor<Zone>,
                     PowersetPrinter<Zone>>(hybridAutomatonFileName, timedWordFileName);
    } else if (vm.count("c_polyhedron")) {
      using Zone = Parma_Polyhedra_Library::C_Polyhedron;
      using Variables = Parma_Polyhedra_Library::Variables_Set;
      return execute<HybridAutomaton<Zone, Variables>,
                     BoostHybridAutomaton<Parma_Polyhedra_Library::Constraint_System, Variables>,
                     Zone,
                     mpq_class,
                     PowersetMonitor<Zone>,
                     PowersetPrinter<Zone>>(hybridAutomatonFileName, timedWordFileName);
      // } else if (vm.count("octagonal")) {
      //   using Zone = Parma_Polyhedra_Library::Octagonal_Shape<mpq_class>;
      //   using Variables = Parma_Polyhedra_Library::Variables_Set;
      //   return execute<HybridAutomaton<Zone, Variables>,
      //                  BoostHybridAutomaton<Parma_Polyhedra_Library::Constraint_System, Variables>,
      //                  Zone,
      //                  Parma_Polyhedra_Library::Coefficient,
      //                  PowersetMonitor<Zone>,
      //                  PowersetPrinter<Zone>>(hybridAutomatonFileName, timedWordFileName);
    } else if (vm.count("bds")) {
      using Zone = Parma_Polyhedra_Library::BD_Shape<mpq_class>;
      using Variables = Parma_Polyhedra_Library::Variables_Set;
      return execute<HybridAutomaton<Zone, Variables>,
                     BoostHybridAutomaton<Parma_Polyhedra_Library::Constraint_System, Variables>,
                     Zone,
                     mpq_class,
                     PowersetMonitor<Zone>,
                     PowersetPrinter<Zone>>(hybridAutomatonFileName, timedWordFileName);
    } else if (vm.count("box")) {
      using Zone = Parma_Polyhedra_Library::Box<Parma_Polyhedra_Library::Rational_Interval>;
      using Variables = Parma_Polyhedra_Library::Variables_Set;
      return execute<HybridAutomaton<Zone, Variables>,
                     BoostHybridAutomaton<Parma_Polyhedra_Library::Constraint_System, Variables>,
                     Zone,
                     mpq_class,
                     PowersetMonitor<Zone>,
                     PowersetPrinter<Zone>>(hybridAutomatonFileName, timedWordFileName);
    }
    else {
      die("Invalid options", 1);
    }
  } else if (vm.count("vector")) {
    if (vm.count("polyhedron")) {
      using Zone = Parma_Polyhedra_Library::NNC_Polyhedron;
      using Variables = Parma_Polyhedra_Library::Variables_Set;
      return execute<HybridAutomaton<Zone, Variables>,
                     BoostHybridAutomaton<Parma_Polyhedra_Library::Constraint_System, Variables>,
                     Zone,
                     mpq_class,
                     VectorMonitor<Zone>,
                     VectorPrinter<Zone>>(hybridAutomatonFileName, timedWordFileName);
    } else if (vm.count("c_polyhedron")) {
      using Zone = Parma_Polyhedra_Library::C_Polyhedron;
      using Variables = Parma_Polyhedra_Library::Variables_Set;
      return execute<HybridAutomaton<Zone, Variables>,
                     BoostHybridAutomaton<Parma_Polyhedra_Library::Constraint_System, Variables>,
                     Zone,
                     mpq_class,
                     VectorMonitor<Zone>,
                     VectorPrinter<Zone>>(hybridAutomatonFileName, timedWordFileName);
      // } else if (vm.count("octagonal")) {
      //   using Zone = Parma_Polyhedra_Library::Octagonal_Shape<mpq_class>;
      //   using Variables = Parma_Polyhedra_Library::Variables_Set;
      //   return execute<HybridAutomaton<Zone, Variables>,
      //                  BoostHybridAutomaton<Parma_Polyhedra_Library::Constraint_System, Variables>,
      //                  Zone,
      //                  mpq_class,
      //                  VectorMonitor<Zone>,
      //                  VectorPrinter<Zone>>(hybridAutomatonFileName, timedWordFileName);
    } else if (vm.count("bds")) {
      using Zone = Parma_Polyhedra_Library::BD_Shape<mpq_class>;
      using Variables = Parma_Polyhedra_Library::Variables_Set;
      return execute<HybridAutomaton<Zone, Variables>,
                     BoostHybridAutomaton<Parma_Polyhedra_Library::Constraint_System, Variables>,
                     Zone,
                     mpq_class,
                     VectorMonitor<Zone>,
                     VectorPrinter<Zone>>(hybridAutomatonFileName, timedWordFileName);
    } else if (vm.count("box")) {
      using Zone = Parma_Polyhedra_Library::Box<Parma_Polyhedra_Library::Rational_Interval>;
      using Variables = Parma_Polyhedra_Library::Variables_Set;
      return execute<HybridAutomaton<Zone, Variables>,
                     BoostHybridAutomaton<Parma_Polyhedra_Library::Constraint_System, Variables>,
                     Zone,
                     mpq_class,
                     VectorMonitor<Zone>,
                     VectorPrinter<Zone>>(hybridAutomatonFileName, timedWordFileName);
    }
    else {
      die("Invalid options", 1);
    }
  } else {
    die("Invalid options", 1);
  }
 
  return 0;
}
