#pragma once

#include <vector>
#include <boost/unordered_map.hpp>
#include <memory>

#include "common_types.hh"

template<typename Zone, typename Variables>
struct AutomatonState;

/*!
  @brief A transition of a hybrid automaton
 */
template<typename Zone, typename Variables, typename = void>
struct AutomatonTransition {
  //! @brief The guard for this transition
  Zone guard;
  //! @brief The variables updated after this transition.
  Variables updatedVars;
  //! @brief The updated Zone.
  Zone updatedZone;
  std::weak_ptr<AutomatonState<Zone, Variables>> target;
};

/*! 
  @brief Evaluate the zone and update if the transition is possible.
  
  @param[in] zone The zone to be evaluated. zone is destroyed.
  @retval true If the transition is possible.
*/
template<typename HAZone, typename ConfZone>
bool transit(const AutomatonTransition<HAZone, Parma_Polyhedra_Library::Variables_Set> &transition, ConfZone& confZone) {
  static_assert(hasUnconstrain<ConfZone>::value, "ConfZone::unconstrain is unavailable");
  static_assert(hasIntersectionAssign<ConfZone, HAZone>::value, "ConfZone::intersection_assign(HAZone) is unavailable");

  confZone.intersection_assign(transition.guard);
  if (confZone.is_empty()) {
    return false;
  }
  confZone.unconstrain(transition.updatedVars);
  confZone.intersection_assign(transition.updatedZone);

  // Ensure that the invariant at the target state is satisfied.
  confZone.intersection_assign(transition.target.lock()->invariant);
  if (confZone.is_empty()) {
    return false;
  }
  return true;
}

/*!
  @brief A state of a hybrid automaton
 */
template<typename Zone, typename Variables>
struct AutomatonState {
  //! @brief The value is true if and only if the state is an accepting state.
  bool isMatch;
  //! @brief The invariant at this state
  Zone invariant;
  //! @brief The flow at this state
  Zone flow;
  //! @brief The initial valuations at this state
  Zone initZone;
  /*! 
    @brief An mapping of a character to the transitions.
    @note Because of non-determinism, the second element is a vector.
   */
  std::vector<AutomatonTransition<Zone, Variables>> next;
};

/*! 
  @brief Evaluate the flow
  
  @param[in] state The state to stay.
  @param[in] zone The zone to be evaluated. zone is destroyed.
*/
template<typename HAZone, typename ConfZone>
void flow(const std::shared_ptr<AutomatonState<HAZone, Parma_Polyhedra_Library::Variables_Set>> state, ConfZone& confZone) {
  static_assert(hasTimeElapseAssign<ConfZone, HAZone>::value, "ConfZone::time_elapse_assign(HAZone) is unavailable");
  static_assert(hasIntersectionAssign<ConfZone, HAZone>::value, "ConfZone::intersection_assign(HAZone) is unavailable");

  confZone.time_elapse_assign(state->flow);
  confZone.intersection_assign(state->invariant);
}


/*!
  @brief A hybrid automaton
 */
template<typename Zone, typename Variables>
struct HybridAutomaton : public Automaton<AutomatonState<Zone, Variables>> {
  //! @brief The dimension of the dynamics. The special dimension for time is excluded.
  std::size_t dimension;
  HybridAutomaton(const std::size_t dimension) :dimension(dimension) {}
  HybridAutomaton() :dimension(0) {}
};

template<typename ITV>
using BoxHA = HybridAutomaton<Parma_Polyhedra_Library::Box<ITV>, Parma_Polyhedra_Library::Variables_Set>;
template<typename ITV>
using BoxHAState = AutomatonState<Parma_Polyhedra_Library::Box<ITV>, Parma_Polyhedra_Library::Variables_Set>;

using PolyhedraHA = HybridAutomaton<Parma_Polyhedra_Library::NNC_Polyhedron, Parma_Polyhedra_Library::Variables_Set>;
using PolyhedraHAState = AutomatonState<Parma_Polyhedra_Library::NNC_Polyhedron, Parma_Polyhedra_Library::Variables_Set>;
