#!/usr/bin/env stack
-- stack runghc
{-# LANGUAGE TupleSections #-}

import System.Environment (getArgs)
import Data.List (intercalate)
import Control.Monad (mapM)

invariant :: Int -> Bool -> String
-- isCrs == True
invariant i True = "-x" ++ show (i - 1) ++ " + x" ++ show i ++ " <= -1"
-- isCrs == False
invariant i False = "x" ++ show (i - 1) ++ " - x" ++ show i ++ " <= 3"

flow :: Bool -> Int -> Bool -> String
-- isCrs == True
flow _ i True = "x" ++ pn ++ " - x" ++ n ++ " <= 1, -1 <= x" ++ pn ++ " - x" ++ n
  where
    n = show i
    pn = show $ i - 1
-- isCrs == False && isU == True
flow True i False = "10 * x" ++ pn ++ " - 10 * x" ++ n ++ " - 9 <= 10, -10 <= 10 * x" ++ pn ++ " - 10 * x" ++ n ++ " - 9"
  where
    n = show i
    pn = show $ i - 1
-- isCrs == False && isU == False
flow False i False = "x" ++ pn ++ " - x" ++ n ++ " - 2 <= 1, -1 <= x" ++ pn ++ " - x" ++ n ++ " - 2"
  where
    n = show i
    pn = show $ i - 1

generateName :: [Bool] -> Bool -> String
generateName isCrss isCrash = crashString ++ intercalate "_" bitsNameList
  where
    crashString = if isCrash then "crash_" else ""
    bitsNameList = map (\isCrs -> if isCrs then "crs" else "rcv") isCrss

-- | The function to generate the location
generateLoc :: [Bool] -- ^ the list of isCrs
            -> Bool -- ^ isU
            -> Bool -- ^ isCrash
            -> Bool -- ^ isSafety
            -> String -- ^ the result
generateLoc isCrss isU isCrash isSafety = unlines [_matchLine, _invariantLine, _flowLine, _initZoneLine ++ "];"]
  where
    _name = generateName isCrss isCrash
    _matchLine = _name ++ " [match=" ++ (if isSafety && not isCrash then "0" else "1") ++ ","
    _invariants = zipWith invariant [1..] isCrss
    _invariantLine = "invariant=\"{" ++ intercalate ", " _invariants ++ "};\","
    _flowPositive = intercalate ", " $ map (\i -> "0 <= x" ++ show i) [1..length isCrss]
    _flows = intercalate  ", " $ zipWith (flow isU) [1..] isCrss
    _flowLine = "flow=\"{x0 == 36, " ++ _flowPositive ++ ", " ++ _flows ++ "};\","
    initZoneValue = reverse $ map (3 *) [0..length isCrss]
    _initZones = intercalate ", " $ zipWith (\i v -> "x"++ show i ++ " == " ++ show v) [0..length isCrss] initZoneValue
    _initZoneLine = "init_zone=\"" ++ (if and isCrss && not isCrash then "{" ++ _initZones ++ "};" else "{false};") ++ "\""

generateEdge :: [Bool] -> [Bool] -> Bool -> String
generateEdge srcBits tgtBits isCrash = srcName ++ " -> " ++ tgtName ++ " [" ++ guardString ++ ", " ++ updatedVars ++ ", "++ updatedZone ++ "];"
  where
    srcName = generateName srcBits isCrash
    tgtName = generateName tgtBits isCrash
    changedBits :: [(Bool, Int)]
    changedBits = map (\ (a, _, c) -> (a, c)) $ filter (\ (srcBit, tgtBit, _) -> srcBit /= tgtBit) $ zip3 srcBits tgtBits [1..]
    makeGuard :: Bool -> Int -> String
    -- fromCrc == True
    makeGuard True i = "x" ++ show (i - 1) ++ " - x" ++ show i ++ " <= 2"
    -- fromCrc == False
    makeGuard False i = "x" ++ show (i - 1) ++ " - x" ++ show i ++ " >= 2"
    guards = map (uncurry makeGuard) changedBits
    guardString = "guard=\"{" ++ intercalate ", " guards ++ "};\""
    updatedVars = "updated_vars=\"{}\""
    updatedZone = "updated_zone=\"{true};\""

generateCrashEdges :: [Bool] -> [String]
generateCrashEdges isCrss = map (\ guard -> srcName ++ " -> " ++ tgtName ++ " [" ++ guard ++ ", " ++ updatedVars ++ ", "++ updatedZone ++ "];") guards
  where
    srcIsCrash = False
    tgtIsCrash = True
    srcName = generateName isCrss srcIsCrash
    tgtName = generateName isCrss tgtIsCrash
    guards = map (\ i -> "guard=\"{x" ++ show (i - 1) ++ " <= x" ++ show i ++ "};\"") [1..length isCrss]
    updatedVars = "updated_vars=\"{}\""
    updatedZone = "updated_zone=\"{true};\""

allCombinations :: Int -> [[Bool]]
allCombinations size = mapM (const [True, False]) [1..size]

main :: IO ()
main = do
  args <- getArgs
  if length args < 3 then
    putStrLn "Usage: ./genACCDot.hs [U|S] [safety|conformance] [DIMENSION]"
    else
      let isU = "U" == head args
          isSafety = (args !! 1) == "safety"
          dimension = (read $ args !! 2 :: Int) in
        if dimension < 1 then
          putStrLn "Error: dimension should be a positive integer."
        else 
          let header = "digraph ACC" ++ (if isU then "U" else "S") ++ show dimension ++ " {"

              dimLine = "    graph [dimension = " ++ show dimension ++ "];"
              allBits = allCombinations $ dimension - 1
              noCrashLocations :: [String]
              noCrashLocations = map (\ isCrss -> generateLoc isCrss isU False isSafety) allBits
              crashLocations = map (\ isCrss -> generateLoc isCrss isU True isSafety) allBits
              locations :: [String]
              locations = noCrashLocations  ++ if isSafety then crashLocations else []
              noCrashEdges = map (\ (srcBits, tgtBits) -> generateEdge srcBits tgtBits False) $ filter (uncurry (/=)) $ concatMap (\a -> map (a,) allBits) allBits
              crashedEdges = map (\ (srcBits, tgtBits) -> generateEdge srcBits tgtBits True) $ filter (uncurry (/=)) $ concatMap (\a -> map (a,) allBits) allBits
              crashEdges = concatMap generateCrashEdges allBits
              edges = noCrashEdges ++ if isSafety then crashedEdges ++ crashEdges else []
              loc_edge_separator = "//TRANSITIONS"
              footer = "}" in
            putStrLn $ unlines $ [header, dimLine] ++ locations ++ [loc_edge_separator] ++ edges ++ [footer]
