(************************************************************
 *
 *                       Graph2MC
 *
 * Université de Lorraine, CNRS, Inria, LORIA, Nancy, France
 *
 * Module description: Convert a graph to PHAVerLite syntax
 *
 * File contributors : Étienne André
 * Created           : 2020/01/20
 * Last modified     : 2020/01/23
 *
 ************************************************************)


(************************************************************)
(** Internal modules *)
(************************************************************)
open OCamlUtilities
open Exceptions
open ParsingStructure

(************************************************************)
(** Constants *)
(************************************************************)
let string_of_true	= "true"
let string_of_false	= "false"

let sPECIAL_COMMENT = "-- HERE COMES THE WORD"

let string_of_var_index variable_index = "x" ^ (string_of_int variable_index)


(************************************************************)
(** Header *)
(************************************************************)
let model_header parsed_graph =
	"-- PHAVerLite model generated automatically by " ^ program_name ^ " from '" ^ Sys.argv.(1) ^ "' on " ^ (now())
	^
	"\n" ^ "automaton " ^ parsed_graph.name ^ "\n"


let string_of_synclabs = "\nsynclabs: tau;"

(* Goal: {{{ contr_var: x0, x1, x2, x3, x4; }}} *)
let string_of_variables_declaration (parsed_graph : ParsingStructure.parsed_graph) =
	(* Create interval *)
	let interval = list_of_interval 0 (parsed_graph.nb_dimensions - 1) in
	
	(* Map to strings *)
	let variable_names = List.map (fun index -> "x" ^ (string_of_int index)) interval in
	
	(* Separate *)
	let variable_names_with_sep = string_of_list_of_string_with_sep ", " variable_names in
	
	(* Result *)
	"\ncontr_var: " ^ variable_names_with_sep ^ ";"


(************************************************************)
(** Zones*)
(************************************************************)

let string_of_relop = function
	| PARSED_OP_L	-> "<"
	| PARSED_OP_LEQ	-> "<="
	| PARSED_OP_EQ	-> "=="
	| PARSED_OP_GEQ	-> ">="
	| PARSED_OP_G	-> ">"

let string_of_variable is_primed variable_name =
	if is_primed then variable_name ^ "'" else variable_name


let string_of_linear_term is_primed = function
	| Parsed_integer integer -> string_of_int integer
	| Parsed_variable (coef, variable_name) ->
		if coef = 0 then "0"
		else if coef = 1 then (string_of_variable is_primed variable_name)
		else (string_of_int coef) ^ " * " ^ (string_of_variable is_primed variable_name)


let rec string_of_linear_expression is_primed = function
	| Parsed_linear_term parsed_linear_term -> string_of_linear_term is_primed parsed_linear_term
	| Parsed_linear_plus_expression (parsed_linear_expression , parsed_linear_term) ->
		(string_of_linear_expression is_primed parsed_linear_expression)
		^ " + " ^ 
		(string_of_linear_term is_primed parsed_linear_term)
		
	| Parsed_linear_minus_expression (parsed_linear_expression , parsed_linear_term) ->
		(string_of_linear_expression is_primed parsed_linear_expression)
		^ " - " ^ 
		(string_of_linear_term is_primed parsed_linear_term)


let string_of_inequality is_left_primed is_right_primed (left, op, right) =
	(string_of_linear_expression is_left_primed left)
	^ " " ^ (string_of_relop op) ^ " "
	^ (string_of_linear_expression is_right_primed right)

	

let string_of_inequalities is_left_primed is_right_primed inequalities =
	string_of_list_of_string_with_sep " & " (List.map (string_of_inequality is_left_primed is_right_primed) inequalities)

let string_of_zone is_left_primed is_right_primed = function
	| Parsed_inequalities inequalities -> string_of_inequalities is_left_primed is_right_primed inequalities
	| Parsed_false_zone -> string_of_false
	| Parsed_true_zone -> string_of_true


(* Guard: no prime variable *)
let string_of_guard = string_of_zone false false

(* Invariant: no prime variable *)
let string_of_invariant = string_of_zone false false

(* Initially: no prime variable *)
let string_of_init_zone = string_of_zone false false

(* Update: prime to the left *)
(*** HACK: since we don't allow x1' == x2, but we may have 2 <= x1', we just set primes both on the right and on the left ***)
let string_of_update = string_of_zone true true

(* Flow: prime everywhere *)
let string_of_flow = string_of_zone true true


(************************************************************)
(** Transitions *)
(************************************************************)

(* Add identity-like updates (x' == x) for variables not explicitly updated in an update *)
(*** NOTE: an assumption of the input models is that all variables are named xi, with 0 <= i < nb_dimensions ***)
let string_of_complement_update nb_dimensions updated_vars =
	(* Compute all variable names *)
	let all_vars = List.map string_of_var_index (list_of_interval 0 (nb_dimensions - 1)) in
	(* Remove updated variables *)
	let non_updated_vars = list_diff all_vars updated_vars in
	(* Build updates *)
	if non_updated_vars = [] then ""
	else
		let updates = List.map (fun variable_name -> variable_name ^ "' == " ^ variable_name) non_updated_vars in
		" & " ^ (string_of_list_of_string_with_sep " & " updates)


let string_of_transition (parsed_graph : ParsingStructure.parsed_graph) (transition : parsed_transition) =
	(* Guard *)
	"    when " ^ (string_of_guard transition.guard)
	(* Action: none here *)
	^ " sync tau"
	(* Update *)
	(* Do not print update if true *)
	^ (if transition.updated_zone = Parsed_true_zone then "" else
	" do {" ^ (string_of_update transition.updated_zone) ^ "" ^ (string_of_complement_update parsed_graph.nb_dimensions transition.updated_vars) ^ "}")
	^ " goto " ^ transition.target
	^ ";"

(* Transitions for a given location *)
let string_of_transitions (parsed_graph : ParsingStructure.parsed_graph) (location_name : string) =
	(* First select only the transitions starting from this location *)
	let locations_starting_from_this_location = List.filter (fun transition -> (transition.source : string) = location_name) parsed_graph.transitions in
	(* Convert transitions *)
	string_of_list_of_string_with_sep "\n" (List.map (string_of_transition parsed_graph) locations_starting_from_this_location)


(************************************************************)
(** Locations *)
(************************************************************)

let string_of_location (parsed_graph : ParsingStructure.parsed_graph) (location : parsed_location) =
	(* Name *)
	"\nloc " ^ location.name ^ " : "
	(* Invariant *)
	^ "\n  while " ^ (string_of_invariant location.invariant)
	(* Flow *)
	^ " wait {" ^ (string_of_flow location.flow) ^ "};"
	(* Transitions for this location *)
	^ "\n" ^ (string_of_transitions parsed_graph location.name)


let string_of_locations (parsed_graph : ParsingStructure.parsed_graph) =
	let (locations_str : string list) = List.map (string_of_location parsed_graph) parsed_graph.locations in
	string_of_list_of_string_with_sep "\n" locations_str


(************************************************************)
(** Locations *)
(************************************************************)

let string_of_initially (parsed_graph : ParsingStructure.parsed_graph) =
	(* Look for initial location *)
	let initial_locations = List.filter (fun (location : ParsingStructure.parsed_location) -> location.init_zone <> Parsed_false_zone) parsed_graph.locations in
	
	(* Small check *)
	if List.length initial_locations <> 1 then raise (ModelError ("Exactly one initial zone was expected; found: " ^ (string_of_int (List.length initial_locations)) ^ ""));
	
	(* Get initial location *)
	let initial_location = List.hd initial_locations in
	(* Convert *)
	"\n\ninitially: "
		^ (initial_location.name)
		^ " & "
		^ (string_of_init_zone initial_location.init_zone)
		^ ";"


(************************************************************)
(** Automaton *)
(************************************************************)

let string_of_graph (parsed_graph : ParsingStructure.parsed_graph) =
	(model_header parsed_graph)
	^ (string_of_variables_declaration parsed_graph)

	^ (string_of_synclabs)

	(* Enumerate on locations *)
	^ (string_of_locations parsed_graph)
	
	(* Initial location *)
	^ (string_of_initially parsed_graph)
	
	(* The end *)
	^ "\nend"
	
	(* Add-on for us: adding a special comment for the word (to be replaced with the actual automaton by an external script) *)
	^ "\n\n" ^ sPECIAL_COMMENT ^ "\n"
	
	(* Add-on for us: composing the model and the word *)
	^ "\n\nsystem = " ^ parsed_graph.name ^ " & word;\n"


(************************************************************)
(** Property *)
(************************************************************)

let property_header parsed_graph =
	"-- PHAVerLite property generated automatically by " ^ program_name ^ " from '" ^ Sys.argv.(1) ^ "' on " ^ (now())


let string_of_property (parsed_graph : ParsingStructure.parsed_graph) =
	(* Find accepting locations *)
	let accepting_locations = List.filter (fun location -> location.accepting) parsed_graph.locations in
	
	(* Convert to string *)
	(* Goal: {{{ <loc name>~$ & tlast = 0 }}} *)
	let accepting_location_conditions = List.map (fun (location : ParsingStructure.parsed_location) -> location.name ^ "~$ & tlast == 0") accepting_locations in
	
	let accepting_condition = string_of_list_of_string_with_sep "\n\t, " accepting_location_conditions in
	
	(property_header parsed_graph)
	^ 
	"\nforbidden = system.{ " ^ accepting_condition ^ " };

	echo \"\";
	echo \"Computing reachable states:\";

	reach = system.reachable;

	
	/*
	echo \"\";
	echo \"Reachable states:\";
	reach.print;
	echo \"\";
	*/

	res = reach;
	res.intersection_assign(forbidden);

	echo \"\";
	echo \"Reachable forbidden states empty?\";
	res.is_empty;
	echo \"\";

	echo \"\";
	echo \"Reachable forbidden states:\";
	res.print;
	echo \"\";"

