(************************************************************
 *
 *                       Graph2MC
 *
 * Université de Lorraine, CNRS, Inria, LORIA, Nancy, France
 *
 * Module description: Useful OCaml functions
 *
 * File contributors : Étienne André
 * Created           : 2020/01/20
 * Last modified     : 2020/01/23
 *
 ************************************************************)
 

(************************************************************)
(** Modules *)
(************************************************************)
open Unix
open Exceptions


(************************************************************)
(** Useful functions on lists *)
(************************************************************)

(** list_of_interval l u Create a fresh new list filled with elements [l, l+1, ..., u-1, u] *)
let rec list_of_interval l u =
	if ( l > u )
		then []
	else l :: (list_of_interval ( l + 1 ) u)
(*	(* Check if the interval is valid *)
	if a > b then [] else(
		(* Create an array (more efficient?) *)
		let ar = Array.make (b - a + 1) a in
		(* Fill it (no need to update index 0) *)
		for index = 1 to b - a do
			ar.(index) <- index + a
		done;
		(* Return a list *)
		Array.to_list ar
	)*)


(** Returns l1 minus l2, with assumption that all elements of l1 are different *)
let list_diff (l1 : 'a list) (l2 : 'a list) : 'a list =
	(* Optimize a little *)
	if l2 = [] then l1
	else (if l1 = [] then []
	else
		List.filter (fun elt -> not (List.mem elt l2)) l1
	)


(************************************************************)
(** Useful functions on string *)
(************************************************************)
(* Convert an array of string into a string *)
let string_of_array_of_string =
	Array.fold_left (fun the_string s -> the_string ^ s) ""

(* Returns a fresh string made of 'n' times 's' *)
let string_n_times n s =
	string_of_array_of_string (Array.make n s)

(* Convert an array of string into a string with separators *)
let string_of_array_of_string_with_sep sep a =
	let length = Array.length a in
	if length = 0 then "" else(
		let the_string = ref "" in
		for i = 0 to length - 2 do
			the_string := (!the_string) ^ a.(i) ^ sep
		done;
		!the_string ^ a.(length - 1)
	)

(** Convert a list of string into a string with separators (uses an internal conversion to array) *)
let string_of_list_of_string_with_sep sep l =
	string_of_array_of_string_with_sep sep (Array.of_list l)


(************************************************************)
(** Printing time functions *)
(************************************************************)
let days = [| "Sun"; "Mon"; "Tue"; "Wed"; "Thu"; "Fri"; "Sat" |]
let months = [| "Jan"; "Feb"; "Mar"; "Apr"; "May"; "Jun";
				"Jul"; "Aug"; "Sep"; "Oct"; "Nov"; "Dec" |]

(* 'add_digits n i' adds (m-n) '0' in front of 'i', if 'i' is an integer with only 'm' digits; result is always a string *)
let add_digits n i =
	(* Convert to string *)
	let str_i = string_of_int i in
	(* Count the number of digits *)
	let size_i = String.length str_i in
	(
		(* Add more *)
		if size_i <= n then
			(string_n_times (n - size_i) "0")
		(* Otherwise keep unchanged *)
		else ""
	) ^ str_i


(* Adds a zero if a number has only 1 digit *)
let two_digits = add_digits 2
(*	(* Add a 0 if needed *)
	(if i <= 9 then "0" else "")
	^ (string_of_int i)*)

let format_time time =
  let tm = localtime time in
(*  sprintf "%s %s %2d %02d:%02d:%02d %04d"
    days.(tm.tm_wday)
    months.(tm.tm_mon)
    tm.tm_mday
    tm.tm_hour
    tm.tm_min
    tm.tm_sec
    (tm.tm_year + 1900)*)
    (days.(tm.tm_wday))
    ^ " " ^ (months.(tm.tm_mon))
    ^ " " ^ (string_of_int tm.tm_mday)
    ^ ", " ^ (string_of_int (tm.tm_year + 1900))
    ^ " " ^ (two_digits tm.tm_hour)
    ^ ":" ^ (two_digits tm.tm_min)
    ^ ":" ^ (two_digits tm.tm_sec)
 
(*let time = fst (Unix.mktime {tm_sec=50; tm_min=45; tm_hour=3;
		tm_mday=18; tm_mon=0; tm_year=73;
		tm_wday=0; tm_yday=0; tm_isdst=false})*)

(* Print the current date and time under the form of a string *)
let now () = "" ^ (format_time (Unix.gettimeofday ()))
(* printf "format_time gives: %s\n" (format_time time) *)


(**************************************************)
(** System functions *)
(**************************************************)

(** `write_to_file file_name file_content` will create a file `file_name` with content `file_content` *)
let write_to_file file_name file_content =
	(*** TODO: test for file existence! ***)
	let oc = open_out file_name in
	(* Write file *)
	output_string oc file_content;
	(* Close channel *)
	close_out oc;
	()



(************************************************************)
(************************************************************)
(* Constants *)
(************************************************************)
(************************************************************)

let program_name = "Graph2MC"



(************************************************************)
(************************************************************)
(* General useful functions *)
(************************************************************)
(************************************************************)

let print_error msg = print_string ("\n *** ERROR " ^ msg)

let print_message msg = print_string ("\n" ^ msg)

(* Abort program *)
let abort_program () =
	print_error (program_name ^ " aborted");
	(*** NOTE: print new line to stderr ***)
	flush Pervasives.stderr;
	flush Pervasives.stdout;
	prerr_newline();
	exit(1)


(* Terminate program *)
let terminate_program () =
	print_newline();
	print_string (program_name ^ " successfully terminated");

	(* The end *)
	flush Pervasives.stderr;
	flush Pervasives.stdout;
	print_newline();
	exit(0)




(************************************************************)
(************************************************************)
(* Parsing functions *)
(************************************************************)
(************************************************************)

(* Generic parser that returns the abstract structure *)
let parser_lexer_gen the_parser the_lexer lexbuf string_of_input file_name =
	(* Parsing *)
(* 	print_message ("Preparing actual parsing…"); *)
	let parsing_structure = try (
		let absolute_filename = FilePath.make_absolute (FileUtil.pwd ()) file_name in
(* 		print_message ("Created absolute file name '" ^ absolute_filename ^ "'."); *)
		
(* 		print_message ("Assigning lex_curr_p…"); *)
		lexbuf.Lexing.lex_curr_p <- { lexbuf.Lexing.lex_curr_p with Lexing.pos_fname = absolute_filename };
		
(* 		print_message ("Assigning lex_start_p…"); *)
		lexbuf.Lexing.lex_start_p <- { lexbuf.Lexing.lex_start_p with Lexing.pos_fname = absolute_filename };

(* 		print_message ("Starting actual parsing of '" ^ absolute_filename ^ "'…"); *)
		
		let parsing_structure = the_parser the_lexer lexbuf in
(* 		print_message ("Parsing structure created"); *)
		parsing_structure
	) with
		| ParsingError (symbol_start, symbol_end) ->
			print_message ("Parsing error detected. Processing…");
			
			(* Convert the in_channel into a string *)
			let file_string = string_of_input () in
			(* Create the error message *)
			let error_message =
				if symbol_start >= 0 && symbol_end >= symbol_start then (
					(* Get the symbol *)
					let error_symbol = (String.sub file_string symbol_start (symbol_end - symbol_start)) in
					(* Resize it if too big *)
					let error_symbol =
						if (String.length error_symbol > 25) then
							"…" ^ (String.sub error_symbol (String.length error_symbol - 25) 25)
						else error_symbol
					in
					(* Get the line *)
					let beginning_of_the_file = String.sub file_string 0 symbol_end in
					let lines = Str.split (Str.regexp "\n") beginning_of_the_file in
					let line = List.length lines in
					(* Make the message *)
					"near '" ^ error_symbol ^ "' at line " ^ (string_of_int line) ^ ".")
				else "somewhere in the file, most probably in the very beginning."
			in
			(* Print the error message *)
			print_error ("Parsing error in file " ^ file_name ^ " " ^ error_message); abort_program (); exit(1)

		| UnexpectedToken c ->
			print_message ("Parsing error detected 'UnexpectedToken'. Processing…");
			print_error ("Parsing error in file " ^ file_name ^ ": unexpected token '" ^ (Char.escaped c) ^ "'."); abort_program (); exit(1)
		
		| Failure f ->
			print_message ("Parsing error detected 'Failure'. Processing…");
			print_error ("Parsing error ('failure') in file " ^ file_name ^ ": " ^ f); abort_program (); exit(1)
	in
	parsing_structure


(* Parse a file and return the abstract structure *)
let parser_lexer_from_file the_parser the_lexer file_name =
	(* Open file *)
(* 	print_message ("Opening in_channel…"); *)
	let in_channel = try (open_in file_name) with
		| Sys_error e -> print_error ("The file '" ^ file_name ^ "' could not be opened.\n" ^ e); abort_program (); exit(1)
	in
	(* Lexing *)
(* 		print_message ("Lexing…"); *)
	let lexbuf = try (Lexing.from_channel in_channel) with
		| Failure f -> print_error ("Lexing error in file " ^ file_name ^ ": " ^ f); abort_program (); exit(1)
	in
	(* Function to convert a in_channel to a string (in case of parsing error) *)
	let string_of_input () =
		(* Convert the file into a string *)
		let extlib_input = IO.input_channel (open_in file_name) in
			IO.read_all extlib_input
	in
	(* Generic function *)
(* 	print_message ("Calling parser lexer…"); *)
	parser_lexer_gen the_parser the_lexer lexbuf string_of_input file_name




(************************************************************)
(** Parse into a parsing structure *)
(************************************************************)
let parse_graph file_name =

	(* Parsing the main model *)
	print_message ("Parsing graph file " ^ file_name ^ "…");
	let parsed_graph : ParsingStructure.parsed_graph = parser_lexer_from_file GraphParser.main GraphLexer.token file_name in

	print_message ("\nModel parsing completed .");
	

	(*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*)
	(* return *)
	(*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*)
	parsed_graph


