/************************************************************
 *
 *                       Graph2MC
 *
 * Université de Lorraine, CNRS, Inria, LORIA, Nancy, France
 *
 * Module description: Parser for the input graph
 *
 * File contributors : Étienne André
 * Created           : 2020/01/20
 * Last modified     : 2020/01/22
 *
 ************************************************************/


%{
open Exceptions;;
open ParsingStructure;;


let parse_error s =
	let symbol_start = symbol_start () in
	let symbol_end = symbol_end () in
	raise (ParsingError (symbol_start, symbol_end))
;;


%}

%token <int> INT
%token <string> NAME

%token OP_PLUS OP_MINUS OP_MUL OP_DIV
%token OP_L OP_LEQ OP_EQ OP_GEQ OP_G
%token EQUALS ARROW DOUBLEQUOTE

%token LPAREN RPAREN LBRACE RBRACE LSQBRA RSQBRA
%token COLON COMMA SEMICOLON

%token
	CT_DIGRAPH
	CT_DIMENSION
	CT_FALSE CT_FLOW
	CT_GRAPH CT_GUARD
	CT_INIT_ZONE CT_INVARIANT
	CT_MATCH
	CT_TRUE
	CT_UPDATED_VARS CT_UPDATED_ZONE
	
	CT_SEPARATOR_TO_SOLVE_BUG


%token EOF

%left PIPE CT_OR        /* lowest precedence */
%left AMPERSAND CT_AND  /* medium precedence */
%left DOUBLEDOT         /* high precedence */
%nonassoc CT_NOT        /* highest precedence */

%left OP_PLUS OP_MINUS  /* lowest precedence */
%left OP_MUL OP_DIV     /* highest precedence */


%start main             /* the entry point */
%type <ParsingStructure.parsed_graph> main
%%

/************************************************************/
main:
	/* Example: {{{digraph motorcar_template{ … }}}} */
	CT_DIGRAPH NAME LBRACE header locations CT_SEPARATOR_TO_SOLVE_BUG transitions RBRACE EOF
	{
		{
			name			= $2;
			nb_dimensions	= $4;
			locations		= $5;
			transitions		= $7;
		}
	}
;


/************************************************************/

header:
	/* Example: {{{ graph [dimension=5]; }}} */
	| CT_GRAPH LSQBRA CT_DIMENSION EQUALS pos_integer RSQBRA SEMICOLON { $5 }
;

/************************************************************/

locations:
	| location locations { $1 :: $2 }
	| location { [$1] }
;

location:
	/* Example: {{{v1 [match=0, invariant="{x0-x1>=2, x0-x1<=10, x1-x2>=2, x1-x2<=10, x2-x3>=2, x2-x3<=10, x3-x4>=2, x3-x4<=10};", flow="{x0==8, x1 * 10 == 85, x2==9, x3 * 10 == 95, x4==10};", init_zone="{x0 == 40, x1 == 35, x2 == 30, x3 == 25, x4 == 20};"]}}} */
	| NAME LSQBRA match_accepting COMMA invariant_comma_opt flow COMMA init_zone RSQBRA SEMICOLON
	{
		{
			name			= $1;
			accepting		= $3;
			invariant		= $5;
			flow			= $6;
			init_zone		= $8;
		}
	}
	/*
	// TO DEBUG!
	| NAME LSQBRA RSQBRA SEMICOLON
	{
		{
			name			= $1;
			accepting		= false;
			invariant		= Parsed_true_zone;
			flow			= Parsed_true_zone;
			init_zone		= Parsed_true_zone;
		}
	}
	*/
;

/************************************************************/

match_accepting:
	/* Example: {{{ match=1 }}} */
	| CT_MATCH EQUALS INT {
		(* Return true iff match <> 0 *)
		($3 > 0)
	}
;

/************************************************************/

transitions:
	| transition transitions { $1 :: $2 }
	| { [] }
;


transition:
	/* Example: {{{ v1 -> v2 [guard="{x0-x1<=4};", updated_vars="{}", updated_zone="{true};"]; }}} */
	| NAME ARROW NAME LSQBRA guard COMMA updated_vars COMMA updated_zone RSQBRA SEMICOLON
	{
		{
			source			= $1;
			target			= $3;
			guard			= $5;
			updated_vars	= $7;
			updated_zone	= $9;
		}
	}
;

/************************************************************/

flow:
	/* Example: {{{ flow="{x0 == 36, 0 <= x1, x0 - x1 - 2 <= 1, -1 <= x0 - x1 - 2};" }}} */
	| CT_FLOW EQUALS zone { $3 }
;


/************************************************************/

guard:
	/* Example: {{{ guard="{x1-x2<=4};" */
	| CT_GUARD EQUALS zone { $3 }
;


/************************************************************/

invariant_comma_opt:
	| { Parsed_true_zone }
	/* Example: {{{invariant="{x0 - x1 <= 3};"}}} */
	| CT_INVARIANT EQUALS zone COMMA { $3 }
;

/************************************************************/

updated_zone:
	/* Example: {{{ updated_zone="{true};" }}} */
	| CT_UPDATED_ZONE EQUALS zone { $3 }
;

/************************************************************/

init_zone:
	/* Example: {{{ init_zone="{x0 == 65, x1 == 60, x2 == 55, x3 == 50, x4 == 45, x5 == 40, x6 == 35, x7 == 30, x8 == 25, x9 == 20};" }}} */
	| CT_INIT_ZONE EQUALS zone { $3 }
;



/************************************************************/

updated_vars:
	/* Example: {{{ updated_vars="{}" }}} */
	| CT_UPDATED_VARS EQUALS DOUBLEQUOTE LBRACE name_list RBRACE DOUBLEQUOTE { $5 }
;

name_list:
	| name_nonempty_list { $1 }
	| { [] }
;

name_nonempty_list:
	| NAME COMMA name_nonempty_list { $1 :: $3}
	| NAME { [$1] }
;


/************************************************************/
zone:
	/* Example: {{{ "{x0 == 36, 0 <= x1, x0 - x1 - 2 <= 1, -1 <= x0 - x1 - 2};" }}} */
	| DOUBLEQUOTE LBRACE zone_content RBRACE SEMICOLON DOUBLEQUOTE { $3 }
;

zone_content:
	| inequalities { Parsed_inequalities $1 }
	| CT_FALSE { Parsed_false_zone }
	| CT_TRUE { Parsed_true_zone }
;

/************************************************************/

inequalities:
	/* Example: {{{ x0 == 36, 0 <= x1, x0 - x1 - 2 <= 1, -1 <= x0 - x1 - 2 }}} */
	| inequality COMMA inequalities { $1 :: $3 }
	| inequality { [$1] }
;

inequality:
	/* Example: {{{ x0 == 36 }}} */
	/* Example: {{{ -1 <= x0 - x1 - 2 }}} */
	| linear_expression relop linear_expression { $1, $2, $3 }
;

/************************************************************/

relop:
	  OP_L   { PARSED_OP_L }
	| OP_LEQ { PARSED_OP_LEQ }
	| OP_EQ  { PARSED_OP_EQ }
	| OP_GEQ { PARSED_OP_GEQ }
	| OP_G   { PARSED_OP_G }
;

/************************************************************/


/* Linear expression over variables and integers */
linear_expression:
	| linear_term { Parsed_linear_term $1 }
	| linear_expression OP_PLUS linear_term { Parsed_linear_plus_expression ($1, $3) }
	| linear_expression OP_MINUS linear_term { Parsed_linear_minus_expression ($1, $3) }
;

/* Linear term over variables and rationals (no recursion, no division) */
linear_term:
	| integer { Parsed_integer $1 }
	| integer NAME { Parsed_variable ($1, $2) }
	| integer OP_MUL integer { Parsed_integer ($1 * $3) }
	| integer OP_MUL NAME { Parsed_variable ($1, $3) }
	| NAME OP_MUL integer { Parsed_variable ($3, $1) }
	| OP_MINUS NAME { Parsed_variable (-1, $2) }
	| NAME { Parsed_variable (1, $1) }
	| LPAREN linear_term RPAREN { $2 }
;

integer:
	| pos_integer { $1 }
	| OP_MINUS pos_integer { -$2 }
;

pos_integer:
	| INT { $1 }
;

