(*****************************************************************
 *
 *                       Graph2MC
 *
 * Université de Lorraine, LORIA, CNRS, France
 *
 * File contributors : Étienne André
 * Created           : 2020/01/20
 * Last modified     : 2020/01/20
*****************************************************************)

{
open GraphParser
open Lexing

(* OCaml style comments *)
let comment_depth = ref 0;;

let line=ref 1;;

}

rule token = parse
	  ['\n']             { line := !line + 1 ; token lexbuf }     (* skip new lines *)
	| [' ' '\t']         { token lexbuf }     (* skip blanks *)

	(* Graph-style comments *)
	| "/*"
		{ comment_depth := 1;
		graph_style_comment lexbuf;
		token lexbuf }

	(* Very strange parsing problem, where the tool fails in differentiating locations and then transitions => add a separator commented out in graph syntax *)
	| "//TRANSITIONS"  { CT_SEPARATOR_TO_SOLVE_BUG }

	| "digraph"        { CT_DIGRAPH }
	| "dimension"      { CT_DIMENSION }
	| "False"          { CT_FALSE }
	| "false"          { CT_FALSE }
	| "flow"           { CT_FLOW }
	| "graph"          { CT_GRAPH }
	| "guard"          { CT_GUARD }
	| "init_zone"      { CT_INIT_ZONE }
	| "invariant"      { CT_INVARIANT }
	| "match"          { CT_MATCH }
	| "True"           { CT_TRUE }
	| "true"           { CT_TRUE }
	| "updated_vars"   { CT_UPDATED_VARS }
	| "updated_zone"   { CT_UPDATED_ZONE }



	| ['a'-'z''A'-'Z']['a'-'z''A'-'Z''_''0'-'9']* as lxm { NAME lxm }
	| ['0'-'9']+ as lxm { INT(int_of_string lxm) }

	| "<="             { OP_LEQ }
	| ">="             { OP_GEQ }
	| '<'              { OP_L }
	| "=="             { OP_EQ }
	| '>'              { OP_G }

	| '='              { EQUALS }
	| "->"             { ARROW }

	| '+'              { OP_PLUS }
	| '-'              { OP_MINUS }
	| '*'              { OP_MUL }
	| '/'              { OP_DIV }

	| '('              { LPAREN }
	| ')'              { RPAREN }
	| '{'              { LBRACE }
	| '}'              { RBRACE }
	| '['              { LSQBRA }
	| ']'              { RSQBRA }

	| ','              { COMMA }
	| ':'              { COLON }
	| ';'              { SEMICOLON }
	
	| '"'              { DOUBLEQUOTE }

	| eof              { EOF}
	| _ { failwith("Unexpected symbol '" ^ (Lexing.lexeme lexbuf) ^ "' at line " ^ string_of_int !line)}



(* Graph-style comments *)
and graph_style_comment = parse
    "/*"  { incr comment_depth; graph_style_comment lexbuf }
  | "*/"  { decr comment_depth;
            if !comment_depth == 0 then () else graph_style_comment lexbuf }
  | eof
    { failwith "End of file inside a comment in model." }
  | '\n'  { line := !line + 1 ; graph_style_comment lexbuf }
  | _     { graph_style_comment lexbuf }
