############################################################################
##'
##' Script to analyse faunal composition according to the
##' 'lumping' and 'splitting' procedures presented in Ellingsen et al.
##' 'Long-term environmental monitoring for assessment of change:
##' measurement inconsistencies over time and potential solutions.
##' https://link.springer.com/article/10.1007%2Fs10661-017-6317-4
##'
##' Nigel Yoccoz (nigel.yoccoz@uit.no) and Torkild Tveraa (torkild.tveraa@nina.no)
##' Troms, November 2017
##' 
############################################################################


##' First, read taxa names from first line of data frame
r1bio.sn=read.table("R1Bio_22.09.14.txt",sep="\t",nrows=1,colClasses=rep("character",389))
r1bio.sn

##' Second, read data on observations from the various stations (starting on line 7) in
##' datafile
r1bio.dat=read.table("R1Bio_22.09.14.txt",sep="\t",skip=7)

##' ##' Set names to columns in datafile
names(r1bio.dat)=r1bio.sn[1,]
summary(r1bio.dat)
r1bio.dat[,1]   # first column is a descriptor of station/year/replicate
## Year, Station and Replicate are substrings of first column in data file
r1bio.dat$year=as.numeric(substr(r1bio.dat[,1],1,4))
r1bio.dat$stat=as.factor(substr(r1bio.dat[,1],6,12))
r1bio.dat$rep=as.factor(substr(r1bio.dat[,1],16,16))

##' Read datafile and substract dataframe  containing information on Scientific name,
##' Phylumm, Class etc.

r1bio.syst=read.table("R1Bio_22.09.14.txt",sep="\t",nrows=7)
dim(r1bio.syst)

##' Count number of taxa
(n.taxa=dim(r1bio.syst)[2]-1)
r1bio.syst[1:7,1:4]  # list taxaonimic levels for the first 3 taxa

##' Identify to what taxonomic level identification was done for each reported
##' scientific name and print and summarize results; inf means species level, 7 = genus
systlev=c()
for (i in 1:n.taxa) systlev[i]=min(which(as.vector(r1bio.syst[,i+1])==""))
systlev
table(systlev)  # tabulate number of taxa at different levels

##' Print those identified to the Genus level and count how many there are
which(systlev==7)
(n.gen=length(which(systlev==7)))

##'Have a look at the first and last of the 67 species identified to the Genus level
##' only and count how many observations there are in the different years

(genus=as.character(r1bio.syst[6,which(systlev==7)[1]+1]))
for (i in which(r1bio.syst[6,]==genus)) {
 print(as.character(r1bio.sn[i]))
# print(table(r1bio.dat[,i],r1bio.dat[,c("year")]))
 print(tapply(r1bio.dat[,i],INDEX=r1bio.dat[["year"]],FUN=sum))
}

n.gen
i.gen=67 # should vary from 1 to n.gen
(genus=as.character(r1bio.syst[6,which(systlev==7)[i.gen]+1]))
for (i in which(r1bio.syst[6,]==genus)) {
  print(paste(i,as.character(r1bio.sn[i])))
  print(tapply(r1bio.dat[,i],INDEX=r1bio.dat[["year"]],FUN=sum))
}

##' create two new data sets to include changes made wrt taxonomic levels

r1bio.new1=r1bio.new2=r1bio.dat 
##' for taxa which are ambiguous:
##' 1 is the data set with maximum reduction,
##' 2 is the one with keeping species
##' three possibilities:
##' only genus, no change needed
##' lump species with genus
##' or remove genus and keep all species

##' only genus, no change needed: 8,12,13,14,20,24,29,32,35,37,39,40,41,42,45,49,51,52,53,54,55,63,66,67

##'  ----------- LUMP SPECIES WITH GENUS ---------------
##' 
##' Note that systlev = 7 denotes species identified to the genus level

for (i.gen in c(1,2,4,9,11,15,17,18,19,22,27,30,33,34,36,38,44,47,48,50,57,59,62,64,65)) {
 genus=as.character(r1bio.syst[6,which(systlev==7)[i.gen]+1]) 
 ind.gen=which(r1bio.syst[6,]==genus) # Note that row 6 in r1bio.syst contains Genus
 r1bio.new1[,ind.gen[1]]=r1bio.new2[,ind.gen[1]]=apply(r1bio.dat[,ind.gen],MAR=1,sum) # sum genus and species
 r1bio.new1[,ind.gen[-1]]=r1bio.new2[,ind.gen[-1]]=0 # put 0 for species
}


 apply(r1bio.dat[,ind.gen],MAR=2,sum)    # checking that numbers are right
 apply(r1bio.new1[,ind.gen],MAR=2,sum)    # for the new file should be the sum

##' ------------ REMOVE GENUS, KEEP SPECIES -------------------

for (i.gen in c(3,6,16,21,25,28,46,58,60)){
 genus=as.character(r1bio.syst[6,which(systlev==7)[i.gen]+1])
 print(c('genus: ', genus))
 ind.gen=which(r1bio.syst[6,]==genus)  # Note that row 6 in r1bio.syst contains Genus
 print(c('ind.gen: ', ind.gen))
 r1bio.new1[,ind.gen[1]]=r1bio.new2[,ind.gen[1]]=0 # put 0 for Genus
}

 apply(r1bio.dat[,ind.gen],MAR=2,sum)    # checking that numbers are right
 apply(r1bio.new1[,ind.gen],MAR=2,sum)    # for the new file should be the sum

# Taxa with two alternatives
for (i.gen in c(5,7,10,23,26,31,43,56,61)){
 genus=as.character(r1bio.syst[6,which(systlev==7)[i.gen]+1])
 print(c('genus: ', toString(genus)))
 ind.gen=which(r1bio.syst[6,]==genus)
 print(c('ind.gen: ', toString(ind.gen)))
 r1bio.new1[,ind.gen[1]]=apply(r1bio.dat[,ind.gen],MAR=1,sum)     # sum genus and species in new1
 r1bio.new1[,ind.gen[-1]]=0                                         # put 0 for species
 r1bio.new2[,ind.gen[1]]=0                                      # put 0 for Genus in new2
}

  apply(r1bio.dat[,ind.gen],MAR=2,sum)    # checking that numbers are right
  apply(r1bio.new1[,ind.gen],MAR=2,sum)    # checking that numbers are right
  apply(r1bio.new2[,ind.gen],MAR=2,sum)    # checking that numbers are right
  
sort(c(c(8,12,13,14,20,24,29,32,35,37,39,40,41,42,45,49,51,52,53,54,55,63,66,67),
  c(1,2,4,9,11,15,17,18,19,22,27,30,33,34,36,38,44,47,48,50,57,59,62,64,65),
  c(3,6,16,21,25,28,46,58,60),c(5,7,10,23,26,31,43,56,61))) - (1:67)# checking all species are present

# Eteone flava 1996 vs longa 1999 ff
# shift between genus (1996) and species for Sphaerodorum gracilis (1999 ff)
# Goniada mostly genus in 2005, Goniada maculata other years
# Jasmineira shift between genus and species
# Magelona shift between genus in 1996,99,02 and 2 species later
# Caulleriella Genus only in 2002
# Chaetozone: shift from species to genus in 2008
# Bathyporeia species only in 2002, genus in other years
# Cheirocratus: shift from species to genus in 2008
# Diastylis: species in 1996, mostly genus later on
# Thracia: genus only in 1996, species later
# Philine, species only in 1999 to 2005, genus al years
# Antalis, genus in 1996, 2008 and 2011, species in 1999, 2002, 2005
# Phoronis, genus in 1996, 2008 and 2011, species in 1999, 2002, 2005

# Family
which(systlev==6)
(n.fam=length(which(systlev==6)))

i.fam=15
famil=as.character(r1bio.syst[5,which(systlev==6)[i.fam]+1])
 for (i in which(r1bio.syst[5,]==famil)) {
   print(as.character(r1bio.sn[i]))
   # print(table(r1bio.dat[,i],r1bio.dat[,c("year")]))
   print(tapply(r1bio.dat[,i],INDEX=r1bio.dat[,c("year")],FUN=sum))
}

# Syllidae 1 Ampharetidae 2 Cirratulidae 3 Pectinariidae 4 Amphilochidae 5
# Ischyroceridae 8 Lysianassidae 9 Oedicerotidae 10 removed
# Aoridae 6 Caprellidae 7 Pleustidae 11 Stenothoidae 13 Paguridae 15 lump all together
# Podoceridae 12 Majidae 14 no other group, no change
for (i.fam in c(6,7,11,13,15)) { # c(6,7,11,13,15) lump family and other taxa   
 famil=as.character(r1bio.syst[5,which(systlev==6)[i.fam]+1])
  for (i in which(r1bio.syst[5,]==famil)) {
   print(as.character(r1bio.sn[i]))
   print(tapply(r1bio.dat[,i],INDEX=r1bio.dat[,c("year")],FUN=sum))
  }
 (ind.fam=which(r1bio.syst[5,]==famil))
 (ind.fam.c=which(systlev[ind.fam-1]==6))
 r1bio.new1[,ind.fam[ind.fam.c]]=r1bio.new2[,ind.fam[ind.fam.c]]=apply(r1bio.dat[,ind.fam],MAR=1,sum)     # sum family and taxa
 r1bio.new1[,ind.fam[-ind.fam.c]]=r1bio.new2[,ind.fam[-ind.fam.c]]=0                                         # put 0 for other taxa
}

apply(r1bio.dat[,ind.fam],MAR=2,sum)    # checking that numbers are right
apply(r1bio.new1[,ind.fam],MAR=2,sum)    # for the new file should be the sum

# remove family, keep other taxa
for (i.fam in c(1,2,3,4,5,8,9,10)) {
 famil=as.character(r1bio.syst[5,which(systlev==6)[i.fam]+1])
  for (i in which(r1bio.syst[5,]==famil)) {
   print(as.character(r1bio.sn[i]))
   print(tapply(r1bio.dat[,i],INDEX=r1bio.dat[,c("year")],FUN=sum))
  }
 (ind.fam=which(r1bio.syst[5,]==famil))
 (ind.fam.c=which(systlev[ind.fam-1]==6))
 r1bio.new1[,ind.fam[ind.fam.c]]=r1bio.new2[,ind.fam[ind.fam.c]]=0     # remove family
} 
apply(r1bio.dat[,ind.fam],MAR=2,sum)    # checking that numbers are right
apply(r1bio.new1[,ind.fam],MAR=2,sum)    # for the new file should be 0 for the family removed

# Order
which(systlev==5)
n.ord=length(which(systlev==5))

i.ord=4
(orde=as.character(r1bio.syst[4,which(systlev==5)[i.ord]+1]))
# r1bio.syst[1:5,which(r1bio.syst[4,]==orde)]
for (i in which(r1bio.syst[4,]==orde)) {
  print(as.character(r1bio.sn[i]))
  # print(table(r1bio.dat[,i],r1bio.dat[,c("year")]))
  print(tapply(r1bio.dat[,i],INDEX=r1bio.dat[,c("year")],FUN=sum))
}

# remove Amphipoda 1 Actiniaria 2
# Echiurida 3 only one taxon no change
# Lump together Nudibranchia 4 and Okenia

for (i.ord in c(1,2)) { # c(1,2)
 (orde=as.character(r1bio.syst[4,which(systlev==5)[i.ord]+1]))
  for (i in which(r1bio.syst[4,]==orde)) {
   print(as.character(r1bio.sn[i]))
   print(tapply(r1bio.dat[,i],INDEX=r1bio.dat[,c("year")],FUN=sum))
  }
 (ind.ord=which(r1bio.syst[4,]==orde))
 (ind.ord.c=which(systlev[ind.ord-1]==5))
 r1bio.new1[,ind.ord[ind.ord.c]]=r1bio.new2[,ind.ord[ind.ord.c]]=0     # remove order
}

apply(r1bio.dat[,ind.ord],MAR=2,sum)    # checking that numbers are right
apply(r1bio.new1[,ind.ord],MAR=2,sum)    # for the new file should be 0 for the family removes
apply(r1bio.new2[,ind.ord],MAR=2,sum)    # for the new file should be 0 for the family removes

i.ord=4
(orde=as.character(r1bio.syst[4,which(systlev==5)[i.ord]+1]))
for (i in which(r1bio.syst[4,]==orde)) {
  print(as.character(r1bio.sn[i]))
  print(tapply(r1bio.dat[,i],INDEX=r1bio.dat[,c("year")],FUN=sum))
}
(ind.ord=which(r1bio.syst[4,]==orde))
(ind.ord.c=which(systlev[ind.ord-1]==5))
r1bio.new1[,ind.ord[ind.ord.c]]=r1bio.new2[,ind.ord[ind.ord.c]]=apply(r1bio.dat[,ind.ord],MAR=1,sum)     # sum order and taxa
r1bio.new1[,ind.ord[-ind.ord.c]]=r1bio.new2[,ind.ord[-ind.ord.c]]=0                                         # put 0 for other taxa

apply(r1bio.dat[,ind.ord],MAR=2,sum)    # checking that numbers are right
apply(r1bio.new1[,ind.ord],MAR=2,sum)    # for the new file should be the sum

# Class
which(systlev==4)
n.cla=length(which(systlev==4))

i.cla=7
(clas=as.character(r1bio.syst[3,which(systlev==4)[i.cla]+1]))
for (i in which(r1bio.syst[3,]==clas)) {
  print(as.character(r1bio.sn[i]))
#  print(tapply(r1bio.dat[,i],INDEX=r1bio.dat[,c("year")],FUN=sum))
}
# Clitellata 1 Pycnogonida 2 Caudofoveata 6 only one group
# Priapulida 3 lump class and species
# Ascidia, only 1 in 2008 remove
# Anthozoa: 2011 only, remove

i.cla=3
(clas=as.character(r1bio.syst[3,which(systlev==4)[i.cla]+1]))
for (i in which(r1bio.syst[3,]==clas)) {
  print(as.character(r1bio.sn[i]))
  print(tapply(r1bio.dat[,i],INDEX=r1bio.dat[,c("year")],FUN=sum))
}
(ind.cla=which(r1bio.syst[3,]==clas))
(ind.cla.c=which(systlev[ind.cla-1]==4))
r1bio.new1[,ind.cla[ind.cla.c]]=r1bio.new2[,ind.cla[ind.cla.c]]=apply(r1bio.dat[,ind.cla],MAR=1,sum)     # sum class and taxa
r1bio.new1[,ind.cla[-ind.cla.c]]=r1bio.new2[,ind.cla[-ind.cla.c]]=0                                         # put 0 for other taxa

apply(r1bio.dat[,ind.ord],MAR=2,sum)    # checking that numbers are right
apply(r1bio.new1[,ind.ord],MAR=2,sum)    # for the new file should be the sum

for (i.cla in c(4,5)) { # 4,5
 (clas=as.character(r1bio.syst[3,which(systlev==4)[i.cla]+1]))
 for (i in which(r1bio.syst[3,]==clas)) {
  print(as.character(r1bio.sn[i]))
  print(tapply(r1bio.dat[,i],INDEX=r1bio.dat[,c("year")],FUN=sum))
 }
 (ind.cla=which(r1bio.syst[3,]==clas))
 (ind.cla.c=which(systlev[ind.cla-1]==4))
 r1bio.new1[,ind.cla[ind.cla.c]]=r1bio.new2[,ind.cla[ind.cla.c]]=0    # put 0 for other taxa
}

apply(r1bio.dat[,ind.cla],MAR=2,sum)    # checking that numbers are right
apply(r1bio.new1[,ind.cla],MAR=2,sum)    # for the new file should be the sum

# Phylum
which(systlev==3)
n.ord=length(which(systlev==3))

i.phy=1
(phyl=as.character(r1bio.syst[2,which(systlev==3)[i.phy]+1]))
for (i in which(r1bio.syst[2,]==phyl)) {
  print(as.character(r1bio.sn[i]))
  print(tapply(r1bio.dat[,i],INDEX=r1bio.dat[,c("year")],FUN=sum))
}

# Sipuncula 6, only 2011 remove
# Tunicata 2 only in 1999 to 2005, other groups in 1996, 2008 and 2011, lump
# Chaetognatha 1 Hemichordata 3 Nemertea 4 Platyhelminthes 5 only one group
i.phy=6
(phyl=as.character(r1bio.syst[2,which(systlev==3)[i.phy]+1]))
for (i in which(r1bio.syst[2,]==phyl)) {
  print(as.character(r1bio.sn[i]))
  print(tapply(r1bio.dat[,i],INDEX=r1bio.dat[,c("year")],FUN=sum))
}
(ind.phy=which(r1bio.syst[2,]==phyl))
(ind.phy.c=which(systlev[ind.phy-1]==3))
r1bio.new1[,ind.phy[ind.phy.c]]=r1bio.new2[,ind.phy[ind.phy.c]]=0      # put 0 for other taxa

apply(r1bio.dat[,ind.phy],MAR=2,sum)    # checking that numbers are right
apply(r1bio.new1[,ind.phy],MAR=2,sum)    # for the new file should be the sum

i.phy=2
(phyl=as.character(r1bio.syst[2,which(systlev==3)[i.phy]+1]))
for (i in which(r1bio.syst[2,]==phyl)) {
  print(as.character(r1bio.sn[i]))
  print(tapply(r1bio.dat[,i],INDEX=r1bio.dat[,c("year")],FUN=sum))
}
(ind.phy=which(r1bio.syst[2,]==phyl))
(ind.phy.c=which(systlev[ind.phy-1]==3))
r1bio.new1[,ind.phy[ind.phy.c]]=r1bio.new2[,ind.phy[ind.phy.c]]=apply(r1bio.dat[,ind.phy],MAR=1,sum)     # sum phylum and taxa
r1bio.new1[,ind.phy[-ind.phy.c]]=r1bio.new2[,ind.phy[-ind.phy.c]]=0                                         # put 0 for other taxa
apply(r1bio.dat[,ind.phy],MAR=2,sum)    # checking that numbers are right
apply(r1bio.new1[,ind.phy],MAR=2,sum)    # for the new file should be the sum

##' removing null columns in r1bio.new1 and new2 after adding column names
dim(r1bio.new1)
r1bio.new1[1:3,1:10]
names(r1bio.dat)
names(r1bio.new1)
names(r1bio.new2)
names(r1bio.sn)


 names(r1bio.dat)=names(r1bio.new1)=names(r1bio.new2)=c(r1bio.sn,"year","stat","rep")
(n.tax=length(names(r1bio.new1))-4)
col.dat.0=apply(r1bio.dat[,-c(1,n.tax+2,n.tax+3,n.tax+4)],MAR=2,sum)
col.new1.0=apply(r1bio.new1[,-c(1,n.tax+2,n.tax+3,n.tax+4)],MAR=2,sum)
col.new2.0=apply(r1bio.new2[,-c(1,n.tax+2,n.tax+3,n.tax+4)],MAR=2,sum)
cbind(col.dat.0,col.new1.0,col.new2.0)  # checking that calculations are right
as.numeric(which(col.new1.0!=0))

#'# removing null columns in new1 and new2
c(1,as.vector(which(col.new1.0!=0)),n.tax+2,n.tax+3,n.tax+4)
r1bio.dat.n=r1bio.new1[,c(1,(as.numeric(which(col.dat.0!=0))+1),n.tax+2,n.tax+3,n.tax+4)]
r1bio.new1.n=r1bio.new1[,c(1,(as.numeric(which(col.new1.0!=0))+1),n.tax+2,n.tax+3,n.tax+4)]
r1bio.new2.n=r1bio.new2[,c(1,(as.numeric(which(col.new2.0!=0))+1),n.tax+2,n.tax+3,n.tax+4)]


##' Check number of 'species' in each dataset
a <- data.frame(cbind(org=col.dat.0,dat1=col.new1.0,dat2=col.new2.0))

dim(a[a$org>0,])
dim(a[a$dat1>0,])
dim(a[a$dat2>0,])

head(a)

##' Write alternative datasets to file
write.table(r1bio.dat.n, file="dataOrg.csv", row.names=FALSE, col.names=TRUE, sep=';')
write.table(r1bio.new1.n, file="data1.csv", row.names=FALSE, col.names=TRUE, sep=';')
write.table(r1bio.new2.n, file="data2.csv", row.names=FALSE, col.names=TRUE, sep=';')


######################## -------------------------------------- ###############################
##' Multivariate analyses
##' Run analyses and check plots according to those presented in paper
######################## -------------------------------------- ###############################
require(ade4)
(n.tax1=dim(r1bio.new1.n)[2]-4)
(n.tax2=dim(r1bio.new2.n)[2]-4)

# correspondence analysis
ca.dat=dudi.coa(r1bio.dat[,-c(1,n.tax+2,n.tax+3,n.tax+4)],scan=F)
s.class(ca.dat$li,fac=as.factor(r1bio.dat$year))
library(R.utils)

rownames(ca.dat$co) <- gsub("\\.+", " ", rownames(ca.dat$co), perl=TRUE)
#rownames(ca.dat$co) <- toCamelCase(rownames(ca.dat$co), capitalize=TRUE)
s.label(ca.dat$co,boxes=F)

ca.new1=dudi.coa(r1bio.new1.n[,-c(1,n.tax1+2,n.tax1+3,n.tax1+4)],scan=F)
s.class(ca.new1$li,fac=as.factor(r1bio.new1.n$year))
s.label(ca.new1$co,boxes=F)

# non-symmetric
nsc.dat=dudi.nsc(r1bio.dat[,-c(1,n.tax+2,n.tax+3,n.tax+4)],scan=F)
s.class(nsc.dat$li,fac=as.factor(r1bio.dat$year))
s.label(ca.dat$co,boxes=F)

nsc.new1=dudi.nsc(r1bio.new1.n[,-c(1,n.tax1+2,n.tax1+3,n.tax1+4)],scan=F)
s.class(nsc.new1$li,fac=as.factor(r1bio.dat$year))
s.label(ca.dat$co,boxes=F)

# NMDS
require(vegan)
dat.ord=metaMDS(r1bio.dat[,-c(1,n.tax+2,n.tax+3,n.tax+4)],trymax=100)
plot(dat.ord)
s.class(dat.ord$points,fac=as.factor(r1bio.dat$year))

dat.new1=metaMDS(r1bio.new1.n[,-c(1,n.tax1+2,n.tax1+3,n.tax1+4)],trymax=700)
plot(dat.new1)
s.class(dat.new1$points,fac=as.factor(r1bio.new1.n$year))

dat.new2=metaMDS(r1bio.new2.n[,-c(1,n.tax2+2,n.tax2+3,n.tax2+4)],trymax=500)
plot(dat.new2)
s.class(dat.new2$points,fac=as.factor(r1bio.new2.n$year))

dat.ord.go=metaMDS(r1bio.dat[,-c(1,n.tax+2,n.tax+3,n.tax+4)],
                   distance="gower",trymax=500)
plot(dat.ord.go)
s.class(dat.ord.go$points,fac=as.factor(r1bio.dat$year))

dat.new1.go=metaMDS(r1bio.new1.n[,-c(1,n.tax1+2,n.tax1+3,n.tax1+4)],
    distance="gower",trymax=500)
plot(dat.new1.go)
s.class(dat.new1.go$points,fac=as.factor(r1bio.new1.n$year))

dat.new2.go=metaMDS(r1bio.new2.n[,-c(1,n.tax2+2,n.tax2+3,n.tax2+4)],
    distance="gower",trymax=500)
plot(dat.new2.go)
s.class(dat.new2.go$points,fac=as.factor(r1bio.new2.n$year))


################################################################
################################################################

# Figure 7 in paper
tiff(filename = "Figure7 NMDS.tif", compression=c("lzw"), units="cm",width=14, height=8, res=600)
par(mfrow=c(1,2))
s.class(dat.ord$points,fac=as.factor(r1bio.dat$year),grid=F)
s.class(dat.new1$points,fac=as.factor(r1bio.new1.n$year),grid=F)
dev.off()

# Figure S1 in Online resource 2
tiff(filename = "FigS1 NDMS.tif", compression=c("lzw"), units="cm",width=8, height=8, res=600)
par(mfrow=c(1,1))
s.class(dat.new2$points,fac=as.factor(r1bio.new2.n$year),grid=F)#, sub="Splitting", csub=1.5, possub="topleft")
dev.off()


################################################################
################################################################

##' CCA with years as predictor; check that vegan is not loaded
detach("package:vegan", unload=TRUE)
(.packages()) # check that vegan is no longer attached (use ade4 for cca)
require(ade4) 

iv1.dat=ade4::cca(r1bio.dat[,-c(1,n.tax+2,n.tax+3,n.tax+4)], 
    as.factor(r1bio.dat$year),scan=F, nf=3)
plot(iv1.dat)
s.class(iv1.dat$ls,fac=as.factor(r1bio.dat$year))

iv1.n1=ade4::cca(r1bio.new1.n[,-c(1,n.tax1+2,n.tax1+3,n.tax1+4)],
            as.factor(r1bio.new1.n$year),scan=F,nf=3)
plot(iv1.n1)

iv1.n2=ade4::cca(r1bio.new2.n[,-c(1,n.tax2+2,n.tax2+3,n.tax2+4)],
            as.factor(r1bio.new2.n$year),scan=F,nf=3)
plot(iv1.n1)


################################################################
################################################################


##' Figure 8 CCA faunal data, axes 1, 2 and 3, based on raw (r1bio.dat) and
##' adjusted (r1bio.new1) data as presented in the paper


tiff(filename = "Figure8.tif", compression=c("lzw"), units="cm",width=14, height=14, res=600)
par(mfrow=c(2,2))
s.class(iv1.dat$ls[,1:2],fac=as.factor(r1bio.dat$year),grid=F)
s.class(cbind(iv1.n1$ls[,1],-iv1.n1$ls[,2]),fac=as.factor(r1bio.new1.n$year),grid=F)

s.class(iv1.dat$ls[,c(1,3)],fac=as.factor(r1bio.dat$year),grid=F)
s.class(cbind(iv1.n1$ls[,1],-iv1.n1$ls[,3]),fac=as.factor(r1bio.new1.n$year),grid=F)
dev.off()

##' Figure S2 in online resource 2
tiff(filename = "FigS2 CCA.tif", compression=c("lzw"), units="cm",width=14, height=8, res=600)
par(mfrow=c(1,2))
s.class(cbind(iv1.n2$ls[,1],-iv1.n2$ls[,2]),fac=as.factor(r1bio.new2.n$year),grid=F)
s.class(cbind(iv1.n2$ls[,1],-iv1.n2$ls[,3]),fac=as.factor(r1bio.new2.n$year),grid=F)

dev.off()

##' End of script
##################################################################
##################################################################
