# R script for ANCOVA analysis of ovary volume and rate of change in ovary volume

# Data for analyses are: 
# wildtype_volume for wildtype ovaries from larvae fed on different dietary treatments
# genetic_fed_volume for genetically manipulated ovaries from larvae fed on standard food 
# genetic_starved_volume for genetically manipulated ovaries from larvae fed on 20% sucrose food from 5h AL3E

load("wildtype_volume.RData")
load("genetic_fed_volume.RData")
load("genetic_starved_volume.RData")

#Packages required:
require("lsmeans")

######  WILD-TYPE OVARIES FROM LARVAE FED ON DIFFERENT DIETARY TREATMENTS    ######

#Log(10) transformation of ovary volume.
wildtype.volume <- within(Volume_Diet, {log10.volume=log10(Volume)
	Genotype=factor(Treatment)})

#Compare ovary volume and ratio of change in ovary volume between three groups:
#larvae fed on standard food (FED)
#larvae fed on 20% sucrose food from 5h AL3E (PRE)
#larvae fed on 20% sucrose food from 15h AL3E (POST)

#check for the assumptions of (G)LM models
mod_ancova1 <- lm(Volume ~ Timepoint*Treatment,data= wildtype.volume)
mod_ancova2 <- lm(log10.volume ~ Timepoint*Treatment,data= wildtype.volume)

par(mfrow=c(2,4))

plot(mod_ancova1)
plot(mod_ancova2)

#Perform ANCOVA
mod_ancova <- lm(log10.volume~ Timepoint*Treatment,data= wildtype.volume)
summary(mod_ancova)

#Calculate least square trends to determine differences in the rate of change in ovary volume
mod_trends<- lstrends(mod_ancova, ~Treatment, var="Timepoint")
pairs(mod_trends,adjust="holm")
cld(mod_trends)

#Calculate least square means to determine differences in ovary volume
mod_means<- lsmeans(mod_ancova, ~Treatment, var="Timepoint")
pairs(mod_means,adjust="holm")
cld(mod_means)

######  GENETIC MANIPULATIONS IN STANDARD FOOD    ######

#Log(10) transformation of ovary volume.
genetic.fed.volume <- within(Volume_Genetic_Fed, {log10.volume=log10(Volume)
	Treatment=as.factor("Fed")}) 

#Compare ovary volume and ratio of change in ovary volume within three groups:
#(A) ecdysone signalling manipulations
#(B) IIS manipulations
#(C) IIS and ecdysone signalling manipulations

#(A) ECDYSONE SIGNALLING MANIPULATIONS

#Subsetting the data
tj <- subset (genetic.fed.volume, Genotype=="tj")
tj_EcRdn <-subset(genetic.fed.volume, Genotype=="tj_EcRdn")
tj_EcRi <-subset(genetic.fed.volume, Genotype=="tj_EcRi")

#Join data.frames per rows (vertically)
ecdysone <- rbind(tj,tj_EcRdn,tj_EcRi)

#check for the assumptions of (G)LM models
mod_ancova1 <- lm(Volume ~ Timepoint*Genotype,data= ecdysone)
mod_ancova2 <- lm(log10.volume ~ Timepoint*Genotype,data= ecdysone)

par(mfrow=c(2,4))

plot(mod_ancova1)
plot(mod_ancova2)

#Perform ANCOVA
mod_ancova <- lm(log10.volume~ Timepoint*Genotype,data= ecdysone)
summary(mod_ancova)

#Calculate least square trends to determine differences in the rate of change in ovary volume
mod_trends<- lstrends(mod_ancova, ~Genotype, var="Timepoint")
pairs(mod_trends,adjust="holm")
cld(mod_trends)  

#Calculate least square means to determine differences in ovary volume
mod_means<- lsmeans(mod_ancova, ~Genotype, var="Timepoint")
pairs(mod_means,adjust="holm")
cld(mod_means)

#(C) IIS MANIPULATIONS

#Subsetting the data
tj <- subset (genetic.fed.volume, Genotype=="tj")
tj_InR <-subset(genetic.fed.volume, Genotype=="tj_InR")
tj_PTEN <-subset(genetic.fed.volume, Genotype=="tj_PTEN")

#Join data.frames per rows (vertically)
insulin <- rbind(tj,tj_InR,tj_PTEN)

#check for the assumptions of (G)LM models
mod_ancova1 <- lm(Volume ~ Timepoint*Genotype,data= insulin)
mod_ancova2 <- lm(log10.volume ~ Timepoint*Genotype,data= insulin)

par(mfrow=c(2,4))

plot(mod_ancova1)
plot(mod_ancova2)

#Perform ANCOVA
mod_ancova <- lm(log10.volume~ Timepoint*Genotype,data= insulin)
summary(mod_ancova)

#Calculate least square trends to determine differences in the rate of change in ovary volume
mod_trends<- lstrends(mod_ancova, ~Genotype, var="Timepoint")
pairs(mod_trends,adjust="holm")
cld(mod_trends) 

#Calculate least square means to determine differences in ovary volume
mod_means<- lsmeans(mod_ancova, ~Genotype, var="Timepoint")
pairs(mod_means,adjust="holm")
cld(mod_means)

#(C) IIS AND ECDYSONE SIGNALLING MANIPULATIONS

#Subsetting the data
tj <- subset (genetic.fed.volume, Genotype=="tj")
tj_EcRi_PTEN <-subset(genetic.fed.volume, Genotype=="tj_EcRi_PTEN")
tj_EcRi_InR <-subset(genetic.fed.volume, Genotype=="tj_EcRi_InR")
tj_EcRdn_PTEN <-subset(genetic.fed.volume, Genotype=="tj_EcRdn_PTEN")
tj_EcRdn_InR <-subset(genetic.fed.volume, Genotype=="tj_EcRdn_InR")

#Join data.frames per rows (vertically)
e_i <- rbind(tj,tj_EcRi_InR,tj_EcRi_PTEN,tj_EcRdn_InR,tj_EcRdn_PTEN)

#check for the assumptions of (G)LM models
mod_ancova1 <- lm(Volume ~ Timepoint*Genotype,data= e_i)
mod_ancova2 <- lm(log10.volume ~ Timepoint*Genotype,data= e_i)

par(mfrow=c(2,4))

plot(mod_ancova1)
plot(mod_ancova2)

#Perform ANCOVA
mod_ancova <- lm(log10.volume~ Timepoint*Genotype,data= e_i)
summary(mod_ancova)

#Calculate least square trends to determine differences in the rate of change in ovary volume
mod_trends<- lstrends(mod_ancova, ~Genotype, var="Timepoint")
pairs(mod_trends,adjust="holm")
cld(mod_trends)  

#Calculate least square means to determine differences in ovary volume
mod_means<- lsmeans(mod_ancova, ~Genotype, var="Timepoint")
pairs(mod_means,adjust="holm")
cld(mod_means)

######  GENETIC MANIPULATIONS IN 20% SUCROSE FOOD   ######

#Log(10) transformation of ovary volume.
genetic.starved.volume <- within(Volume_Genetic_Starved, {log10.volume=log10(Volume)
	Treatment=as.factor("Starved")}) 

#Compare ovary volume and ratio of change in ovary volume within three groups:
#(A) ecdysone signalling manipulations
#(B) IIS manipulations
#(C) IIS and ecdysone signalling manipulations

#Perform ANCOVA
mod_ancova <- lm(log10.volume~ Timepoint* Genotype, data= genetic.starved.volume)
summary(mod_ancova)

#Calculate least square trends to determine differences in the rate of change in ovary volume
mod_trends<- lstrends(mod_ancova, ~ Genotype, var="Timepoint")
pairs(mod_trends,adjust="holm")
cld(mod_trends)

#Calculate least square means to determine differences in ovary volume
mod_means<- lsmeans(mod_ancova, ~ Genotype, var="Timepoint")
pairs(mod_means,adjust="holm")
cld(mod_means)

#(A) ECDYSONE SIGNALLING MANIPULATIONS

#Subsetting the data
tj <- subset (genetic.starved.volume, Genotype=="tj")
tj_EcRi <-subset(genetic.starved.volume, Genotype=="tj_EcRi")

#Join data.frames per rows (vertically)
ecdysone_S <- rbind(tj,tj_EcRi)

#check for the assumptions of (G)LM models
mod_ancova1 <- lm(Volume ~ Timepoint*Genotype,data= ecdysone_S)
mod_ancova2 <- lm(log10.volume ~ Timepoint*Genotype,data= ecdysone_S)

par(mfrow=c(2,4))

plot(mod_ancova1)
plot(mod_ancova2)

#Perform ANCOVA
mod_ancova <- lm(log10.volume~ Timepoint*Genotype,data= ecdysone_S)
summary(mod_ancova)

#Calculate least square trends to determine differences in the rate of change in ovary volume
mod_trends<- lstrends(mod_ancova, ~Genotype, var="Timepoint")
pairs(mod_trends,adjust="holm")
cld(mod_trends)  

#Calculate least square means to determine differences in ovary volume
mod_means<- lsmeans(mod_ancova, ~Genotype, var="Timepoint")
pairs(mod_means,adjust="holm")
cld(mod_means)

#(B) IIS MANIPULATIONS

#Subsetting the data
tj <- subset (genetic.starved.volume, Genotype=="tj")
tj_InR <-subset(genetic.starved.volume, Genotype=="tj_InR")

#Join data.frames per rows (vertically)
insulin_S <- rbind(tj,tj_InR)

#check for the assumptions of (G)LM models
mod_ancova1 <- lm(Volume ~ Timepoint*Genotype,data= insulin_S)
mod_ancova2 <- lm(log10.volume ~ Timepoint*Genotype,data= insulin_S)

par(mfrow=c(2,4))

plot(mod_ancova1)
plot(mod_ancova2)

#Perform ANCOVA
mod_ancova <- lm(log10.volume~ Timepoint*Genotype,data= insulin_S)
summary(mod_ancova)

#Calculate least square trends to determine differences in the rate of change in ovary volume
mod_trends<- lstrends(mod_ancova, ~Genotype, var="Timepoint")
pairs(mod_trends,adjust="holm")
cld(mod_trends)  

#Calculate least square means to determine differences in ovary volume
mod_means<- lsmeans(mod_ancova, ~Genotype, var="Timepoint")
pairs(mod_means,adjust="holm")
cld(mod_means)

#(C) IIS AND ECDYSONE SIGNALLING MANIPULATIONS

#Subsetting the data
tj <- subset (genetic.starved.volume, Genotype=="tj")
tj_EcRi_PTEN <-subset(genetic.starved.volume, Genotype=="tj_EcRi_PTEN")
tj_EcRi_InR <-subset(genetic.starved.volume, Genotype=="tj_EcRi_InR")

#Join data.frames per rows (vertically)
e_i_S <- rbind(tj,tj_EcRi_InR,tj_EcRi_PTEN)

#check for the assumptions of (G)LM models
mod_ancova1 <- lm(Volume ~ Timepoint*Genotype,data= e_i_S)
mod_ancova2 <- lm(log10.volume ~ Timepoint*Genotype,data= e_i_S)

par(mfrow=c(2,4))

plot(mod_ancova1)
plot(mod_ancova2)

#Perform ANCOVA
mod_ancova <- lm(log10.volume~ Timepoint*Genotype,data= e_i_S)
summary(mod_ancova)

#Calculate least square trends to determine differences in the rate of change in ovary volume
mod_trends<- lstrends(mod_ancova, ~Genotype, var="Timepoint")
pairs(mod_trends,adjust="holm")
cld(mod_trends)

#Calculate least square means to determine differences in ovary volume
mod_means<- lsmeans(mod_ancova, ~Genotype, var="Timepoint")
pairs(mod_means,adjust="holm")
cld(mod_means)