# R script for ANCOVA analysis of TF number and rate of TF addition

# Data for analyses are: 
# wildtype_TFs for wildtype ovaries from larvae fed on different dietary treatments
# genetic_fed_TFs for genetically manipulated ovaries from larvae fed on standard food 
# genetic_starved_TFs for genetically manipulated ovaries from larvae fed on 20% sucrose food from 5h AL3E

load("wildtype_TFs.RData")
load("genetic_fed_TFs.RData")
load("genetic_starved_TFs.RData")

#Packages required:
require("lsmeans")

######  WILD-TYPE OVARIES FROM LARVAE FED ON DIFFERENT DIETARY TREATMENTS    ######

#Log(10) transformation of TF number. As TF number can be zero, we also perform a sum transformation (TFs+1).
wildtype.TFs <- within(TFs_Diet, {TFs=(TFs+1)
                                 log10.TFs=log10(TFs)}) 

#Compare TF number and ratio of TF addition between three groups:
#larvae fed on standard food (FED)
#larvae fed on 20% sucrose food from 5h AL3E (PRE)
#larvae fed on 20% sucrose food from 15h AL3E (POST)

#check for the assumptions of (G)LM models
mod_ancova1 <- lm(TFs ~ Timepoint*Treatment,data= wildtype.TFs)
mod_ancova2 <- lm(log10.TFs ~ Timepoint*Treatment,data= wildtype.TFs)

par(mfrow=c(2,4))

plot(mod_ancova1)
plot(mod_ancova2)

#Perform ANCOVA
mod_ancova <- lm(log10.TFs ~ Timepoint*Treatment,data= wildtype.TFs)
summary(mod_ancova)

#Calculate least square trends to determine differences in the rate of TF addition
mod_trends<- lstrends(mod_ancova, ~Treatment, var="Timepoint")
pairs(mod_trends,adjust="holm")
cld(mod_trends)

#Calculate least square means to determine differences in TF number
mod_means<- lsmeans(mod_ancova, ~Treatment, var="Timepoint")
pairs(mod_means,adjust="holm")
cld(mod_means)

######  GENETIC MANIPULATIONS IN STANDARD FOOD    ######

#Log(10) transformation of TF number. As TF number can be zero, we also perform a sum transformation (TFs+1).
genetic.fed.TFs <- within(TFs_Genetic_Fed, {TFs=(TFs+1)
                                            log10.TFs=log10(TFs)}) 

#Compare TF number and ratio of TF addition within three groups:
#(A) ecdysone signalling manipulations
#(B) IIS manipulations
#(C) IIS and ecdysone signalling manipulations

# (A) ECDYSONE SIGNALLING MANIPULATIONS   

#Subsetting the data
tj <- subset (genetic.fed.TFs, Genotype=="tj")
tj_EcRdn <-subset(genetic.fed.TFs, Genotype=="tj_EcRdn")
tj_EcRi <-subset(genetic.fed.TFs, Genotype=="tj_EcRi")

#Join data.frames per rows (vertically)
ecdysone<- rbind(tj,tj_EcRdn,tj_EcRi)

#check for the assumptions of (G)LM models
mod_ancova1 <- lm(TFs ~ Timepoint*Genotype,data= ecdysone)
mod_ancova2 <- lm(log10.TFs ~ Timepoint*Genotype,data= ecdysone)

par(mfrow=c(2,4))

plot(mod_ancova1)
plot(mod_ancova2)

#Perform ANCOVA
mod_ancova <- lm(log10.TFs~ Timepoint*Genotype,data= ecdysone)
summary(mod_ancova)

#Calculate least square trends to determine differences in the rate of TF addition
mod_trends<- lstrends(mod_ancova, ~Genotype, var="Timepoint")
pairs(mod_trends,adjust="holm")
cld(mod_trends)  

#Calculate least square means to determine differences in TF number
mod_means<- lsmeans(mod_ancova, ~Genotype, var="Timepoint")
pairs(mod_means,adjust="holm")
cld(mod_means)

#(B) IIS MANIPULATIONS  

#Subsetting the data
tj <- subset (genetic.fed.TFs, Genotype=="tj")
tj_InR <-subset(genetic.fed.TFs, Genotype=="tj_InR")
tj_PTEN <-subset(genetic.fed.TFs, Genotype=="tj_PTEN")

#Join data.frames per rows (vertically)
insulin <- rbind(tj,tj_InR,tj_PTEN)

#check for the assumptions of (G)LM models
mod_ancova1 <- lm(TFs ~ Timepoint*Genotype,data= insulin)
mod_ancova2 <- lm(log10.TFs ~ Timepoint*Genotype,data= insulin)

par(mfrow=c(2,4))

plot(mod_ancova1)
plot(mod_ancova2)

#Perform ANCOVA
mod_ancova <- lm(log10.TFs~ Timepoint*Genotype,data= insulin)
summary(mod_ancova)

#Calculate least square trends to determine differences in the rate of TF addition
mod_trends<- lstrends(mod_ancova, ~Genotype, var="Timepoint")
pairs(mod_trends,adjust="holm")
cld(mod_trends)  

#Calculate least square means to determine differences in TF number
mod_means<- lsmeans(mod_ancova, ~Genotype, var="Timepoint")
pairs(mod_means,adjust="holm")
cld(mod_means)

#(C) IIS AND ECDYSONE SIGNALLING MANIPULATIONS  

#Subsetting the data
tj <- subset (genetic.fed.TFs, Genotype=="tj")
tj_EcRi_PTEN <-subset(genetic.fed.TFs, Genotype=="tj_EcRi_PTEN")
tj_EcRi_InR <-subset(genetic.fed.TFs, Genotype=="tj_EcRi_InR")
tj_EcRdn_PTEN <-subset(genetic.fed.TFs, Genotype=="tj_EcRdn_PTEN")
tj_EcRdn_InR <-subset(genetic.fed.TFs, Genotype=="tj_EcRdn_InR")

#Join data.frames per rows (vertically)
e_i <- rbind(tj,tj_EcRi_InR,tj_EcRi_PTEN,tj_EcRdn_InR,tj_EcRdn_PTEN)

#check for the assumptions of (G)LM models
mod_ancova1 <- lm(TFs ~ Timepoint*Genotype,data= e_i)
mod_ancova2 <- lm(log10.TFs ~ Timepoint*Genotype,data= e_i)

par(mfrow=c(2,4))

plot(mod_ancova1)
plot(mod_ancova2)

#Perform ANCOVA
mod_ancova <- lm(log10.TFs~ Timepoint*Genotype,data= e_i)
summary(mod_ancova)

#Calculate least square trends to determine differences in the rate of TF addition
mod_trends<- lstrends(mod_ancova, ~Genotype, var="Timepoint")
pairs(mod_trends,adjust="holm")
cld(mod_trends)  

#Calculate least square means to determine differences in TF number
mod_means<- lsmeans(mod_ancova, ~Genotype, var="Timepoint")
pairs(mod_means,adjust="holm")
cld(mod_means)

######  GENETIC MANIPULATIONS IN 20% SUCROSE FOOD   ######

#Log(10) transformation of TF number. As TF number can be zero, we also perform a sum transformation (TFs+1).
genetic.starved.TFs <- within(TFs_Genetic_Starved, {TFs=(TFs+1)
                                            log10.TFs=log(TFs)})  

#Compare TF number and ratio of TF addition within three groups:
#(A) ecdysone signalling manipulations
#(B) IIS manipulations
#(C) IIS and ecdysone signalling manipulations

#(A) ECDYSONE SIGNALLING MANIPULATIONS 

#Subsetting the data
tj <- subset (genetic.starved.TFs, Genotype=="tj")
tj_EcRi <-subset(genetic.starved.TFs, Genotype=="tj_EcRi")

#Join data.frames per rows (vertically)
ecdysone_S <- rbind(tj,tj_EcRi)

#check for the assumptions of (G)LM models
mod_ancova1 <- lm(TFs ~ Timepoint*Genotype,data= ecdysone_S)
mod_ancova2 <- lm(log10.TFs ~ Timepoint*Genotype,data= ecdysone_S)

par(mfrow=c(2,4))

plot(mod_ancova1)
plot(mod_ancova2)

#Perform ANCOVA
mod_ancova <- lm(log10.TFs~ Timepoint*Genotype,data= ecdysone_S)
summary(mod_ancova)

#Calculate least square trends to determine differences in the rate of TF addition
mod_trends<- lstrends(mod_ancova, ~Genotype, var="Timepoint")
pairs(mod_trends,adjust="holm")
cld(mod_trends)  

#Calculate least square means to determine differences in TF number
mod_means<- lsmeans(mod_ancova, ~Genotype, var="Timepoint")
pairs(mod_means,adjust="holm")
cld(mod_means)

#(B) IIS MANIPULATIONS 

#Subsetting the data
tj <- subset (genetic.starved.TFs, Genotype=="tj")
tj_InR <-subset(genetic.starved.TFs, Genotype=="tj_InR")

#Join data.frames per rows (vertically)
insulin_S <- rbind(tj,tj_InR)

#check for the assumptions of (G)LM models
mod_ancova1 <- lm(TFs ~ Timepoint*Genotype,data= insulin_S)
mod_ancova2 <- lm(log10.TFs ~ Timepoint*Genotype,data= insulin_S)

par(mfrow=c(2,4))

plot(mod_ancova1)
plot(mod_ancova2)

#Perform ANCOVA
mod_ancova <- lm(log.TFs~ Timepoint*Genotype,data= insulin_S)
summary(mod_ancova)

#Calculate least square trends to determine differences in the rate of TF addition
mod_trends<- lstrends(mod_ancova, ~Genotype, var="Timepoint")
pairs(mod_trends,adjust="holm")
cld(mod_trends)  

#Calculate least square means to determine differences in TF number
mod_means<- lsmeans(mod_ancova, ~Genotype, var="Timepoint")
pairs(mod_means,adjust="holm")
cld(mod_means)

#(C) IIS AND ECDYSONE SIGNALLING MANIPULATIONS 

#Subsetting the data
tj <- subset (genetic.starved.TFs, Genotype=="tj")
tj_EcRi_PTEN <-subset(genetic.starved.TFs, Genotype=="tj_EcRi_PTEN")
tj_EcRi_InR <-subset(genetic.starved.TFs, Genotype=="tj_EcRi_InR")

#Join data.frames per rows (vertically)
e_i_S <- rbind(tj,tj_EcRi_InR,tj_EcRi_PTEN)

#check for the assumptions of (G)LM models
mod_ancova1 <- lm(TFs ~ Timepoint*Genotype,data= e_i_S)
mod_ancova2 <- lm(log10.TFs ~ Timepoint*Genotype,data= e_i_S)

par(mfrow=c(2,4))

plot(mod_ancova1)
plot(mod_ancova2)

#Perform ANCOVA
mod_ancova <- lm(log.TFs~ Timepoint*Genotype,data= e_i_S)
summary(mod_ancova)

#Calculate least square trends to determine differences in the rate of TF addition
mod_trends<- lstrends(mod_ancova, ~Genotype, var="Timepoint")
pairs(mod_trends,adjust="holm")
cld(mod_trends) 

#Calculate least square means to determine differences in TF number
mod_means<- lsmeans(mod_ancova, ~Genotype, var="Timepoint")
pairs(mod_means,adjust="holm")
cld(mod_means)