% Simulate epidemic via renewal model
function Rtrue = getRTrue2(scenNo, tday, nday)

% Assumptions and notes
% - function expects tday to range 1:nday
% - warns if sequence of consecutive zero incidences
% - various R trajectories specified
% - computes incidence and its weekly moving avg


% Possible scenarios available - must match calling function
scenNam = {'cyclic', 'boom-bust', 'bottle' 'linear', 'piecewise', 'rateshift', 'logistic'};
disp(['True R scenario: ' scenNam{scenNo}]);

% Functions for scenarios: R on a daily basis
switch(scenNo)
    case 1
        % Sinusoidal R across time
        Rtrue = 1.1 + 0.5*sind(tday);
    case 2
        % Exponential rise and fall
        Rtrue = zeros(size(tday)); tchange = floor(nday/2);
        trise = 1:tchange; tfall = tchange+1:nday;
        % Exponential rise to max at tchange
        Rtrue(trise) =  exp(0.002*(1:tchange)); Rmax = Rtrue(tchange);
        % Exponential decay from max
        Rtrue(tfall) = Rmax*exp(-0.002*(tfall - tchange));
    case 3
        % A constant, then fall, then back to constant
        Rtrue = zeros(size(tday));
        tchange = [floor(nday/3), floor(2*nday/3)];
        % First high segment
        Rtrue(1:tchange(1)) = 2;
        % Second low segment
        Rtrue(tchange(1)+1:tchange(2)) = 0.5;
        % Third high segment
        Rtrue(tchange(2)+1:nday) = 1;
    case 4
        % Linear increasing R
        Rtrue = 0.005*tday + 1;
        
    case 5
        % Piecewise-constant case with random no. segments
        nSegs = 6; Rsegs = unifrnd(0.5, 2.5, [1 nSegs+1]);
        tchange = sort(datasample(2:nday-1, nSegs, 'Replace',false));
        % Assign random R intensities to each
        Rtrue = zeros(size(tday));
        for i = 1:nSegs+1
            % R indices for current segment
            if i == 1
                idR = 1:tchange(i);
            elseif i == nSegs+1
                idR = tchange(i-1)+1:nday;
            else
                idR = tchange(i-1)+1:tchange(i);
            end
            % Values over this segment
            Rtrue(idR) = Rsegs(i);
        end
        disp(['No. random segments = ' num2str(nSegs+1)]);
    case 6
        % Single rateshift at a random time point - e.g. vaccine
        shift = floor(nday/2);
        Rtrue = zeros(size(tday));
        % Initially high then low
        Rtrue(1:shift) = 1.8;
        Rtrue(shift+1:end) = 0.6;
        
    case 7
        % Logistic fall in R with time
        R0 = 0.75; Rd = 0.5;
        t1 = 1; t2 = floor(nday/2 + 20);
        Rtrue = R0 + Rd*((1 + exp(-t1*t2))./(1 + exp(-t1*(t2 - tday))));
end



