% Function to get best k and R estimate
function [Rlik, Rbic, Rfia, Rqian, Raic, kbest, nGrpbest, FIsum, liksum, fia, bic, aic] = ...
    fraserSubMultFn2(T, ks, nks, Rmax, nGrps, grpSzs, Iday, Lam, nStream)

% Assumptions and notes
% - includes truncation of initial zeros
% - estimates R and performs model selection
% - expects a group of trajectories
% - vectors of size T = nday - 1
% - uses log-likelihood for independent streams
% - Iday and Lam dims depend on nStream


% Reprod nums with time and grouping
Rks = zeros(nks, T); 
% Total likelihood of each trajectory
liksum = zeros(1, nks); 
% Fisher information sum and product
FIsum = zeros(1, nks); FIprod = FIsum;
% FIA, BIC and Qian selection criteria 
fia = zeros(1, nks); bic = fia; qian = fia; aic = fia;
% For log-sum inequality interrogation
l = zeros(1, nks); a = l; b = l;

% For each set of ks get MLE and log-likelihoods
for i = 1:nks
    % Group properties at this k
    nGrp = nGrps(i); grpSz = grpSzs{i};
    
    % Components of group MLEs, liks, criteria
    A = zeros(1, nGrp); B = A; lik = A; 
    FIAcomp = A; BICcomp = A; Rgrp = A;
    
    % Grouped MLEs for Rt and max log-likelihoods
    jstart = 1;
    for j = 1:nGrp
        % End-indices of group 
        jstop = jstart + grpSz(j) - 1;
        ids = jstart:jstop;
        
        % Component sums: A (Lamj) and B (Ij)
        A(j) = sum(sum(Lam(:, ids)));
        B(j) = sum(sum(Iday(:, ids)));

        % Grouped MLE (replicated over t points in group i.e. ids)
        Rks(i, ids) = B(j)/A(j); Rgrp(j) = B(j)/A(j);
        % Maximum log-likelihood (also replicated to match R)
        lik(j) = getLogLikEst(Rgrp(j), B(j));
        
        % Component for FIA, BIC and Qian
        FIAcomp(j) = -lik(j) + 0.5*log(2*Rmax*A(j)/pi);
        BICcomp(j) = -lik(j) + 0.5*log(nStream*T);
        
        % Update end indices
        jstart = jstop + 1;
    end
    
    % Segment log-likelihood
    liksum(i) = sum(lik);
    % Log-sum calculation
    [l(i), a(i), b(i)] = logSumInequal(A, B);
  
    % FI over groups and product
    FI = (A.^2)./B; 
    FIsum(i) = sum(FI); FIprod(i) = prod(FI);
    
    % FIA, BIC and Qian for that model
    fia(i) = sum(FIAcomp); bic(i) = sum(BICcomp);
    qian(i) = -liksum(i) + sum(log(abs(Rgrp) + (nStream*T)^(-0.25))) +...
        0.5*log(FIprod(i));
    aic(i) = -liksum(i) + nGrp;
end

% Check for issues with FI
if any(isinf(FIprod) | isnan(FIprod))
    warning('Issues with FI at this grouping');
    assignin('base', 'FIprodErr', FIprod);
end
% Check on R estimates
if any(any(any(isinf(Rks)))) || any(any(any(isnan(Rks))))
    error('MLEs for R are inf or nan');
end
% Check for a and b sums consistent
if max(abs(diff(a))) + max(abs(diff(b))) > 10^(-6)
    warning('Log-sums seem suspicious');
    disp(['Max diff a = ' num2str(max(abs(diff(a))))]);
    disp(['Max diff b = ' num2str(max(abs(diff(b))))]);
end

% Select best model as min of criteria
[~, bicMod] = min(bic); [~, fiaMod] = min(fia);
[~, qianMod] = min(qian); [~, LMod] = min(-liksum);
[~, aicMod] = min(aic);

% Best ks and nGrps
modID = [LMod bicMod fiaMod qianMod aicMod];
kbest = ks(modID); nGrpbest = nGrps(modID);
% Ouptut results for model selected estimates
Rlik = Rks(modID(1), :); Rbic = Rks(modID(2), :);
Rfia = Rks(modID(3), :); Rqian = Rks(modID(4), :);
Raic = Rks(modID(5), :); assignin('base', 'Rks', Rks);
