% Functional form of fraserEst - estimates R using renewal models
function [Rlik, Rbic, Rfia, Rqian, kbest, nGrpbest, warnTrue, Rtrue] = ...
    fraserEstFnIndep(T, Pomega, ks, nks, Rmax, nSegs, tday, tchange)

% Assumptions and notes
% - estimates R and performs model selection
% - also simulates a single trajectory

%% Simulate epidemic scenarios

% Daily incidence
Iday = zeros(size(T+1)); 
% Infectiousness, Poisson rate 
Lam = Iday; rate = Iday;
% Initialise epidemic
Iday(1) = 10;

% Pre-specified R trajectory to simulate
Rsegs = unifrnd(0.5, 4, [1 nSegs]);
% Vector for true R
Rtrue = zeros(size(tday));
for i = 1:nSegs
    % R indices for current segment
    if i == 1
        idR = 1:tchange(i);
    elseif i == nSegs
        idR = tchange(i-1)+1:T+1;
    else
        idR = tchange(i-1)+1:tchange(i);
    end
    % Values over this segment
    Rtrue(idR) = Rsegs(i);
end

% Iteratively generate renewal epidemic
for i = 2:T+1
    % Relevant part of serial distribution
    Pomegat = Pomega(1:i-1);
    % Total infectiousness
    Lam(i) = Iday(i-1:-1:1)*Pomegat';
    % Rate for ith day incidence
    rate(i) = Lam(i)*Rtrue(i);
    % Renewal incidence
    Iday(i) = poissrnd(rate(i));
end

%% Fraser-Cori MLEs for R and model selection

% Time vector over which R holds (length is data size)
lenRt = T;

% Get group sizes for each k
grps = @(k) getGrpSz(k, lenRt);
grps = arrayfun(grps, ks);

% Reprod nums with time and grouping
Rkset = zeros(nks, lenRt); 
% Total likelihood of each trajectory
liksum = zeros(1, nks); 
% Fisher information sum and product
FIsum = zeros(1, nks); FIprod = FIsum;

% FIA and BIC selection criteria 
fia = zeros(1, nks); bic = fia; qian = fia;

% Group numbers and likelihood on each k
nGrps = zeros(size(ks)); liks = cell(1, nks);
% R values across k and time
Rks = zeros(nks, lenRt); 

% For each set of ks get MLE and log-likelihoods
for i = 1:nks
    % Group properties at this k
    nGrp = grps(i).nGrp; nGrps(i) = nGrp; 
    grpSz = grps(i).grpSz;
    
    % Components of group MLEs, liks, criteria
    A = zeros(1, nGrp); B = A; lik = A; 
    FIAcomp = A; BICcomp = A; Rgrp = A;
    % Start and end indices
    idday = zeros(nGrp, 2);
    
    % Grouped MLEs for Rt and max log-likelihoods
    jstart = 1;
    for j = 1:nGrp
        % End-indices of group 
        jstop = jstart + grpSz(j) - 1;
        ids = jstart:jstop;
        % Ids (start and end) constituting each group
        idday(j, :) = [jstart jstop];
        
        % Component sums: A (Lamj) and B (Ij)
        A(j) = sum(Lam(ids));
        B(j) = sum(Iday(ids));

        % Grouped MLE (replicated over t points in group i.e. ids)
        Rks(i, ids) = B(j)/A(j); Rgrp(j) = B(j)/A(j);
        % Maximum log-likelihood (also replicated to match R)
        lik(j) = getLogLikEst(Rgrp(j), B(j));
        
        % Component for FIA, BIC and Qian
        FIAcomp(j) = -lik(j) + 0.5*log(2*Rmax*A(j)/pi);
        BICcomp(j) = -lik(j) + 0.5*log(lenRt);
        
        % Update end indices
        jstart = jstop + 1;
    end
    
    % Segment log-likelihood
    liksum(i) = sum(lik);
  
    % FI over groups and product
    FI = (A.^2)./B; 
    FIprod(i) = prod(FI);
    
    % FIA, BIC and Qian for that model
    fia(i) = sum(FIAcomp);
    bic(i) = sum(BICcomp);
    qian(i) = -liksum(i) + sum(log(abs(Rgrp) + lenRt^(-0.25))) +...
        0.5*log(FIprod(i));
end

% Check for issues with FI
if any(isinf(FIprod) | isnan(FIprod))
    warnTrue = 1;
else
    warnTrue = 0;
end
% Check on R estimates
if any(any(any(isinf(Rkset)))) || any(any(any(isnan(Rkset))))
    error('MLEs for R are inf or nan');
end

% Select best model as min of criteria
[~, bicMod] = min(bic);
[~, fiaMod] = min(fia);
[~, qianMod] = min(qian);
% Min from likelihood alone
[~, LMod] = min(-liksum);

% Best ks and nGrps
modID = [LMod bicMod fiaMod qianMod];
kbest = ks(modID); nGrpbest = nGrps(modID);

% Ouptut results for model selected estimates
Rlik = Rks(modID(1), :);
Rbic = Rks(modID(2), :);
Rfia = Rks(modID(3), :);
Rqian = Rks(modID(4), :);

