% Model selection over skylines for different square waves
clearvars; clc;
close all; tic;

% Assumptions and notes
% - examines only FIA and dependence on Nmax
% - simple folder structure, 1 wave per folder
% - square wave model selection, different periods

% Aditional plotting/partition package
addpath(genpath('/Users/kp10/Documents/MATLAB'));

% Set figure defaults
set(groot, 'defaultAxesTickLabelInterpreter', 'latex');
set(groot, 'defaultLegendInterpreter', 'latex');
set(0, 'defaultTextInterpreter', 'latex');
set(0, 'defaultAxesFontSize', 16);
grey1 = 0.8*ones(1, 3); grey2 = 0.5*ones(1, 3);

% Figures to plot, data to save
allFigs = 0; savetrue = 1;

% Home and folders to load
thisDir = cd;
mainFol = 'using phylodyn/sq_short/';
% Data controlling all sims
cd(mainFol);
files = dir('sqwaves_*');
nwave = length(files);
cd(thisDir);

% Metrics to be evaluated
metrics = {'$-\log L$', 'BIC', 'FIA', 'QK', 'AIC'};
nmet = length(metrics);

% Max for FIA volume - range
Nmax = logspace(2, 5, 50);
V = log(Nmax); len = length(Nmax);


% Test different FIA volumes
Psums = cell(1, len);
for j = 1:len
    % Store results
    Pdet = cell(1, nwave);
    
    % Run model selection across all files
    for i = 1:nwave
        % File path to load
        pathload = [mainFol files(i).name];
        % Main model selection function
        [~, Pdet{i}, ~] = sqWvFixFn(thisDir, V(j), nmet, pathload);
    end
    
    % Overall P(true)
    Psum = Pdet{1};
    for i = 2:nwave
        Psum = Psum + Pdet{i};
    end
    Psum = Psum/nwave;
    % Only care about FIA (index 3)
    Psums{j} = Psum(:, 3);
end

% Some fixed variables across all trajectories
cd(pathload);
wave = csvread('wave.csv'); nRuns = csvread('numRuns.csv');
M = csvread('numTraj.csv'); sampRuns = csvread('sampRuns.csv');
tsamp = csvread('samptimes.csv'); nsegs = length(tsamp);
cd(thisDir);
% Shift time of samples
tau = unique(diff(tsamp));

% Extract FIA performance 
Pfia = cell2mat(Psums);
Pfiaquant = quantile(Pfia', [0.01 0.99]);
Pfiaquant = Pfiaquant';

% Prob FIA detection with samples
figure;
plot(sampRuns-1, Psum(:, [2 4 5]), '-',  'linewidth', 2, 'color', grey1);
hold on;
plot(sampRuns-1, Pfiaquant, '-', 'linewidth', 2, 'color', grey2);
hold off;
xlabel(['$m$ ($\tau$ = ' num2str(tau) ')']);
ylabel('P(correct)');
grid off; box off;
if savetrue
    saveas(gcf, ['FIA' num2str(len)], 'fig');
end

figure;
plot(sampRuns-1, Psum(:, [2 4 5]), '.-',  'linewidth', 2, 'color', 'c', 'MarkerSize', 30);
hold on;
plot(sampRuns-1, Pfiaquant, '.-', 'linewidth', 2, 'color', grey2, 'MarkerSize', 30);
hold off;
xlabel(['$m$ ($\tau$ = ' num2str(tau) ')']);
ylabel('P(correct)');
grid off; box off;

% Log time
tsim = toc/60;
disp(['Execution time: ' num2str(tsim) ' mins']);

% Save data
if savetrue
    save(['compFIA_' num2str(len) '_' num2str(M) '.mat']);
end
