#####################################################################################################################################################################
###################################### This is a gene dropping simulation in arbitrary pedigrees to estimate the variance in F ######################################
#####################################################################################################################################################################
# Written 09/01/2014
# The simulation takes as input (1) a linkage map, (2) a pedigree file.
## The linkage map file should contain the following columns: "Chromosome","Marker","Pos.Phys","Pos.Genet"
## The pedigree file should contain the following columns: "IndID","FID","MID","Sex"
# The following input parameters must be provided:
## (1) the number of simulations to run ('nSims'),
## (2) whether you want to save the number of crossovers per meiosis in males and females ('save.recombs'),
## (3) the range of crossover interference in bp,
## (4) whether or not you want to remove the sex chromosome (TguZ).
## ------------------------------------------------------------------------------------------------------------------

# The simulation works as follows:
# (1) It creates a new look-up table that contains the recombination probabilities between adjacent markers (if requested, the sex chromosome is removed). 
## To calculate the recombination probability it uses the Kosambi map function, since our marker positions are given in Kosambi cM.
## Thus, r=0.5*((e^(4*d)-1)/(e^(4*d)+1)), in which r is the recombination probability and d is the distance in Morgan.
## Then it calculates the total physical genome length in bp from the map file.
# (2) It checks the pedigree file for inconsistencies in sex and whether it is ordered in that way that parents are placed before their offspring.
## It produces an error message if the pedigree is not correct. If you do not stop the program, it will proceed though and encounter errors later. 
# (3) It creates the output file on the harddrive (empirical F values (F.Emp) for each individual and each simulation run are stored).
## It creates a list in which all individuals are stored (see below for details).
# (4) It generates the founders + their haplotypes (for each chromosome, i.e. for the zebra finch 2*40=80 chromosomes).
# (5) The simulation loop starts:
## (5.1) The first individual that is not a founder is choosen.
## (5.2) Its parents and their haplotypes are selected.
## (5.3) Loop over all chromosomes and introduce cross-overs. The simulation makes sure that at least one cross-over event per chromosome occurs.
### If specified, it takes crossover interference into account (in the zebra finches, Calderon & Pigozzi (2006) counted 45.7 and 45.2 MLH1 foci per oocyte and spermatocyte, respectively).
### The crossover interference works like that: if there are two or more recombination events inside the interference range, then only one is kept randomly.
## (5.4) For each chromosome the individual gets one (recombined) chromosome from its mother and its father.
## (5.5) For each chromosome the length of autozygous stretches (homozygosity for the founder haplotypes) is calculated by taking into account
### (A) that if the stretch starts at the first or last marker it cannot be as long as in the centre
### (B) that crossovers occur randomly between the two adjacent markers.
## (5.6) For each individual the total length of autozygous stretches is calculated and divided by the total genome length --> F.Emp.
## (5.7) F.Emp is stored on the hard drive.
## (5.8) The parents of the individual are removed from the list of individuals if they are (A) not founders and (B) not needed later on as parents again.
## (5.9) Only if the individual is needed later on in the simulation as a parent its haplotypes get stored in the list of individuals.
### Steps (5.8) and (5.9) should speed up the simulation if there is an extended pedigree to simulate.
## (5.10) Proceed at step (5.1) with the next individual.

# ToDo:
# Create plot full-sib mating linkage map 100kb, 10kb, 1kb, 1st generation, 5th generation (mean + SD)

rm(list=ls())
require(pedigreemm)
# ------------------------------------------------------------------------------------------------------------------
# (0) inputs -----------------------------------------------------------------------------------------------------------
# ------------------------------------------------------------------------------------------------------------------
# define path to files here
path = "C:\\Users\\Data\\"
# n simulations
nSims <- 50
# save the number of recombination events generated?
save.recombs <- TRUE
# remove sex chromosome
rem.sex <- TRUE
# n loci for sampling
nLoci <- c(5,10,20,40,80,160)
# mean IBS (homozygosity of markers)
#meanH <- 0.235482579	# empirical
meanH <- 0.1472794		# random
# create summary
smmry <- FALSE
# ---

# ---------------------------------------------------------
for(ww in 1:length(nLoci)) {
# name outfile
outfile <- paste("+++SIM100kb_humans_InbreedingL",nLoci[ww],"_n",nSims,"_Discrepancy_1.txt",sep="")
# ---------------------------------------------------------


options(warn=2)
# files
ped <- read.table(paste(path,"data_SIM.ped3404RAND.txt",sep=""), sep="\t", header=TRUE)
#ped <- read.table(paste(path,"data_SIM.ped3404.txt",sep=""), sep="\t", header=TRUE)
#ped <- read.table(paste(path,"data_SIM.ped.txt",sep=""), sep="\t", header=TRUE)
map <- read.table(paste(path,"data_HumanB37.SIM100kb.map.txt",sep=""), sep="\t", header=TRUE)
intrfrnc <- read.table(paste(path,"data_HumanB37.chromosomes.txt",sep=""), sep="\t", header=TRUE)
intrfrnc <- intrfrnc[,c("Tgu","inter_range")]
addChr <- data.frame(unique(map$Chromosome)[!(unique(map$Chromosome) %in% intrfrnc$Tgu)])
addChr <- cbind(addChr, intrfrnc$inter_range[which(intrfrnc$Tgu=="chr1.1")])
colnames(addChr) <- c("Tgu","inter_range")
intrfrnc <- rbind(intrfrnc,addChr)
splitAt <- function(x, pos) unname(split(x, cumsum(seq_along(x) %in% pos)))

ped_mm <- pedigree(sire = ped$MID, dam  = ped$FID, label= ped$IndID)
FPed <- inbreeding(ped_mm)
ped7 <- cbind(ped,FPed)
ped7 <- subset(ped7,IndID>=12000 & IndID<13000)
meanF <- mean(ped7$FPed)

# ------------------------------------------------------------------------------------------------------------------
# (1) create recombination probabilities for each chromatid strand from linkage map ------------------------------------
# ------------------------------------------------------------------------------------------------------------------
if(rem.sex==TRUE) {
	map <- subset(map, Chromosome!="chrX")
}
prob <- data.frame(matrix(,ncol=5))
colnames(prob) <- c("Pos.Genet.Start","Pos.Genet.End","Pos.Phys.Start","Pos.Phys.End","Rec.Prob")
prob <- prob[-1,]
for(j in 1:length(unique(map$Chromosome))) {
	chr <- subset(map, Chromosome==unique(map$Chromosome)[j])
	m1 <- chr$Marker[-nrow(chr)]
	m2 <- chr$Marker[-1]
	chr <- matrix(c(chr$Pos.Genet[-nrow(chr)],chr$Pos.Genet[-1],chr$Pos.Phys[-nrow(chr)],chr$Pos.Phys[-1]),ncol=4)
	dstnc <- chr[,2]-chr[,1]
	chr <- data.frame(chr)
	chr$Prob <- ((exp(4*((dstnc)/100))-1)/(exp(4*((dstnc)/100))+1))
	chr <- cbind(unique(map$Chromosome)[j],m1,m2,chr)
	colnames(chr) <- c("Chromosome","Marker1","Marker2","Pos.Genet.Start","Pos.Genet.End","Pos.Phys.Start","Pos.Phys.End","Rec.Prob")
	chr$Pos.Phys.End <- chr$Pos.Phys.End-1
	prob <- rbind(prob,chr)
}

# calculate the total physical genome length from the map file
genome.length <- 0
for(aa in 1:length(unique(prob$Chromosome))) {
	ToDo <- subset(prob,Chromosome==unique(prob$Chromosome)[aa])
	ToDo <- max(ToDo$Pos.Phys.End)
	genome.length <- genome.length+ToDo
}


# ------------------------------------------------------------------------------------------------------------------
# (2) check pedigree ---------------------------------------------------------------------------------------------------
# ------------------------------------------------------------------------------------------------------------------
for(i in 1:nrow(ped)) {
	ind <- ped[i, ]
	if(ind$Sex==1) {
		x <- which(ped$FID==ind$IndID)
		y <- which(ped$MID==ind$IndID)
		if(length(x)>0) {
			stop("males and females are mixed up!")
		}
		if(any(y<i)) {
			stop("pedigree is not ordered correctly - offspring preceed parents")
		}
	}
	if(ind$Sex==0) {
		x <- which(ped$FID==ind$IndID)
		y <- which(ped$MID==ind$IndID)
		if(length(y)>0) {
			stop("males and females are mixed up!")
		}
		if(any(y<i)) {
			stop("pedigree is not ordered correctly - offspring preceed parents")
		}
	}
}


# ------------------------------------------------------------------------------------------------------------------
# (3) create a list in which all sublists get stored. Also create textfile for simulation output -----------------------
# ------------------------------------------------------------------------------------------------------------------
INDS <- list()
# save the number of recombination events with the rest if requested
if(save.recombs==TRUE) {
	out <- data.frame("Sim.run","IndID","F.Emp","F.100L.correct","F.100L.error","Recombs.FID","Recombs.MID") } else {
	out <- data.frame("Sim.run","IndID","F.Emp","F.100L.correct","F.100L.error") }
write.table(out,paste(path, outfile, sep=""), append=FALSE, row.names=FALSE, col.names=FALSE, sep="\t", quote=FALSE, eol="\n")


# ------------------------------------------------------------------------------------------------------------------
# (4) generate founder haplotypes --------------------------------------------------------------------------------------
# ------------------------------------------------------------------------------------------------------------------
founders <- subset(ped,FID==0 | MID==0)
k <- 1
for(i in 1:nrow(founders)) {
	F <- NULL
	F <- list()
	F[[1]] <- founders$IndID[i]
	for(j in 1:length(unique(map$Chromosome))) {
		chr <- subset(map, Chromosome==unique(map$Chromosome)[j])
		# generate first haplotype, which is a repetition of a unique number (for each chromosome in an individual it is the same number, but different from the second haplotype)
		Haplo1 <- rep((2*k-1),nrow(chr))
		# generate second haplotype, which is a repetition of a unique number (for each chromosome in an individual it is the same number, but different from the first haplotype)
		Haplo2 <- rep((2*k),nrow(chr))
		F[[2*j]] <- Haplo1
		F[[2*j+1]] <- Haplo2
	}
	INDS[[k]] <- F
	flush.console()
	print(k)
	k <- k+1
}
# Save k as kk in order to restore it in each simulation
kk <- k


# ------------------------------------------------------------------------------------------------------------------
# (5) loop over the whole pedigree (only here the loop for nSims has to start) -----------------------------------------
# ------------------------------------------------------------------------------------------------------------------
for(nn in 1:nSims) {

# ---------------------------------------------------------
# Select n loci randomly in the genome
loci100 <- prob[sample(c(1:nrow(prob)),nLoci[ww],replace=FALSE),]
#loci100 <- prob[which(prob$Marker1 %in% c("chr1.593","chr2.782","chr1A.368","chr4.349")),]
# ---------------------------------------------------------

timeStart = Sys.time()
# restore k
k <- kk
gnrtns <- subset(ped,FID!=0 & MID!=0)
for(i in 1:nrow(gnrtns)) {
	F <- NULL 
	F <- list()
	# save individual ID
	F[[1]] <- gnrtns$IndID[i]
	# get parents + their haplotypes of the individual
	FID <- INDS[[which(lapply(INDS, "[[", 1) == as.character(gnrtns$FID[i]))]]
	MID <- INDS[[which(lapply(INDS, "[[", 1) == as.character(gnrtns$MID[i]))]]
	# create a variable that stores the amount of the genome being autozygous (that value divided by the total length of the genome = FEmp)
	cum.length.genome <- 0
	all.length.genome <- 0
	# ---------------------------------------------------------
	cum.100L.genome.error <- 0
	cum.100L.genome.correct <- 0
	all.100L.genome.error <- 0
	all.100L.genome.correct <- 0
	# ---------------------------------------------------------
	# Variables for saving the number of recombination events
	if(save.recombs==TRUE) {
		xxFID <- 0
		xxMID <- 0
	}
	# loop over all chromosomes
	for(j in 1:length(unique(map$Chromosome))) {
		chr.map <- subset(map, Chromosome==as.character(unique(map$Chromosome)[j]))
		chr.prob <- subset(prob, Chromosome==as.character(unique(map$Chromosome)[j]))
		interf.range <- subset(intrfrnc, Tgu==as.character(unique(map$Chromosome))[j])$inter_range
		# get at least one crossover per chromosome
		xFID.1 <- 0
		xMID.1 <- 0
		# find the position where the crossover occurs in both female and male
		while(sum(xFID.1)==0) {
		xFID.1 <- rbinom(n=nrow(chr.prob), 1, p=chr.prob$Rec.Prob)
		}
		while(sum(xMID.1)==0) {
		xMID.1 <- rbinom(n=nrow(chr.prob), 1, p=chr.prob$Rec.Prob)
		}
		# introduce crossover interference if required
		if(interf.range>0) {
			# introduce interference - FID
			inter.F <- which(xFID.1==1)
			l <- 1
			while(l <= length(inter.F)) {
				sct <- xFID.1[max(inter.F[l]-interf.range/prob$Pos.Phys.End[1],1):min(inter.F[l]+interf.range/prob$Pos.Phys.End[1],length(xFID.1))]
				while(sum(sct)>1) {
					sct[sample(c(which(sct==1)),1,prob=c(1-chr.prob$Rec.Prob[which(sct==1)]))] <- 0  
				}
			xFID.1[max(inter.F[l]-interf.range/prob$Pos.Phys.End[1],1):min(inter.F[l]+interf.range/prob$Pos.Phys.End[1],length(xFID.1))] <- sct
			inter.F <- which(xFID.1==1)
			l <- l+1
			}
			# introduce interference - MID
			inter.M <- which(xMID.1==1)
			l <- 1
			while(l <= length(inter.M)) {
				sct <- xMID.1[max(inter.M[l]-interf.range/prob$Pos.Phys.End[1],1):min(inter.M[l]+interf.range/prob$Pos.Phys.End[1],length(xMID.1))]
				while(sum(sct)>1) {
					sct[sample(c(which(sct==1)),1,prob=c(1-chr.prob$Rec.Prob[which(sct==1)]))] <- 0
				}
			xMID.1[max(inter.M[l]-interf.range/prob$Pos.Phys.End[1],1):min(inter.M[l]+interf.range/prob$Pos.Phys.End[1],length(xMID.1))] <- sct
			inter.M <- which(xMID.1==1)
			l <- l+1
			}			
		}


		# save the number of recombination events
		if(save.recombs==TRUE) {
			xxFID <- xxFID + sum(xFID.1)
			xxMID <- xxMID + sum(xMID.1)
		}

		# choose left (which is equivalent to either maternal or paternal) haplotype randomly - FID
		FID.hapL <- sample(c(0,1),1)
		# split haplotypes at choosen recombination spots for both chromatids - FID
		FID.1.strt <- FID[[2*j+(FID.hapL%%2)]]
		FID.2.strt <- FID[[2*j+(FID.hapL%%2)]]
		FID.1.end <- FID[[2*j+((FID.hapL+1)%%2)]]
		FID.2.end <- FID[[2*j+((FID.hapL+1)%%2)]]
		# split all chromatids
		FID.1.strt <- splitAt(FID.1.strt,(which(xFID.1==1)+1))
		FID.2.strt <- splitAt(FID.2.strt,(which(xFID.1==1)+1))
		FID.1.end <- splitAt(FID.1.end,(which(xFID.1==1)+1))
		FID.2.end <- splitAt(FID.2.end,(which(xFID.1==1)+1))
		
		FID.Haplo1 <- c()
		FID.Haplo2 <- c()
		FID.Haplo3 <- c()
		FID.Haplo4 <- c()
		m <- 1
		haplos <- c(1,2,3,4)
		while(m<length(FID.1.strt)) {
			element1 <- sample(haplos,1)
			if(element1 %in% c(1,2)) {
				element2 <- sample(c(3,4),1)
				rest1 <- c(1,2)[!c(1,2) == element1]
				rest2 <- c(3,4)[!c(3,4) == element2]
			} else {
				element2 <- sample(c(1,2),1)
				rest1 <- c(3,4)[!c(3,4) == element1] 
				rest2 <- c(1,2)[!c(1,2) == element2]}
			if(m==1) {
				haplo1.start <- c(FID.1.strt[2*m-1],FID.2.strt[2*m-1],FID.1.end[2*m-1],FID.2.end[2*m-1])[element1]
				haplo2.start <- c(FID.1.strt[2*m-1],FID.2.strt[2*m-1],FID.1.end[2*m-1],FID.2.end[2*m-1])[element2]
				haplo1.end <- c(FID.1.strt[2*m],FID.2.strt[2*m],FID.1.end[2*m],FID.2.end[2*m])[element2]
				haplo2.end <- c(FID.1.strt[2*m],FID.2.strt[2*m],FID.1.end[2*m],FID.2.end[2*m])[element1]
				haplo3.start <- c(FID.1.strt[2*m-1],FID.2.strt[2*m-1],FID.1.end[2*m-1],FID.2.end[2*m-1])[rest1]
				haplo3.end <- c(FID.1.strt[2*m],FID.2.strt[2*m],FID.1.end[2*m],FID.2.end[2*m])[rest1]
				haplo4.start <- c(FID.1.strt[2*m-1],FID.2.strt[2*m-1],FID.1.end[2*m-1],FID.2.end[2*m-1])[rest2]
				haplo4.end <- c(FID.1.strt[2*m],FID.2.strt[2*m],FID.1.end[2*m],FID.2.end[2*m])[rest2]
				FID.Haplo1.1 <- unlist(c(haplo1.start,haplo1.end))
				FID.Haplo2.1 <- unlist(c(haplo2.start,haplo2.end))
				FID.Haplo3.1 <- unlist(c(haplo3.start,haplo3.end))
				FID.Haplo4.1 <- unlist(c(haplo4.start,haplo4.end))

				if(tail(FID.Haplo1.1,1)<=tail(FID.Haplo2.1,1)) {
					FID.Haplo1 <- FID.Haplo1.1
					FID.Haplo3 <- FID.Haplo2.1}
				if(tail(FID.Haplo1.1,1)>tail(FID.Haplo2.1,1)) {
					FID.Haplo3 <- FID.Haplo1.1
					FID.Haplo1 <- FID.Haplo2.1}				
				if(tail(FID.Haplo3.1,1)<=tail(FID.Haplo4.1,1)) {
					FID.Haplo2 <- FID.Haplo3.1
					FID.Haplo4 <- FID.Haplo4.1}				
				if(tail(FID.Haplo3.1,1)>tail(FID.Haplo4.1,1)) {
					FID.Haplo4 <- FID.Haplo3.1
					FID.Haplo2 <- FID.Haplo4.1}				
				
			} else {
				haplo1.start <- c(FID.1.strt[m+1],FID.2.strt[m+1],FID.1.end[m+1],FID.2.end[m+1])[element1]
				haplo2.start <- c(FID.1.strt[m+1],FID.2.strt[m+1],FID.1.end[m+1],FID.2.end[m+1])[element2]
				haplo3.start <- c(FID.1.strt[m+1],FID.2.strt[m+1],FID.1.end[m+1],FID.2.end[m+1])[rest1]
				haplo4.start <- c(FID.1.strt[m+1],FID.2.strt[m+1],FID.1.end[m+1],FID.2.end[m+1])[rest2]
				HH <- sample(list(FID.Haplo1,FID.Haplo2),2,replace=FALSE)
				FID.Haplo1.1 <- unlist(c(HH[1],haplo1.start))
				FID.Haplo2.1 <- unlist(c(HH[2],haplo4.start))
				HH <- sample(list(FID.Haplo3,FID.Haplo4),2,replace=FALSE)
				FID.Haplo3.1 <- unlist(c(HH[1],haplo2.start))
				FID.Haplo4.1 <- unlist(c(HH[2],haplo3.start))
				
				if(tail(FID.Haplo1.1,1)<=tail(FID.Haplo2.1,1)) {
					FID.Haplo1 <- FID.Haplo1.1
					FID.Haplo3 <- FID.Haplo2.1}
				if(tail(FID.Haplo1.1,1)>tail(FID.Haplo2.1,1)) {
					FID.Haplo3 <- FID.Haplo1.1
					FID.Haplo1 <- FID.Haplo2.1}				
				if(tail(FID.Haplo3.1,1)<=tail(FID.Haplo4.1,1)) {
					FID.Haplo2 <- FID.Haplo3.1
					FID.Haplo4 <- FID.Haplo4.1}				
				if(tail(FID.Haplo3.1,1)>tail(FID.Haplo4.1,1)) {
					FID.Haplo4 <- FID.Haplo3.1
					FID.Haplo2 <- FID.Haplo4.1}
			}
			m <- m+1
		}
		
		
		# choose left (which is equivalent to either maternal or paternal) haplotype randomly - MID
		MID.hapL <- sample(c(0,1),1)
		# split haplotypes at choosen recombination spots - MID
		MID.1.strt <- MID[[2*j+(MID.hapL%%2)]]
		MID.2.strt <- MID[[2*j+(MID.hapL%%2)]]
		MID.1.end <- MID[[2*j+((MID.hapL+1)%%2)]]
		MID.2.end <- MID[[2*j+((MID.hapL+1)%%2)]]
		# split all chromatids
		MID.1.strt <- splitAt(MID.1.strt,(which(xMID.1==1)+1))
		MID.2.strt <- splitAt(MID.2.strt,(which(xMID.1==1)+1))
		MID.1.end <- splitAt(MID.1.end,(which(xMID.1==1)+1))
		MID.2.end <- splitAt(MID.2.end,(which(xMID.1==1)+1))
		
		MID.Haplo1 <- c()
		MID.Haplo2 <- c()
		MID.Haplo3 <- c()
		MID.Haplo4 <- c()
		m <- 1
		haplos <- c(1,2,3,4)
		while(m<length(MID.1.strt)) {
			element1 <- sample(haplos,1)
			if(element1 %in% c(1,2)) {
				element2 <- sample(c(3,4),1)
				rest1 <- c(1,2)[!c(1,2) == element1]
				rest2 <- c(3,4)[!c(3,4) == element2]
			} else {
				element2 <- sample(c(1,2),1)
				rest1 <- c(3,4)[!c(3,4) == element1] 
				rest2 <- c(1,2)[!c(1,2) == element2]}
			if(m==1) {
				haplo1.start <- c(MID.1.strt[2*m-1],MID.2.strt[2*m-1],MID.1.end[2*m-1],MID.2.end[2*m-1])[element1]
				haplo2.start <- c(MID.1.strt[2*m-1],MID.2.strt[2*m-1],MID.1.end[2*m-1],MID.2.end[2*m-1])[element2]
				haplo1.end <- c(MID.1.strt[2*m],MID.2.strt[2*m],MID.1.end[2*m],MID.2.end[2*m])[element2]
				haplo2.end <- c(MID.1.strt[2*m],MID.2.strt[2*m],MID.1.end[2*m],MID.2.end[2*m])[element1]
				haplo3.start <- c(MID.1.strt[2*m-1],MID.2.strt[2*m-1],MID.1.end[2*m-1],MID.2.end[2*m-1])[rest1]
				haplo3.end <- c(MID.1.strt[2*m],MID.2.strt[2*m],MID.1.end[2*m],MID.2.end[2*m])[rest1]
				haplo4.start <- c(MID.1.strt[2*m-1],MID.2.strt[2*m-1],MID.1.end[2*m-1],MID.2.end[2*m-1])[rest2]
				haplo4.end <- c(MID.1.strt[2*m],MID.2.strt[2*m],MID.1.end[2*m],MID.2.end[2*m])[rest2]
				MID.Haplo1.1 <- unlist(c(haplo1.start,haplo1.end))
				MID.Haplo2.1 <- unlist(c(haplo2.start,haplo2.end))
				MID.Haplo3.1 <- unlist(c(haplo3.start,haplo3.end))
				MID.Haplo4.1 <- unlist(c(haplo4.start,haplo4.end))

				if(tail(MID.Haplo1.1,1)<=tail(MID.Haplo2.1,1)) {
					MID.Haplo1 <- MID.Haplo1.1
					MID.Haplo3 <- MID.Haplo2.1}
				if(tail(MID.Haplo1.1,1)>tail(MID.Haplo2.1,1)) {
					MID.Haplo3 <- MID.Haplo1.1
					MID.Haplo1 <- MID.Haplo2.1}				
				if(tail(MID.Haplo3.1,1)<=tail(MID.Haplo4.1,1)) {
					MID.Haplo2 <- MID.Haplo3.1
					MID.Haplo4 <- MID.Haplo4.1}				
				if(tail(MID.Haplo3.1,1)>tail(MID.Haplo4.1,1)) {
					MID.Haplo4 <- MID.Haplo3.1
					MID.Haplo2 <- MID.Haplo4.1}				
				
			} else {
				haplo1.start <- c(MID.1.strt[m+1],MID.2.strt[m+1],MID.1.end[m+1],MID.2.end[m+1])[element1]
				haplo2.start <- c(MID.1.strt[m+1],MID.2.strt[m+1],MID.1.end[m+1],MID.2.end[m+1])[element2]
				haplo3.start <- c(MID.1.strt[m+1],MID.2.strt[m+1],MID.1.end[m+1],MID.2.end[m+1])[rest1]
				haplo4.start <- c(MID.1.strt[m+1],MID.2.strt[m+1],MID.1.end[m+1],MID.2.end[m+1])[rest2]
				HH <- sample(list(MID.Haplo1,MID.Haplo2),2,replace=FALSE)
				MID.Haplo1.1 <- unlist(c(HH[1],haplo1.start))
				MID.Haplo2.1 <- unlist(c(HH[2],haplo4.start))
				HH <- sample(list(MID.Haplo3,MID.Haplo4),2,replace=FALSE)
				MID.Haplo3.1 <- unlist(c(HH[1],haplo2.start))
				MID.Haplo4.1 <- unlist(c(HH[2],haplo3.start))
				
				if(tail(MID.Haplo1.1,1)<=tail(MID.Haplo2.1,1)) {
					MID.Haplo1 <- MID.Haplo1.1
					MID.Haplo3 <- MID.Haplo2.1}
				if(tail(MID.Haplo1.1,1)>tail(MID.Haplo2.1,1)) {
					MID.Haplo3 <- MID.Haplo1.1
					MID.Haplo1 <- MID.Haplo2.1}				
				if(tail(MID.Haplo3.1,1)<=tail(MID.Haplo4.1,1)) {
					MID.Haplo2 <- MID.Haplo3.1
					MID.Haplo4 <- MID.Haplo4.1}				
				if(tail(MID.Haplo3.1,1)>tail(MID.Haplo4.1,1)) {
					MID.Haplo4 <- MID.Haplo3.1
					MID.Haplo2 <- MID.Haplo4.1}
			}
			m <- m+1
		}
		
		# Select one of the four female haplotypes and one of the four male haplotypes for the individual
		Haplo.Ind.1 <- sample(c(1,2,3,4),1, replace=FALSE)
		Haplo.Ind.2 <- sample(c(1,2,3,4),1, replace=FALSE)
		F[[2*j]] <- if(Haplo.Ind.1==1) {FID.Haplo1} else if(Haplo.Ind.1==2) {FID.Haplo2} else if(Haplo.Ind.1==3) {FID.Haplo3} else if(Haplo.Ind.1==4) {FID.Haplo4}
		F[[2*j+1]] <- if(Haplo.Ind.2==1) {MID.Haplo1} else if(Haplo.Ind.2==2) {MID.Haplo2} else if(Haplo.Ind.2==3) {MID.Haplo3} else if(Haplo.Ind.2==4) {MID.Haplo4}	
		
		# Calculate the amount of autozygous stretches in individual
		xx <- ifelse(F[[2*j]]==F[[2*j+1]],1,0)
		change.pnts <- 0
		for(ii in 1:(length(xx)-1)) {
			if(abs(xx[ii+1]-xx[ii])==1) {
				change.pnts <- c(change.pnts,ii)
			}
		}
		change.pnts <- c(change.pnts,length(xx))
		cum.length <- 0
		# loop over all autozygous stretches
		for(jj in 1:(length(change.pnts)-1)) {
			strtch <- xx[(change.pnts[jj]+1):change.pnts[jj+1]]
			add <- 0
			# differentiate between whether the autozygous stretch starts at the first or last marker (then the stretch is shorter). Also choose the exact position between the two markers randomly.
			if(sum(strtch)!=0) {
				if(change.pnts[jj]+1!=1 & change.pnts[jj+1]!=length(xx)) {
					add <- prob$Pos.Phys.End[1]*(sum(strtch)-1)+sample(1:prob$Pos.Phys.End[1],1)+sample(1:prob$Pos.Phys.End[1],1) }
				if(change.pnts[jj]+1==1 & change.pnts[jj+1]!=length(xx)) {
					add <- prob$Pos.Phys.End[1]*(sum(strtch)-1)+sample(1:prob$Pos.Phys.End[1],1) }
				if(change.pnts[jj]+1!=1 & change.pnts[jj+1]==length(xx)) {
					add <- prob$Pos.Phys.End[1]*(sum(strtch)-1)+sample(1:prob$Pos.Phys.End[1],1) }
				if(change.pnts[jj]+1==1 & change.pnts[jj+1]==length(xx)) {
					add <- prob$Pos.Phys.End[1]*(sum(strtch)-1) }
			cum.length <- cum.length+add
			}
		}
		cum.length
		# store the cum.length per chromosome in cum.length.genome, which adds up all cum.length from each chromosome
		cum.length.genome <- cum.length.genome+cum.length
		all.length.genome <- all.length.genome+prob$Pos.Phys.End[1]*length(xMID.1)
		# ---------------------------------------------------------
		# use the n loci for estimating F
		if(as.character(unique(map$Chromosome)[j]) %in% as.character(loci100$Chromosome)) {
			LociChromo <- subset(loci100,Chromosome==as.character(unique(map$Chromosome)[j]))
			LociChromo <- strsplit(as.character(LociChromo$Marker1),".",fixed = TRUE)
			LociChromo <- as.numeric(unlist(lapply(LociChromo,"[[", sapply(LociChromo, length)[1])))
			yy <- ifelse(F[[2*j]][LociChromo]==F[[2*j+1]][LociChromo],1,0)
			# save the loci without false-negative and false-positive rate
			zz <- yy
			# introduce false-positives (depends on the homozygosity (IBS) of the marker) and no false-negatives (depends on the mutation rate of the marker)
			for(mm in 1:length(yy)) {
				if(yy[mm]==0) {yy[mm] <- sample(c(0,1),1,prob=c(1-((meanH-meanF)/(1-meanF)),((meanH-meanF)/(1-meanF))))}	# False-positive rate 
			}
			all.100L.error <- length(yy)
			all.100L.correct <- length(zz)
			cum.100L.error <- sum(yy)
			cum.100L.correct <- sum(zz)
			cum.100L.genome.error <- cum.100L.genome.error + cum.100L.error
			cum.100L.genome.correct <- cum.100L.genome.correct + cum.100L.correct
			all.100L.genome.error <- all.100L.genome.error + all.100L.error
			all.100L.genome.correct <- all.100L.genome.correct + all.100L.correct
		}
		# ---------------------------------------------------------
	}
	# calculate F.Emp, the empirical F of the individual and store it on the harddrive
	FEmp <- cum.length.genome/all.length.genome
	# ---------------------------------------------------------
	# Estimate F.100L (F.Emp from n loci with and without error)
	F.100L.error <- cum.100L.genome.error/all.100L.genome.error
	F.100L.correct <- cum.100L.genome.correct/all.100L.genome.correct
	# ---------------------------------------------------------
	
	# save the number of recombination events with the rest if requested
	if(save.recombs==TRUE) {
	out <- data.frame(nn,gnrtns$IndID[i],FEmp,F.100L.correct,F.100L.error,xxFID,xxMID)} else {
	out <- data.frame(nn,gnrtns$IndID[i],FEmp,F.100L.correct,F.100L.error)}
	write.table(out,paste(path, outfile, sep=""), append=TRUE, row.names=FALSE, col.names=FALSE, sep="\t", quote=FALSE, eol="\n")
# ------------------------------------------------------------------------------------------------------------------	
	# if the parents of the individual are not needed later on as a parent again remove them from INDS to speed up the simulation (beware that you do not delete the founders)
	fol.F <- which(ped$FID==gnrtns$FID[i])
	fol.M <- which(ped$MID==gnrtns$MID[i])
	if(all(fol.F<=(i+nrow(founders))) & !(gnrtns$FID[i] %in% founders$IndID)) {
		INDS[[which(lapply(INDS, "[[", 1) == as.character(gnrtns$FID[i]))]] <- NULL
	}
	if(all(fol.M<=(i+nrow(founders))) & !(gnrtns$MID[i] %in% founders$IndID)) {
		INDS[[which(lapply(INDS, "[[", 1) == as.character(gnrtns$MID[i]))]] <- NULL
	}	
	
	# only if the individual is needed later on as a parent save its haplotypes
	fol.FID <- which(ped$FID==gnrtns$IndID[i])
	fol.MID <- which(ped$MID==gnrtns$IndID[i])
	if(length(fol.FID)==0 & length(fol.MID)==0) {
	flush.console()
#	print(i)
	} else {
	INDS[[k]] <- F
	flush.console()
#	print(i)
	k <- k+1
	}
}	
print(paste("Taking ",round(Sys.time()-timeStart,digit=2)," minutes for simulation number ",nn," out of ",nSims,sep=""))
}

# ---------------------------------------------------------
}
# ---------------------------------------------------------

#####################################################################################################################################################################
### SUMMARY #########################################################################################################################################################
#####################################################################################################################################################################
if(smmry==TRUE) {
rm(list=ls())
library(checkpoint)
checkpoint("2014-10-08")
require(pedigreemm)
require(sqldf)
se <- function(x) sqrt(var(x)/length(x))
# define path to files here
path = "C:\\Users\\Data\\"
ped <- read.table(paste(path,"data_SIM.ped3404RAND.txt",sep=""), sep="\t", header=TRUE)
# get inbreeding coefficient of our pedigree
ped_mm <- pedigree(sire = ped$MID, dam  = ped$FID, label= ped$IndID)
FPed <- inbreeding(ped_mm)
ped <- cbind(ped,FPed)
# summarize simulation
file_list <- list.files(paste(path,"nLoci\\",sep=""))
mLoci <- lapply(strsplit(file_list, "_", fixed = TRUE),"[[",3)
mLoci <- lapply(strsplit(as.character(mLoci), "L", fixed = TRUE),"[[",2)
mLoci <- unlist(mLoci)
mLoci <- as.numeric(mLoci)

for (ll in 1:length(file_list)) {
	# if the merged dataset doesn't exist, create it
	if (!exists("out.sim")) {
		out.sim <- read.table(paste(path,"nLoci\\",file_list[ll],sep=""), header=TRUE, sep="\t")
		out.sim$nLoci <- mLoci[ll]
		out.sim$Sim.run <- out.sim$Sim.run + 100
	}
	# if the merged dataset does exist, append to it
	else if (exists("out.sim")) {
		temp_out <- read.table(paste(path,"nLoci\\",file_list[ll],sep=""), header=TRUE, sep="\t")
		temp_out$nLoci <- mLoci[ll]
		if(mLoci[ll]==mLoci[ll-1]) { temp_out$Sim.run <- temp_out$Sim.run + out.sim[nrow(out.sim),"Sim.run"] } else { temp_out$Sim.run <- temp_out$Sim.run + 100 }
		out.sim <- rbind(out.sim, temp_out)
		rm(temp_out)
	}
}

# Create new Sim.run variable that is unique for each simulation run
L <- nrow(subset(ped,FID!=0 & MID!=0))
k <- 1
n <- 1
m <- 1
p <- 1
r <- 1
s <- 1
for(i in 1:(nrow(out.sim)/L)) {
	if(all(out.sim[(i*L-L+1):(i*L),"nLoci"]==5)) {
		out.sim[(i*L-L+1):(i*L),1] <- rep(k,L)
		k <- k+1
	}
	if(all(out.sim[(i*L-L+1):(i*L),"nLoci"]==10)) {
		out.sim[(i*L-L+1):(i*L),1] <- rep(n,L)
		n <- n+1
	}	
	if(all(out.sim[(i*L-L+1):(i*L),"nLoci"]==20)) {
		out.sim[(i*L-L+1):(i*L),1] <- rep(m,L)
		m <- m+1
	}		
	if(all(out.sim[(i*L-L+1):(i*L),"nLoci"]==40)) {
		out.sim[(i*L-L+1):(i*L),1] <- rep(p,L)
		p <- p+1
	}		
	if(all(out.sim[(i*L-L+1):(i*L),"nLoci"]==80)) {
		out.sim[(i*L-L+1):(i*L),1] <- rep(r,L)
		r <- r+1
	}		
	if(all(out.sim[(i*L-L+1):(i*L),"nLoci"]==160)) {
		out.sim[(i*L-L+1):(i*L),1] <- rep(s,L)
		s <- s+1
	}
	if(length(table(out.sim[(i*L-L+1):(i*L),"nLoci"]))>1) { stop }
	flush.console()
	if(i %% 100 == 0) {print(i)}
}

out.sim <- sqldf("select * from 'out.sim' left join 'ped' using (IndID)")
out.sim$generation <- ifelse(out.sim$IndID<4000,1,ifelse(out.sim$IndID<5000,2,ifelse(out.sim$IndID<7000,3,ifelse(out.sim$IndID<9000,4,ifelse(out.sim$IndID<11000,5,ifelse(out.sim$IndID<12000,6,7))))))

# Save all simulation data on hard drive
write.table(out.sim,paste(path, "nLoci\\+++AllData.humans.Inbreeding.nLoci.simRAND100kb.txt", sep=""), append=FALSE, row.names=FALSE, col.names=TRUE, sep="\t", quote=FALSE, eol="\n")

####
####
####


rm(list=ls())
library(checkpoint)
checkpoint("2014-10-08")
require(pedigreemm)
require(sqldf)
se <- function(x) sqrt(var(x)/length(x))
# define path to files here
path = "C:\\Users\\Data\\"
ped <- read.table(paste(path,"data_SIM.ped3404RAND.txt",sep=""), sep="\t", header=TRUE)
# get inbreeding coefficient of our pedigree
ped_mm <- pedigree(sire = ped$MID, dam  = ped$FID, label= ped$IndID)
FPed <- inbreeding(ped_mm)
ped <- cbind(ped,FPed)

pedF1 <- subset(ped, IndID<13000)
pedF2 <- subset(ped, IndID>=5000 & IndID<13000)
pedF3 <- subset(ped, IndID>=7000 & IndID<13000)
pedF1$FID <- ifelse(pedF1$FID>13000,0,pedF1$FID)
pedF1$MID <- ifelse(pedF1$MID>13000,0,pedF1$MID)
pedF2$FID <- ifelse(pedF2$FID<5000,0,pedF2$FID)
pedF2$MID <- ifelse(pedF2$MID<5000,0,pedF2$MID)
pedF3$FID <- ifelse(pedF3$FID<7000,0,pedF3$FID)
pedF3$MID <- ifelse(pedF3$MID<7000,0,pedF3$MID)
ped_mm_F1 <- pedigree(sire = pedF1$MID, dam  = pedF1$FID, label= pedF1$IndID)
FPed_F1 <- inbreeding(ped_mm_F1)
pedF1 <- cbind(pedF1,FPed_F1)
pedF1 <- pedF1[,c("IndID","FPed_F1")]
ped_mm_F2 <- pedigree(sire = pedF2$MID, dam  = pedF2$FID, label= pedF2$IndID)
FPed_F2 <- inbreeding(ped_mm_F2)
pedF2 <- cbind(pedF2,FPed_F2)
pedF2 <- pedF2[,c("IndID","FPed_F2")]
ped_mm_F3 <- pedigree(sire = pedF3$MID, dam  = pedF3$FID, label= pedF3$IndID)
FPed_F3 <- inbreeding(ped_mm_F3)
pedF3 <- cbind(pedF3,FPed_F3)
pedF3 <- pedF3[,c("IndID","FPed_F3")]

out.sim <- read.table(paste(path,"nLoci\\+++AllData.humans.Inbreeding.nLoci.simRAND100kb.txt",sep=""), sep="\t", header=TRUE)
out.sim <- merge(out.sim,pedF2,by.x="IndID",by.y="IndID",sort=FALSE)
out.sim <- merge(out.sim,pedF3,by.x="IndID",by.y="IndID",sort=FALSE)
out.sim <- subset(out.sim,Sim_run<=1000)
mLoci <- unique(out.sim$nLoci)
summary.out <- data.frame(matrix(c(NA,NA,NA,NA,NA,NA,NA,NA,NA,NA,NA,NA,NA,NA,NA,NA,NA),nrow=1))
colnames(summary.out) <- c("nLoci","SlopePed","SlopePedF2","SlopePedF3","SlopeLociCorrect","SlopeLociError","SlopeLociErrorCorrect","DeterminationPed","DeterminationPedF2","DeterminationPedF3","DeterminationLociCorrect","DeterminationLociError","DeterminationLociErrorCorrect")
kk <- 1

for(mm in 1:length(unique(mLoci))) {
	ToDo.nLoci <- subset(out.sim,nLoci==unique(mLoci)[mm])
	for(nn in 1:length(unique(out.sim$Sim_run))) {
		ToDo.run <- subset(ToDo.nLoci,Sim_run==nn & generation==7)
		cor.Ped.Emp <- cor(ToDo.run$F_Emp,ToDo.run$FPed)
		cor.PedF2.Emp <- cor(ToDo.run$F_Emp,ToDo.run$FPed_F2)
		cor.PedF3.Emp <- cor(ToDo.run$F_Emp,ToDo.run$FPed_F3)
		cor.Emp.100L.correct <- cor(ToDo.run$F_Emp,ToDo.run$F_100L_correct)
		cor.Emp.100L.error <- cor(ToDo.run$F_Emp,ToDo.run$F_100L_error)
		cor.correct.100L.error <- cor(ToDo.run$F_100L_correct,ToDo.run$F_100L_error)
		slope.Ped.Emp <- coef(lm(ToDo.run$F_Emp~ToDo.run$FPed))[2]
		slope.PedF2.Emp <- coef(lm(ToDo.run$F_Emp~ToDo.run$FPed_F2))[2]
		slope.PedF3.Emp <- coef(lm(ToDo.run$F_Emp~ToDo.run$FPed_F3))[2]
		slope.Emp.100L.correct <- coef(lm(ToDo.run$F_Emp~ToDo.run$F_100L_correct))[2]
		slope.Emp.100L.error <- coef(lm(ToDo.run$F_Emp~ToDo.run$F_100L_error))[2]
		slope.correct.100L.error <- coef(lm(ToDo.run$F_100L_error~ToDo.run$F_100L_correct))[2]
		summary.out[kk,"nLoci"] <- unique(mLoci)[mm]
		summary.out[kk,"SlopePed"] <- slope.Ped.Emp
		summary.out[kk,"SlopePedF2"] <- slope.PedF2.Emp
		summary.out[kk,"SlopePedF3"] <- slope.PedF3.Emp
		summary.out[kk,"SlopeLociCorrect"] <- slope.Emp.100L.correct
		summary.out[kk,"SlopeLociError"] <- slope.Emp.100L.error
		summary.out[kk,"SlopeLociErrorCorrect"] <- slope.correct.100L.error
		summary.out[kk,"DeterminationPed"] <- cor.Ped.Emp^2
		summary.out[kk,"DeterminationPedF2"] <- cor.PedF2.Emp^2
		summary.out[kk,"DeterminationPedF3"] <- cor.PedF3.Emp^2
		summary.out[kk,"DeterminationLociCorrect"] <- cor.Emp.100L.correct^2
		summary.out[kk,"DeterminationLociError"] <- cor.Emp.100L.error^2
		summary.out[kk,"DeterminationLociErrorCorrect"] <- cor.correct.100L.error^2
		kk <- kk + 1
	}
}
mean(summary.out$DeterminationPed)
se(summary.out$DeterminationPed)

out <- data.frame(matrix(rep(NA,49),nrow=1))
colnames(out) <- c("nLoci","SlopePed","SlopePedF2","SlopePedF3","SlopeLociCorrect","SlopeLociError","SlopeLociErrorCorrect","r2Ped","r2PedF2","r2PedF3","r2LociCorrect","r2LociError","r2LociErrorCorrect","SESlopePed","SESlopePedF2","SESlopePedF3","SESlopeLociCorrect","SESlopeLociError","SESlopeLociErrorCorrect","SEr2Ped","SEr2PedF2","SEr2PedF3","SEr2LociCorrect","SEr2LociError","SEr2LociErrorCorrect","QRSlopePed_Low","QRSlopePedF2_Low","QRSlopePedF3_Low","QRSlopeLociCorrect_Low","QRSlopeLociError_Low","QRSlopeLociErrorCorrect_Low","QRr2Ped_Low","QRr2PedF2_Low","QRr2PedF3_Low","QRr2LociCorrect_Low","QRr2LociError_Low","QRr2LociErrorCorrect_Low","QRSlopePed_Up","QRSlopePedF2_Up","QRSlopePedF3_Up","QRSlopeLociCorrect_Up","QRSlopeLociError_Up","QRSlopeLociErrorCorrect_Up","QRr2Ped_Up","QRr2PedF2_Up","QRr2PedF3_Up","QRr2LociCorrect_Up","QRr2LociError_Up","QRr2LociErrorCorrect_Up")

for(mm in 1:length(unique(mLoci))) {
	ToDo <- subset(summary.out,nLoci==unique(mLoci)[mm])
	out[mm,"nLoci"] <- unique(mLoci)[mm]
	out[mm,"SlopePed"] <- mean(ToDo$SlopePed)
	out[mm,"SlopePedF2"] <- mean(ToDo$SlopePedF2)
	out[mm,"SlopePedF3"] <- mean(ToDo$SlopePedF3)
	out[mm,"SlopeLociCorrect"] <- mean(ToDo$SlopeLociCorrect)
	out[mm,"SlopeLociError"] <- mean(ToDo$SlopeLociError)
	out[mm,"SlopeLociErrorCorrect"] <- mean(ToDo$SlopeLociErrorCorrect)
	out[mm,"r2Ped"] <- 	mean(ToDo$DeterminationPed)
	out[mm,"r2PedF2"] <- 	mean(ToDo$DeterminationPedF2)
	out[mm,"r2PedF3"] <- 	mean(ToDo$DeterminationPedF3)
	out[mm,"r2LociCorrect"] <- mean(ToDo$DeterminationLociCorrect)
	out[mm,"r2LociError"] <- mean(ToDo$DeterminationLociError)
	out[mm,"r2LociErrorCorrect"] <- mean(ToDo$DeterminationLociErrorCorrect)
	out[mm,"SESlopePed"] <- sd(ToDo$SlopePed)/(sqrt(length(ToDo)))
	out[mm,"SESlopePedF2"] <- sd(ToDo$SlopePedF2)/(sqrt(length(ToDo)))
	out[mm,"SESlopePedF3"] <- sd(ToDo$SlopePedF3)/(sqrt(length(ToDo)))
	out[mm,"SESlopeLociCorrect"] <- sd(ToDo$SlopeLociCorrect)/(sqrt(length(ToDo)))
	out[mm,"SESlopeLociError"] <- sd(ToDo$SlopeLociError)/(sqrt(length(ToDo)))
	out[mm,"SESlopeLociErrorCorrect"] <- sd(ToDo$SlopeLociErrorCorrect)/(sqrt(length(ToDo)))
	out[mm,"SEr2Ped"] <- sd(ToDo$DeterminationPed)/(sqrt(length(ToDo)))
	out[mm,"SEr2PedF2"] <- sd(ToDo$DeterminationPedF2)/(sqrt(length(ToDo)))
	out[mm,"SEr2PedF3"] <- sd(ToDo$DeterminationPedF3)/(sqrt(length(ToDo)))
	out[mm,"SEr2LociCorrect"] <- sd(ToDo$DeterminationLociCorrect)/(sqrt(length(ToDo)))
	out[mm,"SEr2LociError"] <- sd(ToDo$DeterminationLociError)/(sqrt(length(ToDo)))
	out[mm,"SEr2LociErrorCorrect"] <- sd(ToDo$DeterminationLociErrorCorrect)/(sqrt(length(ToDo)))
	out[mm,"QRSlopePed_Low"] <- quantile(ToDo$SlopePed, c(0.025,0.975))[1]
	out[mm,"QRSlopePedF2_Low"] <- quantile(ToDo$SlopePedF2, c(0.025,0.975))[1]
	out[mm,"QRSlopePedF3_Low"] <- quantile(ToDo$SlopePedF3, c(0.025,0.975))[1]
	out[mm,"QRSlopeLociCorrect_Low"] <- quantile(ToDo$SlopeLociCorrect, c(0.025,0.975))[1]
	out[mm,"QRSlopeLociError_Low"] <- quantile(ToDo$SlopeLociError, c(0.025,0.975))[1]
	out[mm,"QRSlopeLociErrorCorrect_Low"] <- quantile(ToDo$SlopeLociErrorCorrect, c(0.025,0.975))[1]
	out[mm,"QRr2Ped_Low"] <- quantile(ToDo$DeterminationPed, c(0.025,0.975))[1]
	out[mm,"QRr2PedF2_Low"] <- quantile(ToDo$DeterminationPedF2, c(0.025,0.975))[1]
	out[mm,"QRr2PedF3_Low"] <- quantile(ToDo$DeterminationPedF3, c(0.025,0.975))[1]
	out[mm,"QRr2LociCorrect_Low"] <- quantile(ToDo$DeterminationLociCorrect, c(0.025,0.975))[1]
	out[mm,"QRr2LociError_Low"] <- quantile(ToDo$DeterminationLociError, c(0.025,0.975))[1]
	out[mm,"QRr2LociErrorCorrect_Low"] <- quantile(ToDo$DeterminationLociErrorCorrect, c(0.025,0.975))[1]	
	out[mm,"QRSlopePed_Up"] <- quantile(ToDo$SlopePed, c(0.025,0.975))[2]
	out[mm,"QRSlopePedF2_Up"] <- quantile(ToDo$SlopePedF2, c(0.025,0.975))[2]
	out[mm,"QRSlopePedF3_Up"] <- quantile(ToDo$SlopePedF3, c(0.025,0.975))[2]
	out[mm,"QRSlopeLociCorrect_Up"] <- quantile(ToDo$SlopeLociCorrect, c(0.025,0.975))[2]
	out[mm,"QRSlopeLociError_Up"] <- quantile(ToDo$SlopeLociError, c(0.025,0.975))[2]
	out[mm,"QRSlopeLociErrorCorrect_Up"] <- quantile(ToDo$SlopeLociErrorCorrect, c(0.025,0.975))[2]
	out[mm,"QRr2Ped_Up"] <- quantile(ToDo$DeterminationPed, c(0.025,0.975))[2]
	out[mm,"QRr2PedF2_Up"] <- quantile(ToDo$DeterminationPedF2, c(0.025,0.975))[2]
	out[mm,"QRr2PedF3_Up"] <- quantile(ToDo$DeterminationPedF3, c(0.025,0.975))[2]
	out[mm,"QRr2LociCorrect_Up"] <- quantile(ToDo$DeterminationLociCorrect, c(0.025,0.975))[2]
	out[mm,"QRr2LociError_Up"] <- quantile(ToDo$DeterminationLociError, c(0.025,0.975))[2]
	out[mm,"QRr2LociErrorCorrect_Up"] <- quantile(ToDo$DeterminationLociErrorCorrect, c(0.025,0.975))[2]
}	

write.table(out,paste(path,"+++summary.n1000.humans.Inbreeding.nLoci.simRAND100kb.txt", sep=""), append=FALSE, row.names=FALSE, col.names=TRUE, sep="\t", quote=FALSE, eol="\n")
plot(out$nLoci,out$r2Ped,ylim=c(0,1))
points(out$nLoci,out$r2LociCorrect, col="red")
points(out$nLoci,out$r2LociError, col="blue")
}
#####################################################################################################################################################################
### END #############################################################################################################################################################
#####################################################################################################################################################################


