# Working with DataFrames

Both `SimpleSDMLayers.jl` and `GBIF.jl` offer an optional integration with the
`DataFrames.jl` package. Therefore, our [previous
example](https://ecojulia.github.io/SimpleSDMLayers.jl/latest/examples/gbif/)
with the kingfisher *Megaceryle alcyon* could also be approached with a
`DataFrame`-centered workflow.

We will illustrate this using the same data and producing the same figures as in
the previous example. To do so, we will use `GBIF.jl` to produce the occurrence
`DataFrame` we will use throughout this example. However, it is also possible to
use a `DataFrame` of your choosing instead of one generated by `GBIF.jl`, as
long as it holds one occurrence per row, a column with the latitude coordinates,
and a column with longitude coordinates. For the rest, it can hold whatever
information you like. Most of our functions assume by default that the
coordinates are stored in columns named `:latitude` and `:longitude` (the order
doesn't matter), but you can generally specify other names with `latitude =
:lat` in case you don't want to rename them (we will show you how below).

So let's start by getting our data:

```@example dataframes
# Load packages
using SimpleSDMLayers
using GBIF
using Plots
using Statistics
# Load DataFrames too
using DataFrames

# Load environmental data
temperature, precipitation = SimpleSDMPredictor(WorldClim, BioClim, [1,12])

# Get GBIF occurrences
kingfisher = GBIF.taxon("Megaceryle alcyon", strict=true)
kf_occurrences = occurrences(kingfisher, 
                             "hasCoordinate" => "true", 
                             "decimalLatitude" => (0.0, 65.0), 
                             "decimalLongitude" => (-180.0, -50.0), 
                             "limit" => 200)
for i in 1:4
  occurrences!(kf_occurrences)
end
@info kf_occurrences

```

Once the data is loaded, we can easily convert the environmental layers to a
`DataFrame` with the corresponding coordinates. We can do this for a single
layer or for multiple layers at the same time:

```@example dataframes
# Single layer
temperature_df = DataFrame(temperature)
# Multiple layers
env_layers = [temperature, precipitation]
env_df = DataFrame(env_layers)
rename!(env_df, :x1 => :temperature, :x2 => :precipitation)
first(env_df, 5)
```

Note that the resulting `DataFrame` will include `missing` values for the 
elements set to `nothing` in the layers. We might want to remove those rows 
using `filter!` or `dropmissing!`:

```@example dataframes
dropmissing!(env_df, [:temperature, :precipitation]);
last(env_df, 5)
```

`GBIF.jl` allows us to convert a set of occurrences to a `DataFrame` just as
easily:

```@example dataframes
kf_df = DataFrame(kf_occurrences)
last(kf_df, 5)
```

We can then extract the temperature values for all the occurrences.

```@example dataframes
temperature[kf_df]
```

Or we can clip the layers according to the occurrences:

```@example dataframes
temperature_clip = clip(temperature, kf_df)
precipitation_clip = clip(precipitation, kf_df)
```

In case your `DataFrame` has different column names for the coordinates, for
example `:lat` and `:lon`, you can clip it like this:

```@example dataframes
kf_df_shortnames = rename(kf_df, :latitude => :lat, :longitude => :lon)
clip(temperature, kf_df_shortnames; latitude = :lat, longitude = :lon)
```

We can finally plot the layer and occurrence values in a similar way to any
`DataFrame` or `Array`.

```@example dataframes
histogram2d(temperature_clip, precipitation_clip, c = :viridis)
scatter!(temperature_clip[kf_df], precipitation_clip[kf_df], 
         lab= "", c = :white, msc = :orange)
```

To plot the occurrence values over space, you can use:

```@example dataframes
contour(temperature_clip, c = :alpine, title = "Temperature", 
        frame = :box, fill = true)
scatter!(kf_df.longitude, kf_df.latitude, 
         lab = "", c = :white, msc = :orange, ms = 2)
```

We can finally make a layer with the number of observations per cells:

```@example dataframes
abundance = mask(precipitation_clip, kf_occurrences, Float32)
```

A useful trick to visualize sites with occurrences, in contrast with sites
without any occurrence, is to use `replace` or `replace!` to set the values
returned as `0` or `true` by the function `mask()` to `nothing`. This allows us
to first plot a background layer with a uniform colour, covering the whole area
to visualize, then plot the occurrence layer on top using a different colour
scale.

```@example dataframes
abundance_nozeros = replace(abundance, 0 => nothing)
plot(precipitation_clip, c = :lightgrey)
plot!(abundance_nozeros, c = :viridis, clim = extrema(abundance_nozeros))
```

Once again, the cells are rather small, and there are few observations, so this
is not necessarily going to be very informative. As in our other example, to
get a better sense of the distribution of observations, we can get the average
number of observations in a radius of 100km around each cell (we will do so for
a zoomed-in part of the map to save time):

```@example dataframes
zoom = abundance[left = -100.0, right = -75.0, top = 43.0, bottom = 20.0]
buffered = slidingwindow(zoom, Statistics.mean, 100.0)
plot(buffered, c = :lapaz, legend = false, frame = :box)
scatter!(kf_df.longitude, kf_df.latitude, 
         lab = "", c = :white, msc = :orange, ms = 2, alpha = 0.5)
```