#!/usr/bin/env perl

# Rene L Warren
# Interactive Genome Mutation Timemaps
# DEC2020/JAN2021
# GPLv3

use strict;

my $op = 0.75;###sets opacity
my $year=2020;
my ($start,$end)=(1,29903);
my $buffer = 725;#(when spike) #4300(genome); ### this controls how much space before axis begins in centre
my $min = 10;    ### minimum number of distinct viral genome with the variant daily
my $mindaily = 2;
my $allmut=1;

if($#ARGV<6){
   print "Usage: $0\n";
   print "\t<variation.txt (required)>\n";
   print "\t<variant effect .tsv (required)>\n";
   print "\t<annotation .gff (required)>\n";
   print "\t<\"continent name\", or NA for all (required)>\n";
   print "\t<nucleotide/a.a. variant(s) (e.g. \"C22227T D614G\"), or NA for none (required)>\n";
   print "\t<scaling factor (required)>\n";
   print "\t<all / Missense variants 1,2 / 0 (1=byJurisdiction,2=byType default=$allmut)>\n";
   print "\t<opacity 0-1 (more to less transparent, default=$op)>\n";
   print "\t<year - optional(default=$year)>\n";
   print "\t<basename - optional>\n";
   print "\t<buffer (axis start from centre - optional, default=$buffer pixels)>\n";
   print "\t<min. total genome support - optional, default=$min>\n";
   print "\t<min. daily genome support - optional, default=$mindaily>\n";
   print "\t<plot data as % (1=yes, 0=no/default - optional)>\n";
   print "\t<genome start coordinate (optional, default=$start)>\n";
   die "\t<genome end coordinate (optional, default=$end)>\n";
}

my $factor = 2.8; ### multiplicative factor, controls the size of variant dot
my $factorprop = 2.8;
my $fop = 0.085;#0.1;
my $continent = "";
my $plotprop = 0;

my $f = $ARGV[0];
my $out = $f . ".svg"; ## this will be the default xml svg output if no "basename" provided
my $vareff = $ARGV[1];
my $gff = $ARGV[2];
my $continent = $ARGV[3];
$continent = ".*" if($continent eq "NA");
my $varlist = $ARGV[4];
$varlist = "NA" if($varlist eq "NA");
my $scale = $ARGV[5];
$allmut = $ARGV[6] if($ARGV[6] ne "");
$op = $ARGV[7] if($ARGV[7] ne "");
$year = $ARGV[8] if($ARGV[8] ne "");
$out = $ARGV[9] . ".svg" if($ARGV[9] ne "");
$buffer = $ARGV[10] if($ARGV[10] ne "");
my $lowbuffer = 5;
my $stackfactor = 4.5;
$stackfactor=8 if($buffer>=4000);

$min = $ARGV[11] if($ARGV[11] ne "");
$mindaily = $ARGV[12] if($ARGV[12] ne "");
$plotprop = $ARGV[13] if($ARGV[13] ne "");
$start = $ARGV[14] if($ARGV[14] ne "");
$end = $ARGV[15] if($ARGV[15] ne "");


#controls the number of labels that will appear on gganimate plots
my $minthreshold = 25;
$minthreshold = 50 if($allmut);
$minthreshold = 50 if($start==1 && $end==29903);

### setting variant type message
my $mutationtype="total distinct variants mapped";
if(! $allmut){
   if($varlist eq "NA"){
      $mutationtype="distinct missense variants mapped";
   }else{
      $mutationtype="distinct custom variants mapped";
   }
}else{### all types
   if($varlist ne "NA"){
      $mutationtype="distinct custom variants mapped";
   }
}

### pre-computed lookup tsv files
my $country = "country.tsv";
my $d = "daynumber$year.tsv";

my $conv;
$conv->{'january'} = "01";
$conv->{'february'} = "02";
$conv->{'march'} = "03";
$conv->{'april'} = "04";
$conv->{'may'} = "05";
$conv->{'june'} = "06";
$conv->{'july'} = "07";
$conv->{'august'} = "08";
$conv->{'september'} = "09";
$conv->{'october'} = "10";
$conv->{'november'} = "11";
$conv->{'december'} = "12";

#### read date to degree
my $deg;
my $dateconversion;
open(IN,$d) || die "Can't read $d -- fatal.";

### Adjustments to fit a calendar year into a circle
my $days=365;
if($year % 4 == 0){
   $days=366;
}
my $yeardegfactor = 360 / $days;

while(<IN>){
   chomp;
   my @a =split(/\t/);
   my @b = split(/\s/,$a[1]);
   if($b[1]<10){$b[1]="0".$b[1];}
   my $dateadj = "$year-" . $conv->{lc($b[0])} . "-" . $b[1];
   #print ":$dateadj:$a[0]:\n";
   my $degree = $a[0] * $yeardegfactor;### make the degree adjustments
   $degree = sprintf "%.2f" , $degree;
   $deg->{$dateadj} = $degree;
   $dateconversion->{$degree} = $dateadj;
}
close IN;

#### read in gff
my $genes;
my $circcol = "grey";
my $circ;
open(IN,$gff) || die "Can't read gff -- fatal\n";
while(<IN>){
   chomp;
   #MN908947.3      Genbank gene    266     21555   .       +       .       ID=gene-orf1ab;Name=orf1ab;gbkey=Gene;gene=orf1ab;gene_biotype=protein_coding
   my @a=split(/\t/);
   my $gene= $1 if($_=~/gene\=([^\;]*)/);
   if($a[2] eq "gene" && (($a[3]>=$start && $a[3]<=$end) || ($a[4]>=$start && $a[4]<=$end))){
      $a[3] = $start if($a[3]<$start);
      $a[4] = $end if($a[4]>$end);
      $genes->{$a[3]}{'end'}=$a[4];
      $genes->{$a[3]}{'name'}=$gene;
      #print "$a[3] - $a[4] $gene\n";

      $circ->{$a[4]}=$circcol;
      $circ->{$a[3]}="black";
      if($circcol eq "gainsboro"){ $circcol="grey"; }else{ $circcol="gainsboro"; }
   }
}#YYY
close IN;

### variant effect map
my $vareffmap;
open(IN,$vareff) || die "Can't read $vareff -- fatal.";
while(<IN>){
   chomp;
   #variant change  genome_sample_count     +%chg   gene    product init_date       init_id init_region
   #C3037T  silent  288804  7.1     orf1ab  orf1ab polyprotein      2020-01-24      EPI_ISL_422425  Zhejiang

   my @a = split(/\t/);
   if($a[2]>=$min){
      $vareffmap->{$a[0]}{'aa'} = $a[1];
      $vareffmap->{$a[0]}{'product'} = $a[5];
   }
}
close IN;

### read country
my $chash;
open(IN,$country) || die "Can't open $country for reading -- fatal.\n";
while(<IN>){
   chomp;
   my @a=split(/\t/);
   $chash->{lc($a[1])}=$a[6];
}
close IN;
#### read variation
my $track;
my $bt;
my $ctvir=0;
open(IN,$f);
my $reglist;
my $muthash;
my $totdailyreglist;### tracks total daily samples by jurisdictions
while(<IN>){
   chomp;
#hCoV-19/Wales/PHWC-16668D/2020|EPI_ISL_494028|2020-06-27;C257T,C3053T,C6516A,A13969G,C14424T,C21258T,A23419G,G26466T,C29293G,
   my @a=split(/\;/);##split ancillary ; mutations
   my @b=split(/\|/,$a[0]); ##split local | id | date
   my @c=split(/\//,$b[0]); ## further split local
   my $degree = $deg->{$b[2]};

   my @var = split(/\,/,$a[1]);

   my $tmpregion=$c[1];#default is to assign a country
   $tmpregion = "French Guiana" if($tmpregion eq "Guyane Francaise");
   $tmpregion = "Puerto Rico" if($tmpregion eq "Puerto_Rico");
   $tmpregion = "South Africa" if($tmpregion eq "SouthAfrica");
   $tmpregion = "Saudi Arabia" if($tmpregion eq "SaudiArabia");
   $tmpregion = "Cote d\'Ivoire" if($tmpregion eq "Cote dIvoire");
   $tmpregion = "DRC" if($tmpregion eq "Congo");
   $tmpregion = "Hong Kong" if($tmpregion eq "HongKong");
   $tmpregion = "Brazil" if($tmpregion eq "Brasil");
   $tmpregion=$chash->{lc($c[1])} if($continent eq ".*");


   ##this is to troubleshoot undefined regions
   if ($chash->{lc($c[1])} eq ""){
      print "$c[1]\tx\tx\tx\tx\t\n";
   }

   ### tracking all by date
   if(! defined $bt->{$b[1]} && $degree ne "" && $c[1] ne "" && $chash->{lc($c[1])} ne "" && $chash->{lc($c[1])}=~/$continent/i && $degree<=360){###only if ID is new
      #print "$c[1] ($chash->{lc($c[1])})\t$b[2] ($degree)\t@var\n"; 

      my $el = 0;
      my $flag = 0;

      VARBASE:
      foreach my $varbase (@var){

         my $next = $el + 1;
         ### EXCEPTIONS XXX

         if($flag){### will skip if the previous varbase was special case
            $flag=0;
            next VARBASE;
         }

         if($varbase eq "AGGATG23400A" && $var[$next] eq "T23401TAGGTG"){
            $varbase = "GGA23401AGG";
            $flag=1;
         }elsif($varbase eq "T21570TGTTTT"){### && $var[$next] eq "TTTCTT21572T"){
            $varbase = "T21570G";### "TTTTTC21570GTTTTT";
            $flag=0;
         }elsif($varbase eq "TTTCTT21572T"){
            $varbase = "C21575T";
            $flag=0;
         }elsif($varbase eq "AGGGGA28880A" && $var[$next] eq "A28881AAACGA"){
            $varbase = "GGG28881AAC";
            $flag=1;
         }elsif($varbase eq "A21550ACTCTA" && $var[$next] eq "CTAAAC21552C"){
            $varbase = "AA21550CT";
            $flag=1;
         }elsif(($varbase eq "G25563GTAAC" && $var[$next] eq "GAGCG25563G") || ($varbase eq "GAGCG25563G" && $var[$next] eq "G25563GTAAC") ){
            $varbase = "GAG25563TAA";
            $flag=1;
         }elsif($varbase eq "A23076AGGAA" && $var[$next] eq "AACCA23079A"){
            $varbase = "ACC23076GGA";
            $flag=1;
         }else{
            $flag=0;
         }
         ### first criteria checks for var existence with certain genome depth AND either all mutation types (allmut 1/2+anything), or only missense (allmut==0 and not silent/not NA)
         if(defined $vareffmap->{$varbase} && ($allmut || ($vareffmap->{$varbase}{'aa'} ne "silent" && $vareffmap->{$varbase}{'aa'} ne "NA"))){### either plot all mutations or missense/stops
      
            $tmpregion = "missense" if($allmut==2 && $vareffmap->{$varbase}{'aa'} ne "silent");
            $tmpregion = "silent" if($allmut==2 && $vareffmap->{$varbase}{'aa'} eq "silent");
            $reglist->{$tmpregion}{'ct'}++;###track most abundant regions terms of mutations
            $track->{$degree}{$varbase}{$tmpregion}{'sum'}++; ### track by degree, mut, country,count
            $track->{$degree}{$varbase}{$tmpregion}{'date'} = $b[2];
            $muthash->{$degree}{$varbase}=$1 if($varbase=~/\D+(\d+)\D+/);
            ### IN WRONG PLACE, SEE BELOW $totdailyreglist->{$degree}{$tmpregion}{$b[1]}++;
         }#missense filters
         $el++;
      }#cycle through mutations for ID
      $ctvir++;### will only count unique non-redundant genome IDs
      $totdailyreglist->{$degree}{$tmpregion}{$b[1]}++; ### tracks number of samples/regions/day
      if($allmut==2){
         $totdailyreglist->{$degree}{"silent"}{$b[1]}++;
         $totdailyreglist->{$degree}{"missense"}{$b[1]}++;
      }
   }#basic data health/quality filters 
   $bt->{$b[1]}=1;###seen the ID, degree less than 360 and country/continent recognized
}
close IN;

## IF RATIO CHOSEN
my ($fhash,$shash)=("sum","prop");
if($plotprop){### when asked to plot as %. Important to arrange circles by prop, largest first
   ($fhash,$shash)=("prop","sum");
}

foreach my $deg1 (keys %$track){
   my $mutlist1 = $track->{$deg1};
   foreach my $mut1 (keys %$mutlist1){
      my $jurlist1 = $mutlist1->{$mut1};
      foreach my $jur1 (keys %$jurlist1){
         my $tmplist = $totdailyreglist->{$deg1}{$jur1};
         my $numbersample = keys(%$tmplist);
         my $propmut = ($track->{$deg1}{$mut1}{$jur1}{'sum'} / $numbersample) * 100;
         $track->{$deg1}{$mut1}{$jur1}{'prop'} = $propmut;
      }
   }
}
##

while($ctvir =~ s/(\d+)(\d\d\d)/$1\,$2/){};### will add thousand separator to virus count

###prepare the country/continent for color assignments
my $colelement=0;
foreach my $country (sort {$reglist->{$b}{'ct'}<=>$reglist->{$a}{'ct'}} keys %$reglist){
   my $corrcountry = $country;
   $corrcountry="Dominic. Rep." if($country=~/Dominican/);
   $corrcountry="Czech Rep." if($country=~/Czech Republic/);
   $corrcountry="N. Ireland" if($country=~/Northern Ireland/);
   $corrcountry="UAE" if($country=~/United Arab/);
   $corrcountry="Equat. Guinea" if($country=~/Equatorial/);
   $corrcountry="St. Barthelemy" if($country=~/Saint Barthelemy/);

   $reglist->{$country}{'el'}=$colelement;
   $reglist->{$country}{'correctedname'}=$corrcountry;

   #print "$country .. $reglist->{$country}{'ct'}..$reglist->{$country}{'el'}\n";
   $colelement++;
}


#================================ WRITE SVG XML ================================
### image settings
my $width = 2400;#3600;
my $height = 2400;#3600;
my $by = $height / 2;
my $ax = $width / 2;
my $xlegend = 0;
my $ylegend= 0;
my $pi = 3.14159265358979323846264338327950288419716939937510582097494459230781640628620899862;

# xml / SVG
open(SVG, ">$out") || die "Can't write to $out -- fatal.\n";
print SVG "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n";
print SVG "<!DOCTYPE svg PUBLIC \"-//W3C//DTD SVG 1.0//EN\" \"http://www.w3.org/TR/2001/REC-SVG-20010904/DTD/svg10.dtd\">\n";
print SVG "<svg width=\"$width\" height=\"$height\" xmlns=\"http://www.w3.org/2000/svg\" style=\"background-color:black\" viewBox=\"0 0 2400 2400\" id=\"map-svg\" onload=\"makeDraggable(evt)\" >\n";

##### CSS 

print SVG "<defs>\n";
print SVG "\t<style type=\"text/css\"><![CDATA[\n";

print SVG "\t.sb {\n";
print SVG "\tstroke-width:0.8;\n";
print SVG "\tstroke-opacity:50%;\n";
print SVG "\t}\n";

print SVG "\t.compass {\n";
print SVG "\tfill: #fff;\n";
print SVG "\tstroke: #000;\n";
print SVG "\tstroke-width: 1;\n";
print SVG "\t}\n";

print SVG "\t.button {\n";
print SVG "\tfill: #225EA8;\n";
print SVG "\tstroke: #0C2C84;\n";
print SVG "\tstroke-width: 0.5;\n";
print SVG "\tstroke-miterlimit:6;\n";
print SVG "\tstroke-linecap: round;\n";
print SVG "\t}\n";

print SVG "\t.button:hover {\n";
print SVG "\tstroke-width: 1;\n";
print SVG "\t}\n";

print SVG "\t.plus-minus {\n";
print SVG "\tfill: #fff;\n";
print SVG "\tpointer-events: none;\n";
print SVG "\t}\n";

print SVG "\t.cv {\n";
print SVG "\tstroke: none;\n";
print SVG "\tfill-opacity:$op;\n";
print SVG "\t}\n";

print SVG "\t.cv:hover {\n";
print SVG "\tfill-opacity:0.8;\n";
print SVG "\t}\n";

print SVG "\t.legendcol {\n";
print SVG "\tstroke: none;\n";
print SVG "\tfill-opacity:0.6;\n";
print SVG "\t}\n";

print SVG "\t.legendcol:hover {\n";
print SVG "\tfill-opacity:0.8;\n";
print SVG "\t}\n";

print SVG "\ta:link, a:visited {\n";
print SVG "\tcursor: pointer;\n";
print SVG "\t}\n";

print SVG "\ta:hover, a:active {\n";
print SVG "\toutline: dotted 2px yellow;\n";
print SVG "\t}\n";

print SVG "\tg:hover, g:active {\n";
print SVG "\toutline: dotted 2px yellow;\n";
print SVG "\t}\n";

print SVG "\t.static {\n";
print SVG "\tcursor: not-allowed;\n";
print SVG "\t}\n";

print SVG "\t.draggable, .draggable-group {\n";
print SVG "\tcursor: pointer;\n";
print SVG "\t}\n";

print SVG "\tsvg text {\n";
print SVG "\tcursor: default;\n";
print SVG "\t-webkit-user-select: none;\n";
print SVG "\t-moz-user-select: none;\n";
print SVG "\t-ms-user-select: none;\n";
print SVG "\tuser-select: none;\n";
print SVG "\t}\n";


print SVG "\t]]></style>\n";
print SVG "</defs>\n";


##### END CSS

my @colarray=("cyan","fuchsia","white","orange","yellow","lime","#FFBBFF","#6495ED","red","#7FFFD4","#FFD700","#ADD8E6","#F08080",'#b2df8a','#CC00FF','#cab2d6','#b15928','#ffffb3','#fb8072','#80b1d3','#fdb462','#b3de69','#fccde5','#d9d9d9','#bc80bd','#ccebc5','#ffed6f',"#FAF0E6","#FF00FF","#B03060","#800000","#66CDAA","#0000CD","#BA55D3","#9370DB","#3CB371","#7B68EE","#00FA9A","#48D1CC","#C71585","#F5FFFA","#FFE4E1","#FFE4B5","#FFDEAD","#000080","#FDF5E6","#808000","#6B8E23","#FFA500","#FF4500","#DA70D6","#EEE8AA","#98FB98","#AFEEEE","#DB7093","#FFEFD5","#FFDAB9","#CD853F","#FFC0CB","#DDA0DD","#B0E0E6","#A020F0","#800080","#663399","#FF0000","#BC8F8F","#4169E1","#8B4513","#FA8072","#F4A460","#2E8B57","#FFF5EE","#A0522D","#C0C0C0","#87CEEB","#6A5ACD","#708090","#FFFAFA","#00FF7F","#4682B4","#D2B48C","#008080","#D8BFD8","#FF6347","#40E0D0","#EE82EE","#F5DEB3","#FFFFFF","#F5F5F5","#FFFF00","#9ACD32");

@colarray=("fuchsia","cyan") if($allmut==2);

my $xtitle = 20;
print SVG "<text font-size=\"3.6em\" x=\"$xtitle\" y=\"55\" fill=\"white\">SARS-CoV-2 variants $year timemap</text>\n";###print title first (will be reprinted with links)

print SVG "<g id=\"matrix-group\" transform=\"matrix(1 0 0 1 0 0)\" class=\"draggable-group\">\n";  ### this sets the zoom/pan area

### concentric "gene" circles
foreach my $lastcoord (sort {$b<=>$a} keys %$circ){
   my $rad = int(($buffer + $lastcoord - $start) / $scale);
   if($circ->{$lastcoord} eq "black"){$fop=1;}else{$fop=0.085;}
   print SVG "<circle cx=\"$ax\" cy=\"$by\" r=\"$rad\" style=\"fill:$circ->{$lastcoord};stroke:black;stroke-width:0;fill-opacity:$fop;\"/>\n";
}###

my ($xleg,$yleg) = (25,190);
my $xlegright = $width-520;

my $xlegtx = $xleg + 15;
my $ylegtx = $yleg + 8;

my $mid = $ax - 35;
my $date = `date +%F`;
chomp($date);

##ticks
my $midaxis = $ax + 10;#was 2
my $midaxistx = $midaxis + 1;
my $adjstart = $by - int($buffer/$scale);
my $adjstart1 = $adjstart + 28;#was 6
my $adjstart2 = $adjstart + 15;
my $adjend = $adjstart - (($end-$start)/$scale);##size of ref
my $adjend1 = $adjend - 15;#was 8

my $adjendjan2 = $adjend - 15;##CHANGE FOR TICKS CLOSER TO AXES
my $adjendjan1 = $adjendjan2 - 50;
my $adjendjan3 = $adjendjan1 - 10;
my $adjendjul1 = $by + int(($buffer + ($end-$start))/$scale) + 15; ##CHANGE FOR TICKS CLOSER TO AXES
my $adjendjul2 = $adjendjul1 + 50;
my $adjendjul3 = $adjendjul2 + 35;

### WINTER
my $midjan = $mid + 11;# - 35;
print SVG "<line x1=\"$ax\" y1=\"$adjendjan1\" x2=\"$ax\" y2=\"$adjendjan2\" style=\"stroke:silver;stroke-width:4\" />\n";
print SVG "<text font-size=\"3.0em\" x=\"$midjan\" y=\"$adjendjan3\" fill=\"white\">Winter</text>\n";

### SPRING
my $aprleft = ($ax + int(($buffer+($end-$start))/$scale)) + 15;##CHANGE FOR TICKS CLOSER TO AXES
my $aprright = $aprleft+50;
my $midapr = $mid + 80;
print SVG "<line x1=\"$aprleft\" y1=\"$by\" x2=\"$aprright\" y2=\"$by\" style=\"stroke:silver;stroke-width:4\" />\n";
print SVG "<text font-size=\"3.0em\" x=\"$aprleft\" y=\"$midapr\" fill=\"white\">Spring</text>\n";

### SUMMER
print SVG "<line x1=\"$ax\" y1=\"$adjendjul1\" x2=\"$ax\" y2=\"$adjendjul2\" style=\"stroke:silver;stroke-width:4\" />\n";
my $midjul = $mid - 120;
print SVG "<text font-size=\"3.0em\" x=\"$midjul\" y=\"$adjendjul3\" fill=\"white\">Summer</text>\n";

### FALL
my $octleft = ($ax - int(($buffer+($end-$start))/$scale)) - 15;##CHANGE FOR TICKS CLOSER TO AXES
my $octright = $octleft-60;
print SVG "<line x1=\"$octleft\" y1=\"$by\" x2=\"$octright\" y2=\"$by\" style=\"stroke:silver;stroke-width:4\" />\n";
$octright -= 14;
my $midoct = $mid + 20;
print SVG "<text font-size=\"3.0em\" x=\"$octright\" y=\"$midoct\" fill=\"white\">Fall</text>\n";

##############
my $start_text = $start if($start<1000);
if($start>=1000){
   $start_text = sprintf("%.1f", ($start/1000));
   $start_text .= "kb";
}

my $end_text = $end if($end<1000);
if($end>=1000){
   $end_text = sprintf("%.1f", ($end/1000));
   $end_text .= "kb";
}
### GENES

my $genex=$ax+8;
my $genecol="grey";
### will draw gene structures on axis
foreach my $init(sort {$a<=>$b} keys %$genes){
   my $genstart = $by - int(($buffer+($init-$start))/$scale);
   my $genend = $by - int(($buffer+($genes->{$init}{'end'}-$start))/$scale);
   print SVG "<g>\n";
   print SVG "\t<title>$genes->{$init}{'name'} gene ($init-$genes->{$init}{'end'})</title>\n";
   print SVG "\t<line x1=\"$genex\" y1=\"$genstart\" x2=\"$genex\" y2=\"$genend\" style=\"stroke:$genecol;stroke-width:8\" />\n";
   print SVG "</g>\n";
   if($genecol eq "gainsboro"){ $genecol="grey"; }else{ $genecol="gainsboro"; }
}#

### GENOME AXIS
print SVG "<line x1=\"$ax\" y1=\"$adjend\" x2=\"$ax\" y2=\"$adjstart\" style=\"stroke:white;stroke-width:3\" />\n";

### draw connector to text
#top
my ($tickend1,$tickend2)=($ax+4,$ax+7);
print SVG "<line x1=\"$tickend1\" y1=\"$adjend\" x2=\"$tickend2\" y2=\"$adjend\" style=\"stroke:silver;stroke-width:2\" />\n";
#side
print SVG "<line x1=\"$tickend2\" y1=\"$adjend\" x2=\"$midaxis\" y2=\"$adjend1\" style=\"stroke:silver;stroke-width:2\" />\n";

#bottom
print SVG "<line x1=\"$tickend1\" y1=\"$adjstart\" x2=\"$tickend2\" y2=\"$adjstart\" style=\"stroke:silver;stroke-width:2\" />\n";
#side
print SVG "<line x1=\"$tickend2\" y1=\"$adjstart\" x2=\"$midaxis\" y2=\"$adjstart2\" style=\"stroke:silver;stroke-width:2\" />\n";

### textticks
print SVG "<text font-size=\"1.25em\" x=\"$midaxistx\" y=\"$adjend1\" fill=\"white\">$end_text</text>\n";
print SVG "<text font-size=\"1.25em\" x=\"$midaxistx\" y=\"$adjstart1\" fill=\"white\">$start_text</text>\n";

###circle
my ($ex,$ey) = ($ax-115,$by-20);
my ($cx,$cy) = ($ax-115,$by+20);
$cx += 50;
$cy += 40;


my $max=0;
my $maxradius=0;
my $blo;
my $bloeu;

open(TSV,">barplot.tsv") || die "Can't write to barplot.tsv";
print TSV "frame\tday\tdate\tvariant\tlab\tposition\tContinent\tpercent\n";

###DISPLAYS THE SCATTER
foreach my $dco (sort {$a<=>$b} keys %$track){

   my $mlist = $muthash->{$dco};

   #### STACKED BAR
   my $base = $lowbuffer;
   my ($fx1,$fy1,$fx2,$fy2) = (0,0,0,0);

   foreach my $jurisdiction(sort {$reglist->{$b}{'ct'}<=>$reglist->{$a}{'ct'}} keys %$reglist){
      my $samplelist = $totdailyreglist->{$dco}{$jurisdiction};
      my $dailyregionsample = keys(%$samplelist);

      if($dailyregionsample >= $mindaily){

         my $stackstart = $base;
         my $stackend = $base + ($stackfactor*(log($dailyregionsample+1)/log(10)));

         if($dco<=90){
            my $adjdeg = (90 - $dco) * $pi / 180;
            my ($x1,$y1,$x2,$y2) = ((cos($adjdeg)*$stackstart),(sin($adjdeg)*$stackstart),(cos($adjdeg)*$stackend),(sin($adjdeg)*$stackend));
            ($fx1,$fy1,$fx2,$fy2) = (($ax + $x1),($by - $y1),($ax + $x2),($by - $y2));
         }elsif($dco<=180){
            my $adjdeg = (180 - $dco) * $pi / 180;
            my ($x1,$y1,$x2,$y2) = ((sin($adjdeg)*$stackstart),(cos($adjdeg)*$stackstart),(sin($adjdeg)*$stackend),(cos($adjdeg)*$stackend));
            ($fx1,$fy1,$fx2,$fy2) = (($ax + $x1),($by + $y1),($ax + $x2),($by + $y2));
         }elsif($dco<=270){
            my $adjdeg = (270 - $dco) * $pi / 180;
            my ($x1,$y1,$x2,$y2) = ((cos($adjdeg)*$stackstart),(sin($adjdeg)*$stackstart),(cos($adjdeg)*$stackend),(sin($adjdeg)*$stackend));
            ($fx1,$fy1,$fx2,$fy2) = (($ax - $x1),($by + $y1),($ax - $x2),($by + $y2));
         }elsif($dco<367){#was 365
            my $adjdeg = (360 - $dco) * $pi / 180;
            my ($x1,$y1,$x2,$y2) = ((sin($adjdeg)*$stackstart),(cos($adjdeg)*$stackstart),(sin($adjdeg)*$stackend),(cos($adjdeg)*$stackend));
            ($fx1,$fy1,$fx2,$fy2) = (($ax - $x1),($by - $y1),($ax - $x2),($by - $y2));
         }
         ###DRAW SAMPLE STACK
         ### text
         my $colbar = $colarray[$reglist->{$jurisdiction}{'el'}];
         if($allmut==2){
            $jurisdiction = "samples";
            $colbar = "white";
         }

         print SVG "<g>\n";
         printf SVG "\t<title>$dateconversion->{$dco} $jurisdiction=$dailyregionsample</title>\n";
         ### lines ZZZ
         printf SVG "\t<line x1=\"%.1f\" y1=\"%.1f\" x2=\"%.1f\" y2=\"%.1f\" style=\"stroke:$colbar;\" class=\"sb\" />\n", ($fx1,$fy1,$fx2,$fy2);
         print SVG "</g>\n";

         $base = $stackend;
      }
   }
   #######


   foreach my $mm (sort {$mlist->{$a}<=>$mlist->{$b}} keys %$mlist){### this will effectively sort/plot mutations in increasing order of position within the genome seq
      #print "\t$mm\n";

      my $clist = $track->{$dco}{$mm};
      my $previousradius = 0;
      my $previouspropmut = 0;
      my $regiontext = "";
      my @spot;
      my @maxreg;

      foreach my $region (sort {$clist->{$b}{$fhash}<=>$clist->{$a}{$fhash} or $clist->{$a}{$shash}<=>$clist->{$b}{$shash}} keys %$clist){### this sorting is important, will place the large points first, then medium, then small, allowing to interact with more data points
         my $coord = $1 if($mm =~ /\D+(\d+)\D+/);

         if($coord>=$start && $coord<=$end){

            my $tmplist = $totdailyreglist->{$dco}{$region};
            my $numbersample = keys(%$tmplist);

            my $propmut = $clist->{$region}{'prop'};

            my $len = ($buffer + $coord - $start) / $scale;
            my ($fx,$fy,$radius) = (0,0,0);

            if($plotprop){###plot as percentage
               #$radius = $factor * (log($propmut+1) / log(10));#sqrt($propmut * $factorprop / $pi); #$factor * (log($propmut+1) / log(10));
               $radius = sqrt($factorprop * $propmut / $pi);###best results, discriminate 99.3 vs 97
            }else{###regular plot
               $radius = $factor * (log($clist->{$region}{'sum'}+1) / log(10));
            }

            $max = $clist->{$region}{'sum'} if ($clist->{$region}{'sum'} >= $max);
            $maxradius = $radius if($radius>=$maxradius);

            if($dco<=90){
               my $adjdeg = (90 - $dco) * $pi / 180;
               my $y = (sin($adjdeg)*$len);
               my $x = (cos($adjdeg)*$len);
               $fx = ($ax + $x);
               $fy = ($by - $y);
            }elsif($dco<=180){
               my $adjdeg = (180 - $dco) * $pi / 180;
               my $x= (sin($adjdeg)*$len);
               my $y = (cos($adjdeg)*$len);
               $fx = ($ax + $x);
               $fy = ($by + $y);
            }elsif($dco<=270){
               my $adjdeg = (270 - $dco) * $pi / 180;
               my $y = (sin($adjdeg)*$len);
               my $x = (cos($adjdeg)*$len);
               $fx = ($ax - $x);
               $fy = ($by + $y);
            }elsif($dco<367){#was 365
               my $adjdeg = (360 - $dco) * $pi / 180;
               my $x = (sin($adjdeg)*$len);
               my $y = (cos($adjdeg)*$len);
               $fx = ($ax - $x);
               $fy = ($by - $y);
            }

            ###MEAT OF THE SVG
            if($clist->{$region}{'sum'} >= $mindaily && ($varlist eq "NA" || ($vareffmap->{$mm}{'aa'}=~/[A-Z]\d+[A-Z]/ && ($varlist=~/$mm/ || $varlist=~/$vareffmap->{$mm}{'aa'}/)))){### additional filter to only plot specified variants

               ###reformat for indels
               my ($was,$is)=($1,$2) if($mm=~/(\D+)\d+(\D+)/);

               my $lab = "";
               if($propmut >= $minthreshold){### only label when matching threshold
                  if(defined $vareffmap->{$mm}){### assign mut effect when defined
                     if($vareffmap->{$mm}{'aa'} ne ""){
                        $lab = $vareffmap->{$mm}{'aa'};
                        $lab .= "/del" if(length($was) > length($is));
                        $lab .= "/ins" if(length($is) > length($was));
                     }else{
                        $lab = "NC";
                     }
                  }else{
                     $lab = "NC";
                  }
               }
               printf TSV "a$dco\t$dco\t$clist->{$region}{'date'}\t$mm\t$lab\t$coord\t$region\t%.1f\n", $propmut;

               $blo->{$mm}+=$clist->{$region}{'sum'};# if($region eq "Europe");
               $bloeu->{$mm}+=$clist->{$region}{'sum'} if($region eq "Europe");
               push @spot, "<g>\n";
               my $effecttext = "";
               my $tmpregtext = sprintf " $region=$clist->{$region}{'sum'} [%.1f%%]", $propmut;
               #if((int($radius)==$previousradius || int($radius+0.25)>=$previousradius) && $previousradius){###circles close in size/proportion. Increase to bunch together, decrease to make distinct. Difficult to adjust on a log10 scale. Could be 0.3 (would add redudancy, which is ok, because decreasing more could make info disappear at current resolution

               if($plotprop && abs($previouspropmut-$propmut)<=5 && $previouspropmut){
                  $regiontext = $tmpregtext . $regiontext;
                  push @maxreg,$regiontext;
               }elsif(! $plotprop && abs(int($previousradius)-int($radius))<0.25 && $previousradius){
                  $regiontext .= $tmpregtext;
                  push @maxreg,$regiontext;
               }else{  ####if($previousradius > (int($radius))){### reset any other possibilities reset text when pt sz smaller
                  $regiontext = $tmpregtext;
               }


               #print "$mm $clist->{$region}{'tot'} or $clist->{$region}{'sum'} : $regiontext\n";


               $effecttext = " $vareffmap->{$mm}{'product'}, $vareffmap->{$mm}{'aa'}" if(defined $vareffmap->{$mm}); ### some nt mutations are within UTRs
               $effecttext .= "/del" if(length($was) > length($is));
               $effecttext .= "/ins" if(length($is) > length($was));

               ### text
               push @spot, "\t<title>$clist->{$region}{'date'}  $mm $regiontext $effecttext</title>\n";

               ### data point
               my $spotpoint = sprintf "\t<circle cx=\"%.1f\" cy=\"%.1f\" r=\"%.1f\" style=\"fill:$colarray[$reglist->{$region}{'el'}];\" class=\"cv\" />\n", ($fx,$fy,$radius);
               push @spot, $spotpoint;
               push @spot, "</g>\n";

               $previouspropmut = $propmut;
               $previousradius = int($radius);
            }### only filtered variants
         }###only show mutants within range     
      }
      ### clean the redundancy to reduce filesize
      foreach my $line(@spot){
         #print "$line\n";
         #title>2021-05-26  C23604G  Europe=3 [42.9%]  surface glycoprotein
         if($line=~/title\>\S+\s+\S+\s+(.*\])\s+([^\]]*)$/){
            my $xtract=$1;
            $xtract=~s/\[/\\\[/g;
            $xtract=~s/\%/\\\%/g;
            $xtract=~s/\]/\\\]/g;
            my $skip=0;
            foreach my $toptext(@maxreg){
               #print "\t**$xtract** ? **$toptext**\n";
               if($toptext=~/$xtract/ && length($xtract)<length($toptext)){
                  #print "\t\t***$xtract SUBSET $toptext\n";
                  $skip=1;
               }
            }
            print SVG $line if(! $skip);
         }else{
            print SVG $line;
         }
      }
   }
}
print SVG "</g>\n"; ### ends zoomable region

print SVG "<rect x=\"4\" y=\"4\" width=\"895\" height=\"120\" rx=\"15\" style=\"fill:#1B1B1B;stroke-width:0;fill-opacity:0.8;\" />\n";

my $url="";
if($year==2020){
   $url = "https://bcgsc.github.io/SARS2";
}else{
   $url = "https://bcgsc.github.io/SARS2_$year";
}

print SVG "<a href=\"$url\">\n";
print SVG "<title>main menu</title>\n";
print SVG "<text font-size=\"3.6em\" x=\"$xtitle\" y=\"55\" fill=\"white\">SARS-CoV-2 variants $year timemap</text>\n";
print SVG "</a>\n";

print SVG "<text font-size=\"2.5em\" x=\"$xtitle\" y=\"105\" fill=\"white\">Genbank MN908947 vs $ctvir GISAID genomes</text>\n";

### print zoom/pan legend (code courtesy : https://github.com/petercollingridge/code-for-blog/tree/master/svg-interaction/pan_and_zoom)

print SVG "<circle cx=\"950\" cy=\"65\" r=\"43\" fill=\"black\" opacity=\"0.8\" />\n";
print SVG "<circle cx=\"950\" cy=\"65\" r=\"42\" fill=\"#225EA8\" opacity=\"0.4\" onclick=\"rotatex(90,$ax,$by)\"/>\n";
print SVG "\t<path class=\"button\" onclick=\"pan( 0, 50)\" d=\"M950 25 l12   20 a40, 70 0 0,0 -24,  0z\" />\n";
print SVG "\t<path class=\"button\" onclick=\"pan( 50, 0)\" d=\"M910 65 l20  -12 a70, 40 0 0,0   0, 24z\" />\n";
print SVG "\t<path class=\"button\" onclick=\"pan( 0,-50)\" d=\"M950 105 l12  -20 a40, 70 0 0,1 -24,  0z\" />\n";
print SVG "\t<path class=\"button\" onclick=\"pan(-50, 0)\" d=\"M990 65 l-20 -12 a70, 40 0 0,1   0, 24z\" />\n";

print SVG "\t<circle class=\"compass\" cx=\"950\" cy=\"65\" r=\"20\"/>\n";
print SVG "\t<circle class=\"button\" cx=\"950\" cy=\"56\" r=\"8\" onclick=\"zoom(0.8)\"/>\n";
print SVG "\t<circle class=\"button\" cx=\"950\" cy=\"74\" r=\"8\" onclick=\"zoom(1.25)\"/>\n";

print SVG "\t<rect class=\"plus-minus\" x=\"946\" y=\"54.5\" width=\"8\" height=\"3\"/>\n";
print SVG "\t<rect class=\"plus-minus\" x=\"946\" y=\"72.5\" width=\"8\" height=\"3\"/>\n";
print SVG "\t<rect class=\"plus-minus\" x=\"948.5\" y=\"70\" width=\"3\" height=\"8\"/>\n";

#####################LEGEND
my $xlegtitle = $xtitle+15;
my $ylegtitle = $yleg - 30;
my $ylegori=$yleg;
my $ylegtxori=$ylegtx;
my ($yleglast,$ylegtxlast)=(0,0);
my $ycorr=0;

if($allmut==2){
   ## black rectangle background
   print SVG "<rect x=\"4\" y=\"125\" width=\"220\" height=\"127\" rx=\"15\" style=\"fill:#1B1B1B;stroke-width:0;fill-opacity:0.8;\" />\n";
   print SVG "<rect x=\"4\" y=\"253\" width=\"220\" height=\"110\" rx=\"15\" style=\"fill:#1B1B1B;stroke-width:0;fill-opacity:0.8;\" />\n";
   print SVG "<rect x=\"4\" y=\"364\" width=\"220\" height=\"51\" rx=\"15\" style=\"fill:#1B1B1B;stroke-width:0;fill-opacity:0.8;\" />\n";#QQQ
   ### BUTTONS TO CHANGE YEAR / TOGGLE #/%
   print SVG "<rect x=\"150\" y=\"375\" width=\"45\" height=\"30\" rx=\"16\" style=\"fill:blue;stroke:black;stroke-width:2;\" />\n";
   print SVG "<rect x=\"30\" y=\"375\" width=\"90\" height=\"30\" rx=\"16\" style=\"fill:blue;stroke:black;stroke-width:2;\" />\n";
   if($year==2020){
      my $newurl = $url . "_2021";
      my $newout = $out;
      $newout=~s/2020/2021/g;
      $newurl .= "/$newout";
      print SVG "<a href=\"$newurl\"><title>change year</title><text font-size=\"1.4em\" x=\"55\" y=\"397\" fill=\"white\">2021</text></a>\n";
   }elsif($year==2021){
      my $newurl = $url;
      my $newout = $out;
      $newout=~s/2021/2020/g;
      $newurl=~s/\_2021//;
      $newurl .= "/$newout";
      print SVG "<a href=\"$newurl\"><title>change year</title><text font-size=\"1.4em\" x=\"55\" y=\"397\" fill=\"white\">2020</text></a>\n";
   }
   if($plotprop){
      my $newout = $out;
      $newout=~s/ratio//g;
      my $newurl = $url . "/$newout";
      print SVG "<a href=\"$newurl\"><title>view counts</title><text font-size=\"1.4em\" x=\"165\" y=\"397\" fill=\"white\">\#</text></a>\n";
   }else{
      my $newout = $out;
      $newout=~s/SARS2/ratioSARS2/g;
      my $newurl = $url . "/$newout";
      print SVG "<a href=\"$newurl\"><title>view ratios</title><text font-size=\"1.4em\" x=\"165\" y=\"397\" fill=\"white\">\%</text></a>\n";
   }

   ###
   print SVG "<text font-size=\"2.1em\" x=\"$xlegtitle\" y=\"$ylegtitle\" fill=\"white\">Variant type</text>\n";
   $ycorr=200;
}else{
   if($continent eq ".*"){
      ## black rectangle background
      print SVG "<rect x=\"4\" y=\"125\" width=\"220\" height=\"227\" rx=\"15\" style=\"fill:#1B1B1B;stroke-width:0;fill-opacity:0.8;\" />\n";
      print SVG "<rect x=\"4\" y=\"353\" width=\"220\" height=\"110\" rx=\"15\" style=\"fill:#1B1B1B;stroke-width:0;fill-opacity:0.8;\" />\n";
      print SVG "<rect x=\"4\" y=\"464\" width=\"220\" height=\"51\" rx=\"15\" style=\"fill:#1B1B1B;stroke-width:0;fill-opacity:0.8;\" />\n";#QQQ
      ### BUTTONS TO CHANGE YEAR / TOGGLE #/%
      print SVG "<rect x=\"150\" y=\"475\" width=\"45\" height=\"30\" rx=\"16\" style=\"fill:blue;stroke:black;stroke-width:2;\" />\n";
      print SVG "<rect x=\"30\" y=\"475\" width=\"90\" height=\"30\" rx=\"16\" style=\"fill:blue;stroke:black;stroke-width:2;\" />\n";
      if($year==2020){
        my $newurl = $url . "_2021";
        my $newout = $out;
        $newout=~s/2020/2021/g;
        $newurl .= "/$newout";
        print SVG "<a href=\"$newurl\"><title>change year</title><text font-size=\"1.4em\" x=\"55\" y=\"497\" fill=\"white\">2021</text></a>\n";
      }elsif($year==2021){
         my $newurl = $url;
         my $newout = $out;
         $newout=~s/2021/2020/g;
         $newurl=~s/\_2021//;
         $newurl .= "/$newout";
         print SVG "<a href=\"$newurl\"><title>change year</title><text font-size=\"1.4em\" x=\"55\" y=\"497\" fill=\"white\">2020</text></a>\n";
      }
      if($plotprop){
         my $newout = $out;
         $newout=~s/ratio//g;
         my $newurl = $url . "/$newout";
         print SVG "<a href=\"$newurl\"><title>view counts</title><text font-size=\"1.4em\" x=\"165\" y=\"497\" fill=\"white\">\#</text></a>\n";
      }else{
         my $newout = $out;
         $newout=~s/SARS2/ratioSARS2/g;
         my $newurl = $url . "/$newout";
         print SVG "<a href=\"$newurl\"><title>view ratios</title><text font-size=\"1.4em\" x=\"165\" y=\"497\" fill=\"white\">\%</text></a>\n";
      }

      ###
      print SVG "<text font-size=\"2.1em\" x=\"$xlegtitle\" y=\"$ylegtitle\" fill=\"white\">Continents</text>\n";
      $ycorr=100;
   }else{
      ## black rectangle background
      print SVG "<rect x=\"4\" y=\"125\" width=\"425\" height=\"327\" rx=\"15\" style=\"fill:#1B1B1B;stroke-width:0;fill-opacity:0.8;\" />\n";
      print SVG "<rect x=\"4\" y=\"453\" width=\"220\" height=\"110\" rx=\"15\" style=\"fill:#1B1B1B;stroke-width:0;fill-opacity:0.8;\" />\n";
      print SVG "<rect x=\"4\" y=\"564\" width=\"220\" height=\"51\" rx=\"15\" style=\"fill:#1B1B1B;stroke-width:0;fill-opacity:0.8;\" />\n";#QQQ
      ### BUTTONS TO CHANGE YEAR / TOGGLE #/%
      print SVG "<rect x=\"150\" y=\"575\" width=\"45\" height=\"30\" rx=\"16\" style=\"fill:blue;stroke:black;stroke-width:2;\" />\n";
      print SVG "<rect x=\"30\" y=\"575\" width=\"90\" height=\"30\" rx=\"16\" style=\"fill:blue;stroke:black;stroke-width:2;\" />\n";
      if($year==2020){
         my $newurl = $url . "_2021";
         my $newout = $out;
         $newout=~s/2020/2021/g;
         $newurl .= "/$newout";
         print SVG "<a href=\"$newurl\"><title>change year</title><text font-size=\"1.4em\" x=\"55\" y=\"597\" fill=\"white\">2021</text></a>\n";
      }elsif($year==2021){
         my $newurl = $url;
         my $newout = $out;
         $newout=~s/2021/2020/g;
         $newurl=~s/\_2021//;
         $newurl .= "/$newout";
         print SVG "<a href=\"$newurl\"><title>change year</title><text font-size=\"1.4em\" x=\"55\" y=\"597\" fill=\"white\">2020</text></a>\n";
      }
      if($plotprop){
         my $newout = $out;
         $newout=~s/ratio//g;
         my $newurl = $url . "/$newout";
         print SVG "<a href=\"$newurl\"><title>view counts</title><text font-size=\"1.4em\" x=\"165\" y=\"597\" fill=\"white\">\#</text></a>\n";
      }else{
         my $newout = $out;
         $newout=~s/SARS2/ratioSARS2/g;
         my $newurl = $url . "/$newout";
         print SVG "<a href=\"$newurl\"><title>view ratios</title><text font-size=\"1.4em\" x=\"165\" y=\"597\" fill=\"white\">\%</text></a>\n";
      }
      ###
      print SVG "<text font-size=\"2.1em\" x=\"$xlegtitle\" y=\"$ylegtitle\" fill=\"white\">Top regions</text>\n";
   }
}

my $ct=0;

CTRY:
foreach my $country (sort {$reglist->{$b}{'ct'}<=>$reglist->{$a}{'ct'}} keys %$reglist){
   print SVG "<circle cx=\"$xleg\" cy=\"$yleg\" r=\"6\" style=\"fill:$colarray[$reglist->{$country}{'el'}];stroke:black;stroke-width:0;\" class=\"legendcol\" />\n";
   print SVG "<text font-size=\"1.6em\" x=\"$xlegtx\" y=\"$ylegtx\" fill=\"white\">$reglist->{$country}{'correctedname'}</text>\n";
   $yleg+=25;
   $ylegtx+=25;
   $ct++;
   if($ct==10){$yleglast=$yleg;$ylegtxlast=$ylegtx;$yleg=$ylegori;$ylegtx=$ylegtxori;$xleg+=220;$xlegtx+=220;}
   last CTRY if($ct==20);
}

if($ct>=10){
   $xleg-=220;
   $xlegtx-=220;
   $ylegtx=$ylegtxlast+60;
   $yleg=$yleglast+15;
}else{
   $ylegtx+=60;
   $yleg+=15;
}
my $xlegori = $xleg - 10;
#$yleg += 10;##space between point size and legend text explanation

#### set the legend position
$yleg=475-$ycorr;
$ylegtx = $yleg + 40;


if($plotprop){###plot point size as a fraction
  my @step = (1,10,100);
   foreach my $size (@step){
      #my $radleg = $factor * (log($size+1)/log(10)); #sqrt($size * $factorprop / $pi); #$factor * (log($size+1)/log(10));
      my $radleg = sqrt($factorprop * $size / $pi);
      printf SVG "<circle cx=\"%.0f\" cy=\"%.0f\" r=\"%.0f\" style=\"fill:white;stroke:lightgrey;stroke-width:0.2;\" />\n", ($xleg,$yleg,$radleg);
      $xleg+=35;
   }
   my $ylegtx2=$ylegtx+30;
   print SVG "<text font-size=\"1.7em\" x=\"$xlegori\" y=\"$ylegtx\" fill=\"white\">1      10        100% daily</text>\n";
   print SVG "<text font-size=\"1.7em\" x=\"$xlegori\" y=\"$ylegtx2\" fill=\"white\">sample fraction</text>\n";
}else{###plot regular
   my @step = ($mindaily,10,100,1000);
   foreach my $size (@step){
      my $radleg = $factor * (log($size+1)/log(10));
      print SVG "<circle cx=\"$xleg\" cy=\"$yleg\" r=\"$radleg\" style=\"fill:white;stroke:lightgrey;stroke-width:0.2;\" />\n";
      $xleg+=36;
   }
   my $ylegtx2=$ylegtx+30;
   print SVG "<text font-size=\"1.8em\" x=\"$xlegori\" y=\"$ylegtx\" fill=\"white\">$mindaily      10        100        1k</text>\n";
   print SVG "<text font-size=\"1.8em\" x=\"$xlegori\" y=\"$ylegtx2\" fill=\"white\">genome samples</text>\n";
}


#### SIDE INFO
printf "MAX genomes covering mutation : $max (MAX radius of dot: %.2f)\n", $maxradius;
my $numvar=keys(%$blo);
my $eunumvar=keys(%$bloeu);
my $numcountry =keys(%$reglist);
print "REGIONS: $numcountry\n";

print "TOTAL DISTINCT VARIANTS: $numvar ($eunumvar europe)\n";
while($numvar =~ s/(\d+)(\d\d\d)/$1\,$2/){};

#### PRINT ADDITIONAL MAP INFO
my $xlegrightbar = $xlegright - 15;
#print SVG "<line x1=\"$xlegrightbar\" y1=\"145\" x2=\"$xlegrightbar\" y2=\"270\" style=\"stroke:gainsboro;stroke-width:10;fill-opacity:$fop;\" />\n";

#### rounded rectangle for labels
print SVG "<rect x=\"$xlegrightbar\" y=\"4\" width=\"500\" height=\"135\" rx=\"15\" style=\"fill:gainsboro;stroke-width:0;fill-opacity:0.15;\" />\n";

if($continent eq ".*"){
   print SVG "<text font-size=\"1.8em\" x=\"$xlegright\" y=\"32\" fill=\"white\">Data from 6 continents</text>\n";
}else{
   print SVG "<text font-size=\"1.8em\" x=\"$xlegright\" y=\"32\" fill=\"white\">Data from $numcountry regions in $continent</text>\n";
}
print SVG "<text font-size=\"1.8em\" x=\"$xlegright\" y=\"62\" fill=\"white\">Total genomes≥$min  Genomes/day≥$mindaily</text>\n";
print SVG "<text font-size=\"1.8em\" x=\"$xlegright\" y=\"92\" fill=\"white\">$numvar $mutationtype</text>\n";
print SVG "<text font-size=\"1.8em\" x=\"$xlegright\" y=\"122\" fill=\"white\">Updated $date</text>\n";

##### JS
print SVG " <script type=\"text/javascript\"><![CDATA[\n";
print SVG "        var transformMatrix = [1, 0, 0, 1, 0, 0];\n";
print SVG "        var svg = document.getElementById(\'map-svg\');\n";
print SVG "        var viewbox = svg.getAttributeNS(null, \"viewBox\").split(\" \");\n";
print SVG "        var centerX = parseFloat(viewbox[2]) / 2;\n";
print SVG "        var centerY = parseFloat(viewbox[3]) / 2;\n";
print SVG "        var matrixGroup = svg.getElementById(\"matrix-group\");\n";
print SVG "        function pan(dx, dy) {\n";
print SVG "            transformMatrix[4] += dx;\n";
print SVG "            transformMatrix[5] += dy;\n";
print SVG "            setMatrix();\n";
print SVG "        }\n";
print SVG "        function zoom(scale) {\n";
print SVG "            for (var i = 0; i < 6; i++) {\n";
print SVG "                transformMatrix[i] *= scale;\n";
print SVG "            }\n";
print SVG "            transformMatrix[4] += (1 - scale) * centerX;\n";
print SVG "            transformMatrix[5] += (1 - scale) * centerY;\n";
print SVG "            setMatrix();\n";
print SVG "        }\n";
print SVG "        function setMatrix() {\n";
print SVG "            var newMatrix = \"matrix(\" +  transformMatrix.join(\' \') + \")\";\n";
print SVG "            newMatrix += \" rotate(\" + 0 + \",\" + 1200 + \",\" + 1200 + \")\";\n";
print SVG "            matrixGroup.setAttributeNS(null, \"transform\", newMatrix);\n";
print SVG "        }\n";
print SVG "        function rotatex(angle,dx,dy) {\n";
print SVG "        var rt = \"matrix(\" +  transformMatrix.join(\' \') + \")\";\n";
print SVG "        rt += \' rotate(\' + angle + \',\' + dx + \',\' + dy + \')\';\n";
print SVG "        matrixGroup.setAttributeNS(null, \"transform\", rt);\n";
print SVG "        }\n";

### below draggable functions

print SVG "      function makeDraggable(evt) {\n";
print SVG "        var svg = evt.target;\n";

print SVG "        svg.addEventListener(\'mousedown\', startDrag);\n";
print SVG "        svg.addEventListener(\'mousemove\', drag);\n";
print SVG "        svg.addEventListener(\'mouseup\', endDrag);\n";
print SVG "        svg.addEventListener(\'mouseleave\', endDrag);\n";
print SVG "        svg.addEventListener(\'touchstart\', startDrag);\n";
print SVG "        svg.addEventListener(\'touchmove\', drag);\n";
print SVG "        svg.addEventListener(\'touchend\', endDrag);\n";
print SVG "        svg.addEventListener(\'touchleave\', endDrag);\n";
print SVG "        svg.addEventListener(\'touchcancel\', endDrag);\n";

print SVG "        function getMousePosition(evt) {\n";
print SVG "          var CTM = svg.getScreenCTM();\n";
print SVG "          if (evt.touches) { evt = evt.touches[0]; }\n";
print SVG "          return {\n";
print SVG "            x: (evt.clientX - CTM.e) / CTM.a,\n";
print SVG "            y: (evt.clientY - CTM.f) / CTM.d\n";
print SVG "          };\n";
print SVG "        }\n";

print SVG "       var selectedElement, offset, transform;\n";

print SVG "        function initialiseDragging(evt) {\n";
print SVG "            offset = getMousePosition(evt);\n";

print SVG "            // Make sure the first transform on the element is a translate transform\n";
print SVG "            var transforms = selectedElement.transform.baseVal;\n";

print SVG "            if (transforms.length === 0 || transforms.getItem(0).type !== SVGTransform.SVG_TRANSFORM_TRANSLATE) {\n";
print SVG "              // Create an transform that translates by (0, 0)\n";
print SVG "              var translate = svg.createSVGTransform();\n";
print SVG "              translate.setTranslate(0, 0);\n";
print SVG "              selectedElement.transform.baseVal.insertItemBefore(translate, 0);\n";
print SVG "            }\n";

print SVG "            // Get initial translation\n";
print SVG "            transform = transforms.getItem(0);\n";
print SVG "            offset.x -= transform.matrix.e;\n";
print SVG "            offset.y -= transform.matrix.f;\n";
print SVG "        }\n";

print SVG "       function startDrag(evt) {\n";
print SVG "          if (evt.target.classList.contains(\'draggable\')) {\n";
print SVG "            selectedElement = evt.target;\n";
print SVG "            initialiseDragging(evt);\n";
print SVG "          } else if (evt.target.parentNode.classList.contains(\'draggable-group\')) {\n";
print SVG "            selectedElement = evt.target.parentNode;\n";
print SVG "            initialiseDragging(evt);\n";
print SVG "          }\n";
print SVG "        }\n";

print SVG "        function drag(evt) {\n";
print SVG "          if (selectedElement) {\n";
print SVG "            evt.preventDefault();\n";
print SVG "            var coord = getMousePosition(evt);\n";
print SVG "            transform.setTranslate(coord.x - offset.x, coord.y - offset.y);\n";
print SVG "          }\n";
print SVG "        }\n";

print SVG "        function endDrag(evt) {\n";
print SVG "          selectedElement = false;\n";
print SVG "        }\n";
print SVG "      }\n";

print SVG "    ]]></script>\n";

print SVG "</svg>\n";

close TSV;
close SVG;

print "$0 output graph in $out\nTimemaps can be freely distributed under CC-BY license.\n";
print "\nThis software is distributed under GPLv3. If you use this software or the images it produces, please cite us:\n";
print "\n\nWarren RL and Birol I. Interactive SARS-CoV-2 mutation timemaps [version 1; peer review: 2 approved with reservations]. F1000Research 2021, 10:68 (https://doi.org/10.12688/f1000research.50857.1)\nhttps://bcgsc.github.io/SARS2\n\n";

exit;

