subroutine init_cavity_ts_use_profile
  ! Initialize the T/S under cavity by specifying a profile
  ! So this routine is not a general routine for all purpose.	
  ! Here I specify T/S in a selected region (read an extra file).
  ! For cavity alone, one can use the cavity_flag_nod2d index.
  ! 20.01.2012, Qiang
  !-----------------------------------------------------------
  
  use o_param
  use o_array
  use o_mesh
  use o_elements
  use g_config
  use g_parfe
  implicit none

  integer              :: k, n, n3, fileID
  integer              :: n_loc, num_nod
  integer, allocatable :: temp_arr2d(:), nodes_release(:)
  real(kind=8)         :: z
  character(100)       :: file_name

  allocate(temp_arr2d(nod2d))
  temp_arr2d=0
  do n=1, ToDim_nod2D
     temp_arr2d(myList_nod2D(n))=n
  end do

  file_name=trim(meshpath)//'FRIS_FD_region.out'
  fileID=170
  open(fileID, file=file_name)
  read(fileID,*) num_nod
  allocate(nodes_release(num_nod))
  read(fileID,*) nodes_release
  close(fileID)

  do n=1,num_nod
     n_loc=temp_arr2d(nodes_release(n))
     if(n_loc>0) then

        do k=1,num_layers_below_nod2d(n_loc)+1
           n3=nod3d_below_nod2d(k,n_loc)
	   z=coord_nod3d(3,n3)
           if(z>-2000.0) then
              tracer(n3,1)=-2.3 + (2.3-1.9)/2000.*(z+2000.0)
              tracer(n3,2)=34.8 + (34.5-34.8)/2000.*(z+2000.0)
           else
              tracer(n3,1)=-2.3
              tracer(n3,2)=34.8
           end if
        end do
     end if
  end do

  deallocate(nodes_release)
  deallocate(temp_arr2d)

end subroutine init_cavity_ts_use_profile
!
!-----------------------------------------------------------------
!
subroutine init_cavity_ts_extrapolate
  ! fill the ocean under the cavity with T/S at the cavity edge
  ! 
  ! Coded by Qiang Wang
  ! Reviewed by ??
  !------------------------------------------------------------

  use o_mesh
  use o_param
  use o_array
  use g_config
  use g_PARFE
  implicit none

  integer                     :: i, j, n, n2, n_bd, row, ind
  integer                     :: num_lat_reg, num_lon_reg, num_lay_reg
  integer, allocatable        :: ind_op(:), ind_op_glo(:), nearest_bd_nod(:)
  real(kind=8)                :: pp, pr, aux1,aux2, lon, lat, d, d_min
  real(kind=8)                :: x, y, temp_x, temp_y, temp_z
  real(kind=8), external      :: theta
  real(kind=8), allocatable   :: lon_reg(:), lat_reg(:), lay_reg(:)
  real(kind=8), allocatable   :: raw_data_T(:,:,:), raw_data_S(:,:,:)
  real(kind=8), allocatable   :: x_op(:), y_op(:), dep_op(:)
  real(kind=8), allocatable   :: x_op_glo(:), y_op_glo(:), dep_op_glo(:)

  ! define the cavity boundary line
  allocate(ind_op(nod2d), x_op(nod2d), y_op(nod2d), dep_op(nod2d))
  ind_op=0
  x_op=0.0
  y_op=0.0
  dep_op=0.0
  allocate(ind_op_glo(nod2d), x_op_glo(nod2d), y_op_glo(nod2d), dep_op_glo(nod2d))
  ind_op_glo=0
  x_op_glo=0.0
  y_op_glo=0.0
  dep_op_glo=0.0

  do row=1,myDim_nod2d  ! should not include eDim_nod2d 
     if(cavity_flag_nod2d(row)==0 .and. &
          any(cavity_flag_nod2d(nghbr_nod2d(row)%addresses)==1)) then 
        n=myList_nod2d(row)
        ind_op(n)=1
        x_op(n)=geolon2d(row)
        y_op(n)=geolat2d(row)
        dep_op(n)=coord_nod3d(3,bt_nds(row))
     end if
  end do

  call MPI_Barrier(MPI_COMM_WORLD,MPIerr)
  call MPI_AllREDUCE(ind_op, ind_op_glo, &
       nod2d, MPI_INTEGER, MPI_SUM, &
       MPI_COMM_WORLD, MPIerr)
  call MPI_AllREDUCE(x_op, x_op_glo, &
       nod2d, MPI_DOUBLE_PRECISION, MPI_SUM, &
       MPI_COMM_WORLD, MPIerr)
  call MPI_AllREDUCE(y_op, y_op_glo, &
       nod2d, MPI_DOUBLE_PRECISION, MPI_SUM, &
       MPI_COMM_WORLD, MPIerr)
  call MPI_AllREDUCE(dep_op, dep_op_glo, &
       nod2d, MPI_DOUBLE_PRECISION, MPI_SUM, &
       MPI_COMM_WORLD, MPIerr)

  deallocate(ind_op, x_op, y_op, dep_op)

  allocate(nearest_bd_nod(myDim_nod2d+eDim_nod2d))
  do row=1,myDim_nod2d+eDim_nod2d
     if(cavity_flag_nod2d(row)==0) cycle

     x=geolon2d(row)
     y=geolat2d(row)

     d_min=3000.0e3   !dist. in m
     do n=1,nod2d
        if(ind_op_glo(n)/=1) cycle
        call dist_on_earth(x, y, x_op_glo(n), y_op_glo(n), d)
        if(d<d_min) then
           ind=n
           d_min=d
        end if
     end do
     nearest_bd_nod(row)=ind
  end do

  ! open global T/S dataset files
  open(19,file=trim(ClimateDataPath)//trim(OceClimaDataName), status='old')
  ! read reg. grid
  read(19,*) num_lon_reg, num_lat_reg, num_lay_reg
  allocate(lon_reg(num_lon_reg))
  allocate(lat_reg(num_lat_reg))
  allocate(lay_reg(num_lay_reg))
  read(19,*) lon_reg
  read(19,*) lat_reg
  read(19,*) lay_reg
  allocate(raw_data_T(num_lon_reg,num_lat_reg,num_lay_reg))
  allocate(raw_data_S(num_lon_reg,num_lat_reg,num_lay_reg))
  ! read raw data: T
  do i=1, num_lon_reg
     do j=1, num_lat_reg
        read(19, *) raw_data_T(i,j,1:num_lay_reg)         
     end do
  end do
  ! read raw data: S
  do i=1, num_lon_reg
     do j=1, num_lat_reg
        read(19, *) raw_data_S(i,j,1:num_lay_reg)         
     end do
  end do
  close(19) 

  ! interpolate to the nodes under cavity
  do row=1, myDim_nod3d+eDim_nod3D  
     n2=nod2d_corresp_to_nod3d(row)
     if(cavity_flag_nod2d(n2)==0) cycle
     n_bd=nearest_bd_nod(n2)
     temp_z=coord_nod3d(3,row)
     if(temp_z<dep_op_glo(n_bd)) temp_z=dep_op_glo(n_bd)
     temp_x=x_op_glo(n_bd)/rad
     temp_y=y_op_glo(n_bd)/rad
     ! change lon range to [0 360]
     if(temp_x<0.) temp_x=temp_x + 360.0  

     call interp_3d_field(num_lon_reg, num_lat_reg, num_lay_reg, &
          lon_reg, lat_reg, lay_reg, &
          raw_data_T, 1, temp_x, temp_y, temp_z, aux1)

     call interp_3d_field(num_lon_reg, num_lat_reg, num_lay_reg, &
          lon_reg, lat_reg, lay_reg, &
          raw_data_S, 1, temp_x, temp_y, temp_z, aux2)
     tracer(row,2)=aux2

     ! Convert in situ temperature into potential temperature
     pr=0.0_8
     pp=abs(temp_z)
     tracer(row,1)=theta(aux2, aux1, pp, pr)
  end do

  deallocate(ind_op_glo, x_op_glo, y_op_glo, dep_op_glo, nearest_bd_nod)
  deallocate(raw_data_T, raw_data_S, lay_reg, lat_reg, lon_reg)
  
end subroutine init_cavity_ts_extrapolate
!
!-------------------------------------------------------------------------------------
!
subroutine cavity_heat_water_fluxes_3eq
  ! The three-equation model of ice-shelf ocean interaction (Hellmer et al., 1997)
  ! Code derived from BRIOS subroutine iceshelf (which goes back to H.Hellmer's 2D ice shelf model code)
  ! adjusted for use in FESOM by Ralph Timmermann, 16.02.2011
  ! Reviewed by ?
  !----------------------------------------------------------------

  use o_mesh
  use o_param
  use o_array
  use i_array
  use g_parfe
  implicit none

  real (kind=8)  :: temp,sal,tin,zice
  real (kind=8)  :: rhow, rhor, rho
  real (kind=8)  :: gats1, gats2, gas, gat
  real (kind=8)  :: ep1,ep2,ep3,ep4,ep5
  real (kind=8)  :: ex1,ex2,ex3,ex4,ex5
  real (kind=8)  :: vt1,sr1,sr2,sf1,sf2,tf1,tf2,tf,sf,seta,re
  integer        :: n, n3, nk

  real(kind=8),parameter ::  rp =   0.                        !reference pressure
  real(kind=8),parameter ::  a   = -0.0575                    !Foldvik&Kvinge (1974)
  real(kind=8),parameter ::  b   =  0.0901
  real(kind=8),parameter ::  c   =  7.61e-4

  real(kind=8),parameter ::  pr  =  13.8                      !Prandtl number      [dimensionless]
  real(kind=8),parameter ::  sc  =  2432.                     !Schmidt number      [dimensionless]
  real(kind=8),parameter ::  ak  =  2.50e-3                   !dimensionless drag coeff.
  real(kind=8),parameter ::  sak1=  sqrt(ak)
  real(kind=8),parameter ::  un  =  1.95e-6                   !kinematic viscosity [m2/s]
  real(kind=8),parameter ::  pr1 =  pr**(2./3.)               !Jenkins (1991)
  real(kind=8),parameter ::  sc1 =  sc**(2./3.)

  real(kind=8),parameter ::  tob=  -20.                       !temperatur at the ice surface
  real(kind=8),parameter ::  rhoi=  920.                      !mean ice density
  real(kind=8),parameter ::  cpw =  4180.0                    !Barnier et al. (1995)
  real(kind=8),parameter ::  lhf =  3.33e+5                   !latent heat of fusion
  real(kind=8),parameter ::  atk =  273.15                    !0 deg C in Kelvin
  real(kind=8),parameter ::  cpi =  152.5+7.122*(atk+tob)     !Paterson:"The Physics of Glaciers"

  real(kind=8),parameter ::  L    = 334000.                   ! [J/Kg]

  ! hemw = helium content of the glacial meltwater
  ! oomw = isotopic fractionation due to melting
  ! oofw = isotopic fractionation due to freezing
  !      hemw=  4.02*14.
  !      oomw= -30.
  !      oofw= -2.5

  n3=myDim_nod3d+eDim_nod3d

  do n=1,myDim_nod2D+eDim_nod2D      
     if(cavity_flag_nod2d(n)==0) cycle   
     nk=nod3d_below_nod2d(1,n)
     temp = tracer(nk,1)	
     sal  = tracer(nk,2)
     zice = coord_nod3d(3,nk)  !(<0)

     ! Calculate the in-situ temperature tin
     !call potit(s(i,j,N,lrhs)+35.0,t(i,j,N,lrhs),-zice(i,j),rp,tin)
     call potit(sal,temp,abs(zice),rp,tin)

     ! Calculate or prescribe the turbulent heat and salt transfer coeff. GAT and GAS
     ! velocity-dependent approach of Jenkins (1991)

     !rt      vt1  = 0.25*sqrt((u(i,j,N,lrhs)+u(i+1,j,N,lrhs))**2
     !rt     &                +(v(i,j,N,lrhs)+v(i,j+1,N,lrhs))**2)
     ! if(vt1.eq.0.) vt1=0.001
     !rt      re   = Hz_r(i,j,N)*ds/un        !Reynolds number

     vt1  = sqrt(uf(nk)*uf(nk)+uf(nk+n3)*uf(nk+n3))
     vt1  = max(vt1,0.001)
     re   = 10./un                   !vt1*re (=velocity times length scale over kinematic viscosity) is the Reynolds number

     gats1= sak1*vt1
     gats2= 2.12*log(gats1*re)-9.
     gat  = gats1/(gats2+12.5*pr1)
     gas  = gats1/(gats2+12.5*sc1)

     !RG3417 gat  = 1.00e-4   ![m/s]  RT: to be replaced by velocity-dependent equations later
     !RG3417 gas  = 5.05e-7   ![m/s]  RT: to be replaced by velocity-dependent equations later

     ! Calculate
     ! density in the boundary layer: rhow
     ! and interface pressure pg [dbar],
     ! Solve a quadratic equation for the interface salinity sb 
     ! to determine the melting/freezing rate seta.

     call fcn_density(temp,sal,zice,rho)
     rhow = rho !fcn_density returns full in-situ density now!
     ! in previous FESOM version, which has density anomaly from fcn_density, so used density_0+rho  
     ! was rhow= rho0+rho(i,j,N) in BRIOS

     rhor= rhoi/rhow

     ep1 = cpw*gat
     ep2 = cpi*gas
     ep3 = lhf*gas
     ep4 = b+c*zice
     ep5 = gas/rhor

     ! negative heat flux term in the ice (due to -kappa/D)
     ex1 = a*(ep1-ep2)
     ex2 = ep1*(ep4-tin)+ep2*(tob+a*sal-ep4)-ep3
     ex3 = sal*(ep2*(ep4-tob)+ep3)
     ex4 = ex2/ex1
     ex5 = ex3/ex1

     sr1 = 0.25*ex4*ex4-ex5
     sr2 = -0.5*ex4
     sf1 = sr2+sqrt(sr1)
     tf1 = a*sf1+ep4
     sf2 = sr2-sqrt(sr1)
     tf2 = a*sf2+ep4

     ! Salinities < 0 psu are not defined, therefore pick the positive of the two solutions:
     if(sf1.gt.0.) then
        tf = tf1
        sf = sf1
     else
        tf = tf2
        sf = sf2
     endif

     ! Calculate the melting/freezing rate [m/s]
     ! seta = ep5*(1.0-sal/sf)     !rt thinks this is not needed

     !rt  t_surf_flux(i,j)=gat*(tf-tin)
     !rt  s_surf_flux(i,j)=gas*(sf-(s(i,j,N,lrhs)+35.0))

     heat_flux(n)  = rhow*cpw*gat*(tin-tf)      ! [W/m2]  ! positive for upward
     water_flux(n) =          gas*(sf-sal)/sf   ! [m/s]   !

     !      qo=-rhor*seta*oofw
     !      if(seta.le.0.) then
     !         qc=rhor*seta*hemw
     !         qo=rhor*seta*oomw
     !      endif

     ! write(*,'(a10,i10,9f10.3)') 'ice shelf',n,zice,rhow,temp,sal,tin,tf,sf,heat_flux(n),water_flux(n)*86400.*365.

     !for saving to output:
     net_heat_flux(n)=-heat_flux(n)   ! positive down
     fresh_wa_flux(n)=-water_flux(n)

  enddo

end subroutine cavity_heat_water_fluxes_3eq
!
!-------------------------------------------------------------------------------------
!
subroutine potit(salz,pt,pres,rfpres,tin)
  ! Berechnet aus dem Salzgehalt[psu] (SALZ), der pot. Temperatur[oC]
  ! (PT) und dem Referenzdruck[dbar] (REFPRES) die in-situ Temperatur
  ! [oC] (TIN) bezogen auf den in-situ Druck[dbar] (PRES) mit Hilfe
  ! eines Iterationsverfahrens aus.

  integer iter
  real salz,pt,pres,rfpres,tin
  real epsi,tpmd,pt1,ptd,pttmpr

  data tpmd / 0.001 /

  epsi = 0.
  do iter=1,100
     tin  = pt+epsi
     pt1  = pttmpr(salz,tin,pres,rfpres)
     ptd  = pt1-pt
     if(abs(ptd).lt.tpmd) return
     epsi = epsi-ptd
  enddo
  write(6,*) ' WARNING!'
  write(6,*) ' in-situ temperature calculation has not converged.'
  stop
  return
end subroutine potit
!
!-------------------------------------------------------------------------------------
!
real function pttmpr(salz,temp,pres,rfpres)
  ! Berechnet aus dem Salzgehalt/psu (SALZ), der in-situ Temperatur/degC
  ! (TEMP) und dem in-situ Druck/dbar (PRES) die potentielle Temperatur/
  ! degC (PTTMPR) bezogen auf den Referenzdruck/dbar (RFPRES). Es wird
  ! ein Runge-Kutta Verfahren vierter Ordnung verwendet.
  ! Checkwert: PTTMPR = 36.89073 DegC
  !       fuer SALZ   =    40.0 psu
  !            TEMP   =    40.0 DegC
  !            PRES   = 10000.000 dbar
  !            RFPRES =     0.000 dbar

  data ct2 ,ct3  /0.29289322 ,  1.707106781/
  data cq2a,cq2b /0.58578644 ,  0.121320344/
  data cq3a,cq3b /3.414213562, -4.121320344/

  real salz,temp,pres,rfpres
  real p,t,dp,dt,q,ct2,ct3,cq2a,cq2b,cq3a,cq3b
  real adlprt

  p  = pres
  t  = temp
  dp = rfpres-pres
  dt = dp*adlprt(salz,t,p)
  t  = t +0.5*dt
  q = dt
  p  = p +0.5*dp
  dt = dp*adlprt(salz,t,p)
  t  = t + ct2*(dt-q)
  q  = cq2a*dt + cq2b*q
  dt = dp*adlprt(salz,t,p)
  t  = t + ct3*(dt-q)
  q  = cq3a*dt + cq3b*q
  p  = rfpres
  dt = dp*adlprt(salz,t,p)

  pttmpr = t + (dt-q-q)/6.0

end function pttmpr
!
!-------------------------------------------------------------------------------------
!
real function adlprt(salz,temp,pres)
  ! Berechnet aus dem Salzgehalt/psu (SALZ), der in-situ Temperatur/degC
  ! (TEMP) und dem in-situ Druck/dbar (PRES) den adiabatischen Temperatur-
  ! gradienten/(K Dbar^-1) ADLPRT.
  ! Checkwert: ADLPRT =     3.255976E-4 K dbar^-1
  !       fuer SALZ   =    40.0 psu
  !            TEMP   =    40.0 DegC
  !            PRES   = 10000.000 dbar

  real salz,temp,pres
  real s0,a0,a1,a2,a3,b0,b1,c0,c1,c2,c3,d0,d1,e0,e1,e2,ds

  data s0 /35.0/
  data a0,a1,a2,a3 /3.5803E-5, 8.5258E-6, -6.8360E-8, 6.6228E-10/
  data b0,b1       /1.8932E-6, -4.2393E-8/
  data c0,c1,c2,c3 /1.8741E-8, -6.7795E-10, 8.7330E-12, -5.4481E-14/
  data d0,d1       /-1.1351E-10, 2.7759E-12/
  data e0,e1,e2    /-4.6206E-13,  1.8676E-14, -2.1687E-16/

  ds = salz-s0
  adlprt = ( ( (e2*temp + e1)*temp + e0 )*pres                     &
       + ( (d1*temp + d0)*ds                                  &
       + ( (c3*temp + c2)*temp + c1 )*temp + c0 ) )*pres   &
       + (b1*temp + b0)*ds +  ( (a3*temp + a2)*temp + a1 )*temp + a0

END function adlprt
!
!----------------------------------------------------------------------------------------
!
subroutine cavity_heat_water_fluxes_2eq
  ! Compute the heat and freshwater fluxes under ice cavity
  ! using simple 2equ.
  !
  ! Coded by Adriana Huerta-Casas
  ! Reviewed by Qiang Wang
  !--------------------------------------------------------

  use o_mesh
  use o_param
  use o_array, only: heat_flux, water_flux, tracer
  use i_array, only: net_heat_flux, fresh_wa_flux
  use g_config
  use g_parfe
  implicit none

  integer        :: m, n, row
  real(kind=8)   :: gama, L, aux
  real(kind=8)   :: c2, c3, c4, c5, c6
  real(kind=8)   :: t_i, s_i, p, t_fz

  ! parameter for computing heat and water fluxes
  gama = 1.0e-4     ! heat exchange velocity [m/s]
  L    = 334000.    ! water to ice latent heat [J/Kg], same as used by the ice model

  ! parameter for computing freezing temperature (UNESCO 1983 equ.)
  c3 = 1.710523e-3
  c4 = -2.154996e-4
  c5 = -0.0575
  c6 = -7.53e-4

  do n=1,ToDim_nod2D      
     if(cavity_flag_nod2d(n)==0) cycle   
     row=nod3d_below_nod2d(1,n)
     t_i = tracer(row,1)	
     s_i = tracer(row,2)
     t_fz = c3*(s_i**(3./2.)) + c4*(s_i**2) + c5*s_i + c6*abs(coord_nod3d(3,row))
     
     heat_flux(n)=vcpw*gama*(t_i - t_fz)  ! Hunter2006 used cpw=3974J/Kg (*rhowat)
     water_flux(n) = -1.0*heat_flux(n)/(L*1000.0)  

     !for saving to output:
     net_heat_flux(n)=-heat_flux(n)
     fresh_wa_flux(n)=-water_flux(n)
  enddo

end subroutine cavity_heat_water_fluxes_2eq
!
!----------------------------------------------------------------------------------------
!
subroutine cavity_momentum_fluxes
  ! Compute the momentum fluxes under ice cavity
  ! Moved to this separated routine by Qiang, 20.1.2012

  use o_mesh
  use o_param
  use o_array, only: stress_x, stress_y
  use i_array, only: u_w, v_w
  use i_dyn_parms
  use g_config
  use g_parfe
  implicit none

  integer        :: m, n, row
  real(kind=8)   :: gama, L, aux
  real(kind=8)   :: c2, c3, c4, c5, c6
  real(kind=8)   :: t_i, s_i, p, t_fz

  do n=1,ToDim_nod2D      
     if(cavity_flag_nod2d(n)==0) cycle   
     row=nod3d_below_nod2d(1,n)

     ! momentum stress:

     ! need to check the sensitivity to the drag coefficient
     ! here I use the bottom stress coefficient, which is 3e-3, for this FO2 work.

     aux=sqrt(u_w(n)**2+v_w(n)**2)*rho0*C_d   
     stress_x(n)=-aux*u_w(n)
     stress_y(n)=-aux*v_w(n)

  enddo

end subroutine cavity_momentum_fluxes
!
!----------------------------------------------------------------------------------------
!
