!1) modules for forcing parameters and arrays
!2) modules for updating forcing in time and interpolation in space.
!------------------------------------------------------------------------

module g_forcing_param
  implicit none
  save   

  ! *** exchange coefficients ***
  real*8    :: Ce_atm_oce=1.75e-3 ! exchange coeff. of latent heat over open water
  real*8    :: Ch_atm_oce=1.75e-3 ! exchange coeff. of sensible heat over open water
  real*8    :: Cd_atm_oce=1.0e-3  ! drag coefficient between atmosphere and water

  real*8    :: Ce_atm_ice=1.75e-3 ! exchange coeff. of latent heat over ice
  real*8    :: Ch_atm_ice=1.75e-3 ! exchange coeff. of sensible heat over ice
  real*8    :: Cd_atm_ice=1.32e-3 ! drag coefficient between atmosphere and ice 

  namelist /forcing_exchange_coeff/ Ce_atm_oce, Ch_atm_oce, Cd_atm_oce, &
       Ce_atm_ice, Ch_atm_ice, Cd_atm_ice


  ! *** forcing source and type ***
  character(10)                 :: wind_data_source='CORE2'
  character(10)                 :: rad_data_source='CORE2'
  character(10)                 :: precip_data_source='CORE2'
  character(10)                 :: runoff_data_source='CORE2'
  character(10)                 :: sss_data_source='CORE2'
  integer                       :: wind_ttp_ind=1
  integer                       :: rad_ttp_ind=2
  integer                       :: precip_ttp_ind=3
  integer                       :: runoff_ttp_ind=0
  integer                       :: sss_ttp_ind=4


  namelist /forcing_source/ wind_data_source, rad_data_source, precip_data_source, &
       runoff_data_source, sss_data_source, wind_ttp_ind, rad_ttp_ind, precip_ttp_ind, &
       runoff_ttp_ind, sss_ttp_ind

  
  ! *** forcing spacial resolution ***
  character(5)        	:: forcing_grid='other'       !'T62', 'other'  
  real(kind=8)   	:: forcing_dx=0.4
  real(kind=8)   	:: forcing_dy=0.4
  integer        	:: forcing_ni=900
  integer        	:: forcing_nj=451  
  
  namelist /forcing_reso/ forcing_grid, forcing_dx, forcing_dy, &
       forcing_ni, forcing_nj
  

  ! *** coefficients in bulk formulae ***
  logical                       :: AOMIP_drag_coeff=.false.
  logical                       :: ncar_bulk_formulae=.false.
 ! _____________________________________________________C.Danek 16.12.2015___
 !|°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°|
 !|___________________________________________________________________________|
  real(kind=8)                   :: ncar_bulk_z_wind=10.0
  real(kind=8)                   :: ncar_bulk_z_tair=10.0
  ! namelist /forcing_bulk/ AOMIP_drag_coeff, ncar_bulk_formulae
  namelist /forcing_bulk/ AOMIP_drag_coeff, ncar_bulk_formulae, ncar_bulk_z_wind, ncar_bulk_z_tair
 !|___________________________________________________________________________|

  ! *** add land ice melt water ***
  logical                       :: use_landice_water=.false.
  integer                       :: landice_start_mon=1
  integer                       :: landice_end_mon=12

  namelist /land_ice/ use_landice_water, landice_start_mon, landice_end_mon

end module g_forcing_param
!
!----------------------------------------------------------------------------
!
module g_forcing_arrays
  implicit none
  save    

  ! forcing arrays
  real(kind=8), allocatable, dimension(:)         :: u_wind, v_wind 
  real(kind=8), allocatable, dimension(:)         :: Tair, shum
  real(kind=8), allocatable, dimension(:,:)       :: u_wind_t, v_wind_t 
  real(kind=8), allocatable, dimension(:,:)       :: Tair_t, shum_t
  real(kind=8), allocatable, dimension(:)         :: shortwave, longwave
  real(kind=8), allocatable, dimension(:)         :: prec_rain, prec_snow
  real(kind=8), allocatable, dimension(:)         :: runoff, evaporation
  ! _____________________________________________________P.Scholz 08.07.2015___
  !|°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°|
  !|___________________________________________________________________________|
  real(kind=8), allocatable, dimension(:,:)       :: shortwave_t, longwave_t
  real(kind=8), allocatable, dimension(:,:)       :: prec_rain_t, evaporation_t
  !|___________________________________________________________________________|
  
  real(kind=8), allocatable, dimension(:)         :: cloudiness, Pair
  
  real(kind=8), allocatable, dimension(:)         :: runoff_landice
  real(kind=8)                                    :: landice_season(12)

  ! shortwave penetration
  real(kind=8), allocatable, dimension(:)         :: chl, sw_3d

  real(kind=8), allocatable, dimension(:)         :: thdgr, thdgrsn, flice
  real(kind=8), allocatable, dimension(:)         :: olat_heat, osen_heat, olwout

  ! drag coefficient Cd_atm_oce and transfer coefficients for evaporation
  ! Ce_atm_oce and sensible heat Ch_atm_oce between atmosphere and ocean
  real(kind=8), allocatable, dimension(:)	  :: Cd_atm_oce_arr
  real(kind=8), allocatable, dimension(:)	  :: Ch_atm_oce_arr
  real(kind=8), allocatable, dimension(:)	  :: Ce_atm_oce_arr

  ! drag coefficient Cd_atm_oce between atmosphere and ice
  real(kind=8), allocatable, dimension(:)	  :: Cd_atm_ice_arr

end module g_forcing_arrays
!
!----------------------------------------------------------------------------
!
module g_forcing_index
  ! Control parameters for updating forcing in time 	
  !
  ! Coded by Qiang Wang
  ! Uncomment the part required for interpolation in time by Qiang Wang, 12,Oct. 2010
  ! Recover the interpolation for 6hourly data, Qiang, 07.02.2012
  ! Reviewed by ??
  ! --------------------------------------------------------
  
  implicit none
  save

  ! arrays for temporal interpolation
  integer                                         :: update_forcing_flag(4)
  integer                                         :: forcing_rec(4)
  real(kind=8)                                    :: interp_coef(4)

contains

  subroutine forcing_index
    use g_clock
    use g_forcing_param
    ! ===================================================
    ! Load modules g_parfe and g_config for providing 
    ! 'mype' and 'istep' this subroutine
    ! C. Danek, September 2015
    use g_parfe
    use g_config
    ! ===================================================
    implicit none

    real(kind=8)          :: sixhour_sec, threehour_sec
    real(kind=8)          :: oneday_sec
    real(kind=8)          :: modtimeold, modtimenew

    data sixhour_sec /21600.0/, threehour_sec /10800.0/, oneday_sec /86400.0/

    ! ===================================================
    ! Adjust forcing time index for NCEP 6-hourly data
    ! C. Danek, September 2015
    ! The following condition needs a better generalization:
    if ((wind_data_source=='NCEP' .and. wind_ttp_ind==1) .or. (rad_data_source=='NCEP' .and. rad_ttp_ind==1) .or. (precip_data_source=='NCEP' .and. precip_ttp_ind==1)) then
        
        !modtimeold=mod(timeold,oneday_sec)
        !modtimenew=mod(timenew,oneday_sec)
        !if(mype==0) write(*,*) 'istep=', istep, ', timeold=', timeold, ', modtimeold=', modtimeold, ', timenew=', timenew, ', modtimenew=', modtimenew, ', daynew=', daynew

        ! if update forcing or not
        update_forcing_flag=0

        if(mod(timenew, sixhour_sec)==0.0) update_forcing_flag(1)=1
        !if(mype==0) write(*,*) 'forcing_index(): update_forcing_flag(1) = ', update_forcing_flag(1)
        
        !if(modtimeold==0.0) update_forcing_flag(2)=1
        !if(day_in_month==1 .and. modtimeold==0.0) update_forcing_flag(3:4)=1
        
        forcing_rec(1) = 2+int((timenew)/sixhour_sec)+4*(daynew-1)
        !if(mype==0) write(*,*) 'forcing_index(): forcing_rec(1) = ', forcing_rec(1)
        forcing_rec(2) = daynew
        forcing_rec(3) = month
        forcing_rec(4) = month
        
        ! interpolation coefficients
        interp_coef(1)=mod(timenew, sixhour_sec)/sixhour_sec
        !if(mype==0) write(*,*) 'forcing_index(): interp_coef(1) = ', interp_coef(1)
        if (mod(istep,logfile_outfreq)==0 .and. mype==0) then
            write(*,*) 'forcing_index(): istep=', istep, ', timenew [s]=', timenew, ', timenew [h]=', timenew/3600, ', update_forcing_flag(1)=', update_forcing_flag(1), ', daynew=', daynew, ', forcing_rec(1)=', forcing_rec(1), ', interp_coef(1)=', interp_coef(1)
        end if
    ! ===================================================
    else
        
        modtimeold=mod(timeold,oneday_sec)
        
        ! if update forcing or not
        update_forcing_flag=0
        
        if(mod(timeold+threehour_sec, sixhour_sec)==0.0) update_forcing_flag(1)=1
        if(modtimeold==0.0) update_forcing_flag(2)=1
        if(day_in_month==1 .and. modtimeold==0.0) update_forcing_flag(3:4)=1
        
        ! which record to read in
        forcing_rec(1) = 1+int((modtimeold+threehour_sec)/sixhour_sec)+4*(daynew-1)
        forcing_rec(2) = daynew
        forcing_rec(3) = month
        forcing_rec(4) = month
        
        ! interpolation coefficients
        interp_coef(1)=mod(timeold+threehour_sec, sixhour_sec)/sixhour_sec
    
    end if
    
    if(interp_coef(1)>1. .or. interp_coef(1)<0.) then
       write(*,*) 'error in interp_coef'
       call par_ex
       stop
    end if

!!$    interp_coef(2)=modtimeold/oneday_sec
!!$    interp_coef(3)=(day_in_month-1.0+modtimeold/oneday_sec) &
!!$         /real(num_day_in_month(fleapyear,month))
!!$    interp_coef(4)=interp_coef(3)
!!$
!!$    if(any(interp_coef>1.) .or. any(interp_coef<0.)) then
!!$       write(*,*) 'error in interp_coef'
!!$       stop
!!$    end if

  end subroutine forcing_index

end module g_forcing_index
!
!----------------------------------------------------------------------------
!
module g_forcing_interp
  ! This module prepare the weights for interpolating 
  ! forcing data Tair, humidity, wind velocities,
  ! precipitation, radiations, etc.
  !
  ! Taken the structure in the old version, but using different 
  ! way of weighting. Currently using bilinear interpolation.
  !
  ! Modified by Qiang Wang for bilinear interpolation
  ! Reviewed by ??
  !------------------------------------------------------------------	

  integer, allocatable      :: lint_ind(:,:,:)
  real(kind=8), allocatable :: lint_weight(:,:)

contains

  !------------------------------------------------------------------
  subroutine  init_forcing_interp

    ! routine to calculate neighbours and weights for linear interpolation
    !
    ! required information
    !    xmod(nmp)  longitudes of model point on geographical grid in degree (no order assumed)
    !    ymod(nmp)  latitudes of model points on geographical grid in degree (no order assumed)
    !         nmp   number of model points where data are needed
    !    cx(ni)     longitudes of data points on regular geographical grid
    !               by now must be in range[0,..360] in ascending order
    !    cy(nj)     latitudes of data points on regular geographical grid 
    !               by now must be in range [-90,..90] in ascending order
    !
    ! OUTPUT
    !    lint_ind(4,2,nmp)   /i,j/ adress for four neighbors of each model node
    !    lint_weight(4,nmp)  interpolation weights

    use o_mesh
    use o_param
    use g_forcing_param
    use g_config
    use g_parfe
    implicit none

    integer	 		:: ni, nj
    integer     		:: i, ii, j, n, row, n2
    real(kind=8)      		:: rlon, rlat, aux
    real(kind=8),allocatable  	:: cx(:), cy(:)
    real(kind=8)      		:: xmod(myDim_nod2D+eDim_nod2D), ymod(myDim_nod2D+eDim_nod2D)
    real(kind=8)        	:: wt(4)

    n2=myDim_nod2D+eDim_nod2D     

    allocate(lint_ind(4,2,n2))   
    allocate(lint_weight(4,n2))  

    if(forcing_grid=='T62') then
       forcing_ni=192
       forcing_nj=94
    end if
    ni=forcing_ni
    nj=forcing_nj

    allocate(cx(ni), cy(nj))

    if(forcing_grid=='T62') then

       cy = (/-88.542, -86.6531, -84.7532, -82.8508, -80.9473, -79.0435, &
            -77.1394, -75.2351, -73.3307, -71.4262, -69.5217, -67.6171, &
            -65.7125, -63.8079, -61.9033, -59.9986, -58.0939, -56.1893, &
            -54.2846, -52.3799, -50.4752, -48.5705, -46.6658, -44.7611,&
            -42.8564, -40.9517, -39.0470, -37.1422, -35.2375, -33.3328, &
            -31.4281, -29.5234, -27.6186, -25.7139, -23.8092, -21.9044, &
            -19.9997, -18.0950, -16.1902, -14.2855, -12.3808, -10.47604, &
            -8.57131, -6.66657, -4.76184, -2.8571, -0.952368, 0.952368, &
            2.8571, 4.76184, 6.66657, 8.57131, 10.47604, 12.3808, &
            14.2855, 16.1902, 18.095, 19.9997, 21.9044, 23.8092, &
            25.7139, 27.6186, 29.5234, 31.4281, 33.3328, 35.2375,&
            37.1422, 39.047,  40.9517, 42.8564, 44.7611, 46.6658,&
            48.5705, 50.4752, 52.3799, 54.2846, 56.1893, 58.0939,&
            59.9986, 61.9033, 63.8079, 65.7125, 67.6171, 69.5217, &
            71.4262, 73.3307, 75.2351, 77.1394, 79.0435, 80.9473, &
            82.8508, 84.7532, 86.6531, 88.542 /)

       cx(1)=0.0
       do i=2,ni
          cx(i)=cx(i-1)+1.875
       enddo

    elseif(forcing_grid=='other') then

       cy(1)=-90.0
       do j=2,nj
          cy(j)=cy(j-1)+forcing_dy
       enddo

       cx(1)=0.0
       do i=2,ni
          cx(i)=cx(i-1)+forcing_dx
       enddo

    end if

!!$    ! some checks, range of cx and cy
!!$    if(cx(ni)-cx(1).gt.360.) then
!!$       write(*,*) 'Forcing interp: x-range gt 360'
!!$       write(*,*) 'cx(ni):', cx(ni)
!!$       call abort
!!$    endif
!!$    if(cy(nj)-cy(1).gt.180.) then
!!$       write(*,*) 'Forcing interp: y-range gt 180'
!!$       write(*,*) 'cy(nj):', cy(nj)
!!$       call abort
!!$    endif
!!$    if(cx(1).ge.360.0) then
!!$       aux=int(cx(1)/360.)*360.
!!$       do i=1,ni
!!$          cx(i)=cx(i)-aux
!!$       enddo
!!$    endif
!!$    if(cx(ni).gt.360.) call abort

    ! in the following we need cx and cy in unit radian
    cx=cx*rad
    cy=cy*rad

    ! model grid coordinate (in radian, between 0 and 2*pi)
    do row=1,n2                    
       xmod(row)=geolon2d(row)
       ymod(row)=geolat2d(row)
       if (xmod(row)<0.0) xmod(row)=2*pi+xmod(row)	
    enddo

    ! linear interpolation: nodes and weight
    do row=1,n2
       if(xmod(row)<cx(ni)) then
          do i=1,ni
             if(xmod(row)<cx(i)) then
                lint_ind(1,1,row)=i-1
                lint_ind(2,1,row)=i-1
                lint_ind(3,1,row)=i
                lint_ind(4,1,row)=i
                aux=(cx(i)-xmod(row))/(cx(i)-cx(i-1))
                exit
             end if
          end do
       else
          lint_ind(1,1,row)=ni
          lint_ind(2,1,row)=ni
          lint_ind(3,1,row)=1
          lint_ind(4,1,row)=1
          aux=(360.0*rad-xmod(row))/(360.0*rad-cx(ni))
       end if
       wt(1)=aux
       wt(2)=aux
       wt(3)=1.0_8-aux
       wt(4)=1.0_8-aux

       if(ymod(row)<cy(nj)) then
          do j=1,nj
             if(ymod(row)<cy(j)) then
                lint_ind(1,2,row)=j-1
                lint_ind(2,2,row)=j
                lint_ind(3,2,row)=j
                lint_ind(4,2,row)=j-1
                aux=(cy(j)-ymod(row))/(cy(j)-cy(j-1))
                exit
             end if
          end do
       else
          lint_ind(1,2,row)=nj
          lint_ind(2,2,row)=nj
          lint_ind(3,2,row)=nj
          lint_ind(4,2,row)=nj
          aux=1.0_8
       end if
       lint_weight(1,row)=wt(1)*aux
       lint_weight(2,row)=wt(2)*(1.0_8-aux)
       lint_weight(3,row)=wt(3)*(1.0_8-aux)
       lint_weight(4,row)=wt(4)*aux
    end do

    deallocate(cx, cy)

    if(mype==0)  write(*,*) 'weights for interpolating forcing / 2D fields computed'
    return     
  end subroutine init_forcing_interp


  !------------------------------------------------------------------------
  subroutine forcing_linear_ip(zd,idim,jdim,ind,weights,zi,nmpt)
    !  this subroutine interpolates data using prepared weights- 
    !  see subroutine init_linear_forcing_interp
    !
    !  INPUT
    !        zd(idim,jdim)         available data set
    !        nmpt                  number of model positions, where data are wanted
    !        indx(4,2,nmpt)        i,j index of four neighbors of each node
    !        weights(4,nmpt)       interpolation weights
    !        
    !  OUTPUT
    !        zi(nmpt)              array of interpolated values for model points

    use g_parfe
    implicit none                                             

    integer      :: idim, jdim, nmpt                          
    integer      :: ind(4,2,nmpt)
    integer      :: i, n                      
    real(kind=8) :: zd(idim,jdim), zi(nmpt)
    real(kind=8) :: weights(4,nmpt)
    real(kind=8) :: fx

    do n=1,nmpt
       fx=0.0
       do i=1,4
          fx=fx+weights(i,n)*zd(ind(i,1,n),ind(i,2,n))
       enddo
       zi(n)=fx
    enddo

    return
  end subroutine forcing_linear_ip

end module g_forcing_interp
