! Routines for initializing and updating ocean surface forcing fields
!-------------------------------------------------------------------------

subroutine init_atm_forcing
  ! read in forcing data that are constant in time, 
  ! and initialize those which are 6-hourly.
  ! Data in other frequency do not need initialization.
  !
  ! Coded by Qiang Wang
  ! Reviewed by ??

  use o_PARAM
  use o_MESH
  use o_array
  use i_therm_parms
  use i_array
  use g_forcing_param
  use g_forcing_arrays
  use g_forcing_index
  use g_forcing_interp
  use g_read_CORE_NetCDF
  use g_read_NCEP_NetCDF
  use g_read_ECMWF_NetCDF
  use g_read_other_NetCDF
  use g_clock
  use g_parfe
  use g_config
  implicit none
  !
  integer        		             :: nci, ncj
  integer                   		     :: itime, i, k, n2, year_first_rec
  integer                                    :: readtype
  character(80)             		     :: file
  character(15)             		     :: vari, filevari
  character(4)				     :: fileyear
  real(kind=8),	dimension(:,:), allocatable  :: array_nc, array_nc2
  real(kind=8), dimension(nod2D)    	     :: array_fe
  real(kind=8), allocatable                  :: aux(:) 
  logical                                    :: check_dummy

  if(forcing_grid=='T62') then
     forcing_ni=192
     forcing_nj=94
  end if  
  nci=forcing_ni
  ncj=forcing_nj
  
  allocate(array_nc(nci, ncj), array_nc2(nci, ncj))

  n2=toDim_nod2D
       
  ! predefinition/correction

  ! for the CORE case:
  if(wind_data_source=='CORE1' .or. wind_data_source=='CORE2') wind_ttp_ind=1
  if(rad_data_source=='CORE1' .or. rad_data_source=='CORE2') rad_ttp_ind=2
  if(precip_data_source=='CORE1' .or. precip_data_source=='CORE2') precip_ttp_ind=3
  if(runoff_data_source=='CORE1' .or. runoff_data_source=='CORE2') runoff_ttp_ind=0
  if(sss_data_source=='CORE1' .or. sss_data_source=='CORE2') sss_ttp_ind=4

  !==========================================================================
  ! forcing index
  call forcing_index

  !==========================================================================
  ! runoff    

  if(runoff_data_source=='CORE1' .or. runoff_data_source=='CORE2' ) then

     ! runoff in CORE is constant in time

     ! Warning: For a global mesh, conservative scheme is to be updated!!

     file=trim(ForcingDataPath)//trim(runoff_data_source)//'/runoff.nc'
     vari='Foxx_o_roff'
     check_dummy=.false.

     itime=1
     call read_other_NetCDF(file, vari, itime, runoff, check_dummy) 
     if (mod(istep,logfile_outfreq)==0 .and. mype==0) then
         write(*,*) 'init_atm_forcing(): istep=', istep, ', itime=', itime, ', file=', file
         !write(*,*) 'read_other_NetCDF(): mean(runoff)=', sum(runoff)/size(runoff)
     end if
     runoff=runoff/1000.0  ! Kg/s/m2 --> m/s
  end if


  !==========================================================================
  ! sss restoring

  if(restore_s_surf>0.) then
     if(sss_data_source=='AAOMIP') then

        ! taking the annual mean of PHC2 SSS

        file=trim(ForcingDataPath)//'CORE2'//'/PHC2_salx.nc'
        vari='SALT'
        check_dummy=.true.

        Ssurf=0.0

        allocate(aux(n2))
        do itime=1,12
           call read_other_NetCDF(file, vari, itime, aux, check_dummy) 
           Ssurf=Ssurf+aux
        end do
        Ssurf=Ssurf/12.0
        deallocate(aux)
     endif
  end if


  !==========================================================================
  ! wind u and v, Tair, and shum    
  ! Only when using 6-hourly data, we do interpolation in time for them


  if(wind_data_source=='NCEP' .and. wind_ttp_ind==1) then

     fileyear=cyearnew
     itime=forcing_rec(wind_ttp_ind)-1
     year_first_rec=yearnew

     if(itime==0) then
        itime=4*ndpyr
        year_first_rec=yearnew-1
        write(fileyear,'(i4)') year_first_rec
     end if

     do i=1,2

        ! 10-m wind m/s ----------------------------------------

        vari='uwnd'
        file=trim(ForcingDataPath)//'NCEP_6hourly/'//'uwnd.10m.gauss.'//fileyear//'.nc'
        if (mod(istep,logfile_outfreq)==0 .and. mype==0) write(*,*) 'init_atm_forcing(): istep=', istep, ', itime=', itime,   ', file=', file
        call read_NCEPv2_NetCDF(file, vari, itime, array_nc)
           !if (mod(istep,logfile_outfreq)==0 .and. mype==0) then
           !    write(*,*) 'init_atm_forcing(): istep=', istep, ', itime=', itime, ', ToDim_nod2d=', ToDim_nod2d, ', size(array_nc)=', size(array_nc), ', file=', file
           !    write(*,*) 'read_NCEPv2_NetCDF(): mean(uwnd)=', sum(array_nc)/size(array_nc)
           !end if
        call upside_down(array_nc,nci,ncj)
 
        vari='vwnd'
        file=trim(ForcingDataPath)//'NCEP_6hourly/'//'vwnd.10m.gauss.'//fileyear//'.nc'
        call read_NCEPv2_NetCDF(file, vari, itime, array_nc2)
           !if (mod(istep,logfile_outfreq)==0 .and. mype==0) then
           !    write(*,*) 'init_atm_forcing(): istep=', istep, ', itime=', itime, ', ToDim_nod2d=', ToDim_nod2d, ', size(array_nc2)=', size(array_nc2), ', file=', file
           !    write(*,*) 'read_NCEPv2_NetCDF(): mean(vwnd)=', sum(array_nc2)/size(array_nc2)
           !end if
        call upside_down(array_nc2,nci,ncj)
         ! rotate wind
        if(rotated_grid) call rotate_wind(array_nc, array_nc2, nci, ncj)

        ! interp wind to model grid
        call forcing_linear_ip(array_nc,nci,ncj,lint_ind,lint_weight,u_wind_t(i,:),n2)   
        call forcing_linear_ip(array_nc2,nci,ncj,lint_ind,lint_weight,v_wind_t(i,:),n2) 
 
		! 2-m temperature --------------------------------------

        vari='air'
        file=trim(ForcingDataPath)//'NCEP_6hourly/'//'air.2m.gauss.'//fileyear//'.nc'
        call read_NCEPv2_NetCDF(file, vari, itime, array_nc)
           !if (mod(istep,logfile_outfreq)==0 .and. mype==0) then
           !    write(*,*) 'init_atm_forcing(): istep=', istep, ', itime=', itime, ', ToDim_nod2d=', ToDim_nod2d, ', size(array_nc)=', size(array_nc), ', file=', file
           !    write(*,*) 'read_NCEPv2_NetCDF(): mean(air)=', sum(array_nc)/size(array_nc)
           !end if
        call upside_down(array_nc,nci,ncj)
        call forcing_linear_ip(array_nc,nci,ncj,lint_ind,lint_weight,Tair_t(i,:),n2)   
        Tair_t(i,:)=Tair_t(i,:)-tmelt  ! Kelvin --> degree Celcius

        ! 2 m specific humdity  Kg/Kg -------------------------

        vari='shum'
        file=trim(ForcingDataPath)//'NCEP_6hourly/'//'shum.2m.gauss.'//fileyear//'.nc'
        call read_NCEPv2_NetCDF(file, vari, itime, array_nc)
           !if (mod(istep,logfile_outfreq)==0 .and. mype==0) then
           !    write(*,*) 'init_atm_forcing(): istep=', istep, ', itime=', itime, ', ToDim_nod2d=', ToDim_nod2d, ', size(array_nc)=', size(array_nc), ', file=', file
           !    write(*,*) 'read_NCEPv2_NetCDF(): mean(shum)=', sum(array_nc)/size(array_nc)
           !end if
        call upside_down(array_nc,nci,ncj)
        call forcing_linear_ip(array_nc,nci,ncj,lint_ind,lint_weight,shum_t(i,:),n2)    

        itime=itime+1
        if (itime>4*ndpyr) then	! to read the first record of the following year
           itime=1
           write(fileyear,'(i4)') year_first_rec+1
        endif

     end do

  elseif(wind_data_source=='ECMWF' .and. wind_ttp_ind==1) then

     fileyear=cyearnew
     itime=forcing_rec(wind_ttp_ind)-1
     year_first_rec=yearnew

     if(itime==0) then
        itime=4*ndpyr
        year_first_rec=yearnew-1
        write(fileyear,'(i4)') year_first_rec
     end if

     do i=1,2

        ! 10-m wind m/s ----------------------------------------

        vari='u10'
        file=trim(ForcingDataPath)//'ECMWF/'//'u10m_'//fileyear//'.nc'
        call read_ECMWF_NetCDF(file, vari, itime, array_nc, nci, ncj)
        call upside_down(array_nc,nci,ncj)

        vari='v10'
        file=trim(ForcingDataPath)//'ECMWF/'//'v10m_'//fileyear//'.nc'
        call read_ECMWF_NetCDF(file, vari, itime, array_nc2, nci, ncj)
        call upside_down(array_nc2,nci,ncj)

        ! rotate wind
        if(rotated_grid) call rotate_wind(array_nc, array_nc2, nci, ncj)

        ! interp wind to model grid
        call forcing_linear_ip(array_nc,nci,ncj,lint_ind,lint_weight,u_wind_t(i,:),n2)   
        call forcing_linear_ip(array_nc2,nci,ncj,lint_ind,lint_weight,v_wind_t(i,:),n2) 

        ! 2-m temperature --------------------------------------

        vari='t2m'
        file=trim(ForcingDataPath)//'ECMWF/'//'T2m_'//fileyear//'.nc'
        call read_ECMWF_NetCDF(file, vari, itime, array_nc, nci, ncj)
        call upside_down(array_nc,nci,ncj)
        call forcing_linear_ip(array_nc,nci,ncj,lint_ind,lint_weight,Tair_t(i,:),n2)   
        Tair_t(i,:)=Tair_t(i,:)-tmelt  ! Kelvin --> degree Celcius

        ! 2 m specific humdity  Kg/Kg -------------------------

        vari='q'
        file=trim(ForcingDataPath)//'ECMWF/'//'q_'//fileyear//'.nc'
        call read_ECMWF_NetCDF(file, vari, itime, array_nc, nci, ncj)
        call upside_down(array_nc,nci,ncj)
        call forcing_linear_ip(array_nc,nci,ncj,lint_ind,lint_weight,shum_t(i,:),n2)    

        itime=itime+1
        if (itime>4*ndpyr) then	! to read the first record of the following year
           itime=1
           write(fileyear,'(i4)') year_first_rec+1
        endif

     end do

  elseif(wind_data_source=='CORE2' .and. wind_ttp_ind==1) then

     fileyear=cyearnew
     itime=forcing_rec(wind_ttp_ind)-1
     year_first_rec=yearnew

     if(itime==0) then
        itime=4*ndpyr
        year_first_rec=yearnew-1
        write(fileyear,'(i4)') year_first_rec
     end if

     do i=1,2

        ! 10-m wind m/s ----------------------------------------

        filevari='u_10.'
        file=trim(ForcingDataPath)//'CORE2/'//trim(filevari)//fileyear//'.nc'
        vari='U_10_MOD'
        if (mod(istep,logfile_outfreq)==0 .and. mype==0) write(*,*) 'init_atm_forcing(): istep=', istep, ', itime=', itime, ', file=', file
        call read_CORE_NetCDF(file, vari, itime, array_nc)

        filevari='v_10.'
        file=trim(ForcingDataPath)//'CORE2/'//trim(filevari)//fileyear//'.nc'
        vari='V_10_MOD'   
        call read_CORE_NetCDF(file, vari, itime, array_nc2)

        ! rotate wind
        if(rotated_grid) call rotate_wind(array_nc, array_nc2, nci, ncj)

        ! interp wind to model grid
        call forcing_linear_ip(array_nc,nci,ncj,lint_ind,lint_weight,u_wind_t(i,:),n2) 
        call forcing_linear_ip(array_nc2,nci,ncj,lint_ind,lint_weight,v_wind_t(i,:),n2) 

        ! 10-m temperature -------------------------------------

        filevari='t_10.'
        file=trim(ForcingDataPath)//'CORE2/'//trim(filevari)//fileyear//'.nc'
        vari='T_10_MOD'
        call read_CORE_NetCDF(file, vari, itime, array_nc)
        call forcing_linear_ip(array_nc,nci,ncj,lint_ind,lint_weight,Tair_t(i,:),n2) 
        Tair_t(i,:)=Tair_t(i,:)-tmelt  ! Kelvin --> degree celcium

        ! 10 m specific humdity  Kg/Kg -------------------------

        filevari='q_10.'
        file=trim(ForcingDataPath)//'CORE2/'//trim(filevari)//fileyear//'.nc'
        vari='Q_10_MOD'
        call read_CORE_NetCDF(file, vari, itime, array_nc)
        call forcing_linear_ip(array_nc,nci,ncj,lint_ind,lint_weight,shum_t(i,:),n2) 

        itime=itime+1
        if (itime>4*ndpyr) then	! to read the first record of the following year
           itime=1
           write(fileyear,'(i4)') year_first_rec+1
        endif

     end do


  elseif(wind_data_source=='CORE1' .and. wind_ttp_ind==1) then

     itime=forcing_rec(wind_ttp_ind)-1

     if(itime==0) then
        itime=4*ndpyr
     end if

     do i=1,2

        ! 10-m wind m/s ----------------------------------------

        filevari='u_10'
        file=trim(ForcingDataPath)//'CORE1/'//trim(filevari)//'.nc'
        vari='U_10_MOD'
        call read_CORE_NetCDF(file, vari, itime, array_nc)

        filevari='v_10'
        file=trim(ForcingDataPath)//'CORE1/'//trim(filevari)//'.nc'
        vari='V_10_MOD'   
        call read_CORE_NetCDF(file, vari, itime, array_nc2)

        ! rotate wind
        if(rotated_grid) call rotate_wind(array_nc, array_nc2, nci, ncj)

        ! interp wind to model grid
        call forcing_linear_ip(array_nc,nci,ncj,lint_ind,lint_weight,u_wind_t(i,:),n2) 
        call forcing_linear_ip(array_nc2,nci,ncj,lint_ind,lint_weight,v_wind_t(i,:),n2) 

        ! 10-m temperature -------------------------------------

        filevari='t_10'
        file=trim(ForcingDataPath)//'CORE1/'//trim(filevari)//'.nc'
        vari='T_10_MOD'
        call read_CORE_NetCDF(file, vari, itime, array_nc)
        call forcing_linear_ip(array_nc,nci,ncj,lint_ind,lint_weight,Tair_t(i,:),n2) 
        Tair_t(i,:)=Tair_t(i,:)-tmelt  ! Kelvin --> degree celcium

        ! 10 m specific humdity  Kg/Kg -------------------------

        filevari='q_10'
        file=trim(ForcingDataPath)//'CORE1/'//trim(filevari)//'.nc'
        vari='Q_10_MOD'
        call read_CORE_NetCDF(file, vari, itime, array_nc)
        call forcing_linear_ip(array_nc,nci,ncj,lint_ind,lint_weight,shum_t(i,:),n2) 

        itime=itime+1
        if (itime>4*ndpyr) then	! to read the first record of the following year
           itime=1
        endif

     end do

  end if
  
  ! _____________________________________________________P.Scholz 09.07.2015___
  !|°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°|
  !|___6 hourly interpolation for  precipitation/evaporation___________________|
  ! if ncep precipitation is 6 hourly forcing do also interpolation in time 
  if(precip_data_source=='NCEP' .and. precip_ttp_ind==1) then

    fileyear=cyearnew
    itime=forcing_rec(wind_ttp_ind)-1
    year_first_rec=yearnew

	if(itime==0) then
        itime=4*ndpyr
        year_first_rec=yearnew-1
        write(fileyear,'(i4)') year_first_rec
    end if
	
	do i=1,2
		! precipitation
		vari='prate'
		file=trim(ForcingDataPath)//'NCEP_6hourly/'//'prate.sfc.gauss.'//fileyear//'.nc'
        call read_NCEPv2_NetCDF(file, vari, itime, array_nc)
        !if (mod(istep,logfile_outfreq)==0 .and. mype==0) then
        !    write(*,*) 'init_atm_forcing(): istep=', istep, ', itime=', itime, ', ToDim_nod2d=', ToDim_nod2d, ', size(array_nc)=', size(array_nc), ', file=', file
        !    write(*,*) 'read_NCEPv2_NetCDF(): mean(prate)=', sum(array_nc)/size(array_nc)
        !end if
		call upside_down(array_nc,nci,ncj)
		call forcing_linear_ip(array_nc,nci,ncj,lint_ind,lint_weight,prec_rain_t(i,:),n2)  
		prec_rain_t(i,:)=prec_rain_t(i,:)/1000.  ! mm/s --> m/s
		
		! evaporation, need m/s, calculate from latent heat 
		vari='lhtfl'
		file=trim(ForcingDataPath)//'NCEP_6hourly/'//'lhtfl.sfc.gauss.'//fileyear//'.nc'
		call read_NCEPv2_NetCDF(file, vari, itime, array_nc)
        !if (mod(istep,logfile_outfreq)==0 .and. mype==0) then
        !    write(*,*) 'init_atm_forcing(): istep=', istep, ', itime=', itime, ', ToDim_nod2d=', ToDim_nod2d, ', size(array_nc)=', size(array_nc), ', file=', file
        !    write(*,*) 'read_NCEPv2_NetCDF(): mean(lhtfl)=', sum(array_nc)/size(array_nc)
        !end if
		call upside_down(array_nc,nci,ncj)
		call forcing_linear_ip(array_nc,nci,ncj,lint_ind,lint_weight,evaporation_t(i,:),n2)
		evaporation_t(i,:)=evaporation_t(i,:)/2.5e6/1000.0  
		! evap [m/s]= Q_l [W/m2] / lh_evap [J/kg] / rho_w [kg/m3]

        itime=itime+1
		if (itime>4*ndpyr) then	! to read the first record of the following year
	        itime=1
		    write(fileyear,'(i4)') year_first_rec+1
		endif
	end do
  endif
  
  ! _____________________________________________________P.Scholz 09.07.2015___
  !|°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°|
  !|___6 hourly interpolation for  shortwave/longwave radiation________________|
  ! if ncep radiation is 6 hourly forcing do also interpolation in time 
  if(rad_data_source=='NCEP' .and. rad_ttp_ind==1) then

    fileyear=cyearnew
    itime=forcing_rec(wind_ttp_ind)-1
    year_first_rec=yearnew

	if(itime==0) then
        itime=4*ndpyr
        year_first_rec=yearnew-1
        write(fileyear,'(i4)') year_first_rec
    end if
	
	do i=1,2
		! short wave W/m2 ---------------------------------------
		vari='dswrf'
		file=trim(ForcingDataPath)//'NCEP_6hourly/'//'dswrf.sfc.gauss.'//fileyear//'.nc'
        call read_NCEPv2_NetCDF(file, vari, itime, array_nc)
        !if (mod(istep,logfile_outfreq)==0 .and. mype==0) then
        !    write(*,*) 'init_atm_forcing(): istep=', istep, ', itime=', itime, ', ToDim_nod2d=', ToDim_nod2d, ', size(array_nc)=', size(array_nc), ', file=', file
        !    write(*,*) 'read_NCEPv2_NetCDF(): mean(dswrf)=', sum(array_nc)/size(array_nc)
        !end if
		call upside_down(array_nc,nci,ncj)
		call forcing_linear_ip(array_nc,nci,ncj,lint_ind,lint_weight,shortwave_t(i,:),n2) 
		
		! long wave W/m2 ---------------------------------------
		vari='dlwrf'
		file=trim(ForcingDataPath)//'NCEP_6hourly/'//'dlwrf.sfc.gauss.'//fileyear//'.nc'
		call read_NCEPv2_NetCDF(file, vari, itime, array_nc)
        !if (mod(istep,logfile_outfreq)==0 .and. mype==0) then
        !    write(*,*) 'init_atm_forcing(): istep=', istep, ', itime=', itime, ', ToDim_nod2d=', ToDim_nod2d, ', size(array_nc)=', size(array_nc), ', file=', file
        !    write(*,*) 'read_NCEPv2_NetCDF(): mean(dlwrf)=', sum(array_nc)/size(array_nc)
        !end if
		call upside_down(array_nc,nci,ncj)
		call forcing_linear_ip(array_nc,nci,ncj,lint_ind,lint_weight,longwave_t(i,:),n2)  

        itime=itime+1
        if (itime>4*ndpyr) then    ! to read the first record of the following year
           itime=1
           write(fileyear,'(i4)') year_first_rec+1
        endif
	end do
  endif  
  ! _____________________________________________________P.Scholz 09.07.2015___
  !|°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°|
  !|___________________________________________________________________________|  

  deallocate(array_nc, array_nc2)

  if(mype==0) write(*,*) 'Forcing (constant and 6-hourly which need interp in time) read in.'

end subroutine init_atm_forcing
!
!------------------------------------------------------------------------------------------
!
subroutine update_atm_forcing
  ! update atmospheric forcing data
  !
  ! Coded by Qiang Wang
  ! Reviewed by ??
  !---------------------------------------------------------
  
  use o_PARAM
  use o_MESH
  use o_array
  use i_array
  use i_dyn_parms
  use i_therm_parms
  use g_forcing_param
  use g_forcing_arrays
  use g_forcing_index
  use g_parfe
  use g_clock
  use g_config
  implicit none

  integer		:: i
  real(kind=8)		:: i_coef, aux
  real(kind=8)		:: dux, dvy
  real              	:: t1, t2

  t1=MPI_Wtime()  

  ! first, read forcing data
  call read_new_atm_forcing

  ! second, do time interpolation 
  ! Only for wind/Tair/shum with 6-hourly data.
  ! For forcing with other frequency, I do not do interpolation in time,
  ! that is, they are already stored in the proper arrays.
  ! wind, Tair, shum
  if(wind_ttp_ind==1) then
     i_coef=interp_coef(wind_ttp_ind)
     !if (mod(istep,logfile_outfreq)==0 .and. mype==0) then
     !    write(*,*) 'update_atm_forcing(): istep=', istep, ', i_coef=', i_coef
     !end if
     do i=1,ToDim_nod2d                                        
        u_wind(i)=u_wind_t(1,i)+i_coef*(u_wind_t(2,i)-u_wind_t(1,i))
        v_wind(i)=v_wind_t(1,i)+i_coef*(v_wind_t(2,i)-v_wind_t(1,i))
        Tair(i)=Tair_t(1,i)+i_coef*(Tair_t(2,i)-Tair_t(1,i))
        shum(i)=shum_t(1,i)+i_coef*(shum_t(2,i)-shum_t(1,i))
     end do
 end if
  
  ! _____________________________________________________P.Scholz 09.07.2015___
  !|°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°|
  !|___6 hourly interpolation for  shortwave/longwave radiation________________|
  ! if ncep radiation is 6 hourly forcing do also interpolation in time 
  if(rad_ttp_ind==1) then
     i_coef=interp_coef(rad_ttp_ind)
     !if (mod(istep,logfile_outfreq)==0 .and. mype==0) then
     !    write(*,*) 'update_atm_forcing(): istep=', istep, ', i_coef=', i_coef, ' (shortwave, longwave)'
     !end if
     do i=1,ToDim_nod2d                                        
        shortwave(i)=shortwave_t(1,i)+i_coef*(shortwave_t(2,i)-shortwave_t(1,i))
        longwave(i)=longwave_t(1,i)+i_coef*(longwave_t(2,i)-longwave_t(1,i))
     end do
  end if
  
  ! _____________________________________________________P.Scholz 09.07.2015___
  !|°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°|
  !|___6 hourly interpolation for  precipitation/evaporation___________________|
  ! if ncep precipitation is 6 hourly forcing do also interpolation in time 
  if(precip_ttp_ind==1) then
     i_coef=interp_coef(precip_ttp_ind)
     !if (mod(istep,logfile_outfreq)==0 .and. mype==0) then
     !    write(*,*) 'update_atm_forcing(): istep=', istep, ', i_coef=', i_coef, ' (prec_rain, evaporation)'
     !end if
     do i=1,ToDim_nod2d                                        
        prec_rain(i)=prec_rain_t(1,i)+i_coef*(prec_rain_t(2,i)-prec_rain_t(1,i))
        evaporation(i)=evaporation_t(1,i)+i_coef*(evaporation_t(2,i)-evaporation_t(1,i))
     end do
  end if
  
  ! ___________________________________________________________________________
  !|°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°|
  !|___________________________________________________________________________|
  
  ! third, compute exchange coefficients
  ! 1) drag coefficient 
  if(AOMIP_drag_coeff) then
     call cal_wind_drag_coeff
  end if
  ! 2) drag coeff. and heat exchange coeff. over ocean in case using ncar formulae
  if(ncar_bulk_formulae) then
     call ncar_ocean_fluxes_mode
  elseif(AOMIP_drag_coeff) then
     cd_atm_oce_arr=cd_atm_ice_arr
  end if

  ! fourth, compute wind stress
  do i=1,myDim_nod2d+eDim_nod2d     
     dux=u_wind(i)-u_w(i) 
     dvy=v_wind(i)-v_w(i)
     aux=sqrt(dux**2+dvy**2)*rhoair
     stress_atmoce_x(i) = Cd_atm_oce_arr(i)*aux*dux
     stress_atmoce_y(i) = Cd_atm_oce_arr(i)*aux*dvy
     dux=u_wind(i)-u_ice(i) 
     dvy=v_wind(i)-v_ice(i)
     aux=sqrt(dux**2+dvy**2)*rhoair
     stress_atmice_x(i) = Cd_atm_ice_arr(i)*aux*dux
     stress_atmice_y(i) = Cd_atm_ice_arr(i)*aux*dvy
  end do

  ! heat and fresh water fluxes are treated in i_therm and ice2ocean

  t2=MPI_Wtime()

  if (mod(istep,logfile_outfreq)==0 .and. mype==0) then
     write(*,*) 'update forcing data took', t2-t1
  end if

end subroutine update_atm_forcing
!
!------------------------------------------------------------------------------------
!
subroutine read_new_atm_forcing
  ! update atmospheric forcing, SSS, Chl etc. 
  ! assume forcing data on T62 NCEP/NCAR grid
  !
  ! Coded by Qiang Wang
  ! Reviewed by ??

  use o_PARAM
  use o_MESH
  use o_array
  use i_therm_parms
  use g_forcing_param
  use g_forcing_arrays
  use g_forcing_index
  use g_forcing_interp
  use g_read_CORE_NetCDF
  use g_read_NCEP_NetCDF
  use g_read_ECMWF_NetCDF
  use g_read_other_NetCDF
  use g_clock
  use g_parfe
  use g_config
  implicit none
  !
  integer        		        :: nci, ncj 
  integer                   		:: itime, m, i, k, n2
  integer                               :: readtype
  character(80)             		:: file
  character(15)             		:: vari, filevari
  character(4)				:: fileyear
  real(kind=8), allocatable	        :: array_nc(:,:), array_nc2(:,:)
  real(kind=8), dimension(nod2D)    	:: array_fe
  real(kind=8), allocatable             :: aux(:)       
  logical                               :: check_dummy


  nci=forcing_ni
  ncj=forcing_nj

  allocate(array_nc(nci, ncj), array_nc2(nci, ncj))

  n2=ToDim_nod2D

  !==========================================================================
  ! wind u and v, Tair, and shum   

  if(wind_data_source=='NCEP') then

     if(update_forcing_flag(wind_ttp_ind)==1) then

        fileyear=cyearnew
        itime=forcing_rec(wind_ttp_ind)

        ! three temporal types (6 hourly, daily and monthly) are possible 

        if(wind_ttp_ind==1) then ! 6 hourly data

           ! in case of 6-hourly wind, we do interpolation in time,
           ! so the routine is different from in other cases.

           !save the second record to the first record
           do i=1,ToDim_nod2d       
              u_wind_t(1,i)=u_wind_t(2,i)
              v_wind_t(1,i)=v_wind_t(2,i)
              Tair_t(1,i)=Tair_t(2,i)
              shum_t(1,i)=shum_t(2,i)
           end do

           if (itime>4*ndpyr) then	! to read the first record of the next year
              ! ==========================
              ! Adjust NCEP 6-hourly forcing time index 'itime'
              ! Oct 2015, cdanek@awi.de
              !itime=1
              itime=itime-(4*ndpyr)
              ! ==========================
              write(fileyear,'(i4)') yearnew+1
           endif

           ! 10-m wind m/s ----------------------------------------

           vari='uwnd'
           file=trim(ForcingDataPath)//'NCEP_6hourly/'//'uwnd.10m.gauss.'//fileyear//'.nc'
           if (mod(istep,logfile_outfreq)==0 .and. mype==0) write(*,*) 'read_new_atm_forcing(): istep=', istep, ', itime=',   itime, ', file=', file
           call read_NCEPv2_NetCDF(file, vari, itime, array_nc)
           !if (mod(istep,logfile_outfreq)==0 .and. mype==0) then
           !    write(*,*) 'read_new_atm_forcing(): istep=', istep, ', itime=', itime, ', ToDim_nod2d=', ToDim_nod2d, ', size(array_nc)=', size(array_nc), ', file=', file
           !    write(*,*) 'read_NCEPv2_NetCDF(): mean(uwnd)=', sum(array_nc)/size(array_nc)
           !end if
           call upside_down(array_nc,nci,ncj)

           vari='vwnd'
           file=trim(ForcingDataPath)//'NCEP_6hourly/'//'vwnd.10m.gauss.'//fileyear//'.nc'
           call read_NCEPv2_NetCDF(file, vari, itime, array_nc2)
           !if (mod(istep,logfile_outfreq)==0 .and. mype==0) then
           !    write(*,*) 'read_new_atm_forcing(): istep=', istep, ', itime=', itime, ', ToDim_nod2d=', ToDim_nod2d, ', size(array_nc2)=', size(array_nc2), ', file=', file
           !    write(*,*) 'read_NCEPv2_NetCDF(): mean(vwnd)=', sum(array_nc2)/size(array_nc2)
           !end if
           call upside_down(array_nc2,nci,ncj)

           ! rotate wind
           if(rotated_grid) call rotate_wind(array_nc, array_nc2, nci, ncj)

           ! interp wind to model grid
           call forcing_linear_ip(array_nc,nci,ncj,lint_ind,lint_weight,u_wind_t(2,:),n2)   
           call forcing_linear_ip(array_nc2,nci,ncj,lint_ind,lint_weight,v_wind_t(2,:),n2) 

           ! 2-m temperature --------------------------------------

           vari='air'
           file=trim(ForcingDataPath)//'NCEP_6hourly/'//'air.2m.gauss.'//fileyear//'.nc'
           call read_NCEPv2_NetCDF(file, vari, itime, array_nc)
           !if (mod(istep,logfile_outfreq)==0 .and. mype==0) then
           !    write(*,*) 'read_new_atm_forcing(): istep=', istep, ', itime=', itime, ', ToDim_nod2d=', ToDim_nod2d, ', size(array_nc)=', size(array_nc), ', file=', file
           !    write(*,*) 'read_NCEPv2_NetCDF(): mean(air)=', sum(array_nc)/size(array_nc)
           !end if
           call upside_down(array_nc,nci,ncj)
           call forcing_linear_ip(array_nc,nci,ncj,lint_ind,lint_weight,Tair_t(2,:),n2)   
           Tair_t(2,:)=Tair_t(2,:)-tmelt  ! Kelvin --> degree Celcius

           ! 2 m specific humdity  Kg/Kg -------------------------

           vari='shum'
           file=trim(ForcingDataPath)//'NCEP_6hourly/'//'shum.2m.gauss.'//fileyear//'.nc'
           call read_NCEPv2_NetCDF(file, vari, itime, array_nc)
           !if (mod(istep,logfile_outfreq)==0 .and. mype==0) then
           !    write(*,*) 'read_new_atm_forcing(): istep=', istep, ', itime=', itime, ', ToDim_nod2d=', ToDim_nod2d, ', size(array_nc)=', size(array_nc), ', file=', file
           !    write(*,*) 'read_NCEPv2_NetCDF(): mean(shum)=', sum(array_nc)/size(array_nc)
           !end if
           call upside_down(array_nc,nci,ncj)
           call forcing_linear_ip(array_nc,nci,ncj,lint_ind,lint_weight,shum_t(2,:),n2)   
           
        elseif(wind_ttp_ind==2) then ! daily data      

           ! 10-m wind --------------------------------------------

           vari='uwnd'
           file=trim(ForcingDataPath)//'NCEP_daily/'//'uwnd.10m.gauss.'//fileyear//'.nc'
           call read_NCEP_NetCDF(file, vari, itime, array_nc)
           call upside_down(array_nc,nci,ncj)

           vari='vwnd'
           file=trim(ForcingDataPath)//'NCEP_daily/'//'vwnd.10m.gauss.'//fileyear//'.nc'
           call read_NCEP_NetCDF(file, vari, itime, array_nc2)
           call upside_down(array_nc2,nci,ncj)

           ! rotate wind
           if(rotated_grid) call rotate_wind(array_nc, array_nc2, nci, ncj)

           ! interp wind to model grid
           call forcing_linear_ip(array_nc,nci,ncj,lint_ind,lint_weight,u_wind,n2)  
           call forcing_linear_ip(array_nc2,nci,ncj,lint_ind,lint_weight,v_wind,n2) 

           ! 2-m temperature --------------------------------------

           vari='air'
           file=trim(ForcingDataPath)//'NCEP_daily/'//'air.2m.gauss.'//fileyear//'.nc'
           call read_NCEP_NetCDF(file, vari, itime, array_nc)
           call upside_down(array_nc,nci,ncj)
           call forcing_linear_ip(array_nc,nci,ncj,lint_ind,lint_weight,Tair,n2)   
           Tair=Tair-tmelt  ! Kelvin --> degree Celcius

           ! 2 m specific humdity  Kg/Kg -------------------------

           vari='shum'
           file=trim(ForcingDataPath)//'NCEP_daily/'//'shum.2m.gauss.'//fileyear//'.nc'
           call read_NCEP_NetCDF(file, vari, itime, array_nc)
           call upside_down(array_nc,nci,ncj)
           call forcing_linear_ip(array_nc,nci,ncj,lint_ind,lint_weight,shum,n2) 


        elseif(wind_ttp_ind==3) then ! monthly data

           ! 10-m wind m/s ----------------------------------------

           vari='uwnd'
           file=trim(ForcingDataPath)//'NCEP_monthly/'//'uwnd10m.mon.mean.nc'
           call read_NCEP_NetCDF(file, vari, itime, array_nc)
           call upside_down(array_nc,nci,ncj)

           vari='vwnd'
           file=trim(ForcingDataPath)//'NCEP_monthly/'//'vwnd10m.mon.mean.nc'
           call read_NCEP_NetCDF(file, vari, itime, array_nc2)
           call upside_down(array_nc2,nci,ncj)

           ! rotate wind
           if(rotated_grid) call rotate_wind(array_nc, array_nc2, nci, ncj)

           ! interp wind to model grid
           call forcing_linear_ip(array_nc,nci,ncj,lint_ind,lint_weight,u_wind,n2) 
           call forcing_linear_ip(array_nc2,nci,ncj,lint_ind,lint_weight,v_wind,n2) 

           ! 2-m temperature --------------------------------------

           vari='air'
           file=trim(ForcingDataPath)//'NCEP_monthly/'//'air2m.mon.mean.nc'
           call read_NCEP_NetCDF(file, vari, itime, array_nc)
           call upside_down(array_nc,nci,ncj)
           call forcing_linear_ip(array_nc,nci,ncj,lint_ind,lint_weight,Tair,n2)  
           Tair=Tair-tmelt  ! Kelvin --> degree Celcius

           ! 2 m specific humdity  Kg/Kg -------------------------

           vari='shum'
           file=trim(ForcingDataPath)//'NCEP_monthly/'//'shum2m.mon.mean.nc'
           call read_NCEP_NetCDF(file, vari, itime, array_nc)
           call upside_down(array_nc,nci,ncj)
           call forcing_linear_ip(array_nc,nci,ncj,lint_ind,lint_weight,shum,n2) 

        end if

     end if
           
  elseif(wind_data_source=='ECMWF') then

     if(update_forcing_flag(wind_ttp_ind)==1) then

        fileyear=cyearnew
        itime=forcing_rec(wind_ttp_ind)

        ! 6 hourly used, no check required. !if(wind_ttp_ind==1) then ! 6 hourly

        ! in case of 6-hourly wind, we do interpolation in time

        !save the second record to the first record
        do i=1,ToDim_nod2d       
           u_wind_t(1,i)=u_wind_t(2,i)
           v_wind_t(1,i)=v_wind_t(2,i)
           Tair_t(1,i)=Tair_t(2,i)
           shum_t(1,i)=shum_t(2,i)
        end do

        if (itime>4*ndpyr) then	! to read the first record of the next year
           itime=1
           write(fileyear,'(i4)') yearnew+1
        endif

        ! 10-m wind m/s ----------------------------------------

        vari='u10'
        file=trim(ForcingDataPath)//'ECMWF/'//'u10m_'//fileyear//'.nc'
        call read_ECMWF_NetCDF(file, vari, itime, array_nc, nci, ncj)
        call upside_down(array_nc,nci,ncj)

        vari='v10'
        file=trim(ForcingDataPath)//'ECMWF/'//'v10m_'//fileyear//'.nc'
        call read_ECMWF_NetCDF(file, vari, itime, array_nc2, nci, ncj)
        call upside_down(array_nc2,nci,ncj)

        ! rotate wind
        if(rotated_grid) call rotate_wind(array_nc, array_nc2, nci, ncj)

        ! interp wind to model grid
        call forcing_linear_ip(array_nc,nci,ncj,lint_ind,lint_weight,u_wind_t(2,:),n2)   
        call forcing_linear_ip(array_nc2,nci,ncj,lint_ind,lint_weight,v_wind_t(2,:),n2) 

        ! 2-m temperature --------------------------------------

        vari='t2m'
        file=trim(ForcingDataPath)//'ECMWF/'//'T2m_'//fileyear//'.nc'
        call read_ECMWF_NetCDF(file, vari, itime, array_nc, nci, ncj)
        call upside_down(array_nc,nci,ncj)
        call forcing_linear_ip(array_nc,nci,ncj,lint_ind,lint_weight,Tair_t(2,:),n2)   
        Tair_t(2,:)=Tair_t(2,:)-tmelt  ! Kelvin --> degree Celcius

        ! 2 m specific humdity  Kg/Kg -------------------------

        vari='q'
        file=trim(ForcingDataPath)//'ECMWF/'//'q_'//fileyear//'.nc'
        call read_ECMWF_NetCDF(file, vari, itime, array_nc, nci, ncj)
        call upside_down(array_nc,nci,ncj)
        call forcing_linear_ip(array_nc,nci,ncj,lint_ind,lint_weight,shum_t(2,:),n2)    

        !end if

     end if

  elseif(wind_data_source=='CORE2') then

     ! in CORE 6-hourly wind is used 

     if(update_forcing_flag(wind_ttp_ind)==1) then

        !save the second record to the first record
        do i=1,ToDim_nod2d     
           u_wind_t(1,i)=u_wind_t(2,i)
           v_wind_t(1,i)=v_wind_t(2,i)
           Tair_t(1,i)=Tair_t(2,i)
           shum_t(1,i)=shum_t(2,i)
        end do

        fileyear=cyearnew
        itime=forcing_rec(wind_ttp_ind)

        if (itime>4*ndpyr) then	! to read the first record of the next year
           itime=1
           write(fileyear,'(i4)') yearnew+1
        endif

        ! 10-m wind m/s ----------------------------------------

        filevari='u_10.'
        file=trim(ForcingDataPath)//'CORE2/'//trim(filevari)//fileyear//'.nc'
        vari='U_10_MOD'
        if (mod(istep,logfile_outfreq)==0 .and. mype==0) write(*,*) 'read_new_atm_forcing(): istep=', istep, ', itime=', itime, ', file=', file
        call read_CORE_NetCDF(file, vari, itime, array_nc)

        filevari='v_10.'
        file=trim(ForcingDataPath)//'CORE2/'//trim(filevari)//fileyear//'.nc'
        vari='V_10_MOD'   
        call read_CORE_NetCDF(file, vari, itime, array_nc2)

        ! rotate wind
        if(rotated_grid) call rotate_wind(array_nc, array_nc2, nci, ncj)

        ! interp wind to model grid
        call forcing_linear_ip(array_nc,nci,ncj,lint_ind,lint_weight,u_wind_t(2,:),n2) 
        call forcing_linear_ip(array_nc2,nci,ncj,lint_ind,lint_weight,v_wind_t(2,:),n2) 

        ! 10-m temperature -------------------------------------

        filevari='t_10.'
        file=trim(ForcingDataPath)//'CORE2/'//trim(filevari)//fileyear//'.nc'
        vari='T_10_MOD'
        call read_CORE_NetCDF(file, vari, itime, array_nc)
        call forcing_linear_ip(array_nc,nci,ncj,lint_ind,lint_weight,Tair_t(2,:),n2) 
        Tair_t(2,:)=Tair_t(2,:)-tmelt  ! Kelvin --> degree celcium

        ! 10 m specific humdity  Kg/Kg -------------------------

        filevari='q_10.'
        file=trim(ForcingDataPath)//'CORE2/'//trim(filevari)//fileyear//'.nc'
        vari='Q_10_MOD'
        call read_CORE_NetCDF(file, vari, itime, array_nc)
        call forcing_linear_ip(array_nc,nci,ncj,lint_ind,lint_weight,shum_t(2,:),n2) 

     end if

  elseif(wind_data_source=='CORE1') then

     ! in CORE 6-hourly wind is used 

     if(update_forcing_flag(wind_ttp_ind)==1) then

        !save the second record to the first record
        do i=1,myDim_nod2d+eDim_nod2d           
           u_wind_t(1,i)=u_wind_t(2,i)
           v_wind_t(1,i)=v_wind_t(2,i)
           Tair_t(1,i)=Tair_t(2,i)
           shum_t(1,i)=shum_t(2,i)
        end do

        itime=forcing_rec(wind_ttp_ind)

        if (itime>4*ndpyr) then	! rewind to read the first record
           itime=1
        endif

        ! 10-m wind m/s ----------------------------------------

        filevari='u_10'
        file=trim(ForcingDataPath)//'CORE1/'//trim(filevari)//'.nc'
        vari='U_10_MOD'
        call read_CORE_NetCDF(file, vari, itime, array_nc)

        filevari='v_10'
        file=trim(ForcingDataPath)//'CORE1/'//trim(filevari)//'.nc'
        vari='V_10_MOD'   
        call read_CORE_NetCDF(file, vari, itime, array_nc2)

        ! rotate wind
        if(rotated_grid) call rotate_wind(array_nc, array_nc2, nci, ncj)

        ! interp wind to model grid
        call forcing_linear_ip(array_nc,nci,ncj,lint_ind,lint_weight,u_wind_t(2,:),n2)   
        call forcing_linear_ip(array_nc2,nci,ncj,lint_ind,lint_weight,v_wind_t(2,:),n2) 

        ! 10-m temperature -------------------------------------

        filevari='t_10'
        file=trim(ForcingDataPath)//'CORE1/'//trim(filevari)//'.nc'
        vari='T_10_MOD'
        call read_CORE_NetCDF(file, vari, itime, array_nc)
        call forcing_linear_ip(array_nc,nci,ncj,lint_ind,lint_weight,Tair_t(2,:),n2) 
        Tair_t(2,:)=Tair_t(2,:)-tmelt  ! Kelvin --> Degree Celcius

        ! 10 m specific humdity  Kg/Kg -------------------------

        filevari='q_10'
        file=trim(ForcingDataPath)//'CORE1/'//trim(filevari)//'.nc'
        vari='Q_10_MOD'
        call read_CORE_NetCDF(file, vari, itime, array_nc)
        call forcing_linear_ip(array_nc,nci,ncj,lint_ind,lint_weight,shum_t(2,:),n2)  

     end if

  endif


  !==========================================================================
  ! radiation 

  if(rad_data_source=='NCEP') then

     if(update_forcing_flag(rad_ttp_ind)==1) then

        fileyear=cyearnew
        itime=forcing_rec(rad_ttp_ind)

        ! temporal types (1: 6 hourly, 2: daily), currently we calculate lw/sw 
        ! from other data; direct reading codes are commented out!

        if(rad_ttp_ind==1) then ! 6 hourly data
        
           ! ____________________________________________P.Scholz 08.07.2015___
           !|°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°|
           !|___6 hourly shortwave/longwave radiation__________________________|
           ! in case of 6-hourly radiation, we do interpolation in time,
           ! so the routine is different from in other cases.
           ! save the second record to the first record
           do i=1,ToDim_nod2d       
              shortwave_t(1,i)=shortwave_t(2,i)
              longwave_t(1,i)=longwave_t(2,i)
           end do

           if (itime>4*ndpyr) then ! to read the first record of the next year
              ! ==========================
              ! Adjust NCEP 6-hourly forcing time index 'itime'
              ! Oct 2015, cdanek@awi.de
              !itime=1
              itime=itime-(4*ndpyr)
              ! =========================
              write(fileyear,'(i4)') yearnew+1
           endif

           ! short wave W/m2 --------------------------------------
           vari='dswrf'
           file=trim(ForcingDataPath)//'NCEP_6hourly/'//'dswrf.sfc.gauss.'//fileyear//'.nc'
           call read_NCEPv2_NetCDF(file, vari, itime, array_nc)
           !if (mod(istep,logfile_outfreq)==0 .and. mype==0) then
           !    write(*,*) 'read_new_atm_forcing(): istep=', istep, ', itime=', itime, ', ToDim_nod2d=', ToDim_nod2d, ', size(array_nc)=', size(array_nc), ', file=', file
           !    write(*,*) 'read_NCEPv2_NetCDF(): mean(dswrf)=', sum(array_nc)/size(array_nc)
           !end if
           call upside_down(array_nc,nci,ncj)
           call forcing_linear_ip(array_nc,nci,ncj,lint_ind,lint_weight,shortwave_t(2,:),n2) 

           ! long wave W/m2 ---------------------------------------
           vari='dlwrf'
           file=trim(ForcingDataPath)//'NCEP_6hourly/'//'dlwrf.sfc.gauss.'//fileyear//'.nc'
           call read_NCEPv2_NetCDF(file, vari, itime, array_nc)
           !if (mod(istep,logfile_outfreq)==0 .and. mype==0) then
           !    write(*,*) 'read_new_atm_forcing(): istep=', istep, ', itime=', itime, ', ToDim_nod2d=', ToDim_nod2d, ', size(array_nc)=', size(array_nc), ', file=', file
           !    write(*,*) 'read_NCEPv2_NetCDF(): mean(dlwrf)=', sum(array_nc)/size(array_nc)
           !end if
           call upside_down(array_nc,nci,ncj)
           call forcing_linear_ip(array_nc,nci,ncj,lint_ind,lint_weight,longwave_t(2,:),n2)  
 
           ! cloudiness -----------------------------------
!           vari='tcdc'
!           file=trim(ForcingDataPath)//'NCEP_6hourly/'//'tcdc.eatm.gauss.'//fileyear//'.nc'
!           call read_NCEPv2_NetCDF(file, vari, itime, array_nc)
!           call upside_down(array_nc,nci,ncj)
!           call forcing_linear_ip(array_nc,nci,ncj,lint_ind,lint_weight,cloudiness,n2)
!           cloudiness=cloudiness/100.0  ! the raw data have unit %

           ! slp Pa -----------------------------------------------

!           vari='pres'
!           file=trim(ForcingDataPath)//'NCEP_6hourly/'//'pres.sfc.gauss.'//fileyear//'.nc'
!           call read_NCEPv2_NetCDF(file, vari, itime, array_nc)
!           call upside_down(array_nc,nci,ncj)
!           call forcing_linear_ip(array_nc,nci,ncj,lint_ind,lint_weight,Pair,n2)

        elseif(rad_ttp_ind==2) then ! daily data

!!$           ! short wave W/m2 --------------------------------------
!!$
!!$           vari='dswrf'
!!!$           file=trim(ForcingDataPath)//'NCEP_daily/'//'dswrf.sfc.gauss.'//fileyear//'.nc'
!!$           call read_NCEP_NetCDF(file, vari, itime, array_nc)
!!$           call upside_down(array_nc,nci,ncj)
!!$           call forcing_linear_ip(array_nc,nci,ncj,lint_ind,lint_weight,shortwave,n2) 
!!$
!!$           ! long wave W/m2 ---------------------------------------
!!$
!!$           vari='dlwrf'
!!$           file=trim(ForcingDataPath)//'NCEP_daily/'//'dlwrf.sfc.gauss.'//fileyear//'.nc'
!!$           call read_NCEP_NetCDF(file, vari, itime, array_nc)
!!$           call upside_down(array_nc,nci,ncj)
!!$           call forcing_linear_ip(array_nc,nci,ncj,lint_ind,lint_weight,longwave,n2) 

           ! calculate from other data

           ! cloudiness -----------------------------------
           vari='tcdc'
           file=trim(ForcingDataPath)//'NCEP_daily/'//'tcdc.eatm.gauss.'//fileyear//'.nc'
           call read_NCEP_NetCDF(file, vari, itime, array_nc)
           call upside_down(array_nc,nci,ncj)
           call forcing_linear_ip(array_nc,nci,ncj,lint_ind,lint_weight,cloudiness,n2)
           cloudiness=cloudiness/100.0   ! the raw data have unit %
	   
	   ! slp Pa -----------------------------------------------

           vari='slp'
           file=trim(ForcingDataPath)//'NCEP_daily/'//'slp.'//fileyear//'.nc'
           call read_NCEP_NetCDF(file, vari, itime, array_nc)
           call upside_down(array_nc,nci,ncj)
           call forcing_linear_ip(array_nc,nci,ncj,lint_ind,lint_weight,Pair,n2)

        end if

        ! ! Comment out if radiation is provided by NCEP forcing: 
        !
        ! longwave radiation according to Koenig-Langlo and Augstein, 1994
        ! function of cloudiness and Tair
        ! longwave=(0.765+0.22*cloudiness**3)*boltzmann*(Tair+tmelt)**4

        ! shortwave radiation according to Parkinson and Washington, 1979
        ! function of cloudiness and evaporation, the latter is function of 
        ! shum and Pair, or function of dew point temperature.
        ! call shortwave_radiation(daynew,0,23,n2)

     end if

  elseif(rad_data_source=='ECMWF') then

     if(update_forcing_flag(rad_ttp_ind)==1) then

        fileyear=cyearnew
        itime=forcing_rec(rad_ttp_ind)

        !use daily data

        ! short wave W/m2 --------------------------------------

        vari='ssrd'
        file=trim(ForcingDataPath)//'ECMWF/'//'ssrd_'//fileyear//'.nc'
        call read_ECMWF_NetCDF(file, vari, itime, array_nc, nci, ncj)
        call upside_down(array_nc,nci,ncj)
        call forcing_linear_ip(array_nc,nci,ncj,lint_ind,lint_weight,shortwave,n2) 

        ! long wave W/m2 ---------------------------------------

        vari='strd'
        file=trim(ForcingDataPath)//'ECMWF/'//'strd_'//fileyear//'.nc'
        call read_ECMWF_NetCDF(file, vari, itime, array_nc, nci, ncj)
        call upside_down(array_nc,nci,ncj)
        call forcing_linear_ip(array_nc,nci,ncj,lint_ind,lint_weight,longwave,n2) 

     end if

  elseif(rad_data_source=='CORE2') then

     ! in CORE daily radiation fluxes are used 

     if(update_forcing_flag(rad_ttp_ind)==1) then

        fileyear=cyearnew
        itime=forcing_rec(rad_ttp_ind)

        ! short wave W/m2 --------------------------------------

        filevari='ncar_rad.'
        file=trim(ForcingDataPath)//'CORE2/'//trim(filevari)//fileyear//'.nc'
        vari='SWDN_MOD'
        call read_CORE_NetCDF(file, vari, itime, array_nc)
        call forcing_linear_ip(array_nc,nci,ncj,lint_ind,lint_weight,shortwave,n2) 

        ! long wave W/m2 ---------------------------------------

        vari='LWDN_MOD'
        call read_CORE_NetCDF(file, vari, itime, array_nc)
        call forcing_linear_ip(array_nc,nci,ncj,lint_ind,lint_weight,longwave,n2) 

     end if

  elseif(rad_data_source=='CORE1') then

     ! in CORE daily radiation fluxes are used 

     if(update_forcing_flag(rad_ttp_ind)==1) then

        itime=forcing_rec(rad_ttp_ind)

        ! short wave W/m2 --------------------------------------

        filevari='ncar_rad'
        file=trim(ForcingDataPath)//'CORE1/'//trim(filevari)//'.nc'
        vari='SWDN_MOD'
        call read_CORE_NetCDF(file, vari, itime, array_nc)
        call forcing_linear_ip(array_nc,nci,ncj,lint_ind,lint_weight,shortwave,n2) 

        ! long wave W/m2 ---------------------------------------

        vari='LWDN_MOD'
        call read_CORE_NetCDF(file, vari, itime, array_nc)
        call forcing_linear_ip(array_nc,nci,ncj,lint_ind,lint_weight,longwave,n2)  

     end if

  end if


  !==========================================================================
  ! precipitation

  if(precip_data_source=='NCEP') then

     if(update_forcing_flag(precip_ttp_ind)==1) then

        fileyear=cyearnew
        itime=forcing_rec(precip_ttp_ind)

        ! four temporal types (6 hourly, daily and monthly, monthly ltm) are possible 

        if(precip_ttp_ind==1) then ! 6 hourly data
        
           ! ____________________________________________P.Scholz 09.07.2015___
           !|°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°°|
           !|___6 hourly precipitation/evaporation_____________________________|
           ! in case of 6-hourly precip/evap, we do interpolation in time,
           ! so the routine is different from in other cases.
           ! save the second record to the first record
           do i=1,ToDim_nod2d       
              prec_rain_t(1,i)=prec_rain_t(2,i)
              evaporation_t(1,i)=evaporation_t(2,i)
           end do

           if (itime>4*ndpyr) then ! to read the first record of the next year
              ! ==========================
              ! Adjust NCEP 6-hourly forcing time index 'itime'
              ! Oct 2015, cdanek@awi.de
              !itime=1
              itime=itime-(4*ndpyr)
              ! =========================
              write(fileyear,'(i4)') yearnew+1
           endif

           ! total precip mm/s, need m/s --------------------------

           vari='prate'
           file=trim(ForcingDataPath)//'NCEP_6hourly/'//'prate.sfc.gauss.'//fileyear//'.nc'
           call read_NCEPv2_NetCDF(file, vari, itime, array_nc)
           !if (mod(istep,logfile_outfreq)==0 .and. mype==0) then
           !    write(*,*) 'read_new_atm_forcing(): istep=', istep, ', itime=', itime, ', ToDim_nod2d=', ToDim_nod2d, ', size(array_nc)=', size(array_nc), ', file=', file
           !    write(*,*) 'read_NCEPv2_NetCDF(): mean(prate)=', sum(array_nc)/size(array_nc)
           !end if
           call upside_down(array_nc,nci,ncj)
           call forcing_linear_ip(array_nc,nci,ncj,lint_ind,lint_weight,prec_rain_t(2,:),n2)  
           prec_rain_t(2,:)=prec_rain_t(2,:)/1000.  ! mm/s --> m/s

           ! evaporation, need m/s, calculate from latent heat 
           vari='lhtfl'
           file=trim(ForcingDataPath)//'NCEP_6hourly/'//'lhtfl.sfc.gauss.'//fileyear//'.nc'
           call read_NCEPv2_NetCDF(file, vari, itime, array_nc)
           !if (mod(istep,logfile_outfreq)==0 .and. mype==0) then
           !    write(*,*) 'read_new_atm_forcing(): istep=', istep, ', itime=', itime, ', ToDim_nod2d=', ToDim_nod2d, ', size(array_nc)=', size(array_nc), ', file=', file
           !    write(*,*) 'read_NCEPv2_NetCDF(): mean(lhtfl)=', sum(array_nc)/size(array_nc)
           !end if
           call upside_down(array_nc,nci,ncj)
           call forcing_linear_ip(array_nc,nci,ncj,lint_ind,lint_weight,evaporation_t(2,:),n2)
           evaporation_t(2,:)=evaporation_t(2,:)/2.5e6/1000.0  
           ! evap [m/s]= Q_l [W/m2] / lh_evap [J/kg] / rho_w [kg/m3]

        elseif(precip_ttp_ind==2) then ! daily data      

           ! total precip mm/s, need m/s --------------------------

           vari='prate'
           file=trim(ForcingDataPath)//'NCEP_daily/'//'prate.sfc.gauss.'//fileyear//'.nc'
           call read_NCEP_NetCDF(file, vari, itime, array_nc)
           call upside_down(array_nc,nci,ncj)
           call forcing_linear_ip(array_nc,nci,ncj,lint_ind,lint_weight,prec_rain,n2) 
           prec_rain=prec_rain/1000.  ! mm/s --> m/s

           ! evaporation, need m/s, calculate from latent heat 

           vari='lhtfl'
           file=trim(ForcingDataPath)//'NCEP_daily/'//'lhtfl.sfc.gauss.'//fileyear//'.nc'
           call read_NCEP_NetCDF(file, vari, itime, array_nc)
           call upside_down(array_nc,nci,ncj)
           call forcing_linear_ip(array_nc,nci,ncj,lint_ind,lint_weight,evaporation,n2)
           evaporation=evaporation/2.5e6/1000.0
           ! evap [m/s]= Q_l [W/m2] / lh_evap [J/kg] / rho_w [kg/m3]

        elseif(precip_ttp_ind==3) then ! monthly data 

           ! total precip mm/s ------------------------------------

           vari='prate'
           file=trim(ForcingDataPath)//'NCEP_monthly/'//'prate.mon.mean.nc'
           call read_NCEP_NetCDF(file, vari, itime, array_nc)
           call upside_down(array_nc,nci,ncj)
           call forcing_linear_ip(array_nc,nci,ncj,lint_ind,lint_weight,prec_rain,n2)
           prec_rain=prec_rain/1000.  ! mm/s --> m/s

        elseif(precip_ttp_ind==4) then ! monthly ltm data 

           ! total precip mm/s ------------------------------------

           vari='prate'
           file=trim(ForcingDataPath)//'NCEP_mon_ltm/'//'prate.mon.mean.nc'
           call read_NCEP_NetCDF(file, vari, itime, array_nc)
           call upside_down(array_nc,nci,ncj)
           call forcing_linear_ip(array_nc,nci,ncj,lint_ind,lint_weight,prec_rain,n2) 
           prec_rain=prec_rain/1000.  ! mm/s --> m/s

        end if

     end if

  elseif(precip_data_source=='ECMWF') then

     if(update_forcing_flag(precip_ttp_ind)==1) then

        fileyear=cyearnew
        itime=forcing_rec(precip_ttp_ind)

        ! use daily data

        ! total precip kg/m2/s, need m/s --------------------------

        vari='tp'
        file=trim(ForcingDataPath)//'ECMWF/'//'tp_'//fileyear//'.nc'
        call read_ECMWF_NetCDF(file, vari, itime, array_nc, nci, ncj)
        call upside_down(array_nc,nci,ncj)
        call forcing_linear_ip(array_nc,nci,ncj,lint_ind,lint_weight,prec_rain,n2) 
        prec_rain=prec_rain/1000.  ! kg/m2/s --> m/s

     end if

  elseif(precip_data_source=='CORE2') then

     ! in CORE monthly precipitation is used; 
     ! And rain and snow are separated.

     if(update_forcing_flag(precip_ttp_ind)==1) then

        fileyear=cyearnew
        itime=forcing_rec(precip_ttp_ind)

        ! rain mm/s --------------------------------------------

        filevari='ncar_precip.'
        file=trim(ForcingDataPath)//'CORE2/'//trim(filevari)//fileyear//'.nc'
        vari='RAIN'
        call read_CORE_NetCDF(file, vari, itime, array_nc)
        call forcing_linear_ip(array_nc,nci,ncj,lint_ind,lint_weight,prec_rain,n2) 
        prec_rain=prec_rain/1000.  ! mm/s --> m/s

        ! snow mm/s --------------------------------------------

        vari='SNOW'
        call read_CORE_NetCDF(file, vari, itime, array_nc)
        call forcing_linear_ip(array_nc,nci,ncj,lint_ind,lint_weight,prec_snow,n2)  
        prec_snow=prec_snow/1000.  ! mm/s --> m/s

     end if

  elseif(precip_data_source=='CORE1') then

     ! in CORE monthly precipitation is used; 
     ! And rain and snow are separated.

     if(update_forcing_flag(precip_ttp_ind)==1) then

        itime=forcing_rec(precip_ttp_ind)

        ! rain mm/s --------------------------------------------

        filevari='ncar_precip'
        file=trim(ForcingDataPath)//'CORE1/'//trim(filevari)//'.nc'
        vari='RAIN'
        call read_CORE_NetCDF(file, vari, itime, array_nc)
        call forcing_linear_ip(array_nc,nci,ncj,lint_ind,lint_weight,prec_rain,n2)  
        prec_rain=prec_rain/1000.  ! mm/s --> m/s

        ! snow mm/s --------------------------------------------

        vari='SNOW'
        call read_CORE_NetCDF(file, vari, itime, array_nc)
        call forcing_linear_ip(array_nc,nci,ncj,lint_ind,lint_weight,prec_snow,n2)  
        prec_snow=prec_snow/1000.  ! mm/s --> m/s

     end if

  end if


  !==========================================================================
  ! runoff  

  if(runoff_data_source=='Dai09') then

     if(update_forcing_flag(runoff_ttp_ind)==1) then
        if(runoff_ttp_ind==4) then
           !climatology monthly mean

           itime=forcing_rec(runoff_ttp_ind)
           file=trim(MeshPath)//'forcing_data_on_grid/runoff_clim.nc' 
           vari='runoff'

           call read_2ddata_on_grid_NetCDF(file,vari,itime,runoff)

           !kg/m2/s -> m/s
           runoff=runoff/1000.

        elseif(runoff_ttp_ind==3) then
           !monthly data

           itime=forcing_rec(runoff_ttp_ind)
           file=trim(MeshPath)//'forcing_data_on_grid/runoff_'//fileyear//'.nc' 
           vari='runoff'

           call read_2ddata_on_grid_NetCDF(file,vari,itime,runoff)

           !kg/m2/s -> m/s
           runoff=runoff/1000.

        end if
     end if

  elseif(runoff_data_source=='AAOMIP') then

     ! runoff is monthly ltm in AOMIP/AAOMIP

     if(update_forcing_flag(runoff_ttp_ind)==1) then

        allocate(aux(nod2D))

        itime=forcing_rec(runoff_ttp_ind)

        file=trim(ForcingDataPath)//'AAOMIP'//'/river_runoff.dat'
        if(system==1) then
           readtype=2
        else
           readtype=8
        end if

        open(101,file=trim(file),form='unformatted', access='direct',recl=readtype*nod2d)
        read(101,rec=itime) aux                
        runoff=aux(myList_nod2D)        
        close(101)

        deallocate(aux)
     end if

  end if


  !==========================================================================
  ! sss restoring

  if(restore_s_surf>0.) then
     if(sss_data_source=='CORE1' .or. sss_data_source=='CORE2') then

        ! sss is monthly ltm in CORE cases

        if(update_forcing_flag(sss_ttp_ind)==1) then

           itime=forcing_rec(sss_ttp_ind)

           file=trim(ForcingDataPath)//trim(sss_data_source)//'/PHC2_salx.nc'
           vari='SALT'
           check_dummy=.true.
           call read_other_NetCDF(file, vari, itime, Ssurf, check_dummy)  
           if (mod(istep,logfile_outfreq)==0 .and. mype==0) then
               write(*,*) 'read_new_atm_forcing(): istep=', istep, ', itime=', itime, ', file=', file
               !write(*,*) 'read_other_NetCDF(): mean(Ssurf)=', sum(Ssurf)/size(Ssurf)
           end if

        end if

     end if
  end if


  !==========================================================================
  ! chlorophyll climatology

#ifdef use_sw_pene
  ! currently only one type supported
  ! perpetual monthly chl climatology prepared by Sweeney et al. 2005

  if(update_forcing_flag(4)==1) then

     !climatology monthly mean

     itime=forcing_rec(4)
     file=trim(MeshPath)//'forcing_data_on_grid/chl_clim.nc' 
     vari='chl'

     call read_2ddata_on_grid_NetCDF(file,vari,itime,chl)

  end if

#endif

  deallocate(array_nc, array_nc2)

end subroutine read_new_atm_forcing
!
!---------------------------------------------------------------------------------------------------
!
subroutine rotate_wind(xarray, yarray, ni, nj)
  ! rotate wind from geographical coord. to rotated coordinates.
  !
  ! Coded by Qiang Wang
  ! Reviewed by ??
  
  use o_param
  use g_config
  use g_rotate_grid
  use g_forcing_param
  implicit none

  integer 	        :: i, j, ni, nj
  real(kind=8)      	:: cx(ni), cy(nj), xarray(ni,nj), yarray(ni,nj)

  if(forcing_grid=='T62') then

     !latitude
     cy=(/-88.542, -86.6531, -84.7532, -82.8508, -80.9473, -79.0435, &  
          -77.1394, -75.2351, -73.3307, -71.4262, -69.5217, -67.6171, &  
          -65.7125, -63.8079, -61.9033, -59.9986, -58.0939, -56.1893, &  
          -54.2846, -52.3799, -50.4752, -48.5705, -46.6658, -44.7611,&  
          -42.8564, -40.9517, -39.0470, -37.1422, -35.2375, -33.3328, &  
          -31.4281, -29.5234, -27.6186, -25.7139, -23.8092, -21.9044, &  
          -19.9997, -18.0950, -16.1902, -14.2855, -12.3808, -10.47604, &  
          -8.57131, -6.66657, -4.76184, -2.8571, -0.952368, 0.952368, &  
          2.8571, 4.76184, 6.66657, 8.57131, 10.47604, 12.3808, &  
          14.2855, 16.1902, 18.095, 19.9997, 21.9044, 23.8092, &  
          25.7139, 27.6186, 29.5234, 31.4281, 33.3328, 35.2375,&  
          37.1422, 39.047,  40.9517, 42.8564, 44.7611, 46.6658,&  
          48.5705, 50.4752, 52.3799, 54.2846, 56.1893, 58.0939,&  
          59.9986, 61.9033, 63.8079, 65.7125, 67.6171, 69.5217, &  
          71.4262, 73.3307, 75.2351, 77.1394, 79.0435, 80.9473, &  
          82.8508, 84.7532, 86.6531, 88.542 /)*rad
     
     !longitude
     cx(1)=0.0
     do i=2,ni
        cx(i)=cx(i-1)+1.875*rad
     enddo

  elseif(forcing_grid=='other') then     
    
     cy(1)=-90.0*rad
     do j=2,nj
        cy(j)=cy(j-1)+forcing_dy*rad
     enddo
  
     cx(1)=0.0
     do i=2,ni
        cx(i)=cx(i-1)+forcing_dx*rad
     enddo
  
  end if

  !rotate wind
  !cx cy are in radian
  do i=1,ni
     do j=1,nj
        call vector_g2r(xarray(i,j), yarray(i,j), cx(i), cy(j), 1)
     end do
  end do
end subroutine rotate_wind
!
!-----------------------------------------------------------------------------------------
!
subroutine shortwave_radiation(day,nhi,nhf,n2)
  !
  ! calculates the incoming shortwave radiation following
  ! Parkinson and Washington, 1979
  !
  ! input data:
  ! ndoyr               day of year  (1..365 or 366)
  ! nhi                 initial hour (0...23)
  ! nhf                 final hour   (0...23)
  !                     (last hour to be completely considered)
  !
  ! For instance, for a 6h-timestep:     nhi    nhf
  !                                      0      5
  !                                      6      11
  !                                      12     17
  !                                      18     23
  !
  !            for a daily timestep:     0      23
  !
  !
  ! Derived from subroutine swr of BRIOS sea ice model by Ralph Timmermann, 23.9.2004
  ! Copied to the new FESOM version by Qiang Wang
  ! Reviewed by ??
  !=========================================================================
  USE o_MESH, only: geolat2d
  USE o_PARAM
  USE g_forcing_arrays

  IMPLICIT NONE

  INTEGER                 ::      day, ndoyrd, nhi, nhf
  INTEGER                 ::      it, iday, itd, n2
  real*8, dimension(0:23) ::      cosha
  real*8, dimension(n2)   ::      sinlat, coslat, cosz
  real*8, dimension(n2)   ::      cf, evap, fsh
  real*8                  ::      dec, sindec, cosdec

  ! ----------------------------------------------------------------------
  ! Cosine of hour angle for every hour
  ! [Parkinson and Washington, 1979, eq. (3)]
  ! ----------------------------------------------------------------------
  do it=0,23
     cosha(it)=cos(2*pi*(12-it)/24.)             ! correct for GM
  enddo

  ! ----------------------------------------------------------------------
  ! Declination of the sun for every day
  ! -----------------------------------------------------------------------
  ndoyrd=min(day,365)    !no fleap year
  dec=rad*23.44*cos((174.-(ndoyrd-1))*2*pi/365.) ! 365 days/year
  sindec=sin(dec)
  cosdec=cos(dec)

  !-----------------------------------------------------------------------
  ! Sine and cosine of geographical latitude
  !-----------------------------------------------------------------------
  sinlat=sin(geolat2d)
  coslat=cos(geolat2d)

  !-----------------------------------------------------------------------
  !       fsh             shortwave radiation
  !       cf              cloud factor
  !-----------------------------------------------------------------------
  fsh  = 0.
  evap = shum*Pair*0.01/0.622*1.e-3  ! Pair with unit Pa, *0.01->hPa  
  !ecmwf e_vapor(i,j)=611.e-5*exp(19.*tdew(i,j)/(tdew(i,j)+250.))
  cf = 1.-0.6*cloudiness**3


  !-----------------------------------------------------------------------
  !LOOP for every hour
  !-----------------------------------------------------------------------
  do it=nhi,nhf
     if (it.lt.0) then
        itd=it+24
     else
        itd=it
     endif
     !-----------------------------------------------------------------------
     ! Cosine of zenith distance
     ! Parkinson and Washington, 1979, eq. (2)
     !-----------------------------------------------------------------------
     cosz=sinlat*sindec+coslat*cosdec*cosha(itd)
     !-----------------------------------------------------------------------
     ! At night, when no sun is present and cosz < 0.,
     ! the incoming solar radiation is zero (and not negative).
     !-----------------------------------------------------------------------
     cosz=max(cosz,0.)
     !-----------------------------------------------------------------------
     ! Add up incoming shortwave radiation for each and every hour
     ! Parkinson and Washington, 1979, eq. (1)
     !
     ! Solar constant = 1353 [W/m**2]
     !         56.375 = 1353 / 24 for contributions from 24 hours
     !         1353/float(nhf-nhi+1) for contributions from nhf-nhi hours
     !-----------------------------------------------------------------------
     fsh=fsh+1353./float(nhf-nhi+1)*cosz*cosz*cf(:) &
          /((cosz+2.7)*evap+1.085*cosz+0.1)
     !-----------------------------------------------------------------------
  enddo

  shortwave=fsh

end subroutine shortwave_radiation


