classdef AcquisitionModel < handle
%     Class for PET acquisition model objects.
%     PET acquisition model relates an image x to the simulated acquisition 
%     data y as
%     (F)    y = [1/n](G x + [a]) + [b]
%     where:
%     G is the geometric (ray tracing) projector from the image voxels
%     to the scanner's pairs of detectors (bins);
%     a and b are optional additive and background terms representing
%     the effects of accidental coincidences and scattering; assumed to be 
%     0 if not present;
%     n is an optional bin normalization term representing the inverse of
%     detector (bin) efficiencies; assumed to be 1 if not present.
%     The computation of y for a given x by the above formula (F) is
%     referred to as (forward) projection, and the computation of
%     (B)    z = G' [1/n] y
%     where G' is the transpose of G, is referred to as backprojection.

% SyneRBI Synergistic Image Reconstruction Framework (SIRF).
% Copyright 2015 - 2017 Rutherford Appleton Laboratory STFC.
% 
% This is software developed for the Collaborative Computational
% Project in Synergistic Reconstruction for Biomedical Imaging (formerly CCP PETMR)
% (http://www.ccpsynerbi.ac.uk/).
% 
% Licensed under the Apache License, Version 2.0 (the "License");
% you may not use this file except in compliance with the License.
% You may obtain a copy of the License at
% http://www.apache.org/licenses/LICENSE-2.0
% Unless required by applicable law or agreed to in writing, software
% distributed under the License is distributed on an "AS IS" BASIS,
% WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
% See the License for the specific language governing permissions and
% limitations under the License.

    properties
        name
        handle_
    end
    methods (Static)
        function name = class_name()
            name = 'AcquisitionModel';
        end
    end
    methods
        function self = AcquisitionModel()
            self.handle_ = [];
        end
        function delete(self)
            if ~isempty(self.handle_)
                sirf.Utilities.delete(self.handle_)
                self.handle_ = [];
            end
        end
        function set_additive_term(self, at)
%***SIRF*** sets the additive term a in the acquisition model;
%         at:  an AcquisitionData object containing a.
            sirf.Utilities.assert_validity(at, 'AcquisitionData')
            sirf.STIR.setParameter(self.handle_, 'AcquisitionModel', ...
                'additive_term', at, 'h');
        end
        function set_background_term(self, bt)
%***SIRF*** sets the background term b in the acquisition model;
%         bt:  an AcquisitionData object containing b.
            sirf.Utilities.assert_validity(bt, 'AcquisitionData')
            sirf.STIR.setParameter(self.handle_, 'AcquisitionModel', ...
                'background_term', bt, 'h');
        end
        function set_acquisition_sensitivity(self, asm)
%***SIRF*** sets the acquisition sensitivity model responsible for n in the acquisition model;
%         asm: an AcquisitionSensitivityModel object.
            sirf.Utilities.assert_validity(asm, 'AcquisitionSensitivityModel')
            sirf.STIR.setParameter(self.handle_, 'AcquisitionModel', ...
                'asm', asm, 'h');
        end
        function set_up(self, acq_templ, img_templ)
%***SIRF*** sets up the object with appropriate geometric information.
%         This function needs to be called before performing forward- or 
%         backprojections;
%         Usage: 
%             set_up(acq_templ, img_templ);
%         acq_templ:  an AcquisitionData object used as a template for
%                     creating an AcquisitionData object to store forward
%                     projection;
%         img_templ:  an ImageData object used as a template for creating an
%                     ImageData object to store backprojection.
            sirf.Utilities.assert_validity(acq_templ, 'AcquisitionData')
            sirf.Utilities.assert_validity(img_templ, 'ImageData')
            h = calllib...
                ('mstir', 'mSTIR_setupAcquisitionModel',...
                self.handle_, acq_templ.handle_, img_templ.handle_);
            sirf.Utilities.check_status([self.name ':set_up'], h)
            sirf.Utilities.delete(h)
            %calllib('mutilities', 'mDeleteDataHandle', h)
        end
        function ad = forward(self, x, subset_num, num_subsets)
%***SIRF*** computes the forward projection of x (see AcquisitionModel)
%         Usage: 
%             acq_data = forward(image);
%         x:  an ImageData object.
            sirf.Utilities.assert_validity(x, 'ImageData')
            if nargin < 4
                subset_num = 0;
                num_subsets = 1;
            end
            ad = sirf.STIR.AcquisitionData();
            ad.handle_ = calllib('mstir', 'mSTIR_acquisitionModelFwd',...
                self.handle_, x.handle_, subset_num, num_subsets);
            sirf.Utilities.check_status([self.name ':forward'], ad.handle_)
        end
        function image = backward(self, y, subset_num, num_subsets)
%***SIRF*** returns the backprojection of y (see AcquisitionModel)
%         y:  an AcquisitionData object.
            sirf.Utilities.assert_validity(y, 'AcquisitionData')
            if nargin < 4
                subset_num = 0;
                num_subsets = 1;
            end
            image = sirf.STIR.ImageData();
            image.handle_ = calllib('mstir', 'mSTIR_acquisitionModelBwd',...
                self.handle_, y.handle_, subset_num, num_subsets);
            sirf.Utilities.check_status...
                ([self.name ':backward'], image.handle_)
        end
    end
end