function reconstruct_from_listmode(engine)
% A demo showing reconstruction from raw data in listmode format.
% Notes:
%   output will be in the current working directory.
%   input defaults to the mMR subfolder of pet_data_path

% SyneRBI Synergistic Image Reconstruction Framework (SIRF).
% Copyright 2018 - 2019 Rutherford Appleton Laboratory STFC.
% Copyright 2018 - 2019 University College London.
% 
% This is software developed for the Collaborative Computational
% Project in Synergistic Reconstruction for Biomedical Imaging (formerly CCP PETMR)
% (http://www.ccpsynerbi.ac.uk/).
% 
% Licensed under the Apache License, Version 2.0 (the "License");
% you may not use this file except in compliance with the License.
% You may obtain a copy of the License at
% http://www.apache.org/licenses/LICENSE-2.0
% Unless required by applicable law or agreed to in writing, software
% distributed under the License is distributed on an "AS IS" BASIS,
% WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
% See the License for the specific language governing permissions and
% limitations under the License.

if nargin < 1
    engine = [];
end
% import_str = set_up_PET(engine);
% eval(import_str)
PET = set_up_PET(engine);
AD = PET.AcquisitionData();
AD.set_storage_scheme('memory');
pet_data_path = sirf.Utilities.examples_data_path('PET');

%AcquisitionData.set_storage_scheme('memory');

try
    % direct all information printing to info.txt; warnings to warn.txt
    % error messages to go to Matlab Command Window
    PET.MessageRedirector('info.txt', 'warn.txt');

    % First step is to create AcquisitionData ("sinograms") from the
    % listmode file.
    % See the listmode_to_sinograms demo for some more information on this
    % step.

    % create listmode-to-sinograms converter object
    lm2sino = PET.ListmodeToSinograms();

    default_path = fullfile(pet_data_path, 'mMR');

    [filename, pathname] = uigetfile...
        ('*.l.hdr', 'Select listmode data file', default_path);
    list_file = fullfile(pathname, filename);
    [filename, pathname] = uigetfile...
        ('*.hs', 'Select raw data file to be used as a template', default_path);
    tmpl_file = fullfile(pathname, filename);
    [filename, pathname] = uigetfile...
        ('*.n.hdr', 'Select ECAT8 normalization file', default_path);
    norm_file = fullfile(pathname, filename);
    [filename, pathname] = uigetfile...
        ('*.*hv', 'Select attenuation file', default_path);
    attn_file = fullfile(pathname, filename);
    
    % set input, output and template files
    lm2sino.set_input(list_file)
    lm2sino.set_output_prefix('sinograms')
    lm2sino.set_template(tmpl_file)

    % set interval
    lm2sino.set_time_interval(0, 50)

    % set up the converter
    lm2sino.set_up()

    % convert
    fprintf('converting raw data to sinograms...\n')
    lm2sino.process()

    % estimate randoms
    fprintf('estimating randoms...\n')
    randoms = lm2sino.estimate_randoms();

    % get access to the sinograms
    acq_data = lm2sino.get_output();
    % copy the acquisition data into a Matlab array
    acq_array = acq_data.as_array();
    %acq_dim = size(acq_array);
    acq_dim = acq_data.dimensions();
    fprintf('acquisition data dimensions: %d x %d x %d x %d\n', acq_dim)
    z = round(acq_dim(3)/2);
    sirf.Utilities.show_2D_array(acq_array(:,:,z), ...
        'acquisition data', 'tang. pos.', 'views');

    % read attenuation image
    attn_image = PET.ImageData(attn_file);
    attn_image_as_array = attn_image.as_array();
    % select a slice appropriate for the NEMA acquistion data
    z = 72;
    % z = round(size(attn_image_as_array, 3)/2);
    sirf.Utilities.show_2D_array(attn_image_as_array(:,:,z), ...
        'attenuation image', 'tang. pos.', 'views');

    % create initial image estimate of dimensions and voxel sizes
    % compatible with the scanner geometry (included in the AcquisitionData
    % object acq_data) and initialize each voxel to 1.0
    image = acq_data.create_uniform_image(1.0, 127, 127);
    image_array = image.as_array();
    fprintf('image dimensions: %d x %d x %d\n', size(image_array))

    % select acquisition model that implements the geometric
    % forward projection by a ray tracing matrix multiplication
    acq_model = PET.AcquisitionModelUsingRayTracingMatrix();
    acq_model.set_num_tangential_LORs(10)

    % create acquisition sensitivity model from ECAT8 normalization data
    asm_norm = PET.AcquisitionSensitivityModel(norm_file);
    % create acquisition sensitivity model for attenuation
    asm_attn = PET.AcquisitionSensitivityModel(attn_image, acq_model);
    asm_attn.set_up(acq_data);
    % compute attenuation factors
    bin_eff = PET.AcquisitionData(acq_data);
    bin_eff.fill(1.0);
    fprintf('applying attenuation (please wait, may take a while)...\n')
    asm_attn.unnormalise(bin_eff);
    %store these in a new acquisition sensitivity model
    asm_beff = PET.AcquisitionSensitivityModel(bin_eff);

    % chain attenuation and ECAT8 normalisation
    asm = PET.AcquisitionSensitivityModel(asm_norm, asm_beff);

    acq_model.set_acquisition_sensitivity(asm);
    acq_model.set_background_term(randoms);

    % define objective function to be maximized as a
    % log of the Poisson likelihood
    obj_fun = PET.make_Poisson_loglikelihood(acq_data);
    obj_fun.set_acquisition_model(acq_model)

    % select Ordered Subsets Maximum A-Posteriori One Step Late as the
    % reconstruction algorithm (since we are not using a penalty, or prior, in
    % this example, we actually run OSEM);
    % this algorithm does not converge to the maximum of the objective function
    % but is used in practice to speed-up calculations
    % See the reconstruction demos for more complicated examples     
    num_subsets = 7;
    num_subiterations = 2;
    recon = PET.OSMAPOSLReconstructor();
    recon.set_objective_function(obj_fun);
    recon.set_num_subsets(num_subsets);
    recon.set_num_subiterations(num_subiterations);

    % set up the reconstructor based on a sample image
    % (checks the validity of parameters, sets up objective function
    % and other objects involved in the reconstruction, which involves
    % computing/reading sensitivity image etc etc.)
    fprintf('setting up reconstructor, please wait, may take a while...\n');
    recon.set_up(image);

    % set the initial image estimate
    recon.set_current_estimate(image);

    % reconstruct
    fprintf('reconstructing, please wait...\n')
    recon.process();

    % display the reconstructed image
    image_array = recon.get_current_estimate().as_array();
    the_title = sprintf('Reconstructed image');
    sirf.Utilities.show_2D_array(image_array(:,:,z), the_title, 'x', 'y');

catch err
    % display error information
    fprintf('??? %s\n', err.message)
    fprintf('error id is %s\n', err.identifier)
end
