clear; startup;

% =========================================================================
% =========================================================================
%
% Matlab plotting scripts for the analysis of UKESM1 prepared for:
%
%   Yool, A., Palmiéri, J., Jones, C. G., de Mora, L., Kuhlbrodt, T., 
%   Popova, E. E., Nurser, A. J. G., Hirschi, J., Blaker, A. T., 
%	Coward, A. C., Blockley, E. W., and Sellar, A. A.: Evaluating the 
% 	physical and biogeochemical state of the global ocean component of 
%	UKESM1 in CMIP6 Historical simulations, Geosci. Model Dev. Discuss. 
%	[preprint], https://doi.org/10.5194/gmd-2020-333, in review, 2020.
%
%   Accepted for publication: 20 April 2021
%
% Note: scripts provided to document analysis, and may require editing to
% fully replicate plots in manuscript
%
% =========================================================================
% =========================================================================

% =========================================================================
%
% This script plots Figures 20 and 21, and Supplementary Figures S16, S17,
% S18 and S19
%
% =========================================================================

% =========================================================================

load ukesm1_PD_ensemble

% =========================================================================

% load up background files
load nemo_area100e
load nemo_mask100e
fname = sprintf('%s/mean_nemo_%so_1y_20001201-20101201_grid-T.nc', dname(1).name, rname(1).name);
t1 = ncread(fname, 'deptht_bounds');
nemo_dep = double(t1(1,:)); nemo_dep(end+1) = double(t1(2,end));
t1 = ncread(fname, 'deptht');
nemo_mid = double(t1');

% total grid area
t1 = area100e + nemo_mask100e(:,:,1);
t2 = isfinite(t1);
tot_area100e = sum(t1(t2)); % m2

% extra mask to get rid of Mediterranean, Black and Caspian Seas
t1 = nemo_mask100e(:,:,1) + 1;
t1(isnan(t1)) = 0;
t1(235:249,283:344) = t1(235:249,283:344) + 2;
t1(249:262,291:344) = t1(249:262,291:344) + 4;
t1(262:276,330:344) = t1(262:276,330:344) + 8;
t2 = t1;
t2(t1 < 1) = NaN; t2(t1 > 1) = NaN;
med_mask100e = t2;

nxx = -179.5:1:179.5;  nyy = -89.5:1:89.5;
[lon100, lat100] = meshgrid(nxx, nyy);
nxx = -180:1:180;  nyy = -90:1:90;
[lon100b, lat100b] = meshgrid(nxx, nyy);

load woa_mask
clear t1; t1(:,1:180) = woa_mask(:,181:360); t1(:,181:360) = woa_mask(:,1:180);
woa_mask = t1;

load woa_basin

helvetica;
pngopt = 0;

% =========================================================================

% load up annual 3D fields
for e = 1:1:tsize
  pnom = sprintf('%s/mean_medusa_%so_1y_20001201-20101201_ptrc-T.nc', dname(e).name, rname(e).name);
  fprintf('- Reading %s\n', pnom);
  t1 = ncread(pnom, 'DIN'); t2 = permute(t1, [2 1 3]); mod_fld(:,:,:,1,e) = t2;
  t1 = ncread(pnom, 'SIL'); t2 = permute(t1, [2 1 3]); mod_fld(:,:,:,2,e) = t2;
  t1 = ncread(pnom, 'OXY'); t2 = permute(t1, [2 1 3]); mod_fld(:,:,:,3,e) = t2;
  t1 = ncread(pnom, 'DIC'); t2 = permute(t1, [2 1 3]); mod_fld(:,:,:,4,e) = t2;
  t1 = ncread(pnom, 'ALK'); t2 = permute(t1, [2 1 3]); mod_fld(:,:,:,5,e) = t2;
  t1 = ncread(pnom, 'CFC11'); t2 = permute(t1, [2 1 3]); mod_fld(:,:,:,6,e) = t2 * 1e9;
end

% get WOA mask onto 75 levels
t1 = ncread('woa13_nitrate_OMIPinit.nc', 'no3'); woa102_no3(:,:,:,1) = permute(t1, [2 1 3]);
t1 = ncread('woa13_nitrate_OMIPinit.nc', 'depth'); woa102_dep = t1;
t1 = woa102_no3; clear t2;
for k = 1:1:101
  t2(:,:,k) = (t1(:,:,k) + t1(:,:,k+1)) / 2;
end
t3 = regrid_vert(t2, woa102_dep, nemo_dep);
woa75_no3 = t3;
woa75_mask = woa75_no3 * 0;

% need to regrid each level of these fields to a regular grid
reg_fld = zeros([180 360 75 6 tsize]) + NaN;
for e = 1:1:tsize
for f = 1:1:6, fprintf('- Ensemble member %d, Field %d  \n', e, f);
  for k = 1:1:74, fprintf('.');
    t1 = mod_fld(:,:,k,f,e);
    t2 = scatregrid(yy100e, xx100e, t1, lat100, lon100);
    reg_fld(:,:,k,f,e) = t2 + woa75_mask(:,:,k);
  end
end
end

% Atlantic and Pacific basins
clear atl_fld pac_fld;
for e = 1:1:tsize
for f = 1:1:6
  for k = 1:1:74
    t0 = reg_fld(:,:,k,f,e);
    t1 = t0 + woa_basin2(:,:,7);
    t2 = nanmean(t1, 2);
    atl_fld(k,:,f,e) = permute(t2, [2 1]);
    t1 = t0 + woa_basin2(:,:,8);
    t2 = nanmean(t1, 2);
    pac_fld(k,:,f,e) = permute(t2, [2 1]);
  end
end
end

save ('ukesm1_PD_section_07jan19_MOD', 'atl_fld', 'pac_fld', 'reg_fld', '-v7.3');

% ====

% load up obs data
% WOA13
load woa13_all_BGC
woa13_ann(:,:,:,1) = mean(woa13_nit, 4);
woa13_ann(:,:,:,2) = mean(woa13_sil, 4);
woa13_ann(:,:,:,3) = mean(woa13_oxy, 4);
% GLODAP v2; DIC and ALK
load per_m3_fields.mat
woa_ann(:,:,:,1) = new_totco2;
woa_ann(:,:,:,2) = new_totalk;
% GLODAP v1; CFC-11
load GLODAPv1_per_m3_fields.mat
woa_ann(:,:,:,3) = new_cfc_11;

clear atl_woa13 pac_woa13;
for f = 1:1:3
  for k = 1:1:102, clear t0;
    t0 = woa13_ann(:,:,k,f);
    %     t0(:,1:180) = woa13_ann(:,181:360,k,f);
    %     t0(:,181:360) = woa13_ann(:,1:180,k,f);
    t1 = t0 + woa_basin2(:,:,7);
    t2 = nanmean(t1, 2);
    atl_woa13(k,:,f) = permute(t2, [2 1]);
    t1 = t0 + woa_basin2(:,:,8);
    t2 = nanmean(t1, 2);
    pac_woa13(k,:,f) = permute(t2, [2 1]);
  end
end

clear atl_woa pac_woa;
for f = 1:1:3
  for k = 1:1:33, clear t0;
    t0 = woa_ann(:,:,k,f);
    %     t0(:,1:180) = woa_ann(:,181:360,k,f);
    %     t0(:,181:360) = woa_ann(:,1:180,k,f);
    t1 = t0 + woa_basin2(:,:,7);
    t2 = nanmean(t1, 2);
    atl_woa(k,:,f) = permute(t2, [2 1]);
    t1 = t0 + woa_basin2(:,:,8);
    t2 = nanmean(t1, 2);
    pac_woa(k,:,f) = permute(t2, [2 1]);
  end
end

save ('ukesm1_PD_section_07jan19_OBS', 'atl_woa13', 'pac_woa13', 'atl_woa', 'pac_woa', '-v7.3');

% ====

load ukesm1_PD_section_07jan19_MOD
load ukesm1_PD_section_07jan19_OBS

% load up background files
load nemo_area100e
load nemo_mask100e

nxx = -179.5:1:179.5;  nyy = -89.5:1:89.5;
[lon100, lat100] = meshgrid(nxx, nyy);
nxx = -180:1:180;  nyy = -90:1:90;
[lon100b, lat100b] = meshgrid(nxx, nyy);

load woa_mask
clear t1; t1(:,1:180) = woa_mask(:,181:360); t1(:,181:360) = woa_mask(:,1:180);
woa_mask = t1;

load woa_basin

% plot this stuff
t1 = ncread('/noc/users/axy/Matlab/WOA_2009/nitrate_annual_1deg.nc', 'depth');
t2 = (t1(1:end-1) + t1(2:end)) / 2;
t3 = [0 t2' 5750]; woa_dep = t3;
t1 = ncread('/noc/users/axy/Matlab/WOA_2013/woa13_all_n00_01.nc', 'depth');
t2 = (t1(1:end-1) + t1(2:end)) / 2;
t3 = [0 t2' 5750]; woa13_dep = t3;
lat = -90:1:90;

fname = sprintf('%s/mean_nemo_%so_1y_20001201-20101201_grid-T.nc', dname(1).name, rname(1).name);
t1 = ncread(fname, 'deptht_bounds');
nemo_dep = double(t1(1,:)); nemo_dep(end+1) = double(t1(2,end));
t1 = ncread(fname, 'deptht');
nemo_mid = double(t1');

tstr = char('Observed, Atlantic', 'Simulated, Atlantic', 'Observed, Pacific', 'Simulated, Pacific');
for f = 1:1:4, tlen(f) = lenstr(tstr(f,:)); end

fnom = char('DIN [mmol N m^{-3}]', 'Silicic acid [mmol Si m^{-3}]', ...
  'Dissolved oxygen [mmol O_{2} m^{-3}]', 'DIC [mmol C m^{-3}]', ...
  'Alkalinity [meq m^{-3}]', 'CFC-11 [nmol m^{-3}]');
for f = 1:1:6, flen(f) = lenstr(fnom(f,:)); end

% flds = N, Si, C, A, O2
clim = [0 50; 0 180; 0 350; 2e3 2.5e3; 2.2e3 2.6e3; 0 7];

tstr = char('Observed, Atlantic-Pacific zonal mean', 'Simulated, Atlantic-Pacific zonal mean', ...
  'Simulated, Atlantic-Pacific zonal st.dev.', 'Simulated, Atlantic-Pacific zonal CoV');
for f = 1:1:4, tlen(f) = lenstr(tstr(f,:)); end

fnom = char('DIN [mmol N m^{-3}]', 'Silicic acid [mmol Si m^{-3}]', ...
  'Dissolved O_{2} [mmol O_{2} m^{-3}]', 'DIC [mmol C m^{-3}]', ...
  'Alkalinity [meq m^{-3}]', 'CFC-11 [nmol m^{-3}]');
for f = 1:1:6, flen(f) = lenstr(fnom(f,:)); end

% set subzero values of passive tracers to infinitesimal positive values
small = 1e-6;
atl_fld(atl_fld < small) = small;
pac_fld(pac_fld < small) = small;

posdel = [-0.025 0 -0.075 0];

for f = 1:1:3
  
  t1 = atl_woa13(:,:,f); t2 = flipdim(t1, 2); t3 = t2(:,1:175);
  t4 = pac_woa13(:,:,f); t5 = t4(:,6:160); t6 = [t3 t5]; t6(end+1,:) = NaN; t6(:,end+1) = NaN;
  this_woa = t6;
  t1 = mean(atl_fld(:,:,f,:), 4); t2 = flipdim(t1, 2); t3 = t2(:,1:175);
  t4 = mean(pac_fld(:,:,f,:), 4); t5 = t4(:,6:160); t6 = [t3 t5]; t6(end+1,:) = NaN; t6(end+1,:) = NaN; t6(:,end+1) = NaN;
  this_fld = t6;
  slat = [0:1:330]; rlat = [90 60 30 0 -30 -60 -60 -30 0 30 60]; srlat = [0 30 60 90 120 150 200 230 260 290 320];
  
  ypos = 6.05;
  
  figure(1); clf; turbo(60);
  subplot (2,1,1);
  pcolor (slat, woa13_dep / 1e3, this_woa); shading flat;
  seafloor(slat, woa13_dep / 1e3, this_woa);  caxis ([clim(f,1) clim(f,2)]);
  set (gca, 'YDir', 'reverse', 'YTick', 0:1:6); yaxis ([0 6.5]);
  set (gca, 'XTick', srlat, 'XTickLabel', rlat);
  tx = text (90, ypos, 'Atlantic'); set(tx, 'HorizontalAlignment', 'Right');
  tx = text (260, ypos, 'Pacific'); set(tx, 'HorizontalAlignment', 'Left');
  tx = text (175, ypos, 'Southern'); set(tx, 'HorizontalAlignment', 'Center');
  hold on; plot ([130 130 220 220], [-1 7 7 -1], 'k:');
  pos = get(gca, 'Position'); pos = pos + posdel; set(gca, 'Position', pos);
  % xlabel ('Latitude [{\circ}N]');
  ylabel ('Depth [km]');
  title (tstr(1,1:tlen(1))); topbox;
  %
  subplot (2,1,2);
  pcolor (slat, nemo_dep / 1e3, this_fld); shading flat;
  seafloor(slat, nemo_dep / 1e3, this_fld);  caxis ([clim(f,1) clim(f,2)]);
  set(gca, 'YDir', 'reverse', 'YTick', 0:1:6); yaxis ([0 6.5]);
  set (gca, 'XTick', srlat, 'XTickLabel', rlat);
  tx = text (90, ypos, 'Atlantic'); set(tx, 'HorizontalAlignment', 'Right');
  tx = text (260, ypos, 'Pacific'); set(tx, 'HorizontalAlignment', 'Left');
  tx = text (175, ypos, 'Southern'); set(tx, 'HorizontalAlignment', 'Center');
  hold on; plot ([130 130 220 220], [-1 7 7 -1], 'k:');
  pos = get(gca, 'Position'); pos = pos + posdel; set(gca, 'Position', pos);
  xlabel ('Latitude [{\circ}N]'); ylabel ('Depth [km]');
  title (tstr(2,1:tlen(2))); topbox;
  pos = get(gca, 'Position'); cb = colorbar; cbpos = get(cb, 'Position');
  cbpos(1) = pos(1) + pos(3) + 0.03; set(cb, 'Position', cbpos); set(gca, 'Position', pos);
  cbar_ylabel(cb, fnom(f, 1:flen(f)));
  %
  def_figure; oname = sprintf('FIGS/ukesm1_PD_THC_section_bgc_0%d.png', f); exportgraphics(gcf, oname, 'Resolution', 600);  
  xlabel (''); 
  def_figure; oname = sprintf('FIGS/ukesm1_PD_THC_section_bgc_NOXLAB_0%d.png', f); exportgraphics(gcf, oname, 'Resolution', 600);  

end

for f = 1:1:3
  
  t1 = atl_woa(:,:,f); t2 = flipdim(t1, 2); t3 = t2(:,1:175);
  t4 = pac_woa(:,:,f); t5 = t4(:,6:160); t6 = [t3 t5]; t6(end+1,:) = NaN; t6(:,end+1) = NaN;
  this_woa = t6;
  t1 = mean(atl_fld(:,:,f+3,:), 4); t2 = flipdim(t1, 2); t3 = t2(:,1:175);
  t4 = mean(pac_fld(:,:,f+3,:), 4); t5 = t4(:,6:160); t6 = [t3 t5]; t6(end+1,:) = NaN; t6(end+1,:) = NaN; t6(:,end+1) = NaN;
  this_fld = t6;
  slat = [0:1:330]; rlat = [90 60 30 0 -30 -60 -60 -30 0 30 60]; srlat = [0 30 60 90 120 150 200 230 260 290 320];
  
  ypos = 6.05;
  
  figure(1); clf; turbo(60);
  subplot (2,1,1);
  pcolor (slat, woa_dep / 1e3, this_woa); shading flat;
  seafloor(slat, woa_dep / 1e3, this_woa);  caxis ([clim(f+3,1) clim(f+3,2)]);
  set (gca, 'YDir', 'reverse', 'YTick', 0:1:6); yaxis ([0 6.5]);
  set (gca, 'XTick', srlat, 'XTickLabel', rlat);
  tx = text (90, ypos, 'Atlantic'); set(tx, 'HorizontalAlignment', 'Right');
  tx = text (260, ypos, 'Pacific'); set(tx, 'HorizontalAlignment', 'Left');
  tx = text (175, ypos, 'Southern'); set(tx, 'HorizontalAlignment', 'Center');
  hold on; plot ([130 130 220 220], [-1 7 7 -1], 'k:');
  pos = get(gca, 'Position'); pos = pos + posdel; set(gca, 'Position', pos);
  % xlabel ('Latitude [{\circ}N]');
  ylabel ('Depth [km]');
  title (tstr(1,1:tlen(1))); topbox;
  %
  subplot (2,1,2);
  pcolor (slat, nemo_dep / 1e3, this_fld); shading flat;
  seafloor(slat, nemo_dep / 1e3, this_fld);  caxis ([clim(f+3,1) clim(f+3,2)]);
  set(gca, 'YDir', 'reverse', 'YTick', 0:1:6); yaxis ([0 6.5]);
  set (gca, 'XTick', srlat, 'XTickLabel', rlat);
  tx = text (90, ypos, 'Atlantic'); set(tx, 'HorizontalAlignment', 'Right');
  tx = text (260, ypos, 'Pacific'); set(tx, 'HorizontalAlignment', 'Left');
  tx = text (175, ypos, 'Southern'); set(tx, 'HorizontalAlignment', 'Center');
  hold on; plot ([130 130 220 220], [-1 7 7 -1], 'k:');
  pos = get(gca, 'Position'); pos = pos + posdel; set(gca, 'Position', pos);
  xlabel ('Latitude [{\circ}N]'); ylabel ('Depth [km]');
  title (tstr(2,1:tlen(2))); topbox;
  pos = get(gca, 'Position'); cb = colorbar; cbpos = get(cb, 'Position');
  cbpos(1) = pos(1) + pos(3) + 0.03; set(cb, 'Position', cbpos); set(gca, 'Position', pos);
  cbar_ylabel(cb, fnom(f+3, 1:flen(f+3)));
  %
  def_figure; oname = sprintf('FIGS/ukesm1_PD_THC_section_bgc_0%d.png', f+3); exportgraphics(gcf, oname, 'Resolution', 600);  
  xlabel (''); 
  def_figure; oname = sprintf('FIGS/ukesm1_PD_THC_section_bgc_NOXLAB_0%d.png', f+3); exportgraphics(gcf, oname, 'Resolution', 600);  

end

% repeat, but for St. Dev.
for f = 1:1:3
  
  t1 = mean(atl_fld(:,:,f,:), 4); t2 = flipdim(t1, 2); t3 = t2(:,1:175);
  t4 = mean(pac_fld(:,:,f,:), 4); t5 = t4(:,6:160); t6 = [t3 t5]; t6(end+1,:) = NaN; t6(end+1,:) = NaN; t6(:,end+1) = NaN;
  this_fld = t6;
  t1 = std(atl_fld(:,:,f,:), [], 4); t2 = flipdim(t1, 2); t3 = t2(:,1:175);
  t4 = std(pac_fld(:,:,f,:), [], 4); t5 = t4(:,6:160); t6 = [t3 t5]; t6(end+1,:) = NaN; t6(end+1,:) = NaN; t6(:,end+1) = NaN;
  this_cov = t6;

  slat = [0:1:330]; rlat = [90 60 30 0 -30 -60 -60 -30 0 30 60]; srlat = [0 30 60 90 120 150 200 230 260 290 320];
  
  ypos = 6.05;
  
  figure(1); clf; turbo(60);
  subplot (2,1,1);
  pcolor (slat, nemo_dep / 1e3, this_fld); shading flat;
  seafloor(slat, nemo_dep / 1e3, this_fld);  caxis ([clim(f,1) clim(f,2)]);
  set(gca, 'YDir', 'reverse', 'YTick', 0:1:6); yaxis ([0 6.5]);
  set (gca, 'XTick', srlat, 'XTickLabel', rlat);
  tx = text (90, ypos, 'Atlantic'); set(tx, 'HorizontalAlignment', 'Right');
  tx = text (260, ypos, 'Pacific'); set(tx, 'HorizontalAlignment', 'Left');
  tx = text (175, ypos, 'Southern'); set(tx, 'HorizontalAlignment', 'Center');
  hold on; plot ([130 130 220 220], [-1 7 7 -1], 'k:');
  pos = get(gca, 'Position'); pos(1) = pos(1) - 0.075; set(gca, 'Position', pos);
  xlabel ('Latitude [{\circ}N]'); ylabel ('Depth [km]');
  title (tstr(2,1:tlen(2))); topbox;
  pos = get(gca, 'Position'); cb = colorbar; cbpos = get(cb, 'Position');
  cbpos(1) = pos(1) + pos(3) + 0.03; set(cb, 'Position', cbpos); set(gca, 'Position', pos);
  cbar_ylabel(cb, fnom(f, 1:flen(f)));
  subplot (2,1,2);
  [mn, mx] = plot_range(this_cov, 1); mn = 0 - 1e-6; % if (mx > 2), mx = 2; end;
  pcolor (slat, nemo_dep / 1e3, this_cov); shading flat;
  seafloor(slat, nemo_dep / 1e3, this_cov);  caxis ([mn mx]);
  set(gca, 'YDir', 'reverse', 'YTick', 0:1:6); yaxis ([0 6.5]);
  set (gca, 'XTick', srlat, 'XTickLabel', rlat);
  tx = text (90, ypos, 'Atlantic'); set(tx, 'HorizontalAlignment', 'Right');
  tx = text (260, ypos, 'Pacific'); set(tx, 'HorizontalAlignment', 'Left');
  tx = text (175, ypos, 'Southern'); set(tx, 'HorizontalAlignment', 'Center');
  hold on; plot ([130 130 220 220], [-1 7 7 -1], 'k:');
  pos = get(gca, 'Position'); pos(1) = pos(1) - 0.075; set(gca, 'Position', pos);
  xlabel ('Latitude [{\circ}N]'); ylabel ('Depth [km]');
  title (tstr(3,1:tlen(3))); topbox;
  pos = get(gca, 'Position'); cb = colorbar; cbpos = get(cb, 'Position');
  cbpos(1) = pos(1) + pos(3) + 0.03; set(cb, 'Position', cbpos); set(gca, 'Position', pos);
  cbar_ylabel(cb, fnom(f, 1:flen(f)));
  %
  def_figure; oname = sprintf('ukesm1_PD_THC_section_bgc_std_0%d', f);
  print ('-depsc', '-painters', oname);
  pngconvert(oname, 600, pngopt);
  
end

for f = 1:1:3
  
  t1 = mean(atl_fld(:,:,f+3,:), 4); t2 = flipdim(t1, 2); t3 = t2(:,1:175);
  t4 = mean(pac_fld(:,:,f+3,:), 4); t5 = t4(:,6:160); t6 = [t3 t5]; t6(end+1,:) = NaN; t6(end+1,:) = NaN; t6(:,end+1) = NaN;
  this_fld = t6;
  t1 = std(atl_fld(:,:,f+3,:), [], 4); t2 = flipdim(t1, 2); t3 = t2(:,1:175);
  t4 = std(pac_fld(:,:,f+3,:), [], 4); t5 = t4(:,6:160); t6 = [t3 t5]; t6(end+1,:) = NaN; t6(end+1,:) = NaN; t6(:,end+1) = NaN;
  this_cov = t6;

  slat = [0:1:330]; rlat = [90 60 30 0 -30 -60 -60 -30 0 30 60]; srlat = [0 30 60 90 120 150 200 230 260 290 320];
  
  ypos = 6.05;
  
  figure(1); clf; turbo(60);
  subplot (2,1,1);
  pcolor (slat, nemo_dep / 1e3, this_fld); shading flat;
  seafloor(slat, nemo_dep / 1e3, this_fld);  caxis ([clim(f+3,1) clim(f+3,2)]);
  set(gca, 'YDir', 'reverse', 'YTick', 0:1:6); yaxis ([0 6.5]);
  set (gca, 'XTick', srlat, 'XTickLabel', rlat);
  tx = text (90, ypos, 'Atlantic'); set(tx, 'HorizontalAlignment', 'Right');
  tx = text (260, ypos, 'Pacific'); set(tx, 'HorizontalAlignment', 'Left');
  tx = text (175, ypos, 'Southern'); set(tx, 'HorizontalAlignment', 'Center');
  hold on; plot ([130 130 220 220], [-1 7 7 -1], 'k:');
  pos = get(gca, 'Position'); pos(1) = pos(1) - 0.075; set(gca, 'Position', pos);
  xlabel ('Latitude [{\circ}N]'); ylabel ('Depth [km]');
  title (tstr(2,1:tlen(2))); topbox;
  pos = get(gca, 'Position'); cb = colorbar; cbpos = get(cb, 'Position');
  cbpos(1) = pos(1) + pos(3) + 0.03; set(cb, 'Position', cbpos); set(gca, 'Position', pos);
  cbar_ylabel(cb, fnom(f+3, 1:flen(f+3)));
  subplot (2,1,2);
  [mn, mx] = plot_range(this_cov, 1); mn = 0 - 1e-6; % if (mx > 2), mx = 2; end;
  pcolor (slat, nemo_dep / 1e3, this_cov); shading flat;
  seafloor(slat, nemo_dep / 1e3, this_cov);  caxis ([mn mx]);
  set(gca, 'YDir', 'reverse', 'YTick', 0:1:6); yaxis ([0 6.5]);
  set (gca, 'XTick', srlat, 'XTickLabel', rlat);
  tx = text (90, ypos, 'Atlantic'); set(tx, 'HorizontalAlignment', 'Right');
  tx = text (260, ypos, 'Pacific'); set(tx, 'HorizontalAlignment', 'Left');
  tx = text (175, ypos, 'Southern'); set(tx, 'HorizontalAlignment', 'Center');
  hold on; plot ([130 130 220 220], [-1 7 7 -1], 'k:');
  pos = get(gca, 'Position'); pos(1) = pos(1) - 0.075; set(gca, 'Position', pos);
  xlabel ('Latitude [{\circ}N]'); ylabel ('Depth [km]');
  title (tstr(3,1:tlen(3))); topbox;
  pos = get(gca, 'Position'); cb = colorbar; cbpos = get(cb, 'Position');
  cbpos(1) = pos(1) + pos(3) + 0.03; set(cb, 'Position', cbpos); set(gca, 'Position', pos);
  cbar_ylabel(cb, fnom(f, 1:flen(f)));
  %
  def_figure; oname = sprintf('ukesm1_PD_THC_section_bgc_std_0%d', f+3);
  print ('-depsc', '-painters', oname);
  pngconvert(oname, 600, pngopt);
  
end

% repeat, but for CoV
for f = 1:1:3
  
  t1 = mean(atl_fld(:,:,f,:), 4); t2 = flipdim(t1, 2); t3 = t2(:,1:175);
  t4 = mean(pac_fld(:,:,f,:), 4); t5 = t4(:,6:160); t6 = [t3 t5]; t6(end+1,:) = NaN; t6(end+1,:) = NaN; t6(:,end+1) = NaN;
  this_fld = t6;
  t1 = std(atl_fld(:,:,f,:), [], 4); t2 = flipdim(t1, 2); t3 = t2(:,1:175);
  t4 = std(pac_fld(:,:,f,:), [], 4); t5 = t4(:,6:160); t6 = [t3 t5]; t6(end+1,:) = NaN; t6(end+1,:) = NaN; t6(:,end+1) = NaN;
  this_cov = t6 ./ this_fld;

  slat = [0:1:330]; rlat = [90 60 30 0 -30 -60 -60 -30 0 30 60]; srlat = [0 30 60 90 120 150 200 230 260 290 320];
  
  ypos = 6.05;
  
  figure(1); clf; turbo(60);
  subplot (2,1,1);
  pcolor (slat, nemo_dep / 1e3, this_fld); shading flat;
  seafloor(slat, nemo_dep / 1e3, this_fld);  caxis ([clim(f,1) clim(f,2)]);
  set(gca, 'YDir', 'reverse', 'YTick', 0:1:6); yaxis ([0 6.5]);
  set (gca, 'XTick', srlat, 'XTickLabel', rlat);
  tx = text (90, ypos, 'Atlantic'); set(tx, 'HorizontalAlignment', 'Right');
  tx = text (260, ypos, 'Pacific'); set(tx, 'HorizontalAlignment', 'Left');
  tx = text (175, ypos, 'Southern'); set(tx, 'HorizontalAlignment', 'Center');
  hold on; plot ([130 130 220 220], [-1 7 7 -1], 'k:');
  pos = get(gca, 'Position'); pos(1) = pos(1) - 0.075; set(gca, 'Position', pos);
  xlabel ('Latitude [{\circ}N]'); ylabel ('Depth [km]');
  title (tstr(2,1:tlen(2))); topbox;
  pos = get(gca, 'Position'); cb = colorbar; cbpos = get(cb, 'Position');
  cbpos(1) = pos(1) + pos(3) + 0.03; set(cb, 'Position', cbpos); set(gca, 'Position', pos);
  cbar_ylabel(cb, fnom(f, 1:flen(f)));
  subplot (2,1,2);
  [mn, mx] = plot_range(this_cov, 1, 0.05); mn = 0 - 1e-6; if (mx > 2), mx = 2; end;
  pcolor (slat, nemo_dep / 1e3, this_cov); shading flat;
  seafloor(slat, nemo_dep / 1e3, this_cov);  caxis ([mn mx]);
  set(gca, 'YDir', 'reverse', 'YTick', 0:1:6); yaxis ([0 6.5]);
  set (gca, 'XTick', srlat, 'XTickLabel', rlat);
  tx = text (90, ypos, 'Atlantic'); set(tx, 'HorizontalAlignment', 'Right');
  tx = text (260, ypos, 'Pacific'); set(tx, 'HorizontalAlignment', 'Left');
  tx = text (175, ypos, 'Southern'); set(tx, 'HorizontalAlignment', 'Center');
  hold on; plot ([130 130 220 220], [-1 7 7 -1], 'k:');
  pos = get(gca, 'Position'); pos(1) = pos(1) - 0.075; set(gca, 'Position', pos);
  xlabel ('Latitude [{\circ}N]'); ylabel ('Depth [km]');
  title (tstr(4,1:tlen(4))); topbox;
  pos = get(gca, 'Position'); cb = colorbar; cbpos = get(cb, 'Position');
  cbpos(1) = pos(1) + pos(3) + 0.03; set(cb, 'Position', cbpos); set(gca, 'Position', pos);
  cbar_ylabel(cb, 'CoV [-]');
  %
  def_figure; oname = sprintf('ukesm1_PD_THC_section_bgc_CoV_0%d', f);
  print ('-depsc', '-painters', oname);
  pngconvert(oname, 600, pngopt);
  
end

for f = 1:1:3
  
  t1 = mean(atl_fld(:,:,f+3,:), 4); t2 = flipdim(t1, 2); t3 = t2(:,1:175);
  t4 = mean(pac_fld(:,:,f+3,:), 4); t5 = t4(:,6:160); t6 = [t3 t5]; t6(end+1,:) = NaN; t6(end+1,:) = NaN; t6(:,end+1) = NaN;
  this_fld = t6;
  t1 = std(atl_fld(:,:,f+3,:), [], 4); t2 = flipdim(t1, 2); t3 = t2(:,1:175);
  t4 = std(pac_fld(:,:,f+3,:), [], 4); t5 = t4(:,6:160); t6 = [t3 t5]; t6(end+1,:) = NaN; t6(end+1,:) = NaN; t6(:,end+1) = NaN;
  this_cov = t6 ./ this_fld; 

  slat = [0:1:330]; rlat = [90 60 30 0 -30 -60 -60 -30 0 30 60]; srlat = [0 30 60 90 120 150 200 230 260 290 320];
  
  ypos = 6.05;
  
  figure(1); clf; turbo(60);
  subplot (2,1,1);
  pcolor (slat, nemo_dep / 1e3, this_fld); shading flat;
  seafloor(slat, nemo_dep / 1e3, this_fld);  caxis ([clim(f+3,1) clim(f+3,2)]);
  set(gca, 'YDir', 'reverse', 'YTick', 0:1:6); yaxis ([0 6.5]);
  set (gca, 'XTick', srlat, 'XTickLabel', rlat);
  tx = text (90, ypos, 'Atlantic'); set(tx, 'HorizontalAlignment', 'Right');
  tx = text (260, ypos, 'Pacific'); set(tx, 'HorizontalAlignment', 'Left');
  tx = text (175, ypos, 'Southern'); set(tx, 'HorizontalAlignment', 'Center');
  hold on; plot ([130 130 220 220], [-1 7 7 -1], 'k:');
  pos = get(gca, 'Position'); pos(1) = pos(1) - 0.075; set(gca, 'Position', pos);
  xlabel ('Latitude [{\circ}N]'); ylabel ('Depth [km]');
  title (tstr(2,1:tlen(2))); topbox;
  pos = get(gca, 'Position'); cb = colorbar; cbpos = get(cb, 'Position');
  cbpos(1) = pos(1) + pos(3) + 0.03; set(cb, 'Position', cbpos); set(gca, 'Position', pos);
  cbar_ylabel(cb, fnom(f+3, 1:flen(f+3)));
  subplot (2,1,2);
  [mn, mx] = plot_range(this_cov, 1, 0.05); mn = 0 - 1e-6; if (mx > 2), mx = 2; end;
  pcolor (slat, nemo_dep / 1e3, this_cov); shading flat;
  seafloor(slat, nemo_dep / 1e3, this_cov);  caxis ([mn mx]);
  set(gca, 'YDir', 'reverse', 'YTick', 0:1:6); yaxis ([0 6.5]);
  set (gca, 'XTick', srlat, 'XTickLabel', rlat);
  tx = text (90, ypos, 'Atlantic'); set(tx, 'HorizontalAlignment', 'Right');
  tx = text (260, ypos, 'Pacific'); set(tx, 'HorizontalAlignment', 'Left');
  tx = text (175, ypos, 'Southern'); set(tx, 'HorizontalAlignment', 'Center');
  hold on; plot ([130 130 220 220], [-1 7 7 -1], 'k:');
  pos = get(gca, 'Position'); pos(1) = pos(1) - 0.075; set(gca, 'Position', pos);
  xlabel ('Latitude [{\circ}N]'); ylabel ('Depth [km]');
  title (tstr(4,1:tlen(4))); topbox;
  pos = get(gca, 'Position'); cb = colorbar; cbpos = get(cb, 'Position');
  cbpos(1) = pos(1) + pos(3) + 0.03; set(cb, 'Position', cbpos); set(gca, 'Position', pos);
  cbar_ylabel(cb, 'CoV [-]');
  %
  def_figure; oname = sprintf('ukesm1_PD_THC_section_bgc_CoV_0%d', f+3);
  print ('-depsc', '-painters', oname);
  pngconvert(oname, 600, pngopt);
  
end

% pdfconvert('ukesm1_THC_section_', 'ukesm1_THC_section_plots', 3);

tstr = char('Difference Atlantic-Pacific zonal mean');
for f = 1:1:1, tlen(f) = lenstr(tstr(f,:)); end

fnom = char('{\Delta} DIN [mmol N m^{-3}]', '{\Delta} Silicic acid [mmol Si m^{-3}]', ...
  '{\Delta} Dissolved O_{2} [mmol O_{2} m^{-3}]', '{\Delta} DIC [mmol C m^{-3}]', ...
  '{\Delta} Alkalinity [meq m^{-3}]', '{\Delta} CFC-11 [nmol m^{-3}]');
for f = 1:1:6, flen(f) = lenstr(fnom(f,:)); end

% flds = N, Si, O2
clim = [-15 15; -50 50; -50 50; -100 100; -100 100; -3 3];

for f = 1:1:3
  
  t1 = atl_woa13(:,:,f); t2 = flipdim(t1, 2); t3 = t2(:,1:175);
  t4 = pac_woa13(:,:,f); t5 = t4(:,6:160); t6 = [t3 t5]; t6(end+1,:) = NaN; t6(:,end+1) = NaN;
  this_woa = t6;
  t1 = mean(atl_fld(:,:,f,:), 4); t2 = flipdim(t1, 2); t3 = t2(:,1:175);
  t4 = mean(pac_fld(:,:,f,:), 4); t5 = t4(:,6:160); t6 = [t3 t5]; t6(end+1,:) = NaN; t6(end+1,:) = NaN; t6(:,end+1) = NaN;
  this_fld = t6;
  slat = [0:1:330]; rlat = [90 60 30 0 -30 -60 -60 -30 0 30 60]; srlat = [0 30 60 90 120 150 200 230 260 290 320];
  
  this_msk = this_fld * 0;
  t0 = permute(this_woa, [2 3 1]);
  t1 = regrid_vert(t0, woa13_dep/1e3, nemo_dep/1e3);
  t2 = permute(t1, [3 1 2]); t2(end+1,:) = NaN;
  t3 = floodfill(t2, 0, 4, 0);
  this_woa2 = t3 + this_msk;
  
  this_delta = this_fld - this_woa2;
  
  ypos = 6.05;
  
  figure(1); clf; therm6(31);
  subplot (2,1,2)
  pcolor (slat, nemo_dep / 1e3, this_delta); shading flat;
  seafloor(slat, nemo_dep / 1e3, this_delta);  caxis ([clim(f,1) clim(f,2)]);
  set(gca, 'YDir', 'reverse', 'YTick', 0:1:6); yaxis ([0 6.5]);
  set (gca, 'XTick', srlat, 'XTickLabel', rlat);
  tx = text (90, ypos, 'Atlantic'); set(tx, 'HorizontalAlignment', 'Right');
  tx = text (260, ypos, 'Pacific'); set(tx, 'HorizontalAlignment', 'Left');
  tx = text (175, ypos, 'Southern'); set(tx, 'HorizontalAlignment', 'Center');
  hold on; plot ([130 130 220 220], [-1 7 7 -1], 'k:');
  pos = get(gca, 'Position'); pos = pos + posdel; set(gca, 'Position', pos);
  xlabel ('Latitude [{\circ}N]'); ylabel ('Depth [km]');
  title (tstr(1,1:tlen(1))); topbox;
  pos = get(gca, 'Position'); cb = colorbar; cbpos = get(cb, 'Position');
  cbpos(1) = pos(1) + pos(3) + 0.03; set(cb, 'Position', cbpos); set(gca, 'Position', pos);
  cbar_ylabel(cb, fnom(f, 1:flen(f)));
  %
  def_figure; oname = sprintf('FIGS/ukesm1_PD_THC_section_bgc_delta_0%d.png', f); exportgraphics(gcf, oname, 'Resolution', 600);  
end

for f = 1:1:3
  
  t1 = atl_woa(:,:,f); t2 = flipdim(t1, 2); t3 = t2(:,1:175);
  t4 = pac_woa(:,:,f); t5 = t4(:,6:160); t6 = [t3 t5]; t6(end+1,:) = NaN; t6(:,end+1) = NaN;
  this_woa = t6;
  t1 = mean(atl_fld(:,:,f+3,:), 4); t2 = flipdim(t1, 2); t3 = t2(:,1:175);
  t4 = mean(pac_fld(:,:,f+3,:), 4); t5 = t4(:,6:160); t6 = [t3 t5]; t6(end+1,:) = NaN; t6(end+1,:) = NaN; t6(:,end+1) = NaN;
  this_fld = t6;
  slat = [0:1:330]; rlat = [90 60 30 0 -30 -60 -60 -30 0 30 60]; srlat = [0 30 60 90 120 150 200 230 260 290 320];
  
  this_msk = this_fld * 0;
  t0 = permute(this_woa, [2 3 1]);
  t1 = regrid_vert(t0, woa_dep/1e3, nemo_dep/1e3);
  t2 = permute(t1, [3 1 2]); t2(end+1,:) = NaN;
  t3 = floodfill(t2, 0, 4, 0);
  this_woa2 = t3 + this_msk;
  if f == 3, this_woa2(:,1:25) = NaN; end
  
  this_delta = this_fld - this_woa2;
  
  ypos = 6.05;
  
  figure(1); clf; therm6(31);
  subplot (2,1,2)
  pcolor (slat, nemo_dep / 1e3, this_delta); shading flat;
  seafloor(slat, nemo_dep / 1e3, this_delta);  caxis ([clim(f+3,1) clim(f+3,2)]);
  set(gca, 'YDir', 'reverse', 'YTick', 0:1:6); yaxis ([0 6.5]);
  set (gca, 'XTick', srlat, 'XTickLabel', rlat);
  tx = text (90, ypos, 'Atlantic'); set(tx, 'HorizontalAlignment', 'Right');
  tx = text (260, ypos, 'Pacific'); set(tx, 'HorizontalAlignment', 'Left');
  tx = text (175, ypos, 'Southern'); set(tx, 'HorizontalAlignment', 'Center');
  hold on; plot ([130 130 220 220], [-1 7 7 -1], 'k:');
  pos = get(gca, 'Position'); pos = pos + posdel; set(gca, 'Position', pos);
  xlabel ('Latitude [{\circ}N]'); ylabel ('Depth [km]');
  title (tstr(1,1:tlen(1))); topbox;
  pos = get(gca, 'Position'); cb = colorbar; cbpos = get(cb, 'Position');
  cbpos(1) = pos(1) + pos(3) + 0.03; set(cb, 'Position', cbpos); set(gca, 'Position', pos);
  cbar_ylabel(cb, fnom(f+3, 1:flen(f+3)));
  %
  def_figure; oname = sprintf('FIGS/ukesm1_PD_THC_section_bgc_delta_0%d.png', f+3); exportgraphics(gcf, oname, 'Resolution', 600);  
end
