clear; startup;

% =========================================================================
% =========================================================================
%
% Matlab plotting scripts for the analysis of UKESM1 prepared for:
%
%   Yool, A., Palmiéri, J., Jones, C. G., de Mora, L., Kuhlbrodt, T., 
%   Popova, E. E., Nurser, A. J. G., Hirschi, J., Blaker, A. T., 
%	Coward, A. C., Blockley, E. W., and Sellar, A. A.: Evaluating the 
% 	physical and biogeochemical state of the global ocean component of 
%	UKESM1 in CMIP6 Historical simulations, Geosci. Model Dev. Discuss. 
%	[preprint], https://doi.org/10.5194/gmd-2020-333, in review, 2020.
%
%   Accepted for publication: 20 April 2021
%
% Note: scripts provided to document analysis, and may require editing to
% fully replicate plots in manuscript
%
% =========================================================================
% =========================================================================

% =========================================================================
%
% This script plots Figures 4 and 5
%
% =========================================================================

% =========================================================================

load ukesm1_PD_ensemble

% =========================================================================

% load up background files
load nemo_area100e
load nemo_mask100e
fname = sprintf('%s/mean_nemo_%so_1y_20001201-20101201_grid-T.nc', dname(1).name, rname(1).name);
t1 = ncread(fname, 'deptht_bounds');
nemo_dep = double(t1(1,:)); nemo_dep(end+1) = double(t1(2,end));
t1 = ncread(fname, 'deptht');
nemo_mid = double(t1');

% total grid area
t1 = area100e + nemo_mask100e(:,:,1);
t2 = isfinite(t1);
tot_area100e = sum(t1(t2)); % m2

% extra mask to get rid of Mediterranean, Black and Caspian Seas
t1 = nemo_mask100e(:,:,1) + 1;
t1(isnan(t1)) = 0;
t1(235:249,283:344) = t1(235:249,283:344) + 2;
t1(249:262,291:344) = t1(249:262,291:344) + 4;
t1(262:276,330:344) = t1(262:276,330:344) + 8;
t2 = t1;
t2(t1 < 1) = NaN; t2(t1 > 1) = NaN;
med_mask100e = t2;

nxx = -179.5:1:179.5;  nyy = -89.5:1:89.5;
[lon100, lat100] = meshgrid(nxx, nyy);
nxx = -180:1:180;  nyy = -90:1:90;
[lon100b, lat100b] = meshgrid(nxx, nyy);

load woa_mask
clear t1; t1(:,1:180) = woa_mask(:,181:360); t1(:,181:360) = woa_mask(:,1:180);
woa_mask = t1;

load woa_basin

helvetica;
pngopt = 0;

% ======================================================================
% ======================================================================
% ======================================================================

% load up WOA 2013 (1995-2004)
load woa13_95A4_TS

% ======================================================================

% load up NEMO output
vnom = char('thetao', 'so'); vlen = [6 2];
mod_ptm = zeros([332 362 75 12 tsize]);
mod_sal = zeros([332 362 75 12 tsize]);

for e = 1:1:tsize
  for m = 1:1:12, mn = numprint(m, 2);
    iname = sprintf('%s/clim_nemo_%so_1m_2000%s-2009%s_grid-T.nc', dname(e).name, rname(e).name, mn, mn);
    fprintf('- Reading %s\n', iname);
    for f = 1:1:2
      t1 = ncread(iname, vnom(f,1:vlen(f)));
      t2 = permute(t1, [2 1 3]); land = t2(1,1,1); t2(t2 == land) = NaN;
      if f == 1, mod_ptm(:,:,:,m,e) = t2;
      else, mod_sal(:,:,:,m,e) = t2; end
    end; end
end

dep_imarnet;
t1 = cumsum(imarnet_dep(:,4));
l75_dep = [0; t1];

load nemo_mask100e
mod_lat = repmat(yy100e, [1 1 75 12 tsize]);
l75_mid = (l75_dep(1:1:end-1) + l75_dep(2:1:end)) / 2;
mod_dep = repmat(permute(l75_mid, [2 3 1]), [332 362 1 12 tsize]);

% create pressure array
mod_prs = sw_pres(mod_dep, mod_lat);

% create potential density array
mod_sig = sw_pden(mod_sal, mod_ptm, mod_prs, (mod_prs * 0));

% ======================================================================
% ======================================================================
% ======================================================================

woa13_msk = woa13_tmp(:,:,1,1) * 0;

% what's the greatest depth in WOA?
woa13_zdep = nanmax(woa13_dpt(:,:,:,1), [], 3) + woa13_msk;
t1 = isfinite(woa13_tmp(:,:,:,1));
woa13_kdep = nansum(t1, 3) + woa13_msk;

t1 = nanmean(woa13_dpt(:,:,:,1), 1); t2 = nanmean(t1, 2);
dep = squeeze(t2);

deep_switch = 0;

% set level to reference WOA to; 1 = 0m, 2 = 5m, 3 = 10m, etc.
lev = 2;

% find MLD, T
crit = 0.5;
for m = 1:1:12
  t1 = woa13_ptm(:,:,:,m);
  t2 = repmat(t1(:,:,lev), [1 1 102]) - t1;
  [t3,t4] = oc5_finddensity(t2, crit, dep);
  t5 = isnan(t4); t6 = isfinite(t1(:,:,1)); t7 = t5 + t6; t8 = find(t7 == 2);
  if deep_switch == 1, t4(t8) = woa13_zdep(t8);
  else, t3(t8) = 1; t4(t8) = 0; end
  woa13_kmld1(:,:,m) = t3;
  woa13_zmld1(:,:,m) = t4;
end

% find MLD, sigma
crit = 0.125;
for m = 1:1:12
  t1 = woa13_sig(:,:,:,m);
  t2 = t1 - repmat(t1(:,:,lev), [1 1 102]);
  [t3,t4] = oc5_finddensity(t2, crit, dep);
  t5 = isnan(t4); t6 = isfinite(t1(:,:,1)); t7 = t5 + t6; t8 = find(t7 == 2);
  if deep_switch == 1, t4(t8) = woa13_zdep(t8);
  else, t3(t8) = 1; t4(t8) = 0; end
  woa13_kmld2(:,:,m) = t3;
  woa13_zmld2(:,:,m) = t4;
end

% find MLD, variable sigma
crit = 0.5;
for m = 1:1:12
  t1 = woa13_sig(:,:,:,m);
  q1 = woa13_ptm(:,:,lev,m) - crit;
  % q1 = ((woa13_ptm(:,:,1,m) + woa13_ptm(:,:,2,m)) / 2) - crit;
  q2 = sw_pden(woa13_sal(:,:,lev,m), q1, woa13_prs(:,:,lev,m), woa13_prs(:,:,lev,m) * 0);
  q3 = q2 - t1(:,:,1);
  t2 = (t1 - repmat(t1(:,:,1), [1 1 102])) ./ repmat(q3, [1 1 102]);
  [t3,t4] = oc5_finddensity(t2, 1, dep);
  t5 = isnan(t4); t6 = isfinite(t1(:,:,1)); t7 = t5 + t6; t8 = find(t7 == 2);
  if deep_switch == 1, t4(t8) = woa13_zdep(t8);
  else, t3(t8) = 1; t4(t8) = 0; end
  woa13_kmld3(:,:,m) = t3;
  woa13_zmld3(:,:,m) = t4;
end

% what's the greatest depth in NEMO?
t1 = isfinite(mod_ptm(:,:,:,1,1));
t2 = sum(t1, 3);
t3 = mod_ptm(:,:,1,1,1) * 0;
for k = 1:1:75, t4 = find(t2 == k); t3(t4) = l75_dep(k+1); end
mod_zdep = t3 + mod_ptm(:,:,1,1,1) * 0;
mod_kdep = t2 + mod_ptm(:,:,1,1,1) * 0;

% set level to reference NEMO to; 1 = ~0.5m, 5 = ~5m, 8 = ~10m, etc.
lev = 5;

% find MLD, T
crit = 0.5;
for e = 1:1:tsize
  for m = 1:1:12
    t1 = mod_ptm(:,:,:,m,e);
    t2 = repmat(t1(:,:,lev), [1 1 75]) - t1;
    [t3,t4] = oc5_finddensity(t2, crit, l75_mid);
    t5 = isnan(t4); t6 = isfinite(t1(:,:,1)); t7 = t5 + t6; t8 = find(t7 == 2);
    if deep_switch == 1, t4(t8) = mod_zdep(t8);
    else, t3(t8) = 1; t4(t8) = 0; end
    mod_kmld1(:,:,m,e) = t3;
    mod_zmld1(:,:,m,e) = t4;
  end
end

% find MLD, sigma
crit = 0.125;
for e = 1:1:tsize
  for m = 1:1:12
    t1 = mod_sig(:,:,:,m,e);
    t2 = t1 - repmat(t1(:,:,lev), [1 1 75]);
    [t3,t4] = oc5_finddensity(t2, crit, l75_mid);
    t5 = isnan(t4); t6 = isfinite(t1(:,:,1)); t7 = t5 + t6; t8 = find(t7 == 2);
    if deep_switch == 1, t4(t8) = mod_zdep(t8);
    else, t3(t8) = 1; t4(t8) = 0; end
    mod_kmld2(:,:,m,e) = t3;
    mod_zmld2(:,:,m,e) = t4;
  end
end

% find MLD, variable sigma
crit = 0.5;
for e = 1:1:tsize
  for m = 1:1:12
    t1 = mod_sig(:,:,:,m,e);
    q1 = mod_ptm(:,:,lev,m,e) - crit;
    q2 = sw_pden(mod_sal(:,:,lev,m,e), q1, mod_prs(:,:,lev,1,e), mod_prs(:,:,lev,1,e) * 0);
    q3 = q2 - t1(:,:,1);
    t2 = (t1 - repmat(t1(:,:,1), [1 1 75])) ./ repmat(q3, [1 1 75]);
    [t3,t4] = oc5_finddensity(t2, 1, l75_mid);
    t5 = isnan(t4); t6 = isfinite(t1(:,:,1)); t7 = t5 + t6; t8 = find(t7 == 2);
    if deep_switch == 1, t4(t8) = mod_zdep(t8);
    else, t3(t8) = 1; t4(t8) = 0; end
    mod_kmld3(:,:,m,e) = t3;
    mod_zmld3(:,:,m,e) = t4;
  end
end

ann_woa13_mld(:,:,1) = mean(woa13_zmld1, 3);
ann_woa13_mld(:,:,2) = mean(woa13_zmld2, 3);
ann_woa13_mld(:,:,3) = mean(woa13_zmld3, 3);
ann_mod_mld(:,:,1,:) = mean(mod_zmld1, 3);
ann_mod_mld(:,:,2,:) = mean(mod_zmld2, 3);
ann_mod_mld(:,:,3,:) = mean(mod_zmld3, 3);

% save ('ukesm1_PD_mixedlayer_OBS', 'woa13_zmld*', 'woa13_kmld*', 'ann_woa13_mld', '-V7.3');
% save ('ukesm1_PD_mixedlayer_MOD', 'mod_zmld*', 'mod_kmld*', 'ann_mod_mld', '-V7.3');

% ======================================================================
% ======================================================================
% ======================================================================

% pycnocline

% put WOA into "layers"
for m = 1:1:12, for k = 1:1:101
    t1 = woa13_sig(:,:,k:k+1,m);
    t2 = (t1(:,:,1) + t1(:,:,2)) / 2;
    woa13_sig2(:,:,k,m) = t2;
  end; end

woa13_thk = dep(2:end) - dep(1:end-1);
woa13_mid = (dep(1:end-1) + dep(2:end)) / 2;

clear woa13_pyc;
for m = 1:1:12
  t1 = woa13_sig2(:,:,:,m);
  t2 = max(t1, [], 3);
  tl_sum = t1(:,:,1) * 0;
  bl_sum = t1(:,:,1) * 0;
  for k = 1:1:101
    q1 = t1(:,:,k);
    q2 = (t2 - q1); q2(isnan(q2)) = 0;
    tl_sum = tl_sum + (q2 * woa13_thk(k) * woa13_mid(k));
    bl_sum = bl_sum + (q2 * woa13_thk(k));
  end
  woa13_pyc(:,:,m) = tl_sum ./ bl_sum;
end

l75_thk = imarnet_dep(:,4);

clear mod_pyc
for e = 1:1:tsize
  for m = 1:1:12
    t1 = mod_sig(:,:,:,m,e);
    t2 = max(t1, [], 3);
    tl_sum = t1(:,:,1) * 0;
    bl_sum = t1(:,:,1) * 0;
    for k = 1:1:75
      q1 = t1(:,:,k);
      q2 = (t2 - q1); q2(isnan(q2)) = 0;
      tl_sum = tl_sum + (q2 * l75_thk(k) * l75_mid(k));
      bl_sum = bl_sum + (q2 * l75_thk(k));
    end
    mod_pyc(:,:,m,e) = tl_sum ./ bl_sum;
  end
end

% ======================================================================
% ======================================================================
% ======================================================================

save ('ukesm1_PD_mixedlayer_data', 'mod_kmld1', 'mod_kmld2', 'mod_kmld3', 'mod_zmld1', ...
  'mod_zmld2', 'mod_zmld3', 'mod_pyc', ...
  'woa13_kmld1', 'woa13_kmld2', 'woa13_kmld3', 'woa13_zmld1', ...
  'woa13_zmld2', 'woa13_zmld3', 'woa13_pyc', '-v7.3');

% ======================================================================
% ======================================================================
% ======================================================================

load ukesm1_PD_mixedlayer_data

% ======================================================================

seas = [3 4 5; 6 7 8; 9 10 11; 12 1 2];

for s = 1:1:5
  for v = 1:1:4
    % model
    if v == 1, t0 = mod_zmld1; elseif v == 2, t0 = mod_zmld2;
    elseif v == 3, t0 = mod_zmld3; elseif v == 4, t0 = mod_pyc; end
    if s < 5, t1 = t0(:,:,seas(s,:),:); else, t1 = t0; end
    t2 = mean(t1, 3);
    seas_mod_phys(:,:,s,v,:) = t2;
    % obs
    if v == 1, t0 = woa13_zmld1; elseif v == 2, t0 = woa13_zmld2;
    elseif v == 3, t0 = woa13_zmld3; elseif v == 4, t0 = woa13_pyc; end
    if s < 5, t1 = t0(:,:,seas(s,:)); else, t1 = t0; end
    t2 = mean(t1, 3);
    seas_woa_phys(:,:,s,v) = t2;
  end
end

% ======================================================================

load nemo_mask100e
load nemo_area100e

nxx = -179.5:1:179.5;  nyy = -89.5:1:89.5;
[lon100, lat100] = meshgrid(nxx, nyy);
nxx = -180:1:180;  nyy = -90:1:90;
[lon100b, lat100b] = meshgrid(nxx, nyy);

load woa_mask
clear t1; t1(:,1:180) = woa_mask(:,181:360); t1(:,181:360) = woa_mask(:,1:180);
woa_mask = t1;

load woa_basin

q1 = grid_area([-90:1:90], [-180:1:180]);
area100 = q1(1:180,1:360);

% ======================================================================

for v = 1:1:4
  for s = 1:1:5
    t1 = seas_woa_phys(:,:,s,v);
    t2 = t1 .* area100;
    t3 = isfinite(t2);
    t4 = sum(t2(t3)); t5 = sum(area100(t3));
    mean_seas_woa_phys(s,v) = t4 / t5;
    for e = 1:1:tsize
      t1 = seas_mod_phys(:,:,s,v,e);
      t2 = t1 .* area100e;
      t3 = isfinite(t2);
      t4 = sum(t2(t3)); t5 = sum(area100e(t3));
      mean_seas_mod_phys(s,v,e) = t4 / t5;
    end
  end
end

% ======================================================================

% domain
lon_rng = [-180 180]; lat_rng = [-90 90];
gpos = [-0.075 -0.025 0.135 0.1; -0.05 -0.025 0.135 0.1; -0.075 -0.05 0.135 0.1; -0.05 -0.05 0.135 0.1];
gpos = [-0.075 -0.025 0.135 0.1; -0.075 0.05 0.135 0.1; -0.05 -0.025 0.135 0.1; -0.05 0.05 0.135 0.1];
% gpos = gpos * 0;

t1 = woa_basin2(:,:,1);
t2 = woa_basin2(:,:,6);
t1(isfinite(t2)) = NaN;
arc_mask = t1;

load coast;

cropval = log10(5);

for v = 1:1:3, var = v;
  tstr = char('Observed, JJA', 'Simulated, JJA', 'Observed, DJF', 'Simulated, DJF');
  for f = 1:1:4, tlen(f) = lenstr(tstr(f,:)); end
  figure(1); clf; turbo(60);
  subplot(2,2,1);
  ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
  t0 = seas_woa_phys(:,:,2,var); t1 = log10(t0); t1(t1 < cropval) = NaN;
  pcolorm(lat100, lon100, t1);
  plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
  caxis ([1 3]);
  pos = get(gca, 'Position'); pos = pos + gpos(1,:); set(gca, 'Position', pos);
  tp = title (tstr(1,1:tlen(1))); set(tp, 'Position', [0 1.6 0]);
  subplot(2,2,3);
  ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
  t0 = mean(seas_mod_phys(:,:,2,var,:), 5); t1 = log10(t0); t1(t1 < cropval) = NaN;
  pcolorm(yy100e, xx100e, t1);
  plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
  caxis ([1 3]);
  pos = get(gca, 'Position'); pos = pos + gpos(2,:); set(gca, 'Position', pos);
  tp = title (tstr(2,1:tlen(2))); set(tp, 'Position', [0 1.6 0]);
  %   cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos(1) = cbpos(1) + 0.025; cbpos(2) = cbpos(2) - 0.05; cbpos(3) = cbpos(3) - 0.05;
  %     set(cb, 'Position', cbpos); set(gca, 'Position', pos);
  subplot(2,2,2);
  ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
  t0 = seas_woa_phys(:,:,4,var); t1 = log10(t0); t1(t1 < cropval) = NaN;
  pcolorm(lat100, lon100, t1);
  plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
  caxis ([1 3]);
  pos = get(gca, 'Position'); pos = pos + gpos(3,:); set(gca, 'Position', pos);
  tp = title (tstr(3,1:tlen(3))); set(tp, 'Position', [0 1.6 0]);
  subplot(2,2,4);
  ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
  t0 = mean(seas_mod_phys(:,:,4,var,:), 5); t1 = log10(t0); t1(t1 < cropval) = NaN;
  pcolorm(yy100e, xx100e, t1);
  plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
  caxis ([1 3]);
  pos = get(gca, 'Position'); pos = pos + gpos(4,:); set(gca, 'Position', pos);
  tp = title (tstr(4,1:tlen(4))); set(tp, 'Position', [0 1.6 0]);
  cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos(1) = cbpos(1) + 0.025; cbpos(2) = cbpos(2) - 0.075; cbpos(3) = cbpos(3) - 0.05;
  set(cb, 'Position', cbpos); set(gca, 'Position', pos);
  xt = [10 20 50 100 200 500 1000]; lxt = log10(xt);
  set(cb, 'XTick', lxt, 'XTickLabel', xt);
  def_figure; oname = sprintf('FIGS/ukesm1_PD_obs_mod_mix_%s.png', numprint(v,2)); exportgraphics(gcf, oname, 'Resolution', 600);
end

% do delta SST
for s = 1:1:4
  for f = 1:1:4
    regrid_clim_obs(:,:,s,f) = scatregrid(lat100, lon100, seas_woa_phys(:,:,s,f), yy100e, xx100e) + nemo_mask100e(:,:,1);
  end
end

gpos = [-0.075 0 0.135 0.1; -0.05 0 0.135 0.1];
lon_rng = [-180 180]; lat_rng = [-90 90];

for v = 1:1:4
tstr = char('Difference, JJA', 'Difference, DJF');
for f = 1:1:2, tlen(f) = lenstr(tstr(f,:)); end
figure(1); clf; therm6(31);
subplot(1,2,1);
m1 = mean(seas_mod_phys(:,:,2,v,:), 5);
ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
t0 = m1 - regrid_clim_obs(:,:,2,v); t1 = t0; 
[t2, x2, xt, xtl] = logval(t1, 3, 2);
pcolorm(yy100e, xx100e, t2);
plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
caxis ([xt(1) xt(end)]);
pos = get(gca, 'Position'); pos = pos + gpos(1,:); set(gca, 'Position', pos);
tp = title (tstr(1,1:tlen(1))); set(tp, 'Position', [0 1.6 0]);
subplot(1,2,2);
m1 = mean(seas_mod_phys(:,:,4,v,:), 5);
ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
t0 = m1 - regrid_clim_obs(:,:,4,v); t1 = t0;
[t2, x2, xt, xtl] = logval(t1, 3, 2);
pcolorm(yy100e, xx100e, t2);
plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
caxis ([xt(1) xt(end)]);
pos = get(gca, 'Position'); pos = pos + gpos(2,:); set(gca, 'Position', pos);
tp = title (tstr(2,1:tlen(2))); set(tp, 'Position', [0 1.6 0]);
cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos(1) = cbpos(1) + 0.025; cbpos(2) = cbpos(2) - 0.01; cbpos(3) = cbpos(3) - 0.05;
set(cb, 'XTick', xt, 'XTickLabel', xtl);
xtl = get(cb,'XTickLabel'); xtl(3,:) = char('  0  '); set(cb, 'XTickLabel', xtl);
set(cb, 'Position', cbpos); set(gca, 'Position', pos);
def_figure; oname = sprintf('FIGS/ukesm1_PD_obs_mod_mix_delta_%s.png', numprint(v,2)); exportgraphics(gcf, oname, 'Resolution', 600);
end

% domain
lon_rng = [-180 180]; lat_rng = [-90 90];
gpos = [-0.075 -0.025 0.135 0.1; -0.05 -0.025 0.135 0.1; -0.075 -0.05 0.135 0.1; -0.05 -0.05 0.135 0.1];
gpos = [-0.075 -0.025 0.135 0.1; -0.05 -0.025 0.135 0.1; -0.075 0.05 0.135 0.1; -0.05 0.05 0.135 0.1];
% gpos = gpos * 0;

t1 = woa_basin2(:,:,1);
t2 = woa_basin2(:,:,6);
t1(isfinite(t2)) = NaN;
arc_mask = t1;

load coast;

cropval = log10(5);

for v = 1:1:3, var = v;
  tstr = char('Mean, JJA', 'Mean, DJF', 'St. dev., JJA', 'St. dev., DJF');
  for f = 1:1:4, tlen(f) = lenstr(tstr(f,:)); end
  figure(1); clf; turbo(60);
  subplot(2,2,1);
  ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
  t0 = mean(seas_mod_phys(:,:,2,var,:), 5); t1 = log10(t0); t1(t1 < cropval) = NaN;
  pcolorm(yy100e, xx100e, t1);
  plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
  caxis ([1 3]);
  pos = get(gca, 'Position'); pos = pos + gpos(1,:); set(gca, 'Position', pos);
  tp = title (tstr(1,1:tlen(1))); set(tp, 'Position', [0 1.6 0]);
  %   cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos(1) = cbpos(1) + 0.025; cbpos(2) = cbpos(2) - 0.05; cbpos(3) = cbpos(3) - 0.05;
  %     set(cb, 'Position', cbpos); set(gca, 'Position', pos);
  subplot(2,2,3);
  ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
  t0 = mean(seas_mod_phys(:,:,4,var,:), 5); t1 = log10(t0); t1(t1 < cropval) = NaN;
  pcolorm(yy100e, xx100e, t1);
  plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
  caxis ([1 3]);
  pos = get(gca, 'Position'); pos = pos + gpos(3,:); set(gca, 'Position', pos);
  tp = title (tstr(2,1:tlen(2))); set(tp, 'Position', [0 1.6 0]);
  cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos(1) = cbpos(1) + 0.025; cbpos(2) = cbpos(2) - 0.075; cbpos(3) = cbpos(3) - 0.05;
  set(cb, 'Position', cbpos); set(gca, 'Position', pos);
  xt = [10 20 50 100 200 500 1000]; lxt = log10(xt);
  set(cb, 'XTick', lxt, 'XTickLabel', xt);
  subplot(2,2,2);
  ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
  t0 = std(seas_mod_phys(:,:,2,var,:), [], 5); t1 = log10(t0); % t1(t1 < cropval) = NaN;
  pcolorm(yy100e, xx100e, t1);
  plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
  caxis ([0 3]);
  pos = get(gca, 'Position'); pos = pos + gpos(2,:); set(gca, 'Position', pos);
  tp = title (tstr(3,1:tlen(3))); set(tp, 'Position', [0 1.6 0]);
  %   cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos(1) = cbpos(1) + 0.025; cbpos(2) = cbpos(2) - 0.05; cbpos(3) = cbpos(3) - 0.05;
  %     set(cb, 'Position', cbpos); set(gca, 'Position', pos);
  subplot(2,2,4);
  ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
  t0 = std(seas_mod_phys(:,:,4,var,:), [], 5); t1 = log10(t0); % t1(t1 < cropval) = NaN;
  pcolorm(yy100e, xx100e, t1);
  plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
  caxis ([0 3]);
  pos = get(gca, 'Position'); pos = pos + gpos(4,:); set(gca, 'Position', pos);
  tp = title (tstr(4,1:tlen(4))); set(tp, 'Position', [0 1.6 0]);
  cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos(1) = cbpos(1) + 0.025; cbpos(2) = cbpos(2) - 0.075; cbpos(3) = cbpos(3) - 0.05;
  set(cb, 'Position', cbpos); set(gca, 'Position', pos);
  xt = [1 2 5 10 20 50 100 200 500 1000]; lxt = log10(xt);
  set(cb, 'XTick', lxt, 'XTickLabel', xt);
  def_figure; oname = sprintf('FIGS/ukesm1_PD_obs_mod_mix_std_%s.png', numprint(v,2)); exportgraphics(gcf, oname, 'Resolution', 600);
end

for v = 1:1:3, var = v;
  tstr = char('Mean, JJA', 'Mean, DJF', 'CoV, JJA', 'CoV, DJF');
  for f = 1:1:4, tlen(f) = lenstr(tstr(f,:)); end
  figure(1); clf; turbo(60);
  subplot(2,2,1);
  ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
  t0 = mean(seas_mod_phys(:,:,2,var,:), 5); t1 = log10(t0); t1(t1 < cropval) = NaN;
  pcolorm(yy100e, xx100e, t1);
  plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
  caxis ([1 3]);
  pos = get(gca, 'Position'); pos = pos + gpos(1,:); set(gca, 'Position', pos);
  tp = title (tstr(1,1:tlen(1))); set(tp, 'Position', [0 1.6 0]);
  %   cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos(1) = cbpos(1) + 0.025; cbpos(2) = cbpos(2) - 0.05; cbpos(3) = cbpos(3) - 0.05;
  %     set(cb, 'Position', cbpos); set(gca, 'Position', pos);
  subplot(2,2,3);
  ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
  t0 = mean(seas_mod_phys(:,:,4,var,:), 5); t1 = log10(t0); t1(t1 < cropval) = NaN;
  pcolorm(yy100e, xx100e, t1);
  plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
  caxis ([1 3]);
  pos = get(gca, 'Position'); pos = pos + gpos(3,:); set(gca, 'Position', pos);
  tp = title (tstr(2,1:tlen(2))); set(tp, 'Position', [0 1.6 0]);
  cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos(1) = cbpos(1) + 0.025; cbpos(2) = cbpos(2) - 0.075; cbpos(3) = cbpos(3) - 0.05;
  set(cb, 'Position', cbpos); set(gca, 'Position', pos);
  xt = [10 20 50 100 200 500 1000]; lxt = log10(xt);
  set(cb, 'XTick', lxt, 'XTickLabel', xt);
  subplot(2,2,2);
  ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
  t0a = mean(seas_mod_phys(:,:,2,var,:), 5); t0b = std(seas_mod_phys(:,:,2,var,:), [], 5);
  t1 = t0b ./ t0a; % t1 = log10(t0); % t1(t1 < cropval) = NaN;
  pcolorm(yy100e, xx100e, t1);
  plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
  caxis ([0 0.4]);
  pos = get(gca, 'Position'); pos = pos + gpos(2,:); set(gca, 'Position', pos);
  tp = title (tstr(3,1:tlen(3))); set(tp, 'Position', [0 1.6 0]);
  %   cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos(1) = cbpos(1) + 0.025; cbpos(2) = cbpos(2) - 0.05; cbpos(3) = cbpos(3) - 0.05;
  %     set(cb, 'Position', cbpos); set(gca, 'Position', pos);
  subplot(2,2,4);
  ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
  t0a = mean(seas_mod_phys(:,:,4,var,:), 5); t0b = std(seas_mod_phys(:,:,4,var,:), [], 5);
  t1 = t0b ./ t0a; % t1 = log10(t0); % t1(t1 < cropval) = NaN;
  pcolorm(yy100e, xx100e, t1);
  plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
  caxis ([0 0.4]);
  pos = get(gca, 'Position'); pos = pos + gpos(4,:); set(gca, 'Position', pos);
  tp = title (tstr(4,1:tlen(4))); set(tp, 'Position', [0 1.6 0]);
  cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos(1) = cbpos(1) + 0.025; cbpos(2) = cbpos(2) - 0.075; cbpos(3) = cbpos(3) - 0.05;
  set(cb, 'Position', cbpos); set(gca, 'Position', pos);
  % xt = [1 2 5 10 20 50 100 200 500 1000]; lxt = log10(xt);
  % set(cb, 'XTick', lxt, 'XTickLabel', xt);
  def_figure; oname = sprintf('FIGS/ukesm1_PD_obs_mod_mix_cov_%s.png', numprint(v,2)); exportgraphics(gcf, oname, 'Resolution', 600);
end

% -----

lnbin = log10(10):0.02:(log10(2000) + 0.02); nbin = 10.^lnbin;

clear seas_*hist;
for v = 1:1:4
  for s = 1:1:5
    t1 = seas_woa_phys(:,:,s,v);
    t2 = log10(t1);
    [q1, q2, q3] = histwc_axy(t2(:), area100(:)*1e6, lnbin);
    seas_woa_hist(:,s,v) = q1;
    seas_woa_ihist(:,s,v) = q2;
    t1 = mean(seas_mod_phys(:,:,s,v,:), 5);
    t2 = log10(t1);
    [q1, q2, q3] = histwc_axy(t2(:), area100e(:), lnbin);
    seas_mod_hist(:,s,v) = q1;
    seas_mod_ihist(:,s,v) = q2;
    fprintf('.');
  end
  fprintf('\n');
end

for v = 1:1:4
  col = [1 0.4 0.4; 0.4 1 0.4; 0.4 0.4 1; 1 0.4 1; 0 0 0];
  xt = [10 20 50 100 200 500 1000 2000]; lxt = log10(xt);
  
  figure(1); clf
  subplot (2,1,1); hold on;
  for s = 1:1:5, t1 = seas_woa_hist(:,s,v) / 1e12;
    h = plot (lnbin, t1, '-'); set (h, 'Color', col(s,:));
    [t2, t3] = max(t1); h = plot (lnbin(t3), t1(t3), 'o'); set (h, 'Color', col(s,:));
    set (gca, 'XTick', lxt, 'XTickLabel', xt');
  end
  xaxis ([lnbin(1) lnbin(end)]);
  if v < 4, yaxis ([0 20]); else, yaxis ([0 35]); end
  xlabel ('Mixed layer depth [m]'); ylabel ('Area [10^{12} m^{2}]');
  title ('Observations');
  subplot (2,1,2); hold on;
  for s = 1:1:5, t1 = seas_mod_hist(:,s,v) / 1e12;
    h = plot (lnbin, t1, '-'); set (h, 'Color', col(s,:)); end
  for s = 1:1:5, t1 = seas_mod_hist(:,s,v) / 1e12;
    h = plot (lnbin, t1, '-'); set (h, 'Color', col(s,:));
    [t2, t3] = max(t1); h = plot (lnbin(t3), t1(t3), 'o'); set (h, 'Color', col(s,:));
    set (gca, 'XTick', lxt, 'XTickLabel', xt');
  end
  xaxis ([lnbin(1) lnbin(end)]);
  if v < 4, yaxis ([0 20]); else, yaxis ([0 35]); end
  xlabel ('Mixed layer depth [m]'); ylabel ('Area [10^{12} m^{2}]');
  title ('UKESM1');
  legend ('MAM', 'JJA', 'SON', 'DJF', 'Ann', 'Location', 'NorthEast');
  def_figure; oname = sprintf('FIGS/ukesm1_PD_obs_mod_mix_HIST_%s.png', numprint(v,2)); exportgraphics(gcf, oname, 'Resolution', 600);
end

s_alt = [3 4 1 2];

clear seas_*hist;
for v = 1:1:4
  for s = 1:1:5
    clear t1;
    if s == 5
      t1 = seas_woa_phys(:,:,s,v);
    else,
      t1a = seas_woa_phys(:,:,s,v);
      t1b = seas_woa_phys(:,:,s_alt(s),v);
      t1(91:180,:) = t1a(91:180,:);
      t1(1:90,:) = t1b(1:90,:);
    end
    t2 = log10(t1);
    [q1, q2, q3] = histwc_axy(t2(:), area100(:)*1e6, lnbin);
    seas_woa_hist(:,s,v) = q1;
    seas_woa_ihist(:,s,v) = q2;
    clear t1;
    if s == 5
      t1 = mean(seas_mod_phys(:,:,s,v,:), 5);
    else
      t1a = mean(seas_mod_phys(:,:,s,v,:), 5);
      t1b = mean(seas_mod_phys(:,:,s_alt(s),v,:), 5);
      t1(187:332,:) = t1a(187:332,:);
      t1(1:186,:) = t1b(1:186,:);
    end
    t2 = log10(t1);
    [q1, q2, q3] = histwc_axy(t2(:), area100e(:), lnbin);
    seas_mod_hist(:,s,v) = q1;
    seas_mod_ihist(:,s,v) = q2;
    fprintf('.');
  end
  fprintf('\n');
end

for v = 1:1:4
  col = [1 0.4 0.4; 0.4 1 0.4; 0.4 0.4 1; 1 0.4 1; 0 0 0];
  xt = [10 20 50 100 200 500 1000 2000]; lxt = log10(xt);
  
  figure(1); clf
  subplot (2,1,1); hold on;
  for s = 1:1:5, t1 = seas_woa_hist(:,s,v) / 1e12;
    h = plot (lnbin, t1, '-'); set (h, 'Color', col(s,:));
    t2 = cumsum(t1); t3 = t2 / t2(end); t4 = min(find(t3 > 0.5));
    % [t2, t3] = max(t1); h = plot (lnbin(t3), t1(t3), 'o'); set (h, 'Color', col(s,:));
    h = plot (lnbin(t4), t1(t4), 'o'); set (h, 'Color', col(s,:));
    set (gca, 'XTick', lxt, 'XTickLabel', xt');
  end
  xaxis ([lnbin(1) lnbin(end)]);
  if v < 4, yaxis ([0 20]); else, yaxis ([0 35]); end
  % xlabel ('Mixed layer depth [m]'); 
  ylabel ('Area [10^{12} m^{2}]');
  title ('Observations');
  subplot (2,1,2); hold on;
  for s = 1:1:5, t1 = seas_mod_hist(:,s,v) / 1e12;
    h = plot (lnbin, t1, '-'); set (h, 'Color', col(s,:)); end
  for s = 1:1:5, t1 = seas_mod_hist(:,s,v) / 1e12;
    h = plot (lnbin, t1, '-'); set (h, 'Color', col(s,:));
    t2 = cumsum(t1); t3 = t2 / t2(end); t4 = min(find(t3 > 0.5));
    % [t2, t3] = max(t1); h = plot (lnbin(t3), t1(t3), 'o'); set (h, 'Color', col(s,:));
    h = plot (lnbin(t4), t1(t4), 'o'); set (h, 'Color', col(s,:));
    set (gca, 'XTick', lxt, 'XTickLabel', xt');
  end
  xaxis ([lnbin(1) lnbin(end)]);
  if v < 4, yaxis ([0 20]); else, yaxis ([0 35]); end
  xlabel ('Mixed layer depth [m]'); ylabel ('Area [10^{12} m^{2}]');
  title ('UKESM1');
  % legend ('MAM', 'JJA', 'SON', 'DJF', 'Ann', 'Location', 'NorthEast');
  lg = legend ('Spr', 'Sum', 'Aut', 'Win', 'Ann', 'Location', 'NorthEast');
  lgpos = get(lg, 'Position'); lgpos(1) = lgpos(1) - 0.05; lgpos(2) = lgpos(2) -0.02; lgpos(3) = lgpos(3) + 0.05; set(lg, 'Position', lgpos);
  def_figure; oname = sprintf('FIGS/ukesm1_PD_obs_mod_mix_HIST_SEAS_%s.png', numprint(v,2)); exportgraphics(gcf, oname, 'Resolution', 600);
end

gpos = [-0.075 0 0.135 0.1; -0.05 0 0.135 0.1];

v = 5; var = 4;
tstr = char('Observed', 'Simulated');
for f = 1:1:2, tlen(f) = lenstr(tstr(f,:)); end
figure(1); clf; turbo(60);
subplot(1,2,1);
ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
t0 = seas_woa_phys(:,:,5,var); t1 = t0; % t1 = log10(t0); t1(t1 < 1) = NaN;
pcolorm(lat100, lon100, t1);
plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
caxis ([-0.001 1000.001]);
pos = get(gca, 'Position'); pos = pos + gpos(1,:); set(gca, 'Position', pos);
tp = title (tstr(1,1:tlen(1))); set(tp, 'Position', [0 1.6 0]);
subplot(1,2,2);
ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
t0 = mean(seas_mod_phys(:,:,5,var,:), 5); t1 = t0; % t1 = log10(t0); t1(t1 < 1) = NaN;
pcolorm(yy100e, xx100e, t1);
plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
caxis ([-0.001 1000.001]);
pos = get(gca, 'Position'); pos = pos + gpos(2,:); set(gca, 'Position', pos);
tp = title (tstr(2,1:tlen(2))); set(tp, 'Position', [0 1.6 0]);
cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos(1) = cbpos(1) + 0.025; cbpos(2) = cbpos(2) - 0.01; cbpos(3) = cbpos(3) - 0.05;
set(cb, 'Position', cbpos); set(gca, 'Position', pos);
xt = [10 20 50 100 200 500 1000]; lxt = log10(xt);
set(cb, 'XTick', 0:200:1e3);
def_figure; oname = sprintf('FIGS/ukesm1_PD_obs_mod_mix_%s.png', numprint(v,2)); exportgraphics(gcf, oname, 'Resolution', 600);
