clear; startup;

% =========================================================================
% =========================================================================
%
% Matlab plotting scripts for the analysis of UKESM1 prepared for:
%
%   Yool, A., Palmiéri, J., Jones, C. G., de Mora, L., Kuhlbrodt, T., 
%   Popova, E. E., Nurser, A. J. G., Hirschi, J., Blaker, A. T., 
%	Coward, A. C., Blockley, E. W., and Sellar, A. A.: Evaluating the 
% 	physical and biogeochemical state of the global ocean component of 
%	UKESM1 in CMIP6 Historical simulations, Geosci. Model Dev. Discuss. 
%	[preprint], https://doi.org/10.5194/gmd-2020-333, in review, 2020.
%
%   Accepted for publication: 20 April 2021
%
% Note: scripts provided to document analysis, and may require editing to
% fully replicate plots in manuscript
%
% =========================================================================
% =========================================================================

% =========================================================================
%
% This script plots Figures 1 and 2, and Supplementary Figure S5
%
% =========================================================================

% =========================================================================

load ukesm1_PD_ensemble

% =========================================================================

% load up background files
load nemo_area100e
load nemo_mask100e
fname = sprintf('%s/mean_nemo_%so_1y_20001201-20101201_grid-T.nc', dname(1).name, rname(1).name);
t1 = ncread(fname, 'deptht_bounds');
nemo_dep = double(t1(1,:)); nemo_dep(end+1) = double(t1(2,end));
t1 = ncread(fname, 'deptht');
nemo_mid = double(t1');

% total grid area
t1 = area100e + nemo_mask100e(:,:,1);
t2 = isfinite(t1);
tot_area100e = sum(t1(t2)); % m2

% extra mask to get rid of Mediterranean, Black and Caspian Seas
t1 = nemo_mask100e(:,:,1) + 1;
t1(isnan(t1)) = 0;
t1(235:249,283:344) = t1(235:249,283:344) + 2;
t1(249:262,291:344) = t1(249:262,291:344) + 4;
t1(262:276,330:344) = t1(262:276,330:344) + 8;
t2 = t1;
t2(t1 < 1) = NaN; t2(t1 > 1) = NaN;
med_mask100e = t2;

nxx = -179.5:1:179.5;  nyy = -89.5:1:89.5;
[lon100, lat100] = meshgrid(nxx, nyy);
nxx = -180:1:180;  nyy = -90:1:90;
[lon100b, lat100b] = meshgrid(nxx, nyy);

load woa_mask
clear t1; t1(:,1:180) = woa_mask(:,181:360); t1(:,181:360) = woa_mask(:,1:180);
woa_mask = t1;

load woa_basin

helvetica;
pngopt = 0;

% ----------------------------------------------------------------------

% load up annual 3D fields
for e = 1:1:tsize
  pnom = sprintf('%s/mean_nemo_%so_1y_20001201-20101201_grid-T.nc', dname(e).name, rname(e).name);
  fprintf('- Reading %s\n', pnom);
  t1 = ncread(pnom, 'thetao'); t2 = permute(t1, [2 1 3]); mod_phys3(:,:,:,1,e) = t2;
  t1 = ncread(pnom, 'so'); t2 = permute(t1, [2 1 3]); mod_phys3(:,:,:,2,e) = t2;
end

% ----------------------------------------------------------------------

% fields of TMP, SAL, ice
surf1 = [1 1 1 1]; surf2 = [362 332 1 1];

for e = 1:1:tsize
for m = 12:-1:1 % loop reversed to reach full size at first iteration
  fprintf('- Ensemble %d, Month %d\n', e, m);
  pnom = sprintf('%s/clim_nemo_%so_1m_2000%s-2009%s_grid-T.nc', dname(e).name, rname(e).name, numprint(m, 2), numprint(m, 2));
  % TMP
  t1 = ncread(pnom, 'thetao', surf1, surf2); t2 = permute(t1, [2 1]); mod_phys(:,:,m,1,e) = t2;
  % SAL
  t1 = ncread(pnom, 'so', surf1, surf2); t2 = permute(t1, [2 1]); mod_phys(:,:,m,2,e) = t2;
  % ice
  t1 = ncread(pnom, 'soicecov', surf1(1:3), surf2(1:3)); t2 = permute(t1, [2 1]); mod_phys(:,:,m,3,e) = t2;
end
end

% =========================================================================

% create seasonal averages

seas = [3 4 5; 6 7 8; 9 10 11; 12 1 2];

for e = 1:1:tsize
for s = 1:1:4
  for v = 1:1:3
    t1 = mod_phys(:,:,seas(s,:),v,e);
    t2 = mean(t1, 3);
    seas_mod_phys(:,:,s,v,e) = t2;
  end
end
end

save ('ukesm1_PD_phys_output', 'mod_phys', 'mod_phys3', 'seas_mod_phys', '-v7.3');

% =========================================================================

load ('ukesm1_PD_phys_output');

% SST and sea-ice
t1 = ncread('HadISST_sst.nc', 'sst');
t2 = flipdim(permute(t1, [2 1 3]), 1);
% ends April 2018; i.e. need to lose 8 years and 4 months from end
t3 = t2(:,:,1:(end - ((8 * 12) + 4)));
t4 = t3(:,:,1561:end);
t4(t4 == -1000) = NaN;
phys_obs(:,:,:,1) = t4;
t1 = ncread('HadISST_ice.nc', 'sic');
t2 = flipdim(permute(t1, [2 1 3]), 1);
% ends April 2018; i.e. need to lose 8 years and 4 months from end
t3 = t2(:,:,1:(end - ((8 * 12) + 4)));
t4 = t3(:,:,1561:end);
t4(t4 == -1000) = NaN;
phys_obs(:,:,:,3) = t4;

for m = 1:1:12
  top = 108 + m; bot = m;
  t1 = nanmean(phys_obs(:,:,bot:12:top,:), 3);
  clim_obs(:,:,m,:) = t1;
end

load woa13_95A4_TS
clim_obs(:,:,:,2) = squeeze(woa13_sal(:,:,1,:));

seas = [3 4 5; 6 7 8; 9 10 11; 12 1 2];

for s = 1:1:4
  for v = 1:1:3
    t1 = clim_obs(:,:,seas(s,:),v);
    t2 = mean(t1, 3);
    seas_clim_obs(:,:,s,v) = t2;
  end
end

% make a HadISST grid
[xx, yy] = meshgrid([-180:1:179], [-90:1:89]);
[xxb, yyb] = meshgrid([-180:1:180], [-90:1:90]);

lonlim = [-180 180]; latlim = [50 90];

t1 = ncread('HadISST_ice.nc', 'sic');

lon_rng = [-180 180]; lat_rng = [-90 90];
% gpos = [-0.05 -0.025 0.125 0.10; -0.025 -0.025 0.125 0.10; -0.05 -0.05 0.125 0.10; -0.025 -0.05 0.125 0.10];
% gpos = [-0.075 -0.025 0.135 0.1; -0.05 -0.025 0.135 0.1; -0.075 -0.05 0.135 0.1; -0.05 -0.05 0.135 0.1];
gpos = [-0.075 -0.025 0.135 0.1; -0.075 0.05 0.135 0.1; -0.05 -0.025 0.135 0.1; -0.05 0.05 0.135 0.1];
% gpos = gpos * 0;

load woa_basin

t1 = woa_basin2(:,:,1);
t2 = woa_basin2(:,:,6);
t1(isfinite(t2)) = NaN;
arc_mask = t1;

load coast;
load nemo_area100e;

tstr = char('Observed, JJA', 'Simulated, JJA', 'Observed, DJF', 'Simulated, DJF');
for f = 1:1:4, tlen(f) = lenstr(tstr(f,:)); end
figure(1); clf; turbo(64);
subplot(2,2,1);
ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
pcolorm(lat100, lon100, seas_clim_obs(:,:,2,1));
plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
caxis ([-2 30]);
pos = get(gca, 'Position'); pos = pos + gpos(1,:); set(gca, 'Position', pos);
tp = title (tstr(1,1:tlen(1))); set(tp, 'Position', [0 1.6 0]);
subplot(2,2,3);
t1 = mean(seas_mod_phys(:,:,2,1,:), 5);
ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
pcolorm(yy100e, xx100e, t1);
plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
caxis ([-2 30]);
pos = get(gca, 'Position'); pos = pos + gpos(2,:); set(gca, 'Position', pos);
tp = title (tstr(2,1:tlen(2))); set(tp, 'Position', [0 1.6 0]);
%   cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos(1) = cbpos(1) + 0.025; cbpos(2) = cbpos(2) - 0.05; cbpos(3) = cbpos(3) - 0.05;
%     set(cb, 'Position', cbpos); set(gca, 'Position', pos);
subplot(2,2,2);
ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
pcolorm(lat100, lon100, seas_clim_obs(:,:,4,1));
plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
caxis ([-2 30]);
pos = get(gca, 'Position'); pos = pos + gpos(3,:); set(gca, 'Position', pos);
tp = title (tstr(3,1:tlen(3))); set(tp, 'Position', [0 1.6 0]);
subplot(2,2,4);
t1 = mean(seas_mod_phys(:,:,4,1,:), 5);
ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
pcolorm(yy100e, xx100e, t1);
plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
caxis ([-2 30]);
pos = get(gca, 'Position'); pos = pos + gpos(4,:); set(gca, 'Position', pos);
tp = title (tstr(4,1:tlen(4))); set(tp, 'Position', [0 1.6 0]);
cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos(1) = cbpos(1) + 0.025; cbpos(2) = cbpos(2) - 0.075; cbpos(3) = cbpos(3) - 0.05;
set(cb, 'Position', cbpos); set(gca, 'Position', pos);
def_figure; oname = sprintf('FIGS/ukesm1_PD_hadisst_01.png'); exportgraphics(gcf, oname, 'Resolution', 600);  

tstr = char('Observed, JJA', 'Simulated, JJA', 'Observed, DJF', 'Simulated, DJF');
for f = 1:1:4, tlen(f) = lenstr(tstr(f,:)); end
figure(1); clf; turbo(60);
subplot(2,2,1);
ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
pcolorm(lat100, lon100, seas_clim_obs(:,:,2,2));
plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
caxis ([30 40]);
pos = get(gca, 'Position'); pos = pos + gpos(1,:); set(gca, 'Position', pos);
tp = title (tstr(1,1:tlen(1))); set(tp, 'Position', [0 1.6 0]);
subplot(2,2,3);
t1 = mean(seas_mod_phys(:,:,2,2,:), 5);
ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
pcolorm(yy100e, xx100e, t1);
plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
caxis ([30 40]);
pos = get(gca, 'Position'); pos = pos + gpos(2,:); set(gca, 'Position', pos);
tp = title (tstr(2,1:tlen(2))); set(tp, 'Position', [0 1.6 0]);
%   cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos(1) = cbpos(1) + 0.025; cbpos(2) = cbpos(2) - 0.05; cbpos(3) = cbpos(3) - 0.05;
%     set(cb, 'Position', cbpos); set(gca, 'Position', pos);
subplot(2,2,2);
ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
pcolorm(lat100, lon100, seas_clim_obs(:,:,4,2));
plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
caxis ([30 40]);
pos = get(gca, 'Position'); pos = pos + gpos(3,:); set(gca, 'Position', pos);
tp = title (tstr(3,1:tlen(3))); set(tp, 'Position', [0 1.6 0]);
subplot(2,2,4);
t1 = mean(seas_mod_phys(:,:,4,2,:), 5);
ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
pcolorm(yy100e, xx100e, t1);
plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
caxis ([30 40]);
pos = get(gca, 'Position'); pos = pos + gpos(4,:); set(gca, 'Position', pos);
tp = title (tstr(4,1:tlen(4))); set(tp, 'Position', [0 1.6 0]);
cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos(1) = cbpos(1) + 0.025; cbpos(2) = cbpos(2) - 0.075; cbpos(3) = cbpos(3) - 0.05;
set(cb, 'Position', cbpos); set(gca, 'Position', pos);
def_figure; oname = sprintf('FIGS/ukesm1_PD_hadisst_02.png'); exportgraphics(gcf, oname, 'Resolution', 600);  

% do delta SST, SSS
for s = 1:1:4
  for f = 1:1:3
    regrid_clim_obs(:,:,s,f) = scatregrid(lat100, lon100, seas_clim_obs(:,:,s,f), yy100e, xx100e) + nemo_mask100e(:,:,1);
  end
end

gpos = [-0.075 0 0.135 0.1; -0.05 0 0.135 0.1];
lon_rng = [-180 180]; lat_rng = [-90 90];

v = 5; var = 4;
tstr = char('Difference, JJA', 'Difference, DJF');
for f = 1:1:2, tlen(f) = lenstr(tstr(f,:)); end
figure(1); clf; therm6(31);
subplot(1,2,1);
m1 = mean(seas_mod_phys(:,:,2,1,:), 5);
ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
t0 = m1 - regrid_clim_obs(:,:,2,1); t1 = t0; % t1 = log10(t0); t1(t1 < 1) = NaN;
pcolorm(yy100e, xx100e, t1);
plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
caxis ([-4.001 4.001]);
pos = get(gca, 'Position'); pos = pos + gpos(1,:); set(gca, 'Position', pos);
tp = title (tstr(1,1:tlen(1))); set(tp, 'Position', [0 1.6 0]);
subplot(1,2,2);
m1 = mean(seas_mod_phys(:,:,4,1,:), 5);
ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
t0 = m1 - regrid_clim_obs(:,:,4,1); t1 = t0; % t1 = log10(t0); t1(t1 < 1) = NaN;
pcolorm(yy100e, xx100e, t1);
plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
caxis ([-4.001 4.001]);
pos = get(gca, 'Position'); pos = pos + gpos(2,:); set(gca, 'Position', pos);
tp = title (tstr(2,1:tlen(2))); set(tp, 'Position', [0 1.6 0]);
cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos(1) = cbpos(1) + 0.025; cbpos(2) = cbpos(2) - 0.01; cbpos(3) = cbpos(3) - 0.05;
set(cb, 'Position', cbpos); set(gca, 'Position', pos);
def_figure; oname = sprintf('FIGS/ukesm1_PD_hadisst_03.png'); exportgraphics(gcf, oname, 'Resolution', 600);  

v = 5; var = 4;
tstr = char('Difference, JJA', 'Difference, DJF');
for f = 1:1:2, tlen(f) = lenstr(tstr(f,:)); end
figure(1); clf; therm6(31);
subplot(1,2,1);
m1 = mean(seas_mod_phys(:,:,2,2,:), 5);
ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
t0 = m1 - regrid_clim_obs(:,:,2,2); t1 = t0; % t1 = log10(t0); t1(t1 < 1) = NaN;
pcolorm(yy100e, xx100e, t1);
plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
caxis ([-3.001 3.001]);
pos = get(gca, 'Position'); pos = pos + gpos(1,:); set(gca, 'Position', pos);
tp = title (tstr(1,1:tlen(1))); set(tp, 'Position', [0 1.6 0]);
subplot(1,2,2);
m1 = mean(seas_mod_phys(:,:,4,2,:), 5);
ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
t0 = m1 - regrid_clim_obs(:,:,4,2); t1 = t0; % t1 = log10(t0); t1(t1 < 1) = NaN;
pcolorm(yy100e, xx100e, t1);
plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
caxis ([-3.001 3.001]);
pos = get(gca, 'Position'); pos = pos + gpos(2,:); set(gca, 'Position', pos);
tp = title (tstr(2,1:tlen(2))); set(tp, 'Position', [0 1.6 0]);
cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos(1) = cbpos(1) + 0.025; cbpos(2) = cbpos(2) - 0.01; cbpos(3) = cbpos(3) - 0.05;
set(cb, 'Position', cbpos); set(gca, 'Position', pos);
def_figure; oname = sprintf('FIGS/ukesm1_PD_hadisst_04.png'); exportgraphics(gcf, oname, 'Resolution', 600);  

gpos = [-0.075 -0.025 0.135 0.1; -0.075 0.05 0.135 0.1; -0.05 -0.025 0.135 0.1; -0.05 0.05 0.135 0.1];

nxx = -539.5:1:539.5;  nyy = -89.5:1:89.5;
[lon100b, lat100b] = meshgrid(nxx, nyy);

tstr = char('Observed, March', 'Simulated, March', 'Observed, September', 'Simulated, September');
for f = 1:1:4, tlen(f) = lenstr(tstr(f,:)); end
figure(1); clf; 
pal0 = turbo(51); pal(1:9,1:3) = 1; pal(10:60,:) = pal0; colormap(pal);
% pal = turbo(60); pal(1:9,1:3) = 1; colormap(pal);
lon_rng = [-180 180]; lat_rng = [50 90];
subplot(2,2,1);
ax = axesm('MapProjection', 'stereo', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
t1 = [clim_obs(:,:,3,3) clim_obs(:,:,3,3) clim_obs(:,:,3,3)]; t1(t1 == 0) = NaN;
t1(t1 < 0.15) = NaN;
pcolorm(lat100b, lon100b, t1);
plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
caxis ([0 1]);
pos = get(gca, 'Position'); pos = pos + gpos(1,:); set(gca, 'Position', pos);
tp = title (tstr(1,1:tlen(1))); set(tp, 'Position', [0 0.80 0]);
subplot(2,2,3);
t1 = mean(mod_phys(:,:,3,3,:), 5); t1(t1 == 0) = NaN;
t1(t1 < 0.15) = NaN;
ax = axesm('MapProjection', 'stereo', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
pcolorm(yy100e, xx100e, t1);
plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
caxis ([0 1]);
pos = get(gca, 'Position'); pos = pos + gpos(2,:); set(gca, 'Position', pos);
tp = title (tstr(2,1:tlen(2))); set(tp, 'Position', [0 0.80 0]);
%   cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos(1) = cbpos(1) + 0.025; cbpos(2) = cbpos(2) - 0.05; cbpos(3) = cbpos(3) - 0.05;
%     set(cb, 'Position', cbpos); set(gca, 'Position', pos);
lon_rng = [-180 180]; lat_rng = [-90 -50];
subplot(2,2,2);
ax = axesm('MapProjection', 'stereo', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
t1 = [clim_obs(:,:,9,3) clim_obs(:,:,9,3) clim_obs(:,:,9,3)]; t1(t1 == 0) = NaN;
t1(t1 < 0.15) = NaN;
pcolorm(lat100b, lon100b, t1);
plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
caxis ([0 1]);
pos = get(gca, 'Position'); pos = pos + gpos(3,:); set(gca, 'Position', pos);
tp = title (tstr(3,1:tlen(3))); set(tp, 'Position', [0 0.80 0]);
subplot(2,2,4);
t1 = mean(mod_phys(:,:,9,3,:), 5); t1(t1 == 0) = NaN;
t1(t1 < 0.15) = NaN;
ax = axesm('MapProjection', 'stereo', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
pcolorm(yy100e, xx100e, t1);
plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
caxis ([0 1]);
pos = get(gca, 'Position'); pos = pos + gpos(4,:); set(gca, 'Position', pos);
tp = title (tstr(4,1:tlen(4))); set(tp, 'Position', [0 0.80 0]);
cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos = [(pos(1)+0.1) (cbpos(2)-0.075) (pos(3)-0.2) cbpos(4)]; 
set(cb, 'Position', cbpos); set(gca, 'Position', pos);
def_figure; 
set(gcf, 'PaperPosition', [4.1217 8.5763 12.7340 12.5506]);
def_figure; oname = sprintf('FIGS/ukesm1_PD_hadisst_05.png'); exportgraphics(gcf, oname, 'Resolution', 600);  

tstr = char('Observed, March', 'Simulated, March', 'Observed, September', 'Simulated, September');
for f = 1:1:4, tlen(f) = lenstr(tstr(f,:)); end
figure(1); clf; 
pal0 = turbo(51); pal(1:9,1:3) = 1; pal(10:60,:) = pal0; colormap(pal);
lon_rng = [-180 180]; lat_rng = [50 90];
subplot(2,2,1);
ax = axesm('MapProjection', 'stereo', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
t1 = [clim_obs(:,:,3,3) clim_obs(:,:,3,3) clim_obs(:,:,3,3)];
t1(t1 < 0.15) = NaN;
pcolorm(lat100b, lon100b, t1);
plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
caxis ([0 1]);
pos = get(gca, 'Position'); pos = pos + gpos(1,:); set(gca, 'Position', pos);
tp = title (tstr(1,1:tlen(1))); set(tp, 'Position', [0 0.80 0]);
subplot(2,2,3);
t1 = mean(mod_phys(:,:,3,3,:), 5);
t1(t1 < 0.15) = NaN;
ax = axesm('MapProjection', 'stereo', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
pcolorm(yy100e, xx100e, t1);
plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
caxis ([0 1]);
pos = get(gca, 'Position'); pos = pos + gpos(2,:); set(gca, 'Position', pos);
tp = title (tstr(2,1:tlen(2))); set(tp, 'Position', [0 0.80 0]);
%   cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos(1) = cbpos(1) + 0.025; cbpos(2) = cbpos(2) - 0.05; cbpos(3) = cbpos(3) - 0.05;
%     set(cb, 'Position', cbpos); set(gca, 'Position', pos);
lon_rng = [-180 180]; lat_rng = [50 90];
subplot(2,2,2);
ax = axesm('MapProjection', 'stereo', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
t1 = [clim_obs(:,:,9,3) clim_obs(:,:,9,3) clim_obs(:,:,9,3)];
t1(t1 < 0.15) = NaN;
pcolorm(lat100b, lon100b, t1);
plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
caxis ([0 1]);
pos = get(gca, 'Position'); pos = pos + gpos(3,:); set(gca, 'Position', pos);
tp = title (tstr(3,1:tlen(3))); set(tp, 'Position', [0 0.80 0]);
subplot(2,2,4);
t1 = mean(mod_phys(:,:,9,3,:), 5);
t1(t1 < 0.15) = NaN;
ax = axesm('MapProjection', 'stereo', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
pcolorm(yy100e, xx100e, t1);
plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
caxis ([0 1]);
pos = get(gca, 'Position'); pos = pos + gpos(4,:); set(gca, 'Position', pos);
tp = title (tstr(4,1:tlen(4))); set(tp, 'Position', [0 0.80 0]);
cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos = [(pos(1)+0.1) (cbpos(2)-0.075) (pos(3)-0.2) cbpos(4)]; 
set(cb, 'Position', cbpos); set(gca, 'Position', pos);
def_figure; 
set(gcf, 'PaperPosition', [4.1217 8.5763 12.7340 12.5506]);
def_figure; oname = sprintf('FIGS/ukesm1_PD_hadisst_06.png'); exportgraphics(gcf, oname, 'Resolution', 600);  

tstr = char('Observed, March', 'Simulated, March', 'Observed, September', 'Simulated, September');
for f = 1:1:4, tlen(f) = lenstr(tstr(f,:)); end
figure(1); clf; 
pal0 = turbo(51); pal(1:9,1:3) = 1; pal(10:60,:) = pal0; colormap(pal);
lon_rng = [-180 180]; lat_rng = [-90 -50];
subplot(2,2,1);
ax = axesm('MapProjection', 'stereo', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
t1 = [clim_obs(:,:,3,3) clim_obs(:,:,3,3) clim_obs(:,:,3,3)]; 
t1(t1 < 0.15) = NaN;
pcolorm(lat100b, lon100b, t1);
plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
caxis ([0 1]);
pos = get(gca, 'Position'); pos = pos + gpos(1,:); set(gca, 'Position', pos);
tp = title (tstr(1,1:tlen(1))); set(tp, 'Position', [0 0.80 0]);
subplot(2,2,3);
t1 = mean(mod_phys(:,:,3,3,:), 5);
t1(t1 < 0.15) = NaN;
ax = axesm('MapProjection', 'stereo', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
pcolorm(yy100e, xx100e, t1);
plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
caxis ([0 1]);
pos = get(gca, 'Position'); pos = pos + gpos(2,:); set(gca, 'Position', pos);
tp = title (tstr(2,1:tlen(2))); set(tp, 'Position', [0 0.80 0]);
%   cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos(1) = cbpos(1) + 0.025; cbpos(2) = cbpos(2) - 0.05; cbpos(3) = cbpos(3) - 0.05;
%     set(cb, 'Position', cbpos); set(gca, 'Position', pos);
lon_rng = [-180 180]; % lat_rng = [50 90];
subplot(2,2,2);
ax = axesm('MapProjection', 'stereo', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
t1 = [clim_obs(:,:,9,3) clim_obs(:,:,9,3) clim_obs(:,:,9,3)];
t1(t1 < 0.15) = NaN;
pcolorm(lat100b, lon100b, t1);
plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
caxis ([0 1]);
pos = get(gca, 'Position'); pos = pos + gpos(3,:); set(gca, 'Position', pos);
tp = title (tstr(3,1:tlen(3))); set(tp, 'Position', [0 0.80 0]);
subplot(2,2,4);
t1 = mean(mod_phys(:,:,9,3,:), 5);
t1(t1 < 0.15) = NaN;
ax = axesm('MapProjection', 'stereo', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
pcolorm(yy100e, xx100e, t1);
plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
caxis ([0 1]);
pos = get(gca, 'Position'); pos = pos + gpos(4,:); set(gca, 'Position', pos);
tp = title (tstr(4,1:tlen(4))); set(tp, 'Position', [0 0.80 0]);
cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos = [(pos(1)+0.1) (cbpos(2)-0.075) (pos(3)-0.2) cbpos(4)]; 
set(cb, 'Position', cbpos); set(gca, 'Position', pos);
def_figure; 
set(gcf, 'PaperPosition', [4.1217 8.5763 12.7340 12.5506]);
def_figure; oname = sprintf('FIGS/ukesm1_PD_hadisst_07.png'); exportgraphics(gcf, oname, 'Resolution', 600);  

% make time-series plots of sea-ice area (total and "extent")
t1 = grid_area([-90:1:90], [-180:1:180]);
had100 = t1(1:end-1,1:end-1) * 1e6;

xarea100e = area100e; xarea100e(332,:) = NaN;

for m = 1:1:12
  t1 = clim_obs(:,:,m,3);
  % north
  t2 = t1 .* had100; t2(1:90,:) = NaN; t2(isnan(t2)) = 0; t3 = nansum(t2(:));
  t2(t1 < 0.15) = NaN; t4 = nansum(t2(:));
  obs_aice(m,1) = t3; obs_eice(m,1) = t4;
  % south
  t2 = t1 .* had100; t2(91:end,:) = NaN; t2(isnan(t2)) = 0; t3 = nansum(t2(:));
  t2(t1 < 0.15) = NaN; t4 = nansum(t2(:));
  obs_aice(m,2) = t3; obs_eice(m,2) = t4;
  %
  t1 = mean(mod_phys(:,:,m,3,:), 5);
  % north
  t2 = t1 .* xarea100e; t2(1:90,:) = NaN; t2(isnan(t2)) = 0; t3 = nansum(t2(:));
  t2(t1 < 0.15) = NaN; t4 = nansum(t2(:));
  mod_aice(m,1) = t3; mod_eice(m,1) = t4;
  % south
  t2 = t1 .* xarea100e; t2(91:end,:) = NaN; t2(isnan(t2)) = 0; t3 = nansum(t2(:));
  t2(t1 < 0.15) = NaN; t4 = nansum(t2(:));
  mod_aice(m,2) = t3; mod_eice(m,2) = t4;
end

obs_aice = obs_aice / 1e12;
obs_eice = obs_eice / 1e12;
mod_aice = mod_aice / 1e12;
mod_eice = mod_eice / 1e12;

monstr = char('J', 'F', 'M', 'A', 'M', 'J', 'J', 'A', 'S', 'O', 'N', 'D');

scl = 0.7; 
tval = [-0.5:1:12.5];
figure(1); clf;
subplot (1,2,1); 
t1 = [obs_aice(12,1); obs_aice(:,1); obs_aice(1,1)];
t2 = [mod_aice(12,1); mod_aice(:,1); mod_aice(1,1)];
plot (tval, t1, 'ro-', tval, t2, 'bs-');
axis ([0 12 0 16.5]);
xlabel ('Time [month]'); ylabel ('Ice area [10^{6} km^{2}]');
set(gca, 'XTick', 0.5:1:11.5, 'XTickLabel', monstr);
title ('Arctic');
legend ('Observed', 'Simulated', 'Location', 'SouthWest');
pos = get(gca, 'Position'); pos(2) = pos(2) + (pos(4) * ((1-scl)/2)); pos(4) = pos(4) * scl; set (gca, 'Position', pos);
subplot (1,2,2); 
t1 = [obs_aice(12,2); obs_aice(:,2); obs_aice(1,2)];
t2 = [mod_aice(12,2); mod_aice(:,2); mod_aice(1,2)];
plot (tval, t1, 'ro-', tval, t2, 'bs-');
axis ([0 12 0 16.5]);
xlabel ('Time [month]'); ylabel ('Ice area [10^{6} km^{2}]');
set(gca, 'XTick', 0.5:1:11.5, 'XTickLabel', monstr);
title ('Antarctic');
pos = get(gca, 'Position'); pos(2) = pos(2) + (pos(4) * ((1-scl)/2)); pos(4) = pos(4) * scl; set (gca, 'Position', pos);
def_figure; % set(gcf, 'PaperPosition', [2.1217 9.0763 16.7340 9.5506]);
def_figure; oname = sprintf('FIGS/ukesm1_PD_hadisst_08.png'); exportgraphics(gcf, oname, 'Resolution', 600);  

% OBS     Arctic    Antarctic
% Max:    13.9078   16.2899
% Min:     4.6744    2.6408
% 
% MOD
% Max:    15.2812   11.7590
% Min:     8.6518    2.7010

tval = [-0.5:1:12.5];
figure(1); clf;
subplot (1,2,1); 
t1 = [obs_eice(12,1); obs_eice(:,1); obs_eice(1,1)];
t2 = [mod_eice(12,1); mod_eice(:,1); mod_eice(1,1)];
plot (tval, t1, 'ro-', tval, t2, 'bs-');
axis ([0 12 0 16.5]);
xlabel ('Time [month]'); ylabel ('Ice extent [10^{6} km^{2}]');
set(gca, 'XTick', 0.5:1:11.5, 'XTickLabel', monstr);
title ('Arctic');
legend ('Observed', 'Simulated', 'Location', 'SouthWest');
pos = get(gca, 'Position'); pos(2) = pos(2) + (pos(4) * ((1-scl)/2)); pos(4) = pos(4) * scl; set (gca, 'Position', pos);
subplot (1,2,2); 
t1 = [obs_eice(12,2); obs_eice(:,2); obs_eice(1,2)];
t2 = [mod_eice(12,2); mod_eice(:,2); mod_eice(1,2)];
plot (tval, t1, 'ro-', tval, t2, 'bs-');
axis ([0 12 0 16.5]);
xlabel ('Time [month]'); ylabel ('Ice extent [10^{6} km^{2}]');
set(gca, 'XTick', 0.5:1:11.5, 'XTickLabel', monstr);
title ('Antarctic');
pos = get(gca, 'Position'); pos(2) = pos(2) + (pos(4) * ((1-scl)/2)); pos(4) = pos(4) * scl; set (gca, 'Position', pos);
def_figure; % set(gcf, 'PaperPosition', [2.1217 9.0763 16.7340 9.5506]);
def_figure; oname = sprintf('FIGS/ukesm1_PD_hadisst_09.png'); exportgraphics(gcf, oname, 'Resolution', 600);  

