clear; startup;

% =========================================================================
% =========================================================================
%
% Matlab plotting scripts for the analysis of UKESM1 prepared for:
%
%   Yool, A., Palmiéri, J., Jones, C. G., de Mora, L., Kuhlbrodt, T., 
%   Popova, E. E., Nurser, A. J. G., Hirschi, J., Blaker, A. T., 
%	Coward, A. C., Blockley, E. W., and Sellar, A. A.: Evaluating the 
% 	physical and biogeochemical state of the global ocean component of 
%	UKESM1 in CMIP6 Historical simulations, Geosci. Model Dev. Discuss. 
%	[preprint], https://doi.org/10.5194/gmd-2020-333, in review, 2020.
%
%   Accepted for publication: 20 April 2021
%
% Note: scripts provided to document analysis, and may require editing to
% fully replicate plots in manuscript
%
% =========================================================================
% =========================================================================

% =========================================================================
%
% This script plots Figures 6 and 7, and Supplementary Figure S7
%
% =========================================================================

% =========================================================================

load ukesm1_PD_ensemble

% =========================================================================

% load up background files
load nemo_area100e
load nemo_mask100e
fname = sprintf('%s/mean_nemo_%so_1y_20001201-20101201_grid-T.nc', dname(1).name, rname(1).name);
t1 = ncread(fname, 'deptht_bounds');
nemo_dep = double(t1(1,:)); nemo_dep(end+1) = double(t1(2,end));
t1 = ncread(fname, 'deptht');
nemo_mid = double(t1');

% total grid area
t1 = area100e + nemo_mask100e(:,:,1);
t2 = isfinite(t1);
tot_area100e = sum(t1(t2)); % m2

% extra mask to get rid of Mediterranean, Black and Caspian Seas
t1 = nemo_mask100e(:,:,1) + 1;
t1(isnan(t1)) = 0;
t1(235:249,283:344) = t1(235:249,283:344) + 2;
t1(249:262,291:344) = t1(249:262,291:344) + 4;
t1(262:276,330:344) = t1(262:276,330:344) + 8;
t2 = t1;
t2(t1 < 1) = NaN; t2(t1 > 1) = NaN;
med_mask100e = t2;

nxx = -179.5:1:179.5;  nyy = -89.5:1:89.5;
[lon100, lat100] = meshgrid(nxx, nyy);
nxx = -180:1:180;  nyy = -90:1:90;
[lon100b, lat100b] = meshgrid(nxx, nyy);

load woa_mask
clear t1; t1(:,1:180) = woa_mask(:,181:360); t1(:,181:360) = woa_mask(:,1:180);
woa_mask = t1;

load woa_basin

helvetica;
pngopt = 0;

% =========================================================================

% load up annual 3D fields
for e = 1:1:tsize
  pnom = sprintf('%s/mean_nemo_%so_1y_20001201-20101201_grid-T.nc', dname(e).name, rname(e).name);
  fprintf('- Reading %s\n', pnom);
  t1 = ncread(pnom, 'thetao'); t2 = permute(t1, [2 1 3]); mod_fld(:,:,:,1,e) = t2;
  t1 = ncread(pnom, 'so'); t2 = permute(t1, [2 1 3]); mod_fld(:,:,:,2,e) = t2;
end

% get sigma
dep_imarnet;
t1 = cumsum(imarnet_dep(:,4));
l75_dep = [0; t1];

load nemo_mask100e
mod_lat = repmat(yy100e, [1 1 75]);
l75_mid = (l75_dep(1:1:end-1) + l75_dep(2:1:end)) / 2;
mod_dep = repmat(permute(l75_mid, [2 3 1]), [332 362 1]);

% create pressure array
mod_prs = sw_pres(mod_dep, mod_lat);

mod_lat = repmat(mod_lat, [1 1 1 1 tsize]);
mod_prs = repmat(mod_prs, [1 1 1 1 tsize]);

% create potential density array
mod_sig = sw_pden(mod_fld(:,:,:,2,:), mod_fld(:,:,:,1,:), mod_prs, (mod_prs * 0));
mod_fld(:,:,:,3,:) = mod_sig;

% get WOA mask onto 75 levels
t1 = ncread('woa13_nitrate_OMIPinit.nc', 'no3'); woa102_no3(:,:,:,1) = permute(t1, [2 1 3]);
t1 = ncread('woa13_nitrate_OMIPinit.nc', 'depth'); woa102_dep = t1;
t1 = woa102_no3; clear t2;
for k = 1:1:101
  t2(:,:,k) = (t1(:,:,k) + t1(:,:,k+1)) / 2;
end
t3 = regrid_vert(t2, woa102_dep, nemo_dep);
woa75_no3 = t3;
woa75_mask = woa75_no3 * 0;

% need to regrid each level of these fields to a regular grid
reg_fld = zeros([180 360 75 6 tsize]) + NaN;
for e = 1:1:tsize
for f = 1:1:3, fprintf('- Ensemble member %d, Field %d  \n', e, f);
  for k = 1:1:74, fprintf('.');
    t1 = mod_fld(:,:,k,f,e);
    t2 = scatregrid(yy100e, xx100e, t1, lat100, lon100);
    reg_fld(:,:,k,f,e) = t2 + woa75_mask(:,:,k);
  end
end
end

% Atlantic and Pacific basins
clear atl_fld pac_fld;
for e = 1:1:tsize
for f = 1:1:3
  for k = 1:1:74
    t0 = reg_fld(:,:,k,f,e);
    t1 = t0 + woa_basin2(:,:,7);
    t2 = nanmean(t1, 2);
    atl_fld(k,:,f,e) = permute(t2, [2 1]);
    t1 = t0 + woa_basin2(:,:,8);
    t2 = nanmean(t1, 2);
    pac_fld(k,:,f,e) = permute(t2, [2 1]);
  end
end
end

save ('ukesm1_PD_phys_section_MOD', 'atl_fld', 'pac_fld', 'nemo_dep', '-v7.3');

% =========================================================================

% load up annual 3D fields; obs
% load up WOA 2013 (1995-2004)
load woa13_95A4_TS

woa_ann(:,:,:,1) = mean(woa13_ptm, 4);
woa_ann(:,:,:,2) = mean(woa13_sal, 4);
woa_ann(:,:,:,3) = mean(woa13_sig, 4);

clear atl_woa pac_woa;
for f = 1:1:3
  for k = 1:1:102, clear t0;
    t0 = woa_ann(:,:,k,f);
    t1 = t0 + woa_basin2(:,:,7);
    t2 = nanmean(t1, 2); 
    atl_woa(k,:,f) = permute(t2, [2 1]);
    t1 = t0 + woa_basin2(:,:,8);
    t2 = nanmean(t1, 2); 
    pac_woa(k,:,f) = permute(t2, [2 1]);
  end
end

clear woa_dep dep0; 
for k = 1:1:102, t1 = woa13_dpt(:,:,k,1); dep0(k) = squeeze(nanmax(t1(:))); end
t1 = (dep0(1:end-1) + dep0(2:end)) / 2;
t2 = [0 t1 5750]; woa_dep = t2;

save ('ukesm1_PD_phys_section_OBS', 'atl_woa', 'pac_woa', 'woa_dep', '-v7.3');

% =========================================================================

load ukesm1_PD_phys_section_MOD
load ukesm1_PD_phys_section_OBS

% =========================================================================

tstr = char('Observed, Atlantic-Pacific zonal mean', 'Simulated, Atlantic-Pacific zonal mean', ...
  'Simulated, Atlantic-Pacific zonal st.dev.');
for f = 1:1:3, tlen(f) = lenstr(tstr(f,:)); end

fnom = char('Pot. temperature [{\circ}C]', 'Salinity [PSU]', ...
  'Pot. density [kg m^{-3}]');
for f = 1:1:3, flen(f) = lenstr(fnom(f,:)); end

% flds = T, S, sigma
clim = [-2 30; 33 37; 25 28];

posdel = [-0.025 0 -0.075 0];

for f = 1:1:3
  
  t1 = atl_woa(:,:,f); t2 = flipdim(t1, 2); t3 = t2(:,1:175);
  t4 = pac_woa(:,:,f); t5 = t4(:,6:160); t6 = [t3 t5]; t6(end+1,:) = NaN; t6(:,end+1) = NaN;
  this_woa = t6;
  t1 = mean(atl_fld(:,:,f,:), 4); t2 = flipdim(t1, 2); t3 = t2(:,1:175);
  t4 = mean(pac_fld(:,:,f,:), 4); t5 = t4(:,6:160); t6 = [t3 t5]; t6(end+1,:) = NaN; t6(end+1,:) = NaN; t6(:,end+1) = NaN;
  this_fld = t6;
  slat = [0:1:330]; rlat = [90 60 30 0 -30 -60 -60 -30 0 30 60]; srlat = [0 30 60 90 120 150 200 230 260 290 320]; 
  
  if f == 3
    this_woa = this_woa - 1e3;
    this_fld = this_fld - 1e3;
  end
  
  ypos = 6.05;
  
  figure(1); clf; turbo(60);
  subplot (2,1,1);
  pcolor (slat, woa_dep / 1e3, this_woa); shading flat;
  seafloor(slat, woa_dep / 1e3, this_woa);  caxis ([clim(f,1) clim(f,2)]);
  set (gca, 'YDir', 'reverse', 'YTick', 0:1:6); yaxis ([0 6.5]);
  set (gca, 'XTick', srlat, 'XTickLabel', rlat);
  tx = text (90, ypos, 'Atlantic'); set(tx, 'HorizontalAlignment', 'Right'); 
  tx = text (260, ypos, 'Pacific'); set(tx, 'HorizontalAlignment', 'Left'); 
  tx = text (175, ypos, 'Southern'); set(tx, 'HorizontalAlignment', 'Center'); 
  hold on; plot ([130 130 220 220], [-1 7 7 -1], 'k:');
  pos = get(gca, 'Position'); pos = pos + posdel; set(gca, 'Position', pos);
  % xlabel ('Latitude [{\circ}N]'); 
  ylabel ('Depth [km]');
  title (tstr(1,1:tlen(1))); topbox;
  %
  subplot (2,1,2);
  pcolor (slat, nemo_dep / 1e3, this_fld); shading flat;
  seafloor(slat, nemo_dep / 1e3, this_fld);  caxis ([clim(f,1) clim(f,2)]);
  set(gca, 'YDir', 'reverse', 'YTick', 0:1:6); yaxis ([0 6.5]);
  set (gca, 'XTick', srlat, 'XTickLabel', rlat);
  tx = text (90, ypos, 'Atlantic'); set(tx, 'HorizontalAlignment', 'Right'); 
  tx = text (260, ypos, 'Pacific'); set(tx, 'HorizontalAlignment', 'Left'); 
  tx = text (175, ypos, 'Southern'); set(tx, 'HorizontalAlignment', 'Center'); 
  hold on; plot ([130 130 220 220], [-1 7 7 -1], 'k:');
  pos = get(gca, 'Position'); pos = pos + posdel; set(gca, 'Position', pos);
  xlabel ('Latitude [{\circ}N]'); ylabel ('Depth [km]');
  title (tstr(2,1:tlen(2))); topbox;
  pos = get(gca, 'Position'); cb = colorbar; cbpos = get(cb, 'Position');
  cbpos(1) = pos(1) + pos(3) + 0.03; set(cb, 'Position', cbpos); set(gca, 'Position', pos);
  cbar_ylabel(cb, fnom(f, 1:flen(f)));
  %
  def_figure; oname = sprintf('FIGS/ukesm1_PD_THC_section_phys_%s.png', numprint(f,2)); exportgraphics(gcf, oname, 'Resolution', 600);
  xlabel (''); 
  def_figure; oname = sprintf('FIGS/ukesm1_PD_THC_section_phys_NOXLAB_%s.png', numprint(f,2)); exportgraphics(gcf, oname, 'Resolution', 600);

end

for f = 1:1:3
  
  t1 = mean(atl_fld(:,:,f,:), 4); t2 = flipdim(t1, 2); t3 = t2(:,1:175);
  t4 = mean(pac_fld(:,:,f,:), 4); t5 = t4(:,6:160); t6 = [t3 t5]; t6(end+1,:) = NaN; t6(end+1,:) = NaN; t6(:,end+1) = NaN;
  this_fld = t6;
  t1 = std(atl_fld(:,:,f,:), [], 4); t2 = flipdim(t1, 2); t3 = t2(:,1:175);
  t4 = std(pac_fld(:,:,f,:), [], 4); t5 = t4(:,6:160); t6 = [t3 t5]; t6(end+1,:) = NaN; t6(end+1,:) = NaN; t6(:,end+1) = NaN;
  this_std = t6;
  slat = [0:1:330]; rlat = [90 60 30 0 -30 -60 -60 -30 0 30 60]; srlat = [0 30 60 90 120 150 200 230 260 290 320]; 
  
  if f == 3
    this_fld = this_fld - 1e3;
  end
  
  ypos = 6.05;
  
  figure(1); clf; turbo(60);
  subplot (2,1,1);
  pcolor (slat, nemo_dep / 1e3, this_fld); shading flat;
  seafloor(slat, nemo_dep / 1e3, this_fld);  caxis ([clim(f,1) clim(f,2)]);
  set(gca, 'YDir', 'reverse', 'YTick', 0:1:6); yaxis ([0 6.5]);
  set (gca, 'XTick', srlat, 'XTickLabel', rlat);
  tx = text (90, ypos, 'Atlantic'); set(tx, 'HorizontalAlignment', 'Right'); 
  tx = text (260, ypos, 'Pacific'); set(tx, 'HorizontalAlignment', 'Left'); 
  tx = text (175, ypos, 'Southern'); set(tx, 'HorizontalAlignment', 'Center'); 
  hold on; plot ([130 130 220 220], [-1 7 7 -1], 'k:');
  pos = get(gca, 'Position'); pos = pos + posdel; set(gca, 'Position', pos);
  % xlabel ('Latitude [{\circ}N]'); 
  ylabel ('Depth [km]');
  title (tstr(2,1:tlen(2))); topbox;
  pos = get(gca, 'Position'); cb = colorbar; cbpos = get(cb, 'Position');
  cbpos(1) = pos(1) + pos(3) + 0.03; set(cb, 'Position', cbpos); set(gca, 'Position', pos);
  cbar_ylabel(cb, fnom(f, 1:flen(f)));
  subplot (2,1,2);
  [mn, mx] = plot_range(this_std, 1); mn = 0 - 1e-6; % if (mx > 2), mx = 2; end;
  pcolor (slat, nemo_dep / 1e3, this_std); shading flat;
  seafloor(slat, nemo_dep / 1e3, this_std);  caxis ([mn mx]);
  set(gca, 'YDir', 'reverse', 'YTick', 0:1:6); yaxis ([0 6.5]);
  set (gca, 'XTick', srlat, 'XTickLabel', rlat);
  tx = text (90, ypos, 'Atlantic'); set(tx, 'HorizontalAlignment', 'Right');
  tx = text (260, ypos, 'Pacific'); set(tx, 'HorizontalAlignment', 'Left');
  tx = text (175, ypos, 'Southern'); set(tx, 'HorizontalAlignment', 'Center');
  hold on; plot ([130 130 220 220], [-1 7 7 -1], 'k:');
  pos = get(gca, 'Position'); pos = pos + posdel; set(gca, 'Position', pos);
  xlabel ('Latitude [{\circ}N]'); ylabel ('Depth [km]');
  title (tstr(3,1:tlen(3))); topbox;
  pos = get(gca, 'Position'); cb = colorbar; cbpos = get(cb, 'Position');
  cbpos(1) = pos(1) + pos(3) + 0.03; set(cb, 'Position', cbpos); set(gca, 'Position', pos);
  cbar_ylabel(cb, fnom(f, 1:flen(f)));  
  %
  def_figure; oname = sprintf('FIGS/ukesm1_PD_THC_section_phys_std_%s.png', numprint(f,2)); exportgraphics(gcf, oname, 'Resolution', 600);

end

% =========================================================================

tstr = char('Difference Atlantic-Pacific zonal mean');
for f = 1:1:1, tlen(f) = lenstr(tstr(f,:)); end

fnom = char('{\Delta} Pot. temperature [{\circ}C]', '{\Delta} Salinity [PSU]', ...
  '{\Delta} Pot. density [kg m^{-3}]');
for f = 1:1:3, flen(f) = lenstr(fnom(f,:)); end

% flds = T, S, sigma
clim = [-4 4; -1 1; -0.8 0.8];

for f = 1:1:3
  
  t1 = atl_woa(:,:,f); t2 = flipdim(t1, 2); t3 = t2(:,1:175);
  t4 = pac_woa(:,:,f); t5 = t4(:,6:160); t6 = [t3 t5]; t6(end+1,:) = NaN; t6(:,end+1) = NaN;
  this_woa = t6;
  t1 = mean(atl_fld(:,:,f,:), 4); t2 = flipdim(t1, 2); t3 = t2(:,1:175);
  t4 = mean(pac_fld(:,:,f,:), 4); t5 = t4(:,6:160); t6 = [t3 t5]; t6(end+1,:) = NaN; t6(end+1,:) = NaN; t6(:,end+1) = NaN;
  this_fld = t6;
  slat = [0:1:330]; rlat = [90 60 30 0 -30 -60 -60 -30 0 30 60]; srlat = [0 30 60 90 120 150 200 230 260 290 320]; 

  this_msk = this_fld * 0;
  t0 = permute(this_woa, [2 3 1]);
  t1 = regrid_vert(t0, woa_dep/1e3, nemo_dep/1e3);
  t2 = permute(t1, [3 1 2]); t2(end+1,:) = NaN;
  t3 = floodfill(t2, 0, 4, 0);
  this_woa2 = t3 + this_msk;
  
  this_delta = this_fld - this_woa2;
  
  ypos = 6.05;
  
  figure(1); clf; therm6(31);
  subplot (2,1,2)
  pcolor (slat, nemo_dep / 1e3, this_delta); shading flat;
  seafloor(slat, nemo_dep / 1e3, this_delta);  caxis ([clim(f,1) clim(f,2)]);
  set(gca, 'YDir', 'reverse', 'YTick', 0:1:6); yaxis ([0 6.5]);
  set (gca, 'XTick', srlat, 'XTickLabel', rlat);
  tx = text (90, ypos, 'Atlantic'); set(tx, 'HorizontalAlignment', 'Right'); 
  tx = text (260, ypos, 'Pacific'); set(tx, 'HorizontalAlignment', 'Left'); 
  tx = text (175, ypos, 'Southern'); set(tx, 'HorizontalAlignment', 'Center'); 
  hold on; plot ([130 130 220 220], [-1 7 7 -1], 'k:');
  pos = get(gca, 'Position'); pos = pos + posdel; set(gca, 'Position', pos);
  xlabel ('Latitude [{\circ}N]'); ylabel ('Depth [km]');
  title (tstr(1,1:tlen(1))); topbox;
  pos = get(gca, 'Position'); cb = colorbar; cbpos = get(cb, 'Position');
  cbpos(1) = pos(1) + pos(3) + 0.03; set(cb, 'Position', cbpos); set(gca, 'Position', pos);
  cbar_ylabel(cb, fnom(f, 1:flen(f)));
  %
  def_figure; oname = sprintf('FIGS/ukesm1_PD_THC_section_phys_delta_%s.png', numprint(f,2)); exportgraphics(gcf, oname, 'Resolution', 600);
end

% %   set(gcf, 'PaperPosition', [2.1217 8.5763 16.7340 12.5506]);
%   pp = [2.1217 8.5763 16.7340 12.5506]; pp(2) = pp(2) + (pp(4) / 4); pp(4) = (pp(4) / 1.5);
%   set(gcf, 'PaperPosition', pp);

% =========================================================================

figure(1); clf;
hold on;
theta_sdiag(atl_woa(:,:,1), atl_woa(:,:,2), [1 0 0]);
theta_sdiag(mean(atl_fld(:,:,1,:), 4), mean(atl_fld(:,:,2,:), 4), [0 0 1]);

figure(1); clf;
hold on;
theta_sdiag(pac_woa(:,:,1), pac_woa(:,:,2), [1 0 0]);
theta_sdiag(mean(pac_fld(:,:,1,:), 4), mean(pac_fld(:,:,2,:), 4), [0 0 1]);

figure(1); clf;
subplot (1,2,1);
hold on;
q1 = atl_woa(:,:,1); q2 = atl_woa(:,:,2); q3 = mean(atl_fld(:,:,1,:), 4); q4 = mean(atl_fld(:,:,2,:), 4);
q5 = [q1(:); q3(:)]; q6 = [q2(:); q4(:)];
theta_sdiag(q5, q6, [1 1 1]);
theta_sdiag(q1, q2, [1 0 0], 1);
theta_sdiag(q3, q4, [0 0 1], 1);
xlabel ('Salinity [PSU]'); ylabel ('Potential temperature [{\circ}C]');
title ('Atlantic'); topbox;
subplot (1,2,2);
hold on;
q1 = pac_woa(:,:,1); q2 = pac_woa(:,:,2); q3 = mean(pac_fld(:,:,1,:), 4); q4 = mean(pac_fld(:,:,2,:), 4);
q5 = [q1(:); q3(:)]; q6 = [q2(:); q4(:)];
theta_sdiag(q5, q6, [1 1 1]);
theta_sdiag(q1, q2, [1 0 0], 1);
theta_sdiag(q3, q4, [0 0 1], 1);
xlabel ('Salinity [PSU]'); ylabel ('Potential temperature [{\circ}C]');
title ('Pacific'); topbox;
paper; orient portrait; def_figure;
def_figure; oname = sprintf('FIGS/ukesm1_PD_TandS_plot_%s.png', numprint(1,2)); exportgraphics(gcf, oname, 'Resolution', 600);

% latitude
ypos = 0.25;

wlat = repmat(-89.5:1:89.5, [102 1]);
flat = repmat(-89.5:1:89.5, [74 1]);

twoa = atl_woa(:,:,1); swoa = atl_woa(:,:,2);
tfld = mean(atl_fld(:,:,1,:), 4); sfld = mean(atl_fld(:,:,2,:), 4); 
twoa(101,1) = min(tfld(:)); twoa(102,1) = max(tfld(:)); 
swoa(101,1) = min(sfld(:)); swoa(102,1) = max(sfld(:));
tfld(73,1) = min(twoa(:)); tfld(74,1) = max(twoa(:)); 
sfld(73,1) = min(swoa(:)); sfld(74,1) = max(swoa(:));

figure(1); clf;
pal = turbo(45); 
subplot (1,2,1);
theta_sdiag2(twoa, swoa, 'color', wlat);
xax = get(gca, 'XLim'); yax = get(gca, 'Ylim'); axis ([xax yax]);
caxis ([(-90 - 1e-6) (90 + 1e-6)]);
title ('Observed, Atlantic'); topbox;
pos = get(gca, 'Position'); pos(4) = pos(4) - (ypos - pos(2)); pos(2) = ypos; set(gca, 'Position', pos);
subplot (1,2,2);
theta_sdiag2(tfld, sfld, 'color', flat);
% axis ([xax yax]);
caxis ([(-90 - 1e-6) (90 + 1e-6)]);
title ('Simulated, Atlantic'); topbox;
pos = get(gca, 'Position'); pos(4) = pos(4) - (ypos - pos(2)); pos(2) = ypos; set(gca, 'Position', pos);
pos = get (gca, 'Position');
cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos(2) = 0.1;
set(cb, 'Position', cbpos); set(gca, 'Position', pos); set(cb, 'XTick', -90:30:90);
cbar_xlabel(cb, 'Latitude [{\circ}N]');
paper; orient portrait; def_figure;
def_figure; oname = sprintf('FIGS/ukesm1_PD_TandS_plot_%s.png', numprint(2,2)); exportgraphics(gcf, oname, 'Resolution', 600);

twoa = pac_woa(:,:,1); swoa = pac_woa(:,:,2);
tfld = mean(pac_fld(:,:,1,:), 4); sfld = mean(pac_fld(:,:,2,:), 4); 
twoa(101,1) = min(tfld(:)); twoa(102,1) = max(tfld(:)); 
swoa(101,1) = min(sfld(:)); swoa(102,1) = max(sfld(:));
tfld(73,1) = min(twoa(:)); tfld(74,1) = max(twoa(:)); 
sfld(73,1) = min(swoa(:)); sfld(74,1) = max(swoa(:));

figure(1); clf;
pal = turbo(45); 
subplot (1,2,1);
theta_sdiag2(twoa, swoa, 'color', wlat);
xax = get(gca, 'XLim'); yax = get(gca, 'Ylim'); axis ([xax yax]);
caxis ([(-90 - 1e-6) (90 + 1e-6)]);
title ('Observed, Pacific'); topbox;
pos = get(gca, 'Position'); pos(4) = pos(4) - (ypos - pos(2)); pos(2) = ypos; set(gca, 'Position', pos);
subplot (1,2,2);
theta_sdiag2(tfld, sfld, 'color', flat);
% axis ([xax yax]);
caxis ([(-90 - 1e-6) (90 + 1e-6)]);
title ('Simulated, Pacific'); topbox;
pos = get(gca, 'Position'); pos(4) = pos(4) - (ypos - pos(2)); pos(2) = ypos; set(gca, 'Position', pos);
pos = get (gca, 'Position');
cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos(2) = 0.1;
set(cb, 'Position', cbpos); set(gca, 'Position', pos); set(cb, 'XTick', -90:30:90);
cbar_xlabel(cb, 'Latitude [{\circ}N]');
paper; orient portrait; def_figure;
def_figure; oname = sprintf('FIGS/ukesm1_PD_TandS_plot_%s.png', numprint(3,2)); exportgraphics(gcf, oname, 'Resolution', 600);

% depth
woa_mid = (woa_dep(1:end-1) + woa_dep(2:end)) / 2;
wdep = repmat(woa_mid', [1 180]);
fdep = repmat(nemo_mid(1:end-1)', [1 180]);

dp = [10 100 1000 5000]; ldp = log10(dp);

twoa = atl_woa(:,:,1); swoa = atl_woa(:,:,2);
tfld = mean(atl_fld(:,:,1,:), 4); sfld = mean(atl_fld(:,:,2,:), 4); 
twoa(101,1) = min(tfld(:)); twoa(102,1) = max(tfld(:)); 
swoa(101,1) = min(sfld(:)); swoa(102,1) = max(sfld(:));
tfld(73,1) = min(twoa(:)); tfld(74,1) = max(twoa(:)); 
sfld(73,1) = min(swoa(:)); sfld(74,1) = max(swoa(:));

figure(1); clf;
pal = turbo(45); 
subplot (1,2,1);
theta_sdiag2(twoa, swoa, 'color', log10(wdep));
xax = get(gca, 'XLim'); yax = get(gca, 'Ylim'); axis ([xax yax]);
caxis ([ldp(1) ldp(end)]);
title ('Observed, Atlantic'); topbox;
pos = get(gca, 'Position'); pos(4) = pos(4) - (ypos - pos(2)); pos(2) = ypos; set(gca, 'Position', pos);
subplot (1,2,2);
theta_sdiag2(tfld, sfld, 'color', log10(fdep));
% axis ([xax yax]);
caxis ([ldp(1) ldp(end)]);
title ('Simulated, Atlantic'); topbox;
pos = get(gca, 'Position'); pos(4) = pos(4) - (ypos - pos(2)); pos(2) = ypos; set(gca, 'Position', pos);
pos = get (gca, 'Position');
cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos(2) = 0.1;
set(cb, 'Position', cbpos); set(gca, 'Position', pos); set(cb, 'XTick', ldp, 'XTickLabel', dp);
cbar_xlabel(cb, 'Depth [m]');
paper; orient portrait; def_figure;
def_figure; oname = sprintf('FIGS/ukesm1_PD_TandS_plot_%s.png', numprint(4,2)); exportgraphics(gcf, oname, 'Resolution', 600);
  
twoa = pac_woa(:,:,1); swoa = pac_woa(:,:,2);
tfld = mean(pac_fld(:,:,1,:), 4); sfld = mean(pac_fld(:,:,2,:), 4); 
twoa(101,1) = min(tfld(:)); twoa(102,1) = max(tfld(:)); 
swoa(101,1) = min(sfld(:)); swoa(102,1) = max(sfld(:));
tfld(73,1) = min(twoa(:)); tfld(74,1) = max(twoa(:)); 
sfld(73,1) = min(swoa(:)); sfld(74,1) = max(swoa(:));

figure(1); clf;
pal = turbo(45); 
subplot (1,2,1);
theta_sdiag2(twoa, swoa, 'color', log10(wdep));
xax = get(gca, 'XLim'); yax = get(gca, 'Ylim'); axis ([xax yax]);
caxis ([ldp(1) ldp(end)]);
title ('Observed, Pacific'); topbox;
pos = get(gca, 'Position'); pos(4) = pos(4) - (ypos - pos(2)); pos(2) = ypos; set(gca, 'Position', pos);
subplot (1,2,2);
theta_sdiag2(tfld, sfld, 'color', log10(fdep));
% axis ([xax yax]);
caxis ([ldp(1) ldp(end)]);
title ('Simulated, Pacific'); topbox;
pos = get(gca, 'Position'); pos(4) = pos(4) - (ypos - pos(2)); pos(2) = ypos; set(gca, 'Position', pos);
pos = get (gca, 'Position');
cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos(2) = 0.1;
set(cb, 'Position', cbpos); set(gca, 'Position', pos); set(cb, 'XTick', ldp, 'XTickLabel', dp);
cbar_xlabel(cb, 'Depth [m]');
paper; orient portrait; def_figure;
def_figure; oname = sprintf('FIGS/ukesm1_PD_TandS_plot_%s.png', numprint(5,2)); exportgraphics(gcf, oname, 'Resolution', 600);
