clear; startup;

% =========================================================================
% =========================================================================
%
% Matlab plotting scripts for the analysis of UKESM1 prepared for:
%
%   Yool, A., Palmiéri, J., Jones, C. G., de Mora, L., Kuhlbrodt, T., 
%   Popova, E. E., Nurser, A. J. G., Hirschi, J., Blaker, A. T., 
%	Coward, A. C., Blockley, E. W., and Sellar, A. A.: Evaluating the 
% 	physical and biogeochemical state of the global ocean component of 
%	UKESM1 in CMIP6 Historical simulations, Geosci. Model Dev. Discuss. 
%	[preprint], https://doi.org/10.5194/gmd-2020-333, in review, 2020.
%
%   Accepted for publication: 20 April 2021
%
% Note: scripts provided to document analysis, and may require editing to
% fully replicate plots in manuscript
%
% =========================================================================
% =========================================================================

% =========================================================================
%
% This script plots Supplementary Figure S4
%
% =========================================================================
% =========================================================================

% load up background files
load nemo_area100e
load nemo_mask100e

% total grid area
t1 = area100e + nemo_mask100e(:,:,1);
t2 = isfinite(t1);
tot_area100e = sum(t1(t2)); % m2

% extra mask to get rid of Mediterranean, Black and Caspian Seas
t1 = nemo_mask100e(:,:,1) + 1;
t1(isnan(t1)) = 0;
t1(235:249,283:344) = t1(235:249,283:344) + 2;
t1(249:262,291:344) = t1(249:262,291:344) + 4;
t1(262:276,330:344) = t1(262:276,330:344) + 8;
t2 = t1;
t2(t1 < 1) = NaN; t2(t1 > 1) = NaN;
med_mask100e = t2;

nxx = -179.5:1:179.5;  nyy = -89.5:1:89.5;
[lon100, lat100] = meshgrid(nxx, nyy);
nxx = -180:1:180;  nyy = -90:1:90;
[lon100b, lat100b] = meshgrid(nxx, nyy);

load woa_mask
clear t1; t1(:,1:180) = woa_mask(:,181:360); t1(:,181:360) = woa_mask(:,1:180);
woa_mask = t1;

load woa_basin

helvetica;
pngopt = 0;

% =====

% SST and sea-ice
t1 = ncread('HadISST_sst.nc', 'sst');
t2 = flipdim(permute(t1, [2 1 3]), 1);
% ends April 2018; i.e. need to lose 3 years and 4 months from end
t3 = t2(:,:,1:(end - ((8 * 12) + 4)));
% but only want the 45 year period from 1970-2014 inclusive
t4 = t3(:,:,(end-539):1:end);
t4(t4 == -1000) = NaN;
obs_sst = t4;

mod_sst = zeros([332 362 540 9]);

dnom = dir('u-*');
for r = 1:1:9
  dirnom = dnom(r).name;
  pos = 1;
  for y = 1:1:45, yr = y + 1969;
    evalstr = sprintf('fnom = dir(''%s/nemo_%so_1m_%d*'');', dirnom, dirnom(3:end), yr);
    eval(evalstr);
    for m = 1:1:12, inom = sprintf('%s/%s', dirnom, fnom(m).name);
      fprintf('- Reading %s\n', inom);
      t1 = ncread(inom, 'tos');
      t2 = permute(t1, [2 1]);
      mod_sst(:,:,pos,r) = t2;
      pos = pos + 1;
    end
  end
end

save ('raw_sst', 'obs_sst', 'mod_sst', '-v7.3');

% create seasonal averages

seas = [3 4 5; 6 7 8; 9 10 11; 12 1 2];

seas_mod_sst = zeros([332 362 (45*4) 9]);

for r = 1:1:9
  pos = 1;
  fprintf('- Run %d: ', r);
  for y = 1:1:45, top = y * 12; bot = top - 11;
    t0 = mod_sst(:,:,bot:top,r);
    for s = 1:1:4
      t1 = t0(:,:,seas(s,:));
      t2 = mean(t1, 3);
      seas_mod_sst(:,:,pos,r) = t2;
      pos = pos + 1;
    end, fprintf('.');
  end, fprintf('\n');
end

% same for obs
pos = 1;
for y = 1:1:45, top = y * 12; bot = top - 11;
  t0 = obs_sst(:,:,bot:top);
  for s = 1:1:4
    t1 = t0(:,:,seas(s,:));
    t2 = mean(t1, 3);
    seas_obs_sst(:,:,pos) = t2;
    pos = pos + 1;
  end, fprintf('.');
end, fprintf('\n');

% regrid obs
for s = 1:1:180
  regrid_seas_obs_sst(:,:,s) = scatregrid(lat100, lon100, seas_obs_sst(:,:,s), yy100e, xx100e) + nemo_mask100e(:,:,1);
  fprintf('.'); if mod(s, 12) == 0, fprintf(' (%d)\n', s / 3); end
end

% make climatologies
for r = 1:1:9
  for d = 1:1:4
    top = d * 40; bot = top - 39;
    t0 = seas_mod_sst(:,:,bot:top,r);
    for s = 1:1:4
      clim_mod_sst(:,:,s,d,r) = mean(t0(:,:,s:4:(41-s)), 3);
    end
  end
end

% cross-ensemble
for d = 1:1:4
  for s = 1:1:4
    climX_mod_sst(:,:,s,d) = mean(clim_mod_sst(:,:,s,d,:), 5);
  end
end

for d = 1:1:4
  top = d * 40; bot = top - 39;
  t0 = regrid_seas_obs_sst(:,:,bot:top);
  for s = 1:1:4
    clim_obs_sst(:,:,s,d) = mean(t0(:,:,s:4:(41-s)), 3);
  end
end

gpos = [-0.075 0 0.135 0.1; -0.05 0 0.135 0.1];
lon_rng = [-180 180]; lat_rng = [-90 90];
load coast;
load nemo_area100e;

for d = 1:1:4, dec = (d*10) + 1960;
  o1 = clim_obs_sst(:,:,2:2:4,d);
  m1 = climX_mod_sst(:,:,2:2:4,d);
  
  tstr(1,:) = sprintf('Difference, JJA, %ds', dec); tstr(2,:) = sprintf('Difference, DJF, %ds', dec);
  for f = 1:1:2, tlen(f) = lenstr(tstr(f,:)); end
  
  figure(1); clf; therm6(31);
  subplot(1,2,1);
  ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
  t1 = m1(:,:,1) - o1(:,:,1);
  pcolorm(yy100e, xx100e, t1);
  plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
  caxis ([-4.001 4.001]);
  pos = get(gca, 'Position'); pos = pos + gpos(1,:); set(gca, 'Position', pos);
  tp = title (tstr(1,1:tlen(1))); set(tp, 'Position', [0 1.6 0]);
  subplot(1,2,2);
  ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
  t1 = m1(:,:,2) - o1(:,:,2);
  pcolorm(yy100e, xx100e, t1);
  plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
  caxis ([-4.001 4.001]);
  pos = get(gca, 'Position'); pos = pos + gpos(2,:); set(gca, 'Position', pos);
  tp = title (tstr(2,1:tlen(2))); set(tp, 'Position', [0 1.6 0]);
  cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos(1) = cbpos(1) + 0.025; cbpos(2) = cbpos(2) - 0.01; cbpos(3) = cbpos(3) - 0.05;
  set(cb, 'Position', cbpos); set(gca, 'Position', pos);
  def_figure; oname = sprintf('FIGS/ukesm1_SST_ensembleX_%ds.png', dec); exportgraphics(gcf, oname, 'Resolution', 600);
  set(cb, 'Visible', 'off');
  def_figure; oname = sprintf('FIGS/ukesm1_SST_ensembleX_%ds_NOLEG.png', dec); exportgraphics(gcf, oname, 'Resolution', 600); 
end



