clear; startup;

% =========================================================================
% =========================================================================
%
% Matlab plotting scripts for the analysis of UKESM1 prepared for:
%
%   Yool, A., Palmiéri, J., Jones, C. G., de Mora, L., Kuhlbrodt, T., 
%   Popova, E. E., Nurser, A. J. G., Hirschi, J., Blaker, A. T., 
%	Coward, A. C., Blockley, E. W., and Sellar, A. A.: Evaluating the 
% 	physical and biogeochemical state of the global ocean component of 
%	UKESM1 in CMIP6 Historical simulations, Geosci. Model Dev. Discuss. 
%	[preprint], https://doi.org/10.5194/gmd-2020-333, in review, 2020.
%
%   Accepted for publication: 20 April 2021
%
% Note: scripts provided to document analysis, and may require editing to
% fully replicate plots in manuscript
%
% =========================================================================
% =========================================================================

% =========================================================================
%
% This script plots Figures 18 and 22
%
% =========================================================================

% =========================================================================

load ukesm1_PD_ensemble

% =========================================================================

% load up background files
load nemo_area100e
load nemo_mask100e
fname = sprintf('%s/mean_nemo_%so_1y_20001201-20101201_grid-T.nc', dname(1).name, rname(1).name);
t1 = ncread(fname, 'deptht_bounds');
nemo_dep = double(t1(1,:)); nemo_dep(end+1) = double(t1(2,end));
t1 = ncread(fname, 'deptht');
nemo_mid = double(t1');

% total grid area
t1 = area100e + nemo_mask100e(:,:,1);
t2 = isfinite(t1);
tot_area100e = sum(t1(t2)); % m2

% extra mask to get rid of Mediterranean, Black and Caspian Seas
t1 = nemo_mask100e(:,:,1) + 1;
t1(isnan(t1)) = 0;
t1(235:249,283:344) = t1(235:249,283:344) + 2;
t1(249:262,291:344) = t1(249:262,291:344) + 4;
t1(262:276,330:344) = t1(262:276,330:344) + 8;
t2 = t1;
t2(t1 < 1) = NaN; t2(t1 > 1) = NaN;
med_mask100e = t2;

nxx = -179.5:1:179.5;  nyy = -89.5:1:89.5;
[lon100, lat100] = meshgrid(nxx, nyy);
nxx = -180:1:180;  nyy = -90:1:90;
[lon100b, lat100b] = meshgrid(nxx, nyy);
nxx = -179.5:2:179.5;  nyy = -89.5:2:89.5;
[lon200, lat200] = meshgrid(nxx, nyy);

load woa_mask
clear t1; t1(:,1:180) = woa_mask(:,181:360); t1(:,181:360) = woa_mask(:,1:180);
woa_mask = t1;

load woa_basin

helvetica;
pngopt = 0;

% ----------------------------------------------------------------------

addpath('~/Matlab/GLODAP/');
load woa_basin

load GLODAPv1_per_m3_fields

gd1_pi_dic = new_pi_dic;
gd1_totco2 = new_totco2;
gd1_totalk = new_totalk;
gd1_anthro = new_anthro;
gd1_cfc_11 = new_cfc_11;

addpath('~/Matlab/GLODAP_v2/');
load per_m3_fields

gd2_pi_dic = new_pi_dic;
gd2_totco2 = new_totco2;
gd2_totalk = new_totalk;
gd2_anthro = new_anthro;

obs_dep = ncread('AnthCO2.nc', 'depth');
obs_lon = ncread('AnthCO2.nc', 'longitude');
% t1 = double(ncread('AnthCO2.nc', 'AnthCO2'));
% t2 = t1; % permute(t1, [2 1 3]);
% t2(t2 == -999) = 0;
t2 = gd1_anthro; t2(isnan(t2)) = 0;
t3 = t2(:,:,1) * 0;
for k = 1:1:32
    q1 = (t2(:,:,k) + t2(:,:,k+1)) / 2;
    q2 = obs_dep(k+1) - obs_dep(k);
    q3 = q1 * q2;
    t3 = t3 + q3;
end
t4 = t3; % clear t4; t4(:,1:200) = t3(:,161:360); t4(:,201:360) = t3(:,1:160);
t4(t4 == 0) = NaN;
obs_ant1 = t4 + woa_basin2(:,:,1);

t2 = gd1_totco2; t2(isnan(t2)) = 0;
t3 = t2(:,:,1) * 0;
for k = 1:1:32
    q1 = (t2(:,:,k) + t2(:,:,k+1)) / 2;
    q2 = obs_dep(k+1) - obs_dep(k);
    q3 = q1 * q2;
    t3 = t3 + q3;
end
t4 = t3; % clear t4; t4(:,1:200) = t3(:,161:360); t4(:,201:360) = t3(:,1:160);
t4(t4 == 0) = NaN;
obs_dic1 = t4 + woa_basin2(:,:,1);

% t1 = double(ncread('CFC11.nc', 'CFC11'));
% t2 = t1; % permute(t1, [2 1 3]);
% t2(t2 == -999) = 0;
t2 = gd1_cfc_11; t2(isnan(t2)) = 0;
t3 = t2(:,:,1) * 0;
for k = 1:1:32
    q1 = (t2(:,:,k) + t2(:,:,k+1)) / 2;
    q2 = obs_dep(k+1) - obs_dep(k);
    q3 = q1 * q2;
    t3 = t3 + q3;
end
t4 = t3; % clear t4; t4(:,1:200) = t3(:,161:360); t4(:,201:360) = t3(:,1:160);

t4(t4 == 0) = NaN;
obs_cfc11 = t4 + woa_basin2(:,:,1);

obs_dep = ncread('GLODAPv2.2016b.Cant.nc', 'Depth');
obs_lon = ncread('GLODAPv2.2016b.Cant.nc', 'lon');
% t1 = ncread('GLODAPv2.2016b.Cant.nc', 'Cant');
% t2 = permute(t1, [2 1 3]);
% t2(isnan(t2)) = 0;
t2 = gd2_anthro; t2(isnan(t2)) = 0;
t3 = t2(:,:,1) * 0;
for k = 1:1:32
    q1 = (t2(:,:,k) + t2(:,:,k+1)) / 2;
    q2 = obs_dep(k+1) - obs_dep(k);
    q3 = q1 * q2;
    t3 = t3 + q3;
end
clear t4; t4 = t3; % t4(:,1:200) = t3(:,161:360); t4(:,201:360) = t3(:,1:160);
obs_ant2 = t4 + woa_basin2(:,:,1);

t2 = gd2_totco2; t2(isnan(t2)) = 0;
t3 = t2(:,:,1) * 0;
for k = 1:1:32
    q1 = (t2(:,:,k) + t2(:,:,k+1)) / 2;
    q2 = obs_dep(k+1) - obs_dep(k);
    q3 = q1 * q2;
    t3 = t3 + q3;
end
clear t4; t4 = t3; % t4(:,1:200) = t3(:,161:360); t4(:,201:360) = t3(:,1:160);
obs_dic2 = t4 + woa_basin2(:,:,1);

% ----------------------------------------------------------------------

clear pi_dic1 pi_alk1 pi_dic2 pi_alk2 pd_dic1 pd_dic2 pd_alk1 pd_alk2 pd_cfc11;

for e = 1:1:tsize
  % PIcontrol data
  dnom = sprintf('%s/mean_medusa_aw310o_*ptrc-T.nc', dname(e).name);
  t1 = dir(dnom);
  nom1990s = sprintf('%s/%s', dname(e).name, t1(1).name);
  nom2000s = sprintf('%s/%s', dname(e).name, t1(2).name);
  %
  fprintf('- Reading %s\n', nom1990s);
  t1 = ncread(nom1990s, 'DIC_E3T');
  t2 = permute(t1, [2 1 3]);
  pi_dic1(:,:,e) = nansum(t2, 3) + nemo_mask100e(:,:,1);
  t1 = ncread(nom1990s, 'ALK_E3T');
  t2 = permute(t1, [2 1 3]);
  pi_alk1(:,:,e) = nansum(t2, 3) + nemo_mask100e(:,:,1);
  fprintf('- Reading %s\n', nom2000s);
  t1 = ncread(nom2000s, 'DIC_E3T');
  t2 = permute(t1, [2 1 3]);
  pi_dic2(:,:,e) = nansum(t2, 3) + nemo_mask100e(:,:,1);
  t1 = ncread(nom2000s, 'ALK_E3T');
  t2 = permute(t1, [2 1 3]);
  pi_alk2(:,:,e) = nansum(t2, 3) + nemo_mask100e(:,:,1);
  
  % PD data
  pnom = sprintf('%s/mean_medusa_%so_1y_19901201-20001201_ptrc-T.nc', dname(e).name, rname(e).name);
  fprintf('- Reading %s\n', pnom);
  t1 = ncread(pnom, 'DIC_E3T');
  t2 = permute(t1, [2 1 3]);
  pd_dic1(:,:,e) = nansum(t2, 3) + nemo_mask100e(:,:,1);
  t1 = ncread(pnom, 'ALK_E3T');
  t2 = permute(t1, [2 1 3]);
  pd_alk1(:,:,e) = nansum(t2, 3) + nemo_mask100e(:,:,1);
  t1 = ncread(pnom, 'CFC11_E3T');
  t2 = permute(t1, [2 1 3]);
  pd_cfc11(:,:,e) = nansum(t2, 3) + nemo_mask100e(:,:,1);
  %
  pnom = sprintf('%s/mean_medusa_%so_1y_20001201-20101201_ptrc-T.nc', dname(e).name, rname(e).name);
  fprintf('- Reading %s\n', pnom);
  t1 = ncread(pnom, 'DIC_E3T');
  t2 = permute(t1, [2 1 3]);
  pd_dic2(:,:,e) = nansum(t2, 3) + nemo_mask100e(:,:,1);
  t1 = ncread(pnom, 'ALK_E3T');
  t2 = permute(t1, [2 1 3]);
  pd_alk2(:,:,e) = nansum(t2, 3) + nemo_mask100e(:,:,1);
end

pd_ant1 = pd_dic1 - pi_dic1;
pd_ant2 = pd_dic2 - pi_dic2;

% ----------------------------------------------------------------------

lon_rng = [-180 180]; lat_rng = [-90 90];
gpos = [-0.075 0 0.135 0.1; -0.05 0 0.135 0.1];
load coast;

% GLODAP v1.1

tstr = char('Observed', 'Simulated');
for f = 1:1:2, tlen(f) = lenstr(tstr(f,:)); end
figure(1); clf; turbo(60);
subplot(1,2,1);
ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
t0 = obs_ant1 / 1e3; t1 = t0; t1(t1 == 0) = NaN; % t1 = log10(t0); t1(t1 < 1) = NaN;
pcolorm(lat100, lon100, t1);
plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
caxis ([-0.001 120.001]);
pos = get(gca, 'Position'); pos = pos + gpos(1,:); set(gca, 'Position', pos);
tp = title (tstr(1,1:tlen(1))); set(tp, 'Position', [0 1.6 0]);
subplot(1,2,2);
ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
t0 = mean(pd_ant1, 3) / 1e3; t1 = t0; % t1 = log10(t0); t1(t1 < 1) = NaN;
pcolorm(yy100e, xx100e, t1);
plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
caxis ([-0.001 120.001]);
pos = get(gca, 'Position'); pos = pos + gpos(2,:); set(gca, 'Position', pos);
tp = title (tstr(2,1:tlen(2))); set(tp, 'Position', [0 1.6 0]);
cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos(1) = cbpos(1) + 0.025; cbpos(2) = cbpos(2) - 0.01; cbpos(3) = cbpos(3) - 0.05;
set(cb, 'Position', cbpos); set(gca, 'Position', pos);
set (cb, 'XTick', 0:20:120);
def_figure; oname = sprintf('FIGS/ukesm1_PD_obs_mod_anthro_%s.png', numprint(1,2)); exportgraphics(gcf, oname, 'Resolution', 600);

tstr = char('Observed', 'Simulated', 'Simulated, st. dev.');
for f = 1:1:3, tlen(f) = lenstr(tstr(f,:)); end
figure(1); clf; turbo(60);
subplot(1,2,1);
ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
t0 = mean(pd_ant1, 3) / 1e3; t1 = t0; % t1 = log10(t0); t1(t1 < 1) = NaN;
pcolorm(yy100e, xx100e, t1);
plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
caxis ([-0.001 120.001]);
pos = get(gca, 'Position'); pos = pos + gpos(2,:); set(gca, 'Position', pos);
tp = title (tstr(2,1:tlen(2))); set(tp, 'Position', [0 1.6 0]);
cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos(1) = cbpos(1) + 0.05; cbpos(2) = cbpos(2) - 0.01; cbpos(3) = cbpos(3) - 0.1;
set(cb, 'Position', cbpos); set(gca, 'Position', pos);
set (cb, 'XTick', 0:20:120);
subplot(1,2,2);
ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
t0 = std(pd_ant1, [], 3) / 1e3; t1 = t0; % t1 = log10(t0); t1(t1 < 1) = NaN;
pcolorm(yy100e, xx100e, t1);
plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
caxis ([-0.001 20.001]);
pos = get(gca, 'Position'); pos = pos + gpos(2,:); set(gca, 'Position', pos);
tp = title (tstr(3,1:tlen(3))); set(tp, 'Position', [0 1.6 0]);
cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos(1) = cbpos(1) + 0.05; cbpos(2) = cbpos(2) - 0.01; cbpos(3) = cbpos(3) - 0.1;
set(cb, 'Position', cbpos); set(gca, 'Position', pos);
set (cb, 'XTick', 0:5:20);
def_figure; oname = sprintf('FIGS/ukesm1_PD_obs_mod_anthro_STD_%s.png', numprint(1,2)); exportgraphics(gcf, oname, 'Resolution', 600);

tstr = char('Observed', 'Simulated', 'Simulated, CoV');
for f = 1:1:3, tlen(f) = lenstr(tstr(f,:)); end
figure(1); clf; turbo(60);
subplot(1,2,1);
ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
t0 = mean(pd_ant1, 3) / 1e3; t1 = t0; % t1 = log10(t0); t1(t1 < 1) = NaN;
pcolorm(yy100e, xx100e, t1);
plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
caxis ([-0.001 120.001]);
pos = get(gca, 'Position'); pos = pos + gpos(2,:); set(gca, 'Position', pos);
tp = title (tstr(2,1:tlen(2))); set(tp, 'Position', [0 1.6 0]);
cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos(1) = cbpos(1) + 0.05; cbpos(2) = cbpos(2) - 0.01; cbpos(3) = cbpos(3) - 0.1;
set(cb, 'Position', cbpos); set(gca, 'Position', pos);
set (cb, 'XTick', 0:20:120);
subplot(1,2,2);
ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
t0a = mean(pd_ant1, 3) / 1e3; t0b = std(pd_ant1, [], 3) / 1e3; t1 = t0b ./ t0a; % t1 = log10(t0); t1(t1 < 1) = NaN;
pcolorm(yy100e, xx100e, t1);
plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
caxis ([-0.001 5.001]);
pos = get(gca, 'Position'); pos = pos + gpos(2,:); set(gca, 'Position', pos);
tp = title (tstr(3,1:tlen(3))); set(tp, 'Position', [0 1.6 0]);
cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos(1) = cbpos(1) + 0.05; cbpos(2) = cbpos(2) - 0.01; cbpos(3) = cbpos(3) - 0.1;
set(cb, 'Position', cbpos); set(gca, 'Position', pos);
set (cb, 'XTick', 0:1:5);
def_figure; oname = sprintf('FIGS/ukesm1_PD_obs_mod_anthro_CoV_%s.png', numprint(1,2)); exportgraphics(gcf, oname, 'Resolution', 600);

tstr = char('Observed', 'Simulated');
for f = 1:1:2, tlen(f) = lenstr(tstr(f,:)); end
figure(1); clf; turbo(60);
subplot(1,2,1);
ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
t0 = obs_cfc11 / 1e3; t1 = t0; t1(t1 == 0) = NaN; % t1 = log10(t0); t1(t1 < 1) = NaN;
pcolorm(lat100, lon100, t1);
plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
caxis ([-0.001 10.001]);
pos = get(gca, 'Position'); pos = pos + gpos(1,:); set(gca, 'Position', pos);
tp = title (tstr(1,1:tlen(1))); set(tp, 'Position', [0 1.6 0]);
subplot(1,2,2);
ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
t0 = mean(pd_cfc11, 3) * 1e6; t1 = t0; % t1 = log10(t0); t1(t1 < 1) = NaN;
pcolorm(yy100e, xx100e, t1);
plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
caxis ([-0.001 10.001]);
pos = get(gca, 'Position'); pos = pos + gpos(2,:); set(gca, 'Position', pos);
tp = title (tstr(2,1:tlen(2))); set(tp, 'Position', [0 1.6 0]);
cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos(1) = cbpos(1) + 0.025; cbpos(2) = cbpos(2) - 0.01; cbpos(3) = cbpos(3) - 0.05;
set(cb, 'Position', cbpos); set(gca, 'Position', pos);
set (cb, 'XTick', 0:2:10);
def_figure; oname = sprintf('FIGS/ukesm1_PD_obs_mod_anthro_%s.png', numprint(2,2)); exportgraphics(gcf, oname, 'Resolution', 600);

tstr = char('Observed', 'Simulated', 'Simulated, st. dev.');
for f = 1:1:3, tlen(f) = lenstr(tstr(f,:)); end
figure(1); clf; turbo(60);
subplot(1,2,1);
ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
t0 = mean(pd_cfc11, 3) * 1e6; t1 = t0; % t1 = log10(t0); t1(t1 < 1) = NaN;
pcolorm(yy100e, xx100e, t1);
plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
caxis ([-0.001 10.001]);
pos = get(gca, 'Position'); pos = pos + gpos(2,:); set(gca, 'Position', pos);
tp = title (tstr(2,1:tlen(2))); set(tp, 'Position', [0 1.6 0]);
cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos(1) = cbpos(1) + 0.05; cbpos(2) = cbpos(2) - 0.01; cbpos(3) = cbpos(3) - 0.1;
set(cb, 'Position', cbpos); set(gca, 'Position', pos);
set (cb, 'XTick', 0:2:10);
subplot(1,2,2);
ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
t0 = std(pd_cfc11, [], 3) * 1e6; t1 = t0; % t1 = log10(t0); t1(t1 < 1) = NaN;
pcolorm(yy100e, xx100e, t1);
plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
caxis ([-0.001 1.001]);
pos = get(gca, 'Position'); pos = pos + gpos(2,:); set(gca, 'Position', pos);
tp = title (tstr(3,1:tlen(3))); set(tp, 'Position', [0 1.6 0]);
cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos(1) = cbpos(1) + 0.05; cbpos(2) = cbpos(2) - 0.01; cbpos(3) = cbpos(3) - 0.1;
set(cb, 'Position', cbpos); set(gca, 'Position', pos);
set (cb, 'XTick', 0:0.2:1);
def_figure; oname = sprintf('FIGS/ukesm1_PD_obs_mod_anthro_STD_%s.png', numprint(2,2)); exportgraphics(gcf, oname, 'Resolution', 600);

tstr = char('Observed', 'Simulated', 'Simulated, CoV');
for f = 1:1:3, tlen(f) = lenstr(tstr(f,:)); end
figure(1); clf; turbo(60);
subplot(1,2,1);
ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
t0 = mean(pd_cfc11, 3) * 1e6; t1 = t0; % t1 = log10(t0); t1(t1 < 1) = NaN;
pcolorm(yy100e, xx100e, t1);
plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
caxis ([-0.001 10.001]);
pos = get(gca, 'Position'); pos = pos + gpos(2,:); set(gca, 'Position', pos);
tp = title (tstr(2,1:tlen(2))); set(tp, 'Position', [0 1.6 0]);
cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos(1) = cbpos(1) + 0.05; cbpos(2) = cbpos(2) - 0.01; cbpos(3) = cbpos(3) - 0.1;
set(cb, 'Position', cbpos); set(gca, 'Position', pos);
set (cb, 'XTick', 0:2:10);
subplot(1,2,2);
ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
t0a = mean(pd_cfc11, 3) * 1e6; t0b = std(pd_cfc11, [], 3) * 1e6; t1 = t0b ./ t0a; % t1 = log10(t0); t1(t1 < 1) = NaN;
pcolorm(yy100e, xx100e, t1);
plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
caxis ([-0.0001 0.4001]);
pos = get(gca, 'Position'); pos = pos + gpos(2,:); set(gca, 'Position', pos);
tp = title (tstr(3,1:tlen(3))); set(tp, 'Position', [0 1.6 0]);
cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos(1) = cbpos(1) + 0.05; cbpos(2) = cbpos(2) - 0.01; cbpos(3) = cbpos(3) - 0.1;
set(cb, 'Position', cbpos); set(gca, 'Position', pos);
set (cb, 'XTick', 0:0.1:0.4);
def_figure; oname = sprintf('FIGS/ukesm1_PD_obs_mod_anthro_CoV_%s.png', numprint(2,2)); exportgraphics(gcf, oname, 'Resolution', 600);

% GLODAP v2

tstr = char('Observed', 'Simulated');
for f = 1:1:2, tlen(f) = lenstr(tstr(f,:)); end
figure(1); clf; turbo(60);
subplot(1,2,1);
ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
t0 = obs_ant2 / 1e3; t1 = t0; t1(t1 == 0) = NaN; % t1 = log10(t0); t1(t1 < 1) = NaN;
pcolorm(lat100, lon100, t1);
plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
caxis ([-0.001 140.001]);
pos = get(gca, 'Position'); pos = pos + gpos(1,:); set(gca, 'Position', pos);
tp = title (tstr(1,1:tlen(1))); set(tp, 'Position', [0 1.6 0]);
subplot(1,2,2);
ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
t0 = mean(pd_ant2, 3) / 1e3; t1 = t0; % t1 = log10(t0); t1(t1 < 1) = NaN;
pcolorm(yy100e, xx100e, t1);
plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
caxis ([-0.001 140.001]);
pos = get(gca, 'Position'); pos = pos + gpos(2,:); set(gca, 'Position', pos);
tp = title (tstr(2,1:tlen(2))); set(tp, 'Position', [0 1.6 0]);
cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos(1) = cbpos(1) + 0.025; cbpos(2) = cbpos(2) - 0.01; cbpos(3) = cbpos(3) - 0.05;
set(cb, 'Position', cbpos); set(gca, 'Position', pos);
set (cb, 'XTick', 0:20:140);
def_figure; oname = sprintf('FIGS/ukesm1_PD_obs_mod_anthro_%s.png', numprint(3,2)); exportgraphics(gcf, oname, 'Resolution', 600);

tstr = char('Observed', 'Simulated', 'Simulated, st. dev.');
for f = 1:1:3, tlen(f) = lenstr(tstr(f,:)); end
figure(1); clf; turbo(60);
subplot(1,2,1);
ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
t0 = mean(pd_ant2, 3) / 1e3; t1 = t0; % t1 = log10(t0); t1(t1 < 1) = NaN;
pcolorm(yy100e, xx100e, t1);
plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
caxis ([-0.001 140.001]);
pos = get(gca, 'Position'); pos = pos + gpos(2,:); set(gca, 'Position', pos);
tp = title (tstr(2,1:tlen(2))); set(tp, 'Position', [0 1.6 0]);
cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos(1) = cbpos(1) + 0.05; cbpos(2) = cbpos(2) - 0.01; cbpos(3) = cbpos(3) - 0.1;
set(cb, 'Position', cbpos); set(gca, 'Position', pos);
set (cb, 'XTick', 0:20:140);
subplot(1,2,2);
ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
t0 = std(pd_ant2, [], 3) / 1e3; t1 = t0; % t1 = log10(t0); t1(t1 < 1) = NaN;
pcolorm(yy100e, xx100e, t1);
plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
caxis ([-0.001 25.001]);
pos = get(gca, 'Position'); pos = pos + gpos(2,:); set(gca, 'Position', pos);
tp = title (tstr(3,1:tlen(3))); set(tp, 'Position', [0 1.6 0]);
cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos(1) = cbpos(1) + 0.05; cbpos(2) = cbpos(2) - 0.01; cbpos(3) = cbpos(3) - 0.1;
set(cb, 'Position', cbpos); set(gca, 'Position', pos);
set (cb, 'XTick', 0:5:25);
def_figure; oname = sprintf('FIGS/ukesm1_PD_obs_mod_anthro_STD_%s.png', numprint(3,2)); exportgraphics(gcf, oname, 'Resolution', 600);

tstr = char('Observed', 'Simulated', 'Simulated, CoV');
for f = 1:1:3, tlen(f) = lenstr(tstr(f,:)); end
figure(1); clf; turbo(60);
subplot(1,2,1);
ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
t0 = mean(pd_ant2, 3) / 1e3; t1 = t0; % t1 = log10(t0); t1(t1 < 1) = NaN;
pcolorm(yy100e, xx100e, t1);
plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
caxis ([-0.001 140.001]);
pos = get(gca, 'Position'); pos = pos + gpos(2,:); set(gca, 'Position', pos);
tp = title (tstr(2,1:tlen(2))); set(tp, 'Position', [0 1.6 0]);
cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos(1) = cbpos(1) + 0.05; cbpos(2) = cbpos(2) - 0.01; cbpos(3) = cbpos(3) - 0.1;
set(cb, 'Position', cbpos); set(gca, 'Position', pos);
set (cb, 'XTick', 0:20:140);
subplot(1,2,2);
ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
t0a = mean(pd_ant2, 3) / 1e3; t0b = std(pd_ant2, [], 3) / 1e3; t1 = t0b ./ t0a; % t1 = log10(t0); t1(t1 < 1) = NaN;
pcolorm(yy100e, xx100e, t1);
plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
caxis ([-0.001 5.001]);
pos = get(gca, 'Position'); pos = pos + gpos(2,:); set(gca, 'Position', pos);
tp = title (tstr(3,1:tlen(3))); set(tp, 'Position', [0 1.6 0]);
cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos(1) = cbpos(1) + 0.05; cbpos(2) = cbpos(2) - 0.01; cbpos(3) = cbpos(3) - 0.1;
set(cb, 'Position', cbpos); set(gca, 'Position', pos);
set (cb, 'XTick', 0:1:5);
def_figure; oname = sprintf('FIGS/ukesm1_PD_obs_mod_anthro_CoV_%s.png', numprint(3,2)); exportgraphics(gcf, oname, 'Resolution', 600);

% -----

% integrate up the model and observational fields to estimate total uptake
% of CO2, CFC11
t1 = grid_area([-90:1:90], [-180:1:180]);
area100 = t1(1:end-1,1:end-1) * 1e6;

t1 = area100 + woa_basin2(:,:,1);
t2 = nansum(t1(:))

t1 = area100e + nemo_mask100e(:,:,1);
t2 = nansum(t1(:))

t1 = obs_ant1 .* area100;
invt_obs_ant1 = nansum(t1(:));
t1 = obs_ant2 .* area100;
invt_obs_ant2 = nansum(t1(:));

for e = 1:1:9
t1 = pd_ant1(:,:,e) .* area100e;
invt_mod_ant1(e) = nansum(t1(:));
t1 = pd_ant2(:,:,e) .* area100e;
invt_mod_ant2(e) = nansum(t1(:));

t1 = pd_cfc11(:,:,e) .* area100e;
invt_mod_cfc11(e) = nansum(t1(:));
end
t1 = obs_cfc11 .* area100 / 1e9;
invt_obs_cfc11 = nansum(t1(:));

% BEFORE CORRECTION FOR KG -> M3
% GLODAP v1.1 =  94.7444 Pg C (1990s)
% GLODAP v2   = 159.8830 Pg C (2002)

% AFTER CORRECTION FOR KG -> M3
% GLODAP v1.1 =  96.7005 Pg C (1990s)
% GLODAP v2   = 164.4786 Pg C (2002)

% UKESM1      =  94.1042 Pg C (1990s)
% UKESM1      = 113.9633 Pg C (2000s)

% but need to account for different regions of missing data as well

t1 = obs_ant1 * 0; t1(isnan(t1)) = 1;
q1 = scatregrid(lat100, lon100, t1, yy100e, xx100e);
q1(q1 >= 0.5) = NaN; q1(q1 < 0.5) = 0;
msk1 = q1;

t1 = obs_ant2 * 0; t1(isnan(t1)) = 1;
q1 = scatregrid(lat100, lon100, t1, yy100e, xx100e);
q1(q1 >= 0.5) = NaN; q1(q1 < 0.5) = 0;
msk2 = q1;

for e = 1:1:9
t1 = (pd_ant1(:,:,e) + msk1) .* area100e;
invt_mod_ant1(e) = nansum(t1(:));
t1 = (pd_ant2(:,:,e) + msk2) .* area100e;
invt_mod_ant2(e) = nansum(t1(:));

t1 = (pd_cfc11(:,:,e) + msk1) .* area100e;
invt_mod_cfc11(e) = nansum(t1(:));
end

% AFTER CORRECTION FOR KG -> M3
% GLODAP v1.1 =  96.7005 Pg C (1990s)
% GLODAP v2   = 164.4786 Pg C (2002)

% AFTER CORRECTION FOR MASK
% UKESM1      =  84.7655 Pg C (1990s)
% UKESM1      = 112.2351 Pg C (2000s)

% CFCs        = 5.2927e+08
% UKESM1      = 5.3480e+08

t1 = obs_dic1 .* area100;
invt_obs_dic1 = nansum(t1(:));
t1 = obs_dic2 .* area100;
invt_obs_dic2 = nansum(t1(:));

for e = 1:1:9
t1 = (pd_dic1(:,:,e) + msk1) .* area100e;
invt_mod_dic1(e) = nansum(t1(:));
t1 = (pd_dic2(:,:,e) + msk2) .* area100e;
invt_mod_dic2(e) = nansum(t1(:));
end

odic1 =  invt_obs_dic1 * 1e-3 * 12.011 / 1e15;
odic2 =  invt_obs_dic2 * 1e-3 * 12.011 / 1e15;
mdic1 =  mean(invt_mod_dic1 * 1e-3 * 12.011 / 1e15);
mdic2 =  mean(invt_mod_dic2 * 1e-3 * 12.011 / 1e15);

ddic1 = 1e2 * (mdic1 - odic1) / odic1;
ddic2 = 1e2 * (mdic2 - odic2) / odic2;

% AFTER CORRECTION FOR KG -> M3
% AFTER CORRECTION FOR MASK
% GLODAP v1.1 =  35375 Pg C
% GLODAP v2   =  36912 Pg C

% UKESM1      =  35507 Pg C (1990s)  
% UKESM1      =  36696 Pg C (2000s)  

% So these numbers seem somewhat at odds with what the plots show; I
% suspect it's due in part to differences in ocean volume - the obs used
% stop at 5500m whereas the model continues down to 6000m; instead
% calculate the mean concentration of DIC in the model

t2 = gd1_totco2; % t2(isnan(t2)) = 0;
t3 = t2(:,:,1) * 0;
for k = 1:1:32
    q1 = ((t2(:,:,k) + t2(:,:,k+1)) / 2) .* area100;
    q2 = obs_dep(k+1) - obs_dep(k);
    q3 = q1 * q2; q3(q3 == 0) = NaN;
    q4 = isfinite(q3);
    q5 = sum(q3(q4));
    q6 = sum(area100(q4));
    oprof_dic1(k) = q5 / (q2 * q6);
    oprof_vol1(k) = (q2 * q6);
    oinvt_dic1(k) = q5;
end

t2 = gd1_anthro; % t2(isnan(t2)) = 0;
t3 = t2(:,:,1) * 0;
for k = 1:1:32
    q1 = ((t2(:,:,k) + t2(:,:,k+1)) / 2) .* area100;
    q2 = obs_dep(k+1) - obs_dep(k);
    q3 = q1 * q2; q3(q3 == 0) = NaN;
    q4 = isfinite(q3);
    q5 = sum(q3(q4));
    q6 = sum(area100(q4));
    oprof_ant1(k) = q5 / (q2 * q6);
    oinvt_ant1(k) = q5;
end

t2 = gd1_totalk; % t2(isnan(t2)) = 0;
t3 = t2(:,:,1) * 0;
for k = 1:1:32
    q1 = ((t2(:,:,k) + t2(:,:,k+1)) / 2) .* area100;
    q2 = obs_dep(k+1) - obs_dep(k);
    q3 = q1 * q2; q3(q3 == 0) = NaN;
    q4 = isfinite(q3);
    q5 = sum(q3(q4));
    q6 = sum(area100(q4));
    oprof_alk1(k) = q5 / (q2 * q6);
    oinvt_alk1(k) = q5;
end

t2 = gd1_cfc_11; % t2(isnan(t2)) = 0;
t3 = t2(:,:,1) * 0;
for k = 1:1:32
    q1 = ((t2(:,:,k) + t2(:,:,k+1)) / 2) .* area100;
    q2 = obs_dep(k+1) - obs_dep(k);
    q3 = q1 * q2; q3(q3 == 0) = NaN;
    q4 = isfinite(q3);
    q5 = sum(q3(q4));
    q6 = sum(area100(q4));
    oprof_cfc11(k) = q5 / (q2 * q6);
    oinvt_cfc11(k) = q5;
end

t2 = gd2_totco2; % t2(isnan(t2)) = 0;
t3 = t2(:,:,1) * 0;
for k = 1:1:32
    q1 = ((t2(:,:,k) + t2(:,:,k+1)) / 2) .* area100;
    q2 = obs_dep(k+1) - obs_dep(k);
    q3 = q1 * q2; q3(q3 == 0) = NaN;
    q4 = isfinite(q3);
    q5 = sum(q3(q4));
    q6 = sum(area100(q4));
    oprof_dic2(k) = q5 / (q2 * q6);
    oprof_vol2(k) = (q2 * q6);
    oinvt_dic2(k) = q5;
end

t2 = gd2_anthro; % t2(isnan(t2)) = 0;
t3 = t2(:,:,1) * 0;
for k = 1:1:32
    q1 = ((t2(:,:,k) + t2(:,:,k+1)) / 2) .* area100;
    q2 = obs_dep(k+1) - obs_dep(k);
    q3 = q1 * q2; q3(q3 == 0) = NaN;
    q4 = isfinite(q3);
    q5 = sum(q3(q4));
    q6 = sum(area100(q4));
    oprof_ant2(k) = q5 / (q2 * q6);
    oinvt_ant2(k) = q5;
end

t2 = gd2_totalk; % t2(isnan(t2)) = 0;
t3 = t2(:,:,1) * 0;
for k = 1:1:32
    q1 = ((t2(:,:,k) + t2(:,:,k+1)) / 2) .* area100;
    q2 = obs_dep(k+1) - obs_dep(k);
    q3 = q1 * q2; q3(q3 == 0) = NaN;
    q4 = isfinite(q3);
    q5 = sum(q3(q4));
    q6 = sum(area100(q4));
    oprof_alk2(k) = q5 / (q2 * q6);
    oinvt_alk2(k) = q5;
end

clear pi_dic1 pi_dic2 pi_alk1 pi_alk2 pd_dic1 pd_dic2 pd_alk1 pd_alk2 pd_cfc11 pd_vol1 pd_vol2;

for e = 1:1:tsize
  % PIcontrol data
  dnom = sprintf('%s/mean_medusa_aw310o_*ptrc-T.nc', dname(e).name);
  t1 = dir(dnom);
  nom1990s = sprintf('%s/%s', dname(e).name, t1(1).name);
  nom2000s = sprintf('%s/%s', dname(e).name, t1(2).name);
  %
  fprintf('- Reading %s\n', nom1990s);
  t1 = ncread(nom1990s, 'DIC');
  t2 = permute(t1, [2 1 3]);
  pi_dic1(:,:,:,e) = t2; 
  fprintf('- Reading %s\n', nom2000s);
  t1 = ncread(nom1990s, 'DIC');
  t2 = permute(t1, [2 1 3]);
  pi_dic2(:,:,:,e) = t2;
  
  % PD data
  pnom = sprintf('%s/mean_medusa_%so_1y_19901201-20001201_ptrc-T.nc', dname(e).name, rname(e).name);
  fprintf('- Reading %s\n', pnom);
  t1 = ncread(pnom, 'DIC');
  t2 = permute(t1, [2 1 3]);
  pd_dic1(:,:,:,e) = t2;
  t1 = ncread(pnom, 'ALK');
  t2 = permute(t1, [2 1 3]);
  pd_alk1(:,:,:,e) = t2;
  t1 = ncread(pnom, 'CFC11');
  t2 = permute(t1, [2 1 3]);
  pd_cfc11(:,:,:,e) = t2;
  %
  pnom = sprintf('%s/mean_medusa_%so_1y_20001201-20101201_ptrc-T.nc', dname(e).name, rname(e).name);
  fprintf('- Reading %s\n', pnom);
  t1 = ncread(pnom, 'DIC');
  t2 = permute(t1, [2 1 3]);
  pd_dic2(:,:,:,e) = t2;
  t1 = ncread(pnom, 'ALK');
  t2 = permute(t1, [2 1 3]);
  pd_alk2(:,:,:,e) = t2;
end

pd_ant1 = pd_dic1 - pi_dic1;
pd_ant2 = pd_dic2 - pi_dic2;

fname = sprintf('%s/mean_nemo_%so_1y_20001201-20101201_grid-T.nc', dname(1).name, rname(1).name);
t1 = ncread(fname, 'deptht_bounds');
nemo_dep = double(t1(1,:)); nemo_dep(end+1) = double(t1(2,end));
t1 = ncread(fname, 'deptht');
nemo_mid = double(t1');

zzthk = nemo_dep(2:end) - nemo_dep(1:end-1);

for e = 1:1:9
for k = 1:1:75
  t1 = pd_dic1(:,:,k,e); t2 = t1 .* area100e; t3 = isfinite(t1);
  t4 = sum(t2(t3)); t5 = sum(area100e(t3));
  mprof_dic1(k,e) = t4 / t5;
  mprof_vol1(k,e) = t5 * zzthk(k);
  minvt_dic1(k,e) = t4 * zzthk(k);
  t1 = pd_alk1(:,:,k,e); t2 = t1 .* area100e; t3 = isfinite(t1);
  t4 = sum(t2(t3)); t5 = sum(area100e(t3));
  mprof_alk1(k,e) = t4 / t5;
  minvt_alk1(k,e) = t4 * zzthk(k);
  t1 = pd_ant1(:,:,k,e); t2 = t1 .* area100e; t3 = isfinite(t1);
  t4 = sum(t2(t3)); t5 = sum(area100e(t3));
  mprof_ant1(k,e) = t4 / t5;
  minvt_ant1(k,e) = t4 * zzthk(k);
  t1 = pd_dic2(:,:,k,e); t2 = t1 .* area100e; t3 = isfinite(t1);
  t4 = sum(t2(t3)); t5 = sum(area100e(t3));
  mprof_dic2(k,e) = t4 / t5;
  mprof_vol2(k,e) = t5 * zzthk(k);
  minvt_dic2(k,e) = t4 * zzthk(k);
  t1 = pd_alk2(:,:,k,e); t2 = t1 .* area100e; t3 = isfinite(t1);
  t4 = sum(t2(t3)); t5 = sum(area100e(t3));
  mprof_alk2(k,e) = t4 / t5;
  minvt_alk2(k,e) = t4 * zzthk(k);
  t1 = pd_ant2(:,:,k,e); t2 = t1 .* area100e; t3 = isfinite(t1);
  t4 = sum(t2(t3)); t5 = sum(area100e(t3));
  mprof_ant2(k,e) = t4 / t5;
  minvt_ant2(k,e) = t4 * zzthk(k);
  t1 = pd_cfc11(:,:,k,e); t2 = t1 .* area100e; t3 = isfinite(t1);
  t4 = sum(t2(t3)); t5 = sum(area100e(t3));
  mprof_cfc11(k,e) = t4 / t5;
  minvt_cfc11(k,e) = t4 * zzthk(k);
end
end

% total ocean means

t1 = sum(oinvt_dic1) ./ sum(oprof_vol1); tobs_dic1 = t1;
t1 = sum(oinvt_dic2) ./ sum(oprof_vol2); tobs_dic2 = t1;

t1 = sum(oinvt_alk1) ./ sum(oprof_vol1); tobs_alk1 = t1;
t1 = sum(oinvt_alk2) ./ sum(oprof_vol2); tobs_alk2 = t1;

t1 = sum(oinvt_ant1) ./ sum(oprof_vol1); tobs_ant1 = t1;
t1 = sum(oinvt_ant2) ./ sum(oprof_vol2); tobs_ant2 = t1;

t1 = sum(oinvt_cfc11) ./ sum(oprof_vol1); tobs_cfc11 = t1;

t1 = sum(minvt_dic1,1) ./ sum(mprof_vol1,1); tmod_dic1 = mean(t1);
t1 = sum(minvt_dic2,1) ./ sum(mprof_vol2,1); tmod_dic2 = mean(t1);

t1 = sum(minvt_alk1,1) ./ sum(mprof_vol1,1); tmod_alk1 = mean(t1);
t1 = sum(minvt_alk2,1) ./ sum(mprof_vol2,1); tmod_alk2 = mean(t1);

t1 = sum(minvt_ant1,1) ./ sum(mprof_vol1,1); tmod_ant1 = mean(t1);
t1 = sum(minvt_ant2,1) ./ sum(mprof_vol2,1); tmod_ant2 = mean(t1);

t1 = sum(minvt_cfc11,1) ./ sum(mprof_vol1,1); tmod_cfc11 = mean(t1);

%

obs_mid = (obs_dep(2:end) + obs_dep(1:end-1)) / 2;
yt = [5 10 20 50 100 200 500 1000 2000 5000]; lyt = log10(yt);

figure(1); clf
h = plot (oprof_dic1, obs_mid, 'r-', mprof_dic1, nemo_mid, 'b-');
set(h(1), 'LineWidth', 2);
xlabel ('DIC [mmol m^{-3}]'); ylabel ('Depth [m]');
set (gca, 'YDir', 'reverse', 'YScale', 'log');
set (gca, 'YTick', yt, 'YMinorTick', 'off');
yaxis ([5 6000]);
lg = legend('Observations', 'Model ensemble', 'Location', 'NorthEast');
lgpos = get(lg, 'Position'); lgpos(1) = lgpos(1) - 0.08; lgpos(3) = lgpos(3) + 0.08; set (lg, 'Position', lgpos);
paper; orient portrait; def_figure; 
oname = sprintf('FIGS/ukesm1_BGC_profiles_%s.png', numprint(1,2)); exportgraphics(gcf, oname, 'Resolution', 600);

figure(1); clf
h = plot (oprof_alk1, obs_mid, 'r-', mprof_alk1, nemo_mid, 'b-');
set(h(1), 'LineWidth', 2);
xlabel ('Alkalinity [meq m^{-3}]'); ylabel ('Depth [m]');
set (gca, 'YDir', 'reverse', 'YScale', 'log');
set (gca, 'YTick', yt, 'YMinorTick', 'off');
yaxis ([5 6000]);
lg = legend('Observations', 'Model ensemble', 'Location', 'NorthEast');
lgpos = get(lg, 'Position'); lgpos(1) = lgpos(1) - 0.08; lgpos(3) = lgpos(3) + 0.08; set (lg, 'Position', lgpos);
paper; orient portrait; def_figure; 
oname = sprintf('FIGS/ukesm1_BGC_profiles_%s.png', numprint(2,2)); exportgraphics(gcf, oname, 'Resolution', 600);

figure(1); clf
h = plot (oprof_ant1, obs_mid, 'r-', mprof_ant1, nemo_mid, 'b-');
set(h(1), 'LineWidth', 2);
xlabel ('Anthropogenic CO_{2} [mmol m^{-3}]'); ylabel ('Depth [m]');
set (gca, 'YDir', 'reverse', 'YScale', 'log');
set (gca, 'YTick', yt, 'YMinorTick', 'off');
yaxis ([5 6000]);
lg = legend('Observations', 'Model ensemble', 'Location', 'NorthWest');
lgpos = get(lg, 'Position'); lgpos(3) = lgpos(3) + 0.08; set (lg, 'Position', lgpos);
paper; orient portrait; def_figure; 
oname = sprintf('FIGS/ukesm1_BGC_profiles_%s.png', numprint(3,2)); exportgraphics(gcf, oname, 'Resolution', 600);

figure(1); clf
h = plot (oprof_cfc11, obs_mid, 'r-', mprof_cfc11 * 1e9, nemo_mid, 'b-');
set(h(1), 'LineWidth', 2);
xlabel ('CFC11 [nmol m^{-3}]'); ylabel ('Depth [m]');
set (gca, 'YDir', 'reverse', 'YScale', 'log');
set (gca, 'YTick', yt, 'YMinorTick', 'off');
yaxis ([5 6000]);
lg = legend('Observations', 'Model ensemble', 'Location', 'NorthWest');
lgpos = get(lg, 'Position'); lgpos(3) = lgpos(3) + 0.08; set (lg, 'Position', lgpos);
paper; orient portrait; def_figure; 
oname = sprintf('FIGS/ukesm1_BGC_profiles_%s.png', numprint(4,2)); exportgraphics(gcf, oname, 'Resolution', 600);

figure(1); clf
h = plot (oprof_dic2, obs_mid, 'r-', mprof_dic2, nemo_mid, 'b-');
set(h(1), 'LineWidth', 2);
xlabel ('DIC [mmol m^{-3}]'); ylabel ('Depth [m]');
set (gca, 'YDir', 'reverse', 'YScale', 'log');
set (gca, 'YTick', yt, 'YMinorTick', 'off');
yaxis ([5 6000]);
lg = legend('Observations', 'Model ensemble', 'Location', 'NorthEast');
lgpos = get(lg, 'Position'); lgpos(1) = lgpos(1) - 0.08; lgpos(3) = lgpos(3) + 0.08; set (lg, 'Position', lgpos);
paper; orient portrait; def_figure; 
oname = sprintf('FIGS/ukesm1_BGC_profiles_%s.png', numprint(5,2)); exportgraphics(gcf, oname, 'Resolution', 600);

figure(1); clf
h = plot (oprof_alk2, obs_mid, 'r-', mprof_alk2, nemo_mid, 'b-');
set(h(1), 'LineWidth', 2);
xlabel ('Alkalinity [meq m^{-3}]'); ylabel ('Depth [m]');
set (gca, 'YDir', 'reverse', 'YScale', 'log');
set (gca, 'YTick', yt, 'YMinorTick', 'off');
yaxis ([5 6000]);
lg = legend('Observations', 'Model ensemble', 'Location', 'NorthEast');
lgpos = get(lg, 'Position'); lgpos(1) = lgpos(1) - 0.08; lgpos(3) = lgpos(3) + 0.08; set (lg, 'Position', lgpos);
paper; orient portrait; def_figure; 
oname = sprintf('FIGS/ukesm1_BGC_profiles_%s.png', numprint(6,2)); exportgraphics(gcf, oname, 'Resolution', 600);

figure(1); clf
h = plot (oprof_ant2, obs_mid, 'r-', mprof_ant2, nemo_mid, 'b-');
set(h(1), 'LineWidth', 2);
xlabel ('Anthropogenic CO_{2} [mmol m^{-3}]'); ylabel ('Depth [m]');
set (gca, 'YDir', 'reverse', 'YScale', 'log');
set (gca, 'YTick', yt, 'YMinorTick', 'off');
yaxis ([5 6000]);
lg = legend('Observations', 'Model ensemble', 'Location', 'NorthWest');
lgpos = get(lg, 'Position'); lgpos(3) = lgpos(3) + 0.08; set (lg, 'Position', lgpos);
paper; orient portrait; def_figure; 
oname = sprintf('FIGS/ukesm1_BGC_profiles_%s.png', numprint(7,2)); exportgraphics(gcf, oname, 'Resolution', 600);

% ----------------------------------------------------------------------
% ----------------------------------------------------------------------
% ----------------------------------------------------------------------

t1 = load('uptake_data_for_GFN_2014.txt');
obs_t = t1(:,1);
obs_flx = t1(:,2);
obs_flxmin = t1(:,3);
obs_flxmax = t1(:,4);

jobnom = char('aw310', 'bb277');

for j = 1:1:2
  jnom = sprintf('PD_timeseries/u-%s_TotalAirSeaFluxCO2.shelve.txt', jobnom(j,:));
  t1 = load (jnom);
  evalstr = sprintf('mod_t.j%d = t1(:,1);', j); eval(evalstr);
  evalstr = sprintf('mod_flx.j%d = t1(:,2);', j); eval(evalstr);
end

% u-aw310 data is now from the period 2395 to 2555, i.e. 545 years later
mod_t.j1 = mod_t.j1 - 545;

xdat = [obs_t; flipdim(obs_t, 1)];
ydat = [obs_flxmin; flipdim(obs_flxmax, 1)];

jcol(1,:) = [0 0 0];
t1 = turbo(9); jcol(2,:) = [1 0.25 0.25];

figure(1); clf; hold on;
ho = plot (obs_t, obs_flx, 'k-');
set (ho, 'LineWidth', 2);
for j = 1:1:2
  evalstr = sprintf('h = plot (mod_t.j%d, mod_flx.j%d, ''-'');', j, j);
  eval(evalstr);
  set (h, 'Color', jcol(j,:));
  set (h, 'LineWidth', 2);
  if j == 1, set (h, 'LineStyle', '--'); end
end
hf = fill (xdat, ydat, [0.6 0.6 0.6]);
set (hf, 'EdgeColor', [1 1 1]);
ho = plot (obs_t, obs_flx, 'k-');
set (ho, 'LineWidth', 2);
for j = 1:1:2
  evalstr = sprintf('h = plot (mod_t.j%d, mod_flx.j%d, ''-'');', j, j);
  eval(evalstr);
  set (h, 'Color', jcol(j,:));
  set (h, 'LineWidth', 2);
  if j == 1, set (h, 'LineStyle', '--'); end
end
plot (obs_t, obs_t * 0, 'k:');
axis ([1850 2013 -0.5 3.5]);
lg = legend ('Estimated', 'UKESM1 ctrl', 'UKESM1 hist 1', 'Location', 'NorthWest');
lgpos = get(lg, 'Position'); lgpos(3) = lgpos(3) + 0.05; set(lg, 'Position', lgpos);
ylabel ('Air-sea CO_2 flux [Pg C y^{-1}]');
xlabel ('Time [year]');
topbox;
paper; orient portrait; def_figure; 
oname = sprintf('FIGS/ukesm1_co2flux_history.png'); exportgraphics(gcf, oname, 'Resolution', 600);

t1 = obs_t(1)
t1 = obs_t(87)
t1 = obs_t(238)
t1 = obs_t(245)
t1 = mod_t.j1(1)
t1 = mod_t.j1(158)

t1 = obs_flx(1:86);
sum(t1)
t1 = obs_flx(1:245);
sum(t1)
t1 = obs_flx(1:238);
sum(t1)
t1 = obs_flx(87:238);
sum(t1)
t1 = obs_flx(87:245);
sum(t1)
% t1 = mod_flx.j1(1:159);
% sum(t1)
t1 = mod_flx.j1(546:704);
sum(t1)
t1 = mod_flx.j2(1:159);
sum(t1)

% CO2 flux estimates
% obs flx =  12.8953 Pg C (1765 to end-1850)
% obs flx = 149.1607 Pg C (1765 to end-2009)
% obs flx = 132.3472 Pg C (1765 to end-2002)
% obs flx = 119.4518 Pg C (1851 to end-2002)
% obs flx = 136.2654 Pg C (1851 to end-2009)
% control =  -4.9327 Pg C (1851 to end-2009)
% history = 122.1733 Pg C (1851 to end-2009) u-aw331
% control =  -1.5485 Pg C (1851 to end-2009) later control period
% history = 121.8111 Pg C (1851 to end-2009) u-bb277

% CO2 inventory estimates
% GLODAP v1.1 =  94.7444 Pg C (1990s)
% GLODAP v2   = 159.8830 Pg C (2002)
% UKESM1      =  95.8602 Pg C (1990s)
% UKESM1      = 112.1639 Pg C (2000s)

% CFC11 inventory estimates
% GLODAP v1.1 = 5.2927e+08 (1990s)
% UKESM1      = 5.9814e+08 (199Os)

