clear; startup;

% =========================================================================
% =========================================================================
%
% Matlab plotting scripts for the analysis of UKESM1 prepared for:
%
%   Yool, A., Palmiéri, J., Jones, C. G., de Mora, L., Kuhlbrodt, T., 
%   Popova, E. E., Nurser, A. J. G., Hirschi, J., Blaker, A. T., 
%	Coward, A. C., Blockley, E. W., and Sellar, A. A.: Evaluating the 
% 	physical and biogeochemical state of the global ocean component of 
%	UKESM1 in CMIP6 Historical simulations, Geosci. Model Dev. Discuss. 
%	[preprint], https://doi.org/10.5194/gmd-2020-333, in review, 2020.
%
%   Accepted for publication: 20 April 2021
%
% Note: scripts provided to document analysis, and may require editing to
% fully replicate plots in manuscript
%
% =========================================================================
% =========================================================================

% =========================================================================
%
% This script plots Figure 10, 11, 12, 13, 14, and Supplementary Figure S11
%
% =========================================================================
% =========================================================================

load ukesm1_PD_ensemble

% =========================================================================

% load up background files
load nemo_area100e
load nemo_mask100e
fname = sprintf('%s/mean_nemo_%so_1y_20001201-20101201_grid-T.nc', dname(1).name, rname(1).name);
t1 = ncread(fname, 'deptht_bounds');
nemo_dep = double(t1(1,:)); nemo_dep(end+1) = double(t1(2,end));
t1 = ncread(fname, 'deptht');
nemo_mid = double(t1');

% total grid area
t1 = area100e + nemo_mask100e(:,:,1);
t2 = isfinite(t1);
tot_area100e = sum(t1(t2)); % m2

% extra mask to get rid of Mediterranean, Black and Caspian Seas
t1 = nemo_mask100e(:,:,1) + 1;
t1(isnan(t1)) = 0;
t1(235:249,283:344) = t1(235:249,283:344) + 2;
t1(249:262,291:344) = t1(249:262,291:344) + 4;
t1(262:276,330:344) = t1(262:276,330:344) + 8;
t2 = t1;
t2(t1 < 1) = NaN; t2(t1 > 1) = NaN;
med_mask100e = t2;

nxx = -179.5:1:179.5;  nyy = -89.5:1:89.5;
[lon100, lat100] = meshgrid(nxx, nyy);
nxx = -180:1:180;  nyy = -90:1:90;
[lon100b, lat100b] = meshgrid(nxx, nyy);

load woa_mask
clear t1; t1(:,1:180) = woa_mask(:,181:360); t1(:,181:360) = woa_mask(:,1:180);
woa_mask = t1;

load woa_basin

helvetica;
pngopt = 0;

% =========================================================================

% Aeolian dust

for e = 1:1:tsize
  fnom = sprintf('%s/mean_medusa_%so_1y_20001201-20101201_diad-T.nc', dname(e).name, rname(e).name);
  fprintf('- Reading %s\n', fnom);
  t1 = ncread(fnom, 'AEOLIAN');
  ens_mod_dep(:,:,e) = t1'; % mmol Fe / m2 / d
  t1 = ncread(fnom, 'BENTHIC');
  ens_mod_ben(:,:,e) = t1'; % mmol Fe / m2 / d
end

% t1 = ncread('u-bb277/mean_medusa_bb277o_1y_20001201-20101201_diad-T.nc', 'AEOLIAN');
% mod_dep = t1'; % mmol Fe / m2 / d
% t1 = ncread('u-bb277/mean_medusa_bb277o_1y_20001201-20101201_diad-T.nc', 'BENTHIC');
% mod_ben = t1'; % mmol Fe / m2 / d

mod_dep = mean(ens_mod_dep, 3); mod_dep_std = std(ens_mod_dep, [], 3); mod_dep_cov = mod_dep_std ./ mod_dep;

t1 = mod_dep .* area100e + med_mask100e;
t2 = isfinite(t1);
t3 = sum(t1(t2));
% mmol Fe / m2 / d -> Gmol Fe / y
tot_mod_dep = t3 * 360 / 1e3 / 1e9; % Gmol Fe / y

t1 = ncread('OBS/mahowald.orca100_ext.nc', 'dust'); % kg dust / m2 / s
t2 = mean(permute(double(t1), [2 1 3]), 3);
% kg dust / m2 / s -> mmol Fe / m2 / d
scl = (60 * 60 * 24) * 0.035 * 1e3 * 1e3 / 55.845;
t3 = t2 * scl;
t4 = t3 .* area100e + med_mask100e;
t5 = isfinite(t4);
t6 = sum(t4(t5));
t7 = t6 * 360 / 1e3 / 1e9;
wanttot = 2.6e9;
dust_scl = wanttot / (t7 * 1e9);
obs_dep = t3 * dust_scl;
t1 = obs_dep .* area100e + med_mask100e;
t2 = isfinite(t1);
t3 = sum(t1(t2));
% mmol Fe / m2 / d -> Gmol Fe / y
tot_obs_dep = t3 * 360 / 1e3 / 1e9; % Gmol Fe / y

lon_rng = [-180 180]; lat_rng = [-90 90];
xt = [0.1 0.2 0.5 1 2 5 10 20 50 100]; lxt = log10(xt);

% set(0, 'DefaultFigureRenderer', 'painters');

figure(1); clf; turbo(60);
subplot (2,1,1);
t1 = (obs_dep * 360 * 1e3) + nemo_mask100e(:,:,1);
ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
pcolorm(yy100e, xx100e, log10(t1));
load coast; plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
caxis ([lxt(1) lxt(end)]);
tt = title ('Mahowald (2005)'); 
pos = [0.05 0.475 0.75 0.475]; set(gca, 'Position', pos);
ttpos = get(tt, 'Position'); ttpos(2) = ttpos(2) * 0.8; set (tt, 'Position', ttpos);
subplot (2,1,2);
t1 = (mod_dep * 360 * 1e3) + nemo_mask100e(:,:,1);
ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
pcolorm(yy100e, xx100e, log10(t1));
load coast; plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
caxis ([lxt(1) lxt(end)]);
tt = title ('UKESM1');
pos = [0.05 0.025 0.75 0.475]; set(gca, 'Position', pos);
ttpos = get(tt, 'Position'); ttpos(2) = ttpos(2) * 0.8; set (tt, 'Position', ttpos);
pos = get(gca, 'Position'); cb = colorbar; cbpos = get(cb, 'Position');
set (cb, 'YTick', lxt, 'YTickLabel', xt);
cbar_ylabel(cb, 'Aeolian iron [{\mu}mol m^{-2} y^{-1}]');
cbpos(1) = pos(1) + pos(3);
set(cb, 'Position', cbpos); set(gca, 'Position', pos);
%
paper; orient portrait; def_figure;
fnom0 = char('FIGS/ukesm1_PD_iron_deposition.png'); exportgraphics(gcf, fnom0, 'Resolution', 600);

figure(1); clf; turbo(60);
subplot (2,1,1);
t1 = (mod_dep * 360 * 1e3) + nemo_mask100e(:,:,1);
ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
pcolorm(yy100e, xx100e, log10(t1));
load coast; plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
caxis ([lxt(1) lxt(end)]);
title ('UKESM1');
pos = [0.05 0.5 0.75 0.5]; set(gca, 'Position', pos);
pos = get(gca, 'Position'); cb = colorbar; cbpos = get(cb, 'Position');
set (cb, 'YTick', lxt, 'YTickLabel', xt);
cbar_ylabel(cb, 'Aeolian iron [{\mu}mol m^{-2} y^{-1}]');
cbpos(1) = pos(1) + pos(3);
set(cb, 'Position', cbpos); set(gca, 'Position', pos);
subplot (2,1,2);
t1 = (mod_dep_cov) + nemo_mask100e(:,:,1);
ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
pcolorm(yy100e, xx100e, t1);
load coast; plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
caxis ([0 0.5]);
title ('UKESM1, CoV');
pos = [0.05 0.025 0.75 0.5]; set(gca, 'Position', pos);
pos = get(gca, 'Position'); cb = colorbar; cbpos = get(cb, 'Position');
% set (cb, 'YTick', lxt, 'YTickLabel', xt);
cbar_ylabel(cb, 'Aeolian iron CoV [-]');
cbpos(1) = pos(1) + pos(3);
set(cb, 'Position', cbpos); set(gca, 'Position', pos);
%
paper; orient portrait; def_figure;
set(gcf, 'PaperPosition', [2.1217 6.0763 16.7340 17.5506]);
fnom0 = char('ukesm1_PD_iron_deposition_CoV.png'); exportgraphics(gcf, fnom0, 'Resolution', 600);

% A4 = 210 × 297 mm

figure(1); clf; turbo(60);
t1 = (obs_dep * 360 * 1e3) + nemo_mask100e(:,:,1);
ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
pcolorm(yy100e, xx100e, log10(t1));
load coast; plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
caxis ([lxt(1) lxt(end)]);
title ('Mahowald (2005)');
pos = [0.05 0.10 0.75 0.80]; set(gca, 'Position', pos);
cb = colorbar; cbpos = get(cb, 'Position'); set (cb, 'YTick', lxt, 'YTickLabel', xt);
cbar_ylabel(cb, 'Aeolian iron [{\mu}mol m^{-2} y^{-1}]'); cbpos(1) = pos(1) + pos(3);
set(cb, 'Position', cbpos); set(gca, 'Position', pos); set (cb, 'Visible', 'off');
paper; orient tall; def_figure;
fnom0 = char('ukesm1_PD_iron_deposition_CoV_panel1.png'); exportgraphics(gcf, fnom0, 'Resolution', 600); 

figure(1); clf; turbo(60);
t1 = (mod_dep * 360 * 1e3) + nemo_mask100e(:,:,1);
ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
pcolorm(yy100e, xx100e, log10(t1));
load coast; plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
caxis ([lxt(1) lxt(end)]);
title ('UKESM1');
pos = [0.05 0.10 0.75 0.80]; set(gca, 'Position', pos);
cb = colorbar; cbpos = get(cb, 'Position'); set (cb, 'YTick', lxt, 'YTickLabel', xt);
cbar_ylabel(cb, 'Aeolian iron [{\mu}mol m^{-2} y^{-1}]'); cbpos(1) = pos(1) + pos(3);
set(cb, 'Position', cbpos); set(gca, 'Position', pos);
paper; orient tall; def_figure;
fnom0 = char('ukesm1_PD_iron_deposition_CoV_panel2.png'); exportgraphics(gcf, fnom0, 'Resolution', 600); 

figure(1); clf; turbo(60);
t1 = (mod_dep_cov) + nemo_mask100e(:,:,1);
ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
pcolorm(yy100e, xx100e, t1);
load coast; plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
caxis ([0 0.5]);
title ('UKESM1, CoV');
pos = [0.05 0.10 0.75 0.80]; set(gca, 'Position', pos);
cb = colorbar; cbpos = get(cb, 'Position'); set (cb, 'YTick', lxt, 'YTickLabel', xt);
cbar_ylabel(cb, 'Aeolian iron [{\mu}mol m^{-2} y^{-1}]'); cbpos(1) = pos(1) + pos(3);
set(cb, 'Position', cbpos); set(gca, 'Position', pos);
paper; orient tall; def_figure;
fnom0 = char('ukesm1_PD_iron_deposition_CoV_panel3.png'); exportgraphics(gcf, fnom0, 'Resolution', 600);

% =========================================================================

% fields of DIN, SIL, CHL, TPP, DIC, ALK, pH, omega calcite, CCD, delta
% pCO2, air-sea flux, surface oxygen, Pn, Pd, PRN, PRD, ML_TPP,
% seafloor flux, OPAL, FASTCA, Zmi, Zme, SLOW, FAST, SLOW @ 100m, FAST @
% 100m
surf1 = [1 1 1 1]; surf2 = [362 332 1 1];

for e = 1:1:tsize
  for m = 12:-1:1
    fprintf('- Month %d\n', m);
    pnom = sprintf('%s/clim_medusa_%so_1m_2000%s-2009%s_ptrc-T.nc', dname(e).name, rname(e).name, numprint(m, 2), numprint(m, 2));
    dnom = sprintf('%s/clim_medusa_%so_1m_2000%s-2009%s_diad-T.nc', dname(e).name, rname(e).name, numprint(m, 2), numprint(m, 2));
    fprintf('- Reading %s\n', pnom);
    % DIN
    t1 = ncread(pnom, 'DIN', surf1, surf2); t2 = permute(t1, [2 1]); mod_din_ens(:,:,m,e) = t2;
    % SIL
    t1 = ncread(pnom, 'SIL', surf1, surf2); t2 = permute(t1, [2 1]); mod_sil_ens(:,:,m,e) = t2;
    % FER
    t1 = ncread(pnom, 'FER', surf1, surf2); t2 = permute(t1, [2 1]); mod_fer_ens(:,:,m,e) = t2;
    % CHL
    t1 = ncread(pnom, 'CHN', surf1, surf2); t2 = ncread(pnom, 'CHD', surf1, surf2);
    t3 = permute(t1+t2, [2 1]); mod_chl_ens(:,:,m,e) = t3;
    % TPP
    t1 = ncread(dnom, 'PRN'); t2 = ncread(dnom, 'PRD');
    t3 = permute(t1+t2, [2 1]) * 6.625 * 12.011 * 1e-3; mod_tpp_ens(:,:,m,e) = t3;
    mod_prn_ens(:,:,m,e) = permute(t1, [2 1]) * 6.625 * 12.011 * 1e-3;
    mod_prd_ens(:,:,m,e) = permute(t2, [2 1]) * 6.625 * 12.011 * 1e-3;
    % DIC
    t1 = ncread(pnom, 'DIC', surf1, surf2); t2 = permute(t1, [2 1]); mod_dic_ens(:,:,m,e) = t2;
    % ALK
    t1 = ncread(pnom, 'ALK', surf1, surf2); t2 = permute(t1, [2 1]); mod_alk_ens(:,:,m,e) = t2;
    % pH
    t1 = ncread(dnom, 'OCN_PH'); t2 = permute(t1, [2 1]); mod_ph_ens(:,:,m,e) = t2;
    % omega calcite
    t1 = ncread(dnom, 'OM_CAL'); t2 = permute(t1, [2 1]); mod_cal_ens(:,:,m,e) = t2;
    % CCD
    t1 = ncread(dnom, 'CAL_CCD'); t2 = permute(t1, [2 1]); mod_ccd_ens(:,:,m,e) = t2;
    % delta pCO2
    t1 = ncread(dnom, 'OCN_DPCO2'); t2 = permute(t1, [2 1]); mod_dpc_ens(:,:,m,e) = t2;
    % air-sea flux
    t1 = ncread(dnom, 'CO2FLUX'); t2 = permute(t1, [2 1]); mod_flx_ens(:,:,m,e) = t2;
    % surface O2
    t1 = ncread(pnom, 'OXY', surf1, surf2); t2 = permute(t1, [2 1]); mod_oxy_ens(:,:,m,e) = t2;
    % Pn
    t1 = ncread(pnom, 'PHN', surf1, surf2); t2 = permute(t1, [2 1]); mod_phn_ens(:,:,m,e) = t2;
    % Pd
    t1 = ncread(pnom, 'PHD', surf1, surf2); t2 = permute(t1, [2 1]); mod_phd_ens(:,:,m,e) = t2;
    t1 = ncread(pnom, 'PDS', surf1, surf2); t2 = permute(t1, [2 1]); mod_pds_ens(:,:,m,e) = t2;
    % ML TPP
    t1 = ncread(dnom, 'ML_PRN'); t2 = ncread(dnom, 'ML_PRD');
    t3 = permute(t1+t2, [2 1]) * 6.625 * 12.011 * 1e-3; mod_mlp_ens(:,:,m,e) = t3;
    % seafloor flux
    t1 = ncread(dnom, 'SEAFLRC'); t2 = permute(t1, [2 1]); mod_sfr_ens(:,:,m,e) = t2;
    % OPAL
    t1 = ncread(dnom, 'OPAL'); t2 = permute(t1, [2 1]); mod_opl_ens(:,:,m,e) = t2;
    % FASTCA
    t1 = ncread(dnom, 'FASTCA'); t2 = permute(t1, [2 1]); mod_cal_ens(:,:,m,e) = t2;
    % Zmi
    t1 = ncread(pnom, 'ZMI', surf1, surf2); t2 = permute(t1, [2 1]); mod_zmi_ens(:,:,m,e) = t2;
    % Zme
    t1 = ncread(pnom, 'ZME', surf1, surf2); t2 = permute(t1, [2 1]); mod_zme_ens(:,:,m,e) = t2;
    % DETC
    t1 = ncread(dnom, 'DETC'); t2 = permute(t1, [2 1]); mod_scp_ens(:,:,m,e) = t2;
    % FASTC
    t1 = ncread(dnom, 'FASTC'); t2 = permute(t1, [2 1]); mod_fcp_ens(:,:,m,e) = t2;
    % DETC @ 100m
    t1 = ncread(dnom, 'SDC__100'); t2 = permute(t1, [2 1]); mod_s100_ens(:,:,m,e) = t2;
    % FASTC @ 100m
    t1 = ncread(dnom, 'FASTN'); t2 = permute(t1, [2 1]);
    t3 = ncread(dnom, 'FASTC'); t4 = permute(t3, [2 1]);
    t5 = ncread(dnom, 'FDT__100'); t6 = permute(t5, [2 1]);
    mod_f100_ens(:,:,m,e) = (t4 ./ t2) .* t6;
    % PN LIM
    t1 = ncread(dnom, 'PN_JLIM'); t2 = permute(t1, [2 1]); mod_pnjlim_ens(:,:,m,e) = t2;
    t1 = ncread(dnom, 'PN_NLIM'); t2 = permute(t1, [2 1]); mod_pnnlim_ens(:,:,m,e) = t2;
    t1 = ncread(dnom, 'PN_FELIM'); t2 = permute(t1, [2 1]); mod_pnflim_ens(:,:,m,e) = t2;
    % PD LIM
    t1 = ncread(dnom, 'PD_JLIM'); t2 = permute(t1, [2 1]); mod_pdjlim_ens(:,:,m,e) = t2;
    t1 = ncread(dnom, 'PD_NLIM'); t2 = permute(t1, [2 1]); mod_pdnlim_ens(:,:,m,e) = t2;
    t1 = ncread(dnom, 'PD_FELIM'); t2 = permute(t1, [2 1]); mod_pdflim_ens(:,:,m,e) = t2;
    t1 = ncread(dnom, 'PDSILIM2'); t2 = permute(t1, [2 1]); mod_pdslim_ens(:,:,m,e) = t2;
  end
end

vnom = char('din', 'sil', 'fer', 'chl', 'tpp', 'prn', 'prd', 'dic', 'alk', 'ph', ...
  'cal', 'ccd', 'dpc', 'flx', 'oxy', 'phn', 'phd', 'pds', 'mlp', 'sfr', ...
  'opl', 'cal', 'zmi', 'zme', 'scp', 'fcp', 's100', 'f100', 'pnjlim', 'pnnlim', ...
  'pnflim', 'pdjlim', 'pdnlim', 'pdflim', 'pdslim');
[vtot, q1] = size(vnom);
for v = 1:1:vtot
  vlen(v) = lenstr(vnom(v,:));
end

% annual averages for ensemble
for v = 1:1:vtot
  evalstr = sprintf('t1 = mod_%s_ens;', vnom(v,1:vlen(v))); eval(evalstr);
  t2 = mean(t1, 3);
  evalstr = sprintf('mod_%s_ens(:,:,13,:) = t2;', vnom(v,1:vlen(v))); eval(evalstr);
end

% ensemble averages, etc.
for v = 1:1:vtot, fprintf('- %s: ', vnom(v,1:vlen(v)));
  evalstr = sprintf('t1 = mod_%s_ens;', vnom(v,1:vlen(v))); eval(evalstr);
  for m = 1:1:13, fprintf('.');
    t2 = t1(:,:,m,:); t3 = mean(t2, 4);
    evalstr = sprintf('mod_%s(:,:,%d) = t3;', vnom(v,1:vlen(v)), m); eval(evalstr);
    t4 = std(t2, [], 4);
    evalstr = sprintf('mod_%s_std(:,:,%d) = t4;', vnom(v,1:vlen(v)), m); eval(evalstr);
    t5 = t4 ./ t3;
    evalstr = sprintf('mod_%s_cov(:,:,%d) = t5;', vnom(v,1:vlen(v)), m); eval(evalstr);
  end; fprintf('\n');
end

% get them into a single array
for v = 1:1:vtot
  evalstr = sprintf('states_2d(:,:,:,v) = mod_%s;', vnom(v,1:vlen(v))); eval(evalstr);
  evalstr = sprintf('states_2d_ens(:,:,:,:,v) = mod_%s_ens;', vnom(v,1:vlen(v))); eval(evalstr);
  evalstr = sprintf('states_2d_std(:,:,:,v) = mod_%s_std;', vnom(v,1:vlen(v))); eval(evalstr);
  evalstr = sprintf('states_2d_cov(:,:,:,v) = mod_%s_cov;', vnom(v,1:vlen(v))); eval(evalstr);
end

% =========================================================================

% create seasonal averages
clear seas_states_2d*

seas = [3 4 5; 6 7 8; 9 10 11; 12 1 2];

for s = 1:1:5
  if s < 5, want_seas = seas(s,:); else, want_seas = 1:1:12; end
  for v = 1:1:vtot
    t1 = states_2d(:,:,want_seas,v);
    t2 = mean(t1, 3);
    seas_states_2d(:,:,s,v) = t2;
    t1 = states_2d_std(:,:,want_seas,v);
    t2 = mean(t1, 3);
    seas_states_2d_std(:,:,s,v) = t2;
    t1 = states_2d_cov(:,:,want_seas,v);
    t2 = mean(t1, 3);
    seas_states_2d_cov(:,:,s,v) = t2;
  end
end

save ('ukesm1_PD_output_27nov19', 'states_2d*', 'seas_states_2d*', '-v7.3');

% =========================================================================

% load up model data
load ukesm1_PD_output_27nov19

% load up observational data

load obs_2d_seas_moll_states_2019.mat

% =========================================================================

vnom = char('din', 'sil', 'fer', 'chl', 'tpp', 'prn', 'prd', 'dic', 'alk', 'ph', ...
  'cal', 'ccd', 'dpc', 'flx', 'oxy', 'phn', 'phd', 'pds', 'mlp', 'sfr', ...
  'opl', 'cal', 'zmi', 'zme', 'scp', 'fcp', 's100', 'f100', 'pnjlim', 'pnnlim', ...
  'pnflim', 'pdjlim', 'pdnlim', 'pdflim', 'pdslim');

nxx = -179.5:1:179.5;  nyy = -89.5:1:89.5;
[lon100, lat100] = meshgrid(nxx, nyy);

lon_rng = [-180 180]; lat_rng = [-90 90];
load coast

% gpos = [-0.025 -0.05 0.10 0.10; -0.025 -0.05 0.10 0.10; -0.025 0.0 0.10 0.10; -0.025 0.0 0.10 0.10];
gpos = [-0.075 -0.05 0.135 0.1; -0.05 -0.05 0.135 0.1; -0.075 0.0 0.135 0.1; -0.05 0.0 0.135 0.1];

tstr = char('Observed, JJA', 'Observed, DJF', 'Simulated, JJA', 'Simulated, DJF');
tstr2 = char('Simulated, JJA', 'Simulated, DJF', 'Simulated, JJA, CoV', 'Simulated, DJF, CoV');

% DIN
gpos = [-0.075 -0.05 0.135 0.1; -0.05 -0.05 0.135 0.1; -0.075 0.0 0.135 0.1; -0.05 0.0 0.135 0.1];
clear m1 o1;
v = 1; o1 = moll_obs_2d(:,:,2:2:4,v);
v = 1; m1 = seas_states_2d(:,:,2:2:4,v);
figure(1); clf; turbo(60);
pvar = [1 2]; for i = 1:1:2, subplot(2,2,pvar(i));
  ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
  pcolorm(lat100, lon100, o1(:,:,i));
  plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
  caxis ([0 35]);
  pos = get(gca, 'Position'); pos = pos + gpos(pvar(i),:); set(gca, 'Position', pos);
  tl = title (tstr(i,:)); tlp = get(tl, 'Position'); tlp(2) = tlp(2) - 0.5; set(tl, 'Position', tlp);
end
pvar = [3 4]; for i = 1:1:2, subplot(2,2,pvar(i));
  ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
  pcolorm(yy100e, xx100e, m1(:,:,i));
  plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
  caxis ([0 35]);
  pos = get(gca, 'Position'); pos = pos + gpos(pvar(i),:); set(gca, 'Position', pos);
  tl = title (tstr(i+2,:)); tlp = get(tl, 'Position'); tlp(2) = tlp(2) - 0.5; set(tl, 'Position', tlp);
  if pvar(i) == 4, cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos(1) = cbpos(1) + 0.025; cbpos(2) = cbpos(2) - 0.05; cbpos(3) = cbpos(3) - 0.05;
    set(cb, 'Position', cbpos); set(gca, 'Position', pos); end
end
def_figure; oname = sprintf('FIGS/ukesm1_PD_obs_mod_01.png'); exportgraphics(gcf, oname, 'Resolution', 600);

% SIL
gpos = [-0.075 -0.05 0.135 0.1; -0.05 -0.05 0.135 0.1; -0.075 0.0 0.135 0.1; -0.05 0.0 0.135 0.1];
clear m1 o1;
v = 2; o1 = moll_obs_2d(:,:,2:2:4,v);
v = 2; m1 = seas_states_2d(:,:,2:2:4,v);
figure(1); clf; turbo(60);
pvar = [1 2]; for i = 1:1:2, subplot(2,2,pvar(i));
  ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
  pcolorm(lat100, lon100, o1(:,:,i));
  plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
  caxis ([0 100]);
  pos = get(gca, 'Position'); pos = pos + gpos(pvar(i),:); set(gca, 'Position', pos);
  tl = title (tstr(i,:)); tlp = get(tl, 'Position'); tlp(2) = tlp(2) - 0.5; set(tl, 'Position', tlp);
  title (tstr(i,:));
end
pvar = [3 4]; for i = 1:1:2, subplot(2,2,pvar(i));
  ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
  pcolorm(yy100e, xx100e, m1(:,:,i));
  plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
  caxis ([0 100]);
  pos = get(gca, 'Position'); pos = pos + gpos(pvar(i),:); set(gca, 'Position', pos);
  tl = title (tstr(i+2,:)); tlp = get(tl, 'Position'); tlp(2) = tlp(2) - 0.5; set(tl, 'Position', tlp);
  if pvar(i) == 4, cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos(1) = cbpos(1) + 0.025; cbpos(2) = cbpos(2) - 0.05; cbpos(3) = cbpos(3) - 0.05;
    set(cb, 'Position', cbpos); set(gca, 'Position', pos); end
end
def_figure; oname = sprintf('FIGS/ukesm1_PD_obs_mod_02.png'); exportgraphics(gcf, oname, 'Resolution', 600);

% CHL
gpos = [-0.075 -0.05 0.135 0.1; -0.05 -0.05 0.135 0.1; -0.075 0.0 0.135 0.1; -0.05 0.0 0.135 0.1];
clear m1 o1;
v = 3; o0 = moll_obs_2d(:,:,2:2:4,v); o0(o0 == 0) = NaN; o1 = log10(o0);
v = 4; m1 = log10(seas_states_2d(:,:,2:2:4,v));
figure(1); clf; turbo(60);
pvar = [1 2]; for i = 1:1:2, subplot(2,2,pvar(i));
  ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
  pcolorm(lat100, lon100, o1(:,:,i));
  plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
  caxis ([-2 1]);
  pos = get(gca, 'Position'); pos = pos + gpos(pvar(i),:); set(gca, 'Position', pos);
  tl = title (tstr(i,:)); tlp = get(tl, 'Position'); tlp(2) = tlp(2) - 0.5; set(tl, 'Position', tlp);
end
pvar = [3 4]; for i = 1:1:2, subplot(2,2,pvar(i));
  ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
  pcolorm(yy100e, xx100e, m1(:,:,i));
  plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
  caxis ([-2 1]);
  pos = get(gca, 'Position'); pos = pos + gpos(pvar(i),:); set(gca, 'Position', pos);
  tl = title (tstr(i+2,:)); tlp = get(tl, 'Position'); tlp(2) = tlp(2) - 0.5; set(tl, 'Position', tlp);
  if pvar(i) == 4, cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos(1) = cbpos(1) + 0.025; cbpos(2) = cbpos(2) - 0.05; cbpos(3) = cbpos(3) - 0.05;
    set(cb, 'Position', cbpos); set(gca, 'Position', pos); set(cb, 'XTick', [-2 -1 0 1], 'XTickLabel', [0.01 0.1 1.0 10.0]); end
end
def_figure; oname = sprintf('FIGS/ukesm1_PD_obs_mod_03.png'); exportgraphics(gcf, oname, 'Resolution', 600);

% phytoplankton
gpos = [-0.075 -0.05 0.135 0.1; -0.05 -0.05 0.135 0.1; -0.075 0.0 0.135 0.1; -0.05 0.0 0.135 0.1];
tstr = char('Non-diatom, JJA', 'Non-diatom, DJF', 'Diatom, JJA', 'Diatom, DJF');
clear m1 o1;
v = 16; o1 = seas_states_2d(:,:,2:2:4,v);
v = 17; m1 = seas_states_2d(:,:,2:2:4,v);
figure(1); clf; turbo(60);
pvar = [1 2]; for i = 1:1:2, subplot(2,2,pvar(i));
  ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
  pcolorm(yy100e, xx100e, o1(:,:,i));
  plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
  caxis ([0 1]);
  pos = get(gca, 'Position'); pos = pos + gpos(pvar(i),:); set(gca, 'Position', pos);
  tl = title (tstr(i,:)); tlp = get(tl, 'Position'); tlp(2) = tlp(2) - 0.5; set(tl, 'Position', tlp);
end
pvar = [3 4]; for i = 1:1:2, subplot(2,2,pvar(i));
  ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
  pcolorm(yy100e, xx100e, m1(:,:,i));
  plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
  caxis ([0 1]);
  pos = get(gca, 'Position'); pos = pos + gpos(pvar(i),:); set(gca, 'Position', pos);
  tl = title (tstr(i+2,:)); tlp = get(tl, 'Position'); tlp(2) = tlp(2) - 0.5; set(tl, 'Position', tlp);
  if pvar(i) == 4, cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos(1) = cbpos(1) + 0.025; cbpos(2) = cbpos(2) - 0.05; cbpos(3) = cbpos(3) - 0.05;
    set(cb, 'Position', cbpos); set(gca, 'Position', pos); end
end
def_figure; oname = sprintf('FIGS/ukesm1_PD_mod_01.png'); exportgraphics(gcf, oname, 'Resolution', 600);

% zooplankton
gpos = [-0.075 -0.05 0.135 0.1; -0.05 -0.05 0.135 0.1; -0.075 0.0 0.135 0.1; -0.05 0.0 0.135 0.1];
tstr = char('Microzooplankton, JJA', 'Microzooplankton, DJF', 'Mesozooplankton, JJA', 'Mesozooplankton, DJF');
clear m1 o1;
v = 23; o1 = seas_states_2d(:,:,2:2:4,v);
v = 24; m1 = seas_states_2d(:,:,2:2:4,v);
figure(1); clf; turbo(60);
pvar = [1 2]; for i = 1:1:2, subplot(2,2,pvar(i));
  ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
  pcolorm(yy100e, xx100e, o1(:,:,i));
  plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
  caxis ([0 1]);
  pos = get(gca, 'Position'); pos = pos + gpos(pvar(i),:); set(gca, 'Position', pos);
  tl = title (tstr(i,:)); tlp = get(tl, 'Position'); tlp(2) = tlp(2) - 0.5; set(tl, 'Position', tlp);
end
pvar = [3 4]; for i = 1:1:2, subplot(2,2,pvar(i));
  ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
  pcolorm(yy100e, xx100e, m1(:,:,i));
  plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
  caxis ([0 1]);
  pos = get(gca, 'Position'); pos = pos + gpos(pvar(i),:); set(gca, 'Position', pos);
  tl = title (tstr(i+2,:)); tlp = get(tl, 'Position'); tlp(2) = tlp(2) - 0.5; set(tl, 'Position', tlp);
  if pvar(i) == 4, cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos(1) = cbpos(1) + 0.025; cbpos(2) = cbpos(2) - 0.05; cbpos(3) = cbpos(3) - 0.05;
    set(cb, 'Position', cbpos); set(gca, 'Position', pos); end
end
def_figure; oname = sprintf('FIGS/ukesm1_PD_mod_02.png'); exportgraphics(gcf, oname, 'Resolution', 600);

% phytoplankton
gpos = [-0.075 -0.05 0.135 0.1; -0.05 -0.05 0.135 0.1; -0.075 0.0 0.135 0.1; -0.05 0.0 0.135 0.1];
tstr = char('Non-diatom', 'Diatom', 'Microzooplankton', 'Mesozooplankton');
for i = 1:1:4, tlen(i) = lenstr(tstr(i,:)); end
clear t1;
v = 16; t1(:,:,1) = states_2d(:,:,13,v);
v = 17; t1(:,:,2) = states_2d(:,:,13,v);
v = 23; t1(:,:,3) = states_2d(:,:,13,v);
v = 24; t1(:,:,4) = states_2d(:,:,13,v);
figure(1); clf; turbo(60);
pvar = [1 3 2 4]; for i = 1:1:4, subplot(2,2,pvar(i));
  ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
  pcolorm(yy100e, xx100e, t1(:,:,i));
  plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
  if i < 3, caxis ([0 1]); else, caxis ([0 1]); end
  pos = get(gca, 'Position'); pos = pos + gpos(pvar(i),:); set(gca, 'Position', pos);
  tl = title (tstr(i,1:tlen(i))); tlp = get(tl, 'Position'); tlp(2) = tlp(2) - 0.5; set(tl, 'Position', tlp);
  if pvar(i) > 2, cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos(1) = cbpos(1) + 0.025; cbpos(2) = cbpos(2) - 0.05; cbpos(3) = cbpos(3) - 0.05;
    set(cb, 'Position', cbpos); set(gca, 'Position', pos); end
end
def_figure; oname = sprintf('FIGS/ukesm1_PD_ANN_mod_01.png'); exportgraphics(gcf, oname, 'Resolution', 600);

% Si:N and iron
gpos = [-0.075 -0.05 0.135 0.1; -0.05 -0.05 0.135 0.1; -0.075 0.0 0.135 0.1; -0.05 0.0 0.135 0.1];
tstr = char('Diatom Si:N, JJA', 'Diatom Si:N, DJF', 'Iron, JJA', 'Iron, DJF');
clear m1 o1;
o1 = seas_states_2d(:,:,2:2:4,18) ./ seas_states_2d(:,:,2:2:4,17);
m1 = seas_states_2d(:,:,2:2:4,3) * 1e3;
figure(1); clf; turbo(60);
pvar = [1 2]; for i = 1:1:2, subplot(2,2,pvar(i));
  ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
  pcolorm(yy100e, xx100e, o1(:,:,i));
  plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
  caxis ([0 5.5]);
  pos = get(gca, 'Position'); pos = pos + gpos(pvar(i),:); set(gca, 'Position', pos);
  tl = title (tstr(i,:)); tlp = get(tl, 'Position'); tlp(2) = tlp(2) - 0.5; set(tl, 'Position', tlp);
  if pvar(i) == 3, cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos(1) = cbpos(1) + 0.025; cbpos(2) = cbpos(2) - 0.05; cbpos(3) = cbpos(3) - 0.05;
    set(cb, 'Position', cbpos); set(gca, 'Position', pos); end
end
pvar = [3 4]; for i = 1:1:2, subplot(2,2,pvar(i));
  ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
  pcolorm(yy100e, xx100e, m1(:,:,i));
  plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
  caxis ([0 1.5]);
  pos = get(gca, 'Position'); pos = pos + gpos(pvar(i),:); set(gca, 'Position', pos);
  tl = title (tstr(i+2,:)); tlp = get(tl, 'Position'); tlp(2) = tlp(2) - 0.5; set(tl, 'Position', tlp);
  if pvar(i) == 4, cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos(1) = cbpos(1) + 0.025; cbpos(2) = cbpos(2) - 0.05; cbpos(3) = cbpos(3) - 0.05;
    set(cb, 'Position', cbpos); set(gca, 'Position', pos); end
end
def_figure; oname = sprintf('FIGS/ukesm1_PD_mod_03.png'); exportgraphics(gcf, oname, 'Resolution', 600);

% vnom = char('din', 'sil', 'fer', 'chl', 'tpp', 'prn', 'prd', 'dic', 'alk', 'ph', ...
%   'cal', 'ccd', 'dpc', 'flx', 'oxy', 'phn', 'phd', 'pds', 'mlp', 'sfr', ...
%   'opl', 'cal', 'zmi', 'zme', 'scp', 'fcp', 's100', 'f100', 'pnjlim', 'pnnlim', ...
%   'pnflim', 'pdjlim', 'pdnlim', 'pdflim', 'pdslim');

% TPP, VGPM
gpos = [-0.075 -0.05 0.135 0.1; -0.05 -0.05 0.135 0.1; -0.075 0.0 0.135 0.1; -0.05 0.0 0.135 0.1];
tstr = char('Observed, VGPM, JJA', 'Observed, VGPM, DJF', 'Simulated, JJA', 'Simulated, DJF');
clear m1 o1;

v = 4; o1 = moll_obs_2d(:,:,2:2:4,v);
v = 5; m1 = seas_states_2d(:,:,2:2:4,v);
figure(1); clf; turbo(60);
pvar = [1 2]; for i = 1:1:2, subplot(2,2,pvar(i));
  ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
  pcolorm(lat100, lon100, o1(:,:,i));
  plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
  caxis ([0 1.5]);
  pos = get(gca, 'Position'); pos = pos + gpos(pvar(i),:); set(gca, 'Position', pos);
  tl = title (tstr(i,:)); tlp = get(tl, 'Position'); tlp(2) = tlp(2) - 0.5; set(tl, 'Position', tlp);
end
pvar = [3 4]; for i = 1:1:2, subplot(2,2,pvar(i));
  ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
  pcolorm(yy100e, xx100e, m1(:,:,i));
  plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
  caxis ([0 1.5]);
  pos = get(gca, 'Position'); pos = pos + gpos(pvar(i),:); set(gca, 'Position', pos);
  tl = title (tstr(i+2,:)); tlp = get(tl, 'Position'); tlp(2) = tlp(2) - 0.5; set(tl, 'Position', tlp);
  if pvar(i) == 4, cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos(1) = cbpos(1) + 0.025; cbpos(2) = cbpos(2) - 0.05; cbpos(3) = cbpos(3) - 0.05;
    set(cb, 'Position', cbpos); set(gca, 'Position', pos); end
end
def_figure; oname = sprintf('FIGS/ukesm1_PD_obs_mod_04.png'); exportgraphics(gcf, oname, 'Resolution', 600);

% TPP, CbPM
gpos = [-0.075 -0.05 0.135 0.1; -0.05 -0.05 0.135 0.1; -0.075 0.0 0.135 0.1; -0.05 0.0 0.135 0.1];
tstr = char('Observed, CbPM, JJA', 'Observed, CbPM, DJF', 'Simulated, JJA', 'Simulated, DJF');
clear m1 o1;
v = 5; o1 = moll_obs_2d(:,:,2:2:4,v);
v = 5; m1 = seas_states_2d(:,:,2:2:4,v);
figure(1); clf; turbo(60);
pvar = [1 2]; for i = 1:1:2, subplot(2,2,pvar(i));
  ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
  pcolorm(lat100, lon100, o1(:,:,i));
  plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
  caxis ([0 1.5]);
  pos = get(gca, 'Position'); pos = pos + gpos(pvar(i),:); set(gca, 'Position', pos);
  tl = title (tstr(i,:)); tlp = get(tl, 'Position'); tlp(2) = tlp(2) - 0.5; set(tl, 'Position', tlp);
end
pvar = [3 4]; for i = 1:1:2, subplot(2,2,pvar(i));
  ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
  pcolorm(yy100e, xx100e, m1(:,:,i));
  plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
  caxis ([0 1.5]);
  pos = get(gca, 'Position'); pos = pos + gpos(pvar(i),:); set(gca, 'Position', pos);
  tl = title (tstr(i+2,:)); tlp = get(tl, 'Position'); tlp(2) = tlp(2) - 0.5; set(tl, 'Position', tlp);
  if pvar(i) == 4, cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos(1) = cbpos(1) + 0.025; cbpos(2) = cbpos(2) - 0.05; cbpos(3) = cbpos(3) - 0.05;
    set(cb, 'Position', cbpos); set(gca, 'Position', pos); end
end
def_figure; oname = sprintf('FIGS/ukesm1_PD_obs_mod_05.png'); exportgraphics(gcf, oname, 'Resolution', 600);

% TPP, Eppley
gpos = [-0.075 -0.05 0.135 0.1; -0.05 -0.05 0.135 0.1; -0.075 0.0 0.135 0.1; -0.05 0.0 0.135 0.1];
tstr = char('Observed, Eppley, JJA', 'Observed, Eppley, DJF', 'Simulated, JJA', 'Simulated, DJF');
clear m1 o1;
v = 6; o1 = moll_obs_2d(:,:,2:2:4,v);
v = 5; m1 = seas_states_2d(:,:,2:2:4,v);
figure(1); clf; turbo(60);
pvar = [1 2]; for i = 1:1:2, subplot(2,2,pvar(i));
  ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
  pcolorm(lat100, lon100, o1(:,:,i));
  plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
  caxis ([0 1.5]);
  pos = get(gca, 'Position'); pos = pos + gpos(pvar(i),:); set(gca, 'Position', pos);
  tl = title (tstr(i,:)); tlp = get(tl, 'Position'); tlp(2) = tlp(2) - 0.5; set(tl, 'Position', tlp);
end
pvar = [3 4]; for i = 1:1:2, subplot(2,2,pvar(i));
  ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
  pcolorm(yy100e, xx100e, m1(:,:,i));
  plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
  caxis ([0 1.5]);
  pos = get(gca, 'Position'); pos = pos + gpos(pvar(i),:); set(gca, 'Position', pos);
  tl = title (tstr(i+2,:)); tlp = get(tl, 'Position'); tlp(2) = tlp(2) - 0.5; set(tl, 'Position', tlp);
  if pvar(i) == 4, cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos(1) = cbpos(1) + 0.025; cbpos(2) = cbpos(2) - 0.05; cbpos(3) = cbpos(3) - 0.05;
    set(cb, 'Position', cbpos); set(gca, 'Position', pos); end
end
def_figure; oname = sprintf('FIGS/ukesm1_PD_obs_mod_06.png'); exportgraphics(gcf, oname, 'Resolution', 600);

% TPP, mean of VGPM, CbPM and Eppley
gpos = [-0.075 -0.05 0.135 0.1; -0.05 -0.05 0.135 0.1; -0.075 0.0 0.135 0.1; -0.05 0.0 0.135 0.1];
tstr = char('Observed, NPP, JJA', 'Observed, NPP, DJF', 'Simulated, JJA', 'Simulated, DJF');
clear m1 o1;
o1 = mean(moll_obs_2d(:,:,2:2:4,4:6), 4);
v = 5; m1 = seas_states_2d(:,:,2:2:4,v);
figure(1); clf; turbo(60);
pvar = [1 2]; for i = 1:1:2, subplot(2,2,pvar(i));
  ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
  pcolorm(lat100, lon100, o1(:,:,i));
  plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
  caxis ([0 1.5]);
  pos = get(gca, 'Position'); pos = pos + gpos(pvar(i),:); set(gca, 'Position', pos);
  tl = title (tstr(i,:)); tlp = get(tl, 'Position'); tlp(2) = tlp(2) - 0.5; set(tl, 'Position', tlp);
end
pvar = [3 4]; for i = 1:1:2, subplot(2,2,pvar(i));
  ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
  pcolorm(yy100e, xx100e, m1(:,:,i));
  plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
  caxis ([0 1.5]);
  pos = get(gca, 'Position'); pos = pos + gpos(pvar(i),:); set(gca, 'Position', pos);
  tl = title (tstr(i+2,:)); tlp = get(tl, 'Position'); tlp(2) = tlp(2) - 0.5; set(tl, 'Position', tlp);
  if pvar(i) == 4, cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos(1) = cbpos(1) + 0.025; cbpos(2) = cbpos(2) - 0.05; cbpos(3) = cbpos(3) - 0.05;
    set(cb, 'Position', cbpos); set(gca, 'Position', pos); end
end
def_figure; oname = sprintf('FIGS/ukesm1_PD_obs_mod_07.png'); exportgraphics(gcf, oname, 'Resolution', 600);

% vnom = char('din', 'sil', 'fer', 'chl', 'tpp', 'prn', 'prd', 'dic', 'alk', 'ph', ...
%   'cal', 'ccd', 'dpc', 'flx', 'oxy', 'phn', 'phd', 'pds', 'mlp', 'sfr', ...
%   'opl', 'cal', 'zmi', 'zme', 'scp', 'fcp', 's100', 'f100', 'pnjlim', 'pnnlim', ...
%   'pnflim', 'pdjlim', 'pdnlim', 'pdflim', 'pdslim');

% Primary production split
gpos = [-0.075 -0.05 0.135 0.1; -0.05 -0.05 0.135 0.1; -0.075 0.0 0.135 0.1; -0.05 0.0 0.135 0.1];
tstr = char('Pn production, JJA', 'Pn production, DJF', 'Pd production, JJA', 'Pd production, DJF');
v = 6; o1 = seas_states_2d(:,:,2:2:4,v);
v = 7; m1 = seas_states_2d(:,:,2:2:4,v);
figure(1); clf; turbo(60);
pvar = [1 2]; for i = 1:1:2, subplot(2,2,pvar(i));
  ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
  pcolorm(yy100e, xx100e, o1(:,:,i));
  plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
  caxis ([0 1]);
  pos = get(gca, 'Position'); pos = pos + gpos(pvar(i),:); set(gca, 'Position', pos);
  tl = title (tstr(i,:)); tlp = get(tl, 'Position'); tlp(2) = tlp(2) - 0.5; set(tl, 'Position', tlp);
end
pvar = [3 4]; for i = 1:1:2, subplot(2,2,pvar(i));
  ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
  pcolorm(yy100e, xx100e, m1(:,:,i));
  plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
  caxis ([0 1]);
  pos = get(gca, 'Position'); pos = pos + gpos(pvar(i),:); set(gca, 'Position', pos);
  tl = title (tstr(i+2,:)); tlp = get(tl, 'Position'); tlp(2) = tlp(2) - 0.5; set(tl, 'Position', tlp);
  if pvar(i) == 4, cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos(1) = cbpos(1) + 0.025; cbpos(2) = cbpos(2) - 0.05; cbpos(3) = cbpos(3) - 0.05;
    set(cb, 'Position', cbpos); set(gca, 'Position', pos); end
end
def_figure; oname = sprintf('FIGS/ukesm1_PD_mod_04.png'); exportgraphics(gcf, oname, 'Resolution', 600);

% Primary production split
tstr = char('ML production, JJA', 'ML production, DJF', 'Biogenic opal, JJA', 'Biogenic opal, DJF');
o1 = seas_states_2d(:,:,2:2:4,19) ./ seas_states_2d(:,:,2:2:4,5);
v = 21; m1 = seas_states_2d(:,:,2:2:4,v);
figure(1); clf; turbo(60);
pvar = [1 2]; for i = 1:1:2, subplot(2,2,pvar(i));
  ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
  pcolorm(yy100e, xx100e, o1(:,:,i));
  plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
  caxis ([0 1]);
  pos = get(gca, 'Position'); pos = pos + gpos(pvar(i),:); set(gca, 'Position', pos);
  tl = title (tstr(i,:)); tlp = get(tl, 'Position'); tlp(2) = tlp(2) - 0.5; set(tl, 'Position', tlp);
  if pvar(i) == 3, cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos(1) = cbpos(1) + 0.025; cbpos(2) = cbpos(2) - 0.05; cbpos(3) = cbpos(3) - 0.05;
    set(cb, 'Position', cbpos); set(gca, 'Position', pos); end
end
pvar = [3 4]; for i = 1:1:2, subplot(2,2,pvar(i));
  ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
  pcolorm(yy100e, xx100e, m1(:,:,i));
  plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
  caxis ([0 12]);
  pos = get(gca, 'Position'); pos = pos + gpos(pvar(i),:); set(gca, 'Position', pos);
  tl = title (tstr(i+2,:)); tlp = get(tl, 'Position'); tlp(2) = tlp(2) - 0.5; set(tl, 'Position', tlp);
  if pvar(i) == 4, cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos(1) = cbpos(1) + 0.025; cbpos(2) = cbpos(2) - 0.05; cbpos(3) = cbpos(3) - 0.05;
    set(cb, 'Position', cbpos); set(gca, 'Position', pos); end
end
def_figure; oname = sprintf('FIGS/ukesm1_PD_mod_05.png'); exportgraphics(gcf, oname, 'Resolution', 600);

% Primary production split
tstr = char('Biogenic opal, JJA', 'Biogenic opal, DJF', 'Calcium carbonate, JJA', 'Calcium carbonate, DJF');
v = 21; o1 = seas_states_2d(:,:,2:2:4,v);
v = 22; m1 = seas_states_2d(:,:,2:2:4,v);
figure(1); clf; turbo(60);
pvar = [1 2]; for i = 1:1:2, subplot(2,2,pvar(i));
  ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
  pcolorm(yy100e, xx100e, o1(:,:,i));
  plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
  caxis ([0 12]);
  pos = get(gca, 'Position'); pos = pos + gpos(pvar(i),:); set(gca, 'Position', pos);
  tl = title (tstr(i,:)); tlp = get(tl, 'Position'); tlp(2) = tlp(2) - 0.5; set(tl, 'Position', tlp);
  if pvar(i) == 3, cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos(1) = cbpos(1) + 0.025; cbpos(2) = cbpos(2) - 0.05; cbpos(3) = cbpos(3) - 0.05;
    set(cb, 'Position', cbpos); set(gca, 'Position', pos); end
end
pvar = [3 4]; for i = 1:1:2, subplot(2,2,pvar(i));
  ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
  pcolorm(yy100e, xx100e, m1(:,:,i));
  plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
  caxis ([0 3]);
  pos = get(gca, 'Position'); pos = pos + gpos(pvar(i),:); set(gca, 'Position', pos);
  tl = title (tstr(i+2,:)); tlp = get(tl, 'Position'); tlp(2) = tlp(2) - 0.5; set(tl, 'Position', tlp);
  if pvar(i) == 4, cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos(1) = cbpos(1) + 0.025; cbpos(2) = cbpos(2) - 0.05; cbpos(3) = cbpos(3) - 0.05;
    set(cb, 'Position', cbpos); set(gca, 'Position', pos); end
end
def_figure; oname = sprintf('FIGS/ukesm1_PD_mod_05_v3.png'); exportgraphics(gcf, oname, 'Resolution', 600);

% Primary production split
tstr = char('ML production, JJA', 'ML production, DJF', 'Seafloor flux, JJA', 'Seafloor flux, DJF');
lxt = -3:1:2; xt = [0.001 0.01 0.1 1.0 10.0 100.0];
o1 = seas_states_2d(:,:,2:2:4,19) ./ seas_states_2d(:,:,2:2:4,5);
v = 20; m1 = log10(seas_states_2d(:,:,2:2:4,v) * 12.011);
figure(1); clf; turbo(60);
pvar = [1 2]; for i = 1:1:2, subplot(2,2,pvar(i));
  ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
  pcolorm(yy100e, xx100e, o1(:,:,i));
  plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
  caxis ([0 1]);
  pos = get(gca, 'Position'); pos = pos + gpos(pvar(i),:); set(gca, 'Position', pos);
  tl = title (tstr(i,:)); tlp = get(tl, 'Position'); tlp(2) = tlp(2) - 0.5; set(tl, 'Position', tlp);
  if pvar(i) == 3, cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos(1) = cbpos(1) + 0.025; cbpos(2) = cbpos(2) - 0.05; cbpos(3) = cbpos(3) - 0.05;
    set(cb, 'Position', cbpos); set(gca, 'Position', pos); end
end
pvar = [3 4]; for i = 1:1:2, subplot(2,2,pvar(i));
  ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
  pcolorm(yy100e, xx100e, m1(:,:,i));
  plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
  caxis ([-3.5 2]);
  pos = get(gca, 'Position'); pos = pos + gpos(pvar(i),:); set(gca, 'Position', pos);
  tl = title (tstr(i+2,:)); tlp = get(tl, 'Position'); tlp(2) = tlp(2) - 0.5; set(tl, 'Position', tlp);
  if pvar(i) == 4, cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos(1) = cbpos(1) + 0.025; cbpos(2) = cbpos(2) - 0.05; cbpos(3) = cbpos(3) - 0.05;
    set(cb, 'Position', cbpos); set(gca, 'Position', pos);
    set(cb, 'XTick', lxt, 'XTickLabel', xt); end
end
def_figure; oname = sprintf('FIGS/ukesm1_PD_mod_05_v2.png'); exportgraphics(gcf, oname, 'Resolution', 600);

% vnom = char('din', 'sil', 'fer', 'chl', 'tpp', 'prn', 'prd', 'dic', 'alk', 'ph', ...
%   'cal', 'ccd', 'dpc', 'flx', 'oxy', 'phn', 'phd', 'pds', 'mlp', 'sfr', ...
%   'opl', 'cal', 'zmi', 'zme', 'scp', 'fcp', 's100', 'f100', 'pnjlim', 'pnnlim', ...
%   'pnflim', 'pdjlim', 'pdnlim', 'pdflim', 'pdslim');

% Fast / slow split
tstr = char('Slow detritus, JJA', 'Slow detritus, DJF', 'Fast detritus, JJA', 'Fast detritus, DJF');
clear t1;
v = 25; o1 = seas_states_2d(:,:,2:2:4,v);
v = 26; m1 = seas_states_2d(:,:,2:2:4,v);
figure(1); clf; turbo(60);
pvar = [1 2]; for i = 1:1:2, subplot(2,2,pvar(i));
  ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
  pcolorm(yy100e, xx100e, o1(:,:,i));
  plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
  caxis ([0 70]);
  pos = get(gca, 'Position'); pos = pos + gpos(pvar(i),:); set(gca, 'Position', pos);
  tl = title (tstr(i,:)); tlp = get(tl, 'Position'); tlp(2) = tlp(2) - 0.5; set(tl, 'Position', tlp);
  if pvar(i) == 3, cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos(1) = cbpos(1) + 0.025; cbpos(2) = cbpos(2) - 0.05; cbpos(3) = cbpos(3) - 0.05;
    set(cb, 'Position', cbpos); set(gca, 'Position', pos); end
end
pvar = [3 4]; for i = 1:1:2, subplot(2,2,pvar(i));
  ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
  pcolorm(yy100e, xx100e, m1(:,:,i));
  plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
  caxis ([0 70]);
  pos = get(gca, 'Position'); pos = pos + gpos(pvar(i),:); set(gca, 'Position', pos);
  tl = title (tstr(i+2,:)); tlp = get(tl, 'Position'); tlp(2) = tlp(2) - 0.5; set(tl, 'Position', tlp);
  if pvar(i) == 4, cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos(1) = cbpos(1) + 0.025; cbpos(2) = cbpos(2) - 0.05; cbpos(3) = cbpos(3) - 0.05;
    set(cb, 'Position', cbpos); set(gca, 'Position', pos); end
end
def_figure; oname = sprintf('FIGS/ukesm1_PD_mod_06.png'); exportgraphics(gcf, oname, 'Resolution', 600);

% Fast / slow split, 100m
tstr = char('Slow detritus, 100 m, JJA', 'Slow detritus, 100 m, DJF', 'Fast detritus, 100 m, JJA', 'Fast detritus, 100 m, DJF');
clear t1;
v = 27; o1 = seas_states_2d(:,:,2:2:4,v);
v = 28; m1 = seas_states_2d(:,:,2:2:4,v);
figure(1); clf; turbo(60);
pvar = [1 2]; for i = 1:1:2, subplot(2,2,pvar(i));
  ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
  pcolorm(yy100e, xx100e, o1(:,:,i));
  plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
  caxis ([0 30]);
  pos = get(gca, 'Position'); pos = pos + gpos(pvar(i),:); set(gca, 'Position', pos);
  tl = title (tstr(i,:)); tlp = get(tl, 'Position'); tlp(2) = tlp(2) - 0.5; set(tl, 'Position', tlp);
  if pvar(i) == 3, cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos(1) = cbpos(1) + 0.025; cbpos(2) = cbpos(2) - 0.05; cbpos(3) = cbpos(3) - 0.05;
    set(cb, 'Position', cbpos); set(gca, 'Position', pos); end
end
pvar = [3 4]; for i = 1:1:2, subplot(2,2,pvar(i));
  ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
  pcolorm(yy100e, xx100e, m1(:,:,i));
  plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
  caxis ([0 30]);
  pos = get(gca, 'Position'); pos = pos + gpos(pvar(i),:); set(gca, 'Position', pos);
  tl = title (tstr(i+2,:)); tlp = get(tl, 'Position'); tlp(2) = tlp(2) - 0.5; set(tl, 'Position', tlp);
  if pvar(i) == 4, cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos(1) = cbpos(1) + 0.025; cbpos(2) = cbpos(2) - 0.05; cbpos(3) = cbpos(3) - 0.05;
    set(cb, 'Position', cbpos); set(gca, 'Position', pos); end
end
def_figure; oname = sprintf('FIGS/ukesm1_PD_mod_07.png'); exportgraphics(gcf, oname, 'Resolution', 600);

% limitation terms
% equator at position 187 in yy array
t1 = seas_states_2d(:,:,2,29:35);
t1(1:187,:,:,:) = seas_states_2d(1:187,:,4,29:35);
t2 = squeeze(t1);
pn_lim = t2(:,:,2:3);
pd_lim = t2(:,:,5:7);

tstr = char('Pn N limitation', 'Pd N limitation', 'Pn Fe limitation', 'Pd Fe limitation', 'Pd Si limitation');
gpos = [-0.025 -0.08 0.10 0.08; -0.025 -0.08 0.10 0.08; -0.025 -0.04 0.10 0.08; -0.025 -0.04 0.10 0.08; -0.025 0.0 0.10 0.08];
clear t1;
t1(:,:,1) = pn_lim(:,:,1);
t1(:,:,2) = pd_lim(:,:,1);
t1(:,:,3) = pn_lim(:,:,2);
t1(:,:,4) = pd_lim(:,:,2);
t1(:,:,5) = pd_lim(:,:,3);
t1(end+1,:,:) = NaN; t1(:,end+1,:) = NaN;
t1(t1 == 0) = NaN;
figure(1); clf
set(gcf, 'PaperPosition', [0.8353 2.1412 6.5882 7.4118] * 2.54);
turbo(60); pvar = [1 2 3 4 6];
for i = 1:1:5
  subplot (3,2,pvar(i));
  ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
  pcolorm(yy100e, xx100e, t1(:,:,i));
  plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
  caxis ([0 1]);
  pos = get(gca, 'Position'); pos = pos + gpos(i,:); set(gca, 'Position', pos);
  tl = title (tstr(i,:)); tlp = get(tl, 'Position'); tlp(2) = tlp(2) - 0.1; set(tl, 'Position', tlp);
  if pvar(i) == 6, cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos(1) = cbpos(1) + 0.025; cbpos(2) = cbpos(2) - 0.05; cbpos(3) = cbpos(3) - 0.05;
    set(cb, 'Position', cbpos); set(gca, 'Position', pos); end
end
def_figure; oname = sprintf('FIGS/ukesm1_PD_mod_08_v2.png'); exportgraphics(gcf, oname, 'Resolution', 600);

clear pn_maxlim pd_maxlim;
[t1, t2] = min(pn_lim, [], 3);
pn_maxlim(:,:,1) = t1;
t2(isnan(t1)) = NaN;
pn_maxlim(:,:,2) = t2;
[t1, t2] = min(pd_lim, [], 3);
pd_maxlim(:,:,1) = t1;
t2(isnan(t1)) = NaN;
pd_maxlim(:,:,2) = t2;

% nutrient limitation plots
gpos = [-0.075 -0.05 0.135 0.1; -0.05 -0.05 0.135 0.1; -0.075 0.0 0.135 0.1; -0.05 0.0 0.135 0.1];

tstr = char('Pn summer limitation', 'Pd summer limitation', 'Pn limiting factor', 'Pd limiting factor');
clear o1 m1;
o1(:,:,1) = pn_maxlim(:,:,1); o1(:,:,2) = pd_maxlim(:,:,1);
figure(1); clf; turbo(60);
pvar = [1 2]; for i = 1:1:2, subplot(1,2,pvar(i));
  ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
  pcolorm(yy100e, xx100e, o1(:,:,i));
  plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
  caxis ([0 1]);
  pos = get(gca, 'Position'); pos = pos + gpos(pvar(i),:); set(gca, 'Position', pos);
  tl = title (tstr(i,:)); tlp = get(tl, 'Position'); tlp(2) = tlp(2) - 0.5; set(tl, 'Position', tlp);
  if pvar(i) == 2, cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos(1) = cbpos(1) + 0.025; cbpos(2) = cbpos(2) - 0.05; cbpos(3) = cbpos(3) - 0.05;
    set(cb, 'Position', cbpos); set(gca, 'Position', pos); end
end
def_figure; oname = sprintf('FIGS/ukesm1_PD_mod_08_pt1.png'); exportgraphics(gcf, oname, 'Resolution', 600);

o1(:,:,1) = pn_maxlim(:,:,2); o1(:,:,2) = pd_maxlim(:,:,2);
figure(1); clf; pal = turbo(60); pal2 = pal(10:20:50,:); colormap(pal2);
pvar = [1 2]; for i = 1:1:2, subplot(1,2,pvar(i));
  ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
  pcolorm(yy100e, xx100e, o1(:,:,i));
  plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
  caxis ([0.5 3.5]);
  pos = get(gca, 'Position'); pos = pos + gpos(pvar(i),:); set(gca, 'Position', pos);
  tl = title (tstr(i,:)); tlp = get(tl, 'Position'); tlp(2) = tlp(2) - 0.5; set(tl, 'Position', tlp);
  if pvar(i) == 2, cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos(1) = cbpos(1) + 0.025; cbpos(2) = cbpos(2) - 0.05; cbpos(3) = cbpos(3) - 0.05;
    set(cb, 'XTick', 1:1:3); set(cb, 'XTickLabel', [' N'; 'Fe'; 'Si']);
    set(cb, 'Position', cbpos); set(gca, 'Position', pos); end
end
def_figure; oname = sprintf('FIGS/ukesm1_PD_mod_08_pt2.png'); exportgraphics(gcf, oname, 'Resolution', 600);

% ANN production split
tstr = char('Pn production', 'Pd production');
clear t1;
v = 6; t1(:,:,1) = states_2d(:,:,13,v);
v = 7; t1(:,:,2) = states_2d(:,:,13,v);
figure(1); clf; turbo(60);
pvar = [1 2]; for i = 1:1:2, subplot(1,2,pvar(i));
  ax = axesm('MapProjection', 'mollweid', 'MapLonLim', lon_rng, 'MapLatLim', lat_rng); hold on;
  pcolorm(yy100e, xx100e, t1(:,:,i));
  plotm(lat, long, '-k', 'LineWidth', 0.5); axis off; h = framem; set (h, 'LineWidth', 0.5);
  caxis ([0 1]);
  pos = get(gca, 'Position'); pos = pos + gpos(pvar(i),:); set(gca, 'Position', pos);
  tl = title (tstr(i,:)); tlp = get(tl, 'Position'); tlp(2) = tlp(2) - 0.5; set(tl, 'Position', tlp);
  if pvar(i) == 2, cb = colorbar('horiz'); cbpos = get(cb, 'Position'); cbpos(1) = cbpos(1) + 0.025; cbpos(2) = cbpos(2) - 0.05; cbpos(3) = cbpos(3) - 0.05;
    set(cb, 'Position', cbpos); set(gca, 'Position', pos); end
end
def_figure; oname = sprintf('FIGS/ukesm1_PD_ANN_mod_02.png'); exportgraphics(gcf, oname, 'Resolution', 600);

% =========================================================================
% =========================================================================
% =========================================================================
