clear; startup;

% =========================================================================
% =========================================================================
%
% Matlab plotting scripts for the analysis of UKESM1 prepared for:
%
%   Yool, A., Palmiéri, J., Jones, C. G., de Mora, L., Kuhlbrodt, T., 
%   Popova, E. E., Nurser, A. J. G., Hirschi, J., Blaker, A. T., 
%	Coward, A. C., Blockley, E. W., and Sellar, A. A.: Evaluating the 
% 	physical and biogeochemical state of the global ocean component of 
%	UKESM1 in CMIP6 Historical simulations, Geosci. Model Dev. Discuss. 
%	[preprint], https://doi.org/10.5194/gmd-2020-333, in review, 2020.
%
%   Accepted for publication: 20 April 2021
%
% Note: scripts provided to document analysis, and may require editing to
% fully replicate plots in manuscript
%
% =========================================================================
% =========================================================================

% =========================================================================
%
% This script plots Figure 19
%
% =========================================================================

% =========================================================================

load ukesm1_PD_ensemble

% =========================================================================

% load up background files
load nemo_area100e
load nemo_mask100e
fname = sprintf('%s/mean_nemo_%so_1y_20001201-20101201_grid-T.nc', dname(1).name, rname(1).name);
t1 = ncread(fname, 'deptht_bounds');
nemo_dep = double(t1(1,:)); nemo_dep(end+1) = double(t1(2,end));
t1 = ncread(fname, 'deptht');
nemo_mid = double(t1');

% total grid area
t1 = area100e + nemo_mask100e(:,:,1);
t2 = isfinite(t1);
tot_area100e = sum(t1(t2)); % m2

% extra mask to get rid of Mediterranean, Black and Caspian Seas
t1 = nemo_mask100e(:,:,1) + 1;
t1(isnan(t1)) = 0;
t1(235:249,283:344) = t1(235:249,283:344) + 2;
t1(249:262,291:344) = t1(249:262,291:344) + 4;
t1(262:276,330:344) = t1(262:276,330:344) + 8;
t2 = t1;
t2(t1 < 1) = NaN; t2(t1 > 1) = NaN;
med_mask100e = t2;

nxx = -179.5:1:179.5;  nyy = -89.5:1:89.5;
[lon100, lat100] = meshgrid(nxx, nyy);
nxx = -180:1:180;  nyy = -90:1:90;
[lon100b, lat100b] = meshgrid(nxx, nyy);
nxx = -179.5:2:179.5;  nyy = -89.5:2:89.5;
[lon200, lat200] = meshgrid(nxx, nyy);

load woa_mask
clear t1; t1(:,1:180) = woa_mask(:,181:360); t1(:,181:360) = woa_mask(:,1:180);
woa_mask = t1;

load woa_basin

helvetica;
pngopt = 0;

% ----------------------------------------------------------------------

addpath('~/Matlab/GLODAP/');
load woa_basin

load GLODAPv1_per_m3_fields

gd1_pi_dic = new_pi_dic;
gd1_totco2 = new_totco2;
gd1_totalk = new_totalk;
gd1_anthro = new_anthro;
gd1_cfc_11 = new_cfc_11;

addpath('~/Matlab/GLODAP_v2/');
load per_m3_fields

gd2_pi_dic = new_pi_dic;
gd2_totco2 = new_totco2;
gd2_totalk = new_totalk;
gd2_anthro = new_anthro;

obs_dep = ncread('AnthCO2.nc', 'depth');
obs_lon = ncread('AnthCO2.nc', 'longitude');

% ----------------------------------------------------------------------

clear pi_dic1 pi_dic2 pi_alk1 pi_alk2 pd_dic1 pd_dic2 pd_alk1 pd_alk2 pd_cfc11 pd_vol1 pd_vol2;

for e = 1:1:tsize
  % PIcontrol data
  dnom = sprintf('%s/mean_medusa_aw310o_*ptrc-T.nc', dname(e).name);
  t1 = dir(dnom);
  nom1990s = sprintf('%s/%s', dname(e).name, t1(1).name);
  nom2000s = sprintf('%s/%s', dname(e).name, t1(2).name);
  %
  fprintf('- Reading %s\n', nom1990s);
  t1 = ncread(nom1990s, 'DIC');
  t2 = permute(t1, [2 1 3]);
  pi_dic1(:,:,:,e) = t2;
  fprintf('- Reading %s\n', nom2000s);
  t1 = ncread(nom1990s, 'DIC');
  t2 = permute(t1, [2 1 3]);
  pi_dic2(:,:,:,e) = t2;
  
  % PD data
  pnom = sprintf('%s/mean_medusa_%so_1y_19901201-20001201_ptrc-T.nc', dname(e).name, rname(e).name);
  fprintf('- Reading %s\n', pnom);
  t1 = ncread(pnom, 'DIC');
  t2 = permute(t1, [2 1 3]);
  pd_dic1(:,:,:,e) = t2;
  t1 = ncread(pnom, 'ALK');
  t2 = permute(t1, [2 1 3]);
  pd_alk1(:,:,:,e) = t2;
  t1 = ncread(pnom, 'CFC11');
  t2 = permute(t1, [2 1 3]);
  pd_cfc11(:,:,:,e) = t2;
  %
  pnom = sprintf('%s/mean_medusa_%so_1y_20001201-20101201_ptrc-T.nc', dname(e).name, rname(e).name);
  fprintf('- Reading %s\n', pnom);
  t1 = ncread(pnom, 'DIC');
  t2 = permute(t1, [2 1 3]);
  pd_dic2(:,:,:,e) = t2;
  t1 = ncread(pnom, 'ALK');
  t2 = permute(t1, [2 1 3]);
  pd_alk2(:,:,:,e) = t2;
end

pd_ant1 = pd_dic1 - pi_dic1;
pd_ant2 = pd_dic2 - pi_dic2;

fname = sprintf('%s/mean_nemo_%so_1y_20001201-20101201_grid-T.nc', dname(1).name, rname(1).name);
t1 = ncread(fname, 'deptht_bounds');
nemo_dep = double(t1(1,:)); nemo_dep(end+1) = double(t1(2,end));
t1 = ncread(fname, 'deptht');
nemo_mid = double(t1');

zzthk = nemo_dep(2:end) - nemo_dep(1:end-1);

% ----------------------------------------------------------------------

lon_rng = [-180 180]; lat_rng = [-90 90];
gpos = [-0.075 0 0.135 0.1; -0.05 0 0.135 0.1];
load coast;

% ----------------------------------------------------------------------

load nemo_mask100e

rx100a = -179.5:1:179.5;
ry100a = -89.5:1:89.5;

[rx100, ry100] = meshgrid(rx100a, ry100a);

avg_pd_ant1 = mean(pd_ant1, 4);
avg_pd_ant2 = mean(pd_ant2, 4);
avg_pd_cfc11 = mean(pd_cfc11, 4);

% ----------------------------------------------------------------------

for k = 1:1:75
  t1 = avg_pd_ant1(:,:,k);
  t1(t1 < -10) = 0;
  t1(isnan(t1)) = -1e9;
  t2 = scatregrid(yy100e, xx100e, t1, ry100, rx100);
  t2(t2 < -10) = NaN;
  rg_pd_ant1(:,:,k) = t2;
  t1 = avg_pd_ant2(:,:,k);
  t1(t1 < -10) = 0;
  t1(isnan(t1)) = -1e9;
  t2 = scatregrid(yy100e, xx100e, t1, ry100, rx100);
  t2(t2 < -10) = NaN;
  rg_pd_ant2(:,:,k) = t2;
  t1 = avg_pd_cfc11(:,:,k);
  t1(t1 < -10) = 0;
  t1(isnan(t1)) = -1e9;
  t2 = scatregrid(yy100e, xx100e, t1, ry100, rx100);
  t2(t2 < -10) = NaN;
  rg_pd_cfc11(:,:,k) = t2;
end

% ----------------------------------------------------------------------

save ('ukesm1_PD_NAtl_issues_data', 'rg_pd_*', 'gd1_*', 'gd2_*', 'rx100a', 'ry100a', 'nemo_mid', 'obs_dep', '-v7.3');

% ----------------------------------------------------------------------

load ('ukesm1_PD_NAtl_issues_data');

figure(1); clf;
plot ([0 10], [0 10], 'r-');
xlabel ('CO_{2}', 'FontName', 'helvetica', 'Interpreter', 'latex');
ylabel ('$\textsf{CO_2}$', 'Interpreter', 'LaTeX');
onom = char('arse'); paper; orient portrait; 
set(gcf, 'PaperPosition', [2.1217 8.5763 16.7340 9.5506]);
print ('-depsc', '-painters', onom); pngconvert(onom, 200, pngopt);

ypos = 125;
ysft = 0.075;

figure(1); clf; turbo(60);
subplot (1,2,2);
t1 = permute(squeeze(rg_pd_ant1(ypos,:,:)), [2 1]);
pcolor (rx100a, nemo_mid, t1); shading flat;
seafloor (rx100a, nemo_mid, t1);
set (gca, 'YDir', 'reverse');
axis ([-80 0 0 6e3]);
xlabel ('Longitude [{\circ}E]');
% ylabel ('Depth [m]');
caxis ([0 60]); cb = colorbar('horiz'); cbar_xlabel(cb, 'Anthropogenic CO_{2} [mmol C m^{-3}]');
set(cb, 'FontSize', 12);
title ('Simulated'); topbox;
pos2 = get(gca, 'Position'); pos2 = pos2 + [0 ysft 0 -ysft]; cbpos = get(cb, 'Position'); cbpos = cbpos + [0 ysft/3 0 -cbpos(4)/2];
set (gca, 'Position', pos2); set(cb, 'Position', cbpos);
subplot (1,2,1);
t1 = permute(squeeze(gd1_anthro(ypos,:,:)), [2 1]);
pcolor (rx100a, obs_dep, t1); shading flat;
seafloor (rx100a, obs_dep, t1);
set (gca, 'YDir', 'reverse');
axis ([-80 0 0 6e3]);
xlabel ('Longitude [{\circ}E]');
ylabel ('Depth [m]');
caxis ([0 60]);
title ('Observed'); topbox;
pos1 = get(gca, 'Position'); pos1(2) = pos2(2); pos1(4) = pos2(4); set (gca, 'Position', pos1);
paper; orient portrait; set(gcf, 'PaperPosition', [2.1217 8.5763 16.7340 9.5506]);
onom = sprintf('FIGS/ukesm1_PD_NAtl_issues_%s.png', numprint(1, 2));
exportgraphics(gcf, onom, 'Resolution', 600);

figure(1); clf; turbo(60);
subplot (1,2,2);
t1 = permute(squeeze(rg_pd_cfc11(ypos,:,:) * 1e9), [2 1]);
pcolor (rx100a, nemo_mid, t1); shading flat;
seafloor (rx100a, nemo_mid, t1);
set (gca, 'YDir', 'reverse');
axis ([-80 0 0 6e3]);
xlabel ('Longitude [{\circ}E]');
% ylabel ('Depth [m]');
caxis ([0 3]); cb = colorbar('horiz'); cbar_xlabel(cb, 'CFC-11 [nmol m^{-3}]');
title ('Simulated'); topbox;
pos2 = get(gca, 'Position'); pos2 = pos2 + [0 ysft 0 -ysft]; cbpos = get(cb, 'Position'); cbpos = cbpos + [0 ysft/3 0 -cbpos(4)/2];
set (gca, 'Position', pos2); set(cb, 'Position', cbpos);
subplot (1,2,1);
t1 = permute(squeeze(gd1_cfc_11(ypos,:,:)), [2 1]);
pcolor (rx100a, obs_dep, t1); shading flat;
seafloor (rx100a, obs_dep, t1);
set (gca, 'YDir', 'reverse');
axis ([-80 0 0 6e3]);
xlabel ('Longitude [{\circ}E]');
ylabel ('Depth [m]');
caxis ([0 3]);
title ('Observed'); topbox;
pos1 = get(gca, 'Position'); pos1(2) = pos2(2); pos1(4) = pos2(4); set (gca, 'Position', pos1);
paper; orient portrait; set(gcf, 'PaperPosition', [2.1217 8.5763 16.7340 9.5506]);
onom = sprintf('FIGS/ukesm1_PD_NAtl_issues_%s.png', numprint(2, 2));
exportgraphics(gcf, onom, 'Resolution', 600);

% ----------------------------------------------------------------------

ypos = 120;
ysft = 0.075;

figure(1); clf; turbo(60);
subplot (1,2,2);
t1 = permute(squeeze(rg_pd_ant1(ypos,:,:)), [2 1]);
pcolor (rx100a, nemo_mid, t1); shading flat;
seafloor (rx100a, nemo_mid, t1);
set (gca, 'YDir', 'reverse');
axis ([-80 0 0 6e3]);
xlabel ('Longitude [{\circ}E]');
% ylabel ('Depth [m]');
caxis ([0 60]); cb = colorbar('horiz'); cbar_xlabel(cb, 'Anthropogenic CO_{2} [mmol C m^{-3}]');
title ('Simulated'); topbox;
pos2 = get(gca, 'Position'); pos2 = pos2 + [0 ysft 0 -ysft]; cbpos = get(cb, 'Position'); cbpos = cbpos + [0 ysft/3 0 -cbpos(4)/2];
set (gca, 'Position', pos2); set(cb, 'Position', cbpos);
subplot (1,2,1);
t1 = permute(squeeze(gd1_anthro(ypos,:,:)), [2 1]);
pcolor (rx100a, obs_dep, t1); shading flat;
seafloor (rx100a, obs_dep, t1);
set (gca, 'YDir', 'reverse');
axis ([-80 0 0 6e3]);
xlabel ('Longitude [{\circ}E]');
ylabel ('Depth [m]');
caxis ([0 60]);
title ('Observed'); topbox;
pos1 = get(gca, 'Position'); pos1(2) = pos2(2); pos1(4) = pos2(4); set (gca, 'Position', pos1);
paper; orient portrait; set(gcf, 'PaperPosition', [2.1217 8.5763 16.7340 9.5506]);
onom = sprintf('FIGS/ukesm1_PD_NAtl_issues_%s.png', numprint(3, 2));
exportgraphics(gcf, onom, 'Resolution', 600);

figure(1); clf; turbo(60);
subplot (1,2,2);
t1 = permute(squeeze(rg_pd_cfc11(ypos,:,:) * 1e9), [2 1]);
pcolor (rx100a, nemo_mid, t1); shading flat;
seafloor (rx100a, nemo_mid, t1);
set (gca, 'YDir', 'reverse');
axis ([-80 0 0 6e3]);
xlabel ('Longitude [{\circ}E]');
% ylabel ('Depth [m]');
caxis ([0 3]); cb = colorbar('horiz'); cbar_xlabel(cb, 'CFC-11 [nmol m^{-3}]');
title ('Simulated'); topbox;
pos2 = get(gca, 'Position'); pos2 = pos2 + [0 ysft 0 -ysft]; cbpos = get(cb, 'Position'); cbpos = cbpos + [0 ysft/3 0 -cbpos(4)/2];
set (gca, 'Position', pos2); set(cb, 'Position', cbpos);
subplot (1,2,1);
t1 = permute(squeeze(gd1_cfc_11(ypos,:,:)), [2 1]);
pcolor (rx100a, obs_dep, t1); shading flat;
seafloor (rx100a, obs_dep, t1);
set (gca, 'YDir', 'reverse');
axis ([-80 0 0 6e3]);
xlabel ('Longitude [{\circ}E]');
ylabel ('Depth [m]');
caxis ([0 3]);
title ('Observed'); topbox;
pos1 = get(gca, 'Position'); pos1(2) = pos2(2); pos1(4) = pos2(4); set (gca, 'Position', pos1);
paper; orient portrait; set(gcf, 'PaperPosition', [2.1217 8.5763 16.7340 9.5506]);
onom = sprintf('FIGS/ukesm1_PD_NAtl_issues_%s.png', numprint(4, 2));
exportgraphics(gcf, onom, 'Resolution', 600);

% ----------------------------------------------------------------------

ypos = 60;
ysft = 0.075;

figure(1); clf; turbo(60);
subplot (1,2,2);
t1 = permute(squeeze(rg_pd_ant1(ypos,:,:)), [2 1]);
pcolor (rx100a, nemo_mid, t1); shading flat;
seafloor (rx100a, nemo_mid, t1);
set (gca, 'YDir', 'reverse');
axis ([-52 22 0 6e3]);
xlabel ('Longitude [{\circ}E]');
% ylabel ('Depth [m]');
caxis ([0 60]); cb = colorbar('horiz'); cbar_xlabel(cb, 'Anthropogenic CO_{2} [mmol C m^{-3}]');
title ('Simulated'); topbox;
pos2 = get(gca, 'Position'); pos2 = pos2 + [0 ysft 0 -ysft]; cbpos = get(cb, 'Position'); cbpos = cbpos + [0 ysft/3 0 -cbpos(4)/2];
set (gca, 'Position', pos2); set(cb, 'Position', cbpos);
subplot (1,2,1);
t1 = permute(squeeze(gd1_anthro(ypos,:,:)), [2 1]);
pcolor (rx100a, obs_dep, t1); shading flat;
seafloor (rx100a, obs_dep, t1);
set (gca, 'YDir', 'reverse');
axis ([-52 22 0 6e3]);
xlabel ('Longitude [{\circ}E]');
ylabel ('Depth [m]');
caxis ([0 60]);
title ('Observed'); topbox;
pos1 = get(gca, 'Position'); pos1(2) = pos2(2); pos1(4) = pos2(4); set (gca, 'Position', pos1);
paper; orient portrait; set(gcf, 'PaperPosition', [2.1217 8.5763 16.7340 9.5506]);
onom = sprintf('FIGS/ukesm1_PD_NAtl_issues_%s.png', numprint(5, 2));
exportgraphics(gcf, onom, 'Resolution', 600);

figure(1); clf; turbo(60);
subplot (1,2,2);
t1 = permute(squeeze(rg_pd_cfc11(ypos,:,:) * 1e9), [2 1]);
pcolor (rx100a, nemo_mid, t1); shading flat;
seafloor (rx100a, nemo_mid, t1);
set (gca, 'YDir', 'reverse');
axis ([-52 22 0 6e3]);
xlabel ('Longitude [{\circ}E]');
% ylabel ('Depth [m]');
caxis ([0 3]); cb = colorbar('horiz'); cbar_xlabel(cb, 'CFC-11 [nmol m^{-3}]');
title ('Simulated'); topbox;
pos2 = get(gca, 'Position'); pos2 = pos2 + [0 ysft 0 -ysft]; cbpos = get(cb, 'Position'); cbpos = cbpos + [0 ysft/3 0 -cbpos(4)/2];
set (gca, 'Position', pos2); set(cb, 'Position', cbpos);
subplot (1,2,1);
t1 = permute(squeeze(gd1_cfc_11(ypos,:,:)), [2 1]);
pcolor (rx100a, obs_dep, t1); shading flat;
seafloor (rx100a, obs_dep, t1);
set (gca, 'YDir', 'reverse');
axis ([-52 22 0 6e3]);
xlabel ('Longitude [{\circ}E]');
ylabel ('Depth [m]');
caxis ([0 3]);
title ('Observed'); topbox;
pos1 = get(gca, 'Position'); pos1(2) = pos2(2); pos1(4) = pos2(4); set (gca, 'Position', pos1);
paper; orient portrait; set(gcf, 'PaperPosition', [2.1217 8.5763 16.7340 9.5506]);
onom = sprintf('FIGS/ukesm1_PD_NAtl_issues_%s.png', numprint(6, 2));
exportgraphics(gcf, onom, 'Resolution', 600);
