function [a] = regrid_vert(b, odep, ndep);
% REGRID_VERT  Regrids a 3D field to user-specified grid
%
%	>> new = regrid_vert(old,odep,ndep);
%
%	Where,
%
%		new     = new gridded data
%		old     = old gridded data
%		odep	= old depth array (top to bottom)
%		ndep	= new depth array (top to bottom)
%
%	Depth - input depth arrays need to contain either the
%	depth values for each layer if the old data are sections
%	(e.g. the data on level 9 represents a section at 75 m),
%	or the top edge of each cell plus the bottom edge of the
%	bottom cell if the old data are volumes (e.g. the data
%	on level 9 represents the volumn between 70 and 80 m).
%	By comparing the number of levels in the old data with
%	the number of values in the depth array, this macro works
%	out which of these cases you've supplied it with.
%
%	Note : This code is *NOT* protected against you making 
%	stupid mistakes.  And don't be feeding me any NaN-padded 
%	arrays.
%
%   Andrew Yool (axy@noc.ac.uk)

fprintf('\n'); drawnow;

if nargin < 3
	error (' Not enough inputs - try again');
elseif nargin == 3
	% No mask
	mflag = 0;
else
	error (' Too many inputs - try again');
end

% Is depth positive?  If not, try to fix it
if max(odep) == 0, odep = -1 * odep; fprintf('- Fixing negative odep\n'); end
if max(ndep) == 0, ndep = -1 * ndep; fprintf('- Fixing negative ndep\n'); end

% Are either of the depth fields in km rather than m?
if max(odep) < 10, odep = odep * 1e3; fprintf('- Fixing non-metre odep\n'); end
if max(ndep) < 10, ndep = ndep * 1e3; fprintf('- Fixing non-metre ndep\n'); end

% Are depths top to bottom?  If not, try to fix them
if odep(1) > odep(end)
	odep = flipud(odep);
	b = flipdim(b, 3);
  fprintf('- Flipping original array\n'); 
end
if ndep(1) > ndep(end)
	ndep = flipud(ndep);
  fprintf('- Flipping new array\n'); 
end

% Is the data matrix itself top to bottom?  If not, fix it
t1 = b(:,:,2); t2 = isfinite(t1); t3 = sum(sum(t2));
t1 = b(:,:,end-1); t2 = isfinite(t1); t4 = sum(sum(t2));
if t3 < t4
	fprintf(' - Your input matrix appears upside-down, correcting ...'); drawnow;
	b = flipdim(b,3);
end

% Skip the horizontal regridding bit
[sz1, sz2, sz3] = size(b);
grid2d = b;

% Second, work out if its depth levels or layers
clear t1
sz3b = max(size(odep));
if sz3b == sz3
	% It's depth levels - need to average data
    fprintf('- Data are on levels - need to average to layers\n');
	for k = 1:1:(sz3-1)
		% Replace NaNs in the lower level with zeros
		q1 = grid2d(:,:,k+1); q2 = grid2d(:,:,k); q1(isnan(q1)) = q2(isnan(q1));
		t1(:,:,k) = (grid2d(:,:,k) + q1)/2;
	end
	grid2d = t1;
elseif sz3b == (sz3 + 1)
	% It's depth layers - data's fine as it is
    fprintf('- Data are in layers - no need to average to layers\n');
else
	% Mismatch between 3D array and depth array
	error (' Mismatch between 3D array and depth array');
end

% Third, now calculate overlaps in depth dimension
overlapd = regrid_dep(odep, ndep);

% Fourth, apply depth overlap array to create new grid
fprintf(' - Regriding data vertically\n   Completed level ');
[sz1b, sz2b] = size(grid2d(:,:,1));
sz4 = max(size(ndep));
for k = 1:1:(sz4-1)
	clear t1
	t1(1:1:sz1b,1:1:sz2b) = 0;
	for k2 = 1:1:(sz3b-1)
		q1 = overlapd(k,k2);
		if q1 > 0
			q2 = grid2d(:,:,k2);
			t1(:,:) = t1(:,:) + (q1*q2);
		end	
	end
	t2 = t1 / (sum(overlapd(k,:)));
	grid3d(:,:,k) = t2;
	fprintf('%d ',k); if mod(k,10) == 0, fprintf('\n                   '); end; drawnow;
end
fprintf('\n');

% Apply mask if required
if mflag == 1
	fprintf(' - Applying mask to data\n   Completed level ');
	t1(:,:,1) = grid3d(:,:,1) + mask(:,:,1);
	for k = 2:1:(sz4-1)
		t2 = grid3d(:,:,k); t3 = mask(:,:,k);
		t4 = isnan(t2); t5 = isnan(t3)*2;
		t6 = t4 + t5;
		t7 = grid3d(:,:,k) + mask(:,:,k);
		t8 = t1(:,:,k-1);
		t7(t6 == 1) = t8(t6 == 1);
		t1(:,:,k) = t7;
		fprintf('%d ',k); if mod(k,10) == 0, fprintf('\n                   '); end; drawnow;
	end
	grid3d = t1;
	fprintf('\n');
	
	clear t1 t2 t3 t4
	% Prepare replacement field for integrity check
	for k = 1:1:(sz4-1)
		% Construct a grid with a wrap-around border
		t1(2:1:(sz1b+1),2:1:(sz2b+1)) = grid3d(1:sz1b,1:sz2b,k);
		t1(1,:) = NaN; t1(sz1b+2,:) = NaN;
		t1(2:1:(sz1b+1),1) = t1(1:sz1b,sz2b);
		t1(2:1:(sz1b+1),sz2b+2) = t1(1:sz1b,1);
		% Construct an array storing values of neighbouring cells
		t2(1:sz1b,1:sz2b,1) = t1(3:1:(sz1b+2),2:1:(sz2b+1));
		t2(1:sz1b,1:sz2b,2) = t1(2:1:(sz1b+1),3:1:(sz2b+2));
		t2(1:sz1b,1:sz2b,3) = t1(1:1:sz1b,2:1:(sz2b+1));
		t2(1:sz1b,1:sz2b,4) = t1(2:1:(sz1b+1),1:1:sz2b);
		% Determine how many neighbours are finite
		t3 = isfinite(t2); t4 = sum(t3, 3); t4(t4 == 0) = NaN;
		% Sum neighbours to calculate potential replacement value
		t2(isnan(t2)) = 0; t5 = (sum(t2, 3)) ./ t4;
		replace(:,:,k) = t5;
	end

	fprintf(' - Checking new grid for missing cells\n'); drawnow;
	% Check mask for integrity
	for k = 1:1:(sz4-1)
		t1 = grid3d(:,:,k); t1(isfinite(t1)) = 0; t1(isnan(t1)) = 1;
		t2 = mask(:,:,k); t2(isnan(t2)) = 2;
		t3 = t1 + t2;
		% Look for cells where the mask is sea but the new grid land
		t4 = (t3 == 1);
		totmiss = sum(sum(t4));
		if totmiss > 0, fprintf('   Layer %d has %d missing cells (fixed)\n',k,totmiss); drawnow; end
		% Replace missing cells with calculated replacement value
		t5 = grid3d(:,:,k); t6 = replace(:,:,k);
		t5(t4) = replace(t4);
		grid3d(:,:,k) = t5(:,:);
	end
end

fprintf(' - Regriding complete\n');
a = grid3d;

fprintf('\n'); drawnow;
