#!/usr/bin/perl
# Code based on Onno's isochrone grid code
# The data has been updated to the newer version of the Lejeune library
# using code provided by Peter Anders.
use strict;
use Evert::Misc;
my $nzgr =  5;
my $ntgr = 68;
my $nggr = 19;

my @zgr;
my @tgr;
my @ggr;
my @ubv;

my @a = (1, 2, 3, 4, 6, 8);
my $n;

read_ubvi();
my ($bcsun) = tgz_to_ubv(log(5770)/log(10), 4.43, 0.0);

my $logt = 3.6772;
my $logl = -0.6157;
my $logg = 3.0;
my $z;

if ($#ARGV < 2) {
   print "Need log L, log T and log g (optionally Z)\n";
   print "Returns Mv, B-V, U-B, V-R, V-I\n";
   exit;
}

($logl, $logt, $logg, $z) = @ARGV;

my @a = lt2ubv($logl, $logt, $logg, $z);
print "@a\n";

exit;

my @index_array;
sub indx
{
   my $x = shift;
   my @x = @index_array;
   my $nx = $#x;

   my $hi = $nx+1;
   my $lo = 0;
   
   my $sx = $x[-1] - $x[0];
   
   while ($hi>$lo+1) {
      my $k = int(($hi+$lo)/2);
      if ( ($x - $x[$k])*$sx < 0) {
         $hi = $k;
      } else {
         $lo = $k;
      }
   }
   
   return max(0, min($hi, $nx));
}

# Convert log L, log T and M to UBV colours
# A fourth argument, Z, is optional [default: solar]
sub lt2ubv
{
   my $logl = shift;
   my $logt = shift;
   my $logg = shift;
   my $z = shift;
   my $logz = 0;
   
   if (defined($z)) {
      $logz = log($z/0.02)/log(10);
   }
   
   my $mbol = 4.74 - 2.5*$logl;
   my @ubvri = tgz_to_ubv($logt, $logg, $logz);

   # All calibrations were done with respect to Vega, which turns out to be
   # a horrible star to use for calibrations due to the fact that it is
   # rapidly rotating and seen nearly pole-on. This means all of our
   # bolometric corrections will be wrong, even if the colours are probably
   # fine.
   # So we apply an extra shift for the BC so that we will get the
   # "correct" value (MV=4.81, or BC=-0.07) for the Sun.
   my $mv = $mbol - $ubvri[0] + $bcsun + 0.07;
   
   return ($mv, $ubvri[2], $ubvri[1], $ubvri[3], $ubvri[4]);
}

# Input: log T, log g, log Z
# Returns: Array with (Bol.C, U-B, B-V, V-R, V-I)
sub tgz_to_ubv
{
   my $logt = shift;
   my $logg = shift;
   my $logz = shift;
   my @cm;

   @index_array = @ggr;
   my $ig = indx($logg,[ @ggr ]);
   @index_array = @tgr;
   my $it = indx($logt,[ @tgr ]);
   @index_array = @zgr;
   my $iz = indx($logz,[ @zgr ]);
   
   #print "$it $tgr[$it] $logt $tgr[$it+1]\n";
   #print "$ig $ggr[$ig] $logg $ggr[$ig+1]\n";
   #print "$iz $zgr[$iz] $logz $zgr[$iz+1]\n";

   my $dg1 = ($logg - $ggr[$ig-1])/($ggr[$ig] - $ggr[$ig-1]);
   $dg1 = max(0.0, min(1.0, $dg1));
   my $dg2 = 1.0 - $dg1;
   my $dt1 = ($logt - $tgr[$it-1])/($tgr[$it] - $tgr[$it-1]);
   my $dt2 = 1.0 - $dt1;
   my $dz1 = ($logz - $zgr[$iz-1])/($zgr[$iz] - $zgr[$iz-1]);
   my $dz2 = 1.0 - $dz1;
   my $igg1 = $ggr[$ig];
   my $igg2 = $ggr[$ig-1];

   foreach my $k (0..4) {
      $cm[$k] = (($ubv[$iz][$it][$ig][$k]*$dg1 + $ubv[$iz][$it][$ig-1][$k]*$dg2)*$dt1 +
               ($ubv[$iz][$it-1][$ig][$k]*$dg1 + $ubv[$iz][$it-1][$ig-1][$k]*$dg2)*$dt2)*$dz1
               + 
              (($ubv[$iz-1][$it][$ig][$k]*$dg1 + $ubv[$iz-1][$it][$ig-1][$k]*$dg2)*$dt1 +
               ($ubv[$iz-1][$it-1][$ig][$k]*$dg1 + $ubv[$iz-1][$it-1][$ig-1][$k]*$dg2)*$dt2)*$dz2;
   }
   
   $cm[4] += $cm[3];
   
   # Bol.C, U-B, B-V, V-R, V-I
   return @cm;
}

sub read_ubvi
{
   my @ubvri_lbc = qw(
-1.50  2000 -1.02 -7.05491 -0.29783  1.12030  3.74948  3.02679
-1.50  2000 -0.70 -7.05491 -0.29783  1.12030  3.74948  3.02679
-1.50  2000 -0.51 -7.05491 -0.29783  1.12030  3.74948  3.02679
-1.50  2000 -0.29 -7.05491 -0.29783  1.12030  3.74948  3.02679
-1.50  2000  0.00 -7.05491 -0.29783  1.12030  3.74948  3.02679
-1.50  2000  0.28 -7.05491 -0.29783  1.12030  3.74948  3.02679
-1.50  2000  0.50 -7.05491 -0.29783  1.12030  3.74948  3.02679
-1.50  2000  0.60 -7.05491 -0.29783  1.12030  3.74948  3.02679
-1.50  2000  0.87 -7.05491 -0.29783  1.12030  3.74948  3.02679
-1.50  2000  1.00 -7.05491 -0.29783  1.12030  3.74948  3.02679
-1.50  2000  1.50 -7.05491 -0.29783  1.12030  3.74948  3.02679
-1.50  2000  2.00 -7.05491 -0.29783  1.12030  3.74948  3.02679
-1.50  2000  2.50 -7.05491 -0.29783  1.12030  3.74948  3.02679
-1.50  2000  3.00 -7.05491 -0.29783  1.12030  3.74948  3.02679
-1.50  2000  3.50 -7.05491 -0.29783  1.12030  3.74948  3.02679
-1.50  2000  4.00 -7.19603  0.30737  1.98438  3.83970  3.00860
-1.50  2000  4.50 -7.36220  1.35906  2.55481  3.97758  2.99233
-1.50  2000  5.00 -7.45940  2.69098  2.84060  4.13058  2.93958
-1.50  2000  5.50 -7.37559  3.82038  2.96376  4.22386  2.83140
-1.50  2200 -1.02 -5.74734  0.00490  1.08443  3.18965  2.78974
-1.50  2200 -0.70 -5.74734  0.00490  1.08443  3.18965  2.78974
-1.50  2200 -0.51 -5.74734  0.00490  1.08443  3.18965  2.78974
-1.50  2200 -0.29 -5.74734  0.00490  1.08443  3.18965  2.78974
-1.50  2200  0.00 -5.74734  0.00490  1.08443  3.18965  2.78974
-1.50  2200  0.28 -5.74734  0.00490  1.08443  3.18965  2.78974
-1.50  2200  0.50 -5.74734  0.00490  1.08443  3.18965  2.78974
-1.50  2200  0.60 -5.74734  0.00490  1.08443  3.18965  2.78974
-1.50  2200  0.87 -5.74734  0.00490  1.08443  3.18965  2.78974
-1.50  2200  1.00 -5.74734  0.00490  1.08443  3.18965  2.78974
-1.50  2200  1.50 -5.74734  0.00490  1.08443  3.18965  2.78974
-1.50  2200  2.00 -5.74734  0.00490  1.08443  3.18965  2.78974
-1.50  2200  2.50 -5.74734  0.00490  1.08443  3.18965  2.78974
-1.50  2200  3.00 -5.74734  0.00490  1.08443  3.18965  2.78974
-1.50  2200  3.50 -5.74734  0.00490  1.08443  3.18965  2.78974
-1.50  2200  4.00 -5.83881  0.47459  1.74295  3.24982  2.77385
-1.50  2200  4.50 -6.03314  1.22234  2.23193  3.37866  2.79895
-1.50  2200  5.00 -6.18971  2.28050  2.56496  3.53287  2.77720
-1.50  2200  5.50 -6.26440  3.40841  2.73528  3.68918  2.70344
-1.50  2500 -1.02 -5.99423 -0.23091  1.41747  3.08183  2.87080
-1.50  2500 -0.70 -5.60200 -0.23088  1.40344  2.89877  2.78610
-1.50  2500 -0.51 -5.37788 -0.23088  1.39685  2.79747  2.73588
-1.50  2500 -0.29 -5.11838 -0.23088  1.38922  2.68017  2.67773
-1.50  2500  0.00 -4.94869 -0.23089  1.38976  2.66402  2.63922
-1.50  2500  0.28 -4.78486 -0.23089  1.39029  2.64842  2.60205
-1.50  2500  0.50 -4.76560 -0.18071  1.37907  2.64935  2.59194
-1.50  2500  0.60 -4.75684 -0.15790  1.37396  2.64977  2.58735
-1.50  2500  0.87 -4.73319 -0.09632  1.36019  2.65092  2.57496
-1.50  2500  1.00 -4.72181 -0.06667  1.35355  2.65147  2.56899
-1.50  2500  1.50 -4.67802  0.04737  1.32804  2.65358  2.54603
-1.50  2500  2.00 -4.63424  0.16141  1.30253  2.65570  2.52308
-1.50  2500  2.50 -4.59045  0.27545  1.27702  2.65782  2.50012
-1.50  2500  3.00 -4.54667  0.38949  1.25151  2.65993  2.47717
-1.50  2500  3.50 -4.50288  0.50353  1.22600  2.66205  2.45421
-1.50  2500  4.00 -4.48830  0.86898  1.63556  2.68648  2.40440
-1.50  2500  4.50 -4.57277  1.35470  1.95371  2.74300  2.41254
-1.50  2500  5.00 -4.71326  2.01739  2.21663  2.84811  2.42855
-1.50  2500  5.50 -4.90478  2.87751  2.40505  3.01804  2.42611
-1.50  2800 -1.02 -5.77601  0.06061  1.45009  2.93050  2.86379
-1.50  2800 -0.70 -5.43583  0.06067  1.44188  2.80713  2.76515
-1.50  2800 -0.51 -5.23440  0.06063  1.43772  2.73494  2.70608
-1.50  2800 -0.29 -5.00117  0.06059  1.43290  2.65135  2.63769
-1.50  2800  0.00 -4.74361  0.06062  1.43026  2.58145  2.56286
-1.50  2800  0.28 -4.49493  0.06065  1.42772  2.51396  2.49061
-1.50  2800  0.50 -4.31873  0.06069  1.42664  2.47409  2.44065
-1.50  2800  0.60 -4.23864  0.06072  1.42615  2.45597  2.41793
-1.50  2800  0.87 -4.17625  0.12522  1.42588  2.44120  2.38421
-1.50  2800  1.00 -4.14622  0.15628  1.42575  2.43408  2.36797
-1.50  2800  1.50 -4.03069  0.27573  1.42524  2.40672  2.30551
-1.50  2800  2.00 -3.91516  0.39519  1.42474  2.37937  2.24306
-1.50  2800  2.50 -3.79963  0.51464  1.42423  2.35201  2.18060
-1.50  2800  3.00 -3.68411  0.63410  1.42373  2.32465  2.11815
-1.50  2800  3.50 -3.56858  0.75355  1.42323  2.29729  2.05569
-1.50  2800  4.00 -3.51492  0.96750  1.53825  2.26781  2.07231
-1.50  2800  4.50 -3.49497  1.26795  1.76250  2.28497  2.04218
-1.50  2800  5.00 -3.54027  1.67907  1.93425  2.32187  2.05317
-1.50  2800  5.50 -3.68272  2.24496  2.10360  2.43354  2.07241
-1.50  3000 -1.02 -5.47382  0.36024  1.40529  2.75583  2.84762
-1.50  3000 -0.70 -5.47382  0.36024  1.40529  2.75583  2.84762
-1.50  3000 -0.51 -5.28739  0.36026  1.40347  2.69997  2.78041
-1.50  3000 -0.29 -5.07153  0.36028  1.40136  2.63529  2.70259
-1.50  3000  0.00 -4.80249  0.36034  1.39931  2.54921  2.61283
-1.50  3000  0.28 -4.54273  0.36040  1.39732  2.46610  2.52617
-1.50  3000  0.50 -4.35102  0.36039  1.39610  2.39962  2.46743
-1.50  3000  0.60 -4.26388  0.36039  1.39555  2.36940  2.44073
-1.50  3000  0.87 -4.15648  0.40222  1.38471  2.33666  2.39391
-1.50  3000  1.00 -4.10477  0.42236  1.37950  2.32090  2.37137
-1.50  3000  1.50 -3.90588  0.49983  1.35943  2.26027  2.28466
-1.50  3000  2.00 -3.70700  0.57730  1.33937  2.19964  2.19795
-1.50  3000  2.50 -3.50811  0.65477  1.31931  2.13902  2.11125
-1.50  3000  3.00 -3.30922  0.73224  1.29925  2.07839  2.02454
-1.50  3000  3.50 -3.11034  0.80971  1.27919  2.01776  1.93783
-1.50  3000  4.00 -2.90552  0.93787  1.48985  1.97737  1.83904
-1.50  3000  4.50 -2.83379  1.12392  1.63701  1.96749  1.80477
-1.50  3000  5.00 -2.85733  1.42376  1.74949  1.98494  1.82639
-1.50  3000  5.50 -2.91549  1.80241  1.89095  2.04140  1.83255
-1.50  3200 -1.02 -4.87297  0.92479  1.40658  2.68163  2.58168
-1.50  3200 -0.70 -4.87297  0.92479  1.40658  2.68163  2.58168
-1.50  3200 -0.51 -4.65718  0.92484  1.40395  2.53605  2.53273
-1.50  3200 -0.29 -4.40743  0.92479  1.40102  2.37116  2.47323
-1.50  3200  0.00 -4.10542  0.92479  1.39755  2.18036  2.39733
-1.50  3200  0.28 -3.81383  0.92478  1.39420  1.99615  2.32405
-1.50  3200  0.50 -3.63182  0.92478  1.39169  1.88646  2.27898
-1.50  3200  0.60 -3.54908  0.92478  1.39055  1.83661  2.25850
-1.50  3200  0.87 -3.47284  0.92510  1.36337  1.83084  2.21901
-1.50  3200  1.00 -3.43612  0.92526  1.35029  1.82806  2.19999
-1.50  3200  1.50 -3.29493  0.92586  1.29996  1.81738  2.12687
-1.50  3200  2.00 -3.15373  0.92647  1.24963  1.80670  2.05374
-1.50  3200  2.50 -3.01254  0.92707  1.19930  1.79602  1.98061
-1.50  3200  3.00 -2.87134  0.92767  1.14897  1.78533  1.90748
-1.50  3200  3.50 -2.73015  0.92828  1.09865  1.77465  1.83435
-1.50  3200  4.00 -2.41408  0.97196  1.32609  1.71272  1.68227
-1.50  3200  4.50 -2.27806  1.07727  1.47048  1.68868  1.61161
-1.50  3200  5.00 -2.23010  1.24630  1.58851  1.68878  1.58464
-1.50  3200  5.50 -2.23763  1.49731  1.70081  1.71259  1.58287
-1.50  3350 -1.02 -3.70715  1.27649  1.52669  2.17898  2.15013
-1.50  3350 -0.70 -3.70715  1.27649  1.52669  2.17898  2.15013
-1.50  3350 -0.51 -3.70715  1.27649  1.52669  2.17898  2.15013
-1.50  3350 -0.29 -3.53199  1.27643  1.51682  2.05070  2.10716
-1.50  3350  0.00 -3.33223  1.27647  1.50696  1.90831  2.05377
-1.50  3350  0.28 -3.13937  1.27651  1.49743  1.77084  2.00222
-1.50  3350  0.50 -3.12220  1.27648  1.49498  1.74974  2.00430
-1.50  3350  0.60 -3.11440  1.27647  1.49387  1.74015  2.00524
-1.50  3350  0.87 -3.09301  1.27651  1.49104  1.71444  2.00711
-1.50  3350  1.00 -3.05528  1.26360  1.46738  1.70672  1.99096
-1.50  3350  1.50 -2.91017  1.21392  1.37640  1.67702  1.92887
-1.50  3350  2.00 -2.76507  1.16425  1.28542  1.64733  1.86677
-1.50  3350  2.50 -2.61996  1.11457  1.19444  1.61763  1.80468
-1.50  3350  3.00 -2.47486  1.06490  1.10345  1.58794  1.74259
-1.50  3350  3.50 -2.32975  1.01523  1.01247  1.55824  1.68049
-1.50  3350  4.00 -2.07416  0.99846  1.25525  1.53782  1.53650
-1.50  3350  4.50 -1.88421  1.03459  1.39876  1.50676  1.43530
-1.50  3350  5.00 -1.80441  1.13343  1.50725  1.49931  1.38897
-1.50  3350  5.50 -1.78413  1.29917  1.60987  1.51129  1.37300
-1.50  3500 -1.02 -2.42775  1.88359  1.77156  1.56979  1.77381
-1.50  3500 -0.70 -2.42775  1.88359  1.77156  1.56979  1.77381
-1.50  3500 -0.51 -2.42775  1.88359  1.77156  1.56979  1.77381
-1.50  3500 -0.29 -2.42775  1.88359  1.77156  1.56979  1.77381
-1.50  3500  0.00 -2.42775  1.88359  1.77156  1.56979  1.77381
-1.50  3500  0.28 -2.43414  1.76440  1.70145  1.54976  1.78896
-1.50  3500  0.50 -2.43916  1.67076  1.64637  1.53403  1.80086
-1.50  3500  0.60 -2.44477  1.62737  1.62194  1.52936  1.80697
-1.50  3500  0.87 -2.45990  1.51022  1.55601  1.51675  1.82346
-1.50  3500  1.00 -2.46718  1.45381  1.52426  1.51068  1.83140
-1.50  3500  1.50 -2.50553  1.24714  1.40690  1.49750  1.86228
-1.50  3500  2.00 -2.54910  1.06512  1.29820  1.49249  1.89107
-1.50  3500  2.50 -2.59060  0.92040  1.20629  1.49472  1.91447
-1.50  3500  3.00 -1.81318  1.18365  1.34959  1.26151  1.45418
-1.50  3500  3.50 -1.65651  1.12467  1.38840  1.26877  1.38174
-1.50  3500  4.00 -1.48929  1.08922  1.43312  1.28129  1.29883
-1.50  3500  4.50 -1.41800  1.08681  1.46480  1.30089  1.25386
-1.50  3500  5.00 -1.39016  1.12450  1.50328  1.32803  1.22318
-1.50  3500  5.50 -1.39016  1.12450  1.50328  1.32803  1.22318
-1.50  3750 -1.02 -1.36398  2.30326  1.76094  1.31285  1.08092
-1.50  3750 -0.70 -1.36398  2.30326  1.76094  1.31285  1.08092
-1.50  3750 -0.51 -1.36398  2.30326  1.76094  1.31285  1.08092
-1.50  3750 -0.29 -1.36398  2.30326  1.76094  1.31285  1.08092
-1.50  3750  0.00 -1.36398  2.30326  1.76094  1.31285  1.08092
-1.50  3750  0.28 -1.35620  2.18002  1.71310  1.29809  1.08262
-1.50  3750  0.50 -1.35009  2.08318  1.67552  1.28649  1.08395
-1.50  3750  0.60 -1.34973  2.04079  1.65988  1.28227  1.08609
-1.50  3750  0.87 -1.34878  1.92633  1.61765  1.27087  1.09184
-1.50  3750  1.00 -1.34832  1.87122  1.59732  1.26538  1.09461
-1.50  3750  1.50 -1.35377  1.67709  1.52372  1.24673  1.11042
-1.50  3750  2.00 -1.36668  1.50651  1.45590  1.23111  1.13033
-1.50  3750  2.50 -1.38435  1.36371  1.39491  1.21930  1.15145
-1.50  3750  3.00 -1.20872  1.49917  1.43666  1.10665  1.08403
-1.50  3750  3.50 -1.21269  1.40565  1.41288  1.11225  1.09133
-1.50  3750  4.00 -1.15455  1.30916  1.42289  1.13532  1.06323
-1.50  3750  4.50 -1.08230  1.23087  1.43023  1.15603  1.02768
-1.50  3750  5.00 -1.04779  1.19235  1.43193  1.17145  1.00799
-1.50  3750  5.50 -1.04779  1.19235  1.43193  1.17145  1.00799
-1.50  4000 -1.02 -0.81711  2.24883  1.62711  1.05713  0.80762
-1.50  4000 -0.70 -0.81711  2.24883  1.62711  1.05713  0.80762
-1.50  4000 -0.51 -0.81711  2.24883  1.62711  1.05713  0.80762
-1.50  4000 -0.29 -0.81711  2.24883  1.62711  1.05713  0.80762
-1.50  4000  0.00 -0.81711  2.24883  1.62711  1.05713  0.80762
-1.50  4000  0.28 -0.80948  2.13357  1.58615  1.04999  0.80214
-1.50  4000  0.50 -0.80348  2.04300  1.55397  1.04439  0.79784
-1.50  4000  0.60 -0.80254  2.00487  1.54119  1.04320  0.79685
-1.50  4000  0.87 -0.79998  1.90191  1.50670  1.04001  0.79416
-1.50  4000  1.00 -0.79875  1.85234  1.49009  1.03847  0.79287
-1.50  4000  1.50 -0.79964  1.68481  1.43465  1.03559  0.79193
-1.50  4000  2.00 -0.80641  1.54390  1.38810  1.03535  0.79514
-1.50  4000  2.50 -0.81585  1.43067  1.34834  1.03518  0.80165
-1.50  4000  3.00 -0.74611  1.51124  1.38344  1.02624  0.82268
-1.50  4000  3.50 -0.76105  1.44359  1.36199  1.02891  0.83365
-1.50  4000  4.00 -0.76942  1.37927  1.35399  1.03899  0.84140
-1.50  4000  4.50 -0.75066  1.28899  1.35842  1.05812  0.83639
-1.50  4000  5.00 -0.71849  1.20205  1.35438  1.07079  0.82647
-1.50  4000  5.50 -0.71849  1.20205  1.35438  1.07079  0.82647
-1.50  4250 -1.02 -0.50486  1.97392  1.50808  0.92569  0.66519
-1.50  4250 -0.70 -0.50486  1.97392  1.50808  0.92569  0.66519
-1.50  4250 -0.51 -0.50486  1.97392  1.50808  0.92569  0.66519
-1.50  4250 -0.29 -0.50486  1.97392  1.50808  0.92569  0.66519
-1.50  4250  0.00 -0.50486  1.97392  1.50808  0.92569  0.66519
-1.50  4250  0.28 -0.49749  1.88045  1.47092  0.91762  0.65995
-1.50  4250  0.50 -0.49170  1.80702  1.44172  0.91129  0.65583
-1.50  4250  0.60 -0.49077  1.77542  1.43009  0.91002  0.65453
-1.50  4250  0.87 -0.48826  1.69012  1.39871  0.90660  0.65102
-1.50  4250  1.00 -0.48706  1.64905  1.38359  0.90495  0.64933
-1.50  4250  1.50 -0.48764  1.50816  1.33228  0.90467  0.64477
-1.50  4250  2.00 -0.49300  1.39226  1.28864  0.90887  0.64211
-1.50  4250  2.50 -0.50182  1.30467  1.25352  0.91626  0.64122
-1.50  4250  3.00 -0.48045  1.30400  1.27958  0.86540  0.68609
-1.50  4250  3.50 -0.49404  1.26690  1.26321  0.87686  0.68796
-1.50  4250  4.00 -0.50959  1.24689  1.25639  0.89032  0.69136
-1.50  4250  4.50 -0.52164  1.22208  1.25910  0.90622  0.69656
-1.50  4250  5.00 -0.51770  1.15826  1.26315  0.92147  0.70095
-1.50  4250  5.50 -0.51770  1.15826  1.26315  0.92147  0.70095
-1.50  4500 -1.02 -0.31546  1.67857  1.43260  0.77413  0.61807
-1.50  4500 -0.70 -0.31546  1.67857  1.43260  0.77413  0.61807
-1.50  4500 -0.51 -0.31546  1.67857  1.43260  0.77413  0.61807
-1.50  4500 -0.29 -0.31546  1.67857  1.43260  0.77413  0.61807
-1.50  4500  0.00 -0.31546  1.67857  1.43260  0.77413  0.61807
-1.50  4500  0.28 -0.30914  1.61088  1.40232  0.76447  0.61412
-1.50  4500  0.50 -0.30418  1.55770  1.37853  0.75687  0.61102
-1.50  4500  0.60 -0.30317  1.53405  1.36887  0.75482  0.61006
-1.50  4500  0.87 -0.30044  1.47018  1.34279  0.74929  0.60746
-1.50  4500  1.00 -0.29913  1.43943  1.33024  0.74663  0.60621
-1.50  4500  1.50 -0.29755  1.33027  1.28676  0.74169  0.60249
-1.50  4500  2.00 -0.30069  1.23677  1.24817  0.74194  0.59996
-1.50  4500  2.50 -0.30678  1.16343  1.21420  0.74604  0.59806
-1.50  4500  3.00 -0.31611  1.11301  1.18680  0.75444  0.59689
-1.50  4500  3.50 -0.32752  1.08681  1.16767  0.76603  0.59615
-1.50  4500  4.00 -0.34123  1.08078  1.15756  0.78101  0.59591
-1.50  4500  4.50 -0.35643  1.09000  1.15606  0.79832  0.59629
-1.50  4500  5.00 -0.36921  1.09634  1.16328  0.81667  0.59878
-1.50  4500  5.50 -0.36921  1.09634  1.16328  0.81667  0.59878
-1.50  4750 -1.02 -0.14345  1.33561  1.26815  0.68048  0.52180
-1.50  4750 -0.70 -0.14345  1.33561  1.26815  0.68048  0.52180
-1.50  4750 -0.51 -0.14345  1.33561  1.26815  0.68048  0.52180
-1.50  4750 -0.29 -0.14345  1.33561  1.26815  0.68048  0.52180
-1.50  4750  0.00 -0.14345  1.33561  1.26815  0.68048  0.52180
-1.50  4750  0.28 -0.13935  1.28354  1.24279  0.67061  0.51846
-1.50  4750  0.50 -0.13613  1.24263  1.22286  0.66285  0.51583
-1.50  4750  0.60 -0.13549  1.22415  1.21508  0.66067  0.51514
-1.50  4750  0.87 -0.13377  1.17426  1.19406  0.65481  0.51328
-1.50  4750  1.00 -0.13294  1.15024  1.18395  0.65198  0.51238
-1.50  4750  1.50 -0.13244  1.06993  1.14969  0.64445  0.51005
-1.50  4750  2.00 -0.13397  1.00112  1.11907  0.64043  0.50842
-1.50  4750  2.50 -0.13781  0.94543  1.09133  0.63980  0.50730
-1.50  4750  3.00 -0.14400  0.90653  1.06701  0.64288  0.50652
-1.50  4750  3.50 -0.15229  0.88525  1.04681  0.64987  0.50562
-1.50  4750  4.00 -0.16262  0.88268  1.03246  0.66081  0.50499
-1.50  4750  4.50 -0.17474  0.89653  1.02533  0.67525  0.50431
-1.50  4750  5.00 -0.18784  0.92062  1.02578  0.69231  0.50396
-1.50  4750  5.50 -0.18784  0.92062  1.02578  0.69231  0.50396
-1.50  5000 -1.02 -0.02696  1.00241  1.09667  0.62653  0.46835
-1.50  5000 -0.70 -0.02696  1.00241  1.09667  0.62653  0.46835
-1.50  5000 -0.51 -0.02696  1.00241  1.09667  0.62653  0.46835
-1.50  5000 -0.29 -0.02696  1.00241  1.09667  0.62653  0.46835
-1.50  5000  0.00 -0.02696  1.00241  1.09667  0.62653  0.46835
-1.50  5000  0.28 -0.02570  0.95702  1.07660  0.61905  0.46535
-1.50  5000  0.50 -0.02471  0.92136  1.06084  0.61318  0.46300
-1.50  5000  0.60 -0.02474  0.90596  1.05446  0.61122  0.46235
-1.50  5000  0.87 -0.02484  0.86436  1.03721  0.60593  0.46059
-1.50  5000  1.00 -0.02489  0.84434  1.02891  0.60338  0.45974
-1.50  5000  1.50 -0.02601  0.78072  1.00252  0.59556  0.45749
-1.50  5000  2.00 -0.02881  0.72521  0.97961  0.59119  0.45661
-1.50  5000  2.50 -0.03195  0.68347  0.95989  0.58832  0.45580
-1.50  5000  3.00 -0.03629  0.65445  0.94206  0.58799  0.45529
-1.50  5000  3.50 -0.04144  0.63957  0.92633  0.59009  0.45467
-1.50  5000  4.00 -0.04793  0.64003  0.91447  0.59569  0.45408
-1.50  5000  4.50 -0.05583  0.65505  0.90637  0.60477  0.45331
-1.50  5000  5.00 -0.06531  0.68078  0.90268  0.61757  0.45267
-1.50  5000  5.50 -0.06531  0.68078  0.90268  0.61757  0.45267
-1.50  5250 -1.02  0.04084  0.74138  0.93305  0.58009  0.43356
-1.50  5250 -0.70  0.04084  0.74138  0.93305  0.58009  0.43356
-1.50  5250 -0.51  0.04084  0.74138  0.93305  0.58009  0.43356
-1.50  5250 -0.29  0.04084  0.74138  0.93305  0.58009  0.43356
-1.50  5250  0.00  0.04084  0.74138  0.93305  0.58009  0.43356
-1.50  5250  0.28  0.03905  0.70467  0.91877  0.57553  0.43113
-1.50  5250  0.50  0.03764  0.67583  0.90755  0.57194  0.42921
-1.50  5250  0.60  0.03684  0.66251  0.90274  0.57069  0.42868
-1.50  5250  0.87  0.03466  0.62652  0.88974  0.56732  0.42723
-1.50  5250  1.00  0.03361  0.60920  0.88349  0.56570  0.42653
-1.50  5250  1.50  0.02944  0.55304  0.86344  0.56050  0.42446
-1.50  5250  2.00  0.02458  0.50233  0.84617  0.55732  0.42362
-1.50  5250  2.50  0.02040  0.46491  0.83270  0.55469  0.42301
-1.50  5250  3.00  0.01629  0.43875  0.82168  0.55343  0.42278
-1.50  5250  3.50  0.01224  0.42599  0.81260  0.55355  0.42253
-1.50  5250  4.00  0.00820  0.42726  0.80580  0.55567  0.42193
-1.50  5250  4.50  0.00378  0.44141  0.80091  0.56037  0.42132
-1.50  5250  5.00 -0.00135  0.46719  0.79926  0.56812  0.42059
-1.50  5250  5.50 -0.00135  0.46719  0.79926  0.56812  0.42059
-1.50  5500 -1.02  0.11573  0.57799  0.76998  0.51277  0.39728
-1.50  5500 -0.70  0.11573  0.57799  0.76998  0.51277  0.39728
-1.50  5500 -0.51  0.11573  0.57799  0.76998  0.51277  0.39728
-1.50  5500 -0.29  0.11573  0.57799  0.76998  0.51277  0.39728
-1.50  5500  0.00  0.11573  0.57799  0.76998  0.51277  0.39728
-1.50  5500  0.28  0.10903  0.53782  0.77021  0.51665  0.39615
-1.50  5500  0.50  0.10376  0.50625  0.77039  0.51971  0.39525
-1.50  5500  0.60  0.10227  0.49494  0.76724  0.51925  0.39483
-1.50  5500  0.87  0.09827  0.46439  0.75875  0.51800  0.39367
-1.50  5500  1.00  0.09634  0.44969  0.75466  0.51740  0.39311
-1.50  5500  1.50  0.08934  0.39674  0.74237  0.51588  0.39179
-1.50  5500  2.00  0.08262  0.35115  0.73153  0.51418  0.39100
-1.50  5500  2.50  0.07633  0.31306  0.72356  0.51357  0.39080
-1.50  5500  3.00  0.07092  0.28436  0.71731  0.51304  0.39078
-1.50  5500  3.50  0.06645  0.26783  0.71319  0.51297  0.39046
-1.50  5500  4.00  0.06314  0.26474  0.71068  0.51371  0.39018
-1.50  5500  4.50  0.06058  0.27370  0.71048  0.51616  0.38978
-1.50  5500  5.00  0.05838  0.29569  0.71259  0.52068  0.38919
-1.50  5500  5.50  0.05838  0.29569  0.71259  0.52068  0.38919
-1.50  5750 -1.02  0.18889  0.47545  0.62420  0.42734  0.35408
-1.50  5750 -0.70  0.18889  0.47545  0.62420  0.42734  0.35408
-1.50  5750 -0.51  0.18889  0.47545  0.62420  0.42734  0.35408
-1.50  5750 -0.29  0.18889  0.47545  0.62420  0.42734  0.35408
-1.50  5750  0.00  0.18889  0.47545  0.62420  0.42734  0.35408
-1.50  5750  0.28  0.17945  0.43634  0.62475  0.43409  0.35374
-1.50  5750  0.50  0.17204  0.40562  0.62518  0.43939  0.35348
-1.50  5750  0.60  0.16895  0.39090  0.62801  0.44232  0.35346
-1.50  5750  0.87  0.16061  0.35116  0.63563  0.45024  0.35340
-1.50  5750  1.00  0.15660  0.33203  0.63930  0.45405  0.35338
-1.50  5750  1.50  0.14688  0.28309  0.63229  0.45512  0.35226
-1.50  5750  2.00  0.13759  0.23619  0.62807  0.45682  0.35212
-1.50  5750  2.50  0.12937  0.19646  0.62559  0.45797  0.35218
-1.50  5750  3.00  0.12226  0.16350  0.62431  0.45859  0.35250
-1.50  5750  3.50  0.11639  0.14146  0.62376  0.45934  0.35275
-1.50  5750  4.00  0.11210  0.12955  0.62452  0.46030  0.35290
-1.50  5750  4.50  0.10949  0.12984  0.62740  0.46200  0.35265
-1.50  5750  5.00  0.10830  0.14319  0.63238  0.46503  0.35247
-1.50  5750  5.50  0.10830  0.14319  0.63238  0.46503  0.35247
-1.50  6000 -1.02  0.23554  0.42583  0.50198  0.35269  0.31688
-1.50  6000 -0.70  0.23554  0.42583  0.50198  0.35269  0.31688
-1.50  6000 -0.51  0.23554  0.42583  0.50198  0.35269  0.31688
-1.50  6000 -0.29  0.23554  0.42583  0.50198  0.35269  0.31688
-1.50  6000  0.00  0.23554  0.42583  0.50198  0.35269  0.31688
-1.50  6000  0.28  0.22793  0.40247  0.50784  0.35926  0.31551
-1.50  6000  0.50  0.22195  0.38411  0.51245  0.36443  0.31443
-1.50  6000  0.60  0.21868  0.37216  0.51342  0.36700  0.31439
-1.50  6000  0.87  0.20984  0.33989  0.51602  0.37392  0.31427
-1.50  6000  1.00  0.20559  0.32435  0.51728  0.37726  0.31421
-1.50  6000  1.50  0.18644  0.24186  0.54378  0.39900  0.31634
-1.50  6000  2.00  0.17570  0.19738  0.54391  0.40273  0.31596
-1.50  6000  2.50  0.16553  0.15464  0.54552  0.40578  0.31637
-1.50  6000  3.00  0.15632  0.11696  0.54852  0.40858  0.31716
-1.50  6000  3.50  0.14874  0.08692  0.55224  0.41066  0.31795
-1.50  6000  4.00  0.14287  0.06596  0.55619  0.41224  0.31864
-1.50  6000  4.50  0.13893  0.05589  0.56122  0.41425  0.31914
-1.50  6000  5.00  0.13728  0.05774  0.56801  0.41669  0.31945
-1.50  6000  5.50  0.13728  0.05774  0.56801  0.41669  0.31945
-1.50  6250 -1.02  0.27256  0.37480  0.40930  0.29820  0.27400
-1.50  6250 -0.70  0.27256  0.37480  0.40930  0.29820  0.27400
-1.50  6250 -0.51  0.27256  0.37480  0.40930  0.29820  0.27400
-1.50  6250 -0.29  0.27256  0.37480  0.40930  0.29820  0.27400
-1.50  6250  0.00  0.27256  0.37480  0.40930  0.29820  0.27400
-1.50  6250  0.28  0.27256  0.37480  0.40930  0.29820  0.27400
-1.50  6250  0.50  0.27256  0.37480  0.40930  0.29820  0.27400
-1.50  6250  0.60  0.26995  0.36632  0.41264  0.30100  0.27363
-1.50  6250  0.87  0.26291  0.34341  0.42167  0.30856  0.27262
-1.50  6250  1.00  0.25952  0.33238  0.42601  0.31219  0.27214
-1.50  6250  1.50  0.24147  0.26742  0.44211  0.33013  0.27347
-1.50  6250  2.00  0.22247  0.19122  0.47062  0.35160  0.27610
-1.50  6250  2.50  0.21064  0.14777  0.47608  0.35738  0.27634
-1.50  6250  3.00  0.19963  0.10596  0.48257  0.36199  0.27741
-1.50  6250  3.50  0.18995  0.06865  0.48967  0.36585  0.27865
-1.50  6250  4.00  0.18191  0.03850  0.49655  0.36889  0.28004
-1.50  6250  4.50  0.17628  0.01806  0.50371  0.37147  0.28118
-1.50  6250  5.00  0.17289  0.00720  0.51168  0.37426  0.28221
-1.50  6250  5.50  0.17289  0.00720  0.51168  0.37426  0.28221
-1.50  6500 -1.02  0.30928  0.34905  0.31063  0.23727  0.23850
-1.50  6500 -0.70  0.30928  0.34905  0.31063  0.23727  0.23850
-1.50  6500 -0.51  0.30928  0.34905  0.31063  0.23727  0.23850
-1.50  6500 -0.29  0.30928  0.34905  0.31063  0.23727  0.23850
-1.50  6500  0.00  0.30928  0.34905  0.31063  0.23727  0.23850
-1.50  6500  0.28  0.30928  0.34905  0.31063  0.23727  0.23850
-1.50  6500  0.50  0.30928  0.34905  0.31063  0.23727  0.23850
-1.50  6500  0.60  0.30745  0.34576  0.31529  0.24019  0.23771
-1.50  6500  0.87  0.30250  0.33686  0.32788  0.24807  0.23558
-1.50  6500  1.00  0.30012  0.33257  0.33394  0.25186  0.23456
-1.50  6500  1.50  0.28663  0.28522  0.35822  0.26847  0.23357
-1.50  6500  2.00  0.26699  0.21712  0.38089  0.28888  0.23648
-1.50  6500  2.50  0.24845  0.14567  0.41114  0.30950  0.23926
-1.50  6500  3.00  0.23540  0.10081  0.42110  0.31688  0.24037
-1.50  6500  3.50  0.22316  0.05783  0.43050  0.32264  0.24211
-1.50  6500  4.00  0.21255  0.01954  0.43966  0.32738  0.24419
-1.50  6500  4.50  0.20422 -0.01136  0.44847  0.33121  0.24623
-1.50  6500  5.00  0.19838 -0.03453  0.45767  0.33475  0.24814
-1.50  6500  5.50  0.19838 -0.03453  0.45767  0.33475  0.24814
-1.50  6750 -1.02  0.33384  0.32293  0.23311  0.18676  0.21207
-1.50  6750 -0.70  0.33384  0.32293  0.23311  0.18676  0.21207
-1.50  6750 -0.51  0.33384  0.32293  0.23311  0.18676  0.21207
-1.50  6750 -0.29  0.33384  0.32293  0.23311  0.18676  0.21207
-1.50  6750  0.00  0.33384  0.32293  0.23311  0.18676  0.21207
-1.50  6750  0.28  0.33384  0.32293  0.23311  0.18676  0.21207
-1.50  6750  0.50  0.33384  0.32293  0.23311  0.18676  0.21207
-1.50  6750  0.60  0.33230  0.32472  0.23608  0.18934  0.21118
-1.50  6750  0.87  0.32814  0.32954  0.24410  0.19630  0.20877
-1.50  6750  1.00  0.32614  0.33187  0.24797  0.19965  0.20762
-1.50  6750  1.50  0.31728  0.30981  0.27753  0.21585  0.20399
-1.50  6750  2.00  0.30321  0.25904  0.30720  0.23374  0.20399
-1.50  6750  2.50  0.28223  0.18752  0.33541  0.25603  0.20830
-1.50  6750  3.00  0.26363  0.11906  0.36540  0.27584  0.21145
-1.50  6750  3.50  0.24947  0.07178  0.37830  0.28401  0.21338
-1.50  6750  4.00  0.23617  0.02699  0.38936  0.29059  0.21601
-1.50  6750  4.50  0.22436 -0.01415  0.39937  0.29623  0.21904
-1.50  6750  5.00  0.21518 -0.04903  0.40939  0.30116  0.22189
-1.50  6750  5.50  0.21518 -0.04903  0.40939  0.30116  0.22189
-1.50  7000 -1.02  0.34307  0.27019  0.16408  0.13686  0.18671
-1.50  7000 -0.70  0.34307  0.27019  0.16408  0.13686  0.18671
-1.50  7000 -0.51  0.34307  0.27019  0.16408  0.13686  0.18671
-1.50  7000 -0.29  0.34307  0.27019  0.16408  0.13686  0.18671
-1.50  7000  0.00  0.34307  0.27019  0.16408  0.13686  0.18671
-1.50  7000  0.28  0.34307  0.27019  0.16408  0.13686  0.18671
-1.50  7000  0.50  0.34307  0.27019  0.16408  0.13686  0.18671
-1.50  7000  0.60  0.34195  0.27575  0.16613  0.13904  0.18585
-1.50  7000  0.87  0.33893  0.29075  0.17166  0.14492  0.18353
-1.50  7000  1.00  0.33748  0.29797  0.17433  0.14775  0.18241
-1.50  7000  1.50  0.33082  0.30368  0.19472  0.16057  0.17717
-1.50  7000  2.00  0.32104  0.27627  0.22961  0.17812  0.17427
-1.50  7000  2.50  0.30486  0.21817  0.26369  0.19763  0.17594
-1.50  7000  3.00  0.28166  0.13919  0.29788  0.22266  0.18167
-1.50  7000  3.50  0.26389  0.07767  0.32408  0.23878  0.18470
-1.50  7000  4.00  0.24790  0.02760  0.33838  0.24780  0.18787
-1.50  7000  4.50  0.23304 -0.02067  0.35074  0.25535  0.19149
-1.50  7000  5.00  0.22046 -0.06441  0.36186  0.26177  0.19522
-1.50  7000  5.50  0.22046 -0.06441  0.36186  0.26177  0.19522
-1.50  7250 -1.02  0.35106  0.24981  0.12473  0.09441  0.16200
-1.50  7250 -0.70  0.35106  0.24981  0.12473  0.09441  0.16200
-1.50  7250 -0.51  0.35106  0.24981  0.12473  0.09441  0.16200
-1.50  7250 -0.29  0.35106  0.24981  0.12473  0.09441  0.16200
-1.50  7250  0.00  0.35106  0.24981  0.12473  0.09441  0.16200
-1.50  7250  0.28  0.35106  0.24981  0.12473  0.09441  0.16200
-1.50  7250  0.50  0.35106  0.24981  0.12473  0.09441  0.16200
-1.50  7250  0.60  0.35013  0.25901  0.12388  0.09567  0.16138
-1.50  7250  0.87  0.34764  0.28384  0.12158  0.09907  0.15972
-1.50  7250  1.00  0.34643  0.29579  0.12047  0.10070  0.15892
-1.50  7250  1.50  0.34256  0.31945  0.13774  0.11132  0.15320
-1.50  7250  2.00  0.33586  0.32005  0.16269  0.12449  0.14839
-1.50  7250  2.50  0.32487  0.28525  0.20156  0.14340  0.14677
-1.50  7250  3.00  0.30705  0.21977  0.23895  0.16457  0.15012
-1.50  7250  3.50  0.28396  0.13884  0.27491  0.18992  0.15688
-1.50  7250  4.00  0.26624  0.07943  0.29785  0.20420  0.16059
-1.50  7250  4.50  0.24965  0.02711  0.31310  0.21365  0.16467
-1.50  7250  5.00  0.23441 -0.02346  0.32603  0.22185  0.16924
-1.50  7250  5.50  0.23441 -0.02346  0.32603  0.22185  0.16924
-1.50  7500 -1.02  0.34353  0.19535  0.12433  0.08855  0.15021
-1.50  7500 -0.70  0.34353  0.19535  0.12433  0.08855  0.15021
-1.50  7500 -0.51  0.34353  0.19535  0.12433  0.08855  0.15021
-1.50  7500 -0.29  0.34353  0.19535  0.12433  0.08855  0.15021
-1.50  7500  0.00  0.34353  0.19535  0.12433  0.08855  0.15021
-1.50  7500  0.28  0.34353  0.19535  0.12433  0.08855  0.15021
-1.50  7500  0.50  0.34353  0.19535  0.12433  0.08855  0.15021
-1.50  7500  0.60  0.34264  0.20947  0.11519  0.08680  0.15029
-1.50  7500  0.87  0.34024  0.24760  0.09049  0.08208  0.15049
-1.50  7500  1.00  0.33908  0.26596  0.07860  0.07981  0.15059
-1.50  7500  1.50  0.33764  0.30622  0.08929  0.08738  0.14492
-1.50  7500  2.00  0.33428  0.32560  0.11174  0.09791  0.13880
-1.50  7500  2.50  0.32767  0.32283  0.14130  0.11108  0.13436
-1.50  7500  3.00  0.31538  0.27930  0.18375  0.13130  0.13426
-1.50  7500  3.50  0.29519  0.20399  0.22269  0.15471  0.13994
-1.50  7500  4.00  0.27260  0.12259  0.25723  0.17858  0.14699
-1.50  7500  4.50  0.25540  0.06680  0.27750  0.19061  0.15099
-1.50  7500  5.00  0.23803  0.01065  0.29301  0.20069  0.15636
-1.50  7500  5.50  0.23803  0.01065  0.29301  0.20069  0.15636
-1.50  7750 -1.02  0.31024  0.18386  0.04459  0.07582  0.14856
-1.50  7750 -0.70  0.31024  0.18386  0.04459  0.07582  0.14856
-1.50  7750 -0.51  0.31024  0.18386  0.04459  0.07582  0.14856
-1.50  7750 -0.29  0.31024  0.18386  0.04459  0.07582  0.14856
-1.50  7750  0.00  0.31024  0.18386  0.04459  0.07582  0.14856
-1.50  7750  0.28  0.31024  0.18386  0.04459  0.07582  0.14856
-1.50  7750  0.50  0.31024  0.18386  0.04459  0.07582  0.14856
-1.50  7750  0.60  0.31024  0.18386  0.04459  0.07582  0.14856
-1.50  7750  0.87  0.31024  0.18386  0.04459  0.07582  0.14856
-1.50  7750  1.00  0.31024  0.18386  0.04459  0.07582  0.14856
-1.50  7750  1.50  0.31119  0.24033  0.04071  0.07737  0.14332
-1.50  7750  2.00  0.31060  0.27477  0.06019  0.08555  0.13652
-1.50  7750  2.50  0.30631  0.28723  0.08741  0.09654  0.13108
-1.50  7750  3.00  0.29798  0.27581  0.12066  0.11044  0.12779
-1.50  7750  3.50  0.28322  0.22065  0.16530  0.13174  0.12936
-1.50  7750  4.00  0.26032  0.13556  0.20474  0.15640  0.13683
-1.50  7750  4.50  0.23922  0.06195  0.23492  0.17585  0.14302
-1.50  7750  5.00  0.21974  0.00049  0.25438  0.18833  0.14884
-1.50  7750  5.50  0.21974  0.00049  0.25438  0.18833  0.14884
-1.50  8000 -1.02  0.27535  0.08767  0.03232  0.06843  0.13416
-1.50  8000 -0.70  0.27535  0.08767  0.03232  0.06843  0.13416
-1.50  8000 -0.51  0.27535  0.08767  0.03232  0.06843  0.13416
-1.50  8000 -0.29  0.27535  0.08767  0.03232  0.06843  0.13416
-1.50  8000  0.00  0.27535  0.08767  0.03232  0.06843  0.13416
-1.50  8000  0.28  0.27535  0.08767  0.03232  0.06843  0.13416
-1.50  8000  0.50  0.27535  0.08767  0.03232  0.06843  0.13416
-1.50  8000  0.60  0.27535  0.08767  0.03232  0.06843  0.13416
-1.50  8000  0.87  0.27535  0.08767  0.03232  0.06843  0.13416
-1.50  8000  1.00  0.27535  0.08767  0.03232  0.06843  0.13416
-1.50  8000  1.50  0.28249  0.16526  0.00333  0.05701  0.13000
-1.50  8000  2.00  0.28481  0.21275  0.01513  0.06115  0.12292
-1.50  8000  2.50  0.28365  0.23956  0.04025  0.06969  0.11638
-1.50  8000  3.00  0.27822  0.24564  0.07123  0.08071  0.11140
-1.50  8000  3.50  0.26854  0.22503  0.10905  0.09523  0.10899
-1.50  8000  4.00  0.25082  0.15544  0.15461  0.11791  0.11307
-1.50  8000  4.50  0.22642  0.06416  0.19272  0.14305  0.12177
-1.50  8000  5.00  0.20520 -0.00580  0.21829  0.15882  0.12835
-1.50  8000  5.50  0.20520 -0.00580  0.21829  0.15882  0.12835
-1.50  8250 -1.02  0.21756 -0.02959  0.03372  0.07303  0.11424
-1.50  8250 -0.70  0.21756 -0.02959  0.03372  0.07303  0.11424
-1.50  8250 -0.51  0.21756 -0.02959  0.03372  0.07303  0.11424
-1.50  8250 -0.29  0.21756 -0.02959  0.03372  0.07303  0.11424
-1.50  8250  0.00  0.21756 -0.02959  0.03372  0.07303  0.11424
-1.50  8250  0.28  0.21756 -0.02959  0.03372  0.07303  0.11424
-1.50  8250  0.50  0.21756 -0.02959  0.03372  0.07303  0.11424
-1.50  8250  0.60  0.21756 -0.02959  0.03372  0.07303  0.11424
-1.50  8250  0.87  0.21756 -0.02959  0.03372  0.07303  0.11424
-1.50  8250  1.00  0.21756 -0.02959  0.03372  0.07303  0.11424
-1.50  8250  1.50  0.24162  0.07590 -0.02014  0.04165  0.11205
-1.50  8250  2.00  0.24756  0.13512 -0.01998  0.03989  0.10503
-1.50  8250  2.50  0.24956  0.17419  0.00068  0.04520  0.09748
-1.50  8250  3.00  0.24736  0.19329  0.03043  0.05386  0.09126
-1.50  8250  3.50  0.24022  0.18994  0.06439  0.06489  0.08760
-1.50  8250  4.00  0.22828  0.15465  0.10582  0.08087  0.08702
-1.50  8250  4.50  0.20814  0.07147  0.15106  0.10481  0.09317
-1.50  8250  5.00  0.18322 -0.02147  0.18505  0.12771  0.10290
-1.50  8250  5.50  0.18322 -0.02147  0.18505  0.12771  0.10290
-1.50  8500 -1.02  0.19519 -0.00050 -0.02791  0.04743  0.10486
-1.50  8500 -0.70  0.19519 -0.00050 -0.02791  0.04743  0.10486
-1.50  8500 -0.51  0.19519 -0.00050 -0.02791  0.04743  0.10486
-1.50  8500 -0.29  0.19519 -0.00050 -0.02791  0.04743  0.10486
-1.50  8500  0.00  0.19519 -0.00050 -0.02791  0.04743  0.10486
-1.50  8500  0.28  0.19519 -0.00050 -0.02791  0.04743  0.10486
-1.50  8500  0.50  0.19519 -0.00050 -0.02791  0.04743  0.10486
-1.50  8500  0.60  0.19519 -0.00050 -0.02791  0.04743  0.10486
-1.50  8500  0.87  0.19519 -0.00050 -0.02791  0.04743  0.10486
-1.50  8500  1.00  0.19519 -0.00050 -0.02791  0.04743  0.10486
-1.50  8500  1.50  0.19519 -0.00050 -0.02791  0.04743  0.10486
-1.50  8500  2.00  0.20586  0.06956 -0.04049  0.03797  0.09821
-1.50  8500  2.50  0.21043  0.11711 -0.02667  0.03959  0.09057
-1.50  8500  3.00  0.21139  0.14586 -0.00028  0.04569  0.08358
-1.50  8500  3.50  0.20817  0.15515  0.03279  0.05445  0.07837
-1.50  8500  4.00  0.19978  0.14126  0.06970  0.06582  0.07577
-1.50  8500  4.50  0.18687  0.08939  0.11487  0.08348  0.07685
-1.50  8500  5.00  0.16572 -0.00389  0.15680  0.10754  0.08520
-1.50  8500  5.50  0.16572 -0.00389  0.15680  0.10754  0.08520
-1.50  8750 -1.02  0.16042 -0.00441 -0.05086  0.04387  0.09625
-1.50  8750 -0.70  0.16042 -0.00441 -0.05086  0.04387  0.09625
-1.50  8750 -0.51  0.16042 -0.00441 -0.05086  0.04387  0.09625
-1.50  8750 -0.29  0.16042 -0.00441 -0.05086  0.04387  0.09625
-1.50  8750  0.00  0.16042 -0.00441 -0.05086  0.04387  0.09625
-1.50  8750  0.28  0.16042 -0.00441 -0.05086  0.04387  0.09625
-1.50  8750  0.50  0.16042 -0.00441 -0.05086  0.04387  0.09625
-1.50  8750  0.60  0.16042 -0.00441 -0.05086  0.04387  0.09625
-1.50  8750  0.87  0.16042 -0.00441 -0.05086  0.04387  0.09625
-1.50  8750  1.00  0.16042 -0.00441 -0.05086  0.04387  0.09625
-1.50  8750  1.50  0.16042 -0.00441 -0.05086  0.04387  0.09625
-1.50  8750  2.00  0.16239  0.00878 -0.05254  0.04216  0.09498
-1.50  8750  2.50  0.16894  0.06262 -0.04599  0.03984  0.08739
-1.50  8750  3.00  0.17227  0.09995 -0.02390  0.04308  0.07969
-1.50  8750  3.50  0.17245  0.12000  0.00672  0.04926  0.07308
-1.50  8750  4.00  0.16845  0.11874  0.04294  0.05814  0.06884
-1.50  8750  4.50  0.15932  0.09119  0.08282  0.06998  0.06782
-1.50  8750  5.00  0.14600  0.02026  0.12826  0.08975  0.07144
-1.50  8750  5.50  0.14600  0.02026  0.12826  0.08975  0.07144
-1.50  9000 -1.02  0.09311 -0.13845 -0.03458  0.05938  0.08590
-1.50  9000 -0.70  0.09311 -0.13845 -0.03458  0.05938  0.08590
-1.50  9000 -0.51  0.09311 -0.13845 -0.03458  0.05938  0.08590
-1.50  9000 -0.29  0.09311 -0.13845 -0.03458  0.05938  0.08590
-1.50  9000  0.00  0.09311 -0.13845 -0.03458  0.05938  0.08590
-1.50  9000  0.28  0.09311 -0.13845 -0.03458  0.05938  0.08590
-1.50  9000  0.50  0.09311 -0.13845 -0.03458  0.05938  0.08590
-1.50  9000  0.60  0.09311 -0.13845 -0.03458  0.05938  0.08590
-1.50  9000  0.87  0.09311 -0.13845 -0.03458  0.05938  0.08590
-1.50  9000  1.00  0.09311 -0.13845 -0.03458  0.05938  0.08590
-1.50  9000  1.50  0.09311 -0.13845 -0.03458  0.05938  0.08590
-1.50  9000  2.00  0.11502 -0.04951 -0.06357  0.03683  0.08105
-1.50  9000  2.50  0.12309  0.00972 -0.06371  0.03054  0.07380
-1.50  9000  3.00  0.12747  0.05292 -0.04631  0.03119  0.06596
-1.50  9000  3.50  0.12958  0.08001 -0.01829  0.03538  0.05892
-1.50  9000  4.00  0.12858  0.08742  0.01602  0.04218  0.05376
-1.50  9000  4.50  0.12333  0.07264  0.05394  0.05118  0.05115
-1.50  9000  5.00  0.11244  0.03504  0.09290  0.06214  0.05186
-1.50  9000  5.50  0.11244  0.03504  0.09290  0.06214  0.05186
-1.50  9250 -1.02  0.06799 -0.10334 -0.07097  0.01949  0.05316
-1.50  9250 -0.70  0.06799 -0.10334 -0.07097  0.01949  0.05316
-1.50  9250 -0.51  0.06799 -0.10334 -0.07097  0.01949  0.05316
-1.50  9250 -0.29  0.06799 -0.10334 -0.07097  0.01949  0.05316
-1.50  9250  0.00  0.06799 -0.10334 -0.07097  0.01949  0.05316
-1.50  9250  0.28  0.06799 -0.10334 -0.07097  0.01949  0.05316
-1.50  9250  0.50  0.06799 -0.10334 -0.07097  0.01949  0.05316
-1.50  9250  0.60  0.06799 -0.10334 -0.07097  0.01949  0.05316
-1.50  9250  0.87  0.06799 -0.10334 -0.07097  0.01949  0.05316
-1.50  9250  1.00  0.06799 -0.10334 -0.07097  0.01949  0.05316
-1.50  9250  1.50  0.06799 -0.10334 -0.07097  0.01949  0.05316
-1.50  9250  2.00  0.06799 -0.10334 -0.07097  0.01949  0.05316
-1.50  9250  2.50  0.07739 -0.03948 -0.07619  0.00983  0.04648
-1.50  9250  3.00  0.08191  0.00788 -0.06366  0.00800  0.03895
-1.50  9250  3.50  0.08461  0.04031 -0.03932  0.01032  0.03172
-1.50  9250  4.00  0.08554  0.05562 -0.00713  0.01518  0.02567
-1.50  9250  4.50  0.08367  0.05007  0.03040  0.02216  0.02169
-1.50  9250  5.00  0.07726  0.02136  0.07025  0.03120  0.02093
-1.50  9250  5.50  0.07726  0.02136  0.07025  0.03120  0.02093
-1.50  9500 -1.02  0.02888 -0.14513 -0.07429  0.01380  0.03447
-1.50  9500 -0.70  0.02888 -0.14513 -0.07429  0.01380  0.03447
-1.50  9500 -0.51  0.02888 -0.14513 -0.07429  0.01380  0.03447
-1.50  9500 -0.29  0.02888 -0.14513 -0.07429  0.01380  0.03447
-1.50  9500  0.00  0.02888 -0.14513 -0.07429  0.01380  0.03447
-1.50  9500  0.28  0.02888 -0.14513 -0.07429  0.01380  0.03447
-1.50  9500  0.50  0.02888 -0.14513 -0.07429  0.01380  0.03447
-1.50  9500  0.60  0.02888 -0.14513 -0.07429  0.01380  0.03447
-1.50  9500  0.87  0.02888 -0.14513 -0.07429  0.01380  0.03447
-1.50  9500  1.00  0.02888 -0.14513 -0.07429  0.01380  0.03447
-1.50  9500  1.50  0.02888 -0.14513 -0.07429  0.01380  0.03447
-1.50  9500  2.00  0.02888 -0.14513 -0.07429  0.01380  0.03447
-1.50  9500  2.50  0.03925 -0.07771 -0.08365  0.00155  0.02841
-1.50  9500  3.00  0.04386 -0.02735 -0.07533 -0.00249  0.02122
-1.50  9500  3.50  0.04661  0.01005 -0.05477 -0.00199  0.01396
-1.50  9500  4.00  0.04853  0.03227 -0.02493  0.00115  0.00730
-1.50  9500  4.50  0.04872  0.03486  0.01136  0.00637  0.00243
-1.50  9500  5.00  0.04575  0.01431  0.05157  0.01365  0.00049
-1.50  9500  5.50  0.04575  0.01431  0.05157  0.01365  0.00049
-1.50  9750 -1.02 -0.00947 -0.18433 -0.07567  0.01239  0.02273
-1.50  9750 -0.70 -0.00947 -0.18433 -0.07567  0.01239  0.02273
-1.50  9750 -0.51 -0.00947 -0.18433 -0.07567  0.01239  0.02273
-1.50  9750 -0.29 -0.00947 -0.18433 -0.07567  0.01239  0.02273
-1.50  9750  0.00 -0.00947 -0.18433 -0.07567  0.01239  0.02273
-1.50  9750  0.28 -0.00947 -0.18433 -0.07567  0.01239  0.02273
-1.50  9750  0.50 -0.00947 -0.18433 -0.07567  0.01239  0.02273
-1.50  9750  0.60 -0.00947 -0.18433 -0.07567  0.01239  0.02273
-1.50  9750  0.87 -0.00947 -0.18433 -0.07567  0.01239  0.02273
-1.50  9750  1.00 -0.00947 -0.18433 -0.07567  0.01239  0.02273
-1.50  9750  1.50 -0.00947 -0.18433 -0.07567  0.01239  0.02273
-1.50  9750  2.00 -0.00947 -0.18433 -0.07567  0.01239  0.02273
-1.50  9750  2.50  0.00165 -0.11456 -0.08751 -0.00138  0.01729
-1.50  9750  3.00  0.00610 -0.06200 -0.08274 -0.00717  0.01064
-1.50  9750  3.50  0.00875 -0.02106 -0.06576 -0.00847  0.00337
-1.50  9750  4.00  0.01088  0.00648 -0.03881 -0.00682 -0.00349
-1.50  9750  4.50  0.01222  0.01648 -0.00422 -0.00315 -0.00893
-1.50  9750  5.00  0.01148  0.00352  0.03513  0.00254 -0.01185
-1.50  9750  5.50  0.01148  0.00352  0.03513  0.00254 -0.01185
-1.50  10000 -1.02 -0.06653 -0.23531 -0.08013  0.01421  0.01595
-1.50  10000 -0.70 -0.06653 -0.23531 -0.08013  0.01421  0.01595
-1.50  10000 -0.51 -0.06653 -0.23531 -0.08013  0.01421  0.01595
-1.50  10000 -0.29 -0.06653 -0.23531 -0.08013  0.01421  0.01595
-1.50  10000  0.00 -0.06653 -0.23531 -0.08013  0.01421  0.01595
-1.50  10000  0.28 -0.06653 -0.23531 -0.08013  0.01421  0.01595
-1.50  10000  0.50 -0.06653 -0.23531 -0.08013  0.01421  0.01595
-1.50  10000  0.60 -0.06653 -0.23531 -0.08013  0.01421  0.01595
-1.50  10000  0.87 -0.06653 -0.23531 -0.08013  0.01421  0.01595
-1.50  10000  1.00 -0.06653 -0.23531 -0.08013  0.01421  0.01595
-1.50  10000  1.50 -0.06653 -0.23531 -0.08013  0.01421  0.01595
-1.50  10000  2.00 -0.06653 -0.23531 -0.08013  0.01421  0.01595
-1.50  10000  2.50 -0.05484 -0.16329 -0.09368 -0.00064  0.01084
-1.50  10000  3.00 -0.05063 -0.10912 -0.09150 -0.00778  0.00462
-1.50  10000  3.50 -0.04863 -0.06638 -0.07735 -0.01033 -0.00234
-1.50  10000  4.00 -0.04702 -0.03496 -0.05353 -0.01019 -0.00912
-1.50  10000  4.50 -0.04536 -0.01949 -0.02116 -0.00786 -0.01493
-1.50  10000  5.00 -0.04486 -0.02530  0.01717 -0.00351 -0.01878
-1.50  10000  5.50 -0.04486 -0.02530  0.01717 -0.00351 -0.01878
-1.50  10500 -1.02 -0.18952 -0.33012 -0.09082  0.01605  0.00649
-1.50  10500 -0.70 -0.18952 -0.33012 -0.09082  0.01605  0.00649
-1.50  10500 -0.51 -0.18952 -0.33012 -0.09082  0.01605  0.00649
-1.50  10500 -0.29 -0.18952 -0.33012 -0.09082  0.01605  0.00649
-1.50  10500  0.00 -0.18952 -0.33012 -0.09082  0.01605  0.00649
-1.50  10500  0.28 -0.18952 -0.33012 -0.09082  0.01605  0.00649
-1.50  10500  0.50 -0.18952 -0.33012 -0.09082  0.01605  0.00649
-1.50  10500  0.60 -0.18952 -0.33012 -0.09082  0.01605  0.00649
-1.50  10500  0.87 -0.18952 -0.33012 -0.09082  0.01605  0.00649
-1.50  10500  1.00 -0.18952 -0.33012 -0.09082  0.01605  0.00649
-1.50  10500  1.50 -0.18952 -0.33012 -0.09082  0.01605  0.00649
-1.50  10500  2.00 -0.18952 -0.33012 -0.09082  0.01605  0.00649
-1.50  10500  2.50 -0.17681 -0.25521 -0.10642  0.00042  0.00148
-1.50  10500  3.00 -0.17349 -0.19940 -0.10735 -0.00829 -0.00393
-1.50  10500  3.50 -0.17315 -0.15498 -0.09789 -0.01274 -0.00996
-1.50  10500  4.00 -0.17302 -0.12025 -0.07934 -0.01459 -0.01619
-1.50  10500  4.50 -0.17209 -0.09724 -0.05219 -0.01436 -0.02236
-1.50  10500  5.00 -0.17069 -0.09067 -0.01690 -0.01256 -0.02745
-1.50  10500  5.50 -0.17069 -0.09067 -0.01690 -0.01256 -0.02745
-1.50  11000 -1.02 -0.30053 -0.33416 -0.11952 -0.00102 -0.00985
-1.50  11000 -0.70 -0.30053 -0.33416 -0.11952 -0.00102 -0.00985
-1.50  11000 -0.51 -0.30053 -0.33416 -0.11952 -0.00102 -0.00985
-1.50  11000 -0.29 -0.30053 -0.33416 -0.11952 -0.00102 -0.00985
-1.50  11000  0.00 -0.30053 -0.33416 -0.11952 -0.00102 -0.00985
-1.50  11000  0.28 -0.30053 -0.33416 -0.11952 -0.00102 -0.00985
-1.50  11000  0.50 -0.30053 -0.33416 -0.11952 -0.00102 -0.00985
-1.50  11000  0.60 -0.30053 -0.33416 -0.11952 -0.00102 -0.00985
-1.50  11000  0.87 -0.30053 -0.33416 -0.11952 -0.00102 -0.00985
-1.50  11000  1.00 -0.30053 -0.33416 -0.11952 -0.00102 -0.00985
-1.50  11000  1.50 -0.30053 -0.33416 -0.11952 -0.00102 -0.00985
-1.50  11000  2.00 -0.30053 -0.33416 -0.11952 -0.00102 -0.00985
-1.50  11000  2.50 -0.30053 -0.33416 -0.11952 -0.00102 -0.00985
-1.50  11000  3.00 -0.29782 -0.27799 -0.12217 -0.01008 -0.01501
-1.50  11000  3.50 -0.29884 -0.23306 -0.11540 -0.01561 -0.02037
-1.50  11000  4.00 -0.30051 -0.19685 -0.10074 -0.01873 -0.02599
-1.50  11000  4.50 -0.30115 -0.16998 -0.07821 -0.02015 -0.03175
-1.50  11000  5.00 -0.30035 -0.15610 -0.04739 -0.02011 -0.03708
-1.50  11000  5.50 -0.30035 -0.15610 -0.04739 -0.02011 -0.03708
-1.50  11500 -1.02 -0.48324 -0.42392 -0.14010 -0.00547 -0.02227
-1.50  11500 -0.70 -0.48324 -0.42392 -0.14010 -0.00547 -0.02227
-1.50  11500 -0.51 -0.48324 -0.42392 -0.14010 -0.00547 -0.02227
-1.50  11500 -0.29 -0.48324 -0.42392 -0.14010 -0.00547 -0.02227
-1.50  11500  0.00 -0.48324 -0.42392 -0.14010 -0.00547 -0.02227
-1.50  11500  0.28 -0.48324 -0.42392 -0.14010 -0.00547 -0.02227
-1.50  11500  0.50 -0.48324 -0.42392 -0.14010 -0.00547 -0.02227
-1.50  11500  0.60 -0.48324 -0.42392 -0.14010 -0.00547 -0.02227
-1.50  11500  0.87 -0.48324 -0.42392 -0.14010 -0.00547 -0.02227
-1.50  11500  1.00 -0.48324 -0.42392 -0.14010 -0.00547 -0.02227
-1.50  11500  1.50 -0.48324 -0.42392 -0.14010 -0.00547 -0.02227
-1.50  11500  2.00 -0.48324 -0.42392 -0.14010 -0.00547 -0.02227
-1.50  11500  2.50 -0.48324 -0.42392 -0.14010 -0.00547 -0.02227
-1.50  11500  3.00 -0.48178 -0.36889 -0.14352 -0.01450 -0.02722
-1.50  11500  3.50 -0.48473 -0.32468 -0.13816 -0.02029 -0.03225
-1.50  11500  4.00 -0.48866 -0.28854 -0.12587 -0.02414 -0.03736
-1.50  11500  4.50 -0.49160 -0.26069 -0.10693 -0.02647 -0.04247
-1.50  11500  5.00 -0.49248 -0.24322 -0.08041 -0.02765 -0.04747
-1.50  11500  5.50 -0.49248 -0.24322 -0.08041 -0.02765 -0.04747
-1.50  12000 -1.02 -0.44902 -0.42272 -0.12822 -0.02166 -0.03266
-1.50  12000 -0.70 -0.44902 -0.42272 -0.12822 -0.02166 -0.03266
-1.50  12000 -0.51 -0.44902 -0.42272 -0.12822 -0.02166 -0.03266
-1.50  12000 -0.29 -0.44902 -0.42272 -0.12822 -0.02166 -0.03266
-1.50  12000  0.00 -0.44902 -0.42272 -0.12822 -0.02166 -0.03266
-1.50  12000  0.28 -0.44902 -0.42272 -0.12822 -0.02166 -0.03266
-1.50  12000  0.50 -0.44902 -0.42272 -0.12822 -0.02166 -0.03266
-1.50  12000  0.60 -0.44902 -0.42272 -0.12822 -0.02166 -0.03266
-1.50  12000  0.87 -0.44902 -0.42272 -0.12822 -0.02166 -0.03266
-1.50  12000  1.00 -0.44902 -0.42272 -0.12822 -0.02166 -0.03266
-1.50  12000  1.50 -0.44902 -0.42272 -0.12822 -0.02166 -0.03266
-1.50  12000  2.00 -0.44902 -0.42272 -0.12822 -0.02166 -0.03266
-1.50  12000  2.50 -0.44902 -0.42272 -0.12822 -0.02166 -0.03266
-1.50  12000  3.00 -0.44449 -0.36997 -0.13227 -0.03021 -0.03758
-1.50  12000  3.50 -0.44498 -0.32721 -0.12801 -0.03583 -0.04248
-1.50  12000  4.00 -0.44720 -0.29154 -0.11753 -0.03988 -0.04729
-1.50  12000  4.50 -0.44964 -0.26298 -0.10120 -0.04288 -0.05178
-1.50  12000  5.00 -0.45083 -0.24282 -0.07865 -0.04487 -0.05640
-1.50  12000  5.50 -0.45083 -0.24282 -0.07865 -0.04487 -0.05640
-1.50  12500 -1.02 -0.54450 -0.46757 -0.13627 -0.02708 -0.04245
-1.50  12500 -0.70 -0.54450 -0.46757 -0.13627 -0.02708 -0.04245
-1.50  12500 -0.51 -0.54450 -0.46757 -0.13627 -0.02708 -0.04245
-1.50  12500 -0.29 -0.54450 -0.46757 -0.13627 -0.02708 -0.04245
-1.50  12500  0.00 -0.54450 -0.46757 -0.13627 -0.02708 -0.04245
-1.50  12500  0.28 -0.54450 -0.46757 -0.13627 -0.02708 -0.04245
-1.50  12500  0.50 -0.54450 -0.46757 -0.13627 -0.02708 -0.04245
-1.50  12500  0.60 -0.54450 -0.46757 -0.13627 -0.02708 -0.04245
-1.50  12500  0.87 -0.54450 -0.46757 -0.13627 -0.02708 -0.04245
-1.50  12500  1.00 -0.54450 -0.46757 -0.13627 -0.02708 -0.04245
-1.50  12500  1.50 -0.54450 -0.46757 -0.13627 -0.02708 -0.04245
-1.50  12500  2.00 -0.54450 -0.46757 -0.13627 -0.02708 -0.04245
-1.50  12500  2.50 -0.54450 -0.46757 -0.13627 -0.02708 -0.04245
-1.50  12500  3.00 -0.54016 -0.41686 -0.14082 -0.03506 -0.04704
-1.50  12500  3.50 -0.54048 -0.37641 -0.13694 -0.04040 -0.05164
-1.50  12500  4.00 -0.54269 -0.34239 -0.12742 -0.04444 -0.05621
-1.50  12500  4.50 -0.54545 -0.31446 -0.11268 -0.04765 -0.06051
-1.50  12500  5.00 -0.54741 -0.29405 -0.09250 -0.05010 -0.06473
-1.50  12500  5.50 -0.54741 -0.29405 -0.09250 -0.05010 -0.06473
-1.50  13000 -1.02 -0.63802 -0.50764 -0.14389 -0.03253 -0.05139
-1.50  13000 -0.70 -0.63802 -0.50764 -0.14389 -0.03253 -0.05139
-1.50  13000 -0.51 -0.63802 -0.50764 -0.14389 -0.03253 -0.05139
-1.50  13000 -0.29 -0.63802 -0.50764 -0.14389 -0.03253 -0.05139
-1.50  13000  0.00 -0.63802 -0.50764 -0.14389 -0.03253 -0.05139
-1.50  13000  0.28 -0.63802 -0.50764 -0.14389 -0.03253 -0.05139
-1.50  13000  0.50 -0.63802 -0.50764 -0.14389 -0.03253 -0.05139
-1.50  13000  0.60 -0.63802 -0.50764 -0.14389 -0.03253 -0.05139
-1.50  13000  0.87 -0.63802 -0.50764 -0.14389 -0.03253 -0.05139
-1.50  13000  1.00 -0.63802 -0.50764 -0.14389 -0.03253 -0.05139
-1.50  13000  1.50 -0.63802 -0.50764 -0.14389 -0.03253 -0.05139
-1.50  13000  2.00 -0.63802 -0.50764 -0.14389 -0.03253 -0.05139
-1.50  13000  2.50 -0.63802 -0.50764 -0.14389 -0.03253 -0.05139
-1.50  13000  3.00 -0.63417 -0.45813 -0.14875 -0.04024 -0.05559
-1.50  13000  3.50 -0.63469 -0.41979 -0.14542 -0.04505 -0.05997
-1.50  13000  4.00 -0.63678 -0.38792 -0.13646 -0.04880 -0.06433
-1.50  13000  4.50 -0.63939 -0.36146 -0.12262 -0.05206 -0.06851
-1.50  13000  5.00 -0.64179 -0.34138 -0.10407 -0.05468 -0.07256
-1.50  13000  5.50 -0.64179 -0.34138 -0.10407 -0.05468 -0.07256
-1.50  14000 -1.02 -0.83290 -0.66068 -0.13455 -0.02691 -0.06262
-1.50  14000 -0.70 -0.83290 -0.66068 -0.13455 -0.02691 -0.06262
-1.50  14000 -0.51 -0.83290 -0.66068 -0.13455 -0.02691 -0.06262
-1.50  14000 -0.29 -0.83290 -0.66068 -0.13455 -0.02691 -0.06262
-1.50  14000  0.00 -0.83290 -0.66068 -0.13455 -0.02691 -0.06262
-1.50  14000  0.28 -0.83290 -0.66068 -0.13455 -0.02691 -0.06262
-1.50  14000  0.50 -0.83290 -0.66068 -0.13455 -0.02691 -0.06262
-1.50  14000  0.60 -0.83290 -0.66068 -0.13455 -0.02691 -0.06262
-1.50  14000  0.87 -0.83290 -0.66068 -0.13455 -0.02691 -0.06262
-1.50  14000  1.00 -0.83290 -0.66068 -0.13455 -0.02691 -0.06262
-1.50  14000  1.50 -0.83290 -0.66068 -0.13455 -0.02691 -0.06262
-1.50  14000  2.00 -0.83290 -0.66068 -0.13455 -0.02691 -0.06262
-1.50  14000  2.50 -0.81725 -0.57804 -0.15774 -0.04302 -0.06763
-1.50  14000  3.00 -0.81575 -0.52892 -0.16329 -0.05021 -0.07122
-1.50  14000  3.50 -0.81745 -0.49356 -0.16037 -0.05441 -0.07485
-1.50  14000  4.00 -0.81974 -0.46534 -0.15235 -0.05755 -0.07875
-1.50  14000  4.50 -0.82211 -0.44239 -0.13995 -0.06025 -0.08272
-1.50  14000  5.00 -0.82445 -0.42433 -0.12354 -0.06284 -0.08647
-1.50  14000  5.50 -0.82445 -0.42433 -0.12354 -0.06284 -0.08647
-1.50  15000 -1.02 -0.98487 -0.63971 -0.17044 -0.05280 -0.08212
-1.50  15000 -0.70 -0.98487 -0.63971 -0.17044 -0.05280 -0.08212
-1.50  15000 -0.51 -0.98487 -0.63971 -0.17044 -0.05280 -0.08212
-1.50  15000 -0.29 -0.98487 -0.63971 -0.17044 -0.05280 -0.08212
-1.50  15000  0.00 -0.98487 -0.63971 -0.17044 -0.05280 -0.08212
-1.50  15000  0.28 -0.98487 -0.63971 -0.17044 -0.05280 -0.08212
-1.50  15000  0.50 -0.98487 -0.63971 -0.17044 -0.05280 -0.08212
-1.50  15000  0.60 -0.98487 -0.63971 -0.17044 -0.05280 -0.08212
-1.50  15000  0.87 -0.98487 -0.63971 -0.17044 -0.05280 -0.08212
-1.50  15000  1.00 -0.98487 -0.63971 -0.17044 -0.05280 -0.08212
-1.50  15000  1.50 -0.98487 -0.63971 -0.17044 -0.05280 -0.08212
-1.50  15000  2.00 -0.98487 -0.63971 -0.17044 -0.05280 -0.08212
-1.50  15000  2.50 -0.98487 -0.63971 -0.17044 -0.05280 -0.08212
-1.50  15000  3.00 -0.98738 -0.58859 -0.17686 -0.05976 -0.08514
-1.50  15000  3.50 -0.99145 -0.55416 -0.17427 -0.06350 -0.08815
-1.50  15000  4.00 -0.99500 -0.52846 -0.16685 -0.06594 -0.09140
-1.50  15000  4.50 -0.99749 -0.50862 -0.15541 -0.06813 -0.09485
-1.50  15000  5.00 -0.99934 -0.49348 -0.14024 -0.07039 -0.09837
-1.50  15000  5.50 -0.99934 -0.49348 -0.14024 -0.07039 -0.09837
-1.50  16000 -1.02 -1.14059 -0.69618 -0.18181 -0.06158 -0.09532
-1.50  16000 -0.70 -1.14059 -0.69618 -0.18181 -0.06158 -0.09532
-1.50  16000 -0.51 -1.14059 -0.69618 -0.18181 -0.06158 -0.09532
-1.50  16000 -0.29 -1.14059 -0.69618 -0.18181 -0.06158 -0.09532
-1.50  16000  0.00 -1.14059 -0.69618 -0.18181 -0.06158 -0.09532
-1.50  16000  0.28 -1.14059 -0.69618 -0.18181 -0.06158 -0.09532
-1.50  16000  0.50 -1.14059 -0.69618 -0.18181 -0.06158 -0.09532
-1.50  16000  0.60 -1.14059 -0.69618 -0.18181 -0.06158 -0.09532
-1.50  16000  0.87 -1.14059 -0.69618 -0.18181 -0.06158 -0.09532
-1.50  16000  1.00 -1.14059 -0.69618 -0.18181 -0.06158 -0.09532
-1.50  16000  1.50 -1.14059 -0.69618 -0.18181 -0.06158 -0.09532
-1.50  16000  2.00 -1.14059 -0.69618 -0.18181 -0.06158 -0.09532
-1.50  16000  2.50 -1.14059 -0.69618 -0.18181 -0.06158 -0.09532
-1.50  16000  3.00 -1.14804 -0.64119 -0.18953 -0.06869 -0.09787
-1.50  16000  3.50 -1.15569 -0.60632 -0.18733 -0.07197 -0.10016
-1.50  16000  4.00 -1.16114 -0.58189 -0.17999 -0.07421 -0.10281
-1.50  16000  4.50 -1.16461 -0.56379 -0.16938 -0.07589 -0.10577
-1.50  16000  5.00 -1.16642 -0.55119 -0.15513 -0.07779 -0.10888
-1.50  16000  5.50 -1.16642 -0.55119 -0.15513 -0.07779 -0.10888
-1.50  17000 -1.02 -1.28448 -0.75008 -0.19041 -0.06960 -0.10729
-1.50  17000 -0.70 -1.28448 -0.75008 -0.19041 -0.06960 -0.10729
-1.50  17000 -0.51 -1.28448 -0.75008 -0.19041 -0.06960 -0.10729
-1.50  17000 -0.29 -1.28448 -0.75008 -0.19041 -0.06960 -0.10729
-1.50  17000  0.00 -1.28448 -0.75008 -0.19041 -0.06960 -0.10729
-1.50  17000  0.28 -1.28448 -0.75008 -0.19041 -0.06960 -0.10729
-1.50  17000  0.50 -1.28448 -0.75008 -0.19041 -0.06960 -0.10729
-1.50  17000  0.60 -1.28448 -0.75008 -0.19041 -0.06960 -0.10729
-1.50  17000  0.87 -1.28448 -0.75008 -0.19041 -0.06960 -0.10729
-1.50  17000  1.00 -1.28448 -0.75008 -0.19041 -0.06960 -0.10729
-1.50  17000  1.50 -1.28448 -0.75008 -0.19041 -0.06960 -0.10729
-1.50  17000  2.00 -1.28448 -0.75008 -0.19041 -0.06960 -0.10729
-1.50  17000  2.50 -1.28448 -0.75008 -0.19041 -0.06960 -0.10729
-1.50  17000  3.00 -1.29781 -0.68934 -0.20086 -0.07719 -0.10945
-1.50  17000  3.50 -1.30996 -0.65296 -0.19935 -0.08006 -0.11120
-1.50  17000  4.00 -1.31823 -0.62839 -0.19255 -0.08176 -0.11325
-1.50  17000  4.50 -1.32318 -0.61126 -0.18237 -0.08317 -0.11572
-1.50  17000  5.00 -1.32567 -0.59994 -0.16896 -0.08464 -0.11841
-1.50  17000  5.50 -1.32567 -0.59994 -0.16896 -0.08464 -0.11841
-1.50  18000 -1.02 -1.41718 -0.80217 -0.19633 -0.07607 -0.11788
-1.50  18000 -0.70 -1.41718 -0.80217 -0.19633 -0.07607 -0.11788
-1.50  18000 -0.51 -1.41718 -0.80217 -0.19633 -0.07607 -0.11788
-1.50  18000 -0.29 -1.41718 -0.80217 -0.19633 -0.07607 -0.11788
-1.50  18000  0.00 -1.41718 -0.80217 -0.19633 -0.07607 -0.11788
-1.50  18000  0.28 -1.41718 -0.80217 -0.19633 -0.07607 -0.11788
-1.50  18000  0.50 -1.41718 -0.80217 -0.19633 -0.07607 -0.11788
-1.50  18000  0.60 -1.41718 -0.80217 -0.19633 -0.07607 -0.11788
-1.50  18000  0.87 -1.41718 -0.80217 -0.19633 -0.07607 -0.11788
-1.50  18000  1.00 -1.41718 -0.80217 -0.19633 -0.07607 -0.11788
-1.50  18000  1.50 -1.41718 -0.80217 -0.19633 -0.07607 -0.11788
-1.50  18000  2.00 -1.41718 -0.80217 -0.19633 -0.07607 -0.11788
-1.50  18000  2.50 -1.41718 -0.80217 -0.19633 -0.07607 -0.11788
-1.50  18000  3.00 -1.43626 -0.73437 -0.21032 -0.08549 -0.12023
-1.50  18000  3.50 -1.45352 -0.69612 -0.20991 -0.08821 -0.12132
-1.50  18000  4.00 -1.46526 -0.67098 -0.20369 -0.08953 -0.12287
-1.50  18000  4.50 -1.47243 -0.65376 -0.19405 -0.09058 -0.12487
-1.50  18000  5.00 -1.47611 -0.64325 -0.18136 -0.09174 -0.12713
-1.50  18000  5.50 -1.47611 -0.64325 -0.18136 -0.09174 -0.12713
-1.50  19000 -1.02 -1.54102 -0.85382 -0.20018 -0.07975 -0.12634
-1.50  19000 -0.70 -1.54102 -0.85382 -0.20018 -0.07975 -0.12634
-1.50  19000 -0.51 -1.54102 -0.85382 -0.20018 -0.07975 -0.12634
-1.50  19000 -0.29 -1.54102 -0.85382 -0.20018 -0.07975 -0.12634
-1.50  19000  0.00 -1.54102 -0.85382 -0.20018 -0.07975 -0.12634
-1.50  19000  0.28 -1.54102 -0.85382 -0.20018 -0.07975 -0.12634
-1.50  19000  0.50 -1.54102 -0.85382 -0.20018 -0.07975 -0.12634
-1.50  19000  0.60 -1.54102 -0.85382 -0.20018 -0.07975 -0.12634
-1.50  19000  0.87 -1.54102 -0.85382 -0.20018 -0.07975 -0.12634
-1.50  19000  1.00 -1.54102 -0.85382 -0.20018 -0.07975 -0.12634
-1.50  19000  1.50 -1.54102 -0.85382 -0.20018 -0.07975 -0.12634
-1.50  19000  2.00 -1.54102 -0.85382 -0.20018 -0.07975 -0.12634
-1.50  19000  2.50 -1.54102 -0.85382 -0.20018 -0.07975 -0.12634
-1.50  19000  3.00 -1.56467 -0.77668 -0.21841 -0.09257 -0.13037
-1.50  19000  3.50 -1.58751 -0.73563 -0.21967 -0.09562 -0.13101
-1.50  19000  4.00 -1.60329 -0.70966 -0.21411 -0.09661 -0.13191
-1.50  19000  4.50 -1.61296 -0.69233 -0.20486 -0.09751 -0.13339
-1.50  19000  5.00 -1.61827 -0.68185 -0.19272 -0.09849 -0.13530
-1.50  19000  5.50 -1.61827 -0.68185 -0.19272 -0.09849 -0.13530
-1.50  20000 -1.02 -1.68385 -0.81727 -0.22506 -0.09840 -0.13949
-1.50  20000 -0.70 -1.68385 -0.81727 -0.22506 -0.09840 -0.13949
-1.50  20000 -0.51 -1.68385 -0.81727 -0.22506 -0.09840 -0.13949
-1.50  20000 -0.29 -1.68385 -0.81727 -0.22506 -0.09840 -0.13949
-1.50  20000  0.00 -1.68385 -0.81727 -0.22506 -0.09840 -0.13949
-1.50  20000  0.28 -1.68385 -0.81727 -0.22506 -0.09840 -0.13949
-1.50  20000  0.50 -1.68385 -0.81727 -0.22506 -0.09840 -0.13949
-1.50  20000  0.60 -1.68385 -0.81727 -0.22506 -0.09840 -0.13949
-1.50  20000  0.87 -1.68385 -0.81727 -0.22506 -0.09840 -0.13949
-1.50  20000  1.00 -1.68385 -0.81727 -0.22506 -0.09840 -0.13949
-1.50  20000  1.50 -1.68385 -0.81727 -0.22506 -0.09840 -0.13949
-1.50  20000  2.00 -1.68385 -0.81727 -0.22506 -0.09840 -0.13949
-1.50  20000  2.50 -1.68385 -0.81727 -0.22506 -0.09840 -0.13949
-1.50  20000  3.00 -1.68385 -0.81727 -0.22506 -0.09840 -0.13949
-1.50  20000  3.50 -1.71233 -0.77251 -0.22828 -0.10235 -0.14015
-1.50  20000  4.00 -1.73255 -0.74529 -0.22378 -0.10334 -0.14047
-1.50  20000  4.50 -1.74524 -0.72765 -0.21529 -0.10388 -0.14152
-1.50  20000  5.00 -1.75256 -0.71689 -0.20378 -0.10464 -0.14304
-1.50  20000  5.50 -1.75256 -0.71689 -0.20378 -0.10464 -0.14304
-1.50  21000 -1.02 -1.79572 -0.85624 -0.23062 -0.10270 -0.14710
-1.50  21000 -0.70 -1.79572 -0.85624 -0.23062 -0.10270 -0.14710
-1.50  21000 -0.51 -1.79572 -0.85624 -0.23062 -0.10270 -0.14710
-1.50  21000 -0.29 -1.79572 -0.85624 -0.23062 -0.10270 -0.14710
-1.50  21000  0.00 -1.79572 -0.85624 -0.23062 -0.10270 -0.14710
-1.50  21000  0.28 -1.79572 -0.85624 -0.23062 -0.10270 -0.14710
-1.50  21000  0.50 -1.79572 -0.85624 -0.23062 -0.10270 -0.14710
-1.50  21000  0.60 -1.79572 -0.85624 -0.23062 -0.10270 -0.14710
-1.50  21000  0.87 -1.79572 -0.85624 -0.23062 -0.10270 -0.14710
-1.50  21000  1.00 -1.79572 -0.85624 -0.23062 -0.10270 -0.14710
-1.50  21000  1.50 -1.79572 -0.85624 -0.23062 -0.10270 -0.14710
-1.50  21000  2.00 -1.79572 -0.85624 -0.23062 -0.10270 -0.14710
-1.50  21000  2.50 -1.79572 -0.85624 -0.23062 -0.10270 -0.14710
-1.50  21000  3.00 -1.79572 -0.85624 -0.23062 -0.10270 -0.14710
-1.50  21000  3.50 -1.82879 -0.80780 -0.23586 -0.10821 -0.14863
-1.50  21000  4.00 -1.85345 -0.77870 -0.23247 -0.10957 -0.14879
-1.50  21000  4.50 -1.86947 -0.76038 -0.22474 -0.11001 -0.14931
-1.50  21000  5.00 -1.87899 -0.74941 -0.21400 -0.11056 -0.15034
-1.50  21000  5.50 -1.87899 -0.74941 -0.21400 -0.11056 -0.15034
-1.50  22000 -1.02 -1.90195 -0.89185 -0.23574 -0.10618 -0.15340
-1.50  22000 -0.70 -1.90195 -0.89185 -0.23574 -0.10618 -0.15340
-1.50  22000 -0.51 -1.90195 -0.89185 -0.23574 -0.10618 -0.15340
-1.50  22000 -0.29 -1.90195 -0.89185 -0.23574 -0.10618 -0.15340
-1.50  22000  0.00 -1.90195 -0.89185 -0.23574 -0.10618 -0.15340
-1.50  22000  0.28 -1.90195 -0.89185 -0.23574 -0.10618 -0.15340
-1.50  22000  0.50 -1.90195 -0.89185 -0.23574 -0.10618 -0.15340
-1.50  22000  0.60 -1.90195 -0.89185 -0.23574 -0.10618 -0.15340
-1.50  22000  0.87 -1.90195 -0.89185 -0.23574 -0.10618 -0.15340
-1.50  22000  1.00 -1.90195 -0.89185 -0.23574 -0.10618 -0.15340
-1.50  22000  1.50 -1.90195 -0.89185 -0.23574 -0.10618 -0.15340
-1.50  22000  2.00 -1.90195 -0.89185 -0.23574 -0.10618 -0.15340
-1.50  22000  2.50 -1.90195 -0.89185 -0.23574 -0.10618 -0.15340
-1.50  22000  3.00 -1.90195 -0.89185 -0.23574 -0.10618 -0.15340
-1.50  22000  3.50 -1.93782 -0.84145 -0.24209 -0.11312 -0.15611
-1.50  22000  4.00 -1.96652 -0.81038 -0.24015 -0.11522 -0.15651
-1.50  22000  4.50 -1.98614 -0.79109 -0.23348 -0.11574 -0.15675
-1.50  22000  5.00 -1.99816 -0.77972 -0.22350 -0.11618 -0.15746
-1.50  22000  5.50 -1.99816 -0.77972 -0.22350 -0.11618 -0.15746
-1.50  23000 -1.02 -2.00421 -0.92276 -0.24142 -0.10992 -0.15891
-1.50  23000 -0.70 -2.00421 -0.92276 -0.24142 -0.10992 -0.15891
-1.50  23000 -0.51 -2.00421 -0.92276 -0.24142 -0.10992 -0.15891
-1.50  23000 -0.29 -2.00421 -0.92276 -0.24142 -0.10992 -0.15891
-1.50  23000  0.00 -2.00421 -0.92276 -0.24142 -0.10992 -0.15891
-1.50  23000  0.28 -2.00421 -0.92276 -0.24142 -0.10992 -0.15891
-1.50  23000  0.50 -2.00421 -0.92276 -0.24142 -0.10992 -0.15891
-1.50  23000  0.60 -2.00421 -0.92276 -0.24142 -0.10992 -0.15891
-1.50  23000  0.87 -2.00421 -0.92276 -0.24142 -0.10992 -0.15891
-1.50  23000  1.00 -2.00421 -0.92276 -0.24142 -0.10992 -0.15891
-1.50  23000  1.50 -2.00421 -0.92276 -0.24142 -0.10992 -0.15891
-1.50  23000  2.00 -2.00421 -0.92276 -0.24142 -0.10992 -0.15891
-1.50  23000  2.50 -2.00421 -0.92276 -0.24142 -0.10992 -0.15891
-1.50  23000  3.00 -2.00421 -0.92276 -0.24142 -0.10992 -0.15891
-1.50  23000  3.50 -2.04094 -0.87269 -0.24783 -0.11750 -0.16238
-1.50  23000  4.00 -2.07256 -0.84046 -0.24687 -0.12022 -0.16355
-1.50  23000  4.50 -2.09544 -0.81991 -0.24128 -0.12104 -0.16382
-1.50  23000  5.00 -2.11004 -0.80797 -0.23216 -0.12158 -0.16427
-1.50  23000  5.50 -2.11004 -0.80797 -0.23216 -0.12158 -0.16427
-1.50  24000 -1.02 -2.10089 -0.94975 -0.24748 -0.11421 -0.16471
-1.50  24000 -0.70 -2.10089 -0.94975 -0.24748 -0.11421 -0.16471
-1.50  24000 -0.51 -2.10089 -0.94975 -0.24748 -0.11421 -0.16471
-1.50  24000 -0.29 -2.10089 -0.94975 -0.24748 -0.11421 -0.16471
-1.50  24000  0.00 -2.10089 -0.94975 -0.24748 -0.11421 -0.16471
-1.50  24000  0.28 -2.10089 -0.94975 -0.24748 -0.11421 -0.16471
-1.50  24000  0.50 -2.10089 -0.94975 -0.24748 -0.11421 -0.16471
-1.50  24000  0.60 -2.10089 -0.94975 -0.24748 -0.11421 -0.16471
-1.50  24000  0.87 -2.10089 -0.94975 -0.24748 -0.11421 -0.16471
-1.50  24000  1.00 -2.10089 -0.94975 -0.24748 -0.11421 -0.16471
-1.50  24000  1.50 -2.10089 -0.94975 -0.24748 -0.11421 -0.16471
-1.50  24000  2.00 -2.10089 -0.94975 -0.24748 -0.11421 -0.16471
-1.50  24000  2.50 -2.10089 -0.94975 -0.24748 -0.11421 -0.16471
-1.50  24000  3.00 -2.10089 -0.94975 -0.24748 -0.11421 -0.16471
-1.50  24000  3.50 -2.13990 -0.90039 -0.25375 -0.12150 -0.16817
-1.50  24000  4.00 -2.17300 -0.86865 -0.25299 -0.12461 -0.16961
-1.50  24000  4.50 -2.19806 -0.84741 -0.24811 -0.12604 -0.17044
-1.50  24000  5.00 -2.21516 -0.83444 -0.24008 -0.12673 -0.17084
-1.50  24000  5.50 -2.21516 -0.83444 -0.24008 -0.12673 -0.17084
-1.50  25000 -1.02 -2.18947 -0.97592 -0.25234 -0.11774 -0.17067
-1.50  25000 -0.70 -2.18947 -0.97592 -0.25234 -0.11774 -0.17067
-1.50  25000 -0.51 -2.18947 -0.97592 -0.25234 -0.11774 -0.17067
-1.50  25000 -0.29 -2.18947 -0.97592 -0.25234 -0.11774 -0.17067
-1.50  25000  0.00 -2.18947 -0.97592 -0.25234 -0.11774 -0.17067
-1.50  25000  0.28 -2.18947 -0.97592 -0.25234 -0.11774 -0.17067
-1.50  25000  0.50 -2.18947 -0.97592 -0.25234 -0.11774 -0.17067
-1.50  25000  0.60 -2.18947 -0.97592 -0.25234 -0.11774 -0.17067
-1.50  25000  0.87 -2.18947 -0.97592 -0.25234 -0.11774 -0.17067
-1.50  25000  1.00 -2.18947 -0.97592 -0.25234 -0.11774 -0.17067
-1.50  25000  1.50 -2.18947 -0.97592 -0.25234 -0.11774 -0.17067
-1.50  25000  2.00 -2.18947 -0.97592 -0.25234 -0.11774 -0.17067
-1.50  25000  2.50 -2.18947 -0.97592 -0.25234 -0.11774 -0.17067
-1.50  25000  3.00 -2.18947 -0.97592 -0.25234 -0.11774 -0.17067
-1.50  25000  3.50 -2.23446 -0.92441 -0.26023 -0.12590 -0.17388
-1.50  25000  4.00 -2.26891 -0.89416 -0.25905 -0.12875 -0.17522
-1.50  25000  4.50 -2.29524 -0.87323 -0.25464 -0.13042 -0.17612
-1.50  25000  5.00 -2.31406 -0.86004 -0.24720 -0.13145 -0.17691
-1.50  25000  5.50 -2.31406 -0.86004 -0.24720 -0.13145 -0.17691
-1.50  26000 -1.02 -2.27133 -1.00313 -0.25431 -0.11870 -0.17481
-1.50  26000 -0.70 -2.27133 -1.00313 -0.25431 -0.11870 -0.17481
-1.50  26000 -0.51 -2.27133 -1.00313 -0.25431 -0.11870 -0.17481
-1.50  26000 -0.29 -2.27133 -1.00313 -0.25431 -0.11870 -0.17481
-1.50  26000  0.00 -2.27133 -1.00313 -0.25431 -0.11870 -0.17481
-1.50  26000  0.28 -2.27133 -1.00313 -0.25431 -0.11870 -0.17481
-1.50  26000  0.50 -2.27133 -1.00313 -0.25431 -0.11870 -0.17481
-1.50  26000  0.60 -2.27133 -1.00313 -0.25431 -0.11870 -0.17481
-1.50  26000  0.87 -2.27133 -1.00313 -0.25431 -0.11870 -0.17481
-1.50  26000  1.00 -2.27133 -1.00313 -0.25431 -0.11870 -0.17481
-1.50  26000  1.50 -2.27133 -1.00313 -0.25431 -0.11870 -0.17481
-1.50  26000  2.00 -2.27133 -1.00313 -0.25431 -0.11870 -0.17481
-1.50  26000  2.50 -2.27133 -1.00313 -0.25431 -0.11870 -0.17481
-1.50  26000  3.00 -2.27133 -1.00313 -0.25431 -0.11870 -0.17481
-1.50  26000  3.50 -2.32362 -0.94575 -0.26689 -0.13101 -0.17989
-1.50  26000  4.00 -2.36083 -0.91679 -0.26540 -0.13271 -0.18062
-1.50  26000  4.50 -2.38769 -0.89711 -0.26079 -0.13452 -0.18157
-1.50  26000  5.00 -2.40750 -0.88394 -0.25393 -0.13581 -0.18231
-1.50  26000  5.50 -2.40750 -0.88394 -0.25393 -0.13581 -0.18231
-1.50  27000 -1.02 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
-1.50  27000 -0.70 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
-1.50  27000 -0.51 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
-1.50  27000 -0.29 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
-1.50  27000  0.00 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
-1.50  27000  0.28 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
-1.50  27000  0.50 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
-1.50  27000  0.60 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
-1.50  27000  0.87 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
-1.50  27000  1.00 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
-1.50  27000  1.50 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
-1.50  27000  2.00 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
-1.50  27000  2.50 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
-1.50  27000  3.00 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
-1.50  27000  3.50 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
-1.50  27000  4.00 -2.44762 -0.93680 -0.27204 -0.13755 -0.18642
-1.50  27000  4.50 -2.47615 -0.91840 -0.26722 -0.13876 -0.18637
-1.50  27000  5.00 -2.49617 -0.90623 -0.26042 -0.13984 -0.18750
-1.50  27000  5.50 -2.49617 -0.90623 -0.26042 -0.13984 -0.18750
-1.50  28000 -1.02 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
-1.50  28000 -0.70 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
-1.50  28000 -0.51 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
-1.50  28000 -0.29 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
-1.50  28000  0.00 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
-1.50  28000  0.28 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
-1.50  28000  0.50 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
-1.50  28000  0.60 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
-1.50  28000  0.87 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
-1.50  28000  1.00 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
-1.50  28000  1.50 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
-1.50  28000  2.00 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
-1.50  28000  2.50 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
-1.50  28000  3.00 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
-1.50  28000  3.50 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
-1.50  28000  4.00 -2.52892 -0.95542 -0.27879 -0.14239 -0.19191
-1.50  28000  4.50 -2.55968 -0.93767 -0.27357 -0.14305 -0.19198
-1.50  28000  5.00 -2.58074 -0.92661 -0.26689 -0.14394 -0.19221
-1.50  28000  5.50 -2.58074 -0.92661 -0.26689 -0.14394 -0.19221
-1.50  29000 -1.02 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
-1.50  29000 -0.70 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
-1.50  29000 -0.51 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
-1.50  29000 -0.29 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
-1.50  29000  0.00 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
-1.50  29000  0.28 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
-1.50  29000  0.50 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
-1.50  29000  0.60 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
-1.50  29000  0.87 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
-1.50  29000  1.00 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
-1.50  29000  1.50 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
-1.50  29000  2.00 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
-1.50  29000  2.50 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
-1.50  29000  3.00 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
-1.50  29000  3.50 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
-1.50  29000  4.00 -2.60355 -0.97348 -0.28526 -0.14669 -0.19743
-1.50  29000  4.50 -2.63831 -0.95559 -0.28024 -0.14743 -0.19725
-1.50  29000  5.00 -2.66049 -0.94535 -0.27337 -0.14777 -0.19732
-1.50  29000  5.50 -2.66049 -0.94535 -0.27337 -0.14777 -0.19732
-1.50  30000 -1.02 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
-1.50  30000 -0.70 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
-1.50  30000 -0.51 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
-1.50  30000 -0.29 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
-1.50  30000  0.00 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
-1.50  30000  0.28 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
-1.50  30000  0.50 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
-1.50  30000  0.60 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
-1.50  30000  0.87 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
-1.50  30000  1.00 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
-1.50  30000  1.50 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
-1.50  30000  2.00 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
-1.50  30000  2.50 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
-1.50  30000  3.00 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
-1.50  30000  3.50 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
-1.50  30000  4.00 -2.67342 -0.99115 -0.29078 -0.15018 -0.20214
-1.50  30000  4.50 -2.71144 -0.97261 -0.28665 -0.15153 -0.20230
-1.50  30000  5.00 -2.73584 -0.96265 -0.27981 -0.15176 -0.20218
-1.50  30000  5.50 -2.73584 -0.96265 -0.27981 -0.15176 -0.20218
-1.50  31000 -1.02 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
-1.50  31000 -0.70 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
-1.50  31000 -0.51 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
-1.50  31000 -0.29 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
-1.50  31000  0.00 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
-1.50  31000  0.28 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
-1.50  31000  0.50 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
-1.50  31000  0.60 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
-1.50  31000  0.87 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
-1.50  31000  1.00 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
-1.50  31000  1.50 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
-1.50  31000  2.00 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
-1.50  31000  2.50 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
-1.50  31000  3.00 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
-1.50  31000  3.50 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
-1.50  31000  4.00 -2.74139 -1.00792 -0.29531 -0.15236 -0.20587
-1.50  31000  4.50 -2.78037 -0.98904 -0.29260 -0.15499 -0.20676
-1.50  31000  5.00 -2.80661 -0.97910 -0.28612 -0.15544 -0.20660
-1.50  31000  5.50 -2.80661 -0.97910 -0.28612 -0.15544 -0.20660
-1.50  32000 -1.02 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
-1.50  32000 -0.70 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
-1.50  32000 -0.51 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
-1.50  32000 -0.29 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
-1.50  32000  0.00 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
-1.50  32000  0.28 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
-1.50  32000  0.50 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
-1.50  32000  0.60 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
-1.50  32000  0.87 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
-1.50  32000  1.00 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
-1.50  32000  1.50 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
-1.50  32000  2.00 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
-1.50  32000  2.50 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
-1.50  32000  3.00 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
-1.50  32000  3.50 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
-1.50  32000  4.00 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
-1.50  32000  4.50 -2.84690 -1.00455 -0.29779 -0.15760 -0.21048
-1.50  32000  5.00 -2.87420 -0.99452 -0.29207 -0.15854 -0.21056
-1.50  32000  5.50 -2.87420 -0.99452 -0.29207 -0.15854 -0.21056
-1.50  33000 -1.02 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
-1.50  33000 -0.70 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
-1.50  33000 -0.51 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
-1.50  33000 -0.29 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
-1.50  33000  0.00 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
-1.50  33000  0.28 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
-1.50  33000  0.50 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
-1.50  33000  0.60 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
-1.50  33000  0.87 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
-1.50  33000  1.00 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
-1.50  33000  1.50 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
-1.50  33000  2.00 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
-1.50  33000  2.50 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
-1.50  33000  3.00 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
-1.50  33000  3.50 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
-1.50  33000  4.00 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
-1.50  33000  4.50 -2.91195 -1.01898 -0.30219 -0.15942 -0.21359
-1.50  33000  5.00 -2.93942 -1.00904 -0.29737 -0.16111 -0.21400
-1.50  33000  5.50 -2.93942 -1.00904 -0.29737 -0.16111 -0.21400
-1.50  34000 -1.02 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
-1.50  34000 -0.70 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
-1.50  34000 -0.51 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
-1.50  34000 -0.29 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
-1.50  34000  0.00 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
-1.50  34000  0.28 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
-1.50  34000  0.50 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
-1.50  34000  0.60 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
-1.50  34000  0.87 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
-1.50  34000  1.00 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
-1.50  34000  1.50 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
-1.50  34000  2.00 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
-1.50  34000  2.50 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
-1.50  34000  3.00 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
-1.50  34000  3.50 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
-1.50  34000  4.00 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
-1.50  34000  4.50 -2.97899 -1.03181 -0.30566 -0.16027 -0.21590
-1.50  34000  5.00 -3.00391 -1.02231 -0.30213 -0.16293 -0.21688
-1.50  34000  5.50 -3.00391 -1.02231 -0.30213 -0.16293 -0.21688
-1.50  35000 -1.02 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
-1.50  35000 -0.70 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
-1.50  35000 -0.51 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
-1.50  35000 -0.29 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
-1.50  35000  0.00 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
-1.50  35000  0.28 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
-1.50  35000  0.50 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
-1.50  35000  0.60 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
-1.50  35000  0.87 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
-1.50  35000  1.00 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
-1.50  35000  1.50 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
-1.50  35000  2.00 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
-1.50  35000  2.50 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
-1.50  35000  3.00 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
-1.50  35000  3.50 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
-1.50  35000  4.00 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
-1.50  35000  4.50 -3.04809 -1.04294 -0.30835 -0.16031 -0.21740
-1.50  35000  5.00 -3.06963 -1.03422 -0.30595 -0.16402 -0.21917
-1.50  35000  5.50 -3.06963 -1.03422 -0.30595 -0.16402 -0.21917
-1.50  37500 -1.02 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
-1.50  37500 -0.70 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
-1.50  37500 -0.51 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
-1.50  37500 -0.29 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
-1.50  37500  0.00 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
-1.50  37500  0.28 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
-1.50  37500  0.50 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
-1.50  37500  0.60 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
-1.50  37500  0.87 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
-1.50  37500  1.00 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
-1.50  37500  1.50 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
-1.50  37500  2.00 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
-1.50  37500  2.50 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
-1.50  37500  3.00 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
-1.50  37500  3.50 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
-1.50  37500  4.00 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
-1.50  37500  4.50 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
-1.50  37500  5.00 -3.24088 -1.05803 -0.31302 -0.16379 -0.22209
-1.50  37500  5.50 -3.24088 -1.05803 -0.31302 -0.16379 -0.22209
-1.50  40000 -1.02 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
-1.50  40000 -0.70 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
-1.50  40000 -0.51 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
-1.50  40000 -0.29 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
-1.50  40000  0.00 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
-1.50  40000  0.28 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
-1.50  40000  0.50 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
-1.50  40000  0.60 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
-1.50  40000  0.87 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
-1.50  40000  1.00 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
-1.50  40000  1.50 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
-1.50  40000  2.00 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
-1.50  40000  2.50 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
-1.50  40000  3.00 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
-1.50  40000  3.50 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
-1.50  40000  4.00 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
-1.50  40000  4.50 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
-1.50  40000  5.00 -3.42493 -1.07283 -0.31872 -0.16266 -0.22304
-1.50  40000  5.50 -3.42493 -1.07283 -0.31872 -0.16266 -0.22304
-1.50  42500 -1.02 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
-1.50  42500 -0.70 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
-1.50  42500 -0.51 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
-1.50  42500 -0.29 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
-1.50  42500  0.00 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
-1.50  42500  0.28 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
-1.50  42500  0.50 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
-1.50  42500  0.60 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
-1.50  42500  0.87 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
-1.50  42500  1.00 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
-1.50  42500  1.50 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
-1.50  42500  2.00 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
-1.50  42500  2.50 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
-1.50  42500  3.00 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
-1.50  42500  3.50 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
-1.50  42500  4.00 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
-1.50  42500  4.50 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
-1.50  42500  5.00 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
-1.50  42500  5.50 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
-1.50  45000 -1.02 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
-1.50  45000 -0.70 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
-1.50  45000 -0.51 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
-1.50  45000 -0.29 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
-1.50  45000  0.00 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
-1.50  45000  0.28 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
-1.50  45000  0.50 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
-1.50  45000  0.60 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
-1.50  45000  0.87 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
-1.50  45000  1.00 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
-1.50  45000  1.50 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
-1.50  45000  2.00 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
-1.50  45000  2.50 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
-1.50  45000  3.00 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
-1.50  45000  3.50 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
-1.50  45000  4.00 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
-1.50  45000  4.50 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
-1.50  45000  5.00 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
-1.50  45000  5.50 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
-1.50  47500 -1.02 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
-1.50  47500 -0.70 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
-1.50  47500 -0.51 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
-1.50  47500 -0.29 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
-1.50  47500  0.00 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
-1.50  47500  0.28 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
-1.50  47500  0.50 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
-1.50  47500  0.60 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
-1.50  47500  0.87 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
-1.50  47500  1.00 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
-1.50  47500  1.50 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
-1.50  47500  2.00 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
-1.50  47500  2.50 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
-1.50  47500  3.00 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
-1.50  47500  3.50 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
-1.50  47500  4.00 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
-1.50  47500  4.50 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
-1.50  47500  5.00 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
-1.50  47500  5.50 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
-1.50  50000 -1.02 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
-1.50  50000 -0.70 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
-1.50  50000 -0.51 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
-1.50  50000 -0.29 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
-1.50  50000  0.00 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
-1.50  50000  0.28 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
-1.50  50000  0.50 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
-1.50  50000  0.60 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
-1.50  50000  0.87 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
-1.50  50000  1.00 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
-1.50  50000  1.50 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
-1.50  50000  2.00 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
-1.50  50000  2.50 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
-1.50  50000  3.00 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
-1.50  50000  3.50 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
-1.50  50000  4.00 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
-1.50  50000  4.50 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
-1.50  50000  5.00 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
-1.50  50000  5.50 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
-0.50  2000 -1.02 -7.05491 -0.29783  1.12030  3.74948  3.02679
-0.50  2000 -0.70 -7.05491 -0.29783  1.12030  3.74948  3.02679
-0.50  2000 -0.51 -7.05491 -0.29783  1.12030  3.74948  3.02679
-0.50  2000 -0.29 -7.05491 -0.29783  1.12030  3.74948  3.02679
-0.50  2000  0.00 -7.05491 -0.29783  1.12030  3.74948  3.02679
-0.50  2000  0.28 -7.05491 -0.29783  1.12030  3.74948  3.02679
-0.50  2000  0.50 -7.05491 -0.29783  1.12030  3.74948  3.02679
-0.50  2000  0.60 -7.05491 -0.29783  1.12030  3.74948  3.02679
-0.50  2000  0.87 -7.05491 -0.29783  1.12030  3.74948  3.02679
-0.50  2000  1.00 -7.05491 -0.29783  1.12030  3.74948  3.02679
-0.50  2000  1.50 -7.05491 -0.29783  1.12030  3.74948  3.02679
-0.50  2000  2.00 -7.05491 -0.29783  1.12030  3.74948  3.02679
-0.50  2000  2.50 -7.05491 -0.29783  1.12030  3.74948  3.02679
-0.50  2000  3.00 -7.05491 -0.29783  1.12030  3.74948  3.02679
-0.50  2000  3.50 -7.05491 -0.29783  1.12030  3.74948  3.02679
-0.50  2000  4.00 -7.19603  0.30737  1.98438  3.83970  3.00860
-0.50  2000  4.50 -7.36220  1.35906  2.55481  3.97758  2.99233
-0.50  2000  5.00 -7.45940  2.69098  2.84060  4.13058  2.93958
-0.50  2000  5.50 -7.37559  3.82038  2.96376  4.22386  2.83140
-0.50  2200 -1.02 -5.74734  0.00490  1.08443  3.18965  2.78974
-0.50  2200 -0.70 -5.74734  0.00490  1.08443  3.18965  2.78974
-0.50  2200 -0.51 -5.74734  0.00490  1.08443  3.18965  2.78974
-0.50  2200 -0.29 -5.74734  0.00490  1.08443  3.18965  2.78974
-0.50  2200  0.00 -5.74734  0.00490  1.08443  3.18965  2.78974
-0.50  2200  0.28 -5.74734  0.00490  1.08443  3.18965  2.78974
-0.50  2200  0.50 -5.74734  0.00490  1.08443  3.18965  2.78974
-0.50  2200  0.60 -5.74734  0.00490  1.08443  3.18965  2.78974
-0.50  2200  0.87 -5.74734  0.00490  1.08443  3.18965  2.78974
-0.50  2200  1.00 -5.74734  0.00490  1.08443  3.18965  2.78974
-0.50  2200  1.50 -5.74734  0.00490  1.08443  3.18965  2.78974
-0.50  2200  2.00 -5.74734  0.00490  1.08443  3.18965  2.78974
-0.50  2200  2.50 -5.74734  0.00490  1.08443  3.18965  2.78974
-0.50  2200  3.00 -5.74734  0.00490  1.08443  3.18965  2.78974
-0.50  2200  3.50 -5.74734  0.00490  1.08443  3.18965  2.78974
-0.50  2200  4.00 -5.83881  0.47459  1.74295  3.24982  2.77385
-0.50  2200  4.50 -6.03314  1.22234  2.23193  3.37866  2.79895
-0.50  2200  5.00 -6.18971  2.28050  2.56496  3.53287  2.77720
-0.50  2200  5.50 -6.26440  3.40841  2.73528  3.68918  2.70344
-0.50  2500 -1.02 -5.99423 -0.23091  1.41747  3.08183  2.87080
-0.50  2500 -0.70 -5.60200 -0.23088  1.40344  2.89877  2.78610
-0.50  2500 -0.51 -5.37788 -0.23088  1.39685  2.79747  2.73588
-0.50  2500 -0.29 -5.11838 -0.23088  1.38922  2.68017  2.67773
-0.50  2500  0.00 -4.94869 -0.23089  1.38976  2.66402  2.63922
-0.50  2500  0.28 -4.78486 -0.23089  1.39029  2.64842  2.60205
-0.50  2500  0.50 -4.76560 -0.18071  1.37907  2.64935  2.59194
-0.50  2500  0.60 -4.75684 -0.15790  1.37396  2.64977  2.58735
-0.50  2500  0.87 -4.73319 -0.09632  1.36019  2.65092  2.57496
-0.50  2500  1.00 -4.72181 -0.06667  1.35355  2.65147  2.56899
-0.50  2500  1.50 -4.67802  0.04737  1.32804  2.65358  2.54603
-0.50  2500  2.00 -4.63424  0.16141  1.30253  2.65570  2.52308
-0.50  2500  2.50 -4.59045  0.27545  1.27702  2.65782  2.50012
-0.50  2500  3.00 -4.54667  0.38949  1.25151  2.65993  2.47717
-0.50  2500  3.50 -4.50288  0.50353  1.22600  2.66205  2.45421
-0.50  2500  4.00 -4.48830  0.86898  1.63556  2.68648  2.40440
-0.50  2500  4.50 -4.57277  1.35470  1.95371  2.74300  2.41254
-0.50  2500  5.00 -4.71326  2.01739  2.21663  2.84811  2.42855
-0.50  2500  5.50 -4.90478  2.87751  2.40505  3.01804  2.42611
-0.50  2800 -1.02 -5.77601  0.06061  1.45009  2.93050  2.86379
-0.50  2800 -0.70 -5.43583  0.06067  1.44188  2.80713  2.76515
-0.50  2800 -0.51 -5.23440  0.06063  1.43772  2.73494  2.70608
-0.50  2800 -0.29 -5.00117  0.06059  1.43290  2.65135  2.63769
-0.50  2800  0.00 -4.74361  0.06062  1.43026  2.58145  2.56286
-0.50  2800  0.28 -4.49493  0.06065  1.42772  2.51396  2.49061
-0.50  2800  0.50 -4.31873  0.06069  1.42664  2.47409  2.44065
-0.50  2800  0.60 -4.23864  0.06072  1.42615  2.45597  2.41793
-0.50  2800  0.87 -4.17625  0.12522  1.42588  2.44120  2.38421
-0.50  2800  1.00 -4.14622  0.15628  1.42575  2.43408  2.36797
-0.50  2800  1.50 -4.03069  0.27573  1.42524  2.40672  2.30551
-0.50  2800  2.00 -3.91516  0.39519  1.42474  2.37937  2.24306
-0.50  2800  2.50 -3.79963  0.51464  1.42423  2.35201  2.18060
-0.50  2800  3.00 -3.68411  0.63410  1.42373  2.32465  2.11815
-0.50  2800  3.50 -3.56858  0.75355  1.42323  2.29729  2.05569
-0.50  2800  4.00 -3.51492  0.96750  1.53825  2.26781  2.07231
-0.50  2800  4.50 -3.49497  1.26795  1.76250  2.28497  2.04218
-0.50  2800  5.00 -3.54027  1.67907  1.93425  2.32187  2.05317
-0.50  2800  5.50 -3.68272  2.24496  2.10360  2.43354  2.07241
-0.50  3000 -1.02 -5.47382  0.36024  1.40529  2.75583  2.84762
-0.50  3000 -0.70 -5.47382  0.36024  1.40529  2.75583  2.84762
-0.50  3000 -0.51 -5.28739  0.36026  1.40347  2.69997  2.78041
-0.50  3000 -0.29 -5.07153  0.36028  1.40136  2.63529  2.70259
-0.50  3000  0.00 -4.80249  0.36034  1.39931  2.54921  2.61283
-0.50  3000  0.28 -4.54273  0.36040  1.39732  2.46610  2.52617
-0.50  3000  0.50 -4.35102  0.36039  1.39610  2.39962  2.46743
-0.50  3000  0.60 -4.26388  0.36039  1.39555  2.36940  2.44073
-0.50  3000  0.87 -4.15648  0.40222  1.38471  2.33666  2.39391
-0.50  3000  1.00 -4.10477  0.42236  1.37950  2.32090  2.37137
-0.50  3000  1.50 -3.90588  0.49983  1.35943  2.26027  2.28466
-0.50  3000  2.00 -3.70700  0.57730  1.33937  2.19964  2.19795
-0.50  3000  2.50 -3.50811  0.65477  1.31931  2.13902  2.11125
-0.50  3000  3.00 -3.30922  0.73224  1.29925  2.07839  2.02454
-0.50  3000  3.50 -3.11034  0.80971  1.27919  2.01776  1.93783
-0.50  3000  4.00 -2.90552  0.93787  1.48985  1.97737  1.83904
-0.50  3000  4.50 -2.83379  1.12392  1.63701  1.96749  1.80477
-0.50  3000  5.00 -2.85733  1.42376  1.74949  1.98494  1.82639
-0.50  3000  5.50 -2.91549  1.80241  1.89095  2.04140  1.83255
-0.50  3200 -1.02 -4.87297  0.92479  1.40658  2.68163  2.58168
-0.50  3200 -0.70 -4.87297  0.92479  1.40658  2.68163  2.58168
-0.50  3200 -0.51 -4.65718  0.92484  1.40395  2.53605  2.53273
-0.50  3200 -0.29 -4.40743  0.92479  1.40102  2.37116  2.47323
-0.50  3200  0.00 -4.10542  0.92479  1.39755  2.18036  2.39733
-0.50  3200  0.28 -3.81383  0.92478  1.39420  1.99615  2.32405
-0.50  3200  0.50 -3.63182  0.92478  1.39169  1.88646  2.27898
-0.50  3200  0.60 -3.54908  0.92478  1.39055  1.83661  2.25850
-0.50  3200  0.87 -3.47284  0.92510  1.36337  1.83084  2.21901
-0.50  3200  1.00 -3.43612  0.92526  1.35029  1.82806  2.19999
-0.50  3200  1.50 -3.29493  0.92586  1.29996  1.81738  2.12687
-0.50  3200  2.00 -3.15373  0.92647  1.24963  1.80670  2.05374
-0.50  3200  2.50 -3.01254  0.92707  1.19930  1.79602  1.98061
-0.50  3200  3.00 -2.87134  0.92767  1.14897  1.78533  1.90748
-0.50  3200  3.50 -2.73015  0.92828  1.09865  1.77465  1.83435
-0.50  3200  4.00 -2.41408  0.97196  1.32609  1.71272  1.68227
-0.50  3200  4.50 -2.27806  1.07727  1.47048  1.68868  1.61161
-0.50  3200  5.00 -2.23010  1.24630  1.58851  1.68878  1.58464
-0.50  3200  5.50 -2.23763  1.49731  1.70081  1.71259  1.58287
-0.50  3350 -1.02 -3.70715  1.27649  1.52669  2.17898  2.15013
-0.50  3350 -0.70 -3.70715  1.27649  1.52669  2.17898  2.15013
-0.50  3350 -0.51 -3.70715  1.27649  1.52669  2.17898  2.15013
-0.50  3350 -0.29 -3.53199  1.27643  1.51682  2.05070  2.10716
-0.50  3350  0.00 -3.33223  1.27647  1.50696  1.90831  2.05377
-0.50  3350  0.28 -3.13937  1.27651  1.49743  1.77084  2.00222
-0.50  3350  0.50 -3.12220  1.27648  1.49498  1.74974  2.00430
-0.50  3350  0.60 -3.11440  1.27647  1.49387  1.74015  2.00524
-0.50  3350  0.87 -3.09301  1.27651  1.49104  1.71444  2.00711
-0.50  3350  1.00 -3.05528  1.26360  1.46738  1.70672  1.99096
-0.50  3350  1.50 -2.91017  1.21392  1.37640  1.67702  1.92887
-0.50  3350  2.00 -2.76507  1.16425  1.28542  1.64733  1.86677
-0.50  3350  2.50 -2.61996  1.11457  1.19444  1.61763  1.80468
-0.50  3350  3.00 -2.47486  1.06490  1.10345  1.58794  1.74259
-0.50  3350  3.50 -2.32975  1.01523  1.01247  1.55824  1.68049
-0.50  3350  4.00 -2.07416  0.99846  1.25525  1.53782  1.53650
-0.50  3350  4.50 -1.88421  1.03459  1.39876  1.50676  1.43530
-0.50  3350  5.00 -1.80441  1.13343  1.50725  1.49931  1.38897
-0.50  3350  5.50 -1.78413  1.29917  1.60987  1.51129  1.37300
-0.50  3500 -1.02 -2.42775  1.88359  1.77156  1.56979  1.77381
-0.50  3500 -0.70 -2.42775  1.88359  1.77156  1.56979  1.77381
-0.50  3500 -0.51 -2.42775  1.88359  1.77156  1.56979  1.77381
-0.50  3500 -0.29 -2.42775  1.88359  1.77156  1.56979  1.77381
-0.50  3500  0.00 -2.42775  1.88359  1.77156  1.56979  1.77381
-0.50  3500  0.28 -2.43414  1.76440  1.70145  1.54976  1.78896
-0.50  3500  0.50 -2.43916  1.67076  1.64637  1.53403  1.80086
-0.50  3500  0.60 -2.44477  1.62737  1.62194  1.52936  1.80697
-0.50  3500  0.87 -2.45990  1.51022  1.55601  1.51675  1.82346
-0.50  3500  1.00 -2.46718  1.45381  1.52426  1.51068  1.83140
-0.50  3500  1.50 -2.50553  1.24714  1.40690  1.49750  1.86228
-0.50  3500  2.00 -2.54910  1.06512  1.29820  1.49249  1.89107
-0.50  3500  2.50 -2.59060  0.92040  1.20629  1.49472  1.91447
-0.50  3500  3.00 -1.81318  1.18365  1.34959  1.26151  1.45418
-0.50  3500  3.50 -1.65651  1.12467  1.38840  1.26877  1.38174
-0.50  3500  4.00 -1.48929  1.08922  1.43312  1.28129  1.29883
-0.50  3500  4.50 -1.41800  1.08681  1.46480  1.30089  1.25386
-0.50  3500  5.00 -1.39016  1.12450  1.50328  1.32803  1.22318
-0.50  3500  5.50 -1.39016  1.12450  1.50328  1.32803  1.22318
-0.50  3750 -1.02 -1.36398  2.30326  1.76094  1.31285  1.08092
-0.50  3750 -0.70 -1.36398  2.30326  1.76094  1.31285  1.08092
-0.50  3750 -0.51 -1.36398  2.30326  1.76094  1.31285  1.08092
-0.50  3750 -0.29 -1.36398  2.30326  1.76094  1.31285  1.08092
-0.50  3750  0.00 -1.36398  2.30326  1.76094  1.31285  1.08092
-0.50  3750  0.28 -1.35620  2.18002  1.71310  1.29809  1.08262
-0.50  3750  0.50 -1.35009  2.08318  1.67552  1.28649  1.08395
-0.50  3750  0.60 -1.34973  2.04079  1.65988  1.28227  1.08609
-0.50  3750  0.87 -1.34878  1.92633  1.61765  1.27087  1.09184
-0.50  3750  1.00 -1.34832  1.87122  1.59732  1.26538  1.09461
-0.50  3750  1.50 -1.35377  1.67709  1.52372  1.24673  1.11042
-0.50  3750  2.00 -1.36668  1.50651  1.45590  1.23111  1.13033
-0.50  3750  2.50 -1.38435  1.36371  1.39491  1.21930  1.15145
-0.50  3750  3.00 -1.20872  1.49917  1.43666  1.10665  1.08403
-0.50  3750  3.50 -1.21269  1.40565  1.41288  1.11225  1.09133
-0.50  3750  4.00 -1.15455  1.30916  1.42289  1.13532  1.06323
-0.50  3750  4.50 -1.08230  1.23087  1.43023  1.15603  1.02768
-0.50  3750  5.00 -1.04779  1.19235  1.43193  1.17145  1.00799
-0.50  3750  5.50 -1.04779  1.19235  1.43193  1.17145  1.00799
-0.50  4000 -1.02 -0.81711  2.24883  1.62711  1.05713  0.80762
-0.50  4000 -0.70 -0.81711  2.24883  1.62711  1.05713  0.80762
-0.50  4000 -0.51 -0.81711  2.24883  1.62711  1.05713  0.80762
-0.50  4000 -0.29 -0.81711  2.24883  1.62711  1.05713  0.80762
-0.50  4000  0.00 -0.81711  2.24883  1.62711  1.05713  0.80762
-0.50  4000  0.28 -0.80948  2.13357  1.58615  1.04999  0.80214
-0.50  4000  0.50 -0.80348  2.04300  1.55397  1.04439  0.79784
-0.50  4000  0.60 -0.80254  2.00487  1.54119  1.04320  0.79685
-0.50  4000  0.87 -0.79998  1.90191  1.50670  1.04001  0.79416
-0.50  4000  1.00 -0.79875  1.85234  1.49009  1.03847  0.79287
-0.50  4000  1.50 -0.79964  1.68481  1.43465  1.03559  0.79193
-0.50  4000  2.00 -0.80641  1.54390  1.38810  1.03535  0.79514
-0.50  4000  2.50 -0.81585  1.43067  1.34834  1.03518  0.80165
-0.50  4000  3.00 -0.74611  1.51124  1.38344  1.02624  0.82268
-0.50  4000  3.50 -0.76105  1.44359  1.36199  1.02891  0.83365
-0.50  4000  4.00 -0.76942  1.37927  1.35399  1.03899  0.84140
-0.50  4000  4.50 -0.75066  1.28899  1.35842  1.05812  0.83639
-0.50  4000  5.00 -0.71849  1.20205  1.35438  1.07079  0.82647
-0.50  4000  5.50 -0.71849  1.20205  1.35438  1.07079  0.82647
-0.50  4250 -1.02 -0.50486  1.97392  1.50808  0.92569  0.66519
-0.50  4250 -0.70 -0.50486  1.97392  1.50808  0.92569  0.66519
-0.50  4250 -0.51 -0.50486  1.97392  1.50808  0.92569  0.66519
-0.50  4250 -0.29 -0.50486  1.97392  1.50808  0.92569  0.66519
-0.50  4250  0.00 -0.50486  1.97392  1.50808  0.92569  0.66519
-0.50  4250  0.28 -0.49749  1.88045  1.47092  0.91762  0.65995
-0.50  4250  0.50 -0.49170  1.80702  1.44172  0.91129  0.65583
-0.50  4250  0.60 -0.49077  1.77542  1.43009  0.91002  0.65453
-0.50  4250  0.87 -0.48826  1.69012  1.39871  0.90660  0.65102
-0.50  4250  1.00 -0.48706  1.64905  1.38359  0.90495  0.64933
-0.50  4250  1.50 -0.48764  1.50816  1.33228  0.90467  0.64477
-0.50  4250  2.00 -0.49300  1.39226  1.28864  0.90887  0.64211
-0.50  4250  2.50 -0.50182  1.30467  1.25352  0.91626  0.64122
-0.50  4250  3.00 -0.48045  1.30400  1.27958  0.86540  0.68609
-0.50  4250  3.50 -0.49404  1.26690  1.26321  0.87686  0.68796
-0.50  4250  4.00 -0.50959  1.24689  1.25639  0.89032  0.69136
-0.50  4250  4.50 -0.52164  1.22208  1.25910  0.90622  0.69656
-0.50  4250  5.00 -0.51770  1.15826  1.26315  0.92147  0.70095
-0.50  4250  5.50 -0.51770  1.15826  1.26315  0.92147  0.70095
-0.50  4500 -1.02 -0.31546  1.67857  1.43260  0.77413  0.61807
-0.50  4500 -0.70 -0.31546  1.67857  1.43260  0.77413  0.61807
-0.50  4500 -0.51 -0.31546  1.67857  1.43260  0.77413  0.61807
-0.50  4500 -0.29 -0.31546  1.67857  1.43260  0.77413  0.61807
-0.50  4500  0.00 -0.31546  1.67857  1.43260  0.77413  0.61807
-0.50  4500  0.28 -0.30914  1.61088  1.40232  0.76447  0.61412
-0.50  4500  0.50 -0.30418  1.55770  1.37853  0.75687  0.61102
-0.50  4500  0.60 -0.30317  1.53405  1.36887  0.75482  0.61006
-0.50  4500  0.87 -0.30044  1.47018  1.34279  0.74929  0.60746
-0.50  4500  1.00 -0.29913  1.43943  1.33024  0.74663  0.60621
-0.50  4500  1.50 -0.29755  1.33027  1.28676  0.74169  0.60249
-0.50  4500  2.00 -0.30069  1.23677  1.24817  0.74194  0.59996
-0.50  4500  2.50 -0.30678  1.16343  1.21420  0.74604  0.59806
-0.50  4500  3.00 -0.31611  1.11301  1.18680  0.75444  0.59689
-0.50  4500  3.50 -0.32752  1.08681  1.16767  0.76603  0.59615
-0.50  4500  4.00 -0.34123  1.08078  1.15756  0.78101  0.59591
-0.50  4500  4.50 -0.35643  1.09000  1.15606  0.79832  0.59629
-0.50  4500  5.00 -0.36921  1.09634  1.16328  0.81667  0.59878
-0.50  4500  5.50 -0.36921  1.09634  1.16328  0.81667  0.59878
-0.50  4750 -1.02 -0.14345  1.33561  1.26815  0.68048  0.52180
-0.50  4750 -0.70 -0.14345  1.33561  1.26815  0.68048  0.52180
-0.50  4750 -0.51 -0.14345  1.33561  1.26815  0.68048  0.52180
-0.50  4750 -0.29 -0.14345  1.33561  1.26815  0.68048  0.52180
-0.50  4750  0.00 -0.14345  1.33561  1.26815  0.68048  0.52180
-0.50  4750  0.28 -0.13935  1.28354  1.24279  0.67061  0.51846
-0.50  4750  0.50 -0.13613  1.24263  1.22286  0.66285  0.51583
-0.50  4750  0.60 -0.13549  1.22415  1.21508  0.66067  0.51514
-0.50  4750  0.87 -0.13377  1.17426  1.19406  0.65481  0.51328
-0.50  4750  1.00 -0.13294  1.15024  1.18395  0.65198  0.51238
-0.50  4750  1.50 -0.13244  1.06993  1.14969  0.64445  0.51005
-0.50  4750  2.00 -0.13397  1.00112  1.11907  0.64043  0.50842
-0.50  4750  2.50 -0.13781  0.94543  1.09133  0.63980  0.50730
-0.50  4750  3.00 -0.14400  0.90653  1.06701  0.64288  0.50652
-0.50  4750  3.50 -0.15229  0.88525  1.04681  0.64987  0.50562
-0.50  4750  4.00 -0.16262  0.88268  1.03246  0.66081  0.50499
-0.50  4750  4.50 -0.17474  0.89653  1.02533  0.67525  0.50431
-0.50  4750  5.00 -0.18784  0.92062  1.02578  0.69231  0.50396
-0.50  4750  5.50 -0.18784  0.92062  1.02578  0.69231  0.50396
-0.50  5000 -1.02 -0.02696  1.00241  1.09667  0.62653  0.46835
-0.50  5000 -0.70 -0.02696  1.00241  1.09667  0.62653  0.46835
-0.50  5000 -0.51 -0.02696  1.00241  1.09667  0.62653  0.46835
-0.50  5000 -0.29 -0.02696  1.00241  1.09667  0.62653  0.46835
-0.50  5000  0.00 -0.02696  1.00241  1.09667  0.62653  0.46835
-0.50  5000  0.28 -0.02570  0.95702  1.07660  0.61905  0.46535
-0.50  5000  0.50 -0.02471  0.92136  1.06084  0.61318  0.46300
-0.50  5000  0.60 -0.02474  0.90596  1.05446  0.61122  0.46235
-0.50  5000  0.87 -0.02484  0.86436  1.03721  0.60593  0.46059
-0.50  5000  1.00 -0.02489  0.84434  1.02891  0.60338  0.45974
-0.50  5000  1.50 -0.02601  0.78072  1.00252  0.59556  0.45749
-0.50  5000  2.00 -0.02881  0.72521  0.97961  0.59119  0.45661
-0.50  5000  2.50 -0.03195  0.68347  0.95989  0.58832  0.45580
-0.50  5000  3.00 -0.03629  0.65445  0.94206  0.58799  0.45529
-0.50  5000  3.50 -0.04144  0.63957  0.92633  0.59009  0.45467
-0.50  5000  4.00 -0.04793  0.64003  0.91447  0.59569  0.45408
-0.50  5000  4.50 -0.05583  0.65505  0.90637  0.60477  0.45331
-0.50  5000  5.00 -0.06531  0.68078  0.90268  0.61757  0.45267
-0.50  5000  5.50 -0.06531  0.68078  0.90268  0.61757  0.45267
-0.50  5250 -1.02  0.04084  0.74138  0.93305  0.58009  0.43356
-0.50  5250 -0.70  0.04084  0.74138  0.93305  0.58009  0.43356
-0.50  5250 -0.51  0.04084  0.74138  0.93305  0.58009  0.43356
-0.50  5250 -0.29  0.04084  0.74138  0.93305  0.58009  0.43356
-0.50  5250  0.00  0.04084  0.74138  0.93305  0.58009  0.43356
-0.50  5250  0.28  0.03905  0.70467  0.91877  0.57553  0.43113
-0.50  5250  0.50  0.03764  0.67583  0.90755  0.57194  0.42921
-0.50  5250  0.60  0.03684  0.66251  0.90274  0.57069  0.42868
-0.50  5250  0.87  0.03466  0.62652  0.88974  0.56732  0.42723
-0.50  5250  1.00  0.03361  0.60920  0.88349  0.56570  0.42653
-0.50  5250  1.50  0.02944  0.55304  0.86344  0.56050  0.42446
-0.50  5250  2.00  0.02458  0.50233  0.84617  0.55732  0.42362
-0.50  5250  2.50  0.02040  0.46491  0.83270  0.55469  0.42301
-0.50  5250  3.00  0.01629  0.43875  0.82168  0.55343  0.42278
-0.50  5250  3.50  0.01224  0.42599  0.81260  0.55355  0.42253
-0.50  5250  4.00  0.00820  0.42726  0.80580  0.55567  0.42193
-0.50  5250  4.50  0.00378  0.44141  0.80091  0.56037  0.42132
-0.50  5250  5.00 -0.00135  0.46719  0.79926  0.56812  0.42059
-0.50  5250  5.50 -0.00135  0.46719  0.79926  0.56812  0.42059
-0.50  5500 -1.02  0.11573  0.57799  0.76998  0.51277  0.39728
-0.50  5500 -0.70  0.11573  0.57799  0.76998  0.51277  0.39728
-0.50  5500 -0.51  0.11573  0.57799  0.76998  0.51277  0.39728
-0.50  5500 -0.29  0.11573  0.57799  0.76998  0.51277  0.39728
-0.50  5500  0.00  0.11573  0.57799  0.76998  0.51277  0.39728
-0.50  5500  0.28  0.10903  0.53782  0.77021  0.51665  0.39615
-0.50  5500  0.50  0.10376  0.50625  0.77039  0.51971  0.39525
-0.50  5500  0.60  0.10227  0.49494  0.76724  0.51925  0.39483
-0.50  5500  0.87  0.09827  0.46439  0.75875  0.51800  0.39367
-0.50  5500  1.00  0.09634  0.44969  0.75466  0.51740  0.39311
-0.50  5500  1.50  0.08934  0.39674  0.74237  0.51588  0.39179
-0.50  5500  2.00  0.08262  0.35115  0.73153  0.51418  0.39100
-0.50  5500  2.50  0.07633  0.31306  0.72356  0.51357  0.39080
-0.50  5500  3.00  0.07092  0.28436  0.71731  0.51304  0.39078
-0.50  5500  3.50  0.06645  0.26783  0.71319  0.51297  0.39046
-0.50  5500  4.00  0.06314  0.26474  0.71068  0.51371  0.39018
-0.50  5500  4.50  0.06058  0.27370  0.71048  0.51616  0.38978
-0.50  5500  5.00  0.05838  0.29569  0.71259  0.52068  0.38919
-0.50  5500  5.50  0.05838  0.29569  0.71259  0.52068  0.38919
-0.50  5750 -1.02  0.18889  0.47545  0.62420  0.42734  0.35408
-0.50  5750 -0.70  0.18889  0.47545  0.62420  0.42734  0.35408
-0.50  5750 -0.51  0.18889  0.47545  0.62420  0.42734  0.35408
-0.50  5750 -0.29  0.18889  0.47545  0.62420  0.42734  0.35408
-0.50  5750  0.00  0.18889  0.47545  0.62420  0.42734  0.35408
-0.50  5750  0.28  0.17945  0.43634  0.62475  0.43409  0.35374
-0.50  5750  0.50  0.17204  0.40562  0.62518  0.43939  0.35348
-0.50  5750  0.60  0.16895  0.39090  0.62801  0.44232  0.35346
-0.50  5750  0.87  0.16061  0.35116  0.63563  0.45024  0.35340
-0.50  5750  1.00  0.15660  0.33203  0.63930  0.45405  0.35338
-0.50  5750  1.50  0.14688  0.28309  0.63229  0.45512  0.35226
-0.50  5750  2.00  0.13759  0.23619  0.62807  0.45682  0.35212
-0.50  5750  2.50  0.12937  0.19646  0.62559  0.45797  0.35218
-0.50  5750  3.00  0.12226  0.16350  0.62431  0.45859  0.35250
-0.50  5750  3.50  0.11639  0.14146  0.62376  0.45934  0.35275
-0.50  5750  4.00  0.11210  0.12955  0.62452  0.46030  0.35290
-0.50  5750  4.50  0.10949  0.12984  0.62740  0.46200  0.35265
-0.50  5750  5.00  0.10830  0.14319  0.63238  0.46503  0.35247
-0.50  5750  5.50  0.10830  0.14319  0.63238  0.46503  0.35247
-0.50  6000 -1.02  0.23554  0.42583  0.50198  0.35269  0.31688
-0.50  6000 -0.70  0.23554  0.42583  0.50198  0.35269  0.31688
-0.50  6000 -0.51  0.23554  0.42583  0.50198  0.35269  0.31688
-0.50  6000 -0.29  0.23554  0.42583  0.50198  0.35269  0.31688
-0.50  6000  0.00  0.23554  0.42583  0.50198  0.35269  0.31688
-0.50  6000  0.28  0.22793  0.40247  0.50784  0.35926  0.31551
-0.50  6000  0.50  0.22195  0.38411  0.51245  0.36443  0.31443
-0.50  6000  0.60  0.21868  0.37216  0.51342  0.36700  0.31439
-0.50  6000  0.87  0.20984  0.33989  0.51602  0.37392  0.31427
-0.50  6000  1.00  0.20559  0.32435  0.51728  0.37726  0.31421
-0.50  6000  1.50  0.18644  0.24186  0.54378  0.39900  0.31634
-0.50  6000  2.00  0.17570  0.19738  0.54391  0.40273  0.31596
-0.50  6000  2.50  0.16553  0.15464  0.54552  0.40578  0.31637
-0.50  6000  3.00  0.15632  0.11696  0.54852  0.40858  0.31716
-0.50  6000  3.50  0.14874  0.08692  0.55224  0.41066  0.31795
-0.50  6000  4.00  0.14287  0.06596  0.55619  0.41224  0.31864
-0.50  6000  4.50  0.13893  0.05589  0.56122  0.41425  0.31914
-0.50  6000  5.00  0.13728  0.05774  0.56801  0.41669  0.31945
-0.50  6000  5.50  0.13728  0.05774  0.56801  0.41669  0.31945
-0.50  6250 -1.02  0.27256  0.37480  0.40930  0.29820  0.27400
-0.50  6250 -0.70  0.27256  0.37480  0.40930  0.29820  0.27400
-0.50  6250 -0.51  0.27256  0.37480  0.40930  0.29820  0.27400
-0.50  6250 -0.29  0.27256  0.37480  0.40930  0.29820  0.27400
-0.50  6250  0.00  0.27256  0.37480  0.40930  0.29820  0.27400
-0.50  6250  0.28  0.27256  0.37480  0.40930  0.29820  0.27400
-0.50  6250  0.50  0.27256  0.37480  0.40930  0.29820  0.27400
-0.50  6250  0.60  0.26995  0.36632  0.41264  0.30100  0.27363
-0.50  6250  0.87  0.26291  0.34341  0.42167  0.30856  0.27262
-0.50  6250  1.00  0.25952  0.33238  0.42601  0.31219  0.27214
-0.50  6250  1.50  0.24147  0.26742  0.44211  0.33013  0.27347
-0.50  6250  2.00  0.22247  0.19122  0.47062  0.35160  0.27610
-0.50  6250  2.50  0.21064  0.14777  0.47608  0.35738  0.27634
-0.50  6250  3.00  0.19963  0.10596  0.48257  0.36199  0.27741
-0.50  6250  3.50  0.18995  0.06865  0.48967  0.36585  0.27865
-0.50  6250  4.00  0.18191  0.03850  0.49655  0.36889  0.28004
-0.50  6250  4.50  0.17628  0.01806  0.50371  0.37147  0.28118
-0.50  6250  5.00  0.17289  0.00720  0.51168  0.37426  0.28221
-0.50  6250  5.50  0.17289  0.00720  0.51168  0.37426  0.28221
-0.50  6500 -1.02  0.30928  0.34905  0.31063  0.23727  0.23850
-0.50  6500 -0.70  0.30928  0.34905  0.31063  0.23727  0.23850
-0.50  6500 -0.51  0.30928  0.34905  0.31063  0.23727  0.23850
-0.50  6500 -0.29  0.30928  0.34905  0.31063  0.23727  0.23850
-0.50  6500  0.00  0.30928  0.34905  0.31063  0.23727  0.23850
-0.50  6500  0.28  0.30928  0.34905  0.31063  0.23727  0.23850
-0.50  6500  0.50  0.30928  0.34905  0.31063  0.23727  0.23850
-0.50  6500  0.60  0.30745  0.34576  0.31529  0.24019  0.23771
-0.50  6500  0.87  0.30250  0.33686  0.32788  0.24807  0.23558
-0.50  6500  1.00  0.30012  0.33257  0.33394  0.25186  0.23456
-0.50  6500  1.50  0.28663  0.28522  0.35822  0.26847  0.23357
-0.50  6500  2.00  0.26699  0.21712  0.38089  0.28888  0.23648
-0.50  6500  2.50  0.24845  0.14567  0.41114  0.30950  0.23926
-0.50  6500  3.00  0.23540  0.10081  0.42110  0.31688  0.24037
-0.50  6500  3.50  0.22316  0.05783  0.43050  0.32264  0.24211
-0.50  6500  4.00  0.21255  0.01954  0.43966  0.32738  0.24419
-0.50  6500  4.50  0.20422 -0.01136  0.44847  0.33121  0.24623
-0.50  6500  5.00  0.19838 -0.03453  0.45767  0.33475  0.24814
-0.50  6500  5.50  0.19838 -0.03453  0.45767  0.33475  0.24814
-0.50  6750 -1.02  0.33384  0.32293  0.23311  0.18676  0.21207
-0.50  6750 -0.70  0.33384  0.32293  0.23311  0.18676  0.21207
-0.50  6750 -0.51  0.33384  0.32293  0.23311  0.18676  0.21207
-0.50  6750 -0.29  0.33384  0.32293  0.23311  0.18676  0.21207
-0.50  6750  0.00  0.33384  0.32293  0.23311  0.18676  0.21207
-0.50  6750  0.28  0.33384  0.32293  0.23311  0.18676  0.21207
-0.50  6750  0.50  0.33384  0.32293  0.23311  0.18676  0.21207
-0.50  6750  0.60  0.33230  0.32472  0.23608  0.18934  0.21118
-0.50  6750  0.87  0.32814  0.32954  0.24410  0.19630  0.20877
-0.50  6750  1.00  0.32614  0.33187  0.24797  0.19965  0.20762
-0.50  6750  1.50  0.31728  0.30981  0.27753  0.21585  0.20399
-0.50  6750  2.00  0.30321  0.25904  0.30720  0.23374  0.20399
-0.50  6750  2.50  0.28223  0.18752  0.33541  0.25603  0.20830
-0.50  6750  3.00  0.26363  0.11906  0.36540  0.27584  0.21145
-0.50  6750  3.50  0.24947  0.07178  0.37830  0.28401  0.21338
-0.50  6750  4.00  0.23617  0.02699  0.38936  0.29059  0.21601
-0.50  6750  4.50  0.22436 -0.01415  0.39937  0.29623  0.21904
-0.50  6750  5.00  0.21518 -0.04903  0.40939  0.30116  0.22189
-0.50  6750  5.50  0.21518 -0.04903  0.40939  0.30116  0.22189
-0.50  7000 -1.02  0.34307  0.27019  0.16408  0.13686  0.18671
-0.50  7000 -0.70  0.34307  0.27019  0.16408  0.13686  0.18671
-0.50  7000 -0.51  0.34307  0.27019  0.16408  0.13686  0.18671
-0.50  7000 -0.29  0.34307  0.27019  0.16408  0.13686  0.18671
-0.50  7000  0.00  0.34307  0.27019  0.16408  0.13686  0.18671
-0.50  7000  0.28  0.34307  0.27019  0.16408  0.13686  0.18671
-0.50  7000  0.50  0.34307  0.27019  0.16408  0.13686  0.18671
-0.50  7000  0.60  0.34195  0.27575  0.16613  0.13904  0.18585
-0.50  7000  0.87  0.33893  0.29075  0.17166  0.14492  0.18353
-0.50  7000  1.00  0.33748  0.29797  0.17433  0.14775  0.18241
-0.50  7000  1.50  0.33082  0.30368  0.19472  0.16057  0.17717
-0.50  7000  2.00  0.32104  0.27627  0.22961  0.17812  0.17427
-0.50  7000  2.50  0.30486  0.21817  0.26369  0.19763  0.17594
-0.50  7000  3.00  0.28166  0.13919  0.29788  0.22266  0.18167
-0.50  7000  3.50  0.26389  0.07767  0.32408  0.23878  0.18470
-0.50  7000  4.00  0.24790  0.02760  0.33838  0.24780  0.18787
-0.50  7000  4.50  0.23304 -0.02067  0.35074  0.25535  0.19149
-0.50  7000  5.00  0.22046 -0.06441  0.36186  0.26177  0.19522
-0.50  7000  5.50  0.22046 -0.06441  0.36186  0.26177  0.19522
-0.50  7250 -1.02  0.35106  0.24981  0.12473  0.09441  0.16200
-0.50  7250 -0.70  0.35106  0.24981  0.12473  0.09441  0.16200
-0.50  7250 -0.51  0.35106  0.24981  0.12473  0.09441  0.16200
-0.50  7250 -0.29  0.35106  0.24981  0.12473  0.09441  0.16200
-0.50  7250  0.00  0.35106  0.24981  0.12473  0.09441  0.16200
-0.50  7250  0.28  0.35106  0.24981  0.12473  0.09441  0.16200
-0.50  7250  0.50  0.35106  0.24981  0.12473  0.09441  0.16200
-0.50  7250  0.60  0.35013  0.25901  0.12388  0.09567  0.16138
-0.50  7250  0.87  0.34764  0.28384  0.12158  0.09907  0.15972
-0.50  7250  1.00  0.34643  0.29579  0.12047  0.10070  0.15892
-0.50  7250  1.50  0.34256  0.31945  0.13774  0.11132  0.15320
-0.50  7250  2.00  0.33586  0.32005  0.16269  0.12449  0.14839
-0.50  7250  2.50  0.32487  0.28525  0.20156  0.14340  0.14677
-0.50  7250  3.00  0.30705  0.21977  0.23895  0.16457  0.15012
-0.50  7250  3.50  0.28396  0.13884  0.27491  0.18992  0.15688
-0.50  7250  4.00  0.26624  0.07943  0.29785  0.20420  0.16059
-0.50  7250  4.50  0.24965  0.02711  0.31310  0.21365  0.16467
-0.50  7250  5.00  0.23441 -0.02346  0.32603  0.22185  0.16924
-0.50  7250  5.50  0.23441 -0.02346  0.32603  0.22185  0.16924
-0.50  7500 -1.02  0.34353  0.19535  0.12433  0.08855  0.15021
-0.50  7500 -0.70  0.34353  0.19535  0.12433  0.08855  0.15021
-0.50  7500 -0.51  0.34353  0.19535  0.12433  0.08855  0.15021
-0.50  7500 -0.29  0.34353  0.19535  0.12433  0.08855  0.15021
-0.50  7500  0.00  0.34353  0.19535  0.12433  0.08855  0.15021
-0.50  7500  0.28  0.34353  0.19535  0.12433  0.08855  0.15021
-0.50  7500  0.50  0.34353  0.19535  0.12433  0.08855  0.15021
-0.50  7500  0.60  0.34264  0.20947  0.11519  0.08680  0.15029
-0.50  7500  0.87  0.34024  0.24760  0.09049  0.08208  0.15049
-0.50  7500  1.00  0.33908  0.26596  0.07860  0.07981  0.15059
-0.50  7500  1.50  0.33764  0.30622  0.08929  0.08738  0.14492
-0.50  7500  2.00  0.33428  0.32560  0.11174  0.09791  0.13880
-0.50  7500  2.50  0.32767  0.32283  0.14130  0.11108  0.13436
-0.50  7500  3.00  0.31538  0.27930  0.18375  0.13130  0.13426
-0.50  7500  3.50  0.29519  0.20399  0.22269  0.15471  0.13994
-0.50  7500  4.00  0.27260  0.12259  0.25723  0.17858  0.14699
-0.50  7500  4.50  0.25540  0.06680  0.27750  0.19061  0.15099
-0.50  7500  5.00  0.23803  0.01065  0.29301  0.20069  0.15636
-0.50  7500  5.50  0.23803  0.01065  0.29301  0.20069  0.15636
-0.50  7750 -1.02  0.31024  0.18386  0.04459  0.07582  0.14856
-0.50  7750 -0.70  0.31024  0.18386  0.04459  0.07582  0.14856
-0.50  7750 -0.51  0.31024  0.18386  0.04459  0.07582  0.14856
-0.50  7750 -0.29  0.31024  0.18386  0.04459  0.07582  0.14856
-0.50  7750  0.00  0.31024  0.18386  0.04459  0.07582  0.14856
-0.50  7750  0.28  0.31024  0.18386  0.04459  0.07582  0.14856
-0.50  7750  0.50  0.31024  0.18386  0.04459  0.07582  0.14856
-0.50  7750  0.60  0.31024  0.18386  0.04459  0.07582  0.14856
-0.50  7750  0.87  0.31024  0.18386  0.04459  0.07582  0.14856
-0.50  7750  1.00  0.31024  0.18386  0.04459  0.07582  0.14856
-0.50  7750  1.50  0.31119  0.24033  0.04071  0.07737  0.14332
-0.50  7750  2.00  0.31060  0.27477  0.06019  0.08555  0.13652
-0.50  7750  2.50  0.30631  0.28723  0.08741  0.09654  0.13108
-0.50  7750  3.00  0.29798  0.27581  0.12066  0.11044  0.12779
-0.50  7750  3.50  0.28322  0.22065  0.16530  0.13174  0.12936
-0.50  7750  4.00  0.26032  0.13556  0.20474  0.15640  0.13683
-0.50  7750  4.50  0.23922  0.06195  0.23492  0.17585  0.14302
-0.50  7750  5.00  0.21974  0.00049  0.25438  0.18833  0.14884
-0.50  7750  5.50  0.21974  0.00049  0.25438  0.18833  0.14884
-0.50  8000 -1.02  0.27535  0.08767  0.03232  0.06843  0.13416
-0.50  8000 -0.70  0.27535  0.08767  0.03232  0.06843  0.13416
-0.50  8000 -0.51  0.27535  0.08767  0.03232  0.06843  0.13416
-0.50  8000 -0.29  0.27535  0.08767  0.03232  0.06843  0.13416
-0.50  8000  0.00  0.27535  0.08767  0.03232  0.06843  0.13416
-0.50  8000  0.28  0.27535  0.08767  0.03232  0.06843  0.13416
-0.50  8000  0.50  0.27535  0.08767  0.03232  0.06843  0.13416
-0.50  8000  0.60  0.27535  0.08767  0.03232  0.06843  0.13416
-0.50  8000  0.87  0.27535  0.08767  0.03232  0.06843  0.13416
-0.50  8000  1.00  0.27535  0.08767  0.03232  0.06843  0.13416
-0.50  8000  1.50  0.28249  0.16526  0.00333  0.05701  0.13000
-0.50  8000  2.00  0.28481  0.21275  0.01513  0.06115  0.12292
-0.50  8000  2.50  0.28365  0.23956  0.04025  0.06969  0.11638
-0.50  8000  3.00  0.27822  0.24564  0.07123  0.08071  0.11140
-0.50  8000  3.50  0.26854  0.22503  0.10905  0.09523  0.10899
-0.50  8000  4.00  0.25082  0.15544  0.15461  0.11791  0.11307
-0.50  8000  4.50  0.22642  0.06416  0.19272  0.14305  0.12177
-0.50  8000  5.00  0.20520 -0.00580  0.21829  0.15882  0.12835
-0.50  8000  5.50  0.20520 -0.00580  0.21829  0.15882  0.12835
-0.50  8250 -1.02  0.21756 -0.02959  0.03372  0.07303  0.11424
-0.50  8250 -0.70  0.21756 -0.02959  0.03372  0.07303  0.11424
-0.50  8250 -0.51  0.21756 -0.02959  0.03372  0.07303  0.11424
-0.50  8250 -0.29  0.21756 -0.02959  0.03372  0.07303  0.11424
-0.50  8250  0.00  0.21756 -0.02959  0.03372  0.07303  0.11424
-0.50  8250  0.28  0.21756 -0.02959  0.03372  0.07303  0.11424
-0.50  8250  0.50  0.21756 -0.02959  0.03372  0.07303  0.11424
-0.50  8250  0.60  0.21756 -0.02959  0.03372  0.07303  0.11424
-0.50  8250  0.87  0.21756 -0.02959  0.03372  0.07303  0.11424
-0.50  8250  1.00  0.21756 -0.02959  0.03372  0.07303  0.11424
-0.50  8250  1.50  0.24162  0.07590 -0.02014  0.04165  0.11205
-0.50  8250  2.00  0.24756  0.13512 -0.01998  0.03989  0.10503
-0.50  8250  2.50  0.24956  0.17419  0.00068  0.04520  0.09748
-0.50  8250  3.00  0.24736  0.19329  0.03043  0.05386  0.09126
-0.50  8250  3.50  0.24022  0.18994  0.06439  0.06489  0.08760
-0.50  8250  4.00  0.22828  0.15465  0.10582  0.08087  0.08702
-0.50  8250  4.50  0.20814  0.07147  0.15106  0.10481  0.09317
-0.50  8250  5.00  0.18322 -0.02147  0.18505  0.12771  0.10290
-0.50  8250  5.50  0.18322 -0.02147  0.18505  0.12771  0.10290
-0.50  8500 -1.02  0.19519 -0.00050 -0.02791  0.04743  0.10486
-0.50  8500 -0.70  0.19519 -0.00050 -0.02791  0.04743  0.10486
-0.50  8500 -0.51  0.19519 -0.00050 -0.02791  0.04743  0.10486
-0.50  8500 -0.29  0.19519 -0.00050 -0.02791  0.04743  0.10486
-0.50  8500  0.00  0.19519 -0.00050 -0.02791  0.04743  0.10486
-0.50  8500  0.28  0.19519 -0.00050 -0.02791  0.04743  0.10486
-0.50  8500  0.50  0.19519 -0.00050 -0.02791  0.04743  0.10486
-0.50  8500  0.60  0.19519 -0.00050 -0.02791  0.04743  0.10486
-0.50  8500  0.87  0.19519 -0.00050 -0.02791  0.04743  0.10486
-0.50  8500  1.00  0.19519 -0.00050 -0.02791  0.04743  0.10486
-0.50  8500  1.50  0.19519 -0.00050 -0.02791  0.04743  0.10486
-0.50  8500  2.00  0.20586  0.06956 -0.04049  0.03797  0.09821
-0.50  8500  2.50  0.21043  0.11711 -0.02667  0.03959  0.09057
-0.50  8500  3.00  0.21139  0.14586 -0.00028  0.04569  0.08358
-0.50  8500  3.50  0.20817  0.15515  0.03279  0.05445  0.07837
-0.50  8500  4.00  0.19978  0.14126  0.06970  0.06582  0.07577
-0.50  8500  4.50  0.18687  0.08939  0.11487  0.08348  0.07685
-0.50  8500  5.00  0.16572 -0.00389  0.15680  0.10754  0.08520
-0.50  8500  5.50  0.16572 -0.00389  0.15680  0.10754  0.08520
-0.50  8750 -1.02  0.16042 -0.00441 -0.05086  0.04387  0.09625
-0.50  8750 -0.70  0.16042 -0.00441 -0.05086  0.04387  0.09625
-0.50  8750 -0.51  0.16042 -0.00441 -0.05086  0.04387  0.09625
-0.50  8750 -0.29  0.16042 -0.00441 -0.05086  0.04387  0.09625
-0.50  8750  0.00  0.16042 -0.00441 -0.05086  0.04387  0.09625
-0.50  8750  0.28  0.16042 -0.00441 -0.05086  0.04387  0.09625
-0.50  8750  0.50  0.16042 -0.00441 -0.05086  0.04387  0.09625
-0.50  8750  0.60  0.16042 -0.00441 -0.05086  0.04387  0.09625
-0.50  8750  0.87  0.16042 -0.00441 -0.05086  0.04387  0.09625
-0.50  8750  1.00  0.16042 -0.00441 -0.05086  0.04387  0.09625
-0.50  8750  1.50  0.16042 -0.00441 -0.05086  0.04387  0.09625
-0.50  8750  2.00  0.16239  0.00878 -0.05254  0.04216  0.09498
-0.50  8750  2.50  0.16894  0.06262 -0.04599  0.03984  0.08739
-0.50  8750  3.00  0.17227  0.09995 -0.02390  0.04308  0.07969
-0.50  8750  3.50  0.17245  0.12000  0.00672  0.04926  0.07308
-0.50  8750  4.00  0.16845  0.11874  0.04294  0.05814  0.06884
-0.50  8750  4.50  0.15932  0.09119  0.08282  0.06998  0.06782
-0.50  8750  5.00  0.14600  0.02026  0.12826  0.08975  0.07144
-0.50  8750  5.50  0.14600  0.02026  0.12826  0.08975  0.07144
-0.50  9000 -1.02  0.09311 -0.13845 -0.03458  0.05938  0.08590
-0.50  9000 -0.70  0.09311 -0.13845 -0.03458  0.05938  0.08590
-0.50  9000 -0.51  0.09311 -0.13845 -0.03458  0.05938  0.08590
-0.50  9000 -0.29  0.09311 -0.13845 -0.03458  0.05938  0.08590
-0.50  9000  0.00  0.09311 -0.13845 -0.03458  0.05938  0.08590
-0.50  9000  0.28  0.09311 -0.13845 -0.03458  0.05938  0.08590
-0.50  9000  0.50  0.09311 -0.13845 -0.03458  0.05938  0.08590
-0.50  9000  0.60  0.09311 -0.13845 -0.03458  0.05938  0.08590
-0.50  9000  0.87  0.09311 -0.13845 -0.03458  0.05938  0.08590
-0.50  9000  1.00  0.09311 -0.13845 -0.03458  0.05938  0.08590
-0.50  9000  1.50  0.09311 -0.13845 -0.03458  0.05938  0.08590
-0.50  9000  2.00  0.11502 -0.04951 -0.06357  0.03683  0.08105
-0.50  9000  2.50  0.12309  0.00972 -0.06371  0.03054  0.07380
-0.50  9000  3.00  0.12747  0.05292 -0.04631  0.03119  0.06596
-0.50  9000  3.50  0.12958  0.08001 -0.01829  0.03538  0.05892
-0.50  9000  4.00  0.12858  0.08742  0.01602  0.04218  0.05376
-0.50  9000  4.50  0.12333  0.07264  0.05394  0.05118  0.05115
-0.50  9000  5.00  0.11244  0.03504  0.09290  0.06214  0.05186
-0.50  9000  5.50  0.11244  0.03504  0.09290  0.06214  0.05186
-0.50  9250 -1.02  0.06799 -0.10334 -0.07097  0.01949  0.05316
-0.50  9250 -0.70  0.06799 -0.10334 -0.07097  0.01949  0.05316
-0.50  9250 -0.51  0.06799 -0.10334 -0.07097  0.01949  0.05316
-0.50  9250 -0.29  0.06799 -0.10334 -0.07097  0.01949  0.05316
-0.50  9250  0.00  0.06799 -0.10334 -0.07097  0.01949  0.05316
-0.50  9250  0.28  0.06799 -0.10334 -0.07097  0.01949  0.05316
-0.50  9250  0.50  0.06799 -0.10334 -0.07097  0.01949  0.05316
-0.50  9250  0.60  0.06799 -0.10334 -0.07097  0.01949  0.05316
-0.50  9250  0.87  0.06799 -0.10334 -0.07097  0.01949  0.05316
-0.50  9250  1.00  0.06799 -0.10334 -0.07097  0.01949  0.05316
-0.50  9250  1.50  0.06799 -0.10334 -0.07097  0.01949  0.05316
-0.50  9250  2.00  0.06799 -0.10334 -0.07097  0.01949  0.05316
-0.50  9250  2.50  0.07739 -0.03948 -0.07619  0.00983  0.04648
-0.50  9250  3.00  0.08191  0.00788 -0.06366  0.00800  0.03895
-0.50  9250  3.50  0.08461  0.04031 -0.03932  0.01032  0.03172
-0.50  9250  4.00  0.08554  0.05562 -0.00713  0.01518  0.02567
-0.50  9250  4.50  0.08367  0.05007  0.03040  0.02216  0.02169
-0.50  9250  5.00  0.07726  0.02136  0.07025  0.03120  0.02093
-0.50  9250  5.50  0.07726  0.02136  0.07025  0.03120  0.02093
-0.50  9500 -1.02  0.02888 -0.14513 -0.07429  0.01380  0.03447
-0.50  9500 -0.70  0.02888 -0.14513 -0.07429  0.01380  0.03447
-0.50  9500 -0.51  0.02888 -0.14513 -0.07429  0.01380  0.03447
-0.50  9500 -0.29  0.02888 -0.14513 -0.07429  0.01380  0.03447
-0.50  9500  0.00  0.02888 -0.14513 -0.07429  0.01380  0.03447
-0.50  9500  0.28  0.02888 -0.14513 -0.07429  0.01380  0.03447
-0.50  9500  0.50  0.02888 -0.14513 -0.07429  0.01380  0.03447
-0.50  9500  0.60  0.02888 -0.14513 -0.07429  0.01380  0.03447
-0.50  9500  0.87  0.02888 -0.14513 -0.07429  0.01380  0.03447
-0.50  9500  1.00  0.02888 -0.14513 -0.07429  0.01380  0.03447
-0.50  9500  1.50  0.02888 -0.14513 -0.07429  0.01380  0.03447
-0.50  9500  2.00  0.02888 -0.14513 -0.07429  0.01380  0.03447
-0.50  9500  2.50  0.03925 -0.07771 -0.08365  0.00155  0.02841
-0.50  9500  3.00  0.04386 -0.02735 -0.07533 -0.00249  0.02122
-0.50  9500  3.50  0.04661  0.01005 -0.05477 -0.00199  0.01396
-0.50  9500  4.00  0.04853  0.03227 -0.02493  0.00115  0.00730
-0.50  9500  4.50  0.04872  0.03486  0.01136  0.00637  0.00243
-0.50  9500  5.00  0.04575  0.01431  0.05157  0.01365  0.00049
-0.50  9500  5.50  0.04575  0.01431  0.05157  0.01365  0.00049
-0.50  9750 -1.02 -0.00947 -0.18433 -0.07567  0.01239  0.02273
-0.50  9750 -0.70 -0.00947 -0.18433 -0.07567  0.01239  0.02273
-0.50  9750 -0.51 -0.00947 -0.18433 -0.07567  0.01239  0.02273
-0.50  9750 -0.29 -0.00947 -0.18433 -0.07567  0.01239  0.02273
-0.50  9750  0.00 -0.00947 -0.18433 -0.07567  0.01239  0.02273
-0.50  9750  0.28 -0.00947 -0.18433 -0.07567  0.01239  0.02273
-0.50  9750  0.50 -0.00947 -0.18433 -0.07567  0.01239  0.02273
-0.50  9750  0.60 -0.00947 -0.18433 -0.07567  0.01239  0.02273
-0.50  9750  0.87 -0.00947 -0.18433 -0.07567  0.01239  0.02273
-0.50  9750  1.00 -0.00947 -0.18433 -0.07567  0.01239  0.02273
-0.50  9750  1.50 -0.00947 -0.18433 -0.07567  0.01239  0.02273
-0.50  9750  2.00 -0.00947 -0.18433 -0.07567  0.01239  0.02273
-0.50  9750  2.50  0.00165 -0.11456 -0.08751 -0.00138  0.01729
-0.50  9750  3.00  0.00610 -0.06200 -0.08274 -0.00717  0.01064
-0.50  9750  3.50  0.00875 -0.02106 -0.06576 -0.00847  0.00337
-0.50  9750  4.00  0.01088  0.00648 -0.03881 -0.00682 -0.00349
-0.50  9750  4.50  0.01222  0.01648 -0.00422 -0.00315 -0.00893
-0.50  9750  5.00  0.01148  0.00352  0.03513  0.00254 -0.01185
-0.50  9750  5.50  0.01148  0.00352  0.03513  0.00254 -0.01185
-0.50  10000 -1.02 -0.06653 -0.23531 -0.08013  0.01421  0.01595
-0.50  10000 -0.70 -0.06653 -0.23531 -0.08013  0.01421  0.01595
-0.50  10000 -0.51 -0.06653 -0.23531 -0.08013  0.01421  0.01595
-0.50  10000 -0.29 -0.06653 -0.23531 -0.08013  0.01421  0.01595
-0.50  10000  0.00 -0.06653 -0.23531 -0.08013  0.01421  0.01595
-0.50  10000  0.28 -0.06653 -0.23531 -0.08013  0.01421  0.01595
-0.50  10000  0.50 -0.06653 -0.23531 -0.08013  0.01421  0.01595
-0.50  10000  0.60 -0.06653 -0.23531 -0.08013  0.01421  0.01595
-0.50  10000  0.87 -0.06653 -0.23531 -0.08013  0.01421  0.01595
-0.50  10000  1.00 -0.06653 -0.23531 -0.08013  0.01421  0.01595
-0.50  10000  1.50 -0.06653 -0.23531 -0.08013  0.01421  0.01595
-0.50  10000  2.00 -0.06653 -0.23531 -0.08013  0.01421  0.01595
-0.50  10000  2.50 -0.05484 -0.16329 -0.09368 -0.00064  0.01084
-0.50  10000  3.00 -0.05063 -0.10912 -0.09150 -0.00778  0.00462
-0.50  10000  3.50 -0.04863 -0.06638 -0.07735 -0.01033 -0.00234
-0.50  10000  4.00 -0.04702 -0.03496 -0.05353 -0.01019 -0.00912
-0.50  10000  4.50 -0.04536 -0.01949 -0.02116 -0.00786 -0.01493
-0.50  10000  5.00 -0.04486 -0.02530  0.01717 -0.00351 -0.01878
-0.50  10000  5.50 -0.04486 -0.02530  0.01717 -0.00351 -0.01878
-0.50  10500 -1.02 -0.18952 -0.33012 -0.09082  0.01605  0.00649
-0.50  10500 -0.70 -0.18952 -0.33012 -0.09082  0.01605  0.00649
-0.50  10500 -0.51 -0.18952 -0.33012 -0.09082  0.01605  0.00649
-0.50  10500 -0.29 -0.18952 -0.33012 -0.09082  0.01605  0.00649
-0.50  10500  0.00 -0.18952 -0.33012 -0.09082  0.01605  0.00649
-0.50  10500  0.28 -0.18952 -0.33012 -0.09082  0.01605  0.00649
-0.50  10500  0.50 -0.18952 -0.33012 -0.09082  0.01605  0.00649
-0.50  10500  0.60 -0.18952 -0.33012 -0.09082  0.01605  0.00649
-0.50  10500  0.87 -0.18952 -0.33012 -0.09082  0.01605  0.00649
-0.50  10500  1.00 -0.18952 -0.33012 -0.09082  0.01605  0.00649
-0.50  10500  1.50 -0.18952 -0.33012 -0.09082  0.01605  0.00649
-0.50  10500  2.00 -0.18952 -0.33012 -0.09082  0.01605  0.00649
-0.50  10500  2.50 -0.17681 -0.25521 -0.10642  0.00042  0.00148
-0.50  10500  3.00 -0.17349 -0.19940 -0.10735 -0.00829 -0.00393
-0.50  10500  3.50 -0.17315 -0.15498 -0.09789 -0.01274 -0.00996
-0.50  10500  4.00 -0.17302 -0.12025 -0.07934 -0.01459 -0.01619
-0.50  10500  4.50 -0.17209 -0.09724 -0.05219 -0.01436 -0.02236
-0.50  10500  5.00 -0.17069 -0.09067 -0.01690 -0.01256 -0.02745
-0.50  10500  5.50 -0.17069 -0.09067 -0.01690 -0.01256 -0.02745
-0.50  11000 -1.02 -0.30053 -0.33416 -0.11952 -0.00102 -0.00985
-0.50  11000 -0.70 -0.30053 -0.33416 -0.11952 -0.00102 -0.00985
-0.50  11000 -0.51 -0.30053 -0.33416 -0.11952 -0.00102 -0.00985
-0.50  11000 -0.29 -0.30053 -0.33416 -0.11952 -0.00102 -0.00985
-0.50  11000  0.00 -0.30053 -0.33416 -0.11952 -0.00102 -0.00985
-0.50  11000  0.28 -0.30053 -0.33416 -0.11952 -0.00102 -0.00985
-0.50  11000  0.50 -0.30053 -0.33416 -0.11952 -0.00102 -0.00985
-0.50  11000  0.60 -0.30053 -0.33416 -0.11952 -0.00102 -0.00985
-0.50  11000  0.87 -0.30053 -0.33416 -0.11952 -0.00102 -0.00985
-0.50  11000  1.00 -0.30053 -0.33416 -0.11952 -0.00102 -0.00985
-0.50  11000  1.50 -0.30053 -0.33416 -0.11952 -0.00102 -0.00985
-0.50  11000  2.00 -0.30053 -0.33416 -0.11952 -0.00102 -0.00985
-0.50  11000  2.50 -0.30053 -0.33416 -0.11952 -0.00102 -0.00985
-0.50  11000  3.00 -0.29782 -0.27799 -0.12217 -0.01008 -0.01501
-0.50  11000  3.50 -0.29884 -0.23306 -0.11540 -0.01561 -0.02037
-0.50  11000  4.00 -0.30051 -0.19685 -0.10074 -0.01873 -0.02599
-0.50  11000  4.50 -0.30115 -0.16998 -0.07821 -0.02015 -0.03175
-0.50  11000  5.00 -0.30035 -0.15610 -0.04739 -0.02011 -0.03708
-0.50  11000  5.50 -0.30035 -0.15610 -0.04739 -0.02011 -0.03708
-0.50  11500 -1.02 -0.48324 -0.42392 -0.14010 -0.00547 -0.02227
-0.50  11500 -0.70 -0.48324 -0.42392 -0.14010 -0.00547 -0.02227
-0.50  11500 -0.51 -0.48324 -0.42392 -0.14010 -0.00547 -0.02227
-0.50  11500 -0.29 -0.48324 -0.42392 -0.14010 -0.00547 -0.02227
-0.50  11500  0.00 -0.48324 -0.42392 -0.14010 -0.00547 -0.02227
-0.50  11500  0.28 -0.48324 -0.42392 -0.14010 -0.00547 -0.02227
-0.50  11500  0.50 -0.48324 -0.42392 -0.14010 -0.00547 -0.02227
-0.50  11500  0.60 -0.48324 -0.42392 -0.14010 -0.00547 -0.02227
-0.50  11500  0.87 -0.48324 -0.42392 -0.14010 -0.00547 -0.02227
-0.50  11500  1.00 -0.48324 -0.42392 -0.14010 -0.00547 -0.02227
-0.50  11500  1.50 -0.48324 -0.42392 -0.14010 -0.00547 -0.02227
-0.50  11500  2.00 -0.48324 -0.42392 -0.14010 -0.00547 -0.02227
-0.50  11500  2.50 -0.48324 -0.42392 -0.14010 -0.00547 -0.02227
-0.50  11500  3.00 -0.48178 -0.36889 -0.14352 -0.01450 -0.02722
-0.50  11500  3.50 -0.48473 -0.32468 -0.13816 -0.02029 -0.03225
-0.50  11500  4.00 -0.48866 -0.28854 -0.12587 -0.02414 -0.03736
-0.50  11500  4.50 -0.49160 -0.26069 -0.10693 -0.02647 -0.04247
-0.50  11500  5.00 -0.49248 -0.24322 -0.08041 -0.02765 -0.04747
-0.50  11500  5.50 -0.49248 -0.24322 -0.08041 -0.02765 -0.04747
-0.50  12000 -1.02 -0.44902 -0.42272 -0.12822 -0.02166 -0.03266
-0.50  12000 -0.70 -0.44902 -0.42272 -0.12822 -0.02166 -0.03266
-0.50  12000 -0.51 -0.44902 -0.42272 -0.12822 -0.02166 -0.03266
-0.50  12000 -0.29 -0.44902 -0.42272 -0.12822 -0.02166 -0.03266
-0.50  12000  0.00 -0.44902 -0.42272 -0.12822 -0.02166 -0.03266
-0.50  12000  0.28 -0.44902 -0.42272 -0.12822 -0.02166 -0.03266
-0.50  12000  0.50 -0.44902 -0.42272 -0.12822 -0.02166 -0.03266
-0.50  12000  0.60 -0.44902 -0.42272 -0.12822 -0.02166 -0.03266
-0.50  12000  0.87 -0.44902 -0.42272 -0.12822 -0.02166 -0.03266
-0.50  12000  1.00 -0.44902 -0.42272 -0.12822 -0.02166 -0.03266
-0.50  12000  1.50 -0.44902 -0.42272 -0.12822 -0.02166 -0.03266
-0.50  12000  2.00 -0.44902 -0.42272 -0.12822 -0.02166 -0.03266
-0.50  12000  2.50 -0.44902 -0.42272 -0.12822 -0.02166 -0.03266
-0.50  12000  3.00 -0.44449 -0.36997 -0.13227 -0.03021 -0.03758
-0.50  12000  3.50 -0.44498 -0.32721 -0.12801 -0.03583 -0.04248
-0.50  12000  4.00 -0.44720 -0.29154 -0.11753 -0.03988 -0.04729
-0.50  12000  4.50 -0.44964 -0.26298 -0.10120 -0.04288 -0.05178
-0.50  12000  5.00 -0.45083 -0.24282 -0.07865 -0.04487 -0.05640
-0.50  12000  5.50 -0.45083 -0.24282 -0.07865 -0.04487 -0.05640
-0.50  12500 -1.02 -0.54450 -0.46757 -0.13627 -0.02708 -0.04245
-0.50  12500 -0.70 -0.54450 -0.46757 -0.13627 -0.02708 -0.04245
-0.50  12500 -0.51 -0.54450 -0.46757 -0.13627 -0.02708 -0.04245
-0.50  12500 -0.29 -0.54450 -0.46757 -0.13627 -0.02708 -0.04245
-0.50  12500  0.00 -0.54450 -0.46757 -0.13627 -0.02708 -0.04245
-0.50  12500  0.28 -0.54450 -0.46757 -0.13627 -0.02708 -0.04245
-0.50  12500  0.50 -0.54450 -0.46757 -0.13627 -0.02708 -0.04245
-0.50  12500  0.60 -0.54450 -0.46757 -0.13627 -0.02708 -0.04245
-0.50  12500  0.87 -0.54450 -0.46757 -0.13627 -0.02708 -0.04245
-0.50  12500  1.00 -0.54450 -0.46757 -0.13627 -0.02708 -0.04245
-0.50  12500  1.50 -0.54450 -0.46757 -0.13627 -0.02708 -0.04245
-0.50  12500  2.00 -0.54450 -0.46757 -0.13627 -0.02708 -0.04245
-0.50  12500  2.50 -0.54450 -0.46757 -0.13627 -0.02708 -0.04245
-0.50  12500  3.00 -0.54016 -0.41686 -0.14082 -0.03506 -0.04704
-0.50  12500  3.50 -0.54048 -0.37641 -0.13694 -0.04040 -0.05164
-0.50  12500  4.00 -0.54269 -0.34239 -0.12742 -0.04444 -0.05621
-0.50  12500  4.50 -0.54545 -0.31446 -0.11268 -0.04765 -0.06051
-0.50  12500  5.00 -0.54741 -0.29405 -0.09250 -0.05010 -0.06473
-0.50  12500  5.50 -0.54741 -0.29405 -0.09250 -0.05010 -0.06473
-0.50  13000 -1.02 -0.63802 -0.50764 -0.14389 -0.03253 -0.05139
-0.50  13000 -0.70 -0.63802 -0.50764 -0.14389 -0.03253 -0.05139
-0.50  13000 -0.51 -0.63802 -0.50764 -0.14389 -0.03253 -0.05139
-0.50  13000 -0.29 -0.63802 -0.50764 -0.14389 -0.03253 -0.05139
-0.50  13000  0.00 -0.63802 -0.50764 -0.14389 -0.03253 -0.05139
-0.50  13000  0.28 -0.63802 -0.50764 -0.14389 -0.03253 -0.05139
-0.50  13000  0.50 -0.63802 -0.50764 -0.14389 -0.03253 -0.05139
-0.50  13000  0.60 -0.63802 -0.50764 -0.14389 -0.03253 -0.05139
-0.50  13000  0.87 -0.63802 -0.50764 -0.14389 -0.03253 -0.05139
-0.50  13000  1.00 -0.63802 -0.50764 -0.14389 -0.03253 -0.05139
-0.50  13000  1.50 -0.63802 -0.50764 -0.14389 -0.03253 -0.05139
-0.50  13000  2.00 -0.63802 -0.50764 -0.14389 -0.03253 -0.05139
-0.50  13000  2.50 -0.63802 -0.50764 -0.14389 -0.03253 -0.05139
-0.50  13000  3.00 -0.63417 -0.45813 -0.14875 -0.04024 -0.05559
-0.50  13000  3.50 -0.63469 -0.41979 -0.14542 -0.04505 -0.05997
-0.50  13000  4.00 -0.63678 -0.38792 -0.13646 -0.04880 -0.06433
-0.50  13000  4.50 -0.63939 -0.36146 -0.12262 -0.05206 -0.06851
-0.50  13000  5.00 -0.64179 -0.34138 -0.10407 -0.05468 -0.07256
-0.50  13000  5.50 -0.64179 -0.34138 -0.10407 -0.05468 -0.07256
-0.50  14000 -1.02 -0.83290 -0.66068 -0.13455 -0.02691 -0.06262
-0.50  14000 -0.70 -0.83290 -0.66068 -0.13455 -0.02691 -0.06262
-0.50  14000 -0.51 -0.83290 -0.66068 -0.13455 -0.02691 -0.06262
-0.50  14000 -0.29 -0.83290 -0.66068 -0.13455 -0.02691 -0.06262
-0.50  14000  0.00 -0.83290 -0.66068 -0.13455 -0.02691 -0.06262
-0.50  14000  0.28 -0.83290 -0.66068 -0.13455 -0.02691 -0.06262
-0.50  14000  0.50 -0.83290 -0.66068 -0.13455 -0.02691 -0.06262
-0.50  14000  0.60 -0.83290 -0.66068 -0.13455 -0.02691 -0.06262
-0.50  14000  0.87 -0.83290 -0.66068 -0.13455 -0.02691 -0.06262
-0.50  14000  1.00 -0.83290 -0.66068 -0.13455 -0.02691 -0.06262
-0.50  14000  1.50 -0.83290 -0.66068 -0.13455 -0.02691 -0.06262
-0.50  14000  2.00 -0.83290 -0.66068 -0.13455 -0.02691 -0.06262
-0.50  14000  2.50 -0.81725 -0.57804 -0.15774 -0.04302 -0.06763
-0.50  14000  3.00 -0.81575 -0.52892 -0.16329 -0.05021 -0.07122
-0.50  14000  3.50 -0.81745 -0.49356 -0.16037 -0.05441 -0.07485
-0.50  14000  4.00 -0.81974 -0.46534 -0.15235 -0.05755 -0.07875
-0.50  14000  4.50 -0.82211 -0.44239 -0.13995 -0.06025 -0.08272
-0.50  14000  5.00 -0.82445 -0.42433 -0.12354 -0.06284 -0.08647
-0.50  14000  5.50 -0.82445 -0.42433 -0.12354 -0.06284 -0.08647
-0.50  15000 -1.02 -0.98487 -0.63971 -0.17044 -0.05280 -0.08212
-0.50  15000 -0.70 -0.98487 -0.63971 -0.17044 -0.05280 -0.08212
-0.50  15000 -0.51 -0.98487 -0.63971 -0.17044 -0.05280 -0.08212
-0.50  15000 -0.29 -0.98487 -0.63971 -0.17044 -0.05280 -0.08212
-0.50  15000  0.00 -0.98487 -0.63971 -0.17044 -0.05280 -0.08212
-0.50  15000  0.28 -0.98487 -0.63971 -0.17044 -0.05280 -0.08212
-0.50  15000  0.50 -0.98487 -0.63971 -0.17044 -0.05280 -0.08212
-0.50  15000  0.60 -0.98487 -0.63971 -0.17044 -0.05280 -0.08212
-0.50  15000  0.87 -0.98487 -0.63971 -0.17044 -0.05280 -0.08212
-0.50  15000  1.00 -0.98487 -0.63971 -0.17044 -0.05280 -0.08212
-0.50  15000  1.50 -0.98487 -0.63971 -0.17044 -0.05280 -0.08212
-0.50  15000  2.00 -0.98487 -0.63971 -0.17044 -0.05280 -0.08212
-0.50  15000  2.50 -0.98487 -0.63971 -0.17044 -0.05280 -0.08212
-0.50  15000  3.00 -0.98738 -0.58859 -0.17686 -0.05976 -0.08514
-0.50  15000  3.50 -0.99145 -0.55416 -0.17427 -0.06350 -0.08815
-0.50  15000  4.00 -0.99500 -0.52846 -0.16685 -0.06594 -0.09140
-0.50  15000  4.50 -0.99749 -0.50862 -0.15541 -0.06813 -0.09485
-0.50  15000  5.00 -0.99934 -0.49348 -0.14024 -0.07039 -0.09837
-0.50  15000  5.50 -0.99934 -0.49348 -0.14024 -0.07039 -0.09837
-0.50  16000 -1.02 -1.14059 -0.69618 -0.18181 -0.06158 -0.09532
-0.50  16000 -0.70 -1.14059 -0.69618 -0.18181 -0.06158 -0.09532
-0.50  16000 -0.51 -1.14059 -0.69618 -0.18181 -0.06158 -0.09532
-0.50  16000 -0.29 -1.14059 -0.69618 -0.18181 -0.06158 -0.09532
-0.50  16000  0.00 -1.14059 -0.69618 -0.18181 -0.06158 -0.09532
-0.50  16000  0.28 -1.14059 -0.69618 -0.18181 -0.06158 -0.09532
-0.50  16000  0.50 -1.14059 -0.69618 -0.18181 -0.06158 -0.09532
-0.50  16000  0.60 -1.14059 -0.69618 -0.18181 -0.06158 -0.09532
-0.50  16000  0.87 -1.14059 -0.69618 -0.18181 -0.06158 -0.09532
-0.50  16000  1.00 -1.14059 -0.69618 -0.18181 -0.06158 -0.09532
-0.50  16000  1.50 -1.14059 -0.69618 -0.18181 -0.06158 -0.09532
-0.50  16000  2.00 -1.14059 -0.69618 -0.18181 -0.06158 -0.09532
-0.50  16000  2.50 -1.14059 -0.69618 -0.18181 -0.06158 -0.09532
-0.50  16000  3.00 -1.14804 -0.64119 -0.18953 -0.06869 -0.09787
-0.50  16000  3.50 -1.15569 -0.60632 -0.18733 -0.07197 -0.10016
-0.50  16000  4.00 -1.16114 -0.58189 -0.17999 -0.07421 -0.10281
-0.50  16000  4.50 -1.16461 -0.56379 -0.16938 -0.07589 -0.10577
-0.50  16000  5.00 -1.16642 -0.55119 -0.15513 -0.07779 -0.10888
-0.50  16000  5.50 -1.16642 -0.55119 -0.15513 -0.07779 -0.10888
-0.50  17000 -1.02 -1.28448 -0.75008 -0.19041 -0.06960 -0.10729
-0.50  17000 -0.70 -1.28448 -0.75008 -0.19041 -0.06960 -0.10729
-0.50  17000 -0.51 -1.28448 -0.75008 -0.19041 -0.06960 -0.10729
-0.50  17000 -0.29 -1.28448 -0.75008 -0.19041 -0.06960 -0.10729
-0.50  17000  0.00 -1.28448 -0.75008 -0.19041 -0.06960 -0.10729
-0.50  17000  0.28 -1.28448 -0.75008 -0.19041 -0.06960 -0.10729
-0.50  17000  0.50 -1.28448 -0.75008 -0.19041 -0.06960 -0.10729
-0.50  17000  0.60 -1.28448 -0.75008 -0.19041 -0.06960 -0.10729
-0.50  17000  0.87 -1.28448 -0.75008 -0.19041 -0.06960 -0.10729
-0.50  17000  1.00 -1.28448 -0.75008 -0.19041 -0.06960 -0.10729
-0.50  17000  1.50 -1.28448 -0.75008 -0.19041 -0.06960 -0.10729
-0.50  17000  2.00 -1.28448 -0.75008 -0.19041 -0.06960 -0.10729
-0.50  17000  2.50 -1.28448 -0.75008 -0.19041 -0.06960 -0.10729
-0.50  17000  3.00 -1.29781 -0.68934 -0.20086 -0.07719 -0.10945
-0.50  17000  3.50 -1.30996 -0.65296 -0.19935 -0.08006 -0.11120
-0.50  17000  4.00 -1.31823 -0.62839 -0.19255 -0.08176 -0.11325
-0.50  17000  4.50 -1.32318 -0.61126 -0.18237 -0.08317 -0.11572
-0.50  17000  5.00 -1.32567 -0.59994 -0.16896 -0.08464 -0.11841
-0.50  17000  5.50 -1.32567 -0.59994 -0.16896 -0.08464 -0.11841
-0.50  18000 -1.02 -1.41718 -0.80217 -0.19633 -0.07607 -0.11788
-0.50  18000 -0.70 -1.41718 -0.80217 -0.19633 -0.07607 -0.11788
-0.50  18000 -0.51 -1.41718 -0.80217 -0.19633 -0.07607 -0.11788
-0.50  18000 -0.29 -1.41718 -0.80217 -0.19633 -0.07607 -0.11788
-0.50  18000  0.00 -1.41718 -0.80217 -0.19633 -0.07607 -0.11788
-0.50  18000  0.28 -1.41718 -0.80217 -0.19633 -0.07607 -0.11788
-0.50  18000  0.50 -1.41718 -0.80217 -0.19633 -0.07607 -0.11788
-0.50  18000  0.60 -1.41718 -0.80217 -0.19633 -0.07607 -0.11788
-0.50  18000  0.87 -1.41718 -0.80217 -0.19633 -0.07607 -0.11788
-0.50  18000  1.00 -1.41718 -0.80217 -0.19633 -0.07607 -0.11788
-0.50  18000  1.50 -1.41718 -0.80217 -0.19633 -0.07607 -0.11788
-0.50  18000  2.00 -1.41718 -0.80217 -0.19633 -0.07607 -0.11788
-0.50  18000  2.50 -1.41718 -0.80217 -0.19633 -0.07607 -0.11788
-0.50  18000  3.00 -1.43626 -0.73437 -0.21032 -0.08549 -0.12023
-0.50  18000  3.50 -1.45352 -0.69612 -0.20991 -0.08821 -0.12132
-0.50  18000  4.00 -1.46526 -0.67098 -0.20369 -0.08953 -0.12287
-0.50  18000  4.50 -1.47243 -0.65376 -0.19405 -0.09058 -0.12487
-0.50  18000  5.00 -1.47611 -0.64325 -0.18136 -0.09174 -0.12713
-0.50  18000  5.50 -1.47611 -0.64325 -0.18136 -0.09174 -0.12713
-0.50  19000 -1.02 -1.54102 -0.85382 -0.20018 -0.07975 -0.12634
-0.50  19000 -0.70 -1.54102 -0.85382 -0.20018 -0.07975 -0.12634
-0.50  19000 -0.51 -1.54102 -0.85382 -0.20018 -0.07975 -0.12634
-0.50  19000 -0.29 -1.54102 -0.85382 -0.20018 -0.07975 -0.12634
-0.50  19000  0.00 -1.54102 -0.85382 -0.20018 -0.07975 -0.12634
-0.50  19000  0.28 -1.54102 -0.85382 -0.20018 -0.07975 -0.12634
-0.50  19000  0.50 -1.54102 -0.85382 -0.20018 -0.07975 -0.12634
-0.50  19000  0.60 -1.54102 -0.85382 -0.20018 -0.07975 -0.12634
-0.50  19000  0.87 -1.54102 -0.85382 -0.20018 -0.07975 -0.12634
-0.50  19000  1.00 -1.54102 -0.85382 -0.20018 -0.07975 -0.12634
-0.50  19000  1.50 -1.54102 -0.85382 -0.20018 -0.07975 -0.12634
-0.50  19000  2.00 -1.54102 -0.85382 -0.20018 -0.07975 -0.12634
-0.50  19000  2.50 -1.54102 -0.85382 -0.20018 -0.07975 -0.12634
-0.50  19000  3.00 -1.56467 -0.77668 -0.21841 -0.09257 -0.13037
-0.50  19000  3.50 -1.58751 -0.73563 -0.21967 -0.09562 -0.13101
-0.50  19000  4.00 -1.60329 -0.70966 -0.21411 -0.09661 -0.13191
-0.50  19000  4.50 -1.61296 -0.69233 -0.20486 -0.09751 -0.13339
-0.50  19000  5.00 -1.61827 -0.68185 -0.19272 -0.09849 -0.13530
-0.50  19000  5.50 -1.61827 -0.68185 -0.19272 -0.09849 -0.13530
-0.50  20000 -1.02 -1.68385 -0.81727 -0.22506 -0.09840 -0.13949
-0.50  20000 -0.70 -1.68385 -0.81727 -0.22506 -0.09840 -0.13949
-0.50  20000 -0.51 -1.68385 -0.81727 -0.22506 -0.09840 -0.13949
-0.50  20000 -0.29 -1.68385 -0.81727 -0.22506 -0.09840 -0.13949
-0.50  20000  0.00 -1.68385 -0.81727 -0.22506 -0.09840 -0.13949
-0.50  20000  0.28 -1.68385 -0.81727 -0.22506 -0.09840 -0.13949
-0.50  20000  0.50 -1.68385 -0.81727 -0.22506 -0.09840 -0.13949
-0.50  20000  0.60 -1.68385 -0.81727 -0.22506 -0.09840 -0.13949
-0.50  20000  0.87 -1.68385 -0.81727 -0.22506 -0.09840 -0.13949
-0.50  20000  1.00 -1.68385 -0.81727 -0.22506 -0.09840 -0.13949
-0.50  20000  1.50 -1.68385 -0.81727 -0.22506 -0.09840 -0.13949
-0.50  20000  2.00 -1.68385 -0.81727 -0.22506 -0.09840 -0.13949
-0.50  20000  2.50 -1.68385 -0.81727 -0.22506 -0.09840 -0.13949
-0.50  20000  3.00 -1.68385 -0.81727 -0.22506 -0.09840 -0.13949
-0.50  20000  3.50 -1.71233 -0.77251 -0.22828 -0.10235 -0.14015
-0.50  20000  4.00 -1.73255 -0.74529 -0.22378 -0.10334 -0.14047
-0.50  20000  4.50 -1.74524 -0.72765 -0.21529 -0.10388 -0.14152
-0.50  20000  5.00 -1.75256 -0.71689 -0.20378 -0.10464 -0.14304
-0.50  20000  5.50 -1.75256 -0.71689 -0.20378 -0.10464 -0.14304
-0.50  21000 -1.02 -1.79572 -0.85624 -0.23062 -0.10270 -0.14710
-0.50  21000 -0.70 -1.79572 -0.85624 -0.23062 -0.10270 -0.14710
-0.50  21000 -0.51 -1.79572 -0.85624 -0.23062 -0.10270 -0.14710
-0.50  21000 -0.29 -1.79572 -0.85624 -0.23062 -0.10270 -0.14710
-0.50  21000  0.00 -1.79572 -0.85624 -0.23062 -0.10270 -0.14710
-0.50  21000  0.28 -1.79572 -0.85624 -0.23062 -0.10270 -0.14710
-0.50  21000  0.50 -1.79572 -0.85624 -0.23062 -0.10270 -0.14710
-0.50  21000  0.60 -1.79572 -0.85624 -0.23062 -0.10270 -0.14710
-0.50  21000  0.87 -1.79572 -0.85624 -0.23062 -0.10270 -0.14710
-0.50  21000  1.00 -1.79572 -0.85624 -0.23062 -0.10270 -0.14710
-0.50  21000  1.50 -1.79572 -0.85624 -0.23062 -0.10270 -0.14710
-0.50  21000  2.00 -1.79572 -0.85624 -0.23062 -0.10270 -0.14710
-0.50  21000  2.50 -1.79572 -0.85624 -0.23062 -0.10270 -0.14710
-0.50  21000  3.00 -1.79572 -0.85624 -0.23062 -0.10270 -0.14710
-0.50  21000  3.50 -1.82879 -0.80780 -0.23586 -0.10821 -0.14863
-0.50  21000  4.00 -1.85345 -0.77870 -0.23247 -0.10957 -0.14879
-0.50  21000  4.50 -1.86947 -0.76038 -0.22474 -0.11001 -0.14931
-0.50  21000  5.00 -1.87899 -0.74941 -0.21400 -0.11056 -0.15034
-0.50  21000  5.50 -1.87899 -0.74941 -0.21400 -0.11056 -0.15034
-0.50  22000 -1.02 -1.90195 -0.89185 -0.23574 -0.10618 -0.15340
-0.50  22000 -0.70 -1.90195 -0.89185 -0.23574 -0.10618 -0.15340
-0.50  22000 -0.51 -1.90195 -0.89185 -0.23574 -0.10618 -0.15340
-0.50  22000 -0.29 -1.90195 -0.89185 -0.23574 -0.10618 -0.15340
-0.50  22000  0.00 -1.90195 -0.89185 -0.23574 -0.10618 -0.15340
-0.50  22000  0.28 -1.90195 -0.89185 -0.23574 -0.10618 -0.15340
-0.50  22000  0.50 -1.90195 -0.89185 -0.23574 -0.10618 -0.15340
-0.50  22000  0.60 -1.90195 -0.89185 -0.23574 -0.10618 -0.15340
-0.50  22000  0.87 -1.90195 -0.89185 -0.23574 -0.10618 -0.15340
-0.50  22000  1.00 -1.90195 -0.89185 -0.23574 -0.10618 -0.15340
-0.50  22000  1.50 -1.90195 -0.89185 -0.23574 -0.10618 -0.15340
-0.50  22000  2.00 -1.90195 -0.89185 -0.23574 -0.10618 -0.15340
-0.50  22000  2.50 -1.90195 -0.89185 -0.23574 -0.10618 -0.15340
-0.50  22000  3.00 -1.90195 -0.89185 -0.23574 -0.10618 -0.15340
-0.50  22000  3.50 -1.93782 -0.84145 -0.24209 -0.11312 -0.15611
-0.50  22000  4.00 -1.96652 -0.81038 -0.24015 -0.11522 -0.15651
-0.50  22000  4.50 -1.98614 -0.79109 -0.23348 -0.11574 -0.15675
-0.50  22000  5.00 -1.99816 -0.77972 -0.22350 -0.11618 -0.15746
-0.50  22000  5.50 -1.99816 -0.77972 -0.22350 -0.11618 -0.15746
-0.50  23000 -1.02 -2.00421 -0.92276 -0.24142 -0.10992 -0.15891
-0.50  23000 -0.70 -2.00421 -0.92276 -0.24142 -0.10992 -0.15891
-0.50  23000 -0.51 -2.00421 -0.92276 -0.24142 -0.10992 -0.15891
-0.50  23000 -0.29 -2.00421 -0.92276 -0.24142 -0.10992 -0.15891
-0.50  23000  0.00 -2.00421 -0.92276 -0.24142 -0.10992 -0.15891
-0.50  23000  0.28 -2.00421 -0.92276 -0.24142 -0.10992 -0.15891
-0.50  23000  0.50 -2.00421 -0.92276 -0.24142 -0.10992 -0.15891
-0.50  23000  0.60 -2.00421 -0.92276 -0.24142 -0.10992 -0.15891
-0.50  23000  0.87 -2.00421 -0.92276 -0.24142 -0.10992 -0.15891
-0.50  23000  1.00 -2.00421 -0.92276 -0.24142 -0.10992 -0.15891
-0.50  23000  1.50 -2.00421 -0.92276 -0.24142 -0.10992 -0.15891
-0.50  23000  2.00 -2.00421 -0.92276 -0.24142 -0.10992 -0.15891
-0.50  23000  2.50 -2.00421 -0.92276 -0.24142 -0.10992 -0.15891
-0.50  23000  3.00 -2.00421 -0.92276 -0.24142 -0.10992 -0.15891
-0.50  23000  3.50 -2.04094 -0.87269 -0.24783 -0.11750 -0.16238
-0.50  23000  4.00 -2.07256 -0.84046 -0.24687 -0.12022 -0.16355
-0.50  23000  4.50 -2.09544 -0.81991 -0.24128 -0.12104 -0.16382
-0.50  23000  5.00 -2.11004 -0.80797 -0.23216 -0.12158 -0.16427
-0.50  23000  5.50 -2.11004 -0.80797 -0.23216 -0.12158 -0.16427
-0.50  24000 -1.02 -2.10089 -0.94975 -0.24748 -0.11421 -0.16471
-0.50  24000 -0.70 -2.10089 -0.94975 -0.24748 -0.11421 -0.16471
-0.50  24000 -0.51 -2.10089 -0.94975 -0.24748 -0.11421 -0.16471
-0.50  24000 -0.29 -2.10089 -0.94975 -0.24748 -0.11421 -0.16471
-0.50  24000  0.00 -2.10089 -0.94975 -0.24748 -0.11421 -0.16471
-0.50  24000  0.28 -2.10089 -0.94975 -0.24748 -0.11421 -0.16471
-0.50  24000  0.50 -2.10089 -0.94975 -0.24748 -0.11421 -0.16471
-0.50  24000  0.60 -2.10089 -0.94975 -0.24748 -0.11421 -0.16471
-0.50  24000  0.87 -2.10089 -0.94975 -0.24748 -0.11421 -0.16471
-0.50  24000  1.00 -2.10089 -0.94975 -0.24748 -0.11421 -0.16471
-0.50  24000  1.50 -2.10089 -0.94975 -0.24748 -0.11421 -0.16471
-0.50  24000  2.00 -2.10089 -0.94975 -0.24748 -0.11421 -0.16471
-0.50  24000  2.50 -2.10089 -0.94975 -0.24748 -0.11421 -0.16471
-0.50  24000  3.00 -2.10089 -0.94975 -0.24748 -0.11421 -0.16471
-0.50  24000  3.50 -2.13990 -0.90039 -0.25375 -0.12150 -0.16817
-0.50  24000  4.00 -2.17300 -0.86865 -0.25299 -0.12461 -0.16961
-0.50  24000  4.50 -2.19806 -0.84741 -0.24811 -0.12604 -0.17044
-0.50  24000  5.00 -2.21516 -0.83444 -0.24008 -0.12673 -0.17084
-0.50  24000  5.50 -2.21516 -0.83444 -0.24008 -0.12673 -0.17084
-0.50  25000 -1.02 -2.18947 -0.97592 -0.25234 -0.11774 -0.17067
-0.50  25000 -0.70 -2.18947 -0.97592 -0.25234 -0.11774 -0.17067
-0.50  25000 -0.51 -2.18947 -0.97592 -0.25234 -0.11774 -0.17067
-0.50  25000 -0.29 -2.18947 -0.97592 -0.25234 -0.11774 -0.17067
-0.50  25000  0.00 -2.18947 -0.97592 -0.25234 -0.11774 -0.17067
-0.50  25000  0.28 -2.18947 -0.97592 -0.25234 -0.11774 -0.17067
-0.50  25000  0.50 -2.18947 -0.97592 -0.25234 -0.11774 -0.17067
-0.50  25000  0.60 -2.18947 -0.97592 -0.25234 -0.11774 -0.17067
-0.50  25000  0.87 -2.18947 -0.97592 -0.25234 -0.11774 -0.17067
-0.50  25000  1.00 -2.18947 -0.97592 -0.25234 -0.11774 -0.17067
-0.50  25000  1.50 -2.18947 -0.97592 -0.25234 -0.11774 -0.17067
-0.50  25000  2.00 -2.18947 -0.97592 -0.25234 -0.11774 -0.17067
-0.50  25000  2.50 -2.18947 -0.97592 -0.25234 -0.11774 -0.17067
-0.50  25000  3.00 -2.18947 -0.97592 -0.25234 -0.11774 -0.17067
-0.50  25000  3.50 -2.23446 -0.92441 -0.26023 -0.12590 -0.17388
-0.50  25000  4.00 -2.26891 -0.89416 -0.25905 -0.12875 -0.17522
-0.50  25000  4.50 -2.29524 -0.87323 -0.25464 -0.13042 -0.17612
-0.50  25000  5.00 -2.31406 -0.86004 -0.24720 -0.13145 -0.17691
-0.50  25000  5.50 -2.31406 -0.86004 -0.24720 -0.13145 -0.17691
-0.50  26000 -1.02 -2.27133 -1.00313 -0.25431 -0.11870 -0.17481
-0.50  26000 -0.70 -2.27133 -1.00313 -0.25431 -0.11870 -0.17481
-0.50  26000 -0.51 -2.27133 -1.00313 -0.25431 -0.11870 -0.17481
-0.50  26000 -0.29 -2.27133 -1.00313 -0.25431 -0.11870 -0.17481
-0.50  26000  0.00 -2.27133 -1.00313 -0.25431 -0.11870 -0.17481
-0.50  26000  0.28 -2.27133 -1.00313 -0.25431 -0.11870 -0.17481
-0.50  26000  0.50 -2.27133 -1.00313 -0.25431 -0.11870 -0.17481
-0.50  26000  0.60 -2.27133 -1.00313 -0.25431 -0.11870 -0.17481
-0.50  26000  0.87 -2.27133 -1.00313 -0.25431 -0.11870 -0.17481
-0.50  26000  1.00 -2.27133 -1.00313 -0.25431 -0.11870 -0.17481
-0.50  26000  1.50 -2.27133 -1.00313 -0.25431 -0.11870 -0.17481
-0.50  26000  2.00 -2.27133 -1.00313 -0.25431 -0.11870 -0.17481
-0.50  26000  2.50 -2.27133 -1.00313 -0.25431 -0.11870 -0.17481
-0.50  26000  3.00 -2.27133 -1.00313 -0.25431 -0.11870 -0.17481
-0.50  26000  3.50 -2.32362 -0.94575 -0.26689 -0.13101 -0.17989
-0.50  26000  4.00 -2.36083 -0.91679 -0.26540 -0.13271 -0.18062
-0.50  26000  4.50 -2.38769 -0.89711 -0.26079 -0.13452 -0.18157
-0.50  26000  5.00 -2.40750 -0.88394 -0.25393 -0.13581 -0.18231
-0.50  26000  5.50 -2.40750 -0.88394 -0.25393 -0.13581 -0.18231
-0.50  27000 -1.02 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
-0.50  27000 -0.70 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
-0.50  27000 -0.51 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
-0.50  27000 -0.29 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
-0.50  27000  0.00 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
-0.50  27000  0.28 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
-0.50  27000  0.50 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
-0.50  27000  0.60 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
-0.50  27000  0.87 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
-0.50  27000  1.00 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
-0.50  27000  1.50 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
-0.50  27000  2.00 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
-0.50  27000  2.50 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
-0.50  27000  3.00 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
-0.50  27000  3.50 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
-0.50  27000  4.00 -2.44762 -0.93680 -0.27204 -0.13755 -0.18642
-0.50  27000  4.50 -2.47615 -0.91840 -0.26722 -0.13876 -0.18637
-0.50  27000  5.00 -2.49617 -0.90623 -0.26042 -0.13984 -0.18750
-0.50  27000  5.50 -2.49617 -0.90623 -0.26042 -0.13984 -0.18750
-0.50  28000 -1.02 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
-0.50  28000 -0.70 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
-0.50  28000 -0.51 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
-0.50  28000 -0.29 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
-0.50  28000  0.00 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
-0.50  28000  0.28 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
-0.50  28000  0.50 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
-0.50  28000  0.60 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
-0.50  28000  0.87 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
-0.50  28000  1.00 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
-0.50  28000  1.50 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
-0.50  28000  2.00 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
-0.50  28000  2.50 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
-0.50  28000  3.00 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
-0.50  28000  3.50 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
-0.50  28000  4.00 -2.52892 -0.95542 -0.27879 -0.14239 -0.19191
-0.50  28000  4.50 -2.55968 -0.93767 -0.27357 -0.14305 -0.19198
-0.50  28000  5.00 -2.58074 -0.92661 -0.26689 -0.14394 -0.19221
-0.50  28000  5.50 -2.58074 -0.92661 -0.26689 -0.14394 -0.19221
-0.50  29000 -1.02 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
-0.50  29000 -0.70 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
-0.50  29000 -0.51 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
-0.50  29000 -0.29 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
-0.50  29000  0.00 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
-0.50  29000  0.28 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
-0.50  29000  0.50 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
-0.50  29000  0.60 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
-0.50  29000  0.87 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
-0.50  29000  1.00 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
-0.50  29000  1.50 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
-0.50  29000  2.00 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
-0.50  29000  2.50 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
-0.50  29000  3.00 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
-0.50  29000  3.50 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
-0.50  29000  4.00 -2.60355 -0.97348 -0.28526 -0.14669 -0.19743
-0.50  29000  4.50 -2.63831 -0.95559 -0.28024 -0.14743 -0.19725
-0.50  29000  5.00 -2.66049 -0.94535 -0.27337 -0.14777 -0.19732
-0.50  29000  5.50 -2.66049 -0.94535 -0.27337 -0.14777 -0.19732
-0.50  30000 -1.02 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
-0.50  30000 -0.70 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
-0.50  30000 -0.51 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
-0.50  30000 -0.29 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
-0.50  30000  0.00 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
-0.50  30000  0.28 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
-0.50  30000  0.50 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
-0.50  30000  0.60 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
-0.50  30000  0.87 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
-0.50  30000  1.00 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
-0.50  30000  1.50 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
-0.50  30000  2.00 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
-0.50  30000  2.50 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
-0.50  30000  3.00 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
-0.50  30000  3.50 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
-0.50  30000  4.00 -2.67342 -0.99115 -0.29078 -0.15018 -0.20214
-0.50  30000  4.50 -2.71144 -0.97261 -0.28665 -0.15153 -0.20230
-0.50  30000  5.00 -2.73584 -0.96265 -0.27981 -0.15176 -0.20218
-0.50  30000  5.50 -2.73584 -0.96265 -0.27981 -0.15176 -0.20218
-0.50  31000 -1.02 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
-0.50  31000 -0.70 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
-0.50  31000 -0.51 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
-0.50  31000 -0.29 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
-0.50  31000  0.00 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
-0.50  31000  0.28 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
-0.50  31000  0.50 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
-0.50  31000  0.60 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
-0.50  31000  0.87 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
-0.50  31000  1.00 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
-0.50  31000  1.50 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
-0.50  31000  2.00 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
-0.50  31000  2.50 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
-0.50  31000  3.00 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
-0.50  31000  3.50 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
-0.50  31000  4.00 -2.74139 -1.00792 -0.29531 -0.15236 -0.20587
-0.50  31000  4.50 -2.78037 -0.98904 -0.29260 -0.15499 -0.20676
-0.50  31000  5.00 -2.80661 -0.97910 -0.28612 -0.15544 -0.20660
-0.50  31000  5.50 -2.80661 -0.97910 -0.28612 -0.15544 -0.20660
-0.50  32000 -1.02 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
-0.50  32000 -0.70 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
-0.50  32000 -0.51 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
-0.50  32000 -0.29 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
-0.50  32000  0.00 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
-0.50  32000  0.28 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
-0.50  32000  0.50 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
-0.50  32000  0.60 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
-0.50  32000  0.87 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
-0.50  32000  1.00 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
-0.50  32000  1.50 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
-0.50  32000  2.00 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
-0.50  32000  2.50 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
-0.50  32000  3.00 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
-0.50  32000  3.50 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
-0.50  32000  4.00 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
-0.50  32000  4.50 -2.84690 -1.00455 -0.29779 -0.15760 -0.21048
-0.50  32000  5.00 -2.87420 -0.99452 -0.29207 -0.15854 -0.21056
-0.50  32000  5.50 -2.87420 -0.99452 -0.29207 -0.15854 -0.21056
-0.50  33000 -1.02 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
-0.50  33000 -0.70 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
-0.50  33000 -0.51 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
-0.50  33000 -0.29 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
-0.50  33000  0.00 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
-0.50  33000  0.28 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
-0.50  33000  0.50 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
-0.50  33000  0.60 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
-0.50  33000  0.87 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
-0.50  33000  1.00 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
-0.50  33000  1.50 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
-0.50  33000  2.00 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
-0.50  33000  2.50 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
-0.50  33000  3.00 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
-0.50  33000  3.50 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
-0.50  33000  4.00 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
-0.50  33000  4.50 -2.91195 -1.01898 -0.30219 -0.15942 -0.21359
-0.50  33000  5.00 -2.93942 -1.00904 -0.29737 -0.16111 -0.21400
-0.50  33000  5.50 -2.93942 -1.00904 -0.29737 -0.16111 -0.21400
-0.50  34000 -1.02 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
-0.50  34000 -0.70 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
-0.50  34000 -0.51 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
-0.50  34000 -0.29 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
-0.50  34000  0.00 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
-0.50  34000  0.28 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
-0.50  34000  0.50 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
-0.50  34000  0.60 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
-0.50  34000  0.87 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
-0.50  34000  1.00 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
-0.50  34000  1.50 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
-0.50  34000  2.00 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
-0.50  34000  2.50 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
-0.50  34000  3.00 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
-0.50  34000  3.50 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
-0.50  34000  4.00 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
-0.50  34000  4.50 -2.97899 -1.03181 -0.30566 -0.16027 -0.21590
-0.50  34000  5.00 -3.00391 -1.02231 -0.30213 -0.16293 -0.21688
-0.50  34000  5.50 -3.00391 -1.02231 -0.30213 -0.16293 -0.21688
-0.50  35000 -1.02 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
-0.50  35000 -0.70 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
-0.50  35000 -0.51 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
-0.50  35000 -0.29 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
-0.50  35000  0.00 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
-0.50  35000  0.28 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
-0.50  35000  0.50 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
-0.50  35000  0.60 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
-0.50  35000  0.87 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
-0.50  35000  1.00 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
-0.50  35000  1.50 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
-0.50  35000  2.00 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
-0.50  35000  2.50 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
-0.50  35000  3.00 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
-0.50  35000  3.50 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
-0.50  35000  4.00 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
-0.50  35000  4.50 -3.04809 -1.04294 -0.30835 -0.16031 -0.21740
-0.50  35000  5.00 -3.06963 -1.03422 -0.30595 -0.16402 -0.21917
-0.50  35000  5.50 -3.06963 -1.03422 -0.30595 -0.16402 -0.21917
-0.50  37500 -1.02 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
-0.50  37500 -0.70 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
-0.50  37500 -0.51 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
-0.50  37500 -0.29 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
-0.50  37500  0.00 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
-0.50  37500  0.28 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
-0.50  37500  0.50 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
-0.50  37500  0.60 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
-0.50  37500  0.87 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
-0.50  37500  1.00 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
-0.50  37500  1.50 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
-0.50  37500  2.00 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
-0.50  37500  2.50 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
-0.50  37500  3.00 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
-0.50  37500  3.50 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
-0.50  37500  4.00 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
-0.50  37500  4.50 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
-0.50  37500  5.00 -3.24088 -1.05803 -0.31302 -0.16379 -0.22209
-0.50  37500  5.50 -3.24088 -1.05803 -0.31302 -0.16379 -0.22209
-0.50  40000 -1.02 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
-0.50  40000 -0.70 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
-0.50  40000 -0.51 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
-0.50  40000 -0.29 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
-0.50  40000  0.00 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
-0.50  40000  0.28 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
-0.50  40000  0.50 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
-0.50  40000  0.60 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
-0.50  40000  0.87 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
-0.50  40000  1.00 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
-0.50  40000  1.50 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
-0.50  40000  2.00 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
-0.50  40000  2.50 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
-0.50  40000  3.00 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
-0.50  40000  3.50 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
-0.50  40000  4.00 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
-0.50  40000  4.50 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
-0.50  40000  5.00 -3.42493 -1.07283 -0.31872 -0.16266 -0.22304
-0.50  40000  5.50 -3.42493 -1.07283 -0.31872 -0.16266 -0.22304
-0.50  42500 -1.02 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
-0.50  42500 -0.70 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
-0.50  42500 -0.51 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
-0.50  42500 -0.29 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
-0.50  42500  0.00 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
-0.50  42500  0.28 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
-0.50  42500  0.50 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
-0.50  42500  0.60 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
-0.50  42500  0.87 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
-0.50  42500  1.00 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
-0.50  42500  1.50 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
-0.50  42500  2.00 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
-0.50  42500  2.50 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
-0.50  42500  3.00 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
-0.50  42500  3.50 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
-0.50  42500  4.00 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
-0.50  42500  4.50 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
-0.50  42500  5.00 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
-0.50  42500  5.50 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
-0.50  45000 -1.02 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
-0.50  45000 -0.70 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
-0.50  45000 -0.51 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
-0.50  45000 -0.29 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
-0.50  45000  0.00 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
-0.50  45000  0.28 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
-0.50  45000  0.50 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
-0.50  45000  0.60 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
-0.50  45000  0.87 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
-0.50  45000  1.00 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
-0.50  45000  1.50 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
-0.50  45000  2.00 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
-0.50  45000  2.50 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
-0.50  45000  3.00 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
-0.50  45000  3.50 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
-0.50  45000  4.00 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
-0.50  45000  4.50 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
-0.50  45000  5.00 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
-0.50  45000  5.50 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
-0.50  47500 -1.02 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
-0.50  47500 -0.70 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
-0.50  47500 -0.51 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
-0.50  47500 -0.29 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
-0.50  47500  0.00 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
-0.50  47500  0.28 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
-0.50  47500  0.50 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
-0.50  47500  0.60 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
-0.50  47500  0.87 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
-0.50  47500  1.00 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
-0.50  47500  1.50 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
-0.50  47500  2.00 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
-0.50  47500  2.50 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
-0.50  47500  3.00 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
-0.50  47500  3.50 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
-0.50  47500  4.00 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
-0.50  47500  4.50 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
-0.50  47500  5.00 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
-0.50  47500  5.50 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
-0.50  50000 -1.02 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
-0.50  50000 -0.70 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
-0.50  50000 -0.51 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
-0.50  50000 -0.29 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
-0.50  50000  0.00 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
-0.50  50000  0.28 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
-0.50  50000  0.50 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
-0.50  50000  0.60 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
-0.50  50000  0.87 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
-0.50  50000  1.00 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
-0.50  50000  1.50 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
-0.50  50000  2.00 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
-0.50  50000  2.50 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
-0.50  50000  3.00 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
-0.50  50000  3.50 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
-0.50  50000  4.00 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
-0.50  50000  4.50 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
-0.50  50000  5.00 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
-0.50  50000  5.50 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
-0.30  2000 -1.02 -7.05491 -0.29783  1.12030  3.74948  3.02679
-0.30  2000 -0.70 -7.05491 -0.29783  1.12030  3.74948  3.02679
-0.30  2000 -0.51 -7.05491 -0.29783  1.12030  3.74948  3.02679
-0.30  2000 -0.29 -7.05491 -0.29783  1.12030  3.74948  3.02679
-0.30  2000  0.00 -7.05491 -0.29783  1.12030  3.74948  3.02679
-0.30  2000  0.28 -7.05491 -0.29783  1.12030  3.74948  3.02679
-0.30  2000  0.50 -7.05491 -0.29783  1.12030  3.74948  3.02679
-0.30  2000  0.60 -7.05491 -0.29783  1.12030  3.74948  3.02679
-0.30  2000  0.87 -7.05491 -0.29783  1.12030  3.74948  3.02679
-0.30  2000  1.00 -7.05491 -0.29783  1.12030  3.74948  3.02679
-0.30  2000  1.50 -7.05491 -0.29783  1.12030  3.74948  3.02679
-0.30  2000  2.00 -7.05491 -0.29783  1.12030  3.74948  3.02679
-0.30  2000  2.50 -7.05491 -0.29783  1.12030  3.74948  3.02679
-0.30  2000  3.00 -7.05491 -0.29783  1.12030  3.74948  3.02679
-0.30  2000  3.50 -7.05491 -0.29783  1.12030  3.74948  3.02679
-0.30  2000  4.00 -7.19603  0.30737  1.98438  3.83970  3.00860
-0.30  2000  4.50 -7.36220  1.35906  2.55481  3.97758  2.99233
-0.30  2000  5.00 -7.45940  2.69098  2.84060  4.13058  2.93958
-0.30  2000  5.50 -7.37559  3.82038  2.96376  4.22386  2.83140
-0.30  2200 -1.02 -5.74734  0.00490  1.08443  3.18965  2.78974
-0.30  2200 -0.70 -5.74734  0.00490  1.08443  3.18965  2.78974
-0.30  2200 -0.51 -5.74734  0.00490  1.08443  3.18965  2.78974
-0.30  2200 -0.29 -5.74734  0.00490  1.08443  3.18965  2.78974
-0.30  2200  0.00 -5.74734  0.00490  1.08443  3.18965  2.78974
-0.30  2200  0.28 -5.74734  0.00490  1.08443  3.18965  2.78974
-0.30  2200  0.50 -5.74734  0.00490  1.08443  3.18965  2.78974
-0.30  2200  0.60 -5.74734  0.00490  1.08443  3.18965  2.78974
-0.30  2200  0.87 -5.74734  0.00490  1.08443  3.18965  2.78974
-0.30  2200  1.00 -5.74734  0.00490  1.08443  3.18965  2.78974
-0.30  2200  1.50 -5.74734  0.00490  1.08443  3.18965  2.78974
-0.30  2200  2.00 -5.74734  0.00490  1.08443  3.18965  2.78974
-0.30  2200  2.50 -5.74734  0.00490  1.08443  3.18965  2.78974
-0.30  2200  3.00 -5.74734  0.00490  1.08443  3.18965  2.78974
-0.30  2200  3.50 -5.74734  0.00490  1.08443  3.18965  2.78974
-0.30  2200  4.00 -5.83881  0.47459  1.74295  3.24982  2.77385
-0.30  2200  4.50 -6.03314  1.22234  2.23193  3.37866  2.79895
-0.30  2200  5.00 -6.18971  2.28050  2.56496  3.53287  2.77720
-0.30  2200  5.50 -6.26440  3.40841  2.73528  3.68918  2.70344
-0.30  2500 -1.02 -5.99423 -0.23091  1.41747  3.08183  2.87080
-0.30  2500 -0.70 -5.60200 -0.23088  1.40344  2.89877  2.78610
-0.30  2500 -0.51 -5.37788 -0.23088  1.39685  2.79747  2.73588
-0.30  2500 -0.29 -5.11838 -0.23088  1.38922  2.68017  2.67773
-0.30  2500  0.00 -4.94869 -0.23089  1.38976  2.66402  2.63922
-0.30  2500  0.28 -4.78486 -0.23089  1.39029  2.64842  2.60205
-0.30  2500  0.50 -4.76560 -0.18071  1.37907  2.64935  2.59194
-0.30  2500  0.60 -4.75684 -0.15790  1.37396  2.64977  2.58735
-0.30  2500  0.87 -4.73319 -0.09632  1.36019  2.65092  2.57496
-0.30  2500  1.00 -4.72181 -0.06667  1.35355  2.65147  2.56899
-0.30  2500  1.50 -4.67802  0.04737  1.32804  2.65358  2.54603
-0.30  2500  2.00 -4.63424  0.16141  1.30253  2.65570  2.52308
-0.30  2500  2.50 -4.59045  0.27545  1.27702  2.65782  2.50012
-0.30  2500  3.00 -4.54667  0.38949  1.25151  2.65993  2.47717
-0.30  2500  3.50 -4.50288  0.50353  1.22600  2.66205  2.45421
-0.30  2500  4.00 -4.48830  0.86898  1.63556  2.68648  2.40440
-0.30  2500  4.50 -4.57277  1.35470  1.95371  2.74300  2.41254
-0.30  2500  5.00 -4.71326  2.01739  2.21663  2.84811  2.42855
-0.30  2500  5.50 -4.90478  2.87751  2.40505  3.01804  2.42611
-0.30  2800 -1.02 -5.77601  0.06061  1.45009  2.93050  2.86379
-0.30  2800 -0.70 -5.43583  0.06067  1.44188  2.80713  2.76515
-0.30  2800 -0.51 -5.23440  0.06063  1.43772  2.73494  2.70608
-0.30  2800 -0.29 -5.00117  0.06059  1.43290  2.65135  2.63769
-0.30  2800  0.00 -4.74361  0.06062  1.43026  2.58145  2.56286
-0.30  2800  0.28 -4.49493  0.06065  1.42772  2.51396  2.49061
-0.30  2800  0.50 -4.31873  0.06069  1.42664  2.47409  2.44065
-0.30  2800  0.60 -4.23864  0.06072  1.42615  2.45597  2.41793
-0.30  2800  0.87 -4.17625  0.12522  1.42588  2.44120  2.38421
-0.30  2800  1.00 -4.14622  0.15628  1.42575  2.43408  2.36797
-0.30  2800  1.50 -4.03069  0.27573  1.42524  2.40672  2.30551
-0.30  2800  2.00 -3.91516  0.39519  1.42474  2.37937  2.24306
-0.30  2800  2.50 -3.79963  0.51464  1.42423  2.35201  2.18060
-0.30  2800  3.00 -3.68411  0.63410  1.42373  2.32465  2.11815
-0.30  2800  3.50 -3.56858  0.75355  1.42323  2.29729  2.05569
-0.30  2800  4.00 -3.51492  0.96750  1.53825  2.26781  2.07231
-0.30  2800  4.50 -3.49497  1.26795  1.76250  2.28497  2.04218
-0.30  2800  5.00 -3.54027  1.67907  1.93425  2.32187  2.05317
-0.30  2800  5.50 -3.68272  2.24496  2.10360  2.43354  2.07241
-0.30  3000 -1.02 -5.47382  0.36024  1.40529  2.75583  2.84762
-0.30  3000 -0.70 -5.47382  0.36024  1.40529  2.75583  2.84762
-0.30  3000 -0.51 -5.28739  0.36026  1.40347  2.69997  2.78041
-0.30  3000 -0.29 -5.07153  0.36028  1.40136  2.63529  2.70259
-0.30  3000  0.00 -4.80249  0.36034  1.39931  2.54921  2.61283
-0.30  3000  0.28 -4.54273  0.36040  1.39732  2.46610  2.52617
-0.30  3000  0.50 -4.35102  0.36039  1.39610  2.39962  2.46743
-0.30  3000  0.60 -4.26388  0.36039  1.39555  2.36940  2.44073
-0.30  3000  0.87 -4.15648  0.40222  1.38471  2.33666  2.39391
-0.30  3000  1.00 -4.10477  0.42236  1.37950  2.32090  2.37137
-0.30  3000  1.50 -3.90588  0.49983  1.35943  2.26027  2.28466
-0.30  3000  2.00 -3.70700  0.57730  1.33937  2.19964  2.19795
-0.30  3000  2.50 -3.50811  0.65477  1.31931  2.13902  2.11125
-0.30  3000  3.00 -3.30922  0.73224  1.29925  2.07839  2.02454
-0.30  3000  3.50 -3.11034  0.80971  1.27919  2.01776  1.93783
-0.30  3000  4.00 -2.90552  0.93787  1.48985  1.97737  1.83904
-0.30  3000  4.50 -2.83379  1.12392  1.63701  1.96749  1.80477
-0.30  3000  5.00 -2.85733  1.42376  1.74949  1.98494  1.82639
-0.30  3000  5.50 -2.91549  1.80241  1.89095  2.04140  1.83255
-0.30  3200 -1.02 -4.87297  0.92479  1.40658  2.68163  2.58168
-0.30  3200 -0.70 -4.87297  0.92479  1.40658  2.68163  2.58168
-0.30  3200 -0.51 -4.65718  0.92484  1.40395  2.53605  2.53273
-0.30  3200 -0.29 -4.40743  0.92479  1.40102  2.37116  2.47323
-0.30  3200  0.00 -4.10542  0.92479  1.39755  2.18036  2.39733
-0.30  3200  0.28 -3.81383  0.92478  1.39420  1.99615  2.32405
-0.30  3200  0.50 -3.63182  0.92478  1.39169  1.88646  2.27898
-0.30  3200  0.60 -3.54908  0.92478  1.39055  1.83661  2.25850
-0.30  3200  0.87 -3.47284  0.92510  1.36337  1.83084  2.21901
-0.30  3200  1.00 -3.43612  0.92526  1.35029  1.82806  2.19999
-0.30  3200  1.50 -3.29493  0.92586  1.29996  1.81738  2.12687
-0.30  3200  2.00 -3.15373  0.92647  1.24963  1.80670  2.05374
-0.30  3200  2.50 -3.01254  0.92707  1.19930  1.79602  1.98061
-0.30  3200  3.00 -2.87134  0.92767  1.14897  1.78533  1.90748
-0.30  3200  3.50 -2.73015  0.92828  1.09865  1.77465  1.83435
-0.30  3200  4.00 -2.41408  0.97196  1.32609  1.71272  1.68227
-0.30  3200  4.50 -2.27806  1.07727  1.47048  1.68868  1.61161
-0.30  3200  5.00 -2.23010  1.24630  1.58851  1.68878  1.58464
-0.30  3200  5.50 -2.23763  1.49731  1.70081  1.71259  1.58287
-0.30  3350 -1.02 -3.70715  1.27649  1.52669  2.17898  2.15013
-0.30  3350 -0.70 -3.70715  1.27649  1.52669  2.17898  2.15013
-0.30  3350 -0.51 -3.70715  1.27649  1.52669  2.17898  2.15013
-0.30  3350 -0.29 -3.53199  1.27643  1.51682  2.05070  2.10716
-0.30  3350  0.00 -3.33223  1.27647  1.50696  1.90831  2.05377
-0.30  3350  0.28 -3.13937  1.27651  1.49743  1.77084  2.00222
-0.30  3350  0.50 -3.12220  1.27648  1.49498  1.74974  2.00430
-0.30  3350  0.60 -3.11440  1.27647  1.49387  1.74015  2.00524
-0.30  3350  0.87 -3.09301  1.27651  1.49104  1.71444  2.00711
-0.30  3350  1.00 -3.05528  1.26360  1.46738  1.70672  1.99096
-0.30  3350  1.50 -2.91017  1.21392  1.37640  1.67702  1.92887
-0.30  3350  2.00 -2.76507  1.16425  1.28542  1.64733  1.86677
-0.30  3350  2.50 -2.61996  1.11457  1.19444  1.61763  1.80468
-0.30  3350  3.00 -2.47486  1.06490  1.10345  1.58794  1.74259
-0.30  3350  3.50 -2.32975  1.01523  1.01247  1.55824  1.68049
-0.30  3350  4.00 -2.07416  0.99846  1.25525  1.53782  1.53650
-0.30  3350  4.50 -1.88421  1.03459  1.39876  1.50676  1.43530
-0.30  3350  5.00 -1.80441  1.13343  1.50725  1.49931  1.38897
-0.30  3350  5.50 -1.78413  1.29917  1.60987  1.51129  1.37300
-0.30  3500 -1.02 -2.42775  1.88359  1.77156  1.56979  1.77381
-0.30  3500 -0.70 -2.42775  1.88359  1.77156  1.56979  1.77381
-0.30  3500 -0.51 -2.42775  1.88359  1.77156  1.56979  1.77381
-0.30  3500 -0.29 -2.42775  1.88359  1.77156  1.56979  1.77381
-0.30  3500  0.00 -2.42775  1.88359  1.77156  1.56979  1.77381
-0.30  3500  0.28 -2.43414  1.76440  1.70145  1.54976  1.78896
-0.30  3500  0.50 -2.43916  1.67076  1.64637  1.53403  1.80086
-0.30  3500  0.60 -2.44477  1.62737  1.62194  1.52936  1.80697
-0.30  3500  0.87 -2.45990  1.51022  1.55601  1.51675  1.82346
-0.30  3500  1.00 -2.46718  1.45381  1.52426  1.51068  1.83140
-0.30  3500  1.50 -2.50553  1.24714  1.40690  1.49750  1.86228
-0.30  3500  2.00 -2.54910  1.06512  1.29820  1.49249  1.89107
-0.30  3500  2.50 -2.59060  0.92040  1.20629  1.49472  1.91447
-0.30  3500  3.00 -1.81318  1.18365  1.34959  1.26151  1.45418
-0.30  3500  3.50 -1.65651  1.12467  1.38840  1.26877  1.38174
-0.30  3500  4.00 -1.48929  1.08922  1.43312  1.28129  1.29883
-0.30  3500  4.50 -1.41800  1.08681  1.46480  1.30089  1.25386
-0.30  3500  5.00 -1.39016  1.12450  1.50328  1.32803  1.22318
-0.30  3500  5.50 -1.39016  1.12450  1.50328  1.32803  1.22318
-0.30  3750 -1.02 -1.36398  2.30326  1.76094  1.31285  1.08092
-0.30  3750 -0.70 -1.36398  2.30326  1.76094  1.31285  1.08092
-0.30  3750 -0.51 -1.36398  2.30326  1.76094  1.31285  1.08092
-0.30  3750 -0.29 -1.36398  2.30326  1.76094  1.31285  1.08092
-0.30  3750  0.00 -1.36398  2.30326  1.76094  1.31285  1.08092
-0.30  3750  0.28 -1.35620  2.18002  1.71310  1.29809  1.08262
-0.30  3750  0.50 -1.35009  2.08318  1.67552  1.28649  1.08395
-0.30  3750  0.60 -1.34973  2.04079  1.65988  1.28227  1.08609
-0.30  3750  0.87 -1.34878  1.92633  1.61765  1.27087  1.09184
-0.30  3750  1.00 -1.34832  1.87122  1.59732  1.26538  1.09461
-0.30  3750  1.50 -1.35377  1.67709  1.52372  1.24673  1.11042
-0.30  3750  2.00 -1.36668  1.50651  1.45590  1.23111  1.13033
-0.30  3750  2.50 -1.38435  1.36371  1.39491  1.21930  1.15145
-0.30  3750  3.00 -1.20872  1.49917  1.43666  1.10665  1.08403
-0.30  3750  3.50 -1.21269  1.40565  1.41288  1.11225  1.09133
-0.30  3750  4.00 -1.15455  1.30916  1.42289  1.13532  1.06323
-0.30  3750  4.50 -1.08230  1.23087  1.43023  1.15603  1.02768
-0.30  3750  5.00 -1.04779  1.19235  1.43193  1.17145  1.00799
-0.30  3750  5.50 -1.04779  1.19235  1.43193  1.17145  1.00799
-0.30  4000 -1.02 -0.81711  2.24883  1.62711  1.05713  0.80762
-0.30  4000 -0.70 -0.81711  2.24883  1.62711  1.05713  0.80762
-0.30  4000 -0.51 -0.81711  2.24883  1.62711  1.05713  0.80762
-0.30  4000 -0.29 -0.81711  2.24883  1.62711  1.05713  0.80762
-0.30  4000  0.00 -0.81711  2.24883  1.62711  1.05713  0.80762
-0.30  4000  0.28 -0.80948  2.13357  1.58615  1.04999  0.80214
-0.30  4000  0.50 -0.80348  2.04300  1.55397  1.04439  0.79784
-0.30  4000  0.60 -0.80254  2.00487  1.54119  1.04320  0.79685
-0.30  4000  0.87 -0.79998  1.90191  1.50670  1.04001  0.79416
-0.30  4000  1.00 -0.79875  1.85234  1.49009  1.03847  0.79287
-0.30  4000  1.50 -0.79964  1.68481  1.43465  1.03559  0.79193
-0.30  4000  2.00 -0.80641  1.54390  1.38810  1.03535  0.79514
-0.30  4000  2.50 -0.81585  1.43067  1.34834  1.03518  0.80165
-0.30  4000  3.00 -0.74611  1.51124  1.38344  1.02624  0.82268
-0.30  4000  3.50 -0.76105  1.44359  1.36199  1.02891  0.83365
-0.30  4000  4.00 -0.76942  1.37927  1.35399  1.03899  0.84140
-0.30  4000  4.50 -0.75066  1.28899  1.35842  1.05812  0.83639
-0.30  4000  5.00 -0.71849  1.20205  1.35438  1.07079  0.82647
-0.30  4000  5.50 -0.71849  1.20205  1.35438  1.07079  0.82647
-0.30  4250 -1.02 -0.50486  1.97392  1.50808  0.92569  0.66519
-0.30  4250 -0.70 -0.50486  1.97392  1.50808  0.92569  0.66519
-0.30  4250 -0.51 -0.50486  1.97392  1.50808  0.92569  0.66519
-0.30  4250 -0.29 -0.50486  1.97392  1.50808  0.92569  0.66519
-0.30  4250  0.00 -0.50486  1.97392  1.50808  0.92569  0.66519
-0.30  4250  0.28 -0.49749  1.88045  1.47092  0.91762  0.65995
-0.30  4250  0.50 -0.49170  1.80702  1.44172  0.91129  0.65583
-0.30  4250  0.60 -0.49077  1.77542  1.43009  0.91002  0.65453
-0.30  4250  0.87 -0.48826  1.69012  1.39871  0.90660  0.65102
-0.30  4250  1.00 -0.48706  1.64905  1.38359  0.90495  0.64933
-0.30  4250  1.50 -0.48764  1.50816  1.33228  0.90467  0.64477
-0.30  4250  2.00 -0.49300  1.39226  1.28864  0.90887  0.64211
-0.30  4250  2.50 -0.50182  1.30467  1.25352  0.91626  0.64122
-0.30  4250  3.00 -0.48045  1.30400  1.27958  0.86540  0.68609
-0.30  4250  3.50 -0.49404  1.26690  1.26321  0.87686  0.68796
-0.30  4250  4.00 -0.50959  1.24689  1.25639  0.89032  0.69136
-0.30  4250  4.50 -0.52164  1.22208  1.25910  0.90622  0.69656
-0.30  4250  5.00 -0.51770  1.15826  1.26315  0.92147  0.70095
-0.30  4250  5.50 -0.51770  1.15826  1.26315  0.92147  0.70095
-0.30  4500 -1.02 -0.31546  1.67857  1.43260  0.77413  0.61807
-0.30  4500 -0.70 -0.31546  1.67857  1.43260  0.77413  0.61807
-0.30  4500 -0.51 -0.31546  1.67857  1.43260  0.77413  0.61807
-0.30  4500 -0.29 -0.31546  1.67857  1.43260  0.77413  0.61807
-0.30  4500  0.00 -0.31546  1.67857  1.43260  0.77413  0.61807
-0.30  4500  0.28 -0.30914  1.61088  1.40232  0.76447  0.61412
-0.30  4500  0.50 -0.30418  1.55770  1.37853  0.75687  0.61102
-0.30  4500  0.60 -0.30317  1.53405  1.36887  0.75482  0.61006
-0.30  4500  0.87 -0.30044  1.47018  1.34279  0.74929  0.60746
-0.30  4500  1.00 -0.29913  1.43943  1.33024  0.74663  0.60621
-0.30  4500  1.50 -0.29755  1.33027  1.28676  0.74169  0.60249
-0.30  4500  2.00 -0.30069  1.23677  1.24817  0.74194  0.59996
-0.30  4500  2.50 -0.30678  1.16343  1.21420  0.74604  0.59806
-0.30  4500  3.00 -0.31611  1.11301  1.18680  0.75444  0.59689
-0.30  4500  3.50 -0.32752  1.08681  1.16767  0.76603  0.59615
-0.30  4500  4.00 -0.34123  1.08078  1.15756  0.78101  0.59591
-0.30  4500  4.50 -0.35643  1.09000  1.15606  0.79832  0.59629
-0.30  4500  5.00 -0.36921  1.09634  1.16328  0.81667  0.59878
-0.30  4500  5.50 -0.36921  1.09634  1.16328  0.81667  0.59878
-0.30  4750 -1.02 -0.14345  1.33561  1.26815  0.68048  0.52180
-0.30  4750 -0.70 -0.14345  1.33561  1.26815  0.68048  0.52180
-0.30  4750 -0.51 -0.14345  1.33561  1.26815  0.68048  0.52180
-0.30  4750 -0.29 -0.14345  1.33561  1.26815  0.68048  0.52180
-0.30  4750  0.00 -0.14345  1.33561  1.26815  0.68048  0.52180
-0.30  4750  0.28 -0.13935  1.28354  1.24279  0.67061  0.51846
-0.30  4750  0.50 -0.13613  1.24263  1.22286  0.66285  0.51583
-0.30  4750  0.60 -0.13549  1.22415  1.21508  0.66067  0.51514
-0.30  4750  0.87 -0.13377  1.17426  1.19406  0.65481  0.51328
-0.30  4750  1.00 -0.13294  1.15024  1.18395  0.65198  0.51238
-0.30  4750  1.50 -0.13244  1.06993  1.14969  0.64445  0.51005
-0.30  4750  2.00 -0.13397  1.00112  1.11907  0.64043  0.50842
-0.30  4750  2.50 -0.13781  0.94543  1.09133  0.63980  0.50730
-0.30  4750  3.00 -0.14400  0.90653  1.06701  0.64288  0.50652
-0.30  4750  3.50 -0.15229  0.88525  1.04681  0.64987  0.50562
-0.30  4750  4.00 -0.16262  0.88268  1.03246  0.66081  0.50499
-0.30  4750  4.50 -0.17474  0.89653  1.02533  0.67525  0.50431
-0.30  4750  5.00 -0.18784  0.92062  1.02578  0.69231  0.50396
-0.30  4750  5.50 -0.18784  0.92062  1.02578  0.69231  0.50396
-0.30  5000 -1.02 -0.02696  1.00241  1.09667  0.62653  0.46835
-0.30  5000 -0.70 -0.02696  1.00241  1.09667  0.62653  0.46835
-0.30  5000 -0.51 -0.02696  1.00241  1.09667  0.62653  0.46835
-0.30  5000 -0.29 -0.02696  1.00241  1.09667  0.62653  0.46835
-0.30  5000  0.00 -0.02696  1.00241  1.09667  0.62653  0.46835
-0.30  5000  0.28 -0.02570  0.95702  1.07660  0.61905  0.46535
-0.30  5000  0.50 -0.02471  0.92136  1.06084  0.61318  0.46300
-0.30  5000  0.60 -0.02474  0.90596  1.05446  0.61122  0.46235
-0.30  5000  0.87 -0.02484  0.86436  1.03721  0.60593  0.46059
-0.30  5000  1.00 -0.02489  0.84434  1.02891  0.60338  0.45974
-0.30  5000  1.50 -0.02601  0.78072  1.00252  0.59556  0.45749
-0.30  5000  2.00 -0.02881  0.72521  0.97961  0.59119  0.45661
-0.30  5000  2.50 -0.03195  0.68347  0.95989  0.58832  0.45580
-0.30  5000  3.00 -0.03629  0.65445  0.94206  0.58799  0.45529
-0.30  5000  3.50 -0.04144  0.63957  0.92633  0.59009  0.45467
-0.30  5000  4.00 -0.04793  0.64003  0.91447  0.59569  0.45408
-0.30  5000  4.50 -0.05583  0.65505  0.90637  0.60477  0.45331
-0.30  5000  5.00 -0.06531  0.68078  0.90268  0.61757  0.45267
-0.30  5000  5.50 -0.06531  0.68078  0.90268  0.61757  0.45267
-0.30  5250 -1.02  0.04084  0.74138  0.93305  0.58009  0.43356
-0.30  5250 -0.70  0.04084  0.74138  0.93305  0.58009  0.43356
-0.30  5250 -0.51  0.04084  0.74138  0.93305  0.58009  0.43356
-0.30  5250 -0.29  0.04084  0.74138  0.93305  0.58009  0.43356
-0.30  5250  0.00  0.04084  0.74138  0.93305  0.58009  0.43356
-0.30  5250  0.28  0.03905  0.70467  0.91877  0.57553  0.43113
-0.30  5250  0.50  0.03764  0.67583  0.90755  0.57194  0.42921
-0.30  5250  0.60  0.03684  0.66251  0.90274  0.57069  0.42868
-0.30  5250  0.87  0.03466  0.62652  0.88974  0.56732  0.42723
-0.30  5250  1.00  0.03361  0.60920  0.88349  0.56570  0.42653
-0.30  5250  1.50  0.02944  0.55304  0.86344  0.56050  0.42446
-0.30  5250  2.00  0.02458  0.50233  0.84617  0.55732  0.42362
-0.30  5250  2.50  0.02040  0.46491  0.83270  0.55469  0.42301
-0.30  5250  3.00  0.01629  0.43875  0.82168  0.55343  0.42278
-0.30  5250  3.50  0.01224  0.42599  0.81260  0.55355  0.42253
-0.30  5250  4.00  0.00820  0.42726  0.80580  0.55567  0.42193
-0.30  5250  4.50  0.00378  0.44141  0.80091  0.56037  0.42132
-0.30  5250  5.00 -0.00135  0.46719  0.79926  0.56812  0.42059
-0.30  5250  5.50 -0.00135  0.46719  0.79926  0.56812  0.42059
-0.30  5500 -1.02  0.11573  0.57799  0.76998  0.51277  0.39728
-0.30  5500 -0.70  0.11573  0.57799  0.76998  0.51277  0.39728
-0.30  5500 -0.51  0.11573  0.57799  0.76998  0.51277  0.39728
-0.30  5500 -0.29  0.11573  0.57799  0.76998  0.51277  0.39728
-0.30  5500  0.00  0.11573  0.57799  0.76998  0.51277  0.39728
-0.30  5500  0.28  0.10903  0.53782  0.77021  0.51665  0.39615
-0.30  5500  0.50  0.10376  0.50625  0.77039  0.51971  0.39525
-0.30  5500  0.60  0.10227  0.49494  0.76724  0.51925  0.39483
-0.30  5500  0.87  0.09827  0.46439  0.75875  0.51800  0.39367
-0.30  5500  1.00  0.09634  0.44969  0.75466  0.51740  0.39311
-0.30  5500  1.50  0.08934  0.39674  0.74237  0.51588  0.39179
-0.30  5500  2.00  0.08262  0.35115  0.73153  0.51418  0.39100
-0.30  5500  2.50  0.07633  0.31306  0.72356  0.51357  0.39080
-0.30  5500  3.00  0.07092  0.28436  0.71731  0.51304  0.39078
-0.30  5500  3.50  0.06645  0.26783  0.71319  0.51297  0.39046
-0.30  5500  4.00  0.06314  0.26474  0.71068  0.51371  0.39018
-0.30  5500  4.50  0.06058  0.27370  0.71048  0.51616  0.38978
-0.30  5500  5.00  0.05838  0.29569  0.71259  0.52068  0.38919
-0.30  5500  5.50  0.05838  0.29569  0.71259  0.52068  0.38919
-0.30  5750 -1.02  0.18889  0.47545  0.62420  0.42734  0.35408
-0.30  5750 -0.70  0.18889  0.47545  0.62420  0.42734  0.35408
-0.30  5750 -0.51  0.18889  0.47545  0.62420  0.42734  0.35408
-0.30  5750 -0.29  0.18889  0.47545  0.62420  0.42734  0.35408
-0.30  5750  0.00  0.18889  0.47545  0.62420  0.42734  0.35408
-0.30  5750  0.28  0.17945  0.43634  0.62475  0.43409  0.35374
-0.30  5750  0.50  0.17204  0.40562  0.62518  0.43939  0.35348
-0.30  5750  0.60  0.16895  0.39090  0.62801  0.44232  0.35346
-0.30  5750  0.87  0.16061  0.35116  0.63563  0.45024  0.35340
-0.30  5750  1.00  0.15660  0.33203  0.63930  0.45405  0.35338
-0.30  5750  1.50  0.14688  0.28309  0.63229  0.45512  0.35226
-0.30  5750  2.00  0.13759  0.23619  0.62807  0.45682  0.35212
-0.30  5750  2.50  0.12937  0.19646  0.62559  0.45797  0.35218
-0.30  5750  3.00  0.12226  0.16350  0.62431  0.45859  0.35250
-0.30  5750  3.50  0.11639  0.14146  0.62376  0.45934  0.35275
-0.30  5750  4.00  0.11210  0.12955  0.62452  0.46030  0.35290
-0.30  5750  4.50  0.10949  0.12984  0.62740  0.46200  0.35265
-0.30  5750  5.00  0.10830  0.14319  0.63238  0.46503  0.35247
-0.30  5750  5.50  0.10830  0.14319  0.63238  0.46503  0.35247
-0.30  6000 -1.02  0.23554  0.42583  0.50198  0.35269  0.31688
-0.30  6000 -0.70  0.23554  0.42583  0.50198  0.35269  0.31688
-0.30  6000 -0.51  0.23554  0.42583  0.50198  0.35269  0.31688
-0.30  6000 -0.29  0.23554  0.42583  0.50198  0.35269  0.31688
-0.30  6000  0.00  0.23554  0.42583  0.50198  0.35269  0.31688
-0.30  6000  0.28  0.22793  0.40247  0.50784  0.35926  0.31551
-0.30  6000  0.50  0.22195  0.38411  0.51245  0.36443  0.31443
-0.30  6000  0.60  0.21868  0.37216  0.51342  0.36700  0.31439
-0.30  6000  0.87  0.20984  0.33989  0.51602  0.37392  0.31427
-0.30  6000  1.00  0.20559  0.32435  0.51728  0.37726  0.31421
-0.30  6000  1.50  0.18644  0.24186  0.54378  0.39900  0.31634
-0.30  6000  2.00  0.17570  0.19738  0.54391  0.40273  0.31596
-0.30  6000  2.50  0.16553  0.15464  0.54552  0.40578  0.31637
-0.30  6000  3.00  0.15632  0.11696  0.54852  0.40858  0.31716
-0.30  6000  3.50  0.14874  0.08692  0.55224  0.41066  0.31795
-0.30  6000  4.00  0.14287  0.06596  0.55619  0.41224  0.31864
-0.30  6000  4.50  0.13893  0.05589  0.56122  0.41425  0.31914
-0.30  6000  5.00  0.13728  0.05774  0.56801  0.41669  0.31945
-0.30  6000  5.50  0.13728  0.05774  0.56801  0.41669  0.31945
-0.30  6250 -1.02  0.27256  0.37480  0.40930  0.29820  0.27400
-0.30  6250 -0.70  0.27256  0.37480  0.40930  0.29820  0.27400
-0.30  6250 -0.51  0.27256  0.37480  0.40930  0.29820  0.27400
-0.30  6250 -0.29  0.27256  0.37480  0.40930  0.29820  0.27400
-0.30  6250  0.00  0.27256  0.37480  0.40930  0.29820  0.27400
-0.30  6250  0.28  0.27256  0.37480  0.40930  0.29820  0.27400
-0.30  6250  0.50  0.27256  0.37480  0.40930  0.29820  0.27400
-0.30  6250  0.60  0.26995  0.36632  0.41264  0.30100  0.27363
-0.30  6250  0.87  0.26291  0.34341  0.42167  0.30856  0.27262
-0.30  6250  1.00  0.25952  0.33238  0.42601  0.31219  0.27214
-0.30  6250  1.50  0.24147  0.26742  0.44211  0.33013  0.27347
-0.30  6250  2.00  0.22247  0.19122  0.47062  0.35160  0.27610
-0.30  6250  2.50  0.21064  0.14777  0.47608  0.35738  0.27634
-0.30  6250  3.00  0.19963  0.10596  0.48257  0.36199  0.27741
-0.30  6250  3.50  0.18995  0.06865  0.48967  0.36585  0.27865
-0.30  6250  4.00  0.18191  0.03850  0.49655  0.36889  0.28004
-0.30  6250  4.50  0.17628  0.01806  0.50371  0.37147  0.28118
-0.30  6250  5.00  0.17289  0.00720  0.51168  0.37426  0.28221
-0.30  6250  5.50  0.17289  0.00720  0.51168  0.37426  0.28221
-0.30  6500 -1.02  0.30928  0.34905  0.31063  0.23727  0.23850
-0.30  6500 -0.70  0.30928  0.34905  0.31063  0.23727  0.23850
-0.30  6500 -0.51  0.30928  0.34905  0.31063  0.23727  0.23850
-0.30  6500 -0.29  0.30928  0.34905  0.31063  0.23727  0.23850
-0.30  6500  0.00  0.30928  0.34905  0.31063  0.23727  0.23850
-0.30  6500  0.28  0.30928  0.34905  0.31063  0.23727  0.23850
-0.30  6500  0.50  0.30928  0.34905  0.31063  0.23727  0.23850
-0.30  6500  0.60  0.30745  0.34576  0.31529  0.24019  0.23771
-0.30  6500  0.87  0.30250  0.33686  0.32788  0.24807  0.23558
-0.30  6500  1.00  0.30012  0.33257  0.33394  0.25186  0.23456
-0.30  6500  1.50  0.28663  0.28522  0.35822  0.26847  0.23357
-0.30  6500  2.00  0.26699  0.21712  0.38089  0.28888  0.23648
-0.30  6500  2.50  0.24845  0.14567  0.41114  0.30950  0.23926
-0.30  6500  3.00  0.23540  0.10081  0.42110  0.31688  0.24037
-0.30  6500  3.50  0.22316  0.05783  0.43050  0.32264  0.24211
-0.30  6500  4.00  0.21255  0.01954  0.43966  0.32738  0.24419
-0.30  6500  4.50  0.20422 -0.01136  0.44847  0.33121  0.24623
-0.30  6500  5.00  0.19838 -0.03453  0.45767  0.33475  0.24814
-0.30  6500  5.50  0.19838 -0.03453  0.45767  0.33475  0.24814
-0.30  6750 -1.02  0.33384  0.32293  0.23311  0.18676  0.21207
-0.30  6750 -0.70  0.33384  0.32293  0.23311  0.18676  0.21207
-0.30  6750 -0.51  0.33384  0.32293  0.23311  0.18676  0.21207
-0.30  6750 -0.29  0.33384  0.32293  0.23311  0.18676  0.21207
-0.30  6750  0.00  0.33384  0.32293  0.23311  0.18676  0.21207
-0.30  6750  0.28  0.33384  0.32293  0.23311  0.18676  0.21207
-0.30  6750  0.50  0.33384  0.32293  0.23311  0.18676  0.21207
-0.30  6750  0.60  0.33230  0.32472  0.23608  0.18934  0.21118
-0.30  6750  0.87  0.32814  0.32954  0.24410  0.19630  0.20877
-0.30  6750  1.00  0.32614  0.33187  0.24797  0.19965  0.20762
-0.30  6750  1.50  0.31728  0.30981  0.27753  0.21585  0.20399
-0.30  6750  2.00  0.30321  0.25904  0.30720  0.23374  0.20399
-0.30  6750  2.50  0.28223  0.18752  0.33541  0.25603  0.20830
-0.30  6750  3.00  0.26363  0.11906  0.36540  0.27584  0.21145
-0.30  6750  3.50  0.24947  0.07178  0.37830  0.28401  0.21338
-0.30  6750  4.00  0.23617  0.02699  0.38936  0.29059  0.21601
-0.30  6750  4.50  0.22436 -0.01415  0.39937  0.29623  0.21904
-0.30  6750  5.00  0.21518 -0.04903  0.40939  0.30116  0.22189
-0.30  6750  5.50  0.21518 -0.04903  0.40939  0.30116  0.22189
-0.30  7000 -1.02  0.34307  0.27019  0.16408  0.13686  0.18671
-0.30  7000 -0.70  0.34307  0.27019  0.16408  0.13686  0.18671
-0.30  7000 -0.51  0.34307  0.27019  0.16408  0.13686  0.18671
-0.30  7000 -0.29  0.34307  0.27019  0.16408  0.13686  0.18671
-0.30  7000  0.00  0.34307  0.27019  0.16408  0.13686  0.18671
-0.30  7000  0.28  0.34307  0.27019  0.16408  0.13686  0.18671
-0.30  7000  0.50  0.34307  0.27019  0.16408  0.13686  0.18671
-0.30  7000  0.60  0.34195  0.27575  0.16613  0.13904  0.18585
-0.30  7000  0.87  0.33893  0.29075  0.17166  0.14492  0.18353
-0.30  7000  1.00  0.33748  0.29797  0.17433  0.14775  0.18241
-0.30  7000  1.50  0.33082  0.30368  0.19472  0.16057  0.17717
-0.30  7000  2.00  0.32104  0.27627  0.22961  0.17812  0.17427
-0.30  7000  2.50  0.30486  0.21817  0.26369  0.19763  0.17594
-0.30  7000  3.00  0.28166  0.13919  0.29788  0.22266  0.18167
-0.30  7000  3.50  0.26389  0.07767  0.32408  0.23878  0.18470
-0.30  7000  4.00  0.24790  0.02760  0.33838  0.24780  0.18787
-0.30  7000  4.50  0.23304 -0.02067  0.35074  0.25535  0.19149
-0.30  7000  5.00  0.22046 -0.06441  0.36186  0.26177  0.19522
-0.30  7000  5.50  0.22046 -0.06441  0.36186  0.26177  0.19522
-0.30  7250 -1.02  0.35106  0.24981  0.12473  0.09441  0.16200
-0.30  7250 -0.70  0.35106  0.24981  0.12473  0.09441  0.16200
-0.30  7250 -0.51  0.35106  0.24981  0.12473  0.09441  0.16200
-0.30  7250 -0.29  0.35106  0.24981  0.12473  0.09441  0.16200
-0.30  7250  0.00  0.35106  0.24981  0.12473  0.09441  0.16200
-0.30  7250  0.28  0.35106  0.24981  0.12473  0.09441  0.16200
-0.30  7250  0.50  0.35106  0.24981  0.12473  0.09441  0.16200
-0.30  7250  0.60  0.35013  0.25901  0.12388  0.09567  0.16138
-0.30  7250  0.87  0.34764  0.28384  0.12158  0.09907  0.15972
-0.30  7250  1.00  0.34643  0.29579  0.12047  0.10070  0.15892
-0.30  7250  1.50  0.34256  0.31945  0.13774  0.11132  0.15320
-0.30  7250  2.00  0.33586  0.32005  0.16269  0.12449  0.14839
-0.30  7250  2.50  0.32487  0.28525  0.20156  0.14340  0.14677
-0.30  7250  3.00  0.30705  0.21977  0.23895  0.16457  0.15012
-0.30  7250  3.50  0.28396  0.13884  0.27491  0.18992  0.15688
-0.30  7250  4.00  0.26624  0.07943  0.29785  0.20420  0.16059
-0.30  7250  4.50  0.24965  0.02711  0.31310  0.21365  0.16467
-0.30  7250  5.00  0.23441 -0.02346  0.32603  0.22185  0.16924
-0.30  7250  5.50  0.23441 -0.02346  0.32603  0.22185  0.16924
-0.30  7500 -1.02  0.34353  0.19535  0.12433  0.08855  0.15021
-0.30  7500 -0.70  0.34353  0.19535  0.12433  0.08855  0.15021
-0.30  7500 -0.51  0.34353  0.19535  0.12433  0.08855  0.15021
-0.30  7500 -0.29  0.34353  0.19535  0.12433  0.08855  0.15021
-0.30  7500  0.00  0.34353  0.19535  0.12433  0.08855  0.15021
-0.30  7500  0.28  0.34353  0.19535  0.12433  0.08855  0.15021
-0.30  7500  0.50  0.34353  0.19535  0.12433  0.08855  0.15021
-0.30  7500  0.60  0.34264  0.20947  0.11519  0.08680  0.15029
-0.30  7500  0.87  0.34024  0.24760  0.09049  0.08208  0.15049
-0.30  7500  1.00  0.33908  0.26596  0.07860  0.07981  0.15059
-0.30  7500  1.50  0.33764  0.30622  0.08929  0.08738  0.14492
-0.30  7500  2.00  0.33428  0.32560  0.11174  0.09791  0.13880
-0.30  7500  2.50  0.32767  0.32283  0.14130  0.11108  0.13436
-0.30  7500  3.00  0.31538  0.27930  0.18375  0.13130  0.13426
-0.30  7500  3.50  0.29519  0.20399  0.22269  0.15471  0.13994
-0.30  7500  4.00  0.27260  0.12259  0.25723  0.17858  0.14699
-0.30  7500  4.50  0.25540  0.06680  0.27750  0.19061  0.15099
-0.30  7500  5.00  0.23803  0.01065  0.29301  0.20069  0.15636
-0.30  7500  5.50  0.23803  0.01065  0.29301  0.20069  0.15636
-0.30  7750 -1.02  0.31024  0.18386  0.04459  0.07582  0.14856
-0.30  7750 -0.70  0.31024  0.18386  0.04459  0.07582  0.14856
-0.30  7750 -0.51  0.31024  0.18386  0.04459  0.07582  0.14856
-0.30  7750 -0.29  0.31024  0.18386  0.04459  0.07582  0.14856
-0.30  7750  0.00  0.31024  0.18386  0.04459  0.07582  0.14856
-0.30  7750  0.28  0.31024  0.18386  0.04459  0.07582  0.14856
-0.30  7750  0.50  0.31024  0.18386  0.04459  0.07582  0.14856
-0.30  7750  0.60  0.31024  0.18386  0.04459  0.07582  0.14856
-0.30  7750  0.87  0.31024  0.18386  0.04459  0.07582  0.14856
-0.30  7750  1.00  0.31024  0.18386  0.04459  0.07582  0.14856
-0.30  7750  1.50  0.31119  0.24033  0.04071  0.07737  0.14332
-0.30  7750  2.00  0.31060  0.27477  0.06019  0.08555  0.13652
-0.30  7750  2.50  0.30631  0.28723  0.08741  0.09654  0.13108
-0.30  7750  3.00  0.29798  0.27581  0.12066  0.11044  0.12779
-0.30  7750  3.50  0.28322  0.22065  0.16530  0.13174  0.12936
-0.30  7750  4.00  0.26032  0.13556  0.20474  0.15640  0.13683
-0.30  7750  4.50  0.23922  0.06195  0.23492  0.17585  0.14302
-0.30  7750  5.00  0.21974  0.00049  0.25438  0.18833  0.14884
-0.30  7750  5.50  0.21974  0.00049  0.25438  0.18833  0.14884
-0.30  8000 -1.02  0.27535  0.08767  0.03232  0.06843  0.13416
-0.30  8000 -0.70  0.27535  0.08767  0.03232  0.06843  0.13416
-0.30  8000 -0.51  0.27535  0.08767  0.03232  0.06843  0.13416
-0.30  8000 -0.29  0.27535  0.08767  0.03232  0.06843  0.13416
-0.30  8000  0.00  0.27535  0.08767  0.03232  0.06843  0.13416
-0.30  8000  0.28  0.27535  0.08767  0.03232  0.06843  0.13416
-0.30  8000  0.50  0.27535  0.08767  0.03232  0.06843  0.13416
-0.30  8000  0.60  0.27535  0.08767  0.03232  0.06843  0.13416
-0.30  8000  0.87  0.27535  0.08767  0.03232  0.06843  0.13416
-0.30  8000  1.00  0.27535  0.08767  0.03232  0.06843  0.13416
-0.30  8000  1.50  0.28249  0.16526  0.00333  0.05701  0.13000
-0.30  8000  2.00  0.28481  0.21275  0.01513  0.06115  0.12292
-0.30  8000  2.50  0.28365  0.23956  0.04025  0.06969  0.11638
-0.30  8000  3.00  0.27822  0.24564  0.07123  0.08071  0.11140
-0.30  8000  3.50  0.26854  0.22503  0.10905  0.09523  0.10899
-0.30  8000  4.00  0.25082  0.15544  0.15461  0.11791  0.11307
-0.30  8000  4.50  0.22642  0.06416  0.19272  0.14305  0.12177
-0.30  8000  5.00  0.20520 -0.00580  0.21829  0.15882  0.12835
-0.30  8000  5.50  0.20520 -0.00580  0.21829  0.15882  0.12835
-0.30  8250 -1.02  0.21756 -0.02959  0.03372  0.07303  0.11424
-0.30  8250 -0.70  0.21756 -0.02959  0.03372  0.07303  0.11424
-0.30  8250 -0.51  0.21756 -0.02959  0.03372  0.07303  0.11424
-0.30  8250 -0.29  0.21756 -0.02959  0.03372  0.07303  0.11424
-0.30  8250  0.00  0.21756 -0.02959  0.03372  0.07303  0.11424
-0.30  8250  0.28  0.21756 -0.02959  0.03372  0.07303  0.11424
-0.30  8250  0.50  0.21756 -0.02959  0.03372  0.07303  0.11424
-0.30  8250  0.60  0.21756 -0.02959  0.03372  0.07303  0.11424
-0.30  8250  0.87  0.21756 -0.02959  0.03372  0.07303  0.11424
-0.30  8250  1.00  0.21756 -0.02959  0.03372  0.07303  0.11424
-0.30  8250  1.50  0.24162  0.07590 -0.02014  0.04165  0.11205
-0.30  8250  2.00  0.24756  0.13512 -0.01998  0.03989  0.10503
-0.30  8250  2.50  0.24956  0.17419  0.00068  0.04520  0.09748
-0.30  8250  3.00  0.24736  0.19329  0.03043  0.05386  0.09126
-0.30  8250  3.50  0.24022  0.18994  0.06439  0.06489  0.08760
-0.30  8250  4.00  0.22828  0.15465  0.10582  0.08087  0.08702
-0.30  8250  4.50  0.20814  0.07147  0.15106  0.10481  0.09317
-0.30  8250  5.00  0.18322 -0.02147  0.18505  0.12771  0.10290
-0.30  8250  5.50  0.18322 -0.02147  0.18505  0.12771  0.10290
-0.30  8500 -1.02  0.19519 -0.00050 -0.02791  0.04743  0.10486
-0.30  8500 -0.70  0.19519 -0.00050 -0.02791  0.04743  0.10486
-0.30  8500 -0.51  0.19519 -0.00050 -0.02791  0.04743  0.10486
-0.30  8500 -0.29  0.19519 -0.00050 -0.02791  0.04743  0.10486
-0.30  8500  0.00  0.19519 -0.00050 -0.02791  0.04743  0.10486
-0.30  8500  0.28  0.19519 -0.00050 -0.02791  0.04743  0.10486
-0.30  8500  0.50  0.19519 -0.00050 -0.02791  0.04743  0.10486
-0.30  8500  0.60  0.19519 -0.00050 -0.02791  0.04743  0.10486
-0.30  8500  0.87  0.19519 -0.00050 -0.02791  0.04743  0.10486
-0.30  8500  1.00  0.19519 -0.00050 -0.02791  0.04743  0.10486
-0.30  8500  1.50  0.19519 -0.00050 -0.02791  0.04743  0.10486
-0.30  8500  2.00  0.20586  0.06956 -0.04049  0.03797  0.09821
-0.30  8500  2.50  0.21043  0.11711 -0.02667  0.03959  0.09057
-0.30  8500  3.00  0.21139  0.14586 -0.00028  0.04569  0.08358
-0.30  8500  3.50  0.20817  0.15515  0.03279  0.05445  0.07837
-0.30  8500  4.00  0.19978  0.14126  0.06970  0.06582  0.07577
-0.30  8500  4.50  0.18687  0.08939  0.11487  0.08348  0.07685
-0.30  8500  5.00  0.16572 -0.00389  0.15680  0.10754  0.08520
-0.30  8500  5.50  0.16572 -0.00389  0.15680  0.10754  0.08520
-0.30  8750 -1.02  0.16042 -0.00441 -0.05086  0.04387  0.09625
-0.30  8750 -0.70  0.16042 -0.00441 -0.05086  0.04387  0.09625
-0.30  8750 -0.51  0.16042 -0.00441 -0.05086  0.04387  0.09625
-0.30  8750 -0.29  0.16042 -0.00441 -0.05086  0.04387  0.09625
-0.30  8750  0.00  0.16042 -0.00441 -0.05086  0.04387  0.09625
-0.30  8750  0.28  0.16042 -0.00441 -0.05086  0.04387  0.09625
-0.30  8750  0.50  0.16042 -0.00441 -0.05086  0.04387  0.09625
-0.30  8750  0.60  0.16042 -0.00441 -0.05086  0.04387  0.09625
-0.30  8750  0.87  0.16042 -0.00441 -0.05086  0.04387  0.09625
-0.30  8750  1.00  0.16042 -0.00441 -0.05086  0.04387  0.09625
-0.30  8750  1.50  0.16042 -0.00441 -0.05086  0.04387  0.09625
-0.30  8750  2.00  0.16239  0.00878 -0.05254  0.04216  0.09498
-0.30  8750  2.50  0.16894  0.06262 -0.04599  0.03984  0.08739
-0.30  8750  3.00  0.17227  0.09995 -0.02390  0.04308  0.07969
-0.30  8750  3.50  0.17245  0.12000  0.00672  0.04926  0.07308
-0.30  8750  4.00  0.16845  0.11874  0.04294  0.05814  0.06884
-0.30  8750  4.50  0.15932  0.09119  0.08282  0.06998  0.06782
-0.30  8750  5.00  0.14600  0.02026  0.12826  0.08975  0.07144
-0.30  8750  5.50  0.14600  0.02026  0.12826  0.08975  0.07144
-0.30  9000 -1.02  0.09311 -0.13845 -0.03458  0.05938  0.08590
-0.30  9000 -0.70  0.09311 -0.13845 -0.03458  0.05938  0.08590
-0.30  9000 -0.51  0.09311 -0.13845 -0.03458  0.05938  0.08590
-0.30  9000 -0.29  0.09311 -0.13845 -0.03458  0.05938  0.08590
-0.30  9000  0.00  0.09311 -0.13845 -0.03458  0.05938  0.08590
-0.30  9000  0.28  0.09311 -0.13845 -0.03458  0.05938  0.08590
-0.30  9000  0.50  0.09311 -0.13845 -0.03458  0.05938  0.08590
-0.30  9000  0.60  0.09311 -0.13845 -0.03458  0.05938  0.08590
-0.30  9000  0.87  0.09311 -0.13845 -0.03458  0.05938  0.08590
-0.30  9000  1.00  0.09311 -0.13845 -0.03458  0.05938  0.08590
-0.30  9000  1.50  0.09311 -0.13845 -0.03458  0.05938  0.08590
-0.30  9000  2.00  0.11502 -0.04951 -0.06357  0.03683  0.08105
-0.30  9000  2.50  0.12309  0.00972 -0.06371  0.03054  0.07380
-0.30  9000  3.00  0.12747  0.05292 -0.04631  0.03119  0.06596
-0.30  9000  3.50  0.12958  0.08001 -0.01829  0.03538  0.05892
-0.30  9000  4.00  0.12858  0.08742  0.01602  0.04218  0.05376
-0.30  9000  4.50  0.12333  0.07264  0.05394  0.05118  0.05115
-0.30  9000  5.00  0.11244  0.03504  0.09290  0.06214  0.05186
-0.30  9000  5.50  0.11244  0.03504  0.09290  0.06214  0.05186
-0.30  9250 -1.02  0.06799 -0.10334 -0.07097  0.01949  0.05316
-0.30  9250 -0.70  0.06799 -0.10334 -0.07097  0.01949  0.05316
-0.30  9250 -0.51  0.06799 -0.10334 -0.07097  0.01949  0.05316
-0.30  9250 -0.29  0.06799 -0.10334 -0.07097  0.01949  0.05316
-0.30  9250  0.00  0.06799 -0.10334 -0.07097  0.01949  0.05316
-0.30  9250  0.28  0.06799 -0.10334 -0.07097  0.01949  0.05316
-0.30  9250  0.50  0.06799 -0.10334 -0.07097  0.01949  0.05316
-0.30  9250  0.60  0.06799 -0.10334 -0.07097  0.01949  0.05316
-0.30  9250  0.87  0.06799 -0.10334 -0.07097  0.01949  0.05316
-0.30  9250  1.00  0.06799 -0.10334 -0.07097  0.01949  0.05316
-0.30  9250  1.50  0.06799 -0.10334 -0.07097  0.01949  0.05316
-0.30  9250  2.00  0.06799 -0.10334 -0.07097  0.01949  0.05316
-0.30  9250  2.50  0.07739 -0.03948 -0.07619  0.00983  0.04648
-0.30  9250  3.00  0.08191  0.00788 -0.06366  0.00800  0.03895
-0.30  9250  3.50  0.08461  0.04031 -0.03932  0.01032  0.03172
-0.30  9250  4.00  0.08554  0.05562 -0.00713  0.01518  0.02567
-0.30  9250  4.50  0.08367  0.05007  0.03040  0.02216  0.02169
-0.30  9250  5.00  0.07726  0.02136  0.07025  0.03120  0.02093
-0.30  9250  5.50  0.07726  0.02136  0.07025  0.03120  0.02093
-0.30  9500 -1.02  0.02888 -0.14513 -0.07429  0.01380  0.03447
-0.30  9500 -0.70  0.02888 -0.14513 -0.07429  0.01380  0.03447
-0.30  9500 -0.51  0.02888 -0.14513 -0.07429  0.01380  0.03447
-0.30  9500 -0.29  0.02888 -0.14513 -0.07429  0.01380  0.03447
-0.30  9500  0.00  0.02888 -0.14513 -0.07429  0.01380  0.03447
-0.30  9500  0.28  0.02888 -0.14513 -0.07429  0.01380  0.03447
-0.30  9500  0.50  0.02888 -0.14513 -0.07429  0.01380  0.03447
-0.30  9500  0.60  0.02888 -0.14513 -0.07429  0.01380  0.03447
-0.30  9500  0.87  0.02888 -0.14513 -0.07429  0.01380  0.03447
-0.30  9500  1.00  0.02888 -0.14513 -0.07429  0.01380  0.03447
-0.30  9500  1.50  0.02888 -0.14513 -0.07429  0.01380  0.03447
-0.30  9500  2.00  0.02888 -0.14513 -0.07429  0.01380  0.03447
-0.30  9500  2.50  0.03925 -0.07771 -0.08365  0.00155  0.02841
-0.30  9500  3.00  0.04386 -0.02735 -0.07533 -0.00249  0.02122
-0.30  9500  3.50  0.04661  0.01005 -0.05477 -0.00199  0.01396
-0.30  9500  4.00  0.04853  0.03227 -0.02493  0.00115  0.00730
-0.30  9500  4.50  0.04872  0.03486  0.01136  0.00637  0.00243
-0.30  9500  5.00  0.04575  0.01431  0.05157  0.01365  0.00049
-0.30  9500  5.50  0.04575  0.01431  0.05157  0.01365  0.00049
-0.30  9750 -1.02 -0.00947 -0.18433 -0.07567  0.01239  0.02273
-0.30  9750 -0.70 -0.00947 -0.18433 -0.07567  0.01239  0.02273
-0.30  9750 -0.51 -0.00947 -0.18433 -0.07567  0.01239  0.02273
-0.30  9750 -0.29 -0.00947 -0.18433 -0.07567  0.01239  0.02273
-0.30  9750  0.00 -0.00947 -0.18433 -0.07567  0.01239  0.02273
-0.30  9750  0.28 -0.00947 -0.18433 -0.07567  0.01239  0.02273
-0.30  9750  0.50 -0.00947 -0.18433 -0.07567  0.01239  0.02273
-0.30  9750  0.60 -0.00947 -0.18433 -0.07567  0.01239  0.02273
-0.30  9750  0.87 -0.00947 -0.18433 -0.07567  0.01239  0.02273
-0.30  9750  1.00 -0.00947 -0.18433 -0.07567  0.01239  0.02273
-0.30  9750  1.50 -0.00947 -0.18433 -0.07567  0.01239  0.02273
-0.30  9750  2.00 -0.00947 -0.18433 -0.07567  0.01239  0.02273
-0.30  9750  2.50  0.00165 -0.11456 -0.08751 -0.00138  0.01729
-0.30  9750  3.00  0.00610 -0.06200 -0.08274 -0.00717  0.01064
-0.30  9750  3.50  0.00875 -0.02106 -0.06576 -0.00847  0.00337
-0.30  9750  4.00  0.01088  0.00648 -0.03881 -0.00682 -0.00349
-0.30  9750  4.50  0.01222  0.01648 -0.00422 -0.00315 -0.00893
-0.30  9750  5.00  0.01148  0.00352  0.03513  0.00254 -0.01185
-0.30  9750  5.50  0.01148  0.00352  0.03513  0.00254 -0.01185
-0.30  10000 -1.02 -0.06653 -0.23531 -0.08013  0.01421  0.01595
-0.30  10000 -0.70 -0.06653 -0.23531 -0.08013  0.01421  0.01595
-0.30  10000 -0.51 -0.06653 -0.23531 -0.08013  0.01421  0.01595
-0.30  10000 -0.29 -0.06653 -0.23531 -0.08013  0.01421  0.01595
-0.30  10000  0.00 -0.06653 -0.23531 -0.08013  0.01421  0.01595
-0.30  10000  0.28 -0.06653 -0.23531 -0.08013  0.01421  0.01595
-0.30  10000  0.50 -0.06653 -0.23531 -0.08013  0.01421  0.01595
-0.30  10000  0.60 -0.06653 -0.23531 -0.08013  0.01421  0.01595
-0.30  10000  0.87 -0.06653 -0.23531 -0.08013  0.01421  0.01595
-0.30  10000  1.00 -0.06653 -0.23531 -0.08013  0.01421  0.01595
-0.30  10000  1.50 -0.06653 -0.23531 -0.08013  0.01421  0.01595
-0.30  10000  2.00 -0.06653 -0.23531 -0.08013  0.01421  0.01595
-0.30  10000  2.50 -0.05484 -0.16329 -0.09368 -0.00064  0.01084
-0.30  10000  3.00 -0.05063 -0.10912 -0.09150 -0.00778  0.00462
-0.30  10000  3.50 -0.04863 -0.06638 -0.07735 -0.01033 -0.00234
-0.30  10000  4.00 -0.04702 -0.03496 -0.05353 -0.01019 -0.00912
-0.30  10000  4.50 -0.04536 -0.01949 -0.02116 -0.00786 -0.01493
-0.30  10000  5.00 -0.04486 -0.02530  0.01717 -0.00351 -0.01878
-0.30  10000  5.50 -0.04486 -0.02530  0.01717 -0.00351 -0.01878
-0.30  10500 -1.02 -0.18952 -0.33012 -0.09082  0.01605  0.00649
-0.30  10500 -0.70 -0.18952 -0.33012 -0.09082  0.01605  0.00649
-0.30  10500 -0.51 -0.18952 -0.33012 -0.09082  0.01605  0.00649
-0.30  10500 -0.29 -0.18952 -0.33012 -0.09082  0.01605  0.00649
-0.30  10500  0.00 -0.18952 -0.33012 -0.09082  0.01605  0.00649
-0.30  10500  0.28 -0.18952 -0.33012 -0.09082  0.01605  0.00649
-0.30  10500  0.50 -0.18952 -0.33012 -0.09082  0.01605  0.00649
-0.30  10500  0.60 -0.18952 -0.33012 -0.09082  0.01605  0.00649
-0.30  10500  0.87 -0.18952 -0.33012 -0.09082  0.01605  0.00649
-0.30  10500  1.00 -0.18952 -0.33012 -0.09082  0.01605  0.00649
-0.30  10500  1.50 -0.18952 -0.33012 -0.09082  0.01605  0.00649
-0.30  10500  2.00 -0.18952 -0.33012 -0.09082  0.01605  0.00649
-0.30  10500  2.50 -0.17681 -0.25521 -0.10642  0.00042  0.00148
-0.30  10500  3.00 -0.17349 -0.19940 -0.10735 -0.00829 -0.00393
-0.30  10500  3.50 -0.17315 -0.15498 -0.09789 -0.01274 -0.00996
-0.30  10500  4.00 -0.17302 -0.12025 -0.07934 -0.01459 -0.01619
-0.30  10500  4.50 -0.17209 -0.09724 -0.05219 -0.01436 -0.02236
-0.30  10500  5.00 -0.17069 -0.09067 -0.01690 -0.01256 -0.02745
-0.30  10500  5.50 -0.17069 -0.09067 -0.01690 -0.01256 -0.02745
-0.30  11000 -1.02 -0.30053 -0.33416 -0.11952 -0.00102 -0.00985
-0.30  11000 -0.70 -0.30053 -0.33416 -0.11952 -0.00102 -0.00985
-0.30  11000 -0.51 -0.30053 -0.33416 -0.11952 -0.00102 -0.00985
-0.30  11000 -0.29 -0.30053 -0.33416 -0.11952 -0.00102 -0.00985
-0.30  11000  0.00 -0.30053 -0.33416 -0.11952 -0.00102 -0.00985
-0.30  11000  0.28 -0.30053 -0.33416 -0.11952 -0.00102 -0.00985
-0.30  11000  0.50 -0.30053 -0.33416 -0.11952 -0.00102 -0.00985
-0.30  11000  0.60 -0.30053 -0.33416 -0.11952 -0.00102 -0.00985
-0.30  11000  0.87 -0.30053 -0.33416 -0.11952 -0.00102 -0.00985
-0.30  11000  1.00 -0.30053 -0.33416 -0.11952 -0.00102 -0.00985
-0.30  11000  1.50 -0.30053 -0.33416 -0.11952 -0.00102 -0.00985
-0.30  11000  2.00 -0.30053 -0.33416 -0.11952 -0.00102 -0.00985
-0.30  11000  2.50 -0.30053 -0.33416 -0.11952 -0.00102 -0.00985
-0.30  11000  3.00 -0.29782 -0.27799 -0.12217 -0.01008 -0.01501
-0.30  11000  3.50 -0.29884 -0.23306 -0.11540 -0.01561 -0.02037
-0.30  11000  4.00 -0.30051 -0.19685 -0.10074 -0.01873 -0.02599
-0.30  11000  4.50 -0.30115 -0.16998 -0.07821 -0.02015 -0.03175
-0.30  11000  5.00 -0.30035 -0.15610 -0.04739 -0.02011 -0.03708
-0.30  11000  5.50 -0.30035 -0.15610 -0.04739 -0.02011 -0.03708
-0.30  11500 -1.02 -0.48324 -0.42392 -0.14010 -0.00547 -0.02227
-0.30  11500 -0.70 -0.48324 -0.42392 -0.14010 -0.00547 -0.02227
-0.30  11500 -0.51 -0.48324 -0.42392 -0.14010 -0.00547 -0.02227
-0.30  11500 -0.29 -0.48324 -0.42392 -0.14010 -0.00547 -0.02227
-0.30  11500  0.00 -0.48324 -0.42392 -0.14010 -0.00547 -0.02227
-0.30  11500  0.28 -0.48324 -0.42392 -0.14010 -0.00547 -0.02227
-0.30  11500  0.50 -0.48324 -0.42392 -0.14010 -0.00547 -0.02227
-0.30  11500  0.60 -0.48324 -0.42392 -0.14010 -0.00547 -0.02227
-0.30  11500  0.87 -0.48324 -0.42392 -0.14010 -0.00547 -0.02227
-0.30  11500  1.00 -0.48324 -0.42392 -0.14010 -0.00547 -0.02227
-0.30  11500  1.50 -0.48324 -0.42392 -0.14010 -0.00547 -0.02227
-0.30  11500  2.00 -0.48324 -0.42392 -0.14010 -0.00547 -0.02227
-0.30  11500  2.50 -0.48324 -0.42392 -0.14010 -0.00547 -0.02227
-0.30  11500  3.00 -0.48178 -0.36889 -0.14352 -0.01450 -0.02722
-0.30  11500  3.50 -0.48473 -0.32468 -0.13816 -0.02029 -0.03225
-0.30  11500  4.00 -0.48866 -0.28854 -0.12587 -0.02414 -0.03736
-0.30  11500  4.50 -0.49160 -0.26069 -0.10693 -0.02647 -0.04247
-0.30  11500  5.00 -0.49248 -0.24322 -0.08041 -0.02765 -0.04747
-0.30  11500  5.50 -0.49248 -0.24322 -0.08041 -0.02765 -0.04747
-0.30  12000 -1.02 -0.44902 -0.42272 -0.12822 -0.02166 -0.03266
-0.30  12000 -0.70 -0.44902 -0.42272 -0.12822 -0.02166 -0.03266
-0.30  12000 -0.51 -0.44902 -0.42272 -0.12822 -0.02166 -0.03266
-0.30  12000 -0.29 -0.44902 -0.42272 -0.12822 -0.02166 -0.03266
-0.30  12000  0.00 -0.44902 -0.42272 -0.12822 -0.02166 -0.03266
-0.30  12000  0.28 -0.44902 -0.42272 -0.12822 -0.02166 -0.03266
-0.30  12000  0.50 -0.44902 -0.42272 -0.12822 -0.02166 -0.03266
-0.30  12000  0.60 -0.44902 -0.42272 -0.12822 -0.02166 -0.03266
-0.30  12000  0.87 -0.44902 -0.42272 -0.12822 -0.02166 -0.03266
-0.30  12000  1.00 -0.44902 -0.42272 -0.12822 -0.02166 -0.03266
-0.30  12000  1.50 -0.44902 -0.42272 -0.12822 -0.02166 -0.03266
-0.30  12000  2.00 -0.44902 -0.42272 -0.12822 -0.02166 -0.03266
-0.30  12000  2.50 -0.44902 -0.42272 -0.12822 -0.02166 -0.03266
-0.30  12000  3.00 -0.44449 -0.36997 -0.13227 -0.03021 -0.03758
-0.30  12000  3.50 -0.44498 -0.32721 -0.12801 -0.03583 -0.04248
-0.30  12000  4.00 -0.44720 -0.29154 -0.11753 -0.03988 -0.04729
-0.30  12000  4.50 -0.44964 -0.26298 -0.10120 -0.04288 -0.05178
-0.30  12000  5.00 -0.45083 -0.24282 -0.07865 -0.04487 -0.05640
-0.30  12000  5.50 -0.45083 -0.24282 -0.07865 -0.04487 -0.05640
-0.30  12500 -1.02 -0.54450 -0.46757 -0.13627 -0.02708 -0.04245
-0.30  12500 -0.70 -0.54450 -0.46757 -0.13627 -0.02708 -0.04245
-0.30  12500 -0.51 -0.54450 -0.46757 -0.13627 -0.02708 -0.04245
-0.30  12500 -0.29 -0.54450 -0.46757 -0.13627 -0.02708 -0.04245
-0.30  12500  0.00 -0.54450 -0.46757 -0.13627 -0.02708 -0.04245
-0.30  12500  0.28 -0.54450 -0.46757 -0.13627 -0.02708 -0.04245
-0.30  12500  0.50 -0.54450 -0.46757 -0.13627 -0.02708 -0.04245
-0.30  12500  0.60 -0.54450 -0.46757 -0.13627 -0.02708 -0.04245
-0.30  12500  0.87 -0.54450 -0.46757 -0.13627 -0.02708 -0.04245
-0.30  12500  1.00 -0.54450 -0.46757 -0.13627 -0.02708 -0.04245
-0.30  12500  1.50 -0.54450 -0.46757 -0.13627 -0.02708 -0.04245
-0.30  12500  2.00 -0.54450 -0.46757 -0.13627 -0.02708 -0.04245
-0.30  12500  2.50 -0.54450 -0.46757 -0.13627 -0.02708 -0.04245
-0.30  12500  3.00 -0.54016 -0.41686 -0.14082 -0.03506 -0.04704
-0.30  12500  3.50 -0.54048 -0.37641 -0.13694 -0.04040 -0.05164
-0.30  12500  4.00 -0.54269 -0.34239 -0.12742 -0.04444 -0.05621
-0.30  12500  4.50 -0.54545 -0.31446 -0.11268 -0.04765 -0.06051
-0.30  12500  5.00 -0.54741 -0.29405 -0.09250 -0.05010 -0.06473
-0.30  12500  5.50 -0.54741 -0.29405 -0.09250 -0.05010 -0.06473
-0.30  13000 -1.02 -0.63802 -0.50764 -0.14389 -0.03253 -0.05139
-0.30  13000 -0.70 -0.63802 -0.50764 -0.14389 -0.03253 -0.05139
-0.30  13000 -0.51 -0.63802 -0.50764 -0.14389 -0.03253 -0.05139
-0.30  13000 -0.29 -0.63802 -0.50764 -0.14389 -0.03253 -0.05139
-0.30  13000  0.00 -0.63802 -0.50764 -0.14389 -0.03253 -0.05139
-0.30  13000  0.28 -0.63802 -0.50764 -0.14389 -0.03253 -0.05139
-0.30  13000  0.50 -0.63802 -0.50764 -0.14389 -0.03253 -0.05139
-0.30  13000  0.60 -0.63802 -0.50764 -0.14389 -0.03253 -0.05139
-0.30  13000  0.87 -0.63802 -0.50764 -0.14389 -0.03253 -0.05139
-0.30  13000  1.00 -0.63802 -0.50764 -0.14389 -0.03253 -0.05139
-0.30  13000  1.50 -0.63802 -0.50764 -0.14389 -0.03253 -0.05139
-0.30  13000  2.00 -0.63802 -0.50764 -0.14389 -0.03253 -0.05139
-0.30  13000  2.50 -0.63802 -0.50764 -0.14389 -0.03253 -0.05139
-0.30  13000  3.00 -0.63417 -0.45813 -0.14875 -0.04024 -0.05559
-0.30  13000  3.50 -0.63469 -0.41979 -0.14542 -0.04505 -0.05997
-0.30  13000  4.00 -0.63678 -0.38792 -0.13646 -0.04880 -0.06433
-0.30  13000  4.50 -0.63939 -0.36146 -0.12262 -0.05206 -0.06851
-0.30  13000  5.00 -0.64179 -0.34138 -0.10407 -0.05468 -0.07256
-0.30  13000  5.50 -0.64179 -0.34138 -0.10407 -0.05468 -0.07256
-0.30  14000 -1.02 -0.83290 -0.66068 -0.13455 -0.02691 -0.06262
-0.30  14000 -0.70 -0.83290 -0.66068 -0.13455 -0.02691 -0.06262
-0.30  14000 -0.51 -0.83290 -0.66068 -0.13455 -0.02691 -0.06262
-0.30  14000 -0.29 -0.83290 -0.66068 -0.13455 -0.02691 -0.06262
-0.30  14000  0.00 -0.83290 -0.66068 -0.13455 -0.02691 -0.06262
-0.30  14000  0.28 -0.83290 -0.66068 -0.13455 -0.02691 -0.06262
-0.30  14000  0.50 -0.83290 -0.66068 -0.13455 -0.02691 -0.06262
-0.30  14000  0.60 -0.83290 -0.66068 -0.13455 -0.02691 -0.06262
-0.30  14000  0.87 -0.83290 -0.66068 -0.13455 -0.02691 -0.06262
-0.30  14000  1.00 -0.83290 -0.66068 -0.13455 -0.02691 -0.06262
-0.30  14000  1.50 -0.83290 -0.66068 -0.13455 -0.02691 -0.06262
-0.30  14000  2.00 -0.83290 -0.66068 -0.13455 -0.02691 -0.06262
-0.30  14000  2.50 -0.81725 -0.57804 -0.15774 -0.04302 -0.06763
-0.30  14000  3.00 -0.81575 -0.52892 -0.16329 -0.05021 -0.07122
-0.30  14000  3.50 -0.81745 -0.49356 -0.16037 -0.05441 -0.07485
-0.30  14000  4.00 -0.81974 -0.46534 -0.15235 -0.05755 -0.07875
-0.30  14000  4.50 -0.82211 -0.44239 -0.13995 -0.06025 -0.08272
-0.30  14000  5.00 -0.82445 -0.42433 -0.12354 -0.06284 -0.08647
-0.30  14000  5.50 -0.82445 -0.42433 -0.12354 -0.06284 -0.08647
-0.30  15000 -1.02 -0.98487 -0.63971 -0.17044 -0.05280 -0.08212
-0.30  15000 -0.70 -0.98487 -0.63971 -0.17044 -0.05280 -0.08212
-0.30  15000 -0.51 -0.98487 -0.63971 -0.17044 -0.05280 -0.08212
-0.30  15000 -0.29 -0.98487 -0.63971 -0.17044 -0.05280 -0.08212
-0.30  15000  0.00 -0.98487 -0.63971 -0.17044 -0.05280 -0.08212
-0.30  15000  0.28 -0.98487 -0.63971 -0.17044 -0.05280 -0.08212
-0.30  15000  0.50 -0.98487 -0.63971 -0.17044 -0.05280 -0.08212
-0.30  15000  0.60 -0.98487 -0.63971 -0.17044 -0.05280 -0.08212
-0.30  15000  0.87 -0.98487 -0.63971 -0.17044 -0.05280 -0.08212
-0.30  15000  1.00 -0.98487 -0.63971 -0.17044 -0.05280 -0.08212
-0.30  15000  1.50 -0.98487 -0.63971 -0.17044 -0.05280 -0.08212
-0.30  15000  2.00 -0.98487 -0.63971 -0.17044 -0.05280 -0.08212
-0.30  15000  2.50 -0.98487 -0.63971 -0.17044 -0.05280 -0.08212
-0.30  15000  3.00 -0.98738 -0.58859 -0.17686 -0.05976 -0.08514
-0.30  15000  3.50 -0.99145 -0.55416 -0.17427 -0.06350 -0.08815
-0.30  15000  4.00 -0.99500 -0.52846 -0.16685 -0.06594 -0.09140
-0.30  15000  4.50 -0.99749 -0.50862 -0.15541 -0.06813 -0.09485
-0.30  15000  5.00 -0.99934 -0.49348 -0.14024 -0.07039 -0.09837
-0.30  15000  5.50 -0.99934 -0.49348 -0.14024 -0.07039 -0.09837
-0.30  16000 -1.02 -1.14059 -0.69618 -0.18181 -0.06158 -0.09532
-0.30  16000 -0.70 -1.14059 -0.69618 -0.18181 -0.06158 -0.09532
-0.30  16000 -0.51 -1.14059 -0.69618 -0.18181 -0.06158 -0.09532
-0.30  16000 -0.29 -1.14059 -0.69618 -0.18181 -0.06158 -0.09532
-0.30  16000  0.00 -1.14059 -0.69618 -0.18181 -0.06158 -0.09532
-0.30  16000  0.28 -1.14059 -0.69618 -0.18181 -0.06158 -0.09532
-0.30  16000  0.50 -1.14059 -0.69618 -0.18181 -0.06158 -0.09532
-0.30  16000  0.60 -1.14059 -0.69618 -0.18181 -0.06158 -0.09532
-0.30  16000  0.87 -1.14059 -0.69618 -0.18181 -0.06158 -0.09532
-0.30  16000  1.00 -1.14059 -0.69618 -0.18181 -0.06158 -0.09532
-0.30  16000  1.50 -1.14059 -0.69618 -0.18181 -0.06158 -0.09532
-0.30  16000  2.00 -1.14059 -0.69618 -0.18181 -0.06158 -0.09532
-0.30  16000  2.50 -1.14059 -0.69618 -0.18181 -0.06158 -0.09532
-0.30  16000  3.00 -1.14804 -0.64119 -0.18953 -0.06869 -0.09787
-0.30  16000  3.50 -1.15569 -0.60632 -0.18733 -0.07197 -0.10016
-0.30  16000  4.00 -1.16114 -0.58189 -0.17999 -0.07421 -0.10281
-0.30  16000  4.50 -1.16461 -0.56379 -0.16938 -0.07589 -0.10577
-0.30  16000  5.00 -1.16642 -0.55119 -0.15513 -0.07779 -0.10888
-0.30  16000  5.50 -1.16642 -0.55119 -0.15513 -0.07779 -0.10888
-0.30  17000 -1.02 -1.28448 -0.75008 -0.19041 -0.06960 -0.10729
-0.30  17000 -0.70 -1.28448 -0.75008 -0.19041 -0.06960 -0.10729
-0.30  17000 -0.51 -1.28448 -0.75008 -0.19041 -0.06960 -0.10729
-0.30  17000 -0.29 -1.28448 -0.75008 -0.19041 -0.06960 -0.10729
-0.30  17000  0.00 -1.28448 -0.75008 -0.19041 -0.06960 -0.10729
-0.30  17000  0.28 -1.28448 -0.75008 -0.19041 -0.06960 -0.10729
-0.30  17000  0.50 -1.28448 -0.75008 -0.19041 -0.06960 -0.10729
-0.30  17000  0.60 -1.28448 -0.75008 -0.19041 -0.06960 -0.10729
-0.30  17000  0.87 -1.28448 -0.75008 -0.19041 -0.06960 -0.10729
-0.30  17000  1.00 -1.28448 -0.75008 -0.19041 -0.06960 -0.10729
-0.30  17000  1.50 -1.28448 -0.75008 -0.19041 -0.06960 -0.10729
-0.30  17000  2.00 -1.28448 -0.75008 -0.19041 -0.06960 -0.10729
-0.30  17000  2.50 -1.28448 -0.75008 -0.19041 -0.06960 -0.10729
-0.30  17000  3.00 -1.29781 -0.68934 -0.20086 -0.07719 -0.10945
-0.30  17000  3.50 -1.30996 -0.65296 -0.19935 -0.08006 -0.11120
-0.30  17000  4.00 -1.31823 -0.62839 -0.19255 -0.08176 -0.11325
-0.30  17000  4.50 -1.32318 -0.61126 -0.18237 -0.08317 -0.11572
-0.30  17000  5.00 -1.32567 -0.59994 -0.16896 -0.08464 -0.11841
-0.30  17000  5.50 -1.32567 -0.59994 -0.16896 -0.08464 -0.11841
-0.30  18000 -1.02 -1.41718 -0.80217 -0.19633 -0.07607 -0.11788
-0.30  18000 -0.70 -1.41718 -0.80217 -0.19633 -0.07607 -0.11788
-0.30  18000 -0.51 -1.41718 -0.80217 -0.19633 -0.07607 -0.11788
-0.30  18000 -0.29 -1.41718 -0.80217 -0.19633 -0.07607 -0.11788
-0.30  18000  0.00 -1.41718 -0.80217 -0.19633 -0.07607 -0.11788
-0.30  18000  0.28 -1.41718 -0.80217 -0.19633 -0.07607 -0.11788
-0.30  18000  0.50 -1.41718 -0.80217 -0.19633 -0.07607 -0.11788
-0.30  18000  0.60 -1.41718 -0.80217 -0.19633 -0.07607 -0.11788
-0.30  18000  0.87 -1.41718 -0.80217 -0.19633 -0.07607 -0.11788
-0.30  18000  1.00 -1.41718 -0.80217 -0.19633 -0.07607 -0.11788
-0.30  18000  1.50 -1.41718 -0.80217 -0.19633 -0.07607 -0.11788
-0.30  18000  2.00 -1.41718 -0.80217 -0.19633 -0.07607 -0.11788
-0.30  18000  2.50 -1.41718 -0.80217 -0.19633 -0.07607 -0.11788
-0.30  18000  3.00 -1.43626 -0.73437 -0.21032 -0.08549 -0.12023
-0.30  18000  3.50 -1.45352 -0.69612 -0.20991 -0.08821 -0.12132
-0.30  18000  4.00 -1.46526 -0.67098 -0.20369 -0.08953 -0.12287
-0.30  18000  4.50 -1.47243 -0.65376 -0.19405 -0.09058 -0.12487
-0.30  18000  5.00 -1.47611 -0.64325 -0.18136 -0.09174 -0.12713
-0.30  18000  5.50 -1.47611 -0.64325 -0.18136 -0.09174 -0.12713
-0.30  19000 -1.02 -1.54102 -0.85382 -0.20018 -0.07975 -0.12634
-0.30  19000 -0.70 -1.54102 -0.85382 -0.20018 -0.07975 -0.12634
-0.30  19000 -0.51 -1.54102 -0.85382 -0.20018 -0.07975 -0.12634
-0.30  19000 -0.29 -1.54102 -0.85382 -0.20018 -0.07975 -0.12634
-0.30  19000  0.00 -1.54102 -0.85382 -0.20018 -0.07975 -0.12634
-0.30  19000  0.28 -1.54102 -0.85382 -0.20018 -0.07975 -0.12634
-0.30  19000  0.50 -1.54102 -0.85382 -0.20018 -0.07975 -0.12634
-0.30  19000  0.60 -1.54102 -0.85382 -0.20018 -0.07975 -0.12634
-0.30  19000  0.87 -1.54102 -0.85382 -0.20018 -0.07975 -0.12634
-0.30  19000  1.00 -1.54102 -0.85382 -0.20018 -0.07975 -0.12634
-0.30  19000  1.50 -1.54102 -0.85382 -0.20018 -0.07975 -0.12634
-0.30  19000  2.00 -1.54102 -0.85382 -0.20018 -0.07975 -0.12634
-0.30  19000  2.50 -1.54102 -0.85382 -0.20018 -0.07975 -0.12634
-0.30  19000  3.00 -1.56467 -0.77668 -0.21841 -0.09257 -0.13037
-0.30  19000  3.50 -1.58751 -0.73563 -0.21967 -0.09562 -0.13101
-0.30  19000  4.00 -1.60329 -0.70966 -0.21411 -0.09661 -0.13191
-0.30  19000  4.50 -1.61296 -0.69233 -0.20486 -0.09751 -0.13339
-0.30  19000  5.00 -1.61827 -0.68185 -0.19272 -0.09849 -0.13530
-0.30  19000  5.50 -1.61827 -0.68185 -0.19272 -0.09849 -0.13530
-0.30  20000 -1.02 -1.68385 -0.81727 -0.22506 -0.09840 -0.13949
-0.30  20000 -0.70 -1.68385 -0.81727 -0.22506 -0.09840 -0.13949
-0.30  20000 -0.51 -1.68385 -0.81727 -0.22506 -0.09840 -0.13949
-0.30  20000 -0.29 -1.68385 -0.81727 -0.22506 -0.09840 -0.13949
-0.30  20000  0.00 -1.68385 -0.81727 -0.22506 -0.09840 -0.13949
-0.30  20000  0.28 -1.68385 -0.81727 -0.22506 -0.09840 -0.13949
-0.30  20000  0.50 -1.68385 -0.81727 -0.22506 -0.09840 -0.13949
-0.30  20000  0.60 -1.68385 -0.81727 -0.22506 -0.09840 -0.13949
-0.30  20000  0.87 -1.68385 -0.81727 -0.22506 -0.09840 -0.13949
-0.30  20000  1.00 -1.68385 -0.81727 -0.22506 -0.09840 -0.13949
-0.30  20000  1.50 -1.68385 -0.81727 -0.22506 -0.09840 -0.13949
-0.30  20000  2.00 -1.68385 -0.81727 -0.22506 -0.09840 -0.13949
-0.30  20000  2.50 -1.68385 -0.81727 -0.22506 -0.09840 -0.13949
-0.30  20000  3.00 -1.68385 -0.81727 -0.22506 -0.09840 -0.13949
-0.30  20000  3.50 -1.71233 -0.77251 -0.22828 -0.10235 -0.14015
-0.30  20000  4.00 -1.73255 -0.74529 -0.22378 -0.10334 -0.14047
-0.30  20000  4.50 -1.74524 -0.72765 -0.21529 -0.10388 -0.14152
-0.30  20000  5.00 -1.75256 -0.71689 -0.20378 -0.10464 -0.14304
-0.30  20000  5.50 -1.75256 -0.71689 -0.20378 -0.10464 -0.14304
-0.30  21000 -1.02 -1.79572 -0.85624 -0.23062 -0.10270 -0.14710
-0.30  21000 -0.70 -1.79572 -0.85624 -0.23062 -0.10270 -0.14710
-0.30  21000 -0.51 -1.79572 -0.85624 -0.23062 -0.10270 -0.14710
-0.30  21000 -0.29 -1.79572 -0.85624 -0.23062 -0.10270 -0.14710
-0.30  21000  0.00 -1.79572 -0.85624 -0.23062 -0.10270 -0.14710
-0.30  21000  0.28 -1.79572 -0.85624 -0.23062 -0.10270 -0.14710
-0.30  21000  0.50 -1.79572 -0.85624 -0.23062 -0.10270 -0.14710
-0.30  21000  0.60 -1.79572 -0.85624 -0.23062 -0.10270 -0.14710
-0.30  21000  0.87 -1.79572 -0.85624 -0.23062 -0.10270 -0.14710
-0.30  21000  1.00 -1.79572 -0.85624 -0.23062 -0.10270 -0.14710
-0.30  21000  1.50 -1.79572 -0.85624 -0.23062 -0.10270 -0.14710
-0.30  21000  2.00 -1.79572 -0.85624 -0.23062 -0.10270 -0.14710
-0.30  21000  2.50 -1.79572 -0.85624 -0.23062 -0.10270 -0.14710
-0.30  21000  3.00 -1.79572 -0.85624 -0.23062 -0.10270 -0.14710
-0.30  21000  3.50 -1.82879 -0.80780 -0.23586 -0.10821 -0.14863
-0.30  21000  4.00 -1.85345 -0.77870 -0.23247 -0.10957 -0.14879
-0.30  21000  4.50 -1.86947 -0.76038 -0.22474 -0.11001 -0.14931
-0.30  21000  5.00 -1.87899 -0.74941 -0.21400 -0.11056 -0.15034
-0.30  21000  5.50 -1.87899 -0.74941 -0.21400 -0.11056 -0.15034
-0.30  22000 -1.02 -1.90195 -0.89185 -0.23574 -0.10618 -0.15340
-0.30  22000 -0.70 -1.90195 -0.89185 -0.23574 -0.10618 -0.15340
-0.30  22000 -0.51 -1.90195 -0.89185 -0.23574 -0.10618 -0.15340
-0.30  22000 -0.29 -1.90195 -0.89185 -0.23574 -0.10618 -0.15340
-0.30  22000  0.00 -1.90195 -0.89185 -0.23574 -0.10618 -0.15340
-0.30  22000  0.28 -1.90195 -0.89185 -0.23574 -0.10618 -0.15340
-0.30  22000  0.50 -1.90195 -0.89185 -0.23574 -0.10618 -0.15340
-0.30  22000  0.60 -1.90195 -0.89185 -0.23574 -0.10618 -0.15340
-0.30  22000  0.87 -1.90195 -0.89185 -0.23574 -0.10618 -0.15340
-0.30  22000  1.00 -1.90195 -0.89185 -0.23574 -0.10618 -0.15340
-0.30  22000  1.50 -1.90195 -0.89185 -0.23574 -0.10618 -0.15340
-0.30  22000  2.00 -1.90195 -0.89185 -0.23574 -0.10618 -0.15340
-0.30  22000  2.50 -1.90195 -0.89185 -0.23574 -0.10618 -0.15340
-0.30  22000  3.00 -1.90195 -0.89185 -0.23574 -0.10618 -0.15340
-0.30  22000  3.50 -1.93782 -0.84145 -0.24209 -0.11312 -0.15611
-0.30  22000  4.00 -1.96652 -0.81038 -0.24015 -0.11522 -0.15651
-0.30  22000  4.50 -1.98614 -0.79109 -0.23348 -0.11574 -0.15675
-0.30  22000  5.00 -1.99816 -0.77972 -0.22350 -0.11618 -0.15746
-0.30  22000  5.50 -1.99816 -0.77972 -0.22350 -0.11618 -0.15746
-0.30  23000 -1.02 -2.00421 -0.92276 -0.24142 -0.10992 -0.15891
-0.30  23000 -0.70 -2.00421 -0.92276 -0.24142 -0.10992 -0.15891
-0.30  23000 -0.51 -2.00421 -0.92276 -0.24142 -0.10992 -0.15891
-0.30  23000 -0.29 -2.00421 -0.92276 -0.24142 -0.10992 -0.15891
-0.30  23000  0.00 -2.00421 -0.92276 -0.24142 -0.10992 -0.15891
-0.30  23000  0.28 -2.00421 -0.92276 -0.24142 -0.10992 -0.15891
-0.30  23000  0.50 -2.00421 -0.92276 -0.24142 -0.10992 -0.15891
-0.30  23000  0.60 -2.00421 -0.92276 -0.24142 -0.10992 -0.15891
-0.30  23000  0.87 -2.00421 -0.92276 -0.24142 -0.10992 -0.15891
-0.30  23000  1.00 -2.00421 -0.92276 -0.24142 -0.10992 -0.15891
-0.30  23000  1.50 -2.00421 -0.92276 -0.24142 -0.10992 -0.15891
-0.30  23000  2.00 -2.00421 -0.92276 -0.24142 -0.10992 -0.15891
-0.30  23000  2.50 -2.00421 -0.92276 -0.24142 -0.10992 -0.15891
-0.30  23000  3.00 -2.00421 -0.92276 -0.24142 -0.10992 -0.15891
-0.30  23000  3.50 -2.04094 -0.87269 -0.24783 -0.11750 -0.16238
-0.30  23000  4.00 -2.07256 -0.84046 -0.24687 -0.12022 -0.16355
-0.30  23000  4.50 -2.09544 -0.81991 -0.24128 -0.12104 -0.16382
-0.30  23000  5.00 -2.11004 -0.80797 -0.23216 -0.12158 -0.16427
-0.30  23000  5.50 -2.11004 -0.80797 -0.23216 -0.12158 -0.16427
-0.30  24000 -1.02 -2.10089 -0.94975 -0.24748 -0.11421 -0.16471
-0.30  24000 -0.70 -2.10089 -0.94975 -0.24748 -0.11421 -0.16471
-0.30  24000 -0.51 -2.10089 -0.94975 -0.24748 -0.11421 -0.16471
-0.30  24000 -0.29 -2.10089 -0.94975 -0.24748 -0.11421 -0.16471
-0.30  24000  0.00 -2.10089 -0.94975 -0.24748 -0.11421 -0.16471
-0.30  24000  0.28 -2.10089 -0.94975 -0.24748 -0.11421 -0.16471
-0.30  24000  0.50 -2.10089 -0.94975 -0.24748 -0.11421 -0.16471
-0.30  24000  0.60 -2.10089 -0.94975 -0.24748 -0.11421 -0.16471
-0.30  24000  0.87 -2.10089 -0.94975 -0.24748 -0.11421 -0.16471
-0.30  24000  1.00 -2.10089 -0.94975 -0.24748 -0.11421 -0.16471
-0.30  24000  1.50 -2.10089 -0.94975 -0.24748 -0.11421 -0.16471
-0.30  24000  2.00 -2.10089 -0.94975 -0.24748 -0.11421 -0.16471
-0.30  24000  2.50 -2.10089 -0.94975 -0.24748 -0.11421 -0.16471
-0.30  24000  3.00 -2.10089 -0.94975 -0.24748 -0.11421 -0.16471
-0.30  24000  3.50 -2.13990 -0.90039 -0.25375 -0.12150 -0.16817
-0.30  24000  4.00 -2.17300 -0.86865 -0.25299 -0.12461 -0.16961
-0.30  24000  4.50 -2.19806 -0.84741 -0.24811 -0.12604 -0.17044
-0.30  24000  5.00 -2.21516 -0.83444 -0.24008 -0.12673 -0.17084
-0.30  24000  5.50 -2.21516 -0.83444 -0.24008 -0.12673 -0.17084
-0.30  25000 -1.02 -2.18947 -0.97592 -0.25234 -0.11774 -0.17067
-0.30  25000 -0.70 -2.18947 -0.97592 -0.25234 -0.11774 -0.17067
-0.30  25000 -0.51 -2.18947 -0.97592 -0.25234 -0.11774 -0.17067
-0.30  25000 -0.29 -2.18947 -0.97592 -0.25234 -0.11774 -0.17067
-0.30  25000  0.00 -2.18947 -0.97592 -0.25234 -0.11774 -0.17067
-0.30  25000  0.28 -2.18947 -0.97592 -0.25234 -0.11774 -0.17067
-0.30  25000  0.50 -2.18947 -0.97592 -0.25234 -0.11774 -0.17067
-0.30  25000  0.60 -2.18947 -0.97592 -0.25234 -0.11774 -0.17067
-0.30  25000  0.87 -2.18947 -0.97592 -0.25234 -0.11774 -0.17067
-0.30  25000  1.00 -2.18947 -0.97592 -0.25234 -0.11774 -0.17067
-0.30  25000  1.50 -2.18947 -0.97592 -0.25234 -0.11774 -0.17067
-0.30  25000  2.00 -2.18947 -0.97592 -0.25234 -0.11774 -0.17067
-0.30  25000  2.50 -2.18947 -0.97592 -0.25234 -0.11774 -0.17067
-0.30  25000  3.00 -2.18947 -0.97592 -0.25234 -0.11774 -0.17067
-0.30  25000  3.50 -2.23446 -0.92441 -0.26023 -0.12590 -0.17388
-0.30  25000  4.00 -2.26891 -0.89416 -0.25905 -0.12875 -0.17522
-0.30  25000  4.50 -2.29524 -0.87323 -0.25464 -0.13042 -0.17612
-0.30  25000  5.00 -2.31406 -0.86004 -0.24720 -0.13145 -0.17691
-0.30  25000  5.50 -2.31406 -0.86004 -0.24720 -0.13145 -0.17691
-0.30  26000 -1.02 -2.27133 -1.00313 -0.25431 -0.11870 -0.17481
-0.30  26000 -0.70 -2.27133 -1.00313 -0.25431 -0.11870 -0.17481
-0.30  26000 -0.51 -2.27133 -1.00313 -0.25431 -0.11870 -0.17481
-0.30  26000 -0.29 -2.27133 -1.00313 -0.25431 -0.11870 -0.17481
-0.30  26000  0.00 -2.27133 -1.00313 -0.25431 -0.11870 -0.17481
-0.30  26000  0.28 -2.27133 -1.00313 -0.25431 -0.11870 -0.17481
-0.30  26000  0.50 -2.27133 -1.00313 -0.25431 -0.11870 -0.17481
-0.30  26000  0.60 -2.27133 -1.00313 -0.25431 -0.11870 -0.17481
-0.30  26000  0.87 -2.27133 -1.00313 -0.25431 -0.11870 -0.17481
-0.30  26000  1.00 -2.27133 -1.00313 -0.25431 -0.11870 -0.17481
-0.30  26000  1.50 -2.27133 -1.00313 -0.25431 -0.11870 -0.17481
-0.30  26000  2.00 -2.27133 -1.00313 -0.25431 -0.11870 -0.17481
-0.30  26000  2.50 -2.27133 -1.00313 -0.25431 -0.11870 -0.17481
-0.30  26000  3.00 -2.27133 -1.00313 -0.25431 -0.11870 -0.17481
-0.30  26000  3.50 -2.32362 -0.94575 -0.26689 -0.13101 -0.17989
-0.30  26000  4.00 -2.36083 -0.91679 -0.26540 -0.13271 -0.18062
-0.30  26000  4.50 -2.38769 -0.89711 -0.26079 -0.13452 -0.18157
-0.30  26000  5.00 -2.40750 -0.88394 -0.25393 -0.13581 -0.18231
-0.30  26000  5.50 -2.40750 -0.88394 -0.25393 -0.13581 -0.18231
-0.30  27000 -1.02 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
-0.30  27000 -0.70 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
-0.30  27000 -0.51 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
-0.30  27000 -0.29 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
-0.30  27000  0.00 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
-0.30  27000  0.28 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
-0.30  27000  0.50 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
-0.30  27000  0.60 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
-0.30  27000  0.87 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
-0.30  27000  1.00 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
-0.30  27000  1.50 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
-0.30  27000  2.00 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
-0.30  27000  2.50 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
-0.30  27000  3.00 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
-0.30  27000  3.50 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
-0.30  27000  4.00 -2.44762 -0.93680 -0.27204 -0.13755 -0.18642
-0.30  27000  4.50 -2.47615 -0.91840 -0.26722 -0.13876 -0.18637
-0.30  27000  5.00 -2.49617 -0.90623 -0.26042 -0.13984 -0.18750
-0.30  27000  5.50 -2.49617 -0.90623 -0.26042 -0.13984 -0.18750
-0.30  28000 -1.02 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
-0.30  28000 -0.70 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
-0.30  28000 -0.51 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
-0.30  28000 -0.29 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
-0.30  28000  0.00 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
-0.30  28000  0.28 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
-0.30  28000  0.50 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
-0.30  28000  0.60 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
-0.30  28000  0.87 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
-0.30  28000  1.00 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
-0.30  28000  1.50 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
-0.30  28000  2.00 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
-0.30  28000  2.50 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
-0.30  28000  3.00 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
-0.30  28000  3.50 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
-0.30  28000  4.00 -2.52892 -0.95542 -0.27879 -0.14239 -0.19191
-0.30  28000  4.50 -2.55968 -0.93767 -0.27357 -0.14305 -0.19198
-0.30  28000  5.00 -2.58074 -0.92661 -0.26689 -0.14394 -0.19221
-0.30  28000  5.50 -2.58074 -0.92661 -0.26689 -0.14394 -0.19221
-0.30  29000 -1.02 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
-0.30  29000 -0.70 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
-0.30  29000 -0.51 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
-0.30  29000 -0.29 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
-0.30  29000  0.00 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
-0.30  29000  0.28 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
-0.30  29000  0.50 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
-0.30  29000  0.60 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
-0.30  29000  0.87 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
-0.30  29000  1.00 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
-0.30  29000  1.50 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
-0.30  29000  2.00 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
-0.30  29000  2.50 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
-0.30  29000  3.00 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
-0.30  29000  3.50 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
-0.30  29000  4.00 -2.60355 -0.97348 -0.28526 -0.14669 -0.19743
-0.30  29000  4.50 -2.63831 -0.95559 -0.28024 -0.14743 -0.19725
-0.30  29000  5.00 -2.66049 -0.94535 -0.27337 -0.14777 -0.19732
-0.30  29000  5.50 -2.66049 -0.94535 -0.27337 -0.14777 -0.19732
-0.30  30000 -1.02 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
-0.30  30000 -0.70 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
-0.30  30000 -0.51 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
-0.30  30000 -0.29 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
-0.30  30000  0.00 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
-0.30  30000  0.28 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
-0.30  30000  0.50 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
-0.30  30000  0.60 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
-0.30  30000  0.87 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
-0.30  30000  1.00 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
-0.30  30000  1.50 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
-0.30  30000  2.00 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
-0.30  30000  2.50 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
-0.30  30000  3.00 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
-0.30  30000  3.50 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
-0.30  30000  4.00 -2.67342 -0.99115 -0.29078 -0.15018 -0.20214
-0.30  30000  4.50 -2.71144 -0.97261 -0.28665 -0.15153 -0.20230
-0.30  30000  5.00 -2.73584 -0.96265 -0.27981 -0.15176 -0.20218
-0.30  30000  5.50 -2.73584 -0.96265 -0.27981 -0.15176 -0.20218
-0.30  31000 -1.02 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
-0.30  31000 -0.70 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
-0.30  31000 -0.51 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
-0.30  31000 -0.29 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
-0.30  31000  0.00 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
-0.30  31000  0.28 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
-0.30  31000  0.50 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
-0.30  31000  0.60 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
-0.30  31000  0.87 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
-0.30  31000  1.00 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
-0.30  31000  1.50 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
-0.30  31000  2.00 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
-0.30  31000  2.50 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
-0.30  31000  3.00 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
-0.30  31000  3.50 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
-0.30  31000  4.00 -2.74139 -1.00792 -0.29531 -0.15236 -0.20587
-0.30  31000  4.50 -2.78037 -0.98904 -0.29260 -0.15499 -0.20676
-0.30  31000  5.00 -2.80661 -0.97910 -0.28612 -0.15544 -0.20660
-0.30  31000  5.50 -2.80661 -0.97910 -0.28612 -0.15544 -0.20660
-0.30  32000 -1.02 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
-0.30  32000 -0.70 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
-0.30  32000 -0.51 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
-0.30  32000 -0.29 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
-0.30  32000  0.00 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
-0.30  32000  0.28 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
-0.30  32000  0.50 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
-0.30  32000  0.60 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
-0.30  32000  0.87 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
-0.30  32000  1.00 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
-0.30  32000  1.50 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
-0.30  32000  2.00 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
-0.30  32000  2.50 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
-0.30  32000  3.00 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
-0.30  32000  3.50 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
-0.30  32000  4.00 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
-0.30  32000  4.50 -2.84690 -1.00455 -0.29779 -0.15760 -0.21048
-0.30  32000  5.00 -2.87420 -0.99452 -0.29207 -0.15854 -0.21056
-0.30  32000  5.50 -2.87420 -0.99452 -0.29207 -0.15854 -0.21056
-0.30  33000 -1.02 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
-0.30  33000 -0.70 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
-0.30  33000 -0.51 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
-0.30  33000 -0.29 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
-0.30  33000  0.00 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
-0.30  33000  0.28 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
-0.30  33000  0.50 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
-0.30  33000  0.60 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
-0.30  33000  0.87 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
-0.30  33000  1.00 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
-0.30  33000  1.50 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
-0.30  33000  2.00 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
-0.30  33000  2.50 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
-0.30  33000  3.00 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
-0.30  33000  3.50 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
-0.30  33000  4.00 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
-0.30  33000  4.50 -2.91195 -1.01898 -0.30219 -0.15942 -0.21359
-0.30  33000  5.00 -2.93942 -1.00904 -0.29737 -0.16111 -0.21400
-0.30  33000  5.50 -2.93942 -1.00904 -0.29737 -0.16111 -0.21400
-0.30  34000 -1.02 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
-0.30  34000 -0.70 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
-0.30  34000 -0.51 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
-0.30  34000 -0.29 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
-0.30  34000  0.00 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
-0.30  34000  0.28 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
-0.30  34000  0.50 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
-0.30  34000  0.60 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
-0.30  34000  0.87 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
-0.30  34000  1.00 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
-0.30  34000  1.50 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
-0.30  34000  2.00 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
-0.30  34000  2.50 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
-0.30  34000  3.00 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
-0.30  34000  3.50 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
-0.30  34000  4.00 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
-0.30  34000  4.50 -2.97899 -1.03181 -0.30566 -0.16027 -0.21590
-0.30  34000  5.00 -3.00391 -1.02231 -0.30213 -0.16293 -0.21688
-0.30  34000  5.50 -3.00391 -1.02231 -0.30213 -0.16293 -0.21688
-0.30  35000 -1.02 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
-0.30  35000 -0.70 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
-0.30  35000 -0.51 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
-0.30  35000 -0.29 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
-0.30  35000  0.00 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
-0.30  35000  0.28 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
-0.30  35000  0.50 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
-0.30  35000  0.60 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
-0.30  35000  0.87 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
-0.30  35000  1.00 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
-0.30  35000  1.50 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
-0.30  35000  2.00 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
-0.30  35000  2.50 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
-0.30  35000  3.00 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
-0.30  35000  3.50 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
-0.30  35000  4.00 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
-0.30  35000  4.50 -3.04809 -1.04294 -0.30835 -0.16031 -0.21740
-0.30  35000  5.00 -3.06963 -1.03422 -0.30595 -0.16402 -0.21917
-0.30  35000  5.50 -3.06963 -1.03422 -0.30595 -0.16402 -0.21917
-0.30  37500 -1.02 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
-0.30  37500 -0.70 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
-0.30  37500 -0.51 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
-0.30  37500 -0.29 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
-0.30  37500  0.00 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
-0.30  37500  0.28 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
-0.30  37500  0.50 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
-0.30  37500  0.60 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
-0.30  37500  0.87 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
-0.30  37500  1.00 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
-0.30  37500  1.50 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
-0.30  37500  2.00 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
-0.30  37500  2.50 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
-0.30  37500  3.00 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
-0.30  37500  3.50 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
-0.30  37500  4.00 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
-0.30  37500  4.50 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
-0.30  37500  5.00 -3.24088 -1.05803 -0.31302 -0.16379 -0.22209
-0.30  37500  5.50 -3.24088 -1.05803 -0.31302 -0.16379 -0.22209
-0.30  40000 -1.02 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
-0.30  40000 -0.70 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
-0.30  40000 -0.51 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
-0.30  40000 -0.29 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
-0.30  40000  0.00 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
-0.30  40000  0.28 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
-0.30  40000  0.50 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
-0.30  40000  0.60 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
-0.30  40000  0.87 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
-0.30  40000  1.00 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
-0.30  40000  1.50 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
-0.30  40000  2.00 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
-0.30  40000  2.50 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
-0.30  40000  3.00 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
-0.30  40000  3.50 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
-0.30  40000  4.00 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
-0.30  40000  4.50 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
-0.30  40000  5.00 -3.42493 -1.07283 -0.31872 -0.16266 -0.22304
-0.30  40000  5.50 -3.42493 -1.07283 -0.31872 -0.16266 -0.22304
-0.30  42500 -1.02 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
-0.30  42500 -0.70 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
-0.30  42500 -0.51 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
-0.30  42500 -0.29 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
-0.30  42500  0.00 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
-0.30  42500  0.28 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
-0.30  42500  0.50 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
-0.30  42500  0.60 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
-0.30  42500  0.87 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
-0.30  42500  1.00 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
-0.30  42500  1.50 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
-0.30  42500  2.00 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
-0.30  42500  2.50 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
-0.30  42500  3.00 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
-0.30  42500  3.50 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
-0.30  42500  4.00 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
-0.30  42500  4.50 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
-0.30  42500  5.00 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
-0.30  42500  5.50 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
-0.30  45000 -1.02 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
-0.30  45000 -0.70 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
-0.30  45000 -0.51 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
-0.30  45000 -0.29 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
-0.30  45000  0.00 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
-0.30  45000  0.28 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
-0.30  45000  0.50 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
-0.30  45000  0.60 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
-0.30  45000  0.87 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
-0.30  45000  1.00 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
-0.30  45000  1.50 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
-0.30  45000  2.00 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
-0.30  45000  2.50 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
-0.30  45000  3.00 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
-0.30  45000  3.50 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
-0.30  45000  4.00 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
-0.30  45000  4.50 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
-0.30  45000  5.00 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
-0.30  45000  5.50 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
-0.30  47500 -1.02 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
-0.30  47500 -0.70 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
-0.30  47500 -0.51 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
-0.30  47500 -0.29 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
-0.30  47500  0.00 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
-0.30  47500  0.28 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
-0.30  47500  0.50 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
-0.30  47500  0.60 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
-0.30  47500  0.87 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
-0.30  47500  1.00 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
-0.30  47500  1.50 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
-0.30  47500  2.00 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
-0.30  47500  2.50 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
-0.30  47500  3.00 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
-0.30  47500  3.50 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
-0.30  47500  4.00 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
-0.30  47500  4.50 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
-0.30  47500  5.00 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
-0.30  47500  5.50 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
-0.30  50000 -1.02 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
-0.30  50000 -0.70 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
-0.30  50000 -0.51 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
-0.30  50000 -0.29 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
-0.30  50000  0.00 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
-0.30  50000  0.28 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
-0.30  50000  0.50 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
-0.30  50000  0.60 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
-0.30  50000  0.87 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
-0.30  50000  1.00 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
-0.30  50000  1.50 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
-0.30  50000  2.00 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
-0.30  50000  2.50 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
-0.30  50000  3.00 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
-0.30  50000  3.50 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
-0.30  50000  4.00 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
-0.30  50000  4.50 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
-0.30  50000  5.00 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
-0.30  50000  5.50 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
 0.00  2000 -1.02 -7.05491 -0.29783  1.12030  3.74948  3.02679
 0.00  2000 -0.70 -7.05491 -0.29783  1.12030  3.74948  3.02679
 0.00  2000 -0.51 -7.05491 -0.29783  1.12030  3.74948  3.02679
 0.00  2000 -0.29 -7.05491 -0.29783  1.12030  3.74948  3.02679
 0.00  2000  0.00 -7.05491 -0.29783  1.12030  3.74948  3.02679
 0.00  2000  0.28 -7.05491 -0.29783  1.12030  3.74948  3.02679
 0.00  2000  0.50 -7.05491 -0.29783  1.12030  3.74948  3.02679
 0.00  2000  0.60 -7.05491 -0.29783  1.12030  3.74948  3.02679
 0.00  2000  0.87 -7.05491 -0.29783  1.12030  3.74948  3.02679
 0.00  2000  1.00 -7.05491 -0.29783  1.12030  3.74948  3.02679
 0.00  2000  1.50 -7.05491 -0.29783  1.12030  3.74948  3.02679
 0.00  2000  2.00 -7.05491 -0.29783  1.12030  3.74948  3.02679
 0.00  2000  2.50 -7.05491 -0.29783  1.12030  3.74948  3.02679
 0.00  2000  3.00 -7.05491 -0.29783  1.12030  3.74948  3.02679
 0.00  2000  3.50 -7.05491 -0.29783  1.12030  3.74948  3.02679
 0.00  2000  4.00 -7.19603  0.30737  1.98438  3.83970  3.00860
 0.00  2000  4.50 -7.36220  1.35906  2.55481  3.97758  2.99233
 0.00  2000  5.00 -7.45940  2.69098  2.84060  4.13058  2.93958
 0.00  2000  5.50 -7.37559  3.82038  2.96376  4.22386  2.83140
 0.00  2200 -1.02 -5.74734  0.00490  1.08443  3.18965  2.78974
 0.00  2200 -0.70 -5.74734  0.00490  1.08443  3.18965  2.78974
 0.00  2200 -0.51 -5.74734  0.00490  1.08443  3.18965  2.78974
 0.00  2200 -0.29 -5.74734  0.00490  1.08443  3.18965  2.78974
 0.00  2200  0.00 -5.74734  0.00490  1.08443  3.18965  2.78974
 0.00  2200  0.28 -5.74734  0.00490  1.08443  3.18965  2.78974
 0.00  2200  0.50 -5.74734  0.00490  1.08443  3.18965  2.78974
 0.00  2200  0.60 -5.74734  0.00490  1.08443  3.18965  2.78974
 0.00  2200  0.87 -5.74734  0.00490  1.08443  3.18965  2.78974
 0.00  2200  1.00 -5.74734  0.00490  1.08443  3.18965  2.78974
 0.00  2200  1.50 -5.74734  0.00490  1.08443  3.18965  2.78974
 0.00  2200  2.00 -5.74734  0.00490  1.08443  3.18965  2.78974
 0.00  2200  2.50 -5.74734  0.00490  1.08443  3.18965  2.78974
 0.00  2200  3.00 -5.74734  0.00490  1.08443  3.18965  2.78974
 0.00  2200  3.50 -5.74734  0.00490  1.08443  3.18965  2.78974
 0.00  2200  4.00 -5.83881  0.47459  1.74295  3.24982  2.77385
 0.00  2200  4.50 -6.03314  1.22234  2.23193  3.37866  2.79895
 0.00  2200  5.00 -6.18971  2.28050  2.56496  3.53287  2.77720
 0.00  2200  5.50 -6.26440  3.40841  2.73528  3.68918  2.70344
 0.00  2500 -1.02 -5.99423 -0.23091  1.41747  3.08183  2.87080
 0.00  2500 -0.70 -5.60200 -0.23088  1.40344  2.89877  2.78610
 0.00  2500 -0.51 -5.37788 -0.23088  1.39685  2.79747  2.73588
 0.00  2500 -0.29 -5.11838 -0.23088  1.38922  2.68017  2.67773
 0.00  2500  0.00 -4.94869 -0.23089  1.38976  2.66402  2.63922
 0.00  2500  0.28 -4.78486 -0.23089  1.39029  2.64842  2.60205
 0.00  2500  0.50 -4.76560 -0.18071  1.37907  2.64935  2.59194
 0.00  2500  0.60 -4.75684 -0.15790  1.37396  2.64977  2.58735
 0.00  2500  0.87 -4.73319 -0.09632  1.36019  2.65092  2.57496
 0.00  2500  1.00 -4.72181 -0.06667  1.35355  2.65147  2.56899
 0.00  2500  1.50 -4.67802  0.04737  1.32804  2.65358  2.54603
 0.00  2500  2.00 -4.63424  0.16141  1.30253  2.65570  2.52308
 0.00  2500  2.50 -4.59045  0.27545  1.27702  2.65782  2.50012
 0.00  2500  3.00 -4.54667  0.38949  1.25151  2.65993  2.47717
 0.00  2500  3.50 -4.50288  0.50353  1.22600  2.66205  2.45421
 0.00  2500  4.00 -4.48830  0.86898  1.63556  2.68648  2.40440
 0.00  2500  4.50 -4.57277  1.35470  1.95371  2.74300  2.41254
 0.00  2500  5.00 -4.71326  2.01739  2.21663  2.84811  2.42855
 0.00  2500  5.50 -4.90478  2.87751  2.40505  3.01804  2.42611
 0.00  2800 -1.02 -5.77601  0.06061  1.45009  2.93050  2.86379
 0.00  2800 -0.70 -5.43583  0.06067  1.44188  2.80713  2.76515
 0.00  2800 -0.51 -5.23440  0.06063  1.43772  2.73494  2.70608
 0.00  2800 -0.29 -5.00117  0.06059  1.43290  2.65135  2.63769
 0.00  2800  0.00 -4.74361  0.06062  1.43026  2.58145  2.56286
 0.00  2800  0.28 -4.49493  0.06065  1.42772  2.51396  2.49061
 0.00  2800  0.50 -4.31873  0.06069  1.42664  2.47409  2.44065
 0.00  2800  0.60 -4.23864  0.06072  1.42615  2.45597  2.41793
 0.00  2800  0.87 -4.17625  0.12522  1.42588  2.44120  2.38421
 0.00  2800  1.00 -4.14622  0.15628  1.42575  2.43408  2.36797
 0.00  2800  1.50 -4.03069  0.27573  1.42524  2.40672  2.30551
 0.00  2800  2.00 -3.91516  0.39519  1.42474  2.37937  2.24306
 0.00  2800  2.50 -3.79963  0.51464  1.42423  2.35201  2.18060
 0.00  2800  3.00 -3.68411  0.63410  1.42373  2.32465  2.11815
 0.00  2800  3.50 -3.56858  0.75355  1.42323  2.29729  2.05569
 0.00  2800  4.00 -3.51492  0.96750  1.53825  2.26781  2.07231
 0.00  2800  4.50 -3.49497  1.26795  1.76250  2.28497  2.04218
 0.00  2800  5.00 -3.54027  1.67907  1.93425  2.32187  2.05317
 0.00  2800  5.50 -3.68272  2.24496  2.10360  2.43354  2.07241
 0.00  3000 -1.02 -5.47382  0.36024  1.40529  2.75583  2.84762
 0.00  3000 -0.70 -5.47382  0.36024  1.40529  2.75583  2.84762
 0.00  3000 -0.51 -5.28739  0.36026  1.40347  2.69997  2.78041
 0.00  3000 -0.29 -5.07153  0.36028  1.40136  2.63529  2.70259
 0.00  3000  0.00 -4.80249  0.36034  1.39931  2.54921  2.61283
 0.00  3000  0.28 -4.54273  0.36040  1.39732  2.46610  2.52617
 0.00  3000  0.50 -4.35102  0.36039  1.39610  2.39962  2.46743
 0.00  3000  0.60 -4.26388  0.36039  1.39555  2.36940  2.44073
 0.00  3000  0.87 -4.15648  0.40222  1.38471  2.33666  2.39391
 0.00  3000  1.00 -4.10477  0.42236  1.37950  2.32090  2.37137
 0.00  3000  1.50 -3.90588  0.49983  1.35943  2.26027  2.28466
 0.00  3000  2.00 -3.70700  0.57730  1.33937  2.19964  2.19795
 0.00  3000  2.50 -3.50811  0.65477  1.31931  2.13902  2.11125
 0.00  3000  3.00 -3.30922  0.73224  1.29925  2.07839  2.02454
 0.00  3000  3.50 -3.11034  0.80971  1.27919  2.01776  1.93783
 0.00  3000  4.00 -2.90552  0.93787  1.48985  1.97737  1.83904
 0.00  3000  4.50 -2.83379  1.12392  1.63701  1.96749  1.80477
 0.00  3000  5.00 -2.85733  1.42376  1.74949  1.98494  1.82639
 0.00  3000  5.50 -2.91549  1.80241  1.89095  2.04140  1.83255
 0.00  3200 -1.02 -4.87297  0.92479  1.40658  2.68163  2.58168
 0.00  3200 -0.70 -4.87297  0.92479  1.40658  2.68163  2.58168
 0.00  3200 -0.51 -4.65718  0.92484  1.40395  2.53605  2.53273
 0.00  3200 -0.29 -4.40743  0.92479  1.40102  2.37116  2.47323
 0.00  3200  0.00 -4.10542  0.92479  1.39755  2.18036  2.39733
 0.00  3200  0.28 -3.81383  0.92478  1.39420  1.99615  2.32405
 0.00  3200  0.50 -3.63182  0.92478  1.39169  1.88646  2.27898
 0.00  3200  0.60 -3.54908  0.92478  1.39055  1.83661  2.25850
 0.00  3200  0.87 -3.47284  0.92510  1.36337  1.83084  2.21901
 0.00  3200  1.00 -3.43612  0.92526  1.35029  1.82806  2.19999
 0.00  3200  1.50 -3.29493  0.92586  1.29996  1.81738  2.12687
 0.00  3200  2.00 -3.15373  0.92647  1.24963  1.80670  2.05374
 0.00  3200  2.50 -3.01254  0.92707  1.19930  1.79602  1.98061
 0.00  3200  3.00 -2.87134  0.92767  1.14897  1.78533  1.90748
 0.00  3200  3.50 -2.73015  0.92828  1.09865  1.77465  1.83435
 0.00  3200  4.00 -2.41408  0.97196  1.32609  1.71272  1.68227
 0.00  3200  4.50 -2.27806  1.07727  1.47048  1.68868  1.61161
 0.00  3200  5.00 -2.23010  1.24630  1.58851  1.68878  1.58464
 0.00  3200  5.50 -2.23763  1.49731  1.70081  1.71259  1.58287
 0.00  3350 -1.02 -3.70715  1.27649  1.52669  2.17898  2.15013
 0.00  3350 -0.70 -3.70715  1.27649  1.52669  2.17898  2.15013
 0.00  3350 -0.51 -3.70715  1.27649  1.52669  2.17898  2.15013
 0.00  3350 -0.29 -3.53199  1.27643  1.51682  2.05070  2.10716
 0.00  3350  0.00 -3.33223  1.27647  1.50696  1.90831  2.05377
 0.00  3350  0.28 -3.13937  1.27651  1.49743  1.77084  2.00222
 0.00  3350  0.50 -3.12220  1.27648  1.49498  1.74974  2.00430
 0.00  3350  0.60 -3.11440  1.27647  1.49387  1.74015  2.00524
 0.00  3350  0.87 -3.09301  1.27651  1.49104  1.71444  2.00711
 0.00  3350  1.00 -3.05528  1.26360  1.46738  1.70672  1.99096
 0.00  3350  1.50 -2.91017  1.21392  1.37640  1.67702  1.92887
 0.00  3350  2.00 -2.76507  1.16425  1.28542  1.64733  1.86677
 0.00  3350  2.50 -2.61996  1.11457  1.19444  1.61763  1.80468
 0.00  3350  3.00 -2.47486  1.06490  1.10345  1.58794  1.74259
 0.00  3350  3.50 -2.32975  1.01523  1.01247  1.55824  1.68049
 0.00  3350  4.00 -2.07416  0.99846  1.25525  1.53782  1.53650
 0.00  3350  4.50 -1.88421  1.03459  1.39876  1.50676  1.43530
 0.00  3350  5.00 -1.80441  1.13343  1.50725  1.49931  1.38897
 0.00  3350  5.50 -1.78413  1.29917  1.60987  1.51129  1.37300
 0.00  3500 -1.02 -2.42775  1.88359  1.77156  1.56979  1.77381
 0.00  3500 -0.70 -2.42775  1.88359  1.77156  1.56979  1.77381
 0.00  3500 -0.51 -2.42775  1.88359  1.77156  1.56979  1.77381
 0.00  3500 -0.29 -2.42775  1.88359  1.77156  1.56979  1.77381
 0.00  3500  0.00 -2.42775  1.88359  1.77156  1.56979  1.77381
 0.00  3500  0.28 -2.43414  1.76440  1.70145  1.54976  1.78896
 0.00  3500  0.50 -2.43916  1.67076  1.64637  1.53403  1.80086
 0.00  3500  0.60 -2.44477  1.62737  1.62194  1.52936  1.80697
 0.00  3500  0.87 -2.45990  1.51022  1.55601  1.51675  1.82346
 0.00  3500  1.00 -2.46718  1.45381  1.52426  1.51068  1.83140
 0.00  3500  1.50 -2.50553  1.24714  1.40690  1.49750  1.86228
 0.00  3500  2.00 -2.54910  1.06512  1.29820  1.49249  1.89107
 0.00  3500  2.50 -2.59060  0.92040  1.20629  1.49472  1.91447
 0.00  3500  3.00 -1.81318  1.18365  1.34959  1.26151  1.45418
 0.00  3500  3.50 -1.65651  1.12467  1.38840  1.26877  1.38174
 0.00  3500  4.00 -1.48929  1.08922  1.43312  1.28129  1.29883
 0.00  3500  4.50 -1.41800  1.08681  1.46480  1.30089  1.25386
 0.00  3500  5.00 -1.39016  1.12450  1.50328  1.32803  1.22318
 0.00  3500  5.50 -1.39016  1.12450  1.50328  1.32803  1.22318
 0.00  3750 -1.02 -1.36398  2.30326  1.76094  1.31285  1.08092
 0.00  3750 -0.70 -1.36398  2.30326  1.76094  1.31285  1.08092
 0.00  3750 -0.51 -1.36398  2.30326  1.76094  1.31285  1.08092
 0.00  3750 -0.29 -1.36398  2.30326  1.76094  1.31285  1.08092
 0.00  3750  0.00 -1.36398  2.30326  1.76094  1.31285  1.08092
 0.00  3750  0.28 -1.35620  2.18002  1.71310  1.29809  1.08262
 0.00  3750  0.50 -1.35009  2.08318  1.67552  1.28649  1.08395
 0.00  3750  0.60 -1.34973  2.04079  1.65988  1.28227  1.08609
 0.00  3750  0.87 -1.34878  1.92633  1.61765  1.27087  1.09184
 0.00  3750  1.00 -1.34832  1.87122  1.59732  1.26538  1.09461
 0.00  3750  1.50 -1.35377  1.67709  1.52372  1.24673  1.11042
 0.00  3750  2.00 -1.36668  1.50651  1.45590  1.23111  1.13033
 0.00  3750  2.50 -1.38435  1.36371  1.39491  1.21930  1.15145
 0.00  3750  3.00 -1.20872  1.49917  1.43666  1.10665  1.08403
 0.00  3750  3.50 -1.21269  1.40565  1.41288  1.11225  1.09133
 0.00  3750  4.00 -1.15455  1.30916  1.42289  1.13532  1.06323
 0.00  3750  4.50 -1.08230  1.23087  1.43023  1.15603  1.02768
 0.00  3750  5.00 -1.04779  1.19235  1.43193  1.17145  1.00799
 0.00  3750  5.50 -1.04779  1.19235  1.43193  1.17145  1.00799
 0.00  4000 -1.02 -0.81711  2.24883  1.62711  1.05713  0.80762
 0.00  4000 -0.70 -0.81711  2.24883  1.62711  1.05713  0.80762
 0.00  4000 -0.51 -0.81711  2.24883  1.62711  1.05713  0.80762
 0.00  4000 -0.29 -0.81711  2.24883  1.62711  1.05713  0.80762
 0.00  4000  0.00 -0.81711  2.24883  1.62711  1.05713  0.80762
 0.00  4000  0.28 -0.80948  2.13357  1.58615  1.04999  0.80214
 0.00  4000  0.50 -0.80348  2.04300  1.55397  1.04439  0.79784
 0.00  4000  0.60 -0.80254  2.00487  1.54119  1.04320  0.79685
 0.00  4000  0.87 -0.79998  1.90191  1.50670  1.04001  0.79416
 0.00  4000  1.00 -0.79875  1.85234  1.49009  1.03847  0.79287
 0.00  4000  1.50 -0.79964  1.68481  1.43465  1.03559  0.79193
 0.00  4000  2.00 -0.80641  1.54390  1.38810  1.03535  0.79514
 0.00  4000  2.50 -0.81585  1.43067  1.34834  1.03518  0.80165
 0.00  4000  3.00 -0.74611  1.51124  1.38344  1.02624  0.82268
 0.00  4000  3.50 -0.76105  1.44359  1.36199  1.02891  0.83365
 0.00  4000  4.00 -0.76942  1.37927  1.35399  1.03899  0.84140
 0.00  4000  4.50 -0.75066  1.28899  1.35842  1.05812  0.83639
 0.00  4000  5.00 -0.71849  1.20205  1.35438  1.07079  0.82647
 0.00  4000  5.50 -0.71849  1.20205  1.35438  1.07079  0.82647
 0.00  4250 -1.02 -0.50486  1.97392  1.50808  0.92569  0.66519
 0.00  4250 -0.70 -0.50486  1.97392  1.50808  0.92569  0.66519
 0.00  4250 -0.51 -0.50486  1.97392  1.50808  0.92569  0.66519
 0.00  4250 -0.29 -0.50486  1.97392  1.50808  0.92569  0.66519
 0.00  4250  0.00 -0.50486  1.97392  1.50808  0.92569  0.66519
 0.00  4250  0.28 -0.49749  1.88045  1.47092  0.91762  0.65995
 0.00  4250  0.50 -0.49170  1.80702  1.44172  0.91129  0.65583
 0.00  4250  0.60 -0.49077  1.77542  1.43009  0.91002  0.65453
 0.00  4250  0.87 -0.48826  1.69012  1.39871  0.90660  0.65102
 0.00  4250  1.00 -0.48706  1.64905  1.38359  0.90495  0.64933
 0.00  4250  1.50 -0.48764  1.50816  1.33228  0.90467  0.64477
 0.00  4250  2.00 -0.49300  1.39226  1.28864  0.90887  0.64211
 0.00  4250  2.50 -0.50182  1.30467  1.25352  0.91626  0.64122
 0.00  4250  3.00 -0.48045  1.30400  1.27958  0.86540  0.68609
 0.00  4250  3.50 -0.49404  1.26690  1.26321  0.87686  0.68796
 0.00  4250  4.00 -0.50959  1.24689  1.25639  0.89032  0.69136
 0.00  4250  4.50 -0.52164  1.22208  1.25910  0.90622  0.69656
 0.00  4250  5.00 -0.51770  1.15826  1.26315  0.92147  0.70095
 0.00  4250  5.50 -0.51770  1.15826  1.26315  0.92147  0.70095
 0.00  4500 -1.02 -0.31546  1.67857  1.43260  0.77413  0.61807
 0.00  4500 -0.70 -0.31546  1.67857  1.43260  0.77413  0.61807
 0.00  4500 -0.51 -0.31546  1.67857  1.43260  0.77413  0.61807
 0.00  4500 -0.29 -0.31546  1.67857  1.43260  0.77413  0.61807
 0.00  4500  0.00 -0.31546  1.67857  1.43260  0.77413  0.61807
 0.00  4500  0.28 -0.30914  1.61088  1.40232  0.76447  0.61412
 0.00  4500  0.50 -0.30418  1.55770  1.37853  0.75687  0.61102
 0.00  4500  0.60 -0.30317  1.53405  1.36887  0.75482  0.61006
 0.00  4500  0.87 -0.30044  1.47018  1.34279  0.74929  0.60746
 0.00  4500  1.00 -0.29913  1.43943  1.33024  0.74663  0.60621
 0.00  4500  1.50 -0.29755  1.33027  1.28676  0.74169  0.60249
 0.00  4500  2.00 -0.30069  1.23677  1.24817  0.74194  0.59996
 0.00  4500  2.50 -0.30678  1.16343  1.21420  0.74604  0.59806
 0.00  4500  3.00 -0.31611  1.11301  1.18680  0.75444  0.59689
 0.00  4500  3.50 -0.32752  1.08681  1.16767  0.76603  0.59615
 0.00  4500  4.00 -0.34123  1.08078  1.15756  0.78101  0.59591
 0.00  4500  4.50 -0.35643  1.09000  1.15606  0.79832  0.59629
 0.00  4500  5.00 -0.36921  1.09634  1.16328  0.81667  0.59878
 0.00  4500  5.50 -0.36921  1.09634  1.16328  0.81667  0.59878
 0.00  4750 -1.02 -0.14345  1.33561  1.26815  0.68048  0.52180
 0.00  4750 -0.70 -0.14345  1.33561  1.26815  0.68048  0.52180
 0.00  4750 -0.51 -0.14345  1.33561  1.26815  0.68048  0.52180
 0.00  4750 -0.29 -0.14345  1.33561  1.26815  0.68048  0.52180
 0.00  4750  0.00 -0.14345  1.33561  1.26815  0.68048  0.52180
 0.00  4750  0.28 -0.13935  1.28354  1.24279  0.67061  0.51846
 0.00  4750  0.50 -0.13613  1.24263  1.22286  0.66285  0.51583
 0.00  4750  0.60 -0.13549  1.22415  1.21508  0.66067  0.51514
 0.00  4750  0.87 -0.13377  1.17426  1.19406  0.65481  0.51328
 0.00  4750  1.00 -0.13294  1.15024  1.18395  0.65198  0.51238
 0.00  4750  1.50 -0.13244  1.06993  1.14969  0.64445  0.51005
 0.00  4750  2.00 -0.13397  1.00112  1.11907  0.64043  0.50842
 0.00  4750  2.50 -0.13781  0.94543  1.09133  0.63980  0.50730
 0.00  4750  3.00 -0.14400  0.90653  1.06701  0.64288  0.50652
 0.00  4750  3.50 -0.15229  0.88525  1.04681  0.64987  0.50562
 0.00  4750  4.00 -0.16262  0.88268  1.03246  0.66081  0.50499
 0.00  4750  4.50 -0.17474  0.89653  1.02533  0.67525  0.50431
 0.00  4750  5.00 -0.18784  0.92062  1.02578  0.69231  0.50396
 0.00  4750  5.50 -0.18784  0.92062  1.02578  0.69231  0.50396
 0.00  5000 -1.02 -0.02696  1.00241  1.09667  0.62653  0.46835
 0.00  5000 -0.70 -0.02696  1.00241  1.09667  0.62653  0.46835
 0.00  5000 -0.51 -0.02696  1.00241  1.09667  0.62653  0.46835
 0.00  5000 -0.29 -0.02696  1.00241  1.09667  0.62653  0.46835
 0.00  5000  0.00 -0.02696  1.00241  1.09667  0.62653  0.46835
 0.00  5000  0.28 -0.02570  0.95702  1.07660  0.61905  0.46535
 0.00  5000  0.50 -0.02471  0.92136  1.06084  0.61318  0.46300
 0.00  5000  0.60 -0.02474  0.90596  1.05446  0.61122  0.46235
 0.00  5000  0.87 -0.02484  0.86436  1.03721  0.60593  0.46059
 0.00  5000  1.00 -0.02489  0.84434  1.02891  0.60338  0.45974
 0.00  5000  1.50 -0.02601  0.78072  1.00252  0.59556  0.45749
 0.00  5000  2.00 -0.02881  0.72521  0.97961  0.59119  0.45661
 0.00  5000  2.50 -0.03195  0.68347  0.95989  0.58832  0.45580
 0.00  5000  3.00 -0.03629  0.65445  0.94206  0.58799  0.45529
 0.00  5000  3.50 -0.04144  0.63957  0.92633  0.59009  0.45467
 0.00  5000  4.00 -0.04793  0.64003  0.91447  0.59569  0.45408
 0.00  5000  4.50 -0.05583  0.65505  0.90637  0.60477  0.45331
 0.00  5000  5.00 -0.06531  0.68078  0.90268  0.61757  0.45267
 0.00  5000  5.50 -0.06531  0.68078  0.90268  0.61757  0.45267
 0.00  5250 -1.02  0.04084  0.74138  0.93305  0.58009  0.43356
 0.00  5250 -0.70  0.04084  0.74138  0.93305  0.58009  0.43356
 0.00  5250 -0.51  0.04084  0.74138  0.93305  0.58009  0.43356
 0.00  5250 -0.29  0.04084  0.74138  0.93305  0.58009  0.43356
 0.00  5250  0.00  0.04084  0.74138  0.93305  0.58009  0.43356
 0.00  5250  0.28  0.03905  0.70467  0.91877  0.57553  0.43113
 0.00  5250  0.50  0.03764  0.67583  0.90755  0.57194  0.42921
 0.00  5250  0.60  0.03684  0.66251  0.90274  0.57069  0.42868
 0.00  5250  0.87  0.03466  0.62652  0.88974  0.56732  0.42723
 0.00  5250  1.00  0.03361  0.60920  0.88349  0.56570  0.42653
 0.00  5250  1.50  0.02944  0.55304  0.86344  0.56050  0.42446
 0.00  5250  2.00  0.02458  0.50233  0.84617  0.55732  0.42362
 0.00  5250  2.50  0.02040  0.46491  0.83270  0.55469  0.42301
 0.00  5250  3.00  0.01629  0.43875  0.82168  0.55343  0.42278
 0.00  5250  3.50  0.01224  0.42599  0.81260  0.55355  0.42253
 0.00  5250  4.00  0.00820  0.42726  0.80580  0.55567  0.42193
 0.00  5250  4.50  0.00378  0.44141  0.80091  0.56037  0.42132
 0.00  5250  5.00 -0.00135  0.46719  0.79926  0.56812  0.42059
 0.00  5250  5.50 -0.00135  0.46719  0.79926  0.56812  0.42059
 0.00  5500 -1.02  0.11573  0.57799  0.76998  0.51277  0.39728
 0.00  5500 -0.70  0.11573  0.57799  0.76998  0.51277  0.39728
 0.00  5500 -0.51  0.11573  0.57799  0.76998  0.51277  0.39728
 0.00  5500 -0.29  0.11573  0.57799  0.76998  0.51277  0.39728
 0.00  5500  0.00  0.11573  0.57799  0.76998  0.51277  0.39728
 0.00  5500  0.28  0.10903  0.53782  0.77021  0.51665  0.39615
 0.00  5500  0.50  0.10376  0.50625  0.77039  0.51971  0.39525
 0.00  5500  0.60  0.10227  0.49494  0.76724  0.51925  0.39483
 0.00  5500  0.87  0.09827  0.46439  0.75875  0.51800  0.39367
 0.00  5500  1.00  0.09634  0.44969  0.75466  0.51740  0.39311
 0.00  5500  1.50  0.08934  0.39674  0.74237  0.51588  0.39179
 0.00  5500  2.00  0.08262  0.35115  0.73153  0.51418  0.39100
 0.00  5500  2.50  0.07633  0.31306  0.72356  0.51357  0.39080
 0.00  5500  3.00  0.07092  0.28436  0.71731  0.51304  0.39078
 0.00  5500  3.50  0.06645  0.26783  0.71319  0.51297  0.39046
 0.00  5500  4.00  0.06314  0.26474  0.71068  0.51371  0.39018
 0.00  5500  4.50  0.06058  0.27370  0.71048  0.51616  0.38978
 0.00  5500  5.00  0.05838  0.29569  0.71259  0.52068  0.38919
 0.00  5500  5.50  0.05838  0.29569  0.71259  0.52068  0.38919
 0.00  5750 -1.02  0.18889  0.47545  0.62420  0.42734  0.35408
 0.00  5750 -0.70  0.18889  0.47545  0.62420  0.42734  0.35408
 0.00  5750 -0.51  0.18889  0.47545  0.62420  0.42734  0.35408
 0.00  5750 -0.29  0.18889  0.47545  0.62420  0.42734  0.35408
 0.00  5750  0.00  0.18889  0.47545  0.62420  0.42734  0.35408
 0.00  5750  0.28  0.17945  0.43634  0.62475  0.43409  0.35374
 0.00  5750  0.50  0.17204  0.40562  0.62518  0.43939  0.35348
 0.00  5750  0.60  0.16895  0.39090  0.62801  0.44232  0.35346
 0.00  5750  0.87  0.16061  0.35116  0.63563  0.45024  0.35340
 0.00  5750  1.00  0.15660  0.33203  0.63930  0.45405  0.35338
 0.00  5750  1.50  0.14688  0.28309  0.63229  0.45512  0.35226
 0.00  5750  2.00  0.13759  0.23619  0.62807  0.45682  0.35212
 0.00  5750  2.50  0.12937  0.19646  0.62559  0.45797  0.35218
 0.00  5750  3.00  0.12226  0.16350  0.62431  0.45859  0.35250
 0.00  5750  3.50  0.11639  0.14146  0.62376  0.45934  0.35275
 0.00  5750  4.00  0.11210  0.12955  0.62452  0.46030  0.35290
 0.00  5750  4.50  0.10949  0.12984  0.62740  0.46200  0.35265
 0.00  5750  5.00  0.10830  0.14319  0.63238  0.46503  0.35247
 0.00  5750  5.50  0.10830  0.14319  0.63238  0.46503  0.35247
 0.00  6000 -1.02  0.23554  0.42583  0.50198  0.35269  0.31688
 0.00  6000 -0.70  0.23554  0.42583  0.50198  0.35269  0.31688
 0.00  6000 -0.51  0.23554  0.42583  0.50198  0.35269  0.31688
 0.00  6000 -0.29  0.23554  0.42583  0.50198  0.35269  0.31688
 0.00  6000  0.00  0.23554  0.42583  0.50198  0.35269  0.31688
 0.00  6000  0.28  0.22793  0.40247  0.50784  0.35926  0.31551
 0.00  6000  0.50  0.22195  0.38411  0.51245  0.36443  0.31443
 0.00  6000  0.60  0.21868  0.37216  0.51342  0.36700  0.31439
 0.00  6000  0.87  0.20984  0.33989  0.51602  0.37392  0.31427
 0.00  6000  1.00  0.20559  0.32435  0.51728  0.37726  0.31421
 0.00  6000  1.50  0.18644  0.24186  0.54378  0.39900  0.31634
 0.00  6000  2.00  0.17570  0.19738  0.54391  0.40273  0.31596
 0.00  6000  2.50  0.16553  0.15464  0.54552  0.40578  0.31637
 0.00  6000  3.00  0.15632  0.11696  0.54852  0.40858  0.31716
 0.00  6000  3.50  0.14874  0.08692  0.55224  0.41066  0.31795
 0.00  6000  4.00  0.14287  0.06596  0.55619  0.41224  0.31864
 0.00  6000  4.50  0.13893  0.05589  0.56122  0.41425  0.31914
 0.00  6000  5.00  0.13728  0.05774  0.56801  0.41669  0.31945
 0.00  6000  5.50  0.13728  0.05774  0.56801  0.41669  0.31945
 0.00  6250 -1.02  0.27256  0.37480  0.40930  0.29820  0.27400
 0.00  6250 -0.70  0.27256  0.37480  0.40930  0.29820  0.27400
 0.00  6250 -0.51  0.27256  0.37480  0.40930  0.29820  0.27400
 0.00  6250 -0.29  0.27256  0.37480  0.40930  0.29820  0.27400
 0.00  6250  0.00  0.27256  0.37480  0.40930  0.29820  0.27400
 0.00  6250  0.28  0.27256  0.37480  0.40930  0.29820  0.27400
 0.00  6250  0.50  0.27256  0.37480  0.40930  0.29820  0.27400
 0.00  6250  0.60  0.26995  0.36632  0.41264  0.30100  0.27363
 0.00  6250  0.87  0.26291  0.34341  0.42167  0.30856  0.27262
 0.00  6250  1.00  0.25952  0.33238  0.42601  0.31219  0.27214
 0.00  6250  1.50  0.24147  0.26742  0.44211  0.33013  0.27347
 0.00  6250  2.00  0.22247  0.19122  0.47062  0.35160  0.27610
 0.00  6250  2.50  0.21064  0.14777  0.47608  0.35738  0.27634
 0.00  6250  3.00  0.19963  0.10596  0.48257  0.36199  0.27741
 0.00  6250  3.50  0.18995  0.06865  0.48967  0.36585  0.27865
 0.00  6250  4.00  0.18191  0.03850  0.49655  0.36889  0.28004
 0.00  6250  4.50  0.17628  0.01806  0.50371  0.37147  0.28118
 0.00  6250  5.00  0.17289  0.00720  0.51168  0.37426  0.28221
 0.00  6250  5.50  0.17289  0.00720  0.51168  0.37426  0.28221
 0.00  6500 -1.02  0.30928  0.34905  0.31063  0.23727  0.23850
 0.00  6500 -0.70  0.30928  0.34905  0.31063  0.23727  0.23850
 0.00  6500 -0.51  0.30928  0.34905  0.31063  0.23727  0.23850
 0.00  6500 -0.29  0.30928  0.34905  0.31063  0.23727  0.23850
 0.00  6500  0.00  0.30928  0.34905  0.31063  0.23727  0.23850
 0.00  6500  0.28  0.30928  0.34905  0.31063  0.23727  0.23850
 0.00  6500  0.50  0.30928  0.34905  0.31063  0.23727  0.23850
 0.00  6500  0.60  0.30745  0.34576  0.31529  0.24019  0.23771
 0.00  6500  0.87  0.30250  0.33686  0.32788  0.24807  0.23558
 0.00  6500  1.00  0.30012  0.33257  0.33394  0.25186  0.23456
 0.00  6500  1.50  0.28663  0.28522  0.35822  0.26847  0.23357
 0.00  6500  2.00  0.26699  0.21712  0.38089  0.28888  0.23648
 0.00  6500  2.50  0.24845  0.14567  0.41114  0.30950  0.23926
 0.00  6500  3.00  0.23540  0.10081  0.42110  0.31688  0.24037
 0.00  6500  3.50  0.22316  0.05783  0.43050  0.32264  0.24211
 0.00  6500  4.00  0.21255  0.01954  0.43966  0.32738  0.24419
 0.00  6500  4.50  0.20422 -0.01136  0.44847  0.33121  0.24623
 0.00  6500  5.00  0.19838 -0.03453  0.45767  0.33475  0.24814
 0.00  6500  5.50  0.19838 -0.03453  0.45767  0.33475  0.24814
 0.00  6750 -1.02  0.33384  0.32293  0.23311  0.18676  0.21207
 0.00  6750 -0.70  0.33384  0.32293  0.23311  0.18676  0.21207
 0.00  6750 -0.51  0.33384  0.32293  0.23311  0.18676  0.21207
 0.00  6750 -0.29  0.33384  0.32293  0.23311  0.18676  0.21207
 0.00  6750  0.00  0.33384  0.32293  0.23311  0.18676  0.21207
 0.00  6750  0.28  0.33384  0.32293  0.23311  0.18676  0.21207
 0.00  6750  0.50  0.33384  0.32293  0.23311  0.18676  0.21207
 0.00  6750  0.60  0.33230  0.32472  0.23608  0.18934  0.21118
 0.00  6750  0.87  0.32814  0.32954  0.24410  0.19630  0.20877
 0.00  6750  1.00  0.32614  0.33187  0.24797  0.19965  0.20762
 0.00  6750  1.50  0.31728  0.30981  0.27753  0.21585  0.20399
 0.00  6750  2.00  0.30321  0.25904  0.30720  0.23374  0.20399
 0.00  6750  2.50  0.28223  0.18752  0.33541  0.25603  0.20830
 0.00  6750  3.00  0.26363  0.11906  0.36540  0.27584  0.21145
 0.00  6750  3.50  0.24947  0.07178  0.37830  0.28401  0.21338
 0.00  6750  4.00  0.23617  0.02699  0.38936  0.29059  0.21601
 0.00  6750  4.50  0.22436 -0.01415  0.39937  0.29623  0.21904
 0.00  6750  5.00  0.21518 -0.04903  0.40939  0.30116  0.22189
 0.00  6750  5.50  0.21518 -0.04903  0.40939  0.30116  0.22189
 0.00  7000 -1.02  0.34307  0.27019  0.16408  0.13686  0.18671
 0.00  7000 -0.70  0.34307  0.27019  0.16408  0.13686  0.18671
 0.00  7000 -0.51  0.34307  0.27019  0.16408  0.13686  0.18671
 0.00  7000 -0.29  0.34307  0.27019  0.16408  0.13686  0.18671
 0.00  7000  0.00  0.34307  0.27019  0.16408  0.13686  0.18671
 0.00  7000  0.28  0.34307  0.27019  0.16408  0.13686  0.18671
 0.00  7000  0.50  0.34307  0.27019  0.16408  0.13686  0.18671
 0.00  7000  0.60  0.34195  0.27575  0.16613  0.13904  0.18585
 0.00  7000  0.87  0.33893  0.29075  0.17166  0.14492  0.18353
 0.00  7000  1.00  0.33748  0.29797  0.17433  0.14775  0.18241
 0.00  7000  1.50  0.33082  0.30368  0.19472  0.16057  0.17717
 0.00  7000  2.00  0.32104  0.27627  0.22961  0.17812  0.17427
 0.00  7000  2.50  0.30486  0.21817  0.26369  0.19763  0.17594
 0.00  7000  3.00  0.28166  0.13919  0.29788  0.22266  0.18167
 0.00  7000  3.50  0.26389  0.07767  0.32408  0.23878  0.18470
 0.00  7000  4.00  0.24790  0.02760  0.33838  0.24780  0.18787
 0.00  7000  4.50  0.23304 -0.02067  0.35074  0.25535  0.19149
 0.00  7000  5.00  0.22046 -0.06441  0.36186  0.26177  0.19522
 0.00  7000  5.50  0.22046 -0.06441  0.36186  0.26177  0.19522
 0.00  7250 -1.02  0.35106  0.24981  0.12473  0.09441  0.16200
 0.00  7250 -0.70  0.35106  0.24981  0.12473  0.09441  0.16200
 0.00  7250 -0.51  0.35106  0.24981  0.12473  0.09441  0.16200
 0.00  7250 -0.29  0.35106  0.24981  0.12473  0.09441  0.16200
 0.00  7250  0.00  0.35106  0.24981  0.12473  0.09441  0.16200
 0.00  7250  0.28  0.35106  0.24981  0.12473  0.09441  0.16200
 0.00  7250  0.50  0.35106  0.24981  0.12473  0.09441  0.16200
 0.00  7250  0.60  0.35013  0.25901  0.12388  0.09567  0.16138
 0.00  7250  0.87  0.34764  0.28384  0.12158  0.09907  0.15972
 0.00  7250  1.00  0.34643  0.29579  0.12047  0.10070  0.15892
 0.00  7250  1.50  0.34256  0.31945  0.13774  0.11132  0.15320
 0.00  7250  2.00  0.33586  0.32005  0.16269  0.12449  0.14839
 0.00  7250  2.50  0.32487  0.28525  0.20156  0.14340  0.14677
 0.00  7250  3.00  0.30705  0.21977  0.23895  0.16457  0.15012
 0.00  7250  3.50  0.28396  0.13884  0.27491  0.18992  0.15688
 0.00  7250  4.00  0.26624  0.07943  0.29785  0.20420  0.16059
 0.00  7250  4.50  0.24965  0.02711  0.31310  0.21365  0.16467
 0.00  7250  5.00  0.23441 -0.02346  0.32603  0.22185  0.16924
 0.00  7250  5.50  0.23441 -0.02346  0.32603  0.22185  0.16924
 0.00  7500 -1.02  0.34353  0.19535  0.12433  0.08855  0.15021
 0.00  7500 -0.70  0.34353  0.19535  0.12433  0.08855  0.15021
 0.00  7500 -0.51  0.34353  0.19535  0.12433  0.08855  0.15021
 0.00  7500 -0.29  0.34353  0.19535  0.12433  0.08855  0.15021
 0.00  7500  0.00  0.34353  0.19535  0.12433  0.08855  0.15021
 0.00  7500  0.28  0.34353  0.19535  0.12433  0.08855  0.15021
 0.00  7500  0.50  0.34353  0.19535  0.12433  0.08855  0.15021
 0.00  7500  0.60  0.34264  0.20947  0.11519  0.08680  0.15029
 0.00  7500  0.87  0.34024  0.24760  0.09049  0.08208  0.15049
 0.00  7500  1.00  0.33908  0.26596  0.07860  0.07981  0.15059
 0.00  7500  1.50  0.33764  0.30622  0.08929  0.08738  0.14492
 0.00  7500  2.00  0.33428  0.32560  0.11174  0.09791  0.13880
 0.00  7500  2.50  0.32767  0.32283  0.14130  0.11108  0.13436
 0.00  7500  3.00  0.31538  0.27930  0.18375  0.13130  0.13426
 0.00  7500  3.50  0.29519  0.20399  0.22269  0.15471  0.13994
 0.00  7500  4.00  0.27260  0.12259  0.25723  0.17858  0.14699
 0.00  7500  4.50  0.25540  0.06680  0.27750  0.19061  0.15099
 0.00  7500  5.00  0.23803  0.01065  0.29301  0.20069  0.15636
 0.00  7500  5.50  0.23803  0.01065  0.29301  0.20069  0.15636
 0.00  7750 -1.02  0.31024  0.18386  0.04459  0.07582  0.14856
 0.00  7750 -0.70  0.31024  0.18386  0.04459  0.07582  0.14856
 0.00  7750 -0.51  0.31024  0.18386  0.04459  0.07582  0.14856
 0.00  7750 -0.29  0.31024  0.18386  0.04459  0.07582  0.14856
 0.00  7750  0.00  0.31024  0.18386  0.04459  0.07582  0.14856
 0.00  7750  0.28  0.31024  0.18386  0.04459  0.07582  0.14856
 0.00  7750  0.50  0.31024  0.18386  0.04459  0.07582  0.14856
 0.00  7750  0.60  0.31024  0.18386  0.04459  0.07582  0.14856
 0.00  7750  0.87  0.31024  0.18386  0.04459  0.07582  0.14856
 0.00  7750  1.00  0.31024  0.18386  0.04459  0.07582  0.14856
 0.00  7750  1.50  0.31119  0.24033  0.04071  0.07737  0.14332
 0.00  7750  2.00  0.31060  0.27477  0.06019  0.08555  0.13652
 0.00  7750  2.50  0.30631  0.28723  0.08741  0.09654  0.13108
 0.00  7750  3.00  0.29798  0.27581  0.12066  0.11044  0.12779
 0.00  7750  3.50  0.28322  0.22065  0.16530  0.13174  0.12936
 0.00  7750  4.00  0.26032  0.13556  0.20474  0.15640  0.13683
 0.00  7750  4.50  0.23922  0.06195  0.23492  0.17585  0.14302
 0.00  7750  5.00  0.21974  0.00049  0.25438  0.18833  0.14884
 0.00  7750  5.50  0.21974  0.00049  0.25438  0.18833  0.14884
 0.00  8000 -1.02  0.27535  0.08767  0.03232  0.06843  0.13416
 0.00  8000 -0.70  0.27535  0.08767  0.03232  0.06843  0.13416
 0.00  8000 -0.51  0.27535  0.08767  0.03232  0.06843  0.13416
 0.00  8000 -0.29  0.27535  0.08767  0.03232  0.06843  0.13416
 0.00  8000  0.00  0.27535  0.08767  0.03232  0.06843  0.13416
 0.00  8000  0.28  0.27535  0.08767  0.03232  0.06843  0.13416
 0.00  8000  0.50  0.27535  0.08767  0.03232  0.06843  0.13416
 0.00  8000  0.60  0.27535  0.08767  0.03232  0.06843  0.13416
 0.00  8000  0.87  0.27535  0.08767  0.03232  0.06843  0.13416
 0.00  8000  1.00  0.27535  0.08767  0.03232  0.06843  0.13416
 0.00  8000  1.50  0.28249  0.16526  0.00333  0.05701  0.13000
 0.00  8000  2.00  0.28481  0.21275  0.01513  0.06115  0.12292
 0.00  8000  2.50  0.28365  0.23956  0.04025  0.06969  0.11638
 0.00  8000  3.00  0.27822  0.24564  0.07123  0.08071  0.11140
 0.00  8000  3.50  0.26854  0.22503  0.10905  0.09523  0.10899
 0.00  8000  4.00  0.25082  0.15544  0.15461  0.11791  0.11307
 0.00  8000  4.50  0.22642  0.06416  0.19272  0.14305  0.12177
 0.00  8000  5.00  0.20520 -0.00580  0.21829  0.15882  0.12835
 0.00  8000  5.50  0.20520 -0.00580  0.21829  0.15882  0.12835
 0.00  8250 -1.02  0.21756 -0.02959  0.03372  0.07303  0.11424
 0.00  8250 -0.70  0.21756 -0.02959  0.03372  0.07303  0.11424
 0.00  8250 -0.51  0.21756 -0.02959  0.03372  0.07303  0.11424
 0.00  8250 -0.29  0.21756 -0.02959  0.03372  0.07303  0.11424
 0.00  8250  0.00  0.21756 -0.02959  0.03372  0.07303  0.11424
 0.00  8250  0.28  0.21756 -0.02959  0.03372  0.07303  0.11424
 0.00  8250  0.50  0.21756 -0.02959  0.03372  0.07303  0.11424
 0.00  8250  0.60  0.21756 -0.02959  0.03372  0.07303  0.11424
 0.00  8250  0.87  0.21756 -0.02959  0.03372  0.07303  0.11424
 0.00  8250  1.00  0.21756 -0.02959  0.03372  0.07303  0.11424
 0.00  8250  1.50  0.24162  0.07590 -0.02014  0.04165  0.11205
 0.00  8250  2.00  0.24756  0.13512 -0.01998  0.03989  0.10503
 0.00  8250  2.50  0.24956  0.17419  0.00068  0.04520  0.09748
 0.00  8250  3.00  0.24736  0.19329  0.03043  0.05386  0.09126
 0.00  8250  3.50  0.24022  0.18994  0.06439  0.06489  0.08760
 0.00  8250  4.00  0.22828  0.15465  0.10582  0.08087  0.08702
 0.00  8250  4.50  0.20814  0.07147  0.15106  0.10481  0.09317
 0.00  8250  5.00  0.18322 -0.02147  0.18505  0.12771  0.10290
 0.00  8250  5.50  0.18322 -0.02147  0.18505  0.12771  0.10290
 0.00  8500 -1.02  0.19519 -0.00050 -0.02791  0.04743  0.10486
 0.00  8500 -0.70  0.19519 -0.00050 -0.02791  0.04743  0.10486
 0.00  8500 -0.51  0.19519 -0.00050 -0.02791  0.04743  0.10486
 0.00  8500 -0.29  0.19519 -0.00050 -0.02791  0.04743  0.10486
 0.00  8500  0.00  0.19519 -0.00050 -0.02791  0.04743  0.10486
 0.00  8500  0.28  0.19519 -0.00050 -0.02791  0.04743  0.10486
 0.00  8500  0.50  0.19519 -0.00050 -0.02791  0.04743  0.10486
 0.00  8500  0.60  0.19519 -0.00050 -0.02791  0.04743  0.10486
 0.00  8500  0.87  0.19519 -0.00050 -0.02791  0.04743  0.10486
 0.00  8500  1.00  0.19519 -0.00050 -0.02791  0.04743  0.10486
 0.00  8500  1.50  0.19519 -0.00050 -0.02791  0.04743  0.10486
 0.00  8500  2.00  0.20586  0.06956 -0.04049  0.03797  0.09821
 0.00  8500  2.50  0.21043  0.11711 -0.02667  0.03959  0.09057
 0.00  8500  3.00  0.21139  0.14586 -0.00028  0.04569  0.08358
 0.00  8500  3.50  0.20817  0.15515  0.03279  0.05445  0.07837
 0.00  8500  4.00  0.19978  0.14126  0.06970  0.06582  0.07577
 0.00  8500  4.50  0.18687  0.08939  0.11487  0.08348  0.07685
 0.00  8500  5.00  0.16572 -0.00389  0.15680  0.10754  0.08520
 0.00  8500  5.50  0.16572 -0.00389  0.15680  0.10754  0.08520
 0.00  8750 -1.02  0.16042 -0.00441 -0.05086  0.04387  0.09625
 0.00  8750 -0.70  0.16042 -0.00441 -0.05086  0.04387  0.09625
 0.00  8750 -0.51  0.16042 -0.00441 -0.05086  0.04387  0.09625
 0.00  8750 -0.29  0.16042 -0.00441 -0.05086  0.04387  0.09625
 0.00  8750  0.00  0.16042 -0.00441 -0.05086  0.04387  0.09625
 0.00  8750  0.28  0.16042 -0.00441 -0.05086  0.04387  0.09625
 0.00  8750  0.50  0.16042 -0.00441 -0.05086  0.04387  0.09625
 0.00  8750  0.60  0.16042 -0.00441 -0.05086  0.04387  0.09625
 0.00  8750  0.87  0.16042 -0.00441 -0.05086  0.04387  0.09625
 0.00  8750  1.00  0.16042 -0.00441 -0.05086  0.04387  0.09625
 0.00  8750  1.50  0.16042 -0.00441 -0.05086  0.04387  0.09625
 0.00  8750  2.00  0.16239  0.00878 -0.05254  0.04216  0.09498
 0.00  8750  2.50  0.16894  0.06262 -0.04599  0.03984  0.08739
 0.00  8750  3.00  0.17227  0.09995 -0.02390  0.04308  0.07969
 0.00  8750  3.50  0.17245  0.12000  0.00672  0.04926  0.07308
 0.00  8750  4.00  0.16845  0.11874  0.04294  0.05814  0.06884
 0.00  8750  4.50  0.15932  0.09119  0.08282  0.06998  0.06782
 0.00  8750  5.00  0.14600  0.02026  0.12826  0.08975  0.07144
 0.00  8750  5.50  0.14600  0.02026  0.12826  0.08975  0.07144
 0.00  9000 -1.02  0.09311 -0.13845 -0.03458  0.05938  0.08590
 0.00  9000 -0.70  0.09311 -0.13845 -0.03458  0.05938  0.08590
 0.00  9000 -0.51  0.09311 -0.13845 -0.03458  0.05938  0.08590
 0.00  9000 -0.29  0.09311 -0.13845 -0.03458  0.05938  0.08590
 0.00  9000  0.00  0.09311 -0.13845 -0.03458  0.05938  0.08590
 0.00  9000  0.28  0.09311 -0.13845 -0.03458  0.05938  0.08590
 0.00  9000  0.50  0.09311 -0.13845 -0.03458  0.05938  0.08590
 0.00  9000  0.60  0.09311 -0.13845 -0.03458  0.05938  0.08590
 0.00  9000  0.87  0.09311 -0.13845 -0.03458  0.05938  0.08590
 0.00  9000  1.00  0.09311 -0.13845 -0.03458  0.05938  0.08590
 0.00  9000  1.50  0.09311 -0.13845 -0.03458  0.05938  0.08590
 0.00  9000  2.00  0.11502 -0.04951 -0.06357  0.03683  0.08105
 0.00  9000  2.50  0.12309  0.00972 -0.06371  0.03054  0.07380
 0.00  9000  3.00  0.12747  0.05292 -0.04631  0.03119  0.06596
 0.00  9000  3.50  0.12958  0.08001 -0.01829  0.03538  0.05892
 0.00  9000  4.00  0.12858  0.08742  0.01602  0.04218  0.05376
 0.00  9000  4.50  0.12333  0.07264  0.05394  0.05118  0.05115
 0.00  9000  5.00  0.11244  0.03504  0.09290  0.06214  0.05186
 0.00  9000  5.50  0.11244  0.03504  0.09290  0.06214  0.05186
 0.00  9250 -1.02  0.06799 -0.10334 -0.07097  0.01949  0.05316
 0.00  9250 -0.70  0.06799 -0.10334 -0.07097  0.01949  0.05316
 0.00  9250 -0.51  0.06799 -0.10334 -0.07097  0.01949  0.05316
 0.00  9250 -0.29  0.06799 -0.10334 -0.07097  0.01949  0.05316
 0.00  9250  0.00  0.06799 -0.10334 -0.07097  0.01949  0.05316
 0.00  9250  0.28  0.06799 -0.10334 -0.07097  0.01949  0.05316
 0.00  9250  0.50  0.06799 -0.10334 -0.07097  0.01949  0.05316
 0.00  9250  0.60  0.06799 -0.10334 -0.07097  0.01949  0.05316
 0.00  9250  0.87  0.06799 -0.10334 -0.07097  0.01949  0.05316
 0.00  9250  1.00  0.06799 -0.10334 -0.07097  0.01949  0.05316
 0.00  9250  1.50  0.06799 -0.10334 -0.07097  0.01949  0.05316
 0.00  9250  2.00  0.06799 -0.10334 -0.07097  0.01949  0.05316
 0.00  9250  2.50  0.07739 -0.03948 -0.07619  0.00983  0.04648
 0.00  9250  3.00  0.08191  0.00788 -0.06366  0.00800  0.03895
 0.00  9250  3.50  0.08461  0.04031 -0.03932  0.01032  0.03172
 0.00  9250  4.00  0.08554  0.05562 -0.00713  0.01518  0.02567
 0.00  9250  4.50  0.08367  0.05007  0.03040  0.02216  0.02169
 0.00  9250  5.00  0.07726  0.02136  0.07025  0.03120  0.02093
 0.00  9250  5.50  0.07726  0.02136  0.07025  0.03120  0.02093
 0.00  9500 -1.02  0.02888 -0.14513 -0.07429  0.01380  0.03447
 0.00  9500 -0.70  0.02888 -0.14513 -0.07429  0.01380  0.03447
 0.00  9500 -0.51  0.02888 -0.14513 -0.07429  0.01380  0.03447
 0.00  9500 -0.29  0.02888 -0.14513 -0.07429  0.01380  0.03447
 0.00  9500  0.00  0.02888 -0.14513 -0.07429  0.01380  0.03447
 0.00  9500  0.28  0.02888 -0.14513 -0.07429  0.01380  0.03447
 0.00  9500  0.50  0.02888 -0.14513 -0.07429  0.01380  0.03447
 0.00  9500  0.60  0.02888 -0.14513 -0.07429  0.01380  0.03447
 0.00  9500  0.87  0.02888 -0.14513 -0.07429  0.01380  0.03447
 0.00  9500  1.00  0.02888 -0.14513 -0.07429  0.01380  0.03447
 0.00  9500  1.50  0.02888 -0.14513 -0.07429  0.01380  0.03447
 0.00  9500  2.00  0.02888 -0.14513 -0.07429  0.01380  0.03447
 0.00  9500  2.50  0.03925 -0.07771 -0.08365  0.00155  0.02841
 0.00  9500  3.00  0.04386 -0.02735 -0.07533 -0.00249  0.02122
 0.00  9500  3.50  0.04661  0.01005 -0.05477 -0.00199  0.01396
 0.00  9500  4.00  0.04853  0.03227 -0.02493  0.00115  0.00730
 0.00  9500  4.50  0.04872  0.03486  0.01136  0.00637  0.00243
 0.00  9500  5.00  0.04575  0.01431  0.05157  0.01365  0.00049
 0.00  9500  5.50  0.04575  0.01431  0.05157  0.01365  0.00049
 0.00  9750 -1.02 -0.00947 -0.18433 -0.07567  0.01239  0.02273
 0.00  9750 -0.70 -0.00947 -0.18433 -0.07567  0.01239  0.02273
 0.00  9750 -0.51 -0.00947 -0.18433 -0.07567  0.01239  0.02273
 0.00  9750 -0.29 -0.00947 -0.18433 -0.07567  0.01239  0.02273
 0.00  9750  0.00 -0.00947 -0.18433 -0.07567  0.01239  0.02273
 0.00  9750  0.28 -0.00947 -0.18433 -0.07567  0.01239  0.02273
 0.00  9750  0.50 -0.00947 -0.18433 -0.07567  0.01239  0.02273
 0.00  9750  0.60 -0.00947 -0.18433 -0.07567  0.01239  0.02273
 0.00  9750  0.87 -0.00947 -0.18433 -0.07567  0.01239  0.02273
 0.00  9750  1.00 -0.00947 -0.18433 -0.07567  0.01239  0.02273
 0.00  9750  1.50 -0.00947 -0.18433 -0.07567  0.01239  0.02273
 0.00  9750  2.00 -0.00947 -0.18433 -0.07567  0.01239  0.02273
 0.00  9750  2.50  0.00165 -0.11456 -0.08751 -0.00138  0.01729
 0.00  9750  3.00  0.00610 -0.06200 -0.08274 -0.00717  0.01064
 0.00  9750  3.50  0.00875 -0.02106 -0.06576 -0.00847  0.00337
 0.00  9750  4.00  0.01088  0.00648 -0.03881 -0.00682 -0.00349
 0.00  9750  4.50  0.01222  0.01648 -0.00422 -0.00315 -0.00893
 0.00  9750  5.00  0.01148  0.00352  0.03513  0.00254 -0.01185
 0.00  9750  5.50  0.01148  0.00352  0.03513  0.00254 -0.01185
 0.00  10000 -1.02 -0.06653 -0.23531 -0.08013  0.01421  0.01595
 0.00  10000 -0.70 -0.06653 -0.23531 -0.08013  0.01421  0.01595
 0.00  10000 -0.51 -0.06653 -0.23531 -0.08013  0.01421  0.01595
 0.00  10000 -0.29 -0.06653 -0.23531 -0.08013  0.01421  0.01595
 0.00  10000  0.00 -0.06653 -0.23531 -0.08013  0.01421  0.01595
 0.00  10000  0.28 -0.06653 -0.23531 -0.08013  0.01421  0.01595
 0.00  10000  0.50 -0.06653 -0.23531 -0.08013  0.01421  0.01595
 0.00  10000  0.60 -0.06653 -0.23531 -0.08013  0.01421  0.01595
 0.00  10000  0.87 -0.06653 -0.23531 -0.08013  0.01421  0.01595
 0.00  10000  1.00 -0.06653 -0.23531 -0.08013  0.01421  0.01595
 0.00  10000  1.50 -0.06653 -0.23531 -0.08013  0.01421  0.01595
 0.00  10000  2.00 -0.06653 -0.23531 -0.08013  0.01421  0.01595
 0.00  10000  2.50 -0.05484 -0.16329 -0.09368 -0.00064  0.01084
 0.00  10000  3.00 -0.05063 -0.10912 -0.09150 -0.00778  0.00462
 0.00  10000  3.50 -0.04863 -0.06638 -0.07735 -0.01033 -0.00234
 0.00  10000  4.00 -0.04702 -0.03496 -0.05353 -0.01019 -0.00912
 0.00  10000  4.50 -0.04536 -0.01949 -0.02116 -0.00786 -0.01493
 0.00  10000  5.00 -0.04486 -0.02530  0.01717 -0.00351 -0.01878
 0.00  10000  5.50 -0.04486 -0.02530  0.01717 -0.00351 -0.01878
 0.00  10500 -1.02 -0.18952 -0.33012 -0.09082  0.01605  0.00649
 0.00  10500 -0.70 -0.18952 -0.33012 -0.09082  0.01605  0.00649
 0.00  10500 -0.51 -0.18952 -0.33012 -0.09082  0.01605  0.00649
 0.00  10500 -0.29 -0.18952 -0.33012 -0.09082  0.01605  0.00649
 0.00  10500  0.00 -0.18952 -0.33012 -0.09082  0.01605  0.00649
 0.00  10500  0.28 -0.18952 -0.33012 -0.09082  0.01605  0.00649
 0.00  10500  0.50 -0.18952 -0.33012 -0.09082  0.01605  0.00649
 0.00  10500  0.60 -0.18952 -0.33012 -0.09082  0.01605  0.00649
 0.00  10500  0.87 -0.18952 -0.33012 -0.09082  0.01605  0.00649
 0.00  10500  1.00 -0.18952 -0.33012 -0.09082  0.01605  0.00649
 0.00  10500  1.50 -0.18952 -0.33012 -0.09082  0.01605  0.00649
 0.00  10500  2.00 -0.18952 -0.33012 -0.09082  0.01605  0.00649
 0.00  10500  2.50 -0.17681 -0.25521 -0.10642  0.00042  0.00148
 0.00  10500  3.00 -0.17349 -0.19940 -0.10735 -0.00829 -0.00393
 0.00  10500  3.50 -0.17315 -0.15498 -0.09789 -0.01274 -0.00996
 0.00  10500  4.00 -0.17302 -0.12025 -0.07934 -0.01459 -0.01619
 0.00  10500  4.50 -0.17209 -0.09724 -0.05219 -0.01436 -0.02236
 0.00  10500  5.00 -0.17069 -0.09067 -0.01690 -0.01256 -0.02745
 0.00  10500  5.50 -0.17069 -0.09067 -0.01690 -0.01256 -0.02745
 0.00  11000 -1.02 -0.30053 -0.33416 -0.11952 -0.00102 -0.00985
 0.00  11000 -0.70 -0.30053 -0.33416 -0.11952 -0.00102 -0.00985
 0.00  11000 -0.51 -0.30053 -0.33416 -0.11952 -0.00102 -0.00985
 0.00  11000 -0.29 -0.30053 -0.33416 -0.11952 -0.00102 -0.00985
 0.00  11000  0.00 -0.30053 -0.33416 -0.11952 -0.00102 -0.00985
 0.00  11000  0.28 -0.30053 -0.33416 -0.11952 -0.00102 -0.00985
 0.00  11000  0.50 -0.30053 -0.33416 -0.11952 -0.00102 -0.00985
 0.00  11000  0.60 -0.30053 -0.33416 -0.11952 -0.00102 -0.00985
 0.00  11000  0.87 -0.30053 -0.33416 -0.11952 -0.00102 -0.00985
 0.00  11000  1.00 -0.30053 -0.33416 -0.11952 -0.00102 -0.00985
 0.00  11000  1.50 -0.30053 -0.33416 -0.11952 -0.00102 -0.00985
 0.00  11000  2.00 -0.30053 -0.33416 -0.11952 -0.00102 -0.00985
 0.00  11000  2.50 -0.30053 -0.33416 -0.11952 -0.00102 -0.00985
 0.00  11000  3.00 -0.29782 -0.27799 -0.12217 -0.01008 -0.01501
 0.00  11000  3.50 -0.29884 -0.23306 -0.11540 -0.01561 -0.02037
 0.00  11000  4.00 -0.30051 -0.19685 -0.10074 -0.01873 -0.02599
 0.00  11000  4.50 -0.30115 -0.16998 -0.07821 -0.02015 -0.03175
 0.00  11000  5.00 -0.30035 -0.15610 -0.04739 -0.02011 -0.03708
 0.00  11000  5.50 -0.30035 -0.15610 -0.04739 -0.02011 -0.03708
 0.00  11500 -1.02 -0.48324 -0.42392 -0.14010 -0.00547 -0.02227
 0.00  11500 -0.70 -0.48324 -0.42392 -0.14010 -0.00547 -0.02227
 0.00  11500 -0.51 -0.48324 -0.42392 -0.14010 -0.00547 -0.02227
 0.00  11500 -0.29 -0.48324 -0.42392 -0.14010 -0.00547 -0.02227
 0.00  11500  0.00 -0.48324 -0.42392 -0.14010 -0.00547 -0.02227
 0.00  11500  0.28 -0.48324 -0.42392 -0.14010 -0.00547 -0.02227
 0.00  11500  0.50 -0.48324 -0.42392 -0.14010 -0.00547 -0.02227
 0.00  11500  0.60 -0.48324 -0.42392 -0.14010 -0.00547 -0.02227
 0.00  11500  0.87 -0.48324 -0.42392 -0.14010 -0.00547 -0.02227
 0.00  11500  1.00 -0.48324 -0.42392 -0.14010 -0.00547 -0.02227
 0.00  11500  1.50 -0.48324 -0.42392 -0.14010 -0.00547 -0.02227
 0.00  11500  2.00 -0.48324 -0.42392 -0.14010 -0.00547 -0.02227
 0.00  11500  2.50 -0.48324 -0.42392 -0.14010 -0.00547 -0.02227
 0.00  11500  3.00 -0.48178 -0.36889 -0.14352 -0.01450 -0.02722
 0.00  11500  3.50 -0.48473 -0.32468 -0.13816 -0.02029 -0.03225
 0.00  11500  4.00 -0.48866 -0.28854 -0.12587 -0.02414 -0.03736
 0.00  11500  4.50 -0.49160 -0.26069 -0.10693 -0.02647 -0.04247
 0.00  11500  5.00 -0.49248 -0.24322 -0.08041 -0.02765 -0.04747
 0.00  11500  5.50 -0.49248 -0.24322 -0.08041 -0.02765 -0.04747
 0.00  12000 -1.02 -0.44902 -0.42272 -0.12822 -0.02166 -0.03266
 0.00  12000 -0.70 -0.44902 -0.42272 -0.12822 -0.02166 -0.03266
 0.00  12000 -0.51 -0.44902 -0.42272 -0.12822 -0.02166 -0.03266
 0.00  12000 -0.29 -0.44902 -0.42272 -0.12822 -0.02166 -0.03266
 0.00  12000  0.00 -0.44902 -0.42272 -0.12822 -0.02166 -0.03266
 0.00  12000  0.28 -0.44902 -0.42272 -0.12822 -0.02166 -0.03266
 0.00  12000  0.50 -0.44902 -0.42272 -0.12822 -0.02166 -0.03266
 0.00  12000  0.60 -0.44902 -0.42272 -0.12822 -0.02166 -0.03266
 0.00  12000  0.87 -0.44902 -0.42272 -0.12822 -0.02166 -0.03266
 0.00  12000  1.00 -0.44902 -0.42272 -0.12822 -0.02166 -0.03266
 0.00  12000  1.50 -0.44902 -0.42272 -0.12822 -0.02166 -0.03266
 0.00  12000  2.00 -0.44902 -0.42272 -0.12822 -0.02166 -0.03266
 0.00  12000  2.50 -0.44902 -0.42272 -0.12822 -0.02166 -0.03266
 0.00  12000  3.00 -0.44449 -0.36997 -0.13227 -0.03021 -0.03758
 0.00  12000  3.50 -0.44498 -0.32721 -0.12801 -0.03583 -0.04248
 0.00  12000  4.00 -0.44720 -0.29154 -0.11753 -0.03988 -0.04729
 0.00  12000  4.50 -0.44964 -0.26298 -0.10120 -0.04288 -0.05178
 0.00  12000  5.00 -0.45083 -0.24282 -0.07865 -0.04487 -0.05640
 0.00  12000  5.50 -0.45083 -0.24282 -0.07865 -0.04487 -0.05640
 0.00  12500 -1.02 -0.54450 -0.46757 -0.13627 -0.02708 -0.04245
 0.00  12500 -0.70 -0.54450 -0.46757 -0.13627 -0.02708 -0.04245
 0.00  12500 -0.51 -0.54450 -0.46757 -0.13627 -0.02708 -0.04245
 0.00  12500 -0.29 -0.54450 -0.46757 -0.13627 -0.02708 -0.04245
 0.00  12500  0.00 -0.54450 -0.46757 -0.13627 -0.02708 -0.04245
 0.00  12500  0.28 -0.54450 -0.46757 -0.13627 -0.02708 -0.04245
 0.00  12500  0.50 -0.54450 -0.46757 -0.13627 -0.02708 -0.04245
 0.00  12500  0.60 -0.54450 -0.46757 -0.13627 -0.02708 -0.04245
 0.00  12500  0.87 -0.54450 -0.46757 -0.13627 -0.02708 -0.04245
 0.00  12500  1.00 -0.54450 -0.46757 -0.13627 -0.02708 -0.04245
 0.00  12500  1.50 -0.54450 -0.46757 -0.13627 -0.02708 -0.04245
 0.00  12500  2.00 -0.54450 -0.46757 -0.13627 -0.02708 -0.04245
 0.00  12500  2.50 -0.54450 -0.46757 -0.13627 -0.02708 -0.04245
 0.00  12500  3.00 -0.54016 -0.41686 -0.14082 -0.03506 -0.04704
 0.00  12500  3.50 -0.54048 -0.37641 -0.13694 -0.04040 -0.05164
 0.00  12500  4.00 -0.54269 -0.34239 -0.12742 -0.04444 -0.05621
 0.00  12500  4.50 -0.54545 -0.31446 -0.11268 -0.04765 -0.06051
 0.00  12500  5.00 -0.54741 -0.29405 -0.09250 -0.05010 -0.06473
 0.00  12500  5.50 -0.54741 -0.29405 -0.09250 -0.05010 -0.06473
 0.00  13000 -1.02 -0.63802 -0.50764 -0.14389 -0.03253 -0.05139
 0.00  13000 -0.70 -0.63802 -0.50764 -0.14389 -0.03253 -0.05139
 0.00  13000 -0.51 -0.63802 -0.50764 -0.14389 -0.03253 -0.05139
 0.00  13000 -0.29 -0.63802 -0.50764 -0.14389 -0.03253 -0.05139
 0.00  13000  0.00 -0.63802 -0.50764 -0.14389 -0.03253 -0.05139
 0.00  13000  0.28 -0.63802 -0.50764 -0.14389 -0.03253 -0.05139
 0.00  13000  0.50 -0.63802 -0.50764 -0.14389 -0.03253 -0.05139
 0.00  13000  0.60 -0.63802 -0.50764 -0.14389 -0.03253 -0.05139
 0.00  13000  0.87 -0.63802 -0.50764 -0.14389 -0.03253 -0.05139
 0.00  13000  1.00 -0.63802 -0.50764 -0.14389 -0.03253 -0.05139
 0.00  13000  1.50 -0.63802 -0.50764 -0.14389 -0.03253 -0.05139
 0.00  13000  2.00 -0.63802 -0.50764 -0.14389 -0.03253 -0.05139
 0.00  13000  2.50 -0.63802 -0.50764 -0.14389 -0.03253 -0.05139
 0.00  13000  3.00 -0.63417 -0.45813 -0.14875 -0.04024 -0.05559
 0.00  13000  3.50 -0.63469 -0.41979 -0.14542 -0.04505 -0.05997
 0.00  13000  4.00 -0.63678 -0.38792 -0.13646 -0.04880 -0.06433
 0.00  13000  4.50 -0.63939 -0.36146 -0.12262 -0.05206 -0.06851
 0.00  13000  5.00 -0.64179 -0.34138 -0.10407 -0.05468 -0.07256
 0.00  13000  5.50 -0.64179 -0.34138 -0.10407 -0.05468 -0.07256
 0.00  14000 -1.02 -0.83290 -0.66068 -0.13455 -0.02691 -0.06262
 0.00  14000 -0.70 -0.83290 -0.66068 -0.13455 -0.02691 -0.06262
 0.00  14000 -0.51 -0.83290 -0.66068 -0.13455 -0.02691 -0.06262
 0.00  14000 -0.29 -0.83290 -0.66068 -0.13455 -0.02691 -0.06262
 0.00  14000  0.00 -0.83290 -0.66068 -0.13455 -0.02691 -0.06262
 0.00  14000  0.28 -0.83290 -0.66068 -0.13455 -0.02691 -0.06262
 0.00  14000  0.50 -0.83290 -0.66068 -0.13455 -0.02691 -0.06262
 0.00  14000  0.60 -0.83290 -0.66068 -0.13455 -0.02691 -0.06262
 0.00  14000  0.87 -0.83290 -0.66068 -0.13455 -0.02691 -0.06262
 0.00  14000  1.00 -0.83290 -0.66068 -0.13455 -0.02691 -0.06262
 0.00  14000  1.50 -0.83290 -0.66068 -0.13455 -0.02691 -0.06262
 0.00  14000  2.00 -0.83290 -0.66068 -0.13455 -0.02691 -0.06262
 0.00  14000  2.50 -0.81725 -0.57804 -0.15774 -0.04302 -0.06763
 0.00  14000  3.00 -0.81575 -0.52892 -0.16329 -0.05021 -0.07122
 0.00  14000  3.50 -0.81745 -0.49356 -0.16037 -0.05441 -0.07485
 0.00  14000  4.00 -0.81974 -0.46534 -0.15235 -0.05755 -0.07875
 0.00  14000  4.50 -0.82211 -0.44239 -0.13995 -0.06025 -0.08272
 0.00  14000  5.00 -0.82445 -0.42433 -0.12354 -0.06284 -0.08647
 0.00  14000  5.50 -0.82445 -0.42433 -0.12354 -0.06284 -0.08647
 0.00  15000 -1.02 -0.98487 -0.63971 -0.17044 -0.05280 -0.08212
 0.00  15000 -0.70 -0.98487 -0.63971 -0.17044 -0.05280 -0.08212
 0.00  15000 -0.51 -0.98487 -0.63971 -0.17044 -0.05280 -0.08212
 0.00  15000 -0.29 -0.98487 -0.63971 -0.17044 -0.05280 -0.08212
 0.00  15000  0.00 -0.98487 -0.63971 -0.17044 -0.05280 -0.08212
 0.00  15000  0.28 -0.98487 -0.63971 -0.17044 -0.05280 -0.08212
 0.00  15000  0.50 -0.98487 -0.63971 -0.17044 -0.05280 -0.08212
 0.00  15000  0.60 -0.98487 -0.63971 -0.17044 -0.05280 -0.08212
 0.00  15000  0.87 -0.98487 -0.63971 -0.17044 -0.05280 -0.08212
 0.00  15000  1.00 -0.98487 -0.63971 -0.17044 -0.05280 -0.08212
 0.00  15000  1.50 -0.98487 -0.63971 -0.17044 -0.05280 -0.08212
 0.00  15000  2.00 -0.98487 -0.63971 -0.17044 -0.05280 -0.08212
 0.00  15000  2.50 -0.98487 -0.63971 -0.17044 -0.05280 -0.08212
 0.00  15000  3.00 -0.98738 -0.58859 -0.17686 -0.05976 -0.08514
 0.00  15000  3.50 -0.99145 -0.55416 -0.17427 -0.06350 -0.08815
 0.00  15000  4.00 -0.99500 -0.52846 -0.16685 -0.06594 -0.09140
 0.00  15000  4.50 -0.99749 -0.50862 -0.15541 -0.06813 -0.09485
 0.00  15000  5.00 -0.99934 -0.49348 -0.14024 -0.07039 -0.09837
 0.00  15000  5.50 -0.99934 -0.49348 -0.14024 -0.07039 -0.09837
 0.00  16000 -1.02 -1.14059 -0.69618 -0.18181 -0.06158 -0.09532
 0.00  16000 -0.70 -1.14059 -0.69618 -0.18181 -0.06158 -0.09532
 0.00  16000 -0.51 -1.14059 -0.69618 -0.18181 -0.06158 -0.09532
 0.00  16000 -0.29 -1.14059 -0.69618 -0.18181 -0.06158 -0.09532
 0.00  16000  0.00 -1.14059 -0.69618 -0.18181 -0.06158 -0.09532
 0.00  16000  0.28 -1.14059 -0.69618 -0.18181 -0.06158 -0.09532
 0.00  16000  0.50 -1.14059 -0.69618 -0.18181 -0.06158 -0.09532
 0.00  16000  0.60 -1.14059 -0.69618 -0.18181 -0.06158 -0.09532
 0.00  16000  0.87 -1.14059 -0.69618 -0.18181 -0.06158 -0.09532
 0.00  16000  1.00 -1.14059 -0.69618 -0.18181 -0.06158 -0.09532
 0.00  16000  1.50 -1.14059 -0.69618 -0.18181 -0.06158 -0.09532
 0.00  16000  2.00 -1.14059 -0.69618 -0.18181 -0.06158 -0.09532
 0.00  16000  2.50 -1.14059 -0.69618 -0.18181 -0.06158 -0.09532
 0.00  16000  3.00 -1.14804 -0.64119 -0.18953 -0.06869 -0.09787
 0.00  16000  3.50 -1.15569 -0.60632 -0.18733 -0.07197 -0.10016
 0.00  16000  4.00 -1.16114 -0.58189 -0.17999 -0.07421 -0.10281
 0.00  16000  4.50 -1.16461 -0.56379 -0.16938 -0.07589 -0.10577
 0.00  16000  5.00 -1.16642 -0.55119 -0.15513 -0.07779 -0.10888
 0.00  16000  5.50 -1.16642 -0.55119 -0.15513 -0.07779 -0.10888
 0.00  17000 -1.02 -1.28448 -0.75008 -0.19041 -0.06960 -0.10729
 0.00  17000 -0.70 -1.28448 -0.75008 -0.19041 -0.06960 -0.10729
 0.00  17000 -0.51 -1.28448 -0.75008 -0.19041 -0.06960 -0.10729
 0.00  17000 -0.29 -1.28448 -0.75008 -0.19041 -0.06960 -0.10729
 0.00  17000  0.00 -1.28448 -0.75008 -0.19041 -0.06960 -0.10729
 0.00  17000  0.28 -1.28448 -0.75008 -0.19041 -0.06960 -0.10729
 0.00  17000  0.50 -1.28448 -0.75008 -0.19041 -0.06960 -0.10729
 0.00  17000  0.60 -1.28448 -0.75008 -0.19041 -0.06960 -0.10729
 0.00  17000  0.87 -1.28448 -0.75008 -0.19041 -0.06960 -0.10729
 0.00  17000  1.00 -1.28448 -0.75008 -0.19041 -0.06960 -0.10729
 0.00  17000  1.50 -1.28448 -0.75008 -0.19041 -0.06960 -0.10729
 0.00  17000  2.00 -1.28448 -0.75008 -0.19041 -0.06960 -0.10729
 0.00  17000  2.50 -1.28448 -0.75008 -0.19041 -0.06960 -0.10729
 0.00  17000  3.00 -1.29781 -0.68934 -0.20086 -0.07719 -0.10945
 0.00  17000  3.50 -1.30996 -0.65296 -0.19935 -0.08006 -0.11120
 0.00  17000  4.00 -1.31823 -0.62839 -0.19255 -0.08176 -0.11325
 0.00  17000  4.50 -1.32318 -0.61126 -0.18237 -0.08317 -0.11572
 0.00  17000  5.00 -1.32567 -0.59994 -0.16896 -0.08464 -0.11841
 0.00  17000  5.50 -1.32567 -0.59994 -0.16896 -0.08464 -0.11841
 0.00  18000 -1.02 -1.41718 -0.80217 -0.19633 -0.07607 -0.11788
 0.00  18000 -0.70 -1.41718 -0.80217 -0.19633 -0.07607 -0.11788
 0.00  18000 -0.51 -1.41718 -0.80217 -0.19633 -0.07607 -0.11788
 0.00  18000 -0.29 -1.41718 -0.80217 -0.19633 -0.07607 -0.11788
 0.00  18000  0.00 -1.41718 -0.80217 -0.19633 -0.07607 -0.11788
 0.00  18000  0.28 -1.41718 -0.80217 -0.19633 -0.07607 -0.11788
 0.00  18000  0.50 -1.41718 -0.80217 -0.19633 -0.07607 -0.11788
 0.00  18000  0.60 -1.41718 -0.80217 -0.19633 -0.07607 -0.11788
 0.00  18000  0.87 -1.41718 -0.80217 -0.19633 -0.07607 -0.11788
 0.00  18000  1.00 -1.41718 -0.80217 -0.19633 -0.07607 -0.11788
 0.00  18000  1.50 -1.41718 -0.80217 -0.19633 -0.07607 -0.11788
 0.00  18000  2.00 -1.41718 -0.80217 -0.19633 -0.07607 -0.11788
 0.00  18000  2.50 -1.41718 -0.80217 -0.19633 -0.07607 -0.11788
 0.00  18000  3.00 -1.43626 -0.73437 -0.21032 -0.08549 -0.12023
 0.00  18000  3.50 -1.45352 -0.69612 -0.20991 -0.08821 -0.12132
 0.00  18000  4.00 -1.46526 -0.67098 -0.20369 -0.08953 -0.12287
 0.00  18000  4.50 -1.47243 -0.65376 -0.19405 -0.09058 -0.12487
 0.00  18000  5.00 -1.47611 -0.64325 -0.18136 -0.09174 -0.12713
 0.00  18000  5.50 -1.47611 -0.64325 -0.18136 -0.09174 -0.12713
 0.00  19000 -1.02 -1.54102 -0.85382 -0.20018 -0.07975 -0.12634
 0.00  19000 -0.70 -1.54102 -0.85382 -0.20018 -0.07975 -0.12634
 0.00  19000 -0.51 -1.54102 -0.85382 -0.20018 -0.07975 -0.12634
 0.00  19000 -0.29 -1.54102 -0.85382 -0.20018 -0.07975 -0.12634
 0.00  19000  0.00 -1.54102 -0.85382 -0.20018 -0.07975 -0.12634
 0.00  19000  0.28 -1.54102 -0.85382 -0.20018 -0.07975 -0.12634
 0.00  19000  0.50 -1.54102 -0.85382 -0.20018 -0.07975 -0.12634
 0.00  19000  0.60 -1.54102 -0.85382 -0.20018 -0.07975 -0.12634
 0.00  19000  0.87 -1.54102 -0.85382 -0.20018 -0.07975 -0.12634
 0.00  19000  1.00 -1.54102 -0.85382 -0.20018 -0.07975 -0.12634
 0.00  19000  1.50 -1.54102 -0.85382 -0.20018 -0.07975 -0.12634
 0.00  19000  2.00 -1.54102 -0.85382 -0.20018 -0.07975 -0.12634
 0.00  19000  2.50 -1.54102 -0.85382 -0.20018 -0.07975 -0.12634
 0.00  19000  3.00 -1.56467 -0.77668 -0.21841 -0.09257 -0.13037
 0.00  19000  3.50 -1.58751 -0.73563 -0.21967 -0.09562 -0.13101
 0.00  19000  4.00 -1.60329 -0.70966 -0.21411 -0.09661 -0.13191
 0.00  19000  4.50 -1.61296 -0.69233 -0.20486 -0.09751 -0.13339
 0.00  19000  5.00 -1.61827 -0.68185 -0.19272 -0.09849 -0.13530
 0.00  19000  5.50 -1.61827 -0.68185 -0.19272 -0.09849 -0.13530
 0.00  20000 -1.02 -1.68385 -0.81727 -0.22506 -0.09840 -0.13949
 0.00  20000 -0.70 -1.68385 -0.81727 -0.22506 -0.09840 -0.13949
 0.00  20000 -0.51 -1.68385 -0.81727 -0.22506 -0.09840 -0.13949
 0.00  20000 -0.29 -1.68385 -0.81727 -0.22506 -0.09840 -0.13949
 0.00  20000  0.00 -1.68385 -0.81727 -0.22506 -0.09840 -0.13949
 0.00  20000  0.28 -1.68385 -0.81727 -0.22506 -0.09840 -0.13949
 0.00  20000  0.50 -1.68385 -0.81727 -0.22506 -0.09840 -0.13949
 0.00  20000  0.60 -1.68385 -0.81727 -0.22506 -0.09840 -0.13949
 0.00  20000  0.87 -1.68385 -0.81727 -0.22506 -0.09840 -0.13949
 0.00  20000  1.00 -1.68385 -0.81727 -0.22506 -0.09840 -0.13949
 0.00  20000  1.50 -1.68385 -0.81727 -0.22506 -0.09840 -0.13949
 0.00  20000  2.00 -1.68385 -0.81727 -0.22506 -0.09840 -0.13949
 0.00  20000  2.50 -1.68385 -0.81727 -0.22506 -0.09840 -0.13949
 0.00  20000  3.00 -1.68385 -0.81727 -0.22506 -0.09840 -0.13949
 0.00  20000  3.50 -1.71233 -0.77251 -0.22828 -0.10235 -0.14015
 0.00  20000  4.00 -1.73255 -0.74529 -0.22378 -0.10334 -0.14047
 0.00  20000  4.50 -1.74524 -0.72765 -0.21529 -0.10388 -0.14152
 0.00  20000  5.00 -1.75256 -0.71689 -0.20378 -0.10464 -0.14304
 0.00  20000  5.50 -1.75256 -0.71689 -0.20378 -0.10464 -0.14304
 0.00  21000 -1.02 -1.79572 -0.85624 -0.23062 -0.10270 -0.14710
 0.00  21000 -0.70 -1.79572 -0.85624 -0.23062 -0.10270 -0.14710
 0.00  21000 -0.51 -1.79572 -0.85624 -0.23062 -0.10270 -0.14710
 0.00  21000 -0.29 -1.79572 -0.85624 -0.23062 -0.10270 -0.14710
 0.00  21000  0.00 -1.79572 -0.85624 -0.23062 -0.10270 -0.14710
 0.00  21000  0.28 -1.79572 -0.85624 -0.23062 -0.10270 -0.14710
 0.00  21000  0.50 -1.79572 -0.85624 -0.23062 -0.10270 -0.14710
 0.00  21000  0.60 -1.79572 -0.85624 -0.23062 -0.10270 -0.14710
 0.00  21000  0.87 -1.79572 -0.85624 -0.23062 -0.10270 -0.14710
 0.00  21000  1.00 -1.79572 -0.85624 -0.23062 -0.10270 -0.14710
 0.00  21000  1.50 -1.79572 -0.85624 -0.23062 -0.10270 -0.14710
 0.00  21000  2.00 -1.79572 -0.85624 -0.23062 -0.10270 -0.14710
 0.00  21000  2.50 -1.79572 -0.85624 -0.23062 -0.10270 -0.14710
 0.00  21000  3.00 -1.79572 -0.85624 -0.23062 -0.10270 -0.14710
 0.00  21000  3.50 -1.82879 -0.80780 -0.23586 -0.10821 -0.14863
 0.00  21000  4.00 -1.85345 -0.77870 -0.23247 -0.10957 -0.14879
 0.00  21000  4.50 -1.86947 -0.76038 -0.22474 -0.11001 -0.14931
 0.00  21000  5.00 -1.87899 -0.74941 -0.21400 -0.11056 -0.15034
 0.00  21000  5.50 -1.87899 -0.74941 -0.21400 -0.11056 -0.15034
 0.00  22000 -1.02 -1.90195 -0.89185 -0.23574 -0.10618 -0.15340
 0.00  22000 -0.70 -1.90195 -0.89185 -0.23574 -0.10618 -0.15340
 0.00  22000 -0.51 -1.90195 -0.89185 -0.23574 -0.10618 -0.15340
 0.00  22000 -0.29 -1.90195 -0.89185 -0.23574 -0.10618 -0.15340
 0.00  22000  0.00 -1.90195 -0.89185 -0.23574 -0.10618 -0.15340
 0.00  22000  0.28 -1.90195 -0.89185 -0.23574 -0.10618 -0.15340
 0.00  22000  0.50 -1.90195 -0.89185 -0.23574 -0.10618 -0.15340
 0.00  22000  0.60 -1.90195 -0.89185 -0.23574 -0.10618 -0.15340
 0.00  22000  0.87 -1.90195 -0.89185 -0.23574 -0.10618 -0.15340
 0.00  22000  1.00 -1.90195 -0.89185 -0.23574 -0.10618 -0.15340
 0.00  22000  1.50 -1.90195 -0.89185 -0.23574 -0.10618 -0.15340
 0.00  22000  2.00 -1.90195 -0.89185 -0.23574 -0.10618 -0.15340
 0.00  22000  2.50 -1.90195 -0.89185 -0.23574 -0.10618 -0.15340
 0.00  22000  3.00 -1.90195 -0.89185 -0.23574 -0.10618 -0.15340
 0.00  22000  3.50 -1.93782 -0.84145 -0.24209 -0.11312 -0.15611
 0.00  22000  4.00 -1.96652 -0.81038 -0.24015 -0.11522 -0.15651
 0.00  22000  4.50 -1.98614 -0.79109 -0.23348 -0.11574 -0.15675
 0.00  22000  5.00 -1.99816 -0.77972 -0.22350 -0.11618 -0.15746
 0.00  22000  5.50 -1.99816 -0.77972 -0.22350 -0.11618 -0.15746
 0.00  23000 -1.02 -2.00421 -0.92276 -0.24142 -0.10992 -0.15891
 0.00  23000 -0.70 -2.00421 -0.92276 -0.24142 -0.10992 -0.15891
 0.00  23000 -0.51 -2.00421 -0.92276 -0.24142 -0.10992 -0.15891
 0.00  23000 -0.29 -2.00421 -0.92276 -0.24142 -0.10992 -0.15891
 0.00  23000  0.00 -2.00421 -0.92276 -0.24142 -0.10992 -0.15891
 0.00  23000  0.28 -2.00421 -0.92276 -0.24142 -0.10992 -0.15891
 0.00  23000  0.50 -2.00421 -0.92276 -0.24142 -0.10992 -0.15891
 0.00  23000  0.60 -2.00421 -0.92276 -0.24142 -0.10992 -0.15891
 0.00  23000  0.87 -2.00421 -0.92276 -0.24142 -0.10992 -0.15891
 0.00  23000  1.00 -2.00421 -0.92276 -0.24142 -0.10992 -0.15891
 0.00  23000  1.50 -2.00421 -0.92276 -0.24142 -0.10992 -0.15891
 0.00  23000  2.00 -2.00421 -0.92276 -0.24142 -0.10992 -0.15891
 0.00  23000  2.50 -2.00421 -0.92276 -0.24142 -0.10992 -0.15891
 0.00  23000  3.00 -2.00421 -0.92276 -0.24142 -0.10992 -0.15891
 0.00  23000  3.50 -2.04094 -0.87269 -0.24783 -0.11750 -0.16238
 0.00  23000  4.00 -2.07256 -0.84046 -0.24687 -0.12022 -0.16355
 0.00  23000  4.50 -2.09544 -0.81991 -0.24128 -0.12104 -0.16382
 0.00  23000  5.00 -2.11004 -0.80797 -0.23216 -0.12158 -0.16427
 0.00  23000  5.50 -2.11004 -0.80797 -0.23216 -0.12158 -0.16427
 0.00  24000 -1.02 -2.10089 -0.94975 -0.24748 -0.11421 -0.16471
 0.00  24000 -0.70 -2.10089 -0.94975 -0.24748 -0.11421 -0.16471
 0.00  24000 -0.51 -2.10089 -0.94975 -0.24748 -0.11421 -0.16471
 0.00  24000 -0.29 -2.10089 -0.94975 -0.24748 -0.11421 -0.16471
 0.00  24000  0.00 -2.10089 -0.94975 -0.24748 -0.11421 -0.16471
 0.00  24000  0.28 -2.10089 -0.94975 -0.24748 -0.11421 -0.16471
 0.00  24000  0.50 -2.10089 -0.94975 -0.24748 -0.11421 -0.16471
 0.00  24000  0.60 -2.10089 -0.94975 -0.24748 -0.11421 -0.16471
 0.00  24000  0.87 -2.10089 -0.94975 -0.24748 -0.11421 -0.16471
 0.00  24000  1.00 -2.10089 -0.94975 -0.24748 -0.11421 -0.16471
 0.00  24000  1.50 -2.10089 -0.94975 -0.24748 -0.11421 -0.16471
 0.00  24000  2.00 -2.10089 -0.94975 -0.24748 -0.11421 -0.16471
 0.00  24000  2.50 -2.10089 -0.94975 -0.24748 -0.11421 -0.16471
 0.00  24000  3.00 -2.10089 -0.94975 -0.24748 -0.11421 -0.16471
 0.00  24000  3.50 -2.13990 -0.90039 -0.25375 -0.12150 -0.16817
 0.00  24000  4.00 -2.17300 -0.86865 -0.25299 -0.12461 -0.16961
 0.00  24000  4.50 -2.19806 -0.84741 -0.24811 -0.12604 -0.17044
 0.00  24000  5.00 -2.21516 -0.83444 -0.24008 -0.12673 -0.17084
 0.00  24000  5.50 -2.21516 -0.83444 -0.24008 -0.12673 -0.17084
 0.00  25000 -1.02 -2.18947 -0.97592 -0.25234 -0.11774 -0.17067
 0.00  25000 -0.70 -2.18947 -0.97592 -0.25234 -0.11774 -0.17067
 0.00  25000 -0.51 -2.18947 -0.97592 -0.25234 -0.11774 -0.17067
 0.00  25000 -0.29 -2.18947 -0.97592 -0.25234 -0.11774 -0.17067
 0.00  25000  0.00 -2.18947 -0.97592 -0.25234 -0.11774 -0.17067
 0.00  25000  0.28 -2.18947 -0.97592 -0.25234 -0.11774 -0.17067
 0.00  25000  0.50 -2.18947 -0.97592 -0.25234 -0.11774 -0.17067
 0.00  25000  0.60 -2.18947 -0.97592 -0.25234 -0.11774 -0.17067
 0.00  25000  0.87 -2.18947 -0.97592 -0.25234 -0.11774 -0.17067
 0.00  25000  1.00 -2.18947 -0.97592 -0.25234 -0.11774 -0.17067
 0.00  25000  1.50 -2.18947 -0.97592 -0.25234 -0.11774 -0.17067
 0.00  25000  2.00 -2.18947 -0.97592 -0.25234 -0.11774 -0.17067
 0.00  25000  2.50 -2.18947 -0.97592 -0.25234 -0.11774 -0.17067
 0.00  25000  3.00 -2.18947 -0.97592 -0.25234 -0.11774 -0.17067
 0.00  25000  3.50 -2.23446 -0.92441 -0.26023 -0.12590 -0.17388
 0.00  25000  4.00 -2.26891 -0.89416 -0.25905 -0.12875 -0.17522
 0.00  25000  4.50 -2.29524 -0.87323 -0.25464 -0.13042 -0.17612
 0.00  25000  5.00 -2.31406 -0.86004 -0.24720 -0.13145 -0.17691
 0.00  25000  5.50 -2.31406 -0.86004 -0.24720 -0.13145 -0.17691
 0.00  26000 -1.02 -2.27133 -1.00313 -0.25431 -0.11870 -0.17481
 0.00  26000 -0.70 -2.27133 -1.00313 -0.25431 -0.11870 -0.17481
 0.00  26000 -0.51 -2.27133 -1.00313 -0.25431 -0.11870 -0.17481
 0.00  26000 -0.29 -2.27133 -1.00313 -0.25431 -0.11870 -0.17481
 0.00  26000  0.00 -2.27133 -1.00313 -0.25431 -0.11870 -0.17481
 0.00  26000  0.28 -2.27133 -1.00313 -0.25431 -0.11870 -0.17481
 0.00  26000  0.50 -2.27133 -1.00313 -0.25431 -0.11870 -0.17481
 0.00  26000  0.60 -2.27133 -1.00313 -0.25431 -0.11870 -0.17481
 0.00  26000  0.87 -2.27133 -1.00313 -0.25431 -0.11870 -0.17481
 0.00  26000  1.00 -2.27133 -1.00313 -0.25431 -0.11870 -0.17481
 0.00  26000  1.50 -2.27133 -1.00313 -0.25431 -0.11870 -0.17481
 0.00  26000  2.00 -2.27133 -1.00313 -0.25431 -0.11870 -0.17481
 0.00  26000  2.50 -2.27133 -1.00313 -0.25431 -0.11870 -0.17481
 0.00  26000  3.00 -2.27133 -1.00313 -0.25431 -0.11870 -0.17481
 0.00  26000  3.50 -2.32362 -0.94575 -0.26689 -0.13101 -0.17989
 0.00  26000  4.00 -2.36083 -0.91679 -0.26540 -0.13271 -0.18062
 0.00  26000  4.50 -2.38769 -0.89711 -0.26079 -0.13452 -0.18157
 0.00  26000  5.00 -2.40750 -0.88394 -0.25393 -0.13581 -0.18231
 0.00  26000  5.50 -2.40750 -0.88394 -0.25393 -0.13581 -0.18231
 0.00  27000 -1.02 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
 0.00  27000 -0.70 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
 0.00  27000 -0.51 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
 0.00  27000 -0.29 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
 0.00  27000  0.00 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
 0.00  27000  0.28 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
 0.00  27000  0.50 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
 0.00  27000  0.60 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
 0.00  27000  0.87 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
 0.00  27000  1.00 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
 0.00  27000  1.50 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
 0.00  27000  2.00 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
 0.00  27000  2.50 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
 0.00  27000  3.00 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
 0.00  27000  3.50 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
 0.00  27000  4.00 -2.44762 -0.93680 -0.27204 -0.13755 -0.18642
 0.00  27000  4.50 -2.47615 -0.91840 -0.26722 -0.13876 -0.18637
 0.00  27000  5.00 -2.49617 -0.90623 -0.26042 -0.13984 -0.18750
 0.00  27000  5.50 -2.49617 -0.90623 -0.26042 -0.13984 -0.18750
 0.00  28000 -1.02 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
 0.00  28000 -0.70 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
 0.00  28000 -0.51 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
 0.00  28000 -0.29 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
 0.00  28000  0.00 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
 0.00  28000  0.28 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
 0.00  28000  0.50 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
 0.00  28000  0.60 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
 0.00  28000  0.87 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
 0.00  28000  1.00 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
 0.00  28000  1.50 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
 0.00  28000  2.00 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
 0.00  28000  2.50 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
 0.00  28000  3.00 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
 0.00  28000  3.50 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
 0.00  28000  4.00 -2.52892 -0.95542 -0.27879 -0.14239 -0.19191
 0.00  28000  4.50 -2.55968 -0.93767 -0.27357 -0.14305 -0.19198
 0.00  28000  5.00 -2.58074 -0.92661 -0.26689 -0.14394 -0.19221
 0.00  28000  5.50 -2.58074 -0.92661 -0.26689 -0.14394 -0.19221
 0.00  29000 -1.02 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
 0.00  29000 -0.70 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
 0.00  29000 -0.51 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
 0.00  29000 -0.29 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
 0.00  29000  0.00 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
 0.00  29000  0.28 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
 0.00  29000  0.50 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
 0.00  29000  0.60 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
 0.00  29000  0.87 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
 0.00  29000  1.00 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
 0.00  29000  1.50 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
 0.00  29000  2.00 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
 0.00  29000  2.50 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
 0.00  29000  3.00 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
 0.00  29000  3.50 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
 0.00  29000  4.00 -2.60355 -0.97348 -0.28526 -0.14669 -0.19743
 0.00  29000  4.50 -2.63831 -0.95559 -0.28024 -0.14743 -0.19725
 0.00  29000  5.00 -2.66049 -0.94535 -0.27337 -0.14777 -0.19732
 0.00  29000  5.50 -2.66049 -0.94535 -0.27337 -0.14777 -0.19732
 0.00  30000 -1.02 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
 0.00  30000 -0.70 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
 0.00  30000 -0.51 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
 0.00  30000 -0.29 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
 0.00  30000  0.00 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
 0.00  30000  0.28 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
 0.00  30000  0.50 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
 0.00  30000  0.60 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
 0.00  30000  0.87 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
 0.00  30000  1.00 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
 0.00  30000  1.50 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
 0.00  30000  2.00 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
 0.00  30000  2.50 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
 0.00  30000  3.00 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
 0.00  30000  3.50 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
 0.00  30000  4.00 -2.67342 -0.99115 -0.29078 -0.15018 -0.20214
 0.00  30000  4.50 -2.71144 -0.97261 -0.28665 -0.15153 -0.20230
 0.00  30000  5.00 -2.73584 -0.96265 -0.27981 -0.15176 -0.20218
 0.00  30000  5.50 -2.73584 -0.96265 -0.27981 -0.15176 -0.20218
 0.00  31000 -1.02 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
 0.00  31000 -0.70 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
 0.00  31000 -0.51 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
 0.00  31000 -0.29 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
 0.00  31000  0.00 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
 0.00  31000  0.28 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
 0.00  31000  0.50 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
 0.00  31000  0.60 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
 0.00  31000  0.87 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
 0.00  31000  1.00 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
 0.00  31000  1.50 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
 0.00  31000  2.00 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
 0.00  31000  2.50 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
 0.00  31000  3.00 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
 0.00  31000  3.50 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
 0.00  31000  4.00 -2.74139 -1.00792 -0.29531 -0.15236 -0.20587
 0.00  31000  4.50 -2.78037 -0.98904 -0.29260 -0.15499 -0.20676
 0.00  31000  5.00 -2.80661 -0.97910 -0.28612 -0.15544 -0.20660
 0.00  31000  5.50 -2.80661 -0.97910 -0.28612 -0.15544 -0.20660
 0.00  32000 -1.02 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
 0.00  32000 -0.70 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
 0.00  32000 -0.51 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
 0.00  32000 -0.29 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
 0.00  32000  0.00 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
 0.00  32000  0.28 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
 0.00  32000  0.50 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
 0.00  32000  0.60 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
 0.00  32000  0.87 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
 0.00  32000  1.00 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
 0.00  32000  1.50 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
 0.00  32000  2.00 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
 0.00  32000  2.50 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
 0.00  32000  3.00 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
 0.00  32000  3.50 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
 0.00  32000  4.00 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
 0.00  32000  4.50 -2.84690 -1.00455 -0.29779 -0.15760 -0.21048
 0.00  32000  5.00 -2.87420 -0.99452 -0.29207 -0.15854 -0.21056
 0.00  32000  5.50 -2.87420 -0.99452 -0.29207 -0.15854 -0.21056
 0.00  33000 -1.02 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
 0.00  33000 -0.70 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
 0.00  33000 -0.51 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
 0.00  33000 -0.29 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
 0.00  33000  0.00 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
 0.00  33000  0.28 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
 0.00  33000  0.50 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
 0.00  33000  0.60 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
 0.00  33000  0.87 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
 0.00  33000  1.00 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
 0.00  33000  1.50 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
 0.00  33000  2.00 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
 0.00  33000  2.50 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
 0.00  33000  3.00 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
 0.00  33000  3.50 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
 0.00  33000  4.00 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
 0.00  33000  4.50 -2.91195 -1.01898 -0.30219 -0.15942 -0.21359
 0.00  33000  5.00 -2.93942 -1.00904 -0.29737 -0.16111 -0.21400
 0.00  33000  5.50 -2.93942 -1.00904 -0.29737 -0.16111 -0.21400
 0.00  34000 -1.02 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
 0.00  34000 -0.70 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
 0.00  34000 -0.51 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
 0.00  34000 -0.29 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
 0.00  34000  0.00 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
 0.00  34000  0.28 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
 0.00  34000  0.50 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
 0.00  34000  0.60 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
 0.00  34000  0.87 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
 0.00  34000  1.00 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
 0.00  34000  1.50 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
 0.00  34000  2.00 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
 0.00  34000  2.50 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
 0.00  34000  3.00 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
 0.00  34000  3.50 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
 0.00  34000  4.00 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
 0.00  34000  4.50 -2.97899 -1.03181 -0.30566 -0.16027 -0.21590
 0.00  34000  5.00 -3.00391 -1.02231 -0.30213 -0.16293 -0.21688
 0.00  34000  5.50 -3.00391 -1.02231 -0.30213 -0.16293 -0.21688
 0.00  35000 -1.02 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
 0.00  35000 -0.70 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
 0.00  35000 -0.51 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
 0.00  35000 -0.29 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
 0.00  35000  0.00 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
 0.00  35000  0.28 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
 0.00  35000  0.50 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
 0.00  35000  0.60 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
 0.00  35000  0.87 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
 0.00  35000  1.00 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
 0.00  35000  1.50 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
 0.00  35000  2.00 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
 0.00  35000  2.50 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
 0.00  35000  3.00 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
 0.00  35000  3.50 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
 0.00  35000  4.00 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
 0.00  35000  4.50 -3.04809 -1.04294 -0.30835 -0.16031 -0.21740
 0.00  35000  5.00 -3.06963 -1.03422 -0.30595 -0.16402 -0.21917
 0.00  35000  5.50 -3.06963 -1.03422 -0.30595 -0.16402 -0.21917
 0.00  37500 -1.02 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
 0.00  37500 -0.70 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
 0.00  37500 -0.51 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
 0.00  37500 -0.29 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
 0.00  37500  0.00 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
 0.00  37500  0.28 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
 0.00  37500  0.50 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
 0.00  37500  0.60 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
 0.00  37500  0.87 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
 0.00  37500  1.00 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
 0.00  37500  1.50 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
 0.00  37500  2.00 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
 0.00  37500  2.50 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
 0.00  37500  3.00 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
 0.00  37500  3.50 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
 0.00  37500  4.00 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
 0.00  37500  4.50 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
 0.00  37500  5.00 -3.24088 -1.05803 -0.31302 -0.16379 -0.22209
 0.00  37500  5.50 -3.24088 -1.05803 -0.31302 -0.16379 -0.22209
 0.00  40000 -1.02 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
 0.00  40000 -0.70 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
 0.00  40000 -0.51 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
 0.00  40000 -0.29 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
 0.00  40000  0.00 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
 0.00  40000  0.28 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
 0.00  40000  0.50 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
 0.00  40000  0.60 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
 0.00  40000  0.87 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
 0.00  40000  1.00 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
 0.00  40000  1.50 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
 0.00  40000  2.00 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
 0.00  40000  2.50 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
 0.00  40000  3.00 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
 0.00  40000  3.50 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
 0.00  40000  4.00 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
 0.00  40000  4.50 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
 0.00  40000  5.00 -3.42493 -1.07283 -0.31872 -0.16266 -0.22304
 0.00  40000  5.50 -3.42493 -1.07283 -0.31872 -0.16266 -0.22304
 0.00  42500 -1.02 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
 0.00  42500 -0.70 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
 0.00  42500 -0.51 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
 0.00  42500 -0.29 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
 0.00  42500  0.00 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
 0.00  42500  0.28 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
 0.00  42500  0.50 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
 0.00  42500  0.60 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
 0.00  42500  0.87 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
 0.00  42500  1.00 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
 0.00  42500  1.50 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
 0.00  42500  2.00 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
 0.00  42500  2.50 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
 0.00  42500  3.00 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
 0.00  42500  3.50 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
 0.00  42500  4.00 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
 0.00  42500  4.50 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
 0.00  42500  5.00 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
 0.00  42500  5.50 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
 0.00  45000 -1.02 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
 0.00  45000 -0.70 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
 0.00  45000 -0.51 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
 0.00  45000 -0.29 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
 0.00  45000  0.00 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
 0.00  45000  0.28 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
 0.00  45000  0.50 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
 0.00  45000  0.60 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
 0.00  45000  0.87 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
 0.00  45000  1.00 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
 0.00  45000  1.50 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
 0.00  45000  2.00 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
 0.00  45000  2.50 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
 0.00  45000  3.00 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
 0.00  45000  3.50 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
 0.00  45000  4.00 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
 0.00  45000  4.50 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
 0.00  45000  5.00 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
 0.00  45000  5.50 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
 0.00  47500 -1.02 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
 0.00  47500 -0.70 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
 0.00  47500 -0.51 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
 0.00  47500 -0.29 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
 0.00  47500  0.00 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
 0.00  47500  0.28 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
 0.00  47500  0.50 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
 0.00  47500  0.60 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
 0.00  47500  0.87 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
 0.00  47500  1.00 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
 0.00  47500  1.50 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
 0.00  47500  2.00 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
 0.00  47500  2.50 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
 0.00  47500  3.00 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
 0.00  47500  3.50 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
 0.00  47500  4.00 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
 0.00  47500  4.50 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
 0.00  47500  5.00 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
 0.00  47500  5.50 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
 0.00  50000 -1.02 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
 0.00  50000 -0.70 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
 0.00  50000 -0.51 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
 0.00  50000 -0.29 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
 0.00  50000  0.00 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
 0.00  50000  0.28 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
 0.00  50000  0.50 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
 0.00  50000  0.60 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
 0.00  50000  0.87 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
 0.00  50000  1.00 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
 0.00  50000  1.50 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
 0.00  50000  2.00 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
 0.00  50000  2.50 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
 0.00  50000  3.00 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
 0.00  50000  3.50 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
 0.00  50000  4.00 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
 0.00  50000  4.50 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
 0.00  50000  5.00 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
 0.00  50000  5.50 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
 0.30  2000 -1.02 -7.05491 -0.29783  1.12030  3.74948  3.02679
 0.30  2000 -0.70 -7.05491 -0.29783  1.12030  3.74948  3.02679
 0.30  2000 -0.51 -7.05491 -0.29783  1.12030  3.74948  3.02679
 0.30  2000 -0.29 -7.05491 -0.29783  1.12030  3.74948  3.02679
 0.30  2000  0.00 -7.05491 -0.29783  1.12030  3.74948  3.02679
 0.30  2000  0.28 -7.05491 -0.29783  1.12030  3.74948  3.02679
 0.30  2000  0.50 -7.05491 -0.29783  1.12030  3.74948  3.02679
 0.30  2000  0.60 -7.05491 -0.29783  1.12030  3.74948  3.02679
 0.30  2000  0.87 -7.05491 -0.29783  1.12030  3.74948  3.02679
 0.30  2000  1.00 -7.05491 -0.29783  1.12030  3.74948  3.02679
 0.30  2000  1.50 -7.05491 -0.29783  1.12030  3.74948  3.02679
 0.30  2000  2.00 -7.05491 -0.29783  1.12030  3.74948  3.02679
 0.30  2000  2.50 -7.05491 -0.29783  1.12030  3.74948  3.02679
 0.30  2000  3.00 -7.05491 -0.29783  1.12030  3.74948  3.02679
 0.30  2000  3.50 -7.05491 -0.29783  1.12030  3.74948  3.02679
 0.30  2000  4.00 -7.19603  0.30737  1.98438  3.83970  3.00860
 0.30  2000  4.50 -7.36220  1.35906  2.55481  3.97758  2.99233
 0.30  2000  5.00 -7.45940  2.69098  2.84060  4.13058  2.93958
 0.30  2000  5.50 -7.37559  3.82038  2.96376  4.22386  2.83140
 0.30  2200 -1.02 -5.74734  0.00490  1.08443  3.18965  2.78974
 0.30  2200 -0.70 -5.74734  0.00490  1.08443  3.18965  2.78974
 0.30  2200 -0.51 -5.74734  0.00490  1.08443  3.18965  2.78974
 0.30  2200 -0.29 -5.74734  0.00490  1.08443  3.18965  2.78974
 0.30  2200  0.00 -5.74734  0.00490  1.08443  3.18965  2.78974
 0.30  2200  0.28 -5.74734  0.00490  1.08443  3.18965  2.78974
 0.30  2200  0.50 -5.74734  0.00490  1.08443  3.18965  2.78974
 0.30  2200  0.60 -5.74734  0.00490  1.08443  3.18965  2.78974
 0.30  2200  0.87 -5.74734  0.00490  1.08443  3.18965  2.78974
 0.30  2200  1.00 -5.74734  0.00490  1.08443  3.18965  2.78974
 0.30  2200  1.50 -5.74734  0.00490  1.08443  3.18965  2.78974
 0.30  2200  2.00 -5.74734  0.00490  1.08443  3.18965  2.78974
 0.30  2200  2.50 -5.74734  0.00490  1.08443  3.18965  2.78974
 0.30  2200  3.00 -5.74734  0.00490  1.08443  3.18965  2.78974
 0.30  2200  3.50 -5.74734  0.00490  1.08443  3.18965  2.78974
 0.30  2200  4.00 -5.83881  0.47459  1.74295  3.24982  2.77385
 0.30  2200  4.50 -6.03314  1.22234  2.23193  3.37866  2.79895
 0.30  2200  5.00 -6.18971  2.28050  2.56496  3.53287  2.77720
 0.30  2200  5.50 -6.26440  3.40841  2.73528  3.68918  2.70344
 0.30  2500 -1.02 -5.99423 -0.23091  1.41747  3.08183  2.87080
 0.30  2500 -0.70 -5.60200 -0.23088  1.40344  2.89877  2.78610
 0.30  2500 -0.51 -5.37788 -0.23088  1.39685  2.79747  2.73588
 0.30  2500 -0.29 -5.11838 -0.23088  1.38922  2.68017  2.67773
 0.30  2500  0.00 -4.94869 -0.23089  1.38976  2.66402  2.63922
 0.30  2500  0.28 -4.78486 -0.23089  1.39029  2.64842  2.60205
 0.30  2500  0.50 -4.76560 -0.18071  1.37907  2.64935  2.59194
 0.30  2500  0.60 -4.75684 -0.15790  1.37396  2.64977  2.58735
 0.30  2500  0.87 -4.73319 -0.09632  1.36019  2.65092  2.57496
 0.30  2500  1.00 -4.72181 -0.06667  1.35355  2.65147  2.56899
 0.30  2500  1.50 -4.67802  0.04737  1.32804  2.65358  2.54603
 0.30  2500  2.00 -4.63424  0.16141  1.30253  2.65570  2.52308
 0.30  2500  2.50 -4.59045  0.27545  1.27702  2.65782  2.50012
 0.30  2500  3.00 -4.54667  0.38949  1.25151  2.65993  2.47717
 0.30  2500  3.50 -4.50288  0.50353  1.22600  2.66205  2.45421
 0.30  2500  4.00 -4.48830  0.86898  1.63556  2.68648  2.40440
 0.30  2500  4.50 -4.57277  1.35470  1.95371  2.74300  2.41254
 0.30  2500  5.00 -4.71326  2.01739  2.21663  2.84811  2.42855
 0.30  2500  5.50 -4.90478  2.87751  2.40505  3.01804  2.42611
 0.30  2800 -1.02 -5.77601  0.06061  1.45009  2.93050  2.86379
 0.30  2800 -0.70 -5.43583  0.06067  1.44188  2.80713  2.76515
 0.30  2800 -0.51 -5.23440  0.06063  1.43772  2.73494  2.70608
 0.30  2800 -0.29 -5.00117  0.06059  1.43290  2.65135  2.63769
 0.30  2800  0.00 -4.74361  0.06062  1.43026  2.58145  2.56286
 0.30  2800  0.28 -4.49493  0.06065  1.42772  2.51396  2.49061
 0.30  2800  0.50 -4.31873  0.06069  1.42664  2.47409  2.44065
 0.30  2800  0.60 -4.23864  0.06072  1.42615  2.45597  2.41793
 0.30  2800  0.87 -4.17625  0.12522  1.42588  2.44120  2.38421
 0.30  2800  1.00 -4.14622  0.15628  1.42575  2.43408  2.36797
 0.30  2800  1.50 -4.03069  0.27573  1.42524  2.40672  2.30551
 0.30  2800  2.00 -3.91516  0.39519  1.42474  2.37937  2.24306
 0.30  2800  2.50 -3.79963  0.51464  1.42423  2.35201  2.18060
 0.30  2800  3.00 -3.68411  0.63410  1.42373  2.32465  2.11815
 0.30  2800  3.50 -3.56858  0.75355  1.42323  2.29729  2.05569
 0.30  2800  4.00 -3.51492  0.96750  1.53825  2.26781  2.07231
 0.30  2800  4.50 -3.49497  1.26795  1.76250  2.28497  2.04218
 0.30  2800  5.00 -3.54027  1.67907  1.93425  2.32187  2.05317
 0.30  2800  5.50 -3.68272  2.24496  2.10360  2.43354  2.07241
 0.30  3000 -1.02 -5.47382  0.36024  1.40529  2.75583  2.84762
 0.30  3000 -0.70 -5.47382  0.36024  1.40529  2.75583  2.84762
 0.30  3000 -0.51 -5.28739  0.36026  1.40347  2.69997  2.78041
 0.30  3000 -0.29 -5.07153  0.36028  1.40136  2.63529  2.70259
 0.30  3000  0.00 -4.80249  0.36034  1.39931  2.54921  2.61283
 0.30  3000  0.28 -4.54273  0.36040  1.39732  2.46610  2.52617
 0.30  3000  0.50 -4.35102  0.36039  1.39610  2.39962  2.46743
 0.30  3000  0.60 -4.26388  0.36039  1.39555  2.36940  2.44073
 0.30  3000  0.87 -4.15648  0.40222  1.38471  2.33666  2.39391
 0.30  3000  1.00 -4.10477  0.42236  1.37950  2.32090  2.37137
 0.30  3000  1.50 -3.90588  0.49983  1.35943  2.26027  2.28466
 0.30  3000  2.00 -3.70700  0.57730  1.33937  2.19964  2.19795
 0.30  3000  2.50 -3.50811  0.65477  1.31931  2.13902  2.11125
 0.30  3000  3.00 -3.30922  0.73224  1.29925  2.07839  2.02454
 0.30  3000  3.50 -3.11034  0.80971  1.27919  2.01776  1.93783
 0.30  3000  4.00 -2.90552  0.93787  1.48985  1.97737  1.83904
 0.30  3000  4.50 -2.83379  1.12392  1.63701  1.96749  1.80477
 0.30  3000  5.00 -2.85733  1.42376  1.74949  1.98494  1.82639
 0.30  3000  5.50 -2.91549  1.80241  1.89095  2.04140  1.83255
 0.30  3200 -1.02 -4.87297  0.92479  1.40658  2.68163  2.58168
 0.30  3200 -0.70 -4.87297  0.92479  1.40658  2.68163  2.58168
 0.30  3200 -0.51 -4.65718  0.92484  1.40395  2.53605  2.53273
 0.30  3200 -0.29 -4.40743  0.92479  1.40102  2.37116  2.47323
 0.30  3200  0.00 -4.10542  0.92479  1.39755  2.18036  2.39733
 0.30  3200  0.28 -3.81383  0.92478  1.39420  1.99615  2.32405
 0.30  3200  0.50 -3.63182  0.92478  1.39169  1.88646  2.27898
 0.30  3200  0.60 -3.54908  0.92478  1.39055  1.83661  2.25850
 0.30  3200  0.87 -3.47284  0.92510  1.36337  1.83084  2.21901
 0.30  3200  1.00 -3.43612  0.92526  1.35029  1.82806  2.19999
 0.30  3200  1.50 -3.29493  0.92586  1.29996  1.81738  2.12687
 0.30  3200  2.00 -3.15373  0.92647  1.24963  1.80670  2.05374
 0.30  3200  2.50 -3.01254  0.92707  1.19930  1.79602  1.98061
 0.30  3200  3.00 -2.87134  0.92767  1.14897  1.78533  1.90748
 0.30  3200  3.50 -2.73015  0.92828  1.09865  1.77465  1.83435
 0.30  3200  4.00 -2.41408  0.97196  1.32609  1.71272  1.68227
 0.30  3200  4.50 -2.27806  1.07727  1.47048  1.68868  1.61161
 0.30  3200  5.00 -2.23010  1.24630  1.58851  1.68878  1.58464
 0.30  3200  5.50 -2.23763  1.49731  1.70081  1.71259  1.58287
 0.30  3350 -1.02 -3.70715  1.27649  1.52669  2.17898  2.15013
 0.30  3350 -0.70 -3.70715  1.27649  1.52669  2.17898  2.15013
 0.30  3350 -0.51 -3.70715  1.27649  1.52669  2.17898  2.15013
 0.30  3350 -0.29 -3.53199  1.27643  1.51682  2.05070  2.10716
 0.30  3350  0.00 -3.33223  1.27647  1.50696  1.90831  2.05377
 0.30  3350  0.28 -3.13937  1.27651  1.49743  1.77084  2.00222
 0.30  3350  0.50 -3.12220  1.27648  1.49498  1.74974  2.00430
 0.30  3350  0.60 -3.11440  1.27647  1.49387  1.74015  2.00524
 0.30  3350  0.87 -3.09301  1.27651  1.49104  1.71444  2.00711
 0.30  3350  1.00 -3.05528  1.26360  1.46738  1.70672  1.99096
 0.30  3350  1.50 -2.91017  1.21392  1.37640  1.67702  1.92887
 0.30  3350  2.00 -2.76507  1.16425  1.28542  1.64733  1.86677
 0.30  3350  2.50 -2.61996  1.11457  1.19444  1.61763  1.80468
 0.30  3350  3.00 -2.47486  1.06490  1.10345  1.58794  1.74259
 0.30  3350  3.50 -2.32975  1.01523  1.01247  1.55824  1.68049
 0.30  3350  4.00 -2.07416  0.99846  1.25525  1.53782  1.53650
 0.30  3350  4.50 -1.88421  1.03459  1.39876  1.50676  1.43530
 0.30  3350  5.00 -1.80441  1.13343  1.50725  1.49931  1.38897
 0.30  3350  5.50 -1.78413  1.29917  1.60987  1.51129  1.37300
 0.30  3500 -1.02 -2.42775  1.88359  1.77156  1.56979  1.77381
 0.30  3500 -0.70 -2.42775  1.88359  1.77156  1.56979  1.77381
 0.30  3500 -0.51 -2.42775  1.88359  1.77156  1.56979  1.77381
 0.30  3500 -0.29 -2.42775  1.88359  1.77156  1.56979  1.77381
 0.30  3500  0.00 -2.42775  1.88359  1.77156  1.56979  1.77381
 0.30  3500  0.28 -2.43414  1.76440  1.70145  1.54976  1.78896
 0.30  3500  0.50 -2.43916  1.67076  1.64637  1.53403  1.80086
 0.30  3500  0.60 -2.44477  1.62737  1.62194  1.52936  1.80697
 0.30  3500  0.87 -2.45990  1.51022  1.55601  1.51675  1.82346
 0.30  3500  1.00 -2.46718  1.45381  1.52426  1.51068  1.83140
 0.30  3500  1.50 -2.50553  1.24714  1.40690  1.49750  1.86228
 0.30  3500  2.00 -2.54910  1.06512  1.29820  1.49249  1.89107
 0.30  3500  2.50 -2.59060  0.92040  1.20629  1.49472  1.91447
 0.30  3500  3.00 -1.81318  1.18365  1.34959  1.26151  1.45418
 0.30  3500  3.50 -1.65651  1.12467  1.38840  1.26877  1.38174
 0.30  3500  4.00 -1.48929  1.08922  1.43312  1.28129  1.29883
 0.30  3500  4.50 -1.41800  1.08681  1.46480  1.30089  1.25386
 0.30  3500  5.00 -1.39016  1.12450  1.50328  1.32803  1.22318
 0.30  3500  5.50 -1.39016  1.12450  1.50328  1.32803  1.22318
 0.30  3750 -1.02 -1.36398  2.30326  1.76094  1.31285  1.08092
 0.30  3750 -0.70 -1.36398  2.30326  1.76094  1.31285  1.08092
 0.30  3750 -0.51 -1.36398  2.30326  1.76094  1.31285  1.08092
 0.30  3750 -0.29 -1.36398  2.30326  1.76094  1.31285  1.08092
 0.30  3750  0.00 -1.36398  2.30326  1.76094  1.31285  1.08092
 0.30  3750  0.28 -1.35620  2.18002  1.71310  1.29809  1.08262
 0.30  3750  0.50 -1.35009  2.08318  1.67552  1.28649  1.08395
 0.30  3750  0.60 -1.34973  2.04079  1.65988  1.28227  1.08609
 0.30  3750  0.87 -1.34878  1.92633  1.61765  1.27087  1.09184
 0.30  3750  1.00 -1.34832  1.87122  1.59732  1.26538  1.09461
 0.30  3750  1.50 -1.35377  1.67709  1.52372  1.24673  1.11042
 0.30  3750  2.00 -1.36668  1.50651  1.45590  1.23111  1.13033
 0.30  3750  2.50 -1.38435  1.36371  1.39491  1.21930  1.15145
 0.30  3750  3.00 -1.20872  1.49917  1.43666  1.10665  1.08403
 0.30  3750  3.50 -1.21269  1.40565  1.41288  1.11225  1.09133
 0.30  3750  4.00 -1.15455  1.30916  1.42289  1.13532  1.06323
 0.30  3750  4.50 -1.08230  1.23087  1.43023  1.15603  1.02768
 0.30  3750  5.00 -1.04779  1.19235  1.43193  1.17145  1.00799
 0.30  3750  5.50 -1.04779  1.19235  1.43193  1.17145  1.00799
 0.30  4000 -1.02 -0.81711  2.24883  1.62711  1.05713  0.80762
 0.30  4000 -0.70 -0.81711  2.24883  1.62711  1.05713  0.80762
 0.30  4000 -0.51 -0.81711  2.24883  1.62711  1.05713  0.80762
 0.30  4000 -0.29 -0.81711  2.24883  1.62711  1.05713  0.80762
 0.30  4000  0.00 -0.81711  2.24883  1.62711  1.05713  0.80762
 0.30  4000  0.28 -0.80948  2.13357  1.58615  1.04999  0.80214
 0.30  4000  0.50 -0.80348  2.04300  1.55397  1.04439  0.79784
 0.30  4000  0.60 -0.80254  2.00487  1.54119  1.04320  0.79685
 0.30  4000  0.87 -0.79998  1.90191  1.50670  1.04001  0.79416
 0.30  4000  1.00 -0.79875  1.85234  1.49009  1.03847  0.79287
 0.30  4000  1.50 -0.79964  1.68481  1.43465  1.03559  0.79193
 0.30  4000  2.00 -0.80641  1.54390  1.38810  1.03535  0.79514
 0.30  4000  2.50 -0.81585  1.43067  1.34834  1.03518  0.80165
 0.30  4000  3.00 -0.74611  1.51124  1.38344  1.02624  0.82268
 0.30  4000  3.50 -0.76105  1.44359  1.36199  1.02891  0.83365
 0.30  4000  4.00 -0.76942  1.37927  1.35399  1.03899  0.84140
 0.30  4000  4.50 -0.75066  1.28899  1.35842  1.05812  0.83639
 0.30  4000  5.00 -0.71849  1.20205  1.35438  1.07079  0.82647
 0.30  4000  5.50 -0.71849  1.20205  1.35438  1.07079  0.82647
 0.30  4250 -1.02 -0.50486  1.97392  1.50808  0.92569  0.66519
 0.30  4250 -0.70 -0.50486  1.97392  1.50808  0.92569  0.66519
 0.30  4250 -0.51 -0.50486  1.97392  1.50808  0.92569  0.66519
 0.30  4250 -0.29 -0.50486  1.97392  1.50808  0.92569  0.66519
 0.30  4250  0.00 -0.50486  1.97392  1.50808  0.92569  0.66519
 0.30  4250  0.28 -0.49749  1.88045  1.47092  0.91762  0.65995
 0.30  4250  0.50 -0.49170  1.80702  1.44172  0.91129  0.65583
 0.30  4250  0.60 -0.49077  1.77542  1.43009  0.91002  0.65453
 0.30  4250  0.87 -0.48826  1.69012  1.39871  0.90660  0.65102
 0.30  4250  1.00 -0.48706  1.64905  1.38359  0.90495  0.64933
 0.30  4250  1.50 -0.48764  1.50816  1.33228  0.90467  0.64477
 0.30  4250  2.00 -0.49300  1.39226  1.28864  0.90887  0.64211
 0.30  4250  2.50 -0.50182  1.30467  1.25352  0.91626  0.64122
 0.30  4250  3.00 -0.48045  1.30400  1.27958  0.86540  0.68609
 0.30  4250  3.50 -0.49404  1.26690  1.26321  0.87686  0.68796
 0.30  4250  4.00 -0.50959  1.24689  1.25639  0.89032  0.69136
 0.30  4250  4.50 -0.52164  1.22208  1.25910  0.90622  0.69656
 0.30  4250  5.00 -0.51770  1.15826  1.26315  0.92147  0.70095
 0.30  4250  5.50 -0.51770  1.15826  1.26315  0.92147  0.70095
 0.30  4500 -1.02 -0.31546  1.67857  1.43260  0.77413  0.61807
 0.30  4500 -0.70 -0.31546  1.67857  1.43260  0.77413  0.61807
 0.30  4500 -0.51 -0.31546  1.67857  1.43260  0.77413  0.61807
 0.30  4500 -0.29 -0.31546  1.67857  1.43260  0.77413  0.61807
 0.30  4500  0.00 -0.31546  1.67857  1.43260  0.77413  0.61807
 0.30  4500  0.28 -0.30914  1.61088  1.40232  0.76447  0.61412
 0.30  4500  0.50 -0.30418  1.55770  1.37853  0.75687  0.61102
 0.30  4500  0.60 -0.30317  1.53405  1.36887  0.75482  0.61006
 0.30  4500  0.87 -0.30044  1.47018  1.34279  0.74929  0.60746
 0.30  4500  1.00 -0.29913  1.43943  1.33024  0.74663  0.60621
 0.30  4500  1.50 -0.29755  1.33027  1.28676  0.74169  0.60249
 0.30  4500  2.00 -0.30069  1.23677  1.24817  0.74194  0.59996
 0.30  4500  2.50 -0.30678  1.16343  1.21420  0.74604  0.59806
 0.30  4500  3.00 -0.31611  1.11301  1.18680  0.75444  0.59689
 0.30  4500  3.50 -0.32752  1.08681  1.16767  0.76603  0.59615
 0.30  4500  4.00 -0.34123  1.08078  1.15756  0.78101  0.59591
 0.30  4500  4.50 -0.35643  1.09000  1.15606  0.79832  0.59629
 0.30  4500  5.00 -0.36921  1.09634  1.16328  0.81667  0.59878
 0.30  4500  5.50 -0.36921  1.09634  1.16328  0.81667  0.59878
 0.30  4750 -1.02 -0.14345  1.33561  1.26815  0.68048  0.52180
 0.30  4750 -0.70 -0.14345  1.33561  1.26815  0.68048  0.52180
 0.30  4750 -0.51 -0.14345  1.33561  1.26815  0.68048  0.52180
 0.30  4750 -0.29 -0.14345  1.33561  1.26815  0.68048  0.52180
 0.30  4750  0.00 -0.14345  1.33561  1.26815  0.68048  0.52180
 0.30  4750  0.28 -0.13935  1.28354  1.24279  0.67061  0.51846
 0.30  4750  0.50 -0.13613  1.24263  1.22286  0.66285  0.51583
 0.30  4750  0.60 -0.13549  1.22415  1.21508  0.66067  0.51514
 0.30  4750  0.87 -0.13377  1.17426  1.19406  0.65481  0.51328
 0.30  4750  1.00 -0.13294  1.15024  1.18395  0.65198  0.51238
 0.30  4750  1.50 -0.13244  1.06993  1.14969  0.64445  0.51005
 0.30  4750  2.00 -0.13397  1.00112  1.11907  0.64043  0.50842
 0.30  4750  2.50 -0.13781  0.94543  1.09133  0.63980  0.50730
 0.30  4750  3.00 -0.14400  0.90653  1.06701  0.64288  0.50652
 0.30  4750  3.50 -0.15229  0.88525  1.04681  0.64987  0.50562
 0.30  4750  4.00 -0.16262  0.88268  1.03246  0.66081  0.50499
 0.30  4750  4.50 -0.17474  0.89653  1.02533  0.67525  0.50431
 0.30  4750  5.00 -0.18784  0.92062  1.02578  0.69231  0.50396
 0.30  4750  5.50 -0.18784  0.92062  1.02578  0.69231  0.50396
 0.30  5000 -1.02 -0.02696  1.00241  1.09667  0.62653  0.46835
 0.30  5000 -0.70 -0.02696  1.00241  1.09667  0.62653  0.46835
 0.30  5000 -0.51 -0.02696  1.00241  1.09667  0.62653  0.46835
 0.30  5000 -0.29 -0.02696  1.00241  1.09667  0.62653  0.46835
 0.30  5000  0.00 -0.02696  1.00241  1.09667  0.62653  0.46835
 0.30  5000  0.28 -0.02570  0.95702  1.07660  0.61905  0.46535
 0.30  5000  0.50 -0.02471  0.92136  1.06084  0.61318  0.46300
 0.30  5000  0.60 -0.02474  0.90596  1.05446  0.61122  0.46235
 0.30  5000  0.87 -0.02484  0.86436  1.03721  0.60593  0.46059
 0.30  5000  1.00 -0.02489  0.84434  1.02891  0.60338  0.45974
 0.30  5000  1.50 -0.02601  0.78072  1.00252  0.59556  0.45749
 0.30  5000  2.00 -0.02881  0.72521  0.97961  0.59119  0.45661
 0.30  5000  2.50 -0.03195  0.68347  0.95989  0.58832  0.45580
 0.30  5000  3.00 -0.03629  0.65445  0.94206  0.58799  0.45529
 0.30  5000  3.50 -0.04144  0.63957  0.92633  0.59009  0.45467
 0.30  5000  4.00 -0.04793  0.64003  0.91447  0.59569  0.45408
 0.30  5000  4.50 -0.05583  0.65505  0.90637  0.60477  0.45331
 0.30  5000  5.00 -0.06531  0.68078  0.90268  0.61757  0.45267
 0.30  5000  5.50 -0.06531  0.68078  0.90268  0.61757  0.45267
 0.30  5250 -1.02  0.04084  0.74138  0.93305  0.58009  0.43356
 0.30  5250 -0.70  0.04084  0.74138  0.93305  0.58009  0.43356
 0.30  5250 -0.51  0.04084  0.74138  0.93305  0.58009  0.43356
 0.30  5250 -0.29  0.04084  0.74138  0.93305  0.58009  0.43356
 0.30  5250  0.00  0.04084  0.74138  0.93305  0.58009  0.43356
 0.30  5250  0.28  0.03905  0.70467  0.91877  0.57553  0.43113
 0.30  5250  0.50  0.03764  0.67583  0.90755  0.57194  0.42921
 0.30  5250  0.60  0.03684  0.66251  0.90274  0.57069  0.42868
 0.30  5250  0.87  0.03466  0.62652  0.88974  0.56732  0.42723
 0.30  5250  1.00  0.03361  0.60920  0.88349  0.56570  0.42653
 0.30  5250  1.50  0.02944  0.55304  0.86344  0.56050  0.42446
 0.30  5250  2.00  0.02458  0.50233  0.84617  0.55732  0.42362
 0.30  5250  2.50  0.02040  0.46491  0.83270  0.55469  0.42301
 0.30  5250  3.00  0.01629  0.43875  0.82168  0.55343  0.42278
 0.30  5250  3.50  0.01224  0.42599  0.81260  0.55355  0.42253
 0.30  5250  4.00  0.00820  0.42726  0.80580  0.55567  0.42193
 0.30  5250  4.50  0.00378  0.44141  0.80091  0.56037  0.42132
 0.30  5250  5.00 -0.00135  0.46719  0.79926  0.56812  0.42059
 0.30  5250  5.50 -0.00135  0.46719  0.79926  0.56812  0.42059
 0.30  5500 -1.02  0.11573  0.57799  0.76998  0.51277  0.39728
 0.30  5500 -0.70  0.11573  0.57799  0.76998  0.51277  0.39728
 0.30  5500 -0.51  0.11573  0.57799  0.76998  0.51277  0.39728
 0.30  5500 -0.29  0.11573  0.57799  0.76998  0.51277  0.39728
 0.30  5500  0.00  0.11573  0.57799  0.76998  0.51277  0.39728
 0.30  5500  0.28  0.10903  0.53782  0.77021  0.51665  0.39615
 0.30  5500  0.50  0.10376  0.50625  0.77039  0.51971  0.39525
 0.30  5500  0.60  0.10227  0.49494  0.76724  0.51925  0.39483
 0.30  5500  0.87  0.09827  0.46439  0.75875  0.51800  0.39367
 0.30  5500  1.00  0.09634  0.44969  0.75466  0.51740  0.39311
 0.30  5500  1.50  0.08934  0.39674  0.74237  0.51588  0.39179
 0.30  5500  2.00  0.08262  0.35115  0.73153  0.51418  0.39100
 0.30  5500  2.50  0.07633  0.31306  0.72356  0.51357  0.39080
 0.30  5500  3.00  0.07092  0.28436  0.71731  0.51304  0.39078
 0.30  5500  3.50  0.06645  0.26783  0.71319  0.51297  0.39046
 0.30  5500  4.00  0.06314  0.26474  0.71068  0.51371  0.39018
 0.30  5500  4.50  0.06058  0.27370  0.71048  0.51616  0.38978
 0.30  5500  5.00  0.05838  0.29569  0.71259  0.52068  0.38919
 0.30  5500  5.50  0.05838  0.29569  0.71259  0.52068  0.38919
 0.30  5750 -1.02  0.18889  0.47545  0.62420  0.42734  0.35408
 0.30  5750 -0.70  0.18889  0.47545  0.62420  0.42734  0.35408
 0.30  5750 -0.51  0.18889  0.47545  0.62420  0.42734  0.35408
 0.30  5750 -0.29  0.18889  0.47545  0.62420  0.42734  0.35408
 0.30  5750  0.00  0.18889  0.47545  0.62420  0.42734  0.35408
 0.30  5750  0.28  0.17945  0.43634  0.62475  0.43409  0.35374
 0.30  5750  0.50  0.17204  0.40562  0.62518  0.43939  0.35348
 0.30  5750  0.60  0.16895  0.39090  0.62801  0.44232  0.35346
 0.30  5750  0.87  0.16061  0.35116  0.63563  0.45024  0.35340
 0.30  5750  1.00  0.15660  0.33203  0.63930  0.45405  0.35338
 0.30  5750  1.50  0.14688  0.28309  0.63229  0.45512  0.35226
 0.30  5750  2.00  0.13759  0.23619  0.62807  0.45682  0.35212
 0.30  5750  2.50  0.12937  0.19646  0.62559  0.45797  0.35218
 0.30  5750  3.00  0.12226  0.16350  0.62431  0.45859  0.35250
 0.30  5750  3.50  0.11639  0.14146  0.62376  0.45934  0.35275
 0.30  5750  4.00  0.11210  0.12955  0.62452  0.46030  0.35290
 0.30  5750  4.50  0.10949  0.12984  0.62740  0.46200  0.35265
 0.30  5750  5.00  0.10830  0.14319  0.63238  0.46503  0.35247
 0.30  5750  5.50  0.10830  0.14319  0.63238  0.46503  0.35247
 0.30  6000 -1.02  0.23554  0.42583  0.50198  0.35269  0.31688
 0.30  6000 -0.70  0.23554  0.42583  0.50198  0.35269  0.31688
 0.30  6000 -0.51  0.23554  0.42583  0.50198  0.35269  0.31688
 0.30  6000 -0.29  0.23554  0.42583  0.50198  0.35269  0.31688
 0.30  6000  0.00  0.23554  0.42583  0.50198  0.35269  0.31688
 0.30  6000  0.28  0.22793  0.40247  0.50784  0.35926  0.31551
 0.30  6000  0.50  0.22195  0.38411  0.51245  0.36443  0.31443
 0.30  6000  0.60  0.21868  0.37216  0.51342  0.36700  0.31439
 0.30  6000  0.87  0.20984  0.33989  0.51602  0.37392  0.31427
 0.30  6000  1.00  0.20559  0.32435  0.51728  0.37726  0.31421
 0.30  6000  1.50  0.18644  0.24186  0.54378  0.39900  0.31634
 0.30  6000  2.00  0.17570  0.19738  0.54391  0.40273  0.31596
 0.30  6000  2.50  0.16553  0.15464  0.54552  0.40578  0.31637
 0.30  6000  3.00  0.15632  0.11696  0.54852  0.40858  0.31716
 0.30  6000  3.50  0.14874  0.08692  0.55224  0.41066  0.31795
 0.30  6000  4.00  0.14287  0.06596  0.55619  0.41224  0.31864
 0.30  6000  4.50  0.13893  0.05589  0.56122  0.41425  0.31914
 0.30  6000  5.00  0.13728  0.05774  0.56801  0.41669  0.31945
 0.30  6000  5.50  0.13728  0.05774  0.56801  0.41669  0.31945
 0.30  6250 -1.02  0.27256  0.37480  0.40930  0.29820  0.27400
 0.30  6250 -0.70  0.27256  0.37480  0.40930  0.29820  0.27400
 0.30  6250 -0.51  0.27256  0.37480  0.40930  0.29820  0.27400
 0.30  6250 -0.29  0.27256  0.37480  0.40930  0.29820  0.27400
 0.30  6250  0.00  0.27256  0.37480  0.40930  0.29820  0.27400
 0.30  6250  0.28  0.27256  0.37480  0.40930  0.29820  0.27400
 0.30  6250  0.50  0.27256  0.37480  0.40930  0.29820  0.27400
 0.30  6250  0.60  0.26995  0.36632  0.41264  0.30100  0.27363
 0.30  6250  0.87  0.26291  0.34341  0.42167  0.30856  0.27262
 0.30  6250  1.00  0.25952  0.33238  0.42601  0.31219  0.27214
 0.30  6250  1.50  0.24147  0.26742  0.44211  0.33013  0.27347
 0.30  6250  2.00  0.22247  0.19122  0.47062  0.35160  0.27610
 0.30  6250  2.50  0.21064  0.14777  0.47608  0.35738  0.27634
 0.30  6250  3.00  0.19963  0.10596  0.48257  0.36199  0.27741
 0.30  6250  3.50  0.18995  0.06865  0.48967  0.36585  0.27865
 0.30  6250  4.00  0.18191  0.03850  0.49655  0.36889  0.28004
 0.30  6250  4.50  0.17628  0.01806  0.50371  0.37147  0.28118
 0.30  6250  5.00  0.17289  0.00720  0.51168  0.37426  0.28221
 0.30  6250  5.50  0.17289  0.00720  0.51168  0.37426  0.28221
 0.30  6500 -1.02  0.30928  0.34905  0.31063  0.23727  0.23850
 0.30  6500 -0.70  0.30928  0.34905  0.31063  0.23727  0.23850
 0.30  6500 -0.51  0.30928  0.34905  0.31063  0.23727  0.23850
 0.30  6500 -0.29  0.30928  0.34905  0.31063  0.23727  0.23850
 0.30  6500  0.00  0.30928  0.34905  0.31063  0.23727  0.23850
 0.30  6500  0.28  0.30928  0.34905  0.31063  0.23727  0.23850
 0.30  6500  0.50  0.30928  0.34905  0.31063  0.23727  0.23850
 0.30  6500  0.60  0.30745  0.34576  0.31529  0.24019  0.23771
 0.30  6500  0.87  0.30250  0.33686  0.32788  0.24807  0.23558
 0.30  6500  1.00  0.30012  0.33257  0.33394  0.25186  0.23456
 0.30  6500  1.50  0.28663  0.28522  0.35822  0.26847  0.23357
 0.30  6500  2.00  0.26699  0.21712  0.38089  0.28888  0.23648
 0.30  6500  2.50  0.24845  0.14567  0.41114  0.30950  0.23926
 0.30  6500  3.00  0.23540  0.10081  0.42110  0.31688  0.24037
 0.30  6500  3.50  0.22316  0.05783  0.43050  0.32264  0.24211
 0.30  6500  4.00  0.21255  0.01954  0.43966  0.32738  0.24419
 0.30  6500  4.50  0.20422 -0.01136  0.44847  0.33121  0.24623
 0.30  6500  5.00  0.19838 -0.03453  0.45767  0.33475  0.24814
 0.30  6500  5.50  0.19838 -0.03453  0.45767  0.33475  0.24814
 0.30  6750 -1.02  0.33384  0.32293  0.23311  0.18676  0.21207
 0.30  6750 -0.70  0.33384  0.32293  0.23311  0.18676  0.21207
 0.30  6750 -0.51  0.33384  0.32293  0.23311  0.18676  0.21207
 0.30  6750 -0.29  0.33384  0.32293  0.23311  0.18676  0.21207
 0.30  6750  0.00  0.33384  0.32293  0.23311  0.18676  0.21207
 0.30  6750  0.28  0.33384  0.32293  0.23311  0.18676  0.21207
 0.30  6750  0.50  0.33384  0.32293  0.23311  0.18676  0.21207
 0.30  6750  0.60  0.33230  0.32472  0.23608  0.18934  0.21118
 0.30  6750  0.87  0.32814  0.32954  0.24410  0.19630  0.20877
 0.30  6750  1.00  0.32614  0.33187  0.24797  0.19965  0.20762
 0.30  6750  1.50  0.31728  0.30981  0.27753  0.21585  0.20399
 0.30  6750  2.00  0.30321  0.25904  0.30720  0.23374  0.20399
 0.30  6750  2.50  0.28223  0.18752  0.33541  0.25603  0.20830
 0.30  6750  3.00  0.26363  0.11906  0.36540  0.27584  0.21145
 0.30  6750  3.50  0.24947  0.07178  0.37830  0.28401  0.21338
 0.30  6750  4.00  0.23617  0.02699  0.38936  0.29059  0.21601
 0.30  6750  4.50  0.22436 -0.01415  0.39937  0.29623  0.21904
 0.30  6750  5.00  0.21518 -0.04903  0.40939  0.30116  0.22189
 0.30  6750  5.50  0.21518 -0.04903  0.40939  0.30116  0.22189
 0.30  7000 -1.02  0.34307  0.27019  0.16408  0.13686  0.18671
 0.30  7000 -0.70  0.34307  0.27019  0.16408  0.13686  0.18671
 0.30  7000 -0.51  0.34307  0.27019  0.16408  0.13686  0.18671
 0.30  7000 -0.29  0.34307  0.27019  0.16408  0.13686  0.18671
 0.30  7000  0.00  0.34307  0.27019  0.16408  0.13686  0.18671
 0.30  7000  0.28  0.34307  0.27019  0.16408  0.13686  0.18671
 0.30  7000  0.50  0.34307  0.27019  0.16408  0.13686  0.18671
 0.30  7000  0.60  0.34195  0.27575  0.16613  0.13904  0.18585
 0.30  7000  0.87  0.33893  0.29075  0.17166  0.14492  0.18353
 0.30  7000  1.00  0.33748  0.29797  0.17433  0.14775  0.18241
 0.30  7000  1.50  0.33082  0.30368  0.19472  0.16057  0.17717
 0.30  7000  2.00  0.32104  0.27627  0.22961  0.17812  0.17427
 0.30  7000  2.50  0.30486  0.21817  0.26369  0.19763  0.17594
 0.30  7000  3.00  0.28166  0.13919  0.29788  0.22266  0.18167
 0.30  7000  3.50  0.26389  0.07767  0.32408  0.23878  0.18470
 0.30  7000  4.00  0.24790  0.02760  0.33838  0.24780  0.18787
 0.30  7000  4.50  0.23304 -0.02067  0.35074  0.25535  0.19149
 0.30  7000  5.00  0.22046 -0.06441  0.36186  0.26177  0.19522
 0.30  7000  5.50  0.22046 -0.06441  0.36186  0.26177  0.19522
 0.30  7250 -1.02  0.35106  0.24981  0.12473  0.09441  0.16200
 0.30  7250 -0.70  0.35106  0.24981  0.12473  0.09441  0.16200
 0.30  7250 -0.51  0.35106  0.24981  0.12473  0.09441  0.16200
 0.30  7250 -0.29  0.35106  0.24981  0.12473  0.09441  0.16200
 0.30  7250  0.00  0.35106  0.24981  0.12473  0.09441  0.16200
 0.30  7250  0.28  0.35106  0.24981  0.12473  0.09441  0.16200
 0.30  7250  0.50  0.35106  0.24981  0.12473  0.09441  0.16200
 0.30  7250  0.60  0.35013  0.25901  0.12388  0.09567  0.16138
 0.30  7250  0.87  0.34764  0.28384  0.12158  0.09907  0.15972
 0.30  7250  1.00  0.34643  0.29579  0.12047  0.10070  0.15892
 0.30  7250  1.50  0.34256  0.31945  0.13774  0.11132  0.15320
 0.30  7250  2.00  0.33586  0.32005  0.16269  0.12449  0.14839
 0.30  7250  2.50  0.32487  0.28525  0.20156  0.14340  0.14677
 0.30  7250  3.00  0.30705  0.21977  0.23895  0.16457  0.15012
 0.30  7250  3.50  0.28396  0.13884  0.27491  0.18992  0.15688
 0.30  7250  4.00  0.26624  0.07943  0.29785  0.20420  0.16059
 0.30  7250  4.50  0.24965  0.02711  0.31310  0.21365  0.16467
 0.30  7250  5.00  0.23441 -0.02346  0.32603  0.22185  0.16924
 0.30  7250  5.50  0.23441 -0.02346  0.32603  0.22185  0.16924
 0.30  7500 -1.02  0.34353  0.19535  0.12433  0.08855  0.15021
 0.30  7500 -0.70  0.34353  0.19535  0.12433  0.08855  0.15021
 0.30  7500 -0.51  0.34353  0.19535  0.12433  0.08855  0.15021
 0.30  7500 -0.29  0.34353  0.19535  0.12433  0.08855  0.15021
 0.30  7500  0.00  0.34353  0.19535  0.12433  0.08855  0.15021
 0.30  7500  0.28  0.34353  0.19535  0.12433  0.08855  0.15021
 0.30  7500  0.50  0.34353  0.19535  0.12433  0.08855  0.15021
 0.30  7500  0.60  0.34264  0.20947  0.11519  0.08680  0.15029
 0.30  7500  0.87  0.34024  0.24760  0.09049  0.08208  0.15049
 0.30  7500  1.00  0.33908  0.26596  0.07860  0.07981  0.15059
 0.30  7500  1.50  0.33764  0.30622  0.08929  0.08738  0.14492
 0.30  7500  2.00  0.33428  0.32560  0.11174  0.09791  0.13880
 0.30  7500  2.50  0.32767  0.32283  0.14130  0.11108  0.13436
 0.30  7500  3.00  0.31538  0.27930  0.18375  0.13130  0.13426
 0.30  7500  3.50  0.29519  0.20399  0.22269  0.15471  0.13994
 0.30  7500  4.00  0.27260  0.12259  0.25723  0.17858  0.14699
 0.30  7500  4.50  0.25540  0.06680  0.27750  0.19061  0.15099
 0.30  7500  5.00  0.23803  0.01065  0.29301  0.20069  0.15636
 0.30  7500  5.50  0.23803  0.01065  0.29301  0.20069  0.15636
 0.30  7750 -1.02  0.31024  0.18386  0.04459  0.07582  0.14856
 0.30  7750 -0.70  0.31024  0.18386  0.04459  0.07582  0.14856
 0.30  7750 -0.51  0.31024  0.18386  0.04459  0.07582  0.14856
 0.30  7750 -0.29  0.31024  0.18386  0.04459  0.07582  0.14856
 0.30  7750  0.00  0.31024  0.18386  0.04459  0.07582  0.14856
 0.30  7750  0.28  0.31024  0.18386  0.04459  0.07582  0.14856
 0.30  7750  0.50  0.31024  0.18386  0.04459  0.07582  0.14856
 0.30  7750  0.60  0.31024  0.18386  0.04459  0.07582  0.14856
 0.30  7750  0.87  0.31024  0.18386  0.04459  0.07582  0.14856
 0.30  7750  1.00  0.31024  0.18386  0.04459  0.07582  0.14856
 0.30  7750  1.50  0.31119  0.24033  0.04071  0.07737  0.14332
 0.30  7750  2.00  0.31060  0.27477  0.06019  0.08555  0.13652
 0.30  7750  2.50  0.30631  0.28723  0.08741  0.09654  0.13108
 0.30  7750  3.00  0.29798  0.27581  0.12066  0.11044  0.12779
 0.30  7750  3.50  0.28322  0.22065  0.16530  0.13174  0.12936
 0.30  7750  4.00  0.26032  0.13556  0.20474  0.15640  0.13683
 0.30  7750  4.50  0.23922  0.06195  0.23492  0.17585  0.14302
 0.30  7750  5.00  0.21974  0.00049  0.25438  0.18833  0.14884
 0.30  7750  5.50  0.21974  0.00049  0.25438  0.18833  0.14884
 0.30  8000 -1.02  0.27535  0.08767  0.03232  0.06843  0.13416
 0.30  8000 -0.70  0.27535  0.08767  0.03232  0.06843  0.13416
 0.30  8000 -0.51  0.27535  0.08767  0.03232  0.06843  0.13416
 0.30  8000 -0.29  0.27535  0.08767  0.03232  0.06843  0.13416
 0.30  8000  0.00  0.27535  0.08767  0.03232  0.06843  0.13416
 0.30  8000  0.28  0.27535  0.08767  0.03232  0.06843  0.13416
 0.30  8000  0.50  0.27535  0.08767  0.03232  0.06843  0.13416
 0.30  8000  0.60  0.27535  0.08767  0.03232  0.06843  0.13416
 0.30  8000  0.87  0.27535  0.08767  0.03232  0.06843  0.13416
 0.30  8000  1.00  0.27535  0.08767  0.03232  0.06843  0.13416
 0.30  8000  1.50  0.28249  0.16526  0.00333  0.05701  0.13000
 0.30  8000  2.00  0.28481  0.21275  0.01513  0.06115  0.12292
 0.30  8000  2.50  0.28365  0.23956  0.04025  0.06969  0.11638
 0.30  8000  3.00  0.27822  0.24564  0.07123  0.08071  0.11140
 0.30  8000  3.50  0.26854  0.22503  0.10905  0.09523  0.10899
 0.30  8000  4.00  0.25082  0.15544  0.15461  0.11791  0.11307
 0.30  8000  4.50  0.22642  0.06416  0.19272  0.14305  0.12177
 0.30  8000  5.00  0.20520 -0.00580  0.21829  0.15882  0.12835
 0.30  8000  5.50  0.20520 -0.00580  0.21829  0.15882  0.12835
 0.30  8250 -1.02  0.21756 -0.02959  0.03372  0.07303  0.11424
 0.30  8250 -0.70  0.21756 -0.02959  0.03372  0.07303  0.11424
 0.30  8250 -0.51  0.21756 -0.02959  0.03372  0.07303  0.11424
 0.30  8250 -0.29  0.21756 -0.02959  0.03372  0.07303  0.11424
 0.30  8250  0.00  0.21756 -0.02959  0.03372  0.07303  0.11424
 0.30  8250  0.28  0.21756 -0.02959  0.03372  0.07303  0.11424
 0.30  8250  0.50  0.21756 -0.02959  0.03372  0.07303  0.11424
 0.30  8250  0.60  0.21756 -0.02959  0.03372  0.07303  0.11424
 0.30  8250  0.87  0.21756 -0.02959  0.03372  0.07303  0.11424
 0.30  8250  1.00  0.21756 -0.02959  0.03372  0.07303  0.11424
 0.30  8250  1.50  0.24162  0.07590 -0.02014  0.04165  0.11205
 0.30  8250  2.00  0.24756  0.13512 -0.01998  0.03989  0.10503
 0.30  8250  2.50  0.24956  0.17419  0.00068  0.04520  0.09748
 0.30  8250  3.00  0.24736  0.19329  0.03043  0.05386  0.09126
 0.30  8250  3.50  0.24022  0.18994  0.06439  0.06489  0.08760
 0.30  8250  4.00  0.22828  0.15465  0.10582  0.08087  0.08702
 0.30  8250  4.50  0.20814  0.07147  0.15106  0.10481  0.09317
 0.30  8250  5.00  0.18322 -0.02147  0.18505  0.12771  0.10290
 0.30  8250  5.50  0.18322 -0.02147  0.18505  0.12771  0.10290
 0.30  8500 -1.02  0.19519 -0.00050 -0.02791  0.04743  0.10486
 0.30  8500 -0.70  0.19519 -0.00050 -0.02791  0.04743  0.10486
 0.30  8500 -0.51  0.19519 -0.00050 -0.02791  0.04743  0.10486
 0.30  8500 -0.29  0.19519 -0.00050 -0.02791  0.04743  0.10486
 0.30  8500  0.00  0.19519 -0.00050 -0.02791  0.04743  0.10486
 0.30  8500  0.28  0.19519 -0.00050 -0.02791  0.04743  0.10486
 0.30  8500  0.50  0.19519 -0.00050 -0.02791  0.04743  0.10486
 0.30  8500  0.60  0.19519 -0.00050 -0.02791  0.04743  0.10486
 0.30  8500  0.87  0.19519 -0.00050 -0.02791  0.04743  0.10486
 0.30  8500  1.00  0.19519 -0.00050 -0.02791  0.04743  0.10486
 0.30  8500  1.50  0.19519 -0.00050 -0.02791  0.04743  0.10486
 0.30  8500  2.00  0.20586  0.06956 -0.04049  0.03797  0.09821
 0.30  8500  2.50  0.21043  0.11711 -0.02667  0.03959  0.09057
 0.30  8500  3.00  0.21139  0.14586 -0.00028  0.04569  0.08358
 0.30  8500  3.50  0.20817  0.15515  0.03279  0.05445  0.07837
 0.30  8500  4.00  0.19978  0.14126  0.06970  0.06582  0.07577
 0.30  8500  4.50  0.18687  0.08939  0.11487  0.08348  0.07685
 0.30  8500  5.00  0.16572 -0.00389  0.15680  0.10754  0.08520
 0.30  8500  5.50  0.16572 -0.00389  0.15680  0.10754  0.08520
 0.30  8750 -1.02  0.16042 -0.00441 -0.05086  0.04387  0.09625
 0.30  8750 -0.70  0.16042 -0.00441 -0.05086  0.04387  0.09625
 0.30  8750 -0.51  0.16042 -0.00441 -0.05086  0.04387  0.09625
 0.30  8750 -0.29  0.16042 -0.00441 -0.05086  0.04387  0.09625
 0.30  8750  0.00  0.16042 -0.00441 -0.05086  0.04387  0.09625
 0.30  8750  0.28  0.16042 -0.00441 -0.05086  0.04387  0.09625
 0.30  8750  0.50  0.16042 -0.00441 -0.05086  0.04387  0.09625
 0.30  8750  0.60  0.16042 -0.00441 -0.05086  0.04387  0.09625
 0.30  8750  0.87  0.16042 -0.00441 -0.05086  0.04387  0.09625
 0.30  8750  1.00  0.16042 -0.00441 -0.05086  0.04387  0.09625
 0.30  8750  1.50  0.16042 -0.00441 -0.05086  0.04387  0.09625
 0.30  8750  2.00  0.16239  0.00878 -0.05254  0.04216  0.09498
 0.30  8750  2.50  0.16894  0.06262 -0.04599  0.03984  0.08739
 0.30  8750  3.00  0.17227  0.09995 -0.02390  0.04308  0.07969
 0.30  8750  3.50  0.17245  0.12000  0.00672  0.04926  0.07308
 0.30  8750  4.00  0.16845  0.11874  0.04294  0.05814  0.06884
 0.30  8750  4.50  0.15932  0.09119  0.08282  0.06998  0.06782
 0.30  8750  5.00  0.14600  0.02026  0.12826  0.08975  0.07144
 0.30  8750  5.50  0.14600  0.02026  0.12826  0.08975  0.07144
 0.30  9000 -1.02  0.09311 -0.13845 -0.03458  0.05938  0.08590
 0.30  9000 -0.70  0.09311 -0.13845 -0.03458  0.05938  0.08590
 0.30  9000 -0.51  0.09311 -0.13845 -0.03458  0.05938  0.08590
 0.30  9000 -0.29  0.09311 -0.13845 -0.03458  0.05938  0.08590
 0.30  9000  0.00  0.09311 -0.13845 -0.03458  0.05938  0.08590
 0.30  9000  0.28  0.09311 -0.13845 -0.03458  0.05938  0.08590
 0.30  9000  0.50  0.09311 -0.13845 -0.03458  0.05938  0.08590
 0.30  9000  0.60  0.09311 -0.13845 -0.03458  0.05938  0.08590
 0.30  9000  0.87  0.09311 -0.13845 -0.03458  0.05938  0.08590
 0.30  9000  1.00  0.09311 -0.13845 -0.03458  0.05938  0.08590
 0.30  9000  1.50  0.09311 -0.13845 -0.03458  0.05938  0.08590
 0.30  9000  2.00  0.11502 -0.04951 -0.06357  0.03683  0.08105
 0.30  9000  2.50  0.12309  0.00972 -0.06371  0.03054  0.07380
 0.30  9000  3.00  0.12747  0.05292 -0.04631  0.03119  0.06596
 0.30  9000  3.50  0.12958  0.08001 -0.01829  0.03538  0.05892
 0.30  9000  4.00  0.12858  0.08742  0.01602  0.04218  0.05376
 0.30  9000  4.50  0.12333  0.07264  0.05394  0.05118  0.05115
 0.30  9000  5.00  0.11244  0.03504  0.09290  0.06214  0.05186
 0.30  9000  5.50  0.11244  0.03504  0.09290  0.06214  0.05186
 0.30  9250 -1.02  0.06799 -0.10334 -0.07097  0.01949  0.05316
 0.30  9250 -0.70  0.06799 -0.10334 -0.07097  0.01949  0.05316
 0.30  9250 -0.51  0.06799 -0.10334 -0.07097  0.01949  0.05316
 0.30  9250 -0.29  0.06799 -0.10334 -0.07097  0.01949  0.05316
 0.30  9250  0.00  0.06799 -0.10334 -0.07097  0.01949  0.05316
 0.30  9250  0.28  0.06799 -0.10334 -0.07097  0.01949  0.05316
 0.30  9250  0.50  0.06799 -0.10334 -0.07097  0.01949  0.05316
 0.30  9250  0.60  0.06799 -0.10334 -0.07097  0.01949  0.05316
 0.30  9250  0.87  0.06799 -0.10334 -0.07097  0.01949  0.05316
 0.30  9250  1.00  0.06799 -0.10334 -0.07097  0.01949  0.05316
 0.30  9250  1.50  0.06799 -0.10334 -0.07097  0.01949  0.05316
 0.30  9250  2.00  0.06799 -0.10334 -0.07097  0.01949  0.05316
 0.30  9250  2.50  0.07739 -0.03948 -0.07619  0.00983  0.04648
 0.30  9250  3.00  0.08191  0.00788 -0.06366  0.00800  0.03895
 0.30  9250  3.50  0.08461  0.04031 -0.03932  0.01032  0.03172
 0.30  9250  4.00  0.08554  0.05562 -0.00713  0.01518  0.02567
 0.30  9250  4.50  0.08367  0.05007  0.03040  0.02216  0.02169
 0.30  9250  5.00  0.07726  0.02136  0.07025  0.03120  0.02093
 0.30  9250  5.50  0.07726  0.02136  0.07025  0.03120  0.02093
 0.30  9500 -1.02  0.02888 -0.14513 -0.07429  0.01380  0.03447
 0.30  9500 -0.70  0.02888 -0.14513 -0.07429  0.01380  0.03447
 0.30  9500 -0.51  0.02888 -0.14513 -0.07429  0.01380  0.03447
 0.30  9500 -0.29  0.02888 -0.14513 -0.07429  0.01380  0.03447
 0.30  9500  0.00  0.02888 -0.14513 -0.07429  0.01380  0.03447
 0.30  9500  0.28  0.02888 -0.14513 -0.07429  0.01380  0.03447
 0.30  9500  0.50  0.02888 -0.14513 -0.07429  0.01380  0.03447
 0.30  9500  0.60  0.02888 -0.14513 -0.07429  0.01380  0.03447
 0.30  9500  0.87  0.02888 -0.14513 -0.07429  0.01380  0.03447
 0.30  9500  1.00  0.02888 -0.14513 -0.07429  0.01380  0.03447
 0.30  9500  1.50  0.02888 -0.14513 -0.07429  0.01380  0.03447
 0.30  9500  2.00  0.02888 -0.14513 -0.07429  0.01380  0.03447
 0.30  9500  2.50  0.03925 -0.07771 -0.08365  0.00155  0.02841
 0.30  9500  3.00  0.04386 -0.02735 -0.07533 -0.00249  0.02122
 0.30  9500  3.50  0.04661  0.01005 -0.05477 -0.00199  0.01396
 0.30  9500  4.00  0.04853  0.03227 -0.02493  0.00115  0.00730
 0.30  9500  4.50  0.04872  0.03486  0.01136  0.00637  0.00243
 0.30  9500  5.00  0.04575  0.01431  0.05157  0.01365  0.00049
 0.30  9500  5.50  0.04575  0.01431  0.05157  0.01365  0.00049
 0.30  9750 -1.02 -0.00947 -0.18433 -0.07567  0.01239  0.02273
 0.30  9750 -0.70 -0.00947 -0.18433 -0.07567  0.01239  0.02273
 0.30  9750 -0.51 -0.00947 -0.18433 -0.07567  0.01239  0.02273
 0.30  9750 -0.29 -0.00947 -0.18433 -0.07567  0.01239  0.02273
 0.30  9750  0.00 -0.00947 -0.18433 -0.07567  0.01239  0.02273
 0.30  9750  0.28 -0.00947 -0.18433 -0.07567  0.01239  0.02273
 0.30  9750  0.50 -0.00947 -0.18433 -0.07567  0.01239  0.02273
 0.30  9750  0.60 -0.00947 -0.18433 -0.07567  0.01239  0.02273
 0.30  9750  0.87 -0.00947 -0.18433 -0.07567  0.01239  0.02273
 0.30  9750  1.00 -0.00947 -0.18433 -0.07567  0.01239  0.02273
 0.30  9750  1.50 -0.00947 -0.18433 -0.07567  0.01239  0.02273
 0.30  9750  2.00 -0.00947 -0.18433 -0.07567  0.01239  0.02273
 0.30  9750  2.50  0.00165 -0.11456 -0.08751 -0.00138  0.01729
 0.30  9750  3.00  0.00610 -0.06200 -0.08274 -0.00717  0.01064
 0.30  9750  3.50  0.00875 -0.02106 -0.06576 -0.00847  0.00337
 0.30  9750  4.00  0.01088  0.00648 -0.03881 -0.00682 -0.00349
 0.30  9750  4.50  0.01222  0.01648 -0.00422 -0.00315 -0.00893
 0.30  9750  5.00  0.01148  0.00352  0.03513  0.00254 -0.01185
 0.30  9750  5.50  0.01148  0.00352  0.03513  0.00254 -0.01185
 0.30  10000 -1.02 -0.06653 -0.23531 -0.08013  0.01421  0.01595
 0.30  10000 -0.70 -0.06653 -0.23531 -0.08013  0.01421  0.01595
 0.30  10000 -0.51 -0.06653 -0.23531 -0.08013  0.01421  0.01595
 0.30  10000 -0.29 -0.06653 -0.23531 -0.08013  0.01421  0.01595
 0.30  10000  0.00 -0.06653 -0.23531 -0.08013  0.01421  0.01595
 0.30  10000  0.28 -0.06653 -0.23531 -0.08013  0.01421  0.01595
 0.30  10000  0.50 -0.06653 -0.23531 -0.08013  0.01421  0.01595
 0.30  10000  0.60 -0.06653 -0.23531 -0.08013  0.01421  0.01595
 0.30  10000  0.87 -0.06653 -0.23531 -0.08013  0.01421  0.01595
 0.30  10000  1.00 -0.06653 -0.23531 -0.08013  0.01421  0.01595
 0.30  10000  1.50 -0.06653 -0.23531 -0.08013  0.01421  0.01595
 0.30  10000  2.00 -0.06653 -0.23531 -0.08013  0.01421  0.01595
 0.30  10000  2.50 -0.05484 -0.16329 -0.09368 -0.00064  0.01084
 0.30  10000  3.00 -0.05063 -0.10912 -0.09150 -0.00778  0.00462
 0.30  10000  3.50 -0.04863 -0.06638 -0.07735 -0.01033 -0.00234
 0.30  10000  4.00 -0.04702 -0.03496 -0.05353 -0.01019 -0.00912
 0.30  10000  4.50 -0.04536 -0.01949 -0.02116 -0.00786 -0.01493
 0.30  10000  5.00 -0.04486 -0.02530  0.01717 -0.00351 -0.01878
 0.30  10000  5.50 -0.04486 -0.02530  0.01717 -0.00351 -0.01878
 0.30  10500 -1.02 -0.18952 -0.33012 -0.09082  0.01605  0.00649
 0.30  10500 -0.70 -0.18952 -0.33012 -0.09082  0.01605  0.00649
 0.30  10500 -0.51 -0.18952 -0.33012 -0.09082  0.01605  0.00649
 0.30  10500 -0.29 -0.18952 -0.33012 -0.09082  0.01605  0.00649
 0.30  10500  0.00 -0.18952 -0.33012 -0.09082  0.01605  0.00649
 0.30  10500  0.28 -0.18952 -0.33012 -0.09082  0.01605  0.00649
 0.30  10500  0.50 -0.18952 -0.33012 -0.09082  0.01605  0.00649
 0.30  10500  0.60 -0.18952 -0.33012 -0.09082  0.01605  0.00649
 0.30  10500  0.87 -0.18952 -0.33012 -0.09082  0.01605  0.00649
 0.30  10500  1.00 -0.18952 -0.33012 -0.09082  0.01605  0.00649
 0.30  10500  1.50 -0.18952 -0.33012 -0.09082  0.01605  0.00649
 0.30  10500  2.00 -0.18952 -0.33012 -0.09082  0.01605  0.00649
 0.30  10500  2.50 -0.17681 -0.25521 -0.10642  0.00042  0.00148
 0.30  10500  3.00 -0.17349 -0.19940 -0.10735 -0.00829 -0.00393
 0.30  10500  3.50 -0.17315 -0.15498 -0.09789 -0.01274 -0.00996
 0.30  10500  4.00 -0.17302 -0.12025 -0.07934 -0.01459 -0.01619
 0.30  10500  4.50 -0.17209 -0.09724 -0.05219 -0.01436 -0.02236
 0.30  10500  5.00 -0.17069 -0.09067 -0.01690 -0.01256 -0.02745
 0.30  10500  5.50 -0.17069 -0.09067 -0.01690 -0.01256 -0.02745
 0.30  11000 -1.02 -0.30053 -0.33416 -0.11952 -0.00102 -0.00985
 0.30  11000 -0.70 -0.30053 -0.33416 -0.11952 -0.00102 -0.00985
 0.30  11000 -0.51 -0.30053 -0.33416 -0.11952 -0.00102 -0.00985
 0.30  11000 -0.29 -0.30053 -0.33416 -0.11952 -0.00102 -0.00985
 0.30  11000  0.00 -0.30053 -0.33416 -0.11952 -0.00102 -0.00985
 0.30  11000  0.28 -0.30053 -0.33416 -0.11952 -0.00102 -0.00985
 0.30  11000  0.50 -0.30053 -0.33416 -0.11952 -0.00102 -0.00985
 0.30  11000  0.60 -0.30053 -0.33416 -0.11952 -0.00102 -0.00985
 0.30  11000  0.87 -0.30053 -0.33416 -0.11952 -0.00102 -0.00985
 0.30  11000  1.00 -0.30053 -0.33416 -0.11952 -0.00102 -0.00985
 0.30  11000  1.50 -0.30053 -0.33416 -0.11952 -0.00102 -0.00985
 0.30  11000  2.00 -0.30053 -0.33416 -0.11952 -0.00102 -0.00985
 0.30  11000  2.50 -0.30053 -0.33416 -0.11952 -0.00102 -0.00985
 0.30  11000  3.00 -0.29782 -0.27799 -0.12217 -0.01008 -0.01501
 0.30  11000  3.50 -0.29884 -0.23306 -0.11540 -0.01561 -0.02037
 0.30  11000  4.00 -0.30051 -0.19685 -0.10074 -0.01873 -0.02599
 0.30  11000  4.50 -0.30115 -0.16998 -0.07821 -0.02015 -0.03175
 0.30  11000  5.00 -0.30035 -0.15610 -0.04739 -0.02011 -0.03708
 0.30  11000  5.50 -0.30035 -0.15610 -0.04739 -0.02011 -0.03708
 0.30  11500 -1.02 -0.48324 -0.42392 -0.14010 -0.00547 -0.02227
 0.30  11500 -0.70 -0.48324 -0.42392 -0.14010 -0.00547 -0.02227
 0.30  11500 -0.51 -0.48324 -0.42392 -0.14010 -0.00547 -0.02227
 0.30  11500 -0.29 -0.48324 -0.42392 -0.14010 -0.00547 -0.02227
 0.30  11500  0.00 -0.48324 -0.42392 -0.14010 -0.00547 -0.02227
 0.30  11500  0.28 -0.48324 -0.42392 -0.14010 -0.00547 -0.02227
 0.30  11500  0.50 -0.48324 -0.42392 -0.14010 -0.00547 -0.02227
 0.30  11500  0.60 -0.48324 -0.42392 -0.14010 -0.00547 -0.02227
 0.30  11500  0.87 -0.48324 -0.42392 -0.14010 -0.00547 -0.02227
 0.30  11500  1.00 -0.48324 -0.42392 -0.14010 -0.00547 -0.02227
 0.30  11500  1.50 -0.48324 -0.42392 -0.14010 -0.00547 -0.02227
 0.30  11500  2.00 -0.48324 -0.42392 -0.14010 -0.00547 -0.02227
 0.30  11500  2.50 -0.48324 -0.42392 -0.14010 -0.00547 -0.02227
 0.30  11500  3.00 -0.48178 -0.36889 -0.14352 -0.01450 -0.02722
 0.30  11500  3.50 -0.48473 -0.32468 -0.13816 -0.02029 -0.03225
 0.30  11500  4.00 -0.48866 -0.28854 -0.12587 -0.02414 -0.03736
 0.30  11500  4.50 -0.49160 -0.26069 -0.10693 -0.02647 -0.04247
 0.30  11500  5.00 -0.49248 -0.24322 -0.08041 -0.02765 -0.04747
 0.30  11500  5.50 -0.49248 -0.24322 -0.08041 -0.02765 -0.04747
 0.30  12000 -1.02 -0.44902 -0.42272 -0.12822 -0.02166 -0.03266
 0.30  12000 -0.70 -0.44902 -0.42272 -0.12822 -0.02166 -0.03266
 0.30  12000 -0.51 -0.44902 -0.42272 -0.12822 -0.02166 -0.03266
 0.30  12000 -0.29 -0.44902 -0.42272 -0.12822 -0.02166 -0.03266
 0.30  12000  0.00 -0.44902 -0.42272 -0.12822 -0.02166 -0.03266
 0.30  12000  0.28 -0.44902 -0.42272 -0.12822 -0.02166 -0.03266
 0.30  12000  0.50 -0.44902 -0.42272 -0.12822 -0.02166 -0.03266
 0.30  12000  0.60 -0.44902 -0.42272 -0.12822 -0.02166 -0.03266
 0.30  12000  0.87 -0.44902 -0.42272 -0.12822 -0.02166 -0.03266
 0.30  12000  1.00 -0.44902 -0.42272 -0.12822 -0.02166 -0.03266
 0.30  12000  1.50 -0.44902 -0.42272 -0.12822 -0.02166 -0.03266
 0.30  12000  2.00 -0.44902 -0.42272 -0.12822 -0.02166 -0.03266
 0.30  12000  2.50 -0.44902 -0.42272 -0.12822 -0.02166 -0.03266
 0.30  12000  3.00 -0.44449 -0.36997 -0.13227 -0.03021 -0.03758
 0.30  12000  3.50 -0.44498 -0.32721 -0.12801 -0.03583 -0.04248
 0.30  12000  4.00 -0.44720 -0.29154 -0.11753 -0.03988 -0.04729
 0.30  12000  4.50 -0.44964 -0.26298 -0.10120 -0.04288 -0.05178
 0.30  12000  5.00 -0.45083 -0.24282 -0.07865 -0.04487 -0.05640
 0.30  12000  5.50 -0.45083 -0.24282 -0.07865 -0.04487 -0.05640
 0.30  12500 -1.02 -0.54450 -0.46757 -0.13627 -0.02708 -0.04245
 0.30  12500 -0.70 -0.54450 -0.46757 -0.13627 -0.02708 -0.04245
 0.30  12500 -0.51 -0.54450 -0.46757 -0.13627 -0.02708 -0.04245
 0.30  12500 -0.29 -0.54450 -0.46757 -0.13627 -0.02708 -0.04245
 0.30  12500  0.00 -0.54450 -0.46757 -0.13627 -0.02708 -0.04245
 0.30  12500  0.28 -0.54450 -0.46757 -0.13627 -0.02708 -0.04245
 0.30  12500  0.50 -0.54450 -0.46757 -0.13627 -0.02708 -0.04245
 0.30  12500  0.60 -0.54450 -0.46757 -0.13627 -0.02708 -0.04245
 0.30  12500  0.87 -0.54450 -0.46757 -0.13627 -0.02708 -0.04245
 0.30  12500  1.00 -0.54450 -0.46757 -0.13627 -0.02708 -0.04245
 0.30  12500  1.50 -0.54450 -0.46757 -0.13627 -0.02708 -0.04245
 0.30  12500  2.00 -0.54450 -0.46757 -0.13627 -0.02708 -0.04245
 0.30  12500  2.50 -0.54450 -0.46757 -0.13627 -0.02708 -0.04245
 0.30  12500  3.00 -0.54016 -0.41686 -0.14082 -0.03506 -0.04704
 0.30  12500  3.50 -0.54048 -0.37641 -0.13694 -0.04040 -0.05164
 0.30  12500  4.00 -0.54269 -0.34239 -0.12742 -0.04444 -0.05621
 0.30  12500  4.50 -0.54545 -0.31446 -0.11268 -0.04765 -0.06051
 0.30  12500  5.00 -0.54741 -0.29405 -0.09250 -0.05010 -0.06473
 0.30  12500  5.50 -0.54741 -0.29405 -0.09250 -0.05010 -0.06473
 0.30  13000 -1.02 -0.63802 -0.50764 -0.14389 -0.03253 -0.05139
 0.30  13000 -0.70 -0.63802 -0.50764 -0.14389 -0.03253 -0.05139
 0.30  13000 -0.51 -0.63802 -0.50764 -0.14389 -0.03253 -0.05139
 0.30  13000 -0.29 -0.63802 -0.50764 -0.14389 -0.03253 -0.05139
 0.30  13000  0.00 -0.63802 -0.50764 -0.14389 -0.03253 -0.05139
 0.30  13000  0.28 -0.63802 -0.50764 -0.14389 -0.03253 -0.05139
 0.30  13000  0.50 -0.63802 -0.50764 -0.14389 -0.03253 -0.05139
 0.30  13000  0.60 -0.63802 -0.50764 -0.14389 -0.03253 -0.05139
 0.30  13000  0.87 -0.63802 -0.50764 -0.14389 -0.03253 -0.05139
 0.30  13000  1.00 -0.63802 -0.50764 -0.14389 -0.03253 -0.05139
 0.30  13000  1.50 -0.63802 -0.50764 -0.14389 -0.03253 -0.05139
 0.30  13000  2.00 -0.63802 -0.50764 -0.14389 -0.03253 -0.05139
 0.30  13000  2.50 -0.63802 -0.50764 -0.14389 -0.03253 -0.05139
 0.30  13000  3.00 -0.63417 -0.45813 -0.14875 -0.04024 -0.05559
 0.30  13000  3.50 -0.63469 -0.41979 -0.14542 -0.04505 -0.05997
 0.30  13000  4.00 -0.63678 -0.38792 -0.13646 -0.04880 -0.06433
 0.30  13000  4.50 -0.63939 -0.36146 -0.12262 -0.05206 -0.06851
 0.30  13000  5.00 -0.64179 -0.34138 -0.10407 -0.05468 -0.07256
 0.30  13000  5.50 -0.64179 -0.34138 -0.10407 -0.05468 -0.07256
 0.30  14000 -1.02 -0.83290 -0.66068 -0.13455 -0.02691 -0.06262
 0.30  14000 -0.70 -0.83290 -0.66068 -0.13455 -0.02691 -0.06262
 0.30  14000 -0.51 -0.83290 -0.66068 -0.13455 -0.02691 -0.06262
 0.30  14000 -0.29 -0.83290 -0.66068 -0.13455 -0.02691 -0.06262
 0.30  14000  0.00 -0.83290 -0.66068 -0.13455 -0.02691 -0.06262
 0.30  14000  0.28 -0.83290 -0.66068 -0.13455 -0.02691 -0.06262
 0.30  14000  0.50 -0.83290 -0.66068 -0.13455 -0.02691 -0.06262
 0.30  14000  0.60 -0.83290 -0.66068 -0.13455 -0.02691 -0.06262
 0.30  14000  0.87 -0.83290 -0.66068 -0.13455 -0.02691 -0.06262
 0.30  14000  1.00 -0.83290 -0.66068 -0.13455 -0.02691 -0.06262
 0.30  14000  1.50 -0.83290 -0.66068 -0.13455 -0.02691 -0.06262
 0.30  14000  2.00 -0.83290 -0.66068 -0.13455 -0.02691 -0.06262
 0.30  14000  2.50 -0.81725 -0.57804 -0.15774 -0.04302 -0.06763
 0.30  14000  3.00 -0.81575 -0.52892 -0.16329 -0.05021 -0.07122
 0.30  14000  3.50 -0.81745 -0.49356 -0.16037 -0.05441 -0.07485
 0.30  14000  4.00 -0.81974 -0.46534 -0.15235 -0.05755 -0.07875
 0.30  14000  4.50 -0.82211 -0.44239 -0.13995 -0.06025 -0.08272
 0.30  14000  5.00 -0.82445 -0.42433 -0.12354 -0.06284 -0.08647
 0.30  14000  5.50 -0.82445 -0.42433 -0.12354 -0.06284 -0.08647
 0.30  15000 -1.02 -0.98487 -0.63971 -0.17044 -0.05280 -0.08212
 0.30  15000 -0.70 -0.98487 -0.63971 -0.17044 -0.05280 -0.08212
 0.30  15000 -0.51 -0.98487 -0.63971 -0.17044 -0.05280 -0.08212
 0.30  15000 -0.29 -0.98487 -0.63971 -0.17044 -0.05280 -0.08212
 0.30  15000  0.00 -0.98487 -0.63971 -0.17044 -0.05280 -0.08212
 0.30  15000  0.28 -0.98487 -0.63971 -0.17044 -0.05280 -0.08212
 0.30  15000  0.50 -0.98487 -0.63971 -0.17044 -0.05280 -0.08212
 0.30  15000  0.60 -0.98487 -0.63971 -0.17044 -0.05280 -0.08212
 0.30  15000  0.87 -0.98487 -0.63971 -0.17044 -0.05280 -0.08212
 0.30  15000  1.00 -0.98487 -0.63971 -0.17044 -0.05280 -0.08212
 0.30  15000  1.50 -0.98487 -0.63971 -0.17044 -0.05280 -0.08212
 0.30  15000  2.00 -0.98487 -0.63971 -0.17044 -0.05280 -0.08212
 0.30  15000  2.50 -0.98487 -0.63971 -0.17044 -0.05280 -0.08212
 0.30  15000  3.00 -0.98738 -0.58859 -0.17686 -0.05976 -0.08514
 0.30  15000  3.50 -0.99145 -0.55416 -0.17427 -0.06350 -0.08815
 0.30  15000  4.00 -0.99500 -0.52846 -0.16685 -0.06594 -0.09140
 0.30  15000  4.50 -0.99749 -0.50862 -0.15541 -0.06813 -0.09485
 0.30  15000  5.00 -0.99934 -0.49348 -0.14024 -0.07039 -0.09837
 0.30  15000  5.50 -0.99934 -0.49348 -0.14024 -0.07039 -0.09837
 0.30  16000 -1.02 -1.14059 -0.69618 -0.18181 -0.06158 -0.09532
 0.30  16000 -0.70 -1.14059 -0.69618 -0.18181 -0.06158 -0.09532
 0.30  16000 -0.51 -1.14059 -0.69618 -0.18181 -0.06158 -0.09532
 0.30  16000 -0.29 -1.14059 -0.69618 -0.18181 -0.06158 -0.09532
 0.30  16000  0.00 -1.14059 -0.69618 -0.18181 -0.06158 -0.09532
 0.30  16000  0.28 -1.14059 -0.69618 -0.18181 -0.06158 -0.09532
 0.30  16000  0.50 -1.14059 -0.69618 -0.18181 -0.06158 -0.09532
 0.30  16000  0.60 -1.14059 -0.69618 -0.18181 -0.06158 -0.09532
 0.30  16000  0.87 -1.14059 -0.69618 -0.18181 -0.06158 -0.09532
 0.30  16000  1.00 -1.14059 -0.69618 -0.18181 -0.06158 -0.09532
 0.30  16000  1.50 -1.14059 -0.69618 -0.18181 -0.06158 -0.09532
 0.30  16000  2.00 -1.14059 -0.69618 -0.18181 -0.06158 -0.09532
 0.30  16000  2.50 -1.14059 -0.69618 -0.18181 -0.06158 -0.09532
 0.30  16000  3.00 -1.14804 -0.64119 -0.18953 -0.06869 -0.09787
 0.30  16000  3.50 -1.15569 -0.60632 -0.18733 -0.07197 -0.10016
 0.30  16000  4.00 -1.16114 -0.58189 -0.17999 -0.07421 -0.10281
 0.30  16000  4.50 -1.16461 -0.56379 -0.16938 -0.07589 -0.10577
 0.30  16000  5.00 -1.16642 -0.55119 -0.15513 -0.07779 -0.10888
 0.30  16000  5.50 -1.16642 -0.55119 -0.15513 -0.07779 -0.10888
 0.30  17000 -1.02 -1.28448 -0.75008 -0.19041 -0.06960 -0.10729
 0.30  17000 -0.70 -1.28448 -0.75008 -0.19041 -0.06960 -0.10729
 0.30  17000 -0.51 -1.28448 -0.75008 -0.19041 -0.06960 -0.10729
 0.30  17000 -0.29 -1.28448 -0.75008 -0.19041 -0.06960 -0.10729
 0.30  17000  0.00 -1.28448 -0.75008 -0.19041 -0.06960 -0.10729
 0.30  17000  0.28 -1.28448 -0.75008 -0.19041 -0.06960 -0.10729
 0.30  17000  0.50 -1.28448 -0.75008 -0.19041 -0.06960 -0.10729
 0.30  17000  0.60 -1.28448 -0.75008 -0.19041 -0.06960 -0.10729
 0.30  17000  0.87 -1.28448 -0.75008 -0.19041 -0.06960 -0.10729
 0.30  17000  1.00 -1.28448 -0.75008 -0.19041 -0.06960 -0.10729
 0.30  17000  1.50 -1.28448 -0.75008 -0.19041 -0.06960 -0.10729
 0.30  17000  2.00 -1.28448 -0.75008 -0.19041 -0.06960 -0.10729
 0.30  17000  2.50 -1.28448 -0.75008 -0.19041 -0.06960 -0.10729
 0.30  17000  3.00 -1.29781 -0.68934 -0.20086 -0.07719 -0.10945
 0.30  17000  3.50 -1.30996 -0.65296 -0.19935 -0.08006 -0.11120
 0.30  17000  4.00 -1.31823 -0.62839 -0.19255 -0.08176 -0.11325
 0.30  17000  4.50 -1.32318 -0.61126 -0.18237 -0.08317 -0.11572
 0.30  17000  5.00 -1.32567 -0.59994 -0.16896 -0.08464 -0.11841
 0.30  17000  5.50 -1.32567 -0.59994 -0.16896 -0.08464 -0.11841
 0.30  18000 -1.02 -1.41718 -0.80217 -0.19633 -0.07607 -0.11788
 0.30  18000 -0.70 -1.41718 -0.80217 -0.19633 -0.07607 -0.11788
 0.30  18000 -0.51 -1.41718 -0.80217 -0.19633 -0.07607 -0.11788
 0.30  18000 -0.29 -1.41718 -0.80217 -0.19633 -0.07607 -0.11788
 0.30  18000  0.00 -1.41718 -0.80217 -0.19633 -0.07607 -0.11788
 0.30  18000  0.28 -1.41718 -0.80217 -0.19633 -0.07607 -0.11788
 0.30  18000  0.50 -1.41718 -0.80217 -0.19633 -0.07607 -0.11788
 0.30  18000  0.60 -1.41718 -0.80217 -0.19633 -0.07607 -0.11788
 0.30  18000  0.87 -1.41718 -0.80217 -0.19633 -0.07607 -0.11788
 0.30  18000  1.00 -1.41718 -0.80217 -0.19633 -0.07607 -0.11788
 0.30  18000  1.50 -1.41718 -0.80217 -0.19633 -0.07607 -0.11788
 0.30  18000  2.00 -1.41718 -0.80217 -0.19633 -0.07607 -0.11788
 0.30  18000  2.50 -1.41718 -0.80217 -0.19633 -0.07607 -0.11788
 0.30  18000  3.00 -1.43626 -0.73437 -0.21032 -0.08549 -0.12023
 0.30  18000  3.50 -1.45352 -0.69612 -0.20991 -0.08821 -0.12132
 0.30  18000  4.00 -1.46526 -0.67098 -0.20369 -0.08953 -0.12287
 0.30  18000  4.50 -1.47243 -0.65376 -0.19405 -0.09058 -0.12487
 0.30  18000  5.00 -1.47611 -0.64325 -0.18136 -0.09174 -0.12713
 0.30  18000  5.50 -1.47611 -0.64325 -0.18136 -0.09174 -0.12713
 0.30  19000 -1.02 -1.54102 -0.85382 -0.20018 -0.07975 -0.12634
 0.30  19000 -0.70 -1.54102 -0.85382 -0.20018 -0.07975 -0.12634
 0.30  19000 -0.51 -1.54102 -0.85382 -0.20018 -0.07975 -0.12634
 0.30  19000 -0.29 -1.54102 -0.85382 -0.20018 -0.07975 -0.12634
 0.30  19000  0.00 -1.54102 -0.85382 -0.20018 -0.07975 -0.12634
 0.30  19000  0.28 -1.54102 -0.85382 -0.20018 -0.07975 -0.12634
 0.30  19000  0.50 -1.54102 -0.85382 -0.20018 -0.07975 -0.12634
 0.30  19000  0.60 -1.54102 -0.85382 -0.20018 -0.07975 -0.12634
 0.30  19000  0.87 -1.54102 -0.85382 -0.20018 -0.07975 -0.12634
 0.30  19000  1.00 -1.54102 -0.85382 -0.20018 -0.07975 -0.12634
 0.30  19000  1.50 -1.54102 -0.85382 -0.20018 -0.07975 -0.12634
 0.30  19000  2.00 -1.54102 -0.85382 -0.20018 -0.07975 -0.12634
 0.30  19000  2.50 -1.54102 -0.85382 -0.20018 -0.07975 -0.12634
 0.30  19000  3.00 -1.56467 -0.77668 -0.21841 -0.09257 -0.13037
 0.30  19000  3.50 -1.58751 -0.73563 -0.21967 -0.09562 -0.13101
 0.30  19000  4.00 -1.60329 -0.70966 -0.21411 -0.09661 -0.13191
 0.30  19000  4.50 -1.61296 -0.69233 -0.20486 -0.09751 -0.13339
 0.30  19000  5.00 -1.61827 -0.68185 -0.19272 -0.09849 -0.13530
 0.30  19000  5.50 -1.61827 -0.68185 -0.19272 -0.09849 -0.13530
 0.30  20000 -1.02 -1.68385 -0.81727 -0.22506 -0.09840 -0.13949
 0.30  20000 -0.70 -1.68385 -0.81727 -0.22506 -0.09840 -0.13949
 0.30  20000 -0.51 -1.68385 -0.81727 -0.22506 -0.09840 -0.13949
 0.30  20000 -0.29 -1.68385 -0.81727 -0.22506 -0.09840 -0.13949
 0.30  20000  0.00 -1.68385 -0.81727 -0.22506 -0.09840 -0.13949
 0.30  20000  0.28 -1.68385 -0.81727 -0.22506 -0.09840 -0.13949
 0.30  20000  0.50 -1.68385 -0.81727 -0.22506 -0.09840 -0.13949
 0.30  20000  0.60 -1.68385 -0.81727 -0.22506 -0.09840 -0.13949
 0.30  20000  0.87 -1.68385 -0.81727 -0.22506 -0.09840 -0.13949
 0.30  20000  1.00 -1.68385 -0.81727 -0.22506 -0.09840 -0.13949
 0.30  20000  1.50 -1.68385 -0.81727 -0.22506 -0.09840 -0.13949
 0.30  20000  2.00 -1.68385 -0.81727 -0.22506 -0.09840 -0.13949
 0.30  20000  2.50 -1.68385 -0.81727 -0.22506 -0.09840 -0.13949
 0.30  20000  3.00 -1.68385 -0.81727 -0.22506 -0.09840 -0.13949
 0.30  20000  3.50 -1.71233 -0.77251 -0.22828 -0.10235 -0.14015
 0.30  20000  4.00 -1.73255 -0.74529 -0.22378 -0.10334 -0.14047
 0.30  20000  4.50 -1.74524 -0.72765 -0.21529 -0.10388 -0.14152
 0.30  20000  5.00 -1.75256 -0.71689 -0.20378 -0.10464 -0.14304
 0.30  20000  5.50 -1.75256 -0.71689 -0.20378 -0.10464 -0.14304
 0.30  21000 -1.02 -1.79572 -0.85624 -0.23062 -0.10270 -0.14710
 0.30  21000 -0.70 -1.79572 -0.85624 -0.23062 -0.10270 -0.14710
 0.30  21000 -0.51 -1.79572 -0.85624 -0.23062 -0.10270 -0.14710
 0.30  21000 -0.29 -1.79572 -0.85624 -0.23062 -0.10270 -0.14710
 0.30  21000  0.00 -1.79572 -0.85624 -0.23062 -0.10270 -0.14710
 0.30  21000  0.28 -1.79572 -0.85624 -0.23062 -0.10270 -0.14710
 0.30  21000  0.50 -1.79572 -0.85624 -0.23062 -0.10270 -0.14710
 0.30  21000  0.60 -1.79572 -0.85624 -0.23062 -0.10270 -0.14710
 0.30  21000  0.87 -1.79572 -0.85624 -0.23062 -0.10270 -0.14710
 0.30  21000  1.00 -1.79572 -0.85624 -0.23062 -0.10270 -0.14710
 0.30  21000  1.50 -1.79572 -0.85624 -0.23062 -0.10270 -0.14710
 0.30  21000  2.00 -1.79572 -0.85624 -0.23062 -0.10270 -0.14710
 0.30  21000  2.50 -1.79572 -0.85624 -0.23062 -0.10270 -0.14710
 0.30  21000  3.00 -1.79572 -0.85624 -0.23062 -0.10270 -0.14710
 0.30  21000  3.50 -1.82879 -0.80780 -0.23586 -0.10821 -0.14863
 0.30  21000  4.00 -1.85345 -0.77870 -0.23247 -0.10957 -0.14879
 0.30  21000  4.50 -1.86947 -0.76038 -0.22474 -0.11001 -0.14931
 0.30  21000  5.00 -1.87899 -0.74941 -0.21400 -0.11056 -0.15034
 0.30  21000  5.50 -1.87899 -0.74941 -0.21400 -0.11056 -0.15034
 0.30  22000 -1.02 -1.90195 -0.89185 -0.23574 -0.10618 -0.15340
 0.30  22000 -0.70 -1.90195 -0.89185 -0.23574 -0.10618 -0.15340
 0.30  22000 -0.51 -1.90195 -0.89185 -0.23574 -0.10618 -0.15340
 0.30  22000 -0.29 -1.90195 -0.89185 -0.23574 -0.10618 -0.15340
 0.30  22000  0.00 -1.90195 -0.89185 -0.23574 -0.10618 -0.15340
 0.30  22000  0.28 -1.90195 -0.89185 -0.23574 -0.10618 -0.15340
 0.30  22000  0.50 -1.90195 -0.89185 -0.23574 -0.10618 -0.15340
 0.30  22000  0.60 -1.90195 -0.89185 -0.23574 -0.10618 -0.15340
 0.30  22000  0.87 -1.90195 -0.89185 -0.23574 -0.10618 -0.15340
 0.30  22000  1.00 -1.90195 -0.89185 -0.23574 -0.10618 -0.15340
 0.30  22000  1.50 -1.90195 -0.89185 -0.23574 -0.10618 -0.15340
 0.30  22000  2.00 -1.90195 -0.89185 -0.23574 -0.10618 -0.15340
 0.30  22000  2.50 -1.90195 -0.89185 -0.23574 -0.10618 -0.15340
 0.30  22000  3.00 -1.90195 -0.89185 -0.23574 -0.10618 -0.15340
 0.30  22000  3.50 -1.93782 -0.84145 -0.24209 -0.11312 -0.15611
 0.30  22000  4.00 -1.96652 -0.81038 -0.24015 -0.11522 -0.15651
 0.30  22000  4.50 -1.98614 -0.79109 -0.23348 -0.11574 -0.15675
 0.30  22000  5.00 -1.99816 -0.77972 -0.22350 -0.11618 -0.15746
 0.30  22000  5.50 -1.99816 -0.77972 -0.22350 -0.11618 -0.15746
 0.30  23000 -1.02 -2.00421 -0.92276 -0.24142 -0.10992 -0.15891
 0.30  23000 -0.70 -2.00421 -0.92276 -0.24142 -0.10992 -0.15891
 0.30  23000 -0.51 -2.00421 -0.92276 -0.24142 -0.10992 -0.15891
 0.30  23000 -0.29 -2.00421 -0.92276 -0.24142 -0.10992 -0.15891
 0.30  23000  0.00 -2.00421 -0.92276 -0.24142 -0.10992 -0.15891
 0.30  23000  0.28 -2.00421 -0.92276 -0.24142 -0.10992 -0.15891
 0.30  23000  0.50 -2.00421 -0.92276 -0.24142 -0.10992 -0.15891
 0.30  23000  0.60 -2.00421 -0.92276 -0.24142 -0.10992 -0.15891
 0.30  23000  0.87 -2.00421 -0.92276 -0.24142 -0.10992 -0.15891
 0.30  23000  1.00 -2.00421 -0.92276 -0.24142 -0.10992 -0.15891
 0.30  23000  1.50 -2.00421 -0.92276 -0.24142 -0.10992 -0.15891
 0.30  23000  2.00 -2.00421 -0.92276 -0.24142 -0.10992 -0.15891
 0.30  23000  2.50 -2.00421 -0.92276 -0.24142 -0.10992 -0.15891
 0.30  23000  3.00 -2.00421 -0.92276 -0.24142 -0.10992 -0.15891
 0.30  23000  3.50 -2.04094 -0.87269 -0.24783 -0.11750 -0.16238
 0.30  23000  4.00 -2.07256 -0.84046 -0.24687 -0.12022 -0.16355
 0.30  23000  4.50 -2.09544 -0.81991 -0.24128 -0.12104 -0.16382
 0.30  23000  5.00 -2.11004 -0.80797 -0.23216 -0.12158 -0.16427
 0.30  23000  5.50 -2.11004 -0.80797 -0.23216 -0.12158 -0.16427
 0.30  24000 -1.02 -2.10089 -0.94975 -0.24748 -0.11421 -0.16471
 0.30  24000 -0.70 -2.10089 -0.94975 -0.24748 -0.11421 -0.16471
 0.30  24000 -0.51 -2.10089 -0.94975 -0.24748 -0.11421 -0.16471
 0.30  24000 -0.29 -2.10089 -0.94975 -0.24748 -0.11421 -0.16471
 0.30  24000  0.00 -2.10089 -0.94975 -0.24748 -0.11421 -0.16471
 0.30  24000  0.28 -2.10089 -0.94975 -0.24748 -0.11421 -0.16471
 0.30  24000  0.50 -2.10089 -0.94975 -0.24748 -0.11421 -0.16471
 0.30  24000  0.60 -2.10089 -0.94975 -0.24748 -0.11421 -0.16471
 0.30  24000  0.87 -2.10089 -0.94975 -0.24748 -0.11421 -0.16471
 0.30  24000  1.00 -2.10089 -0.94975 -0.24748 -0.11421 -0.16471
 0.30  24000  1.50 -2.10089 -0.94975 -0.24748 -0.11421 -0.16471
 0.30  24000  2.00 -2.10089 -0.94975 -0.24748 -0.11421 -0.16471
 0.30  24000  2.50 -2.10089 -0.94975 -0.24748 -0.11421 -0.16471
 0.30  24000  3.00 -2.10089 -0.94975 -0.24748 -0.11421 -0.16471
 0.30  24000  3.50 -2.13990 -0.90039 -0.25375 -0.12150 -0.16817
 0.30  24000  4.00 -2.17300 -0.86865 -0.25299 -0.12461 -0.16961
 0.30  24000  4.50 -2.19806 -0.84741 -0.24811 -0.12604 -0.17044
 0.30  24000  5.00 -2.21516 -0.83444 -0.24008 -0.12673 -0.17084
 0.30  24000  5.50 -2.21516 -0.83444 -0.24008 -0.12673 -0.17084
 0.30  25000 -1.02 -2.18947 -0.97592 -0.25234 -0.11774 -0.17067
 0.30  25000 -0.70 -2.18947 -0.97592 -0.25234 -0.11774 -0.17067
 0.30  25000 -0.51 -2.18947 -0.97592 -0.25234 -0.11774 -0.17067
 0.30  25000 -0.29 -2.18947 -0.97592 -0.25234 -0.11774 -0.17067
 0.30  25000  0.00 -2.18947 -0.97592 -0.25234 -0.11774 -0.17067
 0.30  25000  0.28 -2.18947 -0.97592 -0.25234 -0.11774 -0.17067
 0.30  25000  0.50 -2.18947 -0.97592 -0.25234 -0.11774 -0.17067
 0.30  25000  0.60 -2.18947 -0.97592 -0.25234 -0.11774 -0.17067
 0.30  25000  0.87 -2.18947 -0.97592 -0.25234 -0.11774 -0.17067
 0.30  25000  1.00 -2.18947 -0.97592 -0.25234 -0.11774 -0.17067
 0.30  25000  1.50 -2.18947 -0.97592 -0.25234 -0.11774 -0.17067
 0.30  25000  2.00 -2.18947 -0.97592 -0.25234 -0.11774 -0.17067
 0.30  25000  2.50 -2.18947 -0.97592 -0.25234 -0.11774 -0.17067
 0.30  25000  3.00 -2.18947 -0.97592 -0.25234 -0.11774 -0.17067
 0.30  25000  3.50 -2.23446 -0.92441 -0.26023 -0.12590 -0.17388
 0.30  25000  4.00 -2.26891 -0.89416 -0.25905 -0.12875 -0.17522
 0.30  25000  4.50 -2.29524 -0.87323 -0.25464 -0.13042 -0.17612
 0.30  25000  5.00 -2.31406 -0.86004 -0.24720 -0.13145 -0.17691
 0.30  25000  5.50 -2.31406 -0.86004 -0.24720 -0.13145 -0.17691
 0.30  26000 -1.02 -2.27133 -1.00313 -0.25431 -0.11870 -0.17481
 0.30  26000 -0.70 -2.27133 -1.00313 -0.25431 -0.11870 -0.17481
 0.30  26000 -0.51 -2.27133 -1.00313 -0.25431 -0.11870 -0.17481
 0.30  26000 -0.29 -2.27133 -1.00313 -0.25431 -0.11870 -0.17481
 0.30  26000  0.00 -2.27133 -1.00313 -0.25431 -0.11870 -0.17481
 0.30  26000  0.28 -2.27133 -1.00313 -0.25431 -0.11870 -0.17481
 0.30  26000  0.50 -2.27133 -1.00313 -0.25431 -0.11870 -0.17481
 0.30  26000  0.60 -2.27133 -1.00313 -0.25431 -0.11870 -0.17481
 0.30  26000  0.87 -2.27133 -1.00313 -0.25431 -0.11870 -0.17481
 0.30  26000  1.00 -2.27133 -1.00313 -0.25431 -0.11870 -0.17481
 0.30  26000  1.50 -2.27133 -1.00313 -0.25431 -0.11870 -0.17481
 0.30  26000  2.00 -2.27133 -1.00313 -0.25431 -0.11870 -0.17481
 0.30  26000  2.50 -2.27133 -1.00313 -0.25431 -0.11870 -0.17481
 0.30  26000  3.00 -2.27133 -1.00313 -0.25431 -0.11870 -0.17481
 0.30  26000  3.50 -2.32362 -0.94575 -0.26689 -0.13101 -0.17989
 0.30  26000  4.00 -2.36083 -0.91679 -0.26540 -0.13271 -0.18062
 0.30  26000  4.50 -2.38769 -0.89711 -0.26079 -0.13452 -0.18157
 0.30  26000  5.00 -2.40750 -0.88394 -0.25393 -0.13581 -0.18231
 0.30  26000  5.50 -2.40750 -0.88394 -0.25393 -0.13581 -0.18231
 0.30  27000 -1.02 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
 0.30  27000 -0.70 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
 0.30  27000 -0.51 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
 0.30  27000 -0.29 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
 0.30  27000  0.00 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
 0.30  27000  0.28 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
 0.30  27000  0.50 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
 0.30  27000  0.60 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
 0.30  27000  0.87 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
 0.30  27000  1.00 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
 0.30  27000  1.50 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
 0.30  27000  2.00 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
 0.30  27000  2.50 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
 0.30  27000  3.00 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
 0.30  27000  3.50 -2.40524 -0.96623 -0.27337 -0.13569 -0.18593
 0.30  27000  4.00 -2.44762 -0.93680 -0.27204 -0.13755 -0.18642
 0.30  27000  4.50 -2.47615 -0.91840 -0.26722 -0.13876 -0.18637
 0.30  27000  5.00 -2.49617 -0.90623 -0.26042 -0.13984 -0.18750
 0.30  27000  5.50 -2.49617 -0.90623 -0.26042 -0.13984 -0.18750
 0.30  28000 -1.02 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
 0.30  28000 -0.70 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
 0.30  28000 -0.51 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
 0.30  28000 -0.29 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
 0.30  28000  0.00 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
 0.30  28000  0.28 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
 0.30  28000  0.50 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
 0.30  28000  0.60 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
 0.30  28000  0.87 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
 0.30  28000  1.00 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
 0.30  28000  1.50 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
 0.30  28000  2.00 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
 0.30  28000  2.50 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
 0.30  28000  3.00 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
 0.30  28000  3.50 -2.48037 -0.98674 -0.27879 -0.13922 -0.19125
 0.30  28000  4.00 -2.52892 -0.95542 -0.27879 -0.14239 -0.19191
 0.30  28000  4.50 -2.55968 -0.93767 -0.27357 -0.14305 -0.19198
 0.30  28000  5.00 -2.58074 -0.92661 -0.26689 -0.14394 -0.19221
 0.30  28000  5.50 -2.58074 -0.92661 -0.26689 -0.14394 -0.19221
 0.30  29000 -1.02 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
 0.30  29000 -0.70 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
 0.30  29000 -0.51 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
 0.30  29000 -0.29 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
 0.30  29000  0.00 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
 0.30  29000  0.28 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
 0.30  29000  0.50 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
 0.30  29000  0.60 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
 0.30  29000  0.87 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
 0.30  29000  1.00 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
 0.30  29000  1.50 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
 0.30  29000  2.00 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
 0.30  29000  2.50 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
 0.30  29000  3.00 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
 0.30  29000  3.50 -2.55216 -1.00672 -0.28264 -0.14104 -0.19520
 0.30  29000  4.00 -2.60355 -0.97348 -0.28526 -0.14669 -0.19743
 0.30  29000  4.50 -2.63831 -0.95559 -0.28024 -0.14743 -0.19725
 0.30  29000  5.00 -2.66049 -0.94535 -0.27337 -0.14777 -0.19732
 0.30  29000  5.50 -2.66049 -0.94535 -0.27337 -0.14777 -0.19732
 0.30  30000 -1.02 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
 0.30  30000 -0.70 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
 0.30  30000 -0.51 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
 0.30  30000 -0.29 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
 0.30  30000  0.00 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
 0.30  30000  0.28 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
 0.30  30000  0.50 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
 0.30  30000  0.60 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
 0.30  30000  0.87 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
 0.30  30000  1.00 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
 0.30  30000  1.50 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
 0.30  30000  2.00 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
 0.30  30000  2.50 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
 0.30  30000  3.00 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
 0.30  30000  3.50 -2.62439 -1.02508 -0.28483 -0.14096 -0.19745
 0.30  30000  4.00 -2.67342 -0.99115 -0.29078 -0.15018 -0.20214
 0.30  30000  4.50 -2.71144 -0.97261 -0.28665 -0.15153 -0.20230
 0.30  30000  5.00 -2.73584 -0.96265 -0.27981 -0.15176 -0.20218
 0.30  30000  5.50 -2.73584 -0.96265 -0.27981 -0.15176 -0.20218
 0.30  31000 -1.02 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
 0.30  31000 -0.70 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
 0.30  31000 -0.51 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
 0.30  31000 -0.29 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
 0.30  31000  0.00 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
 0.30  31000  0.28 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
 0.30  31000  0.50 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
 0.30  31000  0.60 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
 0.30  31000  0.87 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
 0.30  31000  1.00 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
 0.30  31000  1.50 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
 0.30  31000  2.00 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
 0.30  31000  2.50 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
 0.30  31000  3.00 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
 0.30  31000  3.50 -2.70150 -1.04049 -0.28617 -0.13947 -0.19799
 0.30  31000  4.00 -2.74139 -1.00792 -0.29531 -0.15236 -0.20587
 0.30  31000  4.50 -2.78037 -0.98904 -0.29260 -0.15499 -0.20676
 0.30  31000  5.00 -2.80661 -0.97910 -0.28612 -0.15544 -0.20660
 0.30  31000  5.50 -2.80661 -0.97910 -0.28612 -0.15544 -0.20660
 0.30  32000 -1.02 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
 0.30  32000 -0.70 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
 0.30  32000 -0.51 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
 0.30  32000 -0.29 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
 0.30  32000  0.00 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
 0.30  32000  0.28 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
 0.30  32000  0.50 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
 0.30  32000  0.60 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
 0.30  32000  0.87 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
 0.30  32000  1.00 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
 0.30  32000  1.50 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
 0.30  32000  2.00 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
 0.30  32000  2.50 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
 0.30  32000  3.00 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
 0.30  32000  3.50 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
 0.30  32000  4.00 -2.80898 -1.02334 -0.29875 -0.15347 -0.20868
 0.30  32000  4.50 -2.84690 -1.00455 -0.29779 -0.15760 -0.21048
 0.30  32000  5.00 -2.87420 -0.99452 -0.29207 -0.15854 -0.21056
 0.30  32000  5.50 -2.87420 -0.99452 -0.29207 -0.15854 -0.21056
 0.30  33000 -1.02 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
 0.30  33000 -0.70 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
 0.30  33000 -0.51 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
 0.30  33000 -0.29 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
 0.30  33000  0.00 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
 0.30  33000  0.28 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
 0.30  33000  0.50 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
 0.30  33000  0.60 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
 0.30  33000  0.87 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
 0.30  33000  1.00 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
 0.30  33000  1.50 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
 0.30  33000  2.00 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
 0.30  33000  2.50 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
 0.30  33000  3.00 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
 0.30  33000  3.50 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
 0.30  33000  4.00 -2.87987 -1.03658 -0.30130 -0.15341 -0.21044
 0.30  33000  4.50 -2.91195 -1.01898 -0.30219 -0.15942 -0.21359
 0.30  33000  5.00 -2.93942 -1.00904 -0.29737 -0.16111 -0.21400
 0.30  33000  5.50 -2.93942 -1.00904 -0.29737 -0.16111 -0.21400
 0.30  34000 -1.02 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
 0.30  34000 -0.70 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
 0.30  34000 -0.51 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
 0.30  34000 -0.29 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
 0.30  34000  0.00 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
 0.30  34000  0.28 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
 0.30  34000  0.50 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
 0.30  34000  0.60 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
 0.30  34000  0.87 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
 0.30  34000  1.00 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
 0.30  34000  1.50 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
 0.30  34000  2.00 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
 0.30  34000  2.50 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
 0.30  34000  3.00 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
 0.30  34000  3.50 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
 0.30  34000  4.00 -2.95385 -1.04776 -0.30334 -0.15252 -0.21104
 0.30  34000  4.50 -2.97899 -1.03181 -0.30566 -0.16027 -0.21590
 0.30  34000  5.00 -3.00391 -1.02231 -0.30213 -0.16293 -0.21688
 0.30  34000  5.50 -3.00391 -1.02231 -0.30213 -0.16293 -0.21688
 0.30  35000 -1.02 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
 0.30  35000 -0.70 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
 0.30  35000 -0.51 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
 0.30  35000 -0.29 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
 0.30  35000  0.00 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
 0.30  35000  0.28 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
 0.30  35000  0.50 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
 0.30  35000  0.60 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
 0.30  35000  0.87 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
 0.30  35000  1.00 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
 0.30  35000  1.50 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
 0.30  35000  2.00 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
 0.30  35000  2.50 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
 0.30  35000  3.00 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
 0.30  35000  3.50 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
 0.30  35000  4.00 -3.03124 -1.05687 -0.30522 -0.15133 -0.21089
 0.30  35000  4.50 -3.04809 -1.04294 -0.30835 -0.16031 -0.21740
 0.30  35000  5.00 -3.06963 -1.03422 -0.30595 -0.16402 -0.21917
 0.30  35000  5.50 -3.06963 -1.03422 -0.30595 -0.16402 -0.21917
 0.30  37500 -1.02 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
 0.30  37500 -0.70 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
 0.30  37500 -0.51 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
 0.30  37500 -0.29 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
 0.30  37500  0.00 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
 0.30  37500  0.28 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
 0.30  37500  0.50 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
 0.30  37500  0.60 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
 0.30  37500  0.87 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
 0.30  37500  1.00 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
 0.30  37500  1.50 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
 0.30  37500  2.00 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
 0.30  37500  2.50 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
 0.30  37500  3.00 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
 0.30  37500  3.50 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
 0.30  37500  4.00 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
 0.30  37500  4.50 -3.23200 -1.06372 -0.31400 -0.15851 -0.21827
 0.30  37500  5.00 -3.24088 -1.05803 -0.31302 -0.16379 -0.22209
 0.30  37500  5.50 -3.24088 -1.05803 -0.31302 -0.16379 -0.22209
 0.30  40000 -1.02 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
 0.30  40000 -0.70 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
 0.30  40000 -0.51 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
 0.30  40000 -0.29 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
 0.30  40000  0.00 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
 0.30  40000  0.28 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
 0.30  40000  0.50 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
 0.30  40000  0.60 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
 0.30  40000  0.87 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
 0.30  40000  1.00 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
 0.30  40000  1.50 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
 0.30  40000  2.00 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
 0.30  40000  2.50 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
 0.30  40000  3.00 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
 0.30  40000  3.50 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
 0.30  40000  4.00 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
 0.30  40000  4.50 -3.42597 -1.07634 -0.31968 -0.15850 -0.21954
 0.30  40000  5.00 -3.42493 -1.07283 -0.31872 -0.16266 -0.22304
 0.30  40000  5.50 -3.42493 -1.07283 -0.31872 -0.16266 -0.22304
 0.30  42500 -1.02 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
 0.30  42500 -0.70 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
 0.30  42500 -0.51 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
 0.30  42500 -0.29 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
 0.30  42500  0.00 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
 0.30  42500  0.28 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
 0.30  42500  0.50 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
 0.30  42500  0.60 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
 0.30  42500  0.87 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
 0.30  42500  1.00 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
 0.30  42500  1.50 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
 0.30  42500  2.00 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
 0.30  42500  2.50 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
 0.30  42500  3.00 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
 0.30  42500  3.50 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
 0.30  42500  4.00 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
 0.30  42500  4.50 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
 0.30  42500  5.00 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
 0.30  42500  5.50 -3.61259 -1.08316 -0.32449 -0.16314 -0.22488
 0.30  45000 -1.02 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
 0.30  45000 -0.70 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
 0.30  45000 -0.51 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
 0.30  45000 -0.29 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
 0.30  45000  0.00 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
 0.30  45000  0.28 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
 0.30  45000  0.50 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
 0.30  45000  0.60 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
 0.30  45000  0.87 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
 0.30  45000  1.00 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
 0.30  45000  1.50 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
 0.30  45000  2.00 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
 0.30  45000  2.50 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
 0.30  45000  3.00 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
 0.30  45000  3.50 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
 0.30  45000  4.00 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
 0.30  45000  4.50 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
 0.30  45000  5.00 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
 0.30  45000  5.50 -3.79341 -1.09075 -0.32914 -0.16449 -0.22689
 0.30  47500 -1.02 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
 0.30  47500 -0.70 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
 0.30  47500 -0.51 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
 0.30  47500 -0.29 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
 0.30  47500  0.00 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
 0.30  47500  0.28 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
 0.30  47500  0.50 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
 0.30  47500  0.60 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
 0.30  47500  0.87 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
 0.30  47500  1.00 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
 0.30  47500  1.50 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
 0.30  47500  2.00 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
 0.30  47500  2.50 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
 0.30  47500  3.00 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
 0.30  47500  3.50 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
 0.30  47500  4.00 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
 0.30  47500  4.50 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
 0.30  47500  5.00 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
 0.30  47500  5.50 -3.96500 -1.09723 -0.33315 -0.16581 -0.22859
 0.30  50000 -1.02 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
 0.30  50000 -0.70 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
 0.30  50000 -0.51 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
 0.30  50000 -0.29 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
 0.30  50000  0.00 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
 0.30  50000  0.28 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
 0.30  50000  0.50 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
 0.30  50000  0.60 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
 0.30  50000  0.87 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
 0.30  50000  1.00 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
 0.30  50000  1.50 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
 0.30  50000  2.00 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
 0.30  50000  2.50 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
 0.30  50000  3.00 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
 0.30  50000  3.50 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
 0.30  50000  4.00 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
 0.30  50000  4.50 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
 0.30  50000  5.00 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
 0.30  50000  5.50 -4.12837 -1.10284 -0.33629 -0.16706 -0.22980
);
   my @ubvri = @ubvri_lbc;
   my $log10 = log(10);

   if (-f "<ltg2ubv_new.dat") {
      open FILE, "<ltg2ubv_new.dat";
      $nzgr = <FILE>+1;
      $ntgr = <FILE>+1;
      $nggr = <FILE>+1;
      undef @ubvri;
      while (<FILE>) {
         chomp; split;
         push @ubvri, @_;
      }
      close FILE;
   }

   foreach my $k (0..$nzgr-1) {
      foreach my $i (0..$ntgr-1) {
         foreach my $j (0..$nggr-1) {
            $zgr[$k] = shift @ubvri;
            $tgr[$i] = shift @ubvri;
            $ggr[$j] = shift @ubvri;
            foreach (0..4) {
               $ubv[$k][$i][$j][$_] = shift @ubvri;
            }
         }
         $tgr[$i] = log($tgr[$i])/$log10;
      }
      $zgr[$k] = -log((3.0 + 37.425*10.0**(-$zgr[$k]))/38.0)/$log10;
   }
}
