module paquette_coefficients

   use real_kind
   
   implicit none
   
   ! DC & DD are spline coeffcients for repulsive potential (ion-ion or electron-electron interaction)
   real(double), save :: dc(4, 50, 3), dd(4, 50)

   ! DCAT & DDAT are for attractive potenial (ion-electron)
   real(double), save :: dcat(4, 50, 3), ddat(4, 50)

contains

  ! Initialise collision integrals for atomic diffusion
   subroutine initialise_collision_integrals
      use real_kind
      implicit none
      
      DC(1:4,1:25,1) = reshape( (/   &
             +1.19599D-02,-2.39198D-02,-3.02547D+01,-2.94860D+01,   &
             -2.39198D-02,-1.48010D-02,-2.94860D+01,-2.87231D+01,   &
             -1.48010D-02,-1.77390D-02,-2.87231D+01,-2.79637D+01,   &
             -1.77390D-02,-1.74423D-02,-2.79637D+01,-2.72086D+01,   &
             -1.74423D-02,-1.80040D-02,-2.72086D+01,-2.64576D+01,   &
             -1.80040D-02,-1.83218D-02,-2.64576D+01,-2.57110D+01,   &
             -1.83218D-02,-1.86847D-02,-2.57110D+01,-2.49688D+01,   &
             -1.86847D-02,-1.90073D-02,-2.49688D+01,-2.42310D+01,   &
             -1.90073D-02,-1.93026D-02,-2.42310D+01,-2.34978D+01,   &
             -1.93026D-02,-1.95555D-02,-2.34978D+01,-2.27693D+01,   &
             -1.95555D-02,-1.97557D-02,-2.27693D+01,-2.20454D+01,   &
             -1.97557D-02,-1.98886D-02,-2.20454D+01,-2.13263D+01,   &
             -1.98886D-02,-1.99373D-02,-2.13263D+01,-2.06120D+01,   &
             -1.99373D-02,-1.98810D-02,-2.06120D+01,-1.99024D+01,   &
             -1.98810D-02,-1.96948D-02,-1.99024D+01,-1.91976D+01,   &
             -1.96948D-02,-1.93486D-02,-1.91976D+01,-1.84975D+01,   &
             -1.93486D-02,-1.88059D-02,-1.84975D+01,-1.78021D+01,   &
             -1.88059D-02,-1.80227D-02,-1.78021D+01,-1.71112D+01,   &
             -1.80227D-02,-1.69459D-02,-1.71112D+01,-1.64246D+01,   &
             -1.69459D-02,-1.55109D-02,-1.64246D+01,-1.57421D+01,   &
             -1.55109D-02,-1.36394D-02,-1.57421D+01,-1.50633D+01,   &
             -1.36394D-02,-1.12361D-02,-1.50633D+01,-1.43878D+01,   &
             -1.12361D-02,-8.18466D-03,-1.43878D+01,-1.37150D+01,   &
             -8.18466D-03,-4.34258D-03,-1.37150D+01,-1.30441D+01,   &
             -4.34258D-03,+4.65253D-04,-1.30441D+01,-1.23743D+01    &
             /), (/ 4, 25 /) )

      DC(1:4,26:50,1) = reshape( (/   &
             +4.65253D-04,+6.45493D-03,-1.23743D+01,-1.17044D+01,   &
             +6.45493D-03,+1.38941D-02,-1.17044D+01,-1.10329D+01,   &
             +1.38941D-02,+2.31151D-02,-1.10329D+01,-1.03581D+01,   &
             +2.31151D-02,+3.45317D-02,-1.03581D+01,-9.67777D+00,   &
             +3.45317D-02,+4.86585D-02,-9.67777D+00,-8.98913D+00,   &
             +4.86585D-02,+6.61321D-02,-8.98913D+00,-8.28881D+00,   &
             +6.61321D-02,+8.77309D-02,-8.28881D+00,-7.57261D+00,   &
             +8.77309D-02,+1.14383D-01,-7.57261D+00,-6.83537D+00,   &
             +1.14383D-01,+1.47142D-01,-6.83537D+00,-6.07066D+00,   &
             +1.47142D-01,+1.87092D-01,-6.07066D+00,-5.27065D+00,   &
             +1.87092D-01,+2.35096D-01,-5.27065D+00,-4.42573D+00,   &
             +2.35096D-01,+2.91268D-01,-4.42573D+00,-3.52439D+00,   &
             +2.91268D-01,+3.53977D-01,-3.52439D+00,-2.55315D+00,   &
             +3.53977D-01,+4.18217D-01,-2.55315D+00,-1.49695D+00,   &
             +4.18217D-01,+4.73499D-01,-1.49695D+00,-3.40379D-01,   &
             +4.73499D-01,+5.02343D-01,-3.40379D-01,+9.29832D-01,   &
             +5.02343D-01,+4.82140D-01,+9.29832D-01,+2.32060D+00,   &
             +4.82140D-01,+3.92303D-01,+2.32060D+00,+3.82709D+00,   &
             +3.92303D-01,+2.20401D-01,+3.82709D+00,+5.42773D+00,   &
             +2.20401D-01,-5.31156D-02,+5.42773D+00,+7.08127D+00,   &
             -5.31156D-02,-3.94063D-01,+7.08127D+00,+8.72205D+00,   &
             -3.94063D-01,-5.99574D-01,+8.72205D+00,+1.02683D+01,   &
             -5.99574D-01,-4.71033D-01,+1.02683D+01,+1.16706D+01,   &
             -4.71033D-01,-4.68969D-01,+1.16706D+01,+1.29598D+01,   &
             -4.68969D-01,+2.34484D-01,+1.29598D+01,+1.41366D+01    &
             /), (/ 4, 25 /) )
      
      
      DC(1:4,1:25,2) = reshape( (/   &
             +1.34102D-02,-2.68205D-02,-2.55941D+01,-2.48408D+01,   &
             -2.68205D-02,-1.66309D-02,-2.48408D+01,-2.40939D+01,   &
             -1.66309D-02,-1.99547D-02,-2.40939D+01,-2.33511D+01,   &
             -1.99547D-02,-1.96575D-02,-2.33511D+01,-2.26130D+01,   &
             -1.96575D-02,-2.03264D-02,-2.26130D+01,-2.18796D+01,   &
             -2.03264D-02,-2.07272D-02,-2.18796D+01,-2.11511D+01,   &
             -2.07272D-02,-2.11843D-02,-2.11511D+01,-2.04276D+01,   &
             -2.11843D-02,-2.16034D-02,-2.04276D+01,-1.97091D+01,   &
             -2.16034D-02,-2.20005D-02,-1.97091D+01,-1.89959D+01,   &
             -2.20005D-02,-2.23602D-02,-1.89959D+01,-1.82879D+01,   &
             -2.23602D-02,-2.26732D-02,-1.82879D+01,-1.75853D+01,   &
             -2.26732D-02,-2.29256D-02,-1.75853D+01,-1.68881D+01,   &
             -2.29256D-02,-2.31017D-02,-1.68881D+01,-1.61965D+01,   &
             -2.31017D-02,-2.31822D-02,-1.61965D+01,-1.55103D+01,   &
             -2.31822D-02,-2.31439D-02,-1.55103D+01,-1.48298D+01,   &
             -2.31439D-02,-2.29591D-02,-1.48298D+01,-1.41548D+01,   &
             -2.29591D-02,-2.25945D-02,-1.41548D+01,-1.34853D+01,   &
             -2.25945D-02,-2.20100D-02,-1.34853D+01,-1.28212D+01,   &
             -2.20100D-02,-2.11574D-02,-1.28212D+01,-1.21624D+01,   &
             -2.11574D-02,-1.99787D-02,-1.21624D+01,-1.15087D+01,   &
             -1.99787D-02,-1.84040D-02,-1.15087D+01,-1.08598D+01,   &
             -1.84040D-02,-1.63488D-02,-1.08598D+01,-1.02153D+01,   &
             -1.63488D-02,-1.37108D-02,-1.02153D+01,-9.57474D+00,   &
             -1.37108D-02,-1.03658D-02,-9.57474D+00,-8.93745D+00,   &
             -1.03658D-02,-6.16232D-03,-8.93745D+00,-8.30266D+00    &   
             /), (/ 4, 25 /) )
      
      DC(1:4,26:50,2) = reshape( (/   &
             -6.16232D-03,-9.15489D-04,-8.30266D+00,-7.66934D+00,   &
             -9.15489D-04,+5.60137D-03,-7.66934D+00,-7.03625D+00,   &
             +5.60137D-03,+1.36664D-02,-7.03625D+00,-6.40181D+00,   &
             +1.36664D-02,+2.36209D-02,-6.40181D+00,-5.76409D+00,   &
             +2.36209D-02,+3.58820D-02,-5.76409D+00,-5.12070D+00,   &
             +3.58820D-02,+5.09546D-02,-5.12070D+00,-4.46870D+00,   &
             +5.09546D-02,+6.94376D-02,-4.46870D+00,-3.80447D+00,   &
             +6.94376D-02,+9.20141D-02,-3.80447D+00,-3.12357D+00,   &
             +9.20141D-02,+1.19403D-01,-3.12357D+00,-2.42059D+00,   &
             +1.19403D-01,+1.52233D-01,-2.42059D+00,-1.68896D+00,   &
             +1.52233D-01,+1.90763D-01,-1.68896D+00,-9.20788D-01,   &
             +1.90763D-01,+2.34333D-01,-9.20788D-01,-1.06834D-01,   &
             +2.34333D-01,+2.80392D-01,-1.06834D-01,+7.63360D-01,   &
             +2.80392D-01,+3.23003D-01,+7.63360D-01,+1.70085D+00,   &
             +3.23003D-01,+3.51091D-01,+1.70085D+00,+2.71586D+00,   &
             +3.51091D-01,+3.47669D-01,+2.71586D+00,+3.81513D+00,   &
             +3.47669D-01,+2.93155D-01,+3.81513D+00,+4.99784D+00,   &
             +2.93155D-01,+1.77064D-01,+4.99784D+00,+6.25091D+00,   &
             +1.77064D-01,+1.68817D-02,+6.25091D+00,+7.54647D+00,   &
             +1.68817D-02,-1.37836D-01,+7.54647D+00,+8.84609D+00,   &
             -1.37836D-01,-2.36519D-01,+8.84609D+00,+1.01126D+01,   &
             -2.36519D-01,-2.49035D-01,+1.01126D+01,+1.13224D+01,   &
             -2.49035D-01,-1.95933D-01,+1.13224D+01,+1.24724D+01,   &
             -1.95933D-01,-1.60453D-01,+1.24724D+01,+1.35754D+01,   &
             -1.60453D-01,+8.02267D-02,+1.35754D+01,+1.46398D+01    &
             /), (/ 4, 25 /) )
      
      
      DC(1:4,1:25,3) = reshape( (/   &
             +1.46130D-02,-2.92259D-02,-1.93212D+01,-1.85804D+01,   &
             -2.92259D-02,-1.81472D-02,-1.85804D+01,-1.78467D+01,   &
             -1.81472D-02,-2.17897D-02,-1.78467D+01,-1.71173D+01,   &
             -2.17897D-02,-2.14905D-02,-1.71173D+01,-1.63931D+01,   &
             -2.14905D-02,-2.22463D-02,-1.63931D+01,-1.56741D+01,   &
             -2.22463D-02,-2.27134D-02,-1.56741D+01,-1.49604D+01,   &
             -2.27134D-02,-2.32457D-02,-1.49604D+01,-1.42522D+01,   &
             -2.32457D-02,-2.37412D-02,-1.42522D+01,-1.35496D+01,   &
             -2.37412D-02,-2.42183D-02,-1.35496D+01,-1.28526D+01,   &
             -2.42183D-02,-2.46613D-02,-1.28526D+01,-1.21615D+01,   &
             -2.46613D-02,-2.50615D-02,-1.21615D+01,-1.14763D+01,   &
             -2.50615D-02,-2.54057D-02,-1.14763D+01,-1.07971D+01,   &
             -2.54057D-02,-2.56787D-02,-1.07971D+01,-1.01240D+01,   &
             -2.56787D-02,-2.58623D-02,-1.01240D+01,-9.45709D+00,   &
             -2.58623D-02,-2.59347D-02,-9.45709D+00,-8.79637D+00,   &
             -2.59347D-02,-2.58697D-02,-8.79637D+00,-8.14187D+00,   &
             -2.58697D-02,-2.56362D-02,-8.14187D+00,-7.49358D+00,   &
             -2.56362D-02,-2.51970D-02,-7.49358D+00,-6.85144D+00,   &
             -2.51970D-02,-2.45075D-02,-6.85144D+00,-6.21534D+00,   &
             -2.45075D-02,-2.35146D-02,-6.21534D+00,-5.58513D+00,   &
             -2.35146D-02,-2.21544D-02,-5.58513D+00,-4.96057D+00,   &
             -2.21544D-02,-2.03503D-02,-4.96057D+00,-4.34132D+00,   &
             -2.03503D-02,-1.80104D-02,-4.34132D+00,-3.72695D+00,   &
             -1.80104D-02,-1.50236D-02,-3.72695D+00,-3.11691D+00,   &
             -1.50236D-02,-1.12556D-02,-3.11691D+00,-2.51047D+00    &
             /), (/ 4, 25 /) )
      
      DC(1:4,26:50,3) = reshape( (/   &
             -1.12556D-02,-6.54378D-03,-2.51047D+00,-1.90674D+00,   &
             -6.54378D-03,-6.90518D-04,-1.90674D+00,-1.30457D+00,   &
             -6.90518D-04,+6.54438D-03,-1.30457D+00,-7.02574D-01,   &
             +6.54438D-03,+1.54523D-02,-7.02574D-01,-9.90038D-02,   &
             +1.54523D-02,+2.63848D-02,-9.90038D-02,+5.08275D-01,   &
             +2.63848D-02,+3.97599D-02,+5.08275D-01,+1.12189D+00,   &
             +3.97599D-02,+5.60625D-02,+1.12189D+00,+1.74504D+00,   &
             +5.60625D-02,+7.58259D-02,+1.74504D+00,+2.38165D+00,   &
             +7.58259D-02,+9.95773D-02,+2.38165D+00,+3.03645D+00,   &
             +9.95773D-02,+1.27708D-01,+3.03645D+00,+3.71516D+00,   &
             +1.27708D-01,+1.60207D-01,+3.71516D+00,+4.42451D+00,   &
             +1.60207D-01,+1.96165D-01,+4.42451D+00,+5.17232D+00,   &
             +1.96165D-01,+2.32939D-01,+5.17232D+00,+5.96720D+00,   &
             +2.32939D-01,+2.64985D-01,+5.96720D+00,+6.81799D+00,   &
             +2.64985D-01,+2.82691D-01,+6.81799D+00,+7.73238D+00,   &
             +2.82691D-01,+2.72481D-01,+7.73238D+00,+8.71461D+00,   &
             +2.72481D-01,+2.20588D-01,+8.71461D+00,+9.76224D+00,   &
             +2.20588D-01,+1.23271D-01,+9.76224D+00,+1.08628D+01,   &
             +1.23271D-01,+5.64105D-05,+1.08628D+01,+1.19930D+01,   &
             +5.64105D-05,-1.01669D-01,+1.19930D+01,+1.31231D+01,   &
             -1.01669D-01,-1.45351D-01,+1.31231D+01,+1.42289D+01,   &
             -1.45351D-01,-1.12670D-01,+1.42289D+01,+1.52998D+01,   &
             -1.12670D-01,-1.25510D-01,+1.52998D+01,+1.63436D+01,   &
             -1.25510D-01,+4.14471D-02,+1.63436D+01,+1.73573D+01,   &
             +4.14471D-02,-2.07236D-02,+1.73573D+01,+1.83810D+01    &
             /), (/ 4, 25 /) )
      
      
      DD(1:4,1:25) = reshape( (/   &
             +1.18229D-02,-2.36458D-02,-2.55112D+01,-2.47319D+01,   &
             -2.36458D-02,-1.46794D-02,-2.47319D+01,-2.39583D+01,   &
             -1.46794D-02,-1.76226D-02,-2.39583D+01,-2.31882D+01,   &
             -1.76226D-02,-1.73748D-02,-2.31882D+01,-2.24223D+01,   &
             -1.73748D-02,-1.79780D-02,-2.24223D+01,-2.16606D+01,   &
             -1.79780D-02,-1.83439D-02,-2.16606D+01,-2.09032D+01,   &
             -1.83439D-02,-1.87580D-02,-2.09032D+01,-2.01502D+01,   &
             -1.87580D-02,-1.91359D-02,-2.01502D+01,-1.94017D+01,   &
             -1.91359D-02,-1.94907D-02,-1.94017D+01,-1.86579D+01,   &
             -1.94907D-02,-1.98074D-02,-1.86579D+01,-1.79186D+01,   &
             -1.98074D-02,-2.00758D-02,-1.79186D+01,-1.71842D+01,   &
             -2.00758D-02,-2.02817D-02,-1.71842D+01,-1.64545D+01,   &
             -2.02817D-02,-2.04080D-02,-1.64545D+01,-1.57298D+01,   &
             -2.04080D-02,-2.04344D-02,-1.57298D+01,-1.50099D+01,   &
             -2.04344D-02,-2.03360D-02,-1.50099D+01,-1.42949D+01,   &
             -2.03360D-02,-2.00828D-02,-1.42949D+01,-1.35848D+01,   &
             -2.00828D-02,-1.96388D-02,-1.35848D+01,-1.28796D+01,   &
             -1.96388D-02,-1.89603D-02,-1.28796D+01,-1.21790D+01,   &
             -1.89603D-02,-1.79946D-02,-1.21790D+01,-1.14830D+01,   &
             -1.79946D-02,-1.66781D-02,-1.14830D+01,-1.07913D+01,   &
             -1.66781D-02,-1.49334D-02,-1.07913D+01,-1.01036D+01,   &
             -1.49334D-02,-1.26670D-02,-1.01036D+01,-9.41952D+00,   &
             -1.26670D-02,-9.76488D-03,-9.41952D+00,-8.73845D+00,   &
             -9.76488D-03,-6.08817D-03,-8.73845D+00,-8.05974D+00,   &
             -6.08817D-03,-1.46688D-03,-8.05974D+00,-7.38248D+00    &
             /), (/ 4, 25 /) )

      DD(1:4,26:50) = reshape( (/   &
             -1.46688D-03,+4.30758D-03,-7.38248D+00,-6.70557D+00,   &
             +4.30758D-03,+1.14919D-02,-6.70557D+00,-6.02763D+00,   &
             +1.14919D-02,+2.04026D-02,-6.02763D+00,-5.34694D+00,   &
             +2.04026D-02,+3.14302D-02,-5.34694D+00,-4.66134D+00,   &
             +3.14302D-02,+4.50547D-02,-4.66134D+00,-3.96821D+00,   &
             +4.50547D-02,+6.18616D-02,-3.96821D+00,-3.26426D+00,   &
             +6.18616D-02,+8.25526D-02,-3.26426D+00,-2.54546D+00,   &
             +8.25526D-02,+1.07939D-01,-2.54546D+00,-1.80685D+00,   &
             +1.07939D-01,+1.38897D-01,-1.80685D+00,-1.04234D+00,   &
             +1.38897D-01,+1.76234D-01,-1.04234D+00,-2.44487D-01,   &
             +1.76234D-01,+2.20387D-01,-2.44487D-01,+5.95659D-01,   &
             +2.20387D-01,+2.70814D-01,+5.95659D-01,+1.48870D+00,   &
             +2.70814D-01,+3.24880D-01,+1.48870D+00,+2.44673D+00,   &
             +3.24880D-01,+3.76096D-01,+2.44673D+00,+3.48274D+00,   &
             +3.76096D-01,+4.11895D-01,+3.48274D+00,+4.60901D+00,   &
             +4.11895D-01,+4.12194D-01,+4.60901D+00,+5.83413D+00,   &
             +4.12194D-01,+3.52176D-01,+5.83413D+00,+7.15818D+00,   &
             +3.52176D-01,+2.14860D-01,+7.15818D+00,+8.56675D+00,   &
             +2.14860D-01,+1.56932D-02,+8.56675D+00,+1.00269D+01,   &
             +1.56932D-02,-1.84529D-01,+1.00269D+01,+1.14908D+01,   &
             -1.84529D-01,-3.17740D-01,+1.14908D+01,+1.29104D+01,   &
             -3.17740D-01,-3.51796D-01,+1.29104D+01,+1.42538D+01,   &
             -3.51796D-01,-2.86379D-01,+1.42538D+01,+1.55127D+01,   &
             -2.86379D-01,-2.68214D-01,+1.55127D+01,+1.67029D+01,   &
             -2.68214D-01,+1.34107D-01,+1.67029D+01,+1.78287D+01   &
             /), (/ 4, 25 /) )
      
      
      
      
      
      DCAT(1:4,1:25,1) = reshape( (/   &
             -4.85605D+00,+9.71211D+00,-2.44778D+01,-2.50688D+01,   &
             +9.71211D+00,-9.31384D+00,-2.50688D+01,-2.33288D+01,   &
             -9.31384D+00,+9.24600D+00,-2.33288D+01,-2.38242D+01,   &
             +9.24600D+00,-5.09678D+00,-2.38242D+01,-2.21005D+01,   &
             -5.09678D+00,-3.51130D-01,-2.21005D+01,-2.16001D+01,   &
             -3.51130D-01,-5.73409D-01,-2.16001D+01,-2.11839D+01,   &
             -5.73409D-01,+1.83302D+00,-2.11839D+01,-2.09054D+01,   &
             +1.83302D+00,-7.68878D-01,-2.09054D+01,-2.01869D+01,   &
             -7.68878D-01,+9.36371D-01,-2.01869D+01,-1.96529D+01,   &
             +9.36371D-01,-8.65172D-01,-1.96529D+01,-1.88943D+01,   &
             -8.65172D-01,-1.42540D+00,-1.88943D+01,-1.83432D+01,   &
             -1.42540D+00,+4.34705D+00,-1.83432D+01,-1.81343D+01,   &
             +4.34705D+00,+3.61550D-01,-1.81343D+01,-1.68820D+01,   &
             +3.61550D-01,-3.50574D+00,-1.68820D+01,-1.55430D+01,   &
             -3.50574D+00,+2.59550D+00,-1.55430D+01,-1.50454D+01,   &
             +2.59550D+00,-1.44563D+00,-1.50454D+01,-1.39249D+01,   &
             -1.44563D+00,-8.64470D-01,-1.39249D+01,-1.31513D+01,   &
             -8.64470D-01,+8.82476D-01,-1.31513D+01,-1.25851D+01,   &
             +8.82476D-01,+1.35231D-01,-1.25851D+01,-1.18072D+01,   &
             +1.35231D-01,+8.48986D-02,-1.18072D+01,-1.09968D+01,   &
             +8.48986D-02,-5.89033D-02,-1.09968D+01,-1.01661D+01,   &
             -5.89033D-02,-1.07275D-01,-1.01661D+01,-9.34947D+00,   &
             -1.07275D-01,-1.53774D-01,-9.34947D+00,-8.55860D+00,   &
             -1.53774D-01,-6.39482D-02,-8.55860D+00,-7.80464D+00,   &
             -6.39482D-02,-1.27604D-01,-7.80464D+00,-7.06602D+00    &
             /), (/ 4, 25 /) )
      
      DCAT(1:4,26:50,1) = reshape( (/   &
             -1.27604D-01,-3.16007D-01,-7.06602D+00,-6.35803D+00,   &
             -3.16007D-01,-7.55843D-02,-6.35803D+00,-5.72588D+00,   &
             -7.55843D-02,+6.48945D-02,-5.72588D+00,-5.11188D+00,   &
             +6.48945D-02,+5.15827D-02,-5.11188D+00,-4.48229D+00,   &
             +5.15827D-02,-4.79565D-01,-4.48229D+00,-3.84033D+00,   &
             -4.79565D-01,+3.59783D-01,-3.84033D+00,-3.31347D+00,   &
             +3.59783D-01,-3.04381D-01,-3.31347D+00,-2.70025D+00,   &
             -3.04381D-01,+1.12514D-01,-2.70025D+00,-2.16009D+00,   &
             +1.12514D-01,+1.90268D-01,-2.16009D+00,-1.59292D+00,   &
             +1.90268D-01,+1.96365D-01,-1.59292D+00,-9.80089D-01,   &
             +1.96365D-01,+1.78507D-01,-9.80089D-01,-3.20131D-01,   &
             +1.78507D-01,+1.77190D-01,-3.20131D-01,+3.82669D-01,   &
             +1.77190D-01,+1.92652D-01,+3.82669D-01,+1.12800D+00,   &
             +1.92652D-01,+2.20472D-01,+1.12800D+00,+1.91956D+00,   &
             +2.20472D-01,+2.52940D-01,+1.91956D+00,+2.76403D+00,   &
             +2.52940D-01,+2.82221D-01,+2.76403D+00,+3.66921D+00,   &
             +2.82221D-01,+3.00417D-01,+3.66921D+00,+4.64213D+00,   &
             +3.00417D-01,+2.99089D-01,+4.64213D+00,+5.68714D+00,   &
             +2.99089D-01,+2.69658D-01,+5.68714D+00,+6.80394D+00,   &
             +2.69658D-01,+2.08129D-01,+6.80394D+00,+7.98545D+00,   &
             +2.08129D-01,+1.12949D-01,+7.98545D+00,+9.21692D+00,   &
             +1.12949D-01,+9.86067D-03,+9.21692D+00,+1.04755D+01,   &
             +9.86067D-03,-1.12290D-01,+1.04755D+01,+1.17364D+01,   &
             -1.12290D-01,-2.95924D-02,+1.17364D+01,+1.29704D+01,   &
             -2.95924D-02,+1.47962D-02,+1.29704D+01,+1.41973D+01    &
             /), (/ 4, 25 /) )
      
      
      DCAT(1:4,1:25,2) = reshape( (/   &
             -3.80453D+00,+7.60906D+00,-2.08526D+01,-2.11137D+01,   &
             +7.60906D+00,-1.00677D+01,-2.11137D+01,-1.95486D+01,   &
             -1.00677D+01,+1.09298D+01,-1.95486D+01,-2.03997D+01,   &
             +1.09298D+01,-3.76938D+00,-2.03997D+01,-1.86277D+01,   &
             -3.76938D+00,-2.50868D+00,-1.86277D+01,-1.77604D+01,   &
             -2.50868D+00,+2.83415D+00,-1.77604D+01,-1.74951D+01,   &
             +2.83415D+00,-1.06745D+00,-1.74951D+01,-1.65496D+01,   &
             -1.06745D+00,-2.73597D-01,-1.65496D+01,-1.58604D+01,   &
             -2.73597D-01,+5.70462D-01,-1.58604D+01,-1.52368D+01,   &
             +5.70462D-01,+1.17072D-01,-1.52368D+01,-1.44762D+01,   &
             +1.17072D-01,-4.26755D-01,-1.44762D+01,-1.36876D+01,   &
             -4.26755D-01,+1.84166D+00,-1.36876D+01,-1.30014D+01,   &
             +1.84166D+00,-4.51420D-02,-1.30014D+01,-1.18732D+01,   &
             -4.51420D-02,-1.68106D+00,-1.18732D+01,-1.07559D+01,   &
             -1.68106D+00,+7.40637D-01,-1.07559D+01,-1.00419D+01,   &
             +7.40637D-01,-4.15277D-01,-1.00419D+01,-9.15028D+00,   &
             -4.15277D-01,-5.31841D-01,-9.15028D+00,-8.35829D+00,   &
             -5.31841D-01,+7.54067D-02,-8.35829D+00,-7.69393D+00,   &
             +7.54067D-02,+3.64437D-02,-7.69393D+00,-7.01148D+00,   &
             +3.64437D-02,+8.52374D-03,-7.01148D+00,-6.32029D+00,   &
             +8.52374D-03,-5.24956D-02,-6.32029D+00,-5.62704D+00,   &
             -5.24956D-02,-8.76527D-02,-5.62704D+00,-4.94640D+00,   &
             -8.76527D-02,-1.07693D-01,-4.94640D+00,-4.28679D+00,   &
             -1.07693D-01,-1.02441D-01,-4.28679D+00,-3.65303D+00,   &
             -1.02441D-01,-1.04497D-01,-3.65303D+00,-3.04386D+00    &
             /), (/ 4, 25 /) )
      
      DCAT(1:4,26:50,2) = reshape( (/   &
             -1.04497D-01,-1.16430D-01,-3.04386D+00,-2.45976D+00,   &
             -1.16430D-01,-8.20446D-02,-2.45976D+00,-1.90361D+00,   &
             -8.20446D-02,-5.53768D-02,-1.90361D+00,-1.36715D+00,   &
             -5.53768D-02,-4.31036D-02,-1.36715D+00,-8.43977D-01,   &
             -4.31036D-02,-8.94580D-02,-8.43977D-01,-3.31151D-01,   &
             -8.94580D-02,+1.74820D-02,-3.31151D-01,+1.60205D-01,   &
             +1.74820D-02,-4.17248D-02,+1.60205D-01,+6.55758D-01,   &
             -4.17248D-02,+2.16009D-02,+6.55758D-01,+1.14130D+00,   &
             +2.16009D-02,+5.02965D-02,+1.14130D+00,+1.63202D+00,   &
             +5.02965D-02,+7.34150D-02,+1.63202D+00,+2.13481D+00,   &
             +7.34150D-02,+9.68269D-02,+2.13481D+00,+2.65522D+00,   &
             +9.68269D-02,+1.25372D-01,+2.65522D+00,+3.19888D+00,   &
             +1.25372D-01,+1.59190D-01,+3.19888D+00,+3.77262D+00,   &
             +1.59190D-01,+1.96991D-01,+3.77262D+00,+4.38456D+00,   &
             +1.96991D-01,+2.35595D-01,+4.38456D+00,+5.04379D+00,   &
             +2.35595D-01,+2.69792D-01,+5.04379D+00,+5.75956D+00,   &
             +2.69792D-01,+2.92272D-01,+5.75956D+00,+6.54007D+00,   &
             +2.92272D-01,+2.95067D-01,+6.54007D+00,+7.39073D+00,   &
             +2.95067D-01,+2.70564D-01,+7.39073D+00,+8.31221D+00,   &
             +2.70564D-01,+2.20440D-01,+8.31221D+00,+9.29863D+00,   &
             +2.20440D-01,+1.38511D-01,+9.29863D+00,+1.03379D+01,   &
             +1.38511D-01,+8.80848D-02,+1.03379D+01,+1.14105D+01,   &
             +8.80848D-02,-7.78941D-02,+1.14105D+01,+1.25042D+01,   &
             -7.78941D-02,+3.60071D-01,+1.25042D+01,+1.35792D+01,   &
             +3.60071D-01,-1.80035D-01,+1.35792D+01,+1.47406D+01    &
             /), (/ 4, 25 /) )
      

      DCAT(1:4,1:25,3) = reshape( (/   &
             -2.06818D+00,+4.13636D+00,-1.54059D+01,-1.51150D+01,   &
             +4.13636D+00,-8.93460D+00,-1.51150D+01,-1.38314D+01,   &
             -8.93460D+00,+1.08282D+01,-1.38314D+01,-1.46921D+01,   &
             +1.08282D+01,-2.97377D+00,-1.46921D+01,-1.29540D+01,   &
             -2.97377D+00,-2.25377D+00,-1.29540D+01,-1.19296D+01,   &
             -2.25377D+00,+2.79041D+00,-1.19296D+01,-1.14462D+01,   &
             +2.79041D+00,-1.37253D+00,-1.14462D+01,-1.02930D+01,   &
             -1.37253D+00,-3.87183D-01,-1.02930D+01,-9.46928D+00,   &
             -3.87183D-01,+3.36588D-01,-9.46928D+00,-8.73845D+00,   &
             +3.36588D-01,+2.05509D-01,-8.73845D+00,-7.92684D+00,   &
             +2.05509D-01,-9.13383D-02,-7.92684D+00,-7.06591D+00,   &
             -9.13383D-02,+5.02385D-01,-7.06591D+00,-6.22691D+00,   &
             +5.02385D-01,-1.25938D-01,-6.22691D+00,-5.26733D+00,   &
             -1.25938D-01,-6.87925D-01,-5.26733D+00,-4.33797D+00,   &
             -6.87925D-01,+8.12958D-02,-4.33797D+00,-3.57372D+00,   &
             +8.12958D-02,-2.46616D-01,-3.57372D+00,-2.78995D+00,   &
             -2.46616D-01,-3.03295D-01,-2.78995D+00,-2.06538D+00,   &
             -3.03295D-01,-9.83392D-02,-2.06538D+00,-1.41359D+00,   &
             -9.83392D-02,-7.94911D-02,-1.41359D+00,-7.85408D-01,   &
             -7.94911D-02,-7.66879D-02,-7.85408D-01,-1.76302D-01,   &
             -7.66879D-02,-8.63431D-02,-1.76302D-01,+4.14399D-01,   &
             -8.63431D-02,-9.05049D-02,+4.14399D-01,+9.84378D-01,   &
             -9.05049D-02,-9.00801D-02,+9.84378D-01,+1.53264D+00,   &
             -9.00801D-02,-8.47287D-02,+1.53264D+00,+2.05927D+00,   &
             -8.47287D-02,-7.78775D-02,+2.05927D+00,+2.56558D+00    &
             /), (/ 4, 25 /) )
      
      DCAT(1:4,26:50,3) = reshape( (/   &
             -7.78775D-02,-7.05918D-02,+2.56558D+00,+3.05319D+00,   &
             -7.05918D-02,-5.90727D-02,+3.05319D+00,+3.52386D+00,   &
             -5.90727D-02,-4.74236D-02,+3.52386D+00,+3.98035D+00,   &
             -4.74236D-02,-3.62205D-02,+3.98035D+00,+4.42546D+00,   &
             -3.62205D-02,-2.90446D-02,+4.42546D+00,+4.86188D+00,   &
             -2.90446D-02,-8.69943D-03,+4.86188D+00,+5.29133D+00,   &
             -8.69943D-03,-2.75832D-04,+5.29133D+00,+5.71869D+00,   &
             -2.75832D-04,+1.96726D-02,+5.71869D+00,+6.14599D+00,   &
             +1.96726D-02,+3.91533D-02,+6.14599D+00,+6.57800D+00,   &
             +3.91533D-02,+6.09268D-02,+6.57800D+00,+7.01941D+00,   &
             +6.09268D-02,+8.60213D-02,+7.01941D+00,+7.47545D+00,   &
             +8.60213D-02,+1.15234D-01,+7.47545D+00,+7.95213D+00,   &
             +1.15234D-01,+1.48588D-01,+7.95213D+00,+8.45646D+00,   &
             +1.48588D-01,+1.85161D-01,+8.45646D+00,+8.99646D+00,   &
             +1.85161D-01,+2.22451D-01,+8.99646D+00,+9.58090D+00,   &
             +2.22451D-01,+2.55866D-01,+9.58090D+00,+1.02187D+01,   &
             +2.55866D-01,+2.78501D-01,+1.02187D+01,+1.09180D+01,   &
             +2.78501D-01,+2.83080D-01,+1.09180D+01,+1.16840D+01,   &
             +2.83080D-01,+2.62093D-01,+1.16840D+01,+1.25180D+01,   &
             +2.62093D-01,+2.21026D-01,+1.25180D+01,+1.34149D+01,   &
             +2.21026D-01,+1.44689D-01,+1.34149D+01,+1.43649D+01,   &
             +1.44689D-01,+1.33750D-01,+1.43649D+01,+1.53496D+01,   &
             +1.33750D-01,-1.00279D-01,+1.53496D+01,+1.63664D+01,   &
             -1.00279D-01,+7.07988D-01,+1.63664D+01,+1.73591D+01,   &
             +7.07988D-01,-3.53994D-01,+1.73591D+01,+1.85217D+01    &
             /), (/ 4, 25 /) )
      
      
      DDAT(1:4,1:25) = reshape( (/   &
             -1.70341D+00,+3.40681D+00,-2.30677D+01,-2.36179D+01,   &
             +3.40681D+00,-1.49538D-01,-2.36179D+01,-2.33505D+01,   &
             -1.49538D-01,+5.36125D+00,-2.33505D+01,-2.31190D+01,   &
             +5.36125D+00,-6.27728D+00,-2.31190D+01,-2.16008D+01,   &
             -6.27728D+00,+7.81096D+00,-2.16008D+01,-2.15892D+01,   &
             +7.81096D+00,-6.95942D-01,-2.15892D+01,-1.97029D+01,   &
             -6.95942D-01,-4.07855D+00,-1.97029D+01,-1.79836D+01,   &
             -4.07855D+00,-2.58381D+00,-1.79836D+01,-1.72432D+01,   &
             -2.58381D+00,-9.97977D-01,-1.72432D+01,-1.71229D+01,   &
             -9.97977D-01,+5.38341D+00,-1.71229D+01,-1.72422D+01,   &
             +5.38341D+00,-2.04809D+00,-1.72422D+01,-1.60694D+01,   &
             -2.04809D+00,+3.50576D+00,-1.60694D+01,-1.53881D+01,   &
             +3.50576D+00,-4.79859D+00,-1.53881D+01,-1.38655D+01,   &
             -4.79859D+00,+1.16840D+00,-1.38655D+01,-1.34945D+01,   &
             +1.16840D+00,+8.19462D-01,-1.34945D+01,-1.28431D+01,   &
             +8.19462D-01,-1.82778D+00,-1.28431D+01,-1.19950D+01,   &
             -1.82778D+00,-1.65089D-02,-1.19950D+01,-1.15856D+01,   &
             -1.65089D-02,-4.16448D-01,-1.15856D+01,-1.11802D+01,   &
             -4.16448D-01,+1.88821D+00,-1.11802D+01,-1.08747D+01,   &
             +1.88821D+00,+8.26616D-01,-1.08747D+01,-1.01161D+01,   &
             +8.26616D-01,-2.03576D-01,-1.01161D+01,-9.15902D+00,   &
             -2.03576D-01,-4.49988D-01,-9.15902D+00,-8.25083D+00,   &
             -4.49988D-01,-3.23824D-01,-8.25083D+00,-7.45064D+00,   &
             -3.23824D-01,-1.52158D-01,-7.45064D+00,-6.72817D+00,   &
             -1.52158D-01,-3.19250D-02,-6.72817D+00,-6.04221D+00    &
             /), (/ 4, 25 /) )
             
      DDAT(1:4,26:50) = reshape( (/   &
             -3.19250D-02,+7.02209D-02,-6.04221D+00,-5.36392D+00,   &
             +7.02209D-02,+4.52255D-02,-5.36392D+00,-4.66877D+00,   &
             +4.52255D-02,+1.46629D-01,-4.66877D+00,-3.96278D+00,   &
             +1.46629D-01,+9.38012D-03,-3.96278D+00,-3.22158D+00,   &
             +9.38012D-03,+1.43265D-01,-3.22158D+00,-2.47814D+00,   &
             +1.43265D-01,+7.07177D-02,-2.47814D+00,-1.70032D+00,   &
             +7.07177D-02,+4.40308D-02,-1.70032D+00,-9.05521D-01,   &
             +4.40308D-02,+4.86217D-02,-9.05521D-01,-1.00156D-01,   &
             +4.86217D-02,+1.51036D-01,-1.00156D-01,+7.16878D-01,   &
             +1.51036D-01,+7.56348D-02,+7.16878D-01,+1.57016D+00,   &
             +7.56348D-02,+6.35608D-02,+1.57016D+00,+2.44160D+00,   &
             +6.35608D-02,+1.03660D-01,+2.44160D+00,+3.32829D+00,   &
             +1.03660D-01,+1.48495D-01,+3.32829D+00,+4.23985D+00,   &
             +1.48495D-01,+1.87179D-01,+4.23985D+00,+5.18706D+00,   &
             +1.87179D-01,+2.19549D-01,+5.18706D+00,+6.17919D+00,   &
             +2.19549D-01,+2.34271D-01,+6.17919D+00,+7.22401D+00,   &
             +2.34271D-01,+2.18239D-01,+7.22401D+00,+8.32506D+00,   &
             +2.18239D-01,+1.72842D-01,+8.32506D+00,+9.47848D+00,   &
             +1.72842D-01,+1.06073D-01,+9.47848D+00,+1.06734D+01,   &
             +1.06073D-01,+3.51672D-02,+1.06734D+01,+1.18938D+01,   &
             +3.51672D-02,-3.76546D-02,+1.18938D+01,+1.31226D+01,   &
             -3.76546D-02,-5.81893D-02,+1.31226D+01,+1.43423D+01,   &
             -5.81893D-02,-1.82463D-01,+1.43423D+01,+1.55481D+01,   &
             -1.82463D-01,+2.72096D-01,+1.55481D+01,+1.67101D+01,   &
             +2.72096D-01,-1.36048D-01,+1.67101D+01,+1.79374D+01    &
             /), (/ 4, 25 /) )
      
   end subroutine initialise_collision_integrals

end module

