# rm(list=ls())

#.######################################################################.#
#                                                                        #
#               FASTING ENDURANCE DYNAMIC MODEL                          #
#                             by                                         #
#            L. Monica Trondrud & Gabriel Pigeon                         #
#                                                                        #
#                         July 2020                                      #
#                                                                        #
#.######################################################################.#

### Info & packages -------

# The following model is a modification of the mathematical model by
# "Speakman & Westerterp (2013). A mathematical model of weight loss under 
# total starvation: evidence against the thrifty-gene hypothesis.
# Disease Models & Mechanisms 2013 6: 236-251; doi: 10.1242/dmm.010009"

# This R script details a loop to calculate fasting endurance of an animal
# The model is used to calculate fasting endurance in the following papers:

# Trondrud et al. 2021 Fat storage influences fasting endurance more than body size in an ungulate.
# Accepted in Functional Ecology

# Oldakowski et al. manuscript

# Many of the steps and input can be replaced or modified to other data sets, 
# these are indicated comments

# If the script is used and/or modified to calculate fasting endurance, please
# cite Trondrud et al. 2021 Functional Ecology
# and notify the creators (Trondrud or Pigeon) before publishing.

# Important information #######
# The time it takes for the model to run depends on the length of the data set 
# (assumed duration of fast), the number of individuals used and the number of 
# iterations run per individual.
# We use doParallel and foreach to run paralell calculations on mulitple cores. 
# It is important to adjust the number of cores (x) in "registerDoParallel(x)" 
# to match the capacity of your computers processor. 
# We advise to use n-1 cores (if your processor has 8 cores, use max. 7).

# Relevant packages:
library(graphics)
library(ggplot2)
library(lattice)
library(readxl)
library(lubridate)
library(mgcv)
library(VGAM)
library(lme4)
library(lmerTest)
library(foreach)
library(readr)
library(doParallel)

### 1. LOAD DATA  --------------------------------

#### 1a) Activity data & model --------
#data set
actdat <- read_rds(".\\data\\activity_for_web_model.Rds") 
#model
load(".\\models\\unscaled.activity.model.bam2.Rdata") #model name is act.m2c
summary(act.m2c)

#### 1b) Temperature data & model --------
#data set
tempdat <- read_rds( ".\\data\\tempdat_for_model.Rds") 

#model
load(".\\models\\unscaled.sc.temp.model.bam.Rdata") #model name is temp.mb
summary(temp.m2b)

#### 1c) DEE data & model --------
#data set
dlw<- read_rds(".\\data\\data_dee_model_ffm2.Rds") #data with 21 individuals
dlw2<-read_rds(".\\data\\data_dee_model_subcut_only2_ffm2.Rds") #data with 14 individuals

#model
load(".\\models\\unscaled.dee.m.new.Rdata")#model name is dee.m.new


#### 1d) Body composition data -------
# Culldata (140 individuals)
df1 <- read_rds(".\\data\\culldata_for_model.Rds")

# Example phenotypes used in Trondrud et al. 2021
df2 <- read_rds(".\\data\\phenotypes_for_model.Rds")

# Merge data 
df3 <- rbind(df1,df2)


### 2. PREPARE DATA  --------------------------------


# 2a) Constants that can be changed (should be changed here, not inside the loop)

FoodE <- 0
lambda <- 0.489  # From Speakman & Westerterp (2013)
gamma <- 0.547 # From Speakman & Westerterp (2013)
fat.TAG <- 0.92 # Tricylglycerides(utilizable lipids) contribute 92% of fat mass 
m.o.protein <- 0.22 # Lean mass consists of 22% protein (ca)
m.o.water <- 0.73 # Lean mass consists of 73% water (ca)
m.o.fat <- 0.05 # Lean mass consists of 5% triacylglyrecides (lipids) (ca)
adi.protein <- 0.025 # 2.ca 5% protein in adipose tissue  
EnergyProtein <- 18 # MJ/kg energy content of proteins
EnergyFat <- 39 #MJ/kg energy content of fat/lipids 
dead.percent <- 40 # death occurs at 40% lean mass loss (treshold to break loop)
available <- 0.4 # 40% of total lean mass can be expended during fast


# 2b) Create a dataframe to predict activity and body temperature in loop
simyear <- actdat %>% ungroup() %>%
  # filter(year2 %in% c("2014", "2016", "2017")) %>%
  dplyr::select(jj, jj.c,  collartemp.mean, year2) %>% 
  group_by(jj) %>%  
  summarise_all(mean, na.rm=T) %>% #create mean temperature across 10 years of data
  mutate(id='R320', year2='2015') #randomindividual and year as 

# 2c) Create a list which is used to feed data into the loop
# Splits data frame into list with 1 data frame per individual
bigL <- split(df3, f=df3$id_text) 




### 3. RUN MODEL  --------------------------------

registerDoParallel(7) #remember to change depending on available cores
start_time <- Sys.time() # start time to evaluate how long the loop takes to run

res <- foreach(df=bigL, .errorhandling = "stop", .packages="dplyr") %dopar% {
  # df <-bigL[[1]] # test run (do not run loop but one line at a time)
  list1 <- data.frame()
  list2 <- data.frame()
  list.id <- list()
  # dfdf <- df2[df2$id_text=="Avg",]
  for (knit in 1:100){ # knit determines no. of iterations (1:100) = 100 iter
    
    newdat2 <- simyear #create newdata to predict fasting endurance
    newdat2$id_2 <- df$id_text[1] # set ID

#Create necessary columns in newdata:
    newdat2$boot <- knit       # iteration number
    newdat2$ffm2 <- NA         # lean mass 
    newdat2$bm <- NA           # body mass
    newdat2$Pfat <- NA         # proportion of fat in body
    newdat2$adipose <- NA      # mass of fat stores
    newdat2$fatchem <- NA      # chemichally derived body fat, used to 
                               # separate structural and disposable fat
    newdat2$Efat <- NA         # Energy from fat stores 
    newdat2$Eprot <- NA        # Energy from protein stores 
    newdat2$Efatuse <- NA      # Energy depleted from fat store per day 
    newdat2$Eprotuse <- NA     # Energy depleted from protein store per day 
    newdat2$Rfat <- NA         # R fat ratio from Speakman & Westerterp (2013) 
    newdat2$DEE <- NA          # Daily energy expenditure 
    newdat2$CDEE <- NA         # DEE covered by internal energy reserve
    newdat2$muscleorgans <- NA # Mass of muscle and organs 
                               # (lean mass minus skeleton and skin)
    newdat2$vitals <- NA       # Size of vital muscle and organs 
                               # (not available as energy reserve)
    newdat2$available.m.o <- NA# Size of muscle and organs available as energy
    newdat2$Etot <- NA         # Total internal energy reserve
    
    newdat2$actcol.day<- NA    # Activity for DEE prediction
    newdat2$act <- NA          # Activity for temp prediction (different names in models) 
    newdat2$sc.temp.mean<-NA   # Subcutaneous body temperature 
    newdat2$sc.temp.exp<-NA    # Exponential subcutaneous body temperature 
    newdat2$fat.percent <- NA  # Percent body fat of total body mass  
    newdat2$fat.struct <- NA   # Structural fat (not available as energy)
    newdat2$Efat.adi<- NA      # Energy from fat/lipids contributed by fat mass
    newdat2$Efat.mo<-NA        # Energy from fat/lipids contributed by muscle/lean mass 
    newdat2$Eprot.adi <- NA    # Energy from protein contributed by fat mass
    newdat2$Eprot.mo <- NA     # Energy from protein contributed by muscle/lean mass 
    newdat2$Eprot <- NA        # Total energy from protein 
    newdat2$Eprot.prop.adi <-NA# Relative proportion of energy from protein 
                               # contributed by fat mass
    newdat2$Eprot.prop.mo <-NA # Relative proportion of energy from protein 
                               # contributed by muscle/lean mass
    newdat2$Efat.prop.adi<-NA  # Relative proportion of energy from fat/lipids 
                               # contributed by fat mass
    newdat2$Efat.prop.mo <- NA # Relative proportion of energy from fat/lipids 
                               # contributed by muscle/lean mass 
    
    newdat2$muscle.organ.water <- NA # Water content of muscles and organs 
    
    # newdat2$actcol.day<- 0   # set to zero for resting DEE prediction
    # newdat2$act <- 0           
    
    newdat2$percent.loss <- NA # percent body mass loss since day 1
    newdat2$m.o.loss <- NA     # muscle and organ mass loss since day 1
    
# fixed values throughout loop 
    newdat2$skinhooves <- df$skinhooves[1] # mass of skin & hooves
    newdat2$bone <- df$bone[1]             # mass of skeleton
    newdat2$legl <- df$legl[1]             # length of hindleg (skeletal size)
    
# starting values
    newdat2$ffm2[1] <- df$ffm[1]           # initial lean mass 
    newdat2$adipose[1] <- df$adipose[1]    # initial fat mass for energy depletion
    newdat2$bm[1] <- df$bm[1]              # initial body mass
    newdat2$muscleorgans[1] <- df$muscleorgans[1] # initial mass of muscles and
                                                  # organs
    newdat2$available.m.o[1] <- available*newdat2$muscleorgans[1]# initial mass 
                                                  # of available (40%) muscles 
                                                  # and organs
    newdat2$vitals <- (1-available)*newdat2$muscleorgans[1] # vital organ and 
                              # muscle mass which cannot used as energy reserve
    newdat2$fat.struct <- df$fat.struct[1] # structural fat which cannot
                                           # be used as energy reserve
    
    
    on=1 # indicator to break loop, do not change
    t=1  # starting time 
    
# FASTING ENDURANCE LOOP (1 run = 1 iteration for 1 individual)    
    while(on==1 & t<nrow(newdat2)){ 
      
      # predicting activity levels
      # Resting DEE: activate lines 198+190 and deactivite lines 223-228 
      pred <-  predict(act.m2c, newdata=newdat2[t,],
      se.fit=T,type="response",
      exclude = c("s(id)","s(jj.c,id)")) # global prediction
      
      #add uncertainty around prediction using rnorm()
      newdat2$actcol.day[t] <- pred$fit+rnorm(1, 0, pred$se.fit) #remove rnorm for food prediction
      newdat2$act[t] <- newdat2$actcol.day[t] # add column for temp prediction
      
      # predicting subcut temp 
      pred2 <-  predict(temp.m2b, newdata=newdat2[t,],
                        se.fit=T,type="response",
                        exclude = c("s(id)","s(jj.c,id)"))
     
      #add uncertainty around prediction using rnorm()
      newdat2$sc.temp.mean[t]  <- pred2$fit+rnorm(1, 0, pred2$se.fit) #remove rnorm for food prediction
      newdat2$sc.temp.exp[t] <- exp(newdat2$sc.temp.mean[t]) #raise to exponential to fit DEE model
      
      # predicting daily energy expenditure 
      fit <- predict(dee.m.new, newdat2[t,], type="response",se.fit=T)#remove rnorm for food prediction
      #add uncertainty around prediction using rnorm()
      newdat2$DEE[t] <- fit$fit+rnorm(1, 0, fit$se.fit) 

      # get fat energy content
      newdat2$Efat.adi[t] <- newdat2$adipose[t]*fat.TAG*EnergyFat
      newdat2$Efat.mo[t] <- newdat2$available.m.o[t]*m.o.fat*EnergyFat #available mo is 40%
      newdat2$Efat[t] <- newdat2$Efat.adi[t]+newdat2$Efat.mo[t]
      newdat2$muscle.organ.water[t] <- newdat2$available.m.o[t]*m.o.water
      
      # propotion of energy from fat in each store
      newdat2$Efat.prop.adi[t] <- newdat2$Efat.adi[t]/newdat2$Efat[t] 
      newdat2$Efat.prop.mo[t] <- newdat2$Efat.mo[t]/newdat2$Efat[t] 
      newdat2$fat.percent[t] <- (newdat2$adipose[t]/newdat2$bm[t])*100
      
      # get protein energy content
      newdat2$Eprot.mo[t] <- newdat2$available.m.o[t]*m.o.protein*EnergyProtein
      newdat2$Eprot.adi[t] <- newdat2$adipose[t]*adi.protein*EnergyProtein
      newdat2$Eprot[t] <- newdat2$Eprot.adi[t]+newdat2$Eprot.mo[t]
      
      # proportion of energy from protein in each store
      newdat2$Eprot.prop.adi[t] <- newdat2$Eprot.adi[t]/newdat2$Eprot[t] 
      newdat2$Eprot.prop.mo[t] <- newdat2$Eprot.mo[t]/newdat2$Eprot[t] 
      
      # total energy reserve
      newdat2$Etot[t] <- newdat2$Efat[t]+newdat2$Eprot[t]
      
      # get Pfat ratio 
      newdat2$Pfat[t] <- newdat2$Efat[t]/newdat2$Etot[t]
      
      # make sure Pfat is not negative (when Pfat=0, loop is broken)
      newdat2$Pfat[t] <- ifelse(newdat2$Pfat[t]<0, 0, newdat2$Pfat[t])
     
      # get Rfat
      newdat2$Rfat[t] <- lambda+gamma*newdat2$Pfat[t]
      
      # Rfat cannot be lower than lambda     
      newdat2$Rfat[t] <- ifelse(newdat2$Rfat[t]<lambda, 0, newdat2$Rfat[t])
      
      # Get CDEE (energy from internal stores)
      # If FoodE=0, DEE = CDEE
      newdat2$CDEE[t] <- newdat2$DEE[t]-FoodE
      
      # Get body composition (chemically estimated fat and lean mass)
      # Fatchem = sum of structural and disposable fat
      newdat2$fatchem[t] <- newdat2$fat.struct[t]+newdat2$adipose[t]
      # Lean mass (ffm2) = body mass minus fatchem
      newdat2$ffm2[t] <- newdat2$bm[t]-newdat2$fatchem[t]
      
      # get energy used from fat
      newdat2$Efatuse[t] <-  newdat2$CDEE[t]*newdat2$Rfat[t]
      
      # Efatuse cannot be negative:
      if(newdat2$Efat[t]<=0 ) newdat2$Efatuse[t] <- 0
      
      # get energy used from protein
      newdat2$Eprotuse[t] <-  newdat2$CDEE[t]*(1-newdat2$Rfat[t])
      
# Get energy and body composition for next day (t+1)    

  # Energy reserves on next day = Energy on day t minus energy used on day t
      newdat2$Efat[t+1] <- (newdat2$Efat[t]-newdat2$Efatuse[t])
      newdat2$Eprot[t+1] <- (newdat2$Eprot[t]-newdat2$Eprotuse[t])
      
      
  # Proportion of energy contribution from each reserve
      newdat2$Efat.adi[t+1] <-  newdat2$Efat[t+1]*newdat2$Efat.prop.adi[t]
      newdat2$Efat.mo[t+1] <-  newdat2$Efat[t+1]*newdat2$Efat.prop.mo[t]
      
      newdat2$Eprot.adi[t+1] <-  newdat2$Eprot[t+1]*newdat2$Eprot.prop.adi[t]
      newdat2$Eprot.mo[t+1] <-  newdat2$Eprot[t+1]*newdat2$Eprot.prop.mo[t]

  # Calculate mass of body compartments based on energy depleted 
      # available muscle and organ mass
      newdat2$available.m.o[t+1] <- (((newdat2$Eprot.mo[t+1]/EnergyProtein)+(newdat2$Efat.mo[t+1]/EnergyFat))/((m.o.fat+m.o.protein)*100))*100
      # disposable fat mass
      newdat2$adipose[t+1] <- (((newdat2$Eprot.adi[t+1]/EnergyProtein)+(newdat2$Efat.adi[t+1]/EnergyFat))/((fat.TAG+adi.protein)*100))*100
      #body mass
      newdat2$bm[t+1] <- newdat2$adipose[t+1]+newdat2$bone[t+1]+newdat2$skinhooves[t+1]+newdat2$vitals[t+1]+newdat2$available.m.o[t+1]
      # total body fat 
      newdat2$fatchem[t+1] <- newdat2$adipose[t+1]+newdat2$fat.struct[t+1]
      # total lean mass
      newdat2$ffm2[t+1] <- newdat2$bm[t+1]-newdat2$fatchem[t+1]
      #percent loss of lean mass
      newdat2$percent.loss[t] <- ((newdat2$ffm2[1]-newdat2$ffm2[t])/newdat2$ffm2[1])*100
      #total muscle and organ mass
      newdat2$muscleorgans[t+1] <- newdat2$vitals[t+1]+newdat2$available.m.o[t+1]
      # percent loss of muscle and organ mass
      newdat2$m.o.loss[t] <- ((newdat2$muscleorgans[1]-newdat2$muscleorgans[t])/newdat2$muscleorgans[1])*100
      
      
      #Loop breaks when Energy is depleted of fat percent is less than 0.5
      if (newdat2$Etot[t]<0 | newdat2$fat.percent[t]<0.5)
        on=0
      
      t=t+1
      
    }
    #binds each iteration to a list
    list1 <- rbind(list1, as.data.frame(newdat2)) 
    
  }
  return(list1)
}


end_time <- Sys.time() # get time when model is finished
end_time - start_time  # check duration 

# The resulting data frame is a huge list with all iterations
fe.data <- bind_rows(list1) #bind all rows to get one big data frame

#optional save data
save(fe.data, file=".\\data\\fe.data.Rdata")

### 4. USE DATA  --------------------------------

# Get fasting endurance per ID per iteration

# Filter out missing values for body mass to keep data only until death occurs
# Filter out data when energy expenditure is unrealistically low for the species
# you work with, here we set a treshold at 3 MJ/day for Svalbard reindeer
# Create a summary data file per iteration (boot) with results of interest
death.data <-fe.data %>%  group_by(id_2, boot)  %>% 
  filter(!is.na(bm), CDEE>3) %>%
  summarise(bm.loss=max(bm, na.rm=T)-min(bm, na.rm=T),
            bm.end=min(bm, na.rm=T),
            bm.loss.percent=((max(bm, na.rm=T)-min(bm, na.rm=T))/max(bm, na.rm=T))*100,
            # mo.loss=max(muscleorgans, na.rm=T)-min(muscleorgans, na.rm=T),
            # mo.loss.percent=((max(muscleorgans, na.rm=T)-min(muscleorgans, na.rm=T))/max(muscleorgans, na.rm=T))*100,
            # adipose.loss=max(adipose, na.rm=T)-min(adipose, na.rm=T),
            # adipose.loss.percent=((max(adipose, na.rm=T)-min(adipose, na.rm=T))/max(adipose, na.rm=T))*100,
            day.death=last(jj), # DAY OF DEATH IS YOUR FASTING ENDURANCE RESULT
            adipose.start=first(adipose),
            mo.start=first(muscleorgans),
            bm.start=first(bm),
            ffm.start=first(ffm2),
            legl=first(legl),
            bodycondition=adipose.start/bm.start,
            Etot=first(Etot))

#Create a summary per individual, get mean and standard deviation 
death.summary<- death.data %>% group_by(id_2) %>% 
  summarise_if(is.numeric, c(mean=mean, sd=sd), na.rm=T)


# Plot nice things

examplePlot <-ggplot(death.summary,
                     aes(bm.start_mean, day.death_mean))+  
  geom_point(size=2, shape=1)+
  geom_pointrange(aes(ymin=day.death_mean-day.death_sd, 
                      ymax=day.death_mean+day.death_sd))+
    geom_smooth( method="lm", col="black")+
  # scale_x_continuous("Body mass (kg)",
                     # breaks=seq(50,90,10),
                     # labels=seq(50,90,10),
                     # limits=c(50,90))+
  # scale_y_continuous("Fasting endurance (days)",
  #                    breaks=seq(0,225,25),
  #                    labels=seq(0,225,25),
  #                    limits=c(25,175))+
  # stat_poly_eq(formula = y ~ x ,
  #              aes(label = paste(..eq.label.., adj.rr.label, sep = "~~~")),
  #              parse = TRUE,
  #              vjust=0.5) +
  theme_classic()+
  theme(panel.border = element_rect(fill=NA, colour="black"),
        legend.position = "none")
examplePlot
