!
!    Copyright 2014, 2015, 2017, 2018, 2020, 2021 Guy Munhoven
!
!    This file is part of µXML.
!
!    µXML is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Affero General Public License as
!    published by the Free Software Foundation, either version 3 of
!    the License, or (at your option) any later version.
!
!    µXML is distributed in the hope that it will be useful, but
!    WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!    See the GNU Affero General Public License for more details.
!
!    You should have received a copy of the Affero GNU General Public
!    License along with µXML.  If not, see <https://www.gnu.org/licenses/>.
!


#include <modmxm.h>
!===============================================================================
 MODULE MODMXM_TASKS
!===============================================================================

IMPLICIT NONE

INTEGER, PARAMETER, PRIVATE :: jp_stderr = (MXM_STDERR)


! Parameters
! ----------

! Tasks to be performed: identificators and descriptions

INTEGER,          PARAMETER :: p_nlenmax_tkname = 31



CHARACTER(LEN=*), PARAMETER, PRIVATE :: p_tkname_00 = 'None'
INTEGER, PARAMETER :: p_itask_none                =  0
CHARACTER(LEN=*), PARAMETER, PRIVATE :: p_tkname_01 = 'Proceed to non-SPC'
INTEGER, PARAMETER :: p_itask_proceed_to_nonSPC   =  1
CHARACTER(LEN=*), PARAMETER, PRIVATE :: p_tkname_02 = 'Get Tag'
INTEGER, PARAMETER :: p_itask_get_tag             =  2
CHARACTER(LEN=*), PARAMETER, PRIVATE :: p_tkname_03 = 'Skip to next tag'
INTEGER, PARAMETER :: p_itask_search_next_tag     =  3
CHARACTER(LEN=*), PARAMETER, PRIVATE :: p_tkname_04 = 'Identify tag type'
INTEGER, PARAMETER :: p_itask_identify_tag_type   =  4
CHARACTER(LEN=*), PARAMETER, PRIVATE :: p_tkname_05 = 'Process comment'
INTEGER, PARAMETER :: p_itask_process_comment     =  5
CHARACTER(LEN=*), PARAMETER, PRIVATE :: p_tkname_06 = 'Process CDATA'
INTEGER, PARAMETER :: p_itask_process_cdata       =  6
CHARACTER(LEN=*), PARAMETER, PRIVATE :: p_tkname_07 = 'Process !DOCTYPE'
INTEGER, PARAMETER :: p_itask_process_doctype     =  7
CHARACTER(LEN=*), PARAMETER, PRIVATE :: p_tkname_08 = 'Process description (<? ... ?>)'
INTEGER, PARAMETER :: p_itask_process_desc_tag =  8
CHARACTER(LEN=*), PARAMETER, PRIVATE :: p_tkname_09 = 'Get name'
INTEGER, PARAMETER :: p_itask_get_name            =  9
CHARACTER(LEN=*), PARAMETER, PRIVATE :: p_tkname_10 = 'Get quoted string'
INTEGER, PARAMETER :: p_itask_get_quoted_string   = 10
CHARACTER(LEN=*), PARAMETER, PRIVATE :: p_tkname_11 = 'Process single-quoted string'
INTEGER, PARAMETER :: p_itask_process_sqstr       = 11
CHARACTER(LEN=*), PARAMETER, PRIVATE :: p_tkname_12 = 'Process double-quoted string'
INTEGER, PARAMETER :: p_itask_process_dqstr       = 12
CHARACTER(LEN=*), PARAMETER, PRIVATE :: p_tkname_13 = 'Process element start-tag'
INTEGER, PARAMETER :: p_itask_process_elt_starttag= 13
CHARACTER(LEN=*), PARAMETER, PRIVATE :: p_tkname_14 = 'Process attribute'
INTEGER, PARAMETER :: p_itask_process_attribute   = 14
CHARACTER(LEN=*), PARAMETER, PRIVATE :: p_tkname_15 = 'Process element end-tag'
INTEGER, PARAMETER :: p_itask_process_elt_endtag= 15

INTEGER, PARAMETER, PRIVATE :: p_itask_last                = 15


! Type definitions

TYPE integer_lifo
  INTEGER :: i
  TYPE(integer_lifo), POINTER :: prev
  TYPE(integer_lifo), POINTER :: next
END TYPE


CONTAINS


!-------------------------------------------------------------------------------
FUNCTION TASK_NAME(i_task) RESULT(str_task)
!-------------------------------------------------------------------------------

IMPLICIT NONE

INTEGER, INTENT(IN)             :: i_task
CHARACTER(LEN=p_nlenmax_tkname) :: str_task

SELECT CASE(i_task)
CASE( 0)
  str_task = p_tkname_00
CASE( 1)
  str_task = p_tkname_01
CASE( 2)
  str_task = p_tkname_02
CASE( 3)
  str_task = p_tkname_03
CASE( 4)
  str_task = p_tkname_04
CASE( 5)
  str_task = p_tkname_05
CASE( 6)
  str_task = p_tkname_06
CASE( 7)
  str_task = p_tkname_07
CASE( 8)
  str_task = p_tkname_08
CASE( 9)
  str_task = p_tkname_09
CASE(10)
  str_task = p_tkname_10
CASE(11)
  str_task = p_tkname_11
CASE(12)
  str_task = p_tkname_12
CASE(13)
  str_task = p_tkname_13
CASE(14)
  str_task = p_tkname_14
CASE(15)
  str_task = p_tkname_15
CASE DEFAULT
  str_task = 'Unknown task'
END SELECT

RETURN

!-------------------------------------------------------------------------------
END FUNCTION TASK_NAME
!-------------------------------------------------------------------------------



!-------------------------------------------------------------------------------
SUBROUTINE PULL_TASK(ili_tasks, i_task, i_err)
!-------------------------------------------------------------------------------

IMPLICIT NONE


TYPE(integer_lifo), POINTER :: ili_tasks
INTEGER :: i_task
INTEGER :: i_err

INTEGER :: i_task_pulled


IF (.NOT. ASSOCIATED(ili_tasks)) THEN
  WRITE(jp_stderr,'("[PULL_TASK] Error: task stack not ASSOCIATED -- returning")')
  i_err = 1
ENDIF

i_task_pulled = ili_tasks%i


IF (ASSOCIATED(ili_tasks%prev)) THEN
                  ! There is a more ancient task:
                  ! clear the present task, and go
                  ! back one level. Keep the current
                  ! one for later usage.
  ili_tasks%i = p_itask_none
  ili_tasks => ili_tasks%prev
ELSE
  ili_tasks%i = p_itask_none
ENDIF

IF (i_task_pulled /= i_task) THEN
  WRITE(jp_stderr, '("[PULL_TASK] Error: task stack inconsistency -- returning")')
  WRITE(jp_stderr, '("  expected = ", I0, " (",A,")")') i_task, TRIM(TASK_NAME(i_task))
  WRITE(jp_stderr, '("  pulled = ", I0, " (",A,")")') i_task_pulled, TRIM(TASK_NAME(i_task_pulled))
  i_err = 2
ELSE
  i_err = 0
ENDIF


RETURN



!-------------------------------------------------------------------------------
END SUBROUTINE PULL_TASK
!-------------------------------------------------------------------------------


!-------------------------------------------------------------------------------
SUBROUTINE CURR_TASK(ili_tasks, i_task, i_err)
!-------------------------------------------------------------------------------

IMPLICIT NONE


TYPE(integer_lifo), POINTER :: ili_tasks
INTEGER, INTENT(OUT) :: i_task
INTEGER, INTENT(OUT) :: i_err


IF (.NOT. ASSOCIATED(ili_tasks)) THEN
  WRITE(jp_stderr,'("[CURR_TASK] Error: task stack not ASSOCIATED -- returning")')
  i_err = 1
  i_task = 0
ELSE
  i_task = ili_tasks%i
  i_err = 0
ENDIF

RETURN



!-------------------------------------------------------------------------------
END SUBROUTINE CURR_TASK
!-------------------------------------------------------------------------------



!-------------------------------------------------------------------------------
SUBROUTINE PUSH_TASK(ili_tasks, i_task, i_err)
!-------------------------------------------------------------------------------


IMPLICIT NONE

TYPE(integer_lifo), POINTER :: ili_tasks
INTEGER, INTENT(IN)         :: i_task
INTEGER, INTENT(OUT)        :: i_err

                  ! Check if requested i_task is valid:
                  ! if not RETURN with i_err = 1
IF ((i_task < 0) .OR. (i_task > p_itask_last)) THEN
  WRITE(jp_stderr,'("[PUSH_TASK] Error: unknown task ", I0, " -- returning")') i_task
  i_err = 1
  RETURN
ENDIF

IF (.NOT. ASSOCIATED(ili_tasks)) THEN
                  ! If ili_tasks is not yet associated,
                  ! create a new one and use the new item
                  ! for the task
  ALLOCATE(ili_tasks)
  NULLIFY(ili_tasks%prev)
  NULLIFY(ili_tasks%next)
  ili_tasks%i = i_task
  
ELSE
  IF (.NOT. ASSOCIATED(ili_tasks%prev)) THEN
                  ! If ili_tasks%prev is not associated, we are already
                  ! at the root level. If the currently registered task
                  ! is p_itask_none, override that one, and RETURN
                  ! with i_err = 0 (no error)
    IF (ili_tasks%i == p_itask_none) THEN
      ili_tasks%i = i_task
      i_err = 0
      RETURN
    ENDIF

  ENDIF

                  ! In all other cases, go up one layer (create if necessary)
                  
  IF (ASSOCIATED(ili_tasks%next)) THEN
    ili_tasks => ili_tasks%next
  ELSE
    ALLOCATE(ili_tasks%next)
    NULLIFY(ili_tasks%next%next)
    ili_tasks%next%prev => ili_tasks
    ili_tasks => ili_tasks%next
  ENDIF

  ili_tasks%i = i_task
  i_err = 0

  RETURN

ENDIF
!-------------------------------------------------------------------------------
END SUBROUTINE PUSH_TASK
!-------------------------------------------------------------------------------



!-------------------------------------------------------------------------------
SUBROUTINE DEALLOCATE_TASKSTACK(ili_tasks)
!-------------------------------------------------------------------------------

IMPLICIT NONE


! Argument list variables
! -----------------------

TYPE(integer_lifo), POINTER :: ili_tasks


! Local variables
! ---------------

TYPE(integer_lifo), POINTER :: ili_work


! End of declarations
! -------------------


IF(.NOT. ASSOCIATED(ili_tasks)) RETURN

! First work up the tree from the root
ili_work => ili_tasks

DO WHILE(ASSOCIATED(ili_work%next))
  ili_work => ili_work%next
ENDDO

! Now work back and deallocate branch after branch
DO WHILE(ASSOCIATED(ili_work%prev))
  ili_work => ili_work%prev
  DEALLOCATE(ili_work%next)
ENDDO

! Finally deallocate the root as well
DEALLOCATE(ili_tasks)

NULLIFY(ili_work)

RETURN

!-------------------------------------------------------------------------------
END SUBROUTINE DEALLOCATE_TASKSTACK
!-------------------------------------------------------------------------------



!===============================================================================
END MODULE MODMXM_TASKS
!===============================================================================
