!
!    Copyright 2014, 2015, 2017, 2018, 2020, 2021 Guy Munhoven
!
!    This file is part of µXML.
!
!    µXML is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Affero General Public License as
!    published by the Free Software Foundation, either version 3 of
!    the License, or (at your option) any later version.
!
!    µXML is distributed in the hope that it will be useful, but
!    WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!    See the GNU Affero General Public License for more details.
!
!    You should have received a copy of the Affero GNU General Public
!    License along with µXML.  If not, see <https://www.gnu.org/licenses/>.
!


#include <modmxm.h>
!=======================================================================
 MODULE MODMXM_STKRC
!=======================================================================

USE MODMXM_GENERAL

IMPLICIT NONE

INTEGER, PARAMETER, PRIVATE :: jp_stderr = (MXM_STDERR)
#ifdef DEBUG
INTEGER, PARAMETER, PRIVATE :: jp_stddbg = (MXM_STDDBG)
#endif


TYPE stack_recchunks
  INTEGER :: n_chars
  CHARACTER(LEN=p_maxlen_chunk)   :: str_chunk
  LOGICAL :: l_eor
  TYPE(stack_recchunks), POINTER  :: prev
  TYPE(stack_recchunks), POINTER  :: next
END TYPE


                  ! Encapsulate POINTER to TYPE(stack_recchunks)
                  ! in order to use arrays of pointers
TYPE stkrc_ptr
  TYPE(stack_recchunks), POINTER :: ptr
END TYPE



CONTAINS

!-----------------------------------------------------------------------
 SUBROUTINE STKRC_createRoot(stkrc_any, nlay_stkrc, nlen_stkrc)
!-----------------------------------------------------------------------

IMPLICIT NONE

TYPE(stack_recchunks), POINTER :: stkrc_any
INTEGER, INTENT(OUT), OPTIONAL :: nlay_stkrc
INTEGER, INTENT(OUT), OPTIONAL :: nlen_stkrc

IF (ASSOCIATED(stkrc_any)) THEN
  WRITE(jp_stderr,'("[STKRC_createRoot] Error: ", A)') &
    'stkrc_any already associated -- aborting'
  CALL ABORT()
ENDIF

ALLOCATE(stkrc_any)

stkrc_any%n_chars = 0
stkrc_any%str_chunk = ''
stkrc_any%l_eor = .TRUE.
NULLIFY(stkrc_any%prev)
NULLIFY(stkrc_any%next)

IF (PRESENT(nlay_stkrc)) nlay_stkrc = 1
IF (PRESENT(nlen_stkrc)) nlen_stkrc = 0

RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE STKRC_createRoot
!-----------------------------------------------------------------------




!-----------------------------------------------------------------------
 SUBROUTINE STKRC_createNext(stkrc_any, nlay_stkrc, nlen_stkrc)
!-----------------------------------------------------------------------

IMPLICIT NONE

TYPE(stack_recchunks), POINTER   :: stkrc_any
INTEGER, INTENT(INOUT), OPTIONAL :: nlay_stkrc 
INTEGER, INTENT(INOUT), OPTIONAL :: nlen_stkrc

TYPE(stack_recchunks), SAVE, POINTER :: stkrc_child

IF (.NOT. ASSOCIATED(stkrc_any)) THEN
  WRITE(jp_stderr,'("[STKRC_createNext] Error: ", A)') &
    'stkrc_any not associated, cannot create next -- aborting'
  CALL ABORT()
ELSEIF (ASSOCIATED(stkrc_any%next)) THEN
  WRITE(jp_stderr,'("[STKRC_createNext] Error: ", A)') &
    'stkrc_any already has next, cannot create new one -- aborting'
  CALL ABORT()
ENDIF

NULLIFY(stkrc_child)

ALLOCATE(stkrc_child)

stkrc_child%n_chars = 0
stkrc_child%str_chunk = ''
stkrc_child%l_eor = .FALSE.
stkrc_child%prev => stkrc_any
NULLIFY(stkrc_child%next)

stkrc_any%next => stkrc_child
IF (PRESENT(nlay_stkrc)) nlay_stkrc = nlay_stkrc + 1
IF (PRESENT(nlen_stkrc)) nlen_stkrc = nlen_stkrc + stkrc_any%n_chars

RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE STKRC_createNext
!-----------------------------------------------------------------------




!-----------------------------------------------------------------------
 SUBROUTINE STKRC_deallocateStkrc(stkrc_any, nlay_discarded, nlen_discarded)
!-----------------------------------------------------------------------

IMPLICIT NONE

! Argument list variables
! -----------------------

TYPE(stack_recchunks), POINTER :: stkrc_any
INTEGER, INTENT(OUT), OPTIONAL :: nlay_discarded
INTEGER, INTENT(OUT), OPTIONAL :: nlen_discarded


! Local variables
! ---------------

TYPE(stack_recchunks), POINTER :: stkrc_work
INTEGER :: i_level


! If "stkrc_any" is void (not associated),
! there is nothing to do: RETURN
IF(.NOT. ASSOCIATED(stkrc_any)) RETURN


stkrc_work => stkrc_any
i_level = 0

! Now work forwards (down-node) to the end of the tree
DO WHILE(ASSOCIATED(stkrc_work%next))
  stkrc_work => stkrc_work%next
  i_level = i_level + 1
ENDDO

IF (PRESENT(nlay_discarded)) nlay_discarded = i_level + 1

IF (PRESENT(nlen_discarded)) nlen_discarded = stkrc_work%n_chars
! Now work back and deallocate branch after branch
DO WHILE (i_level > 0)
  stkrc_work => stkrc_work%prev
  i_level = i_level - 1
  DEALLOCATE(stkrc_work%next)
  IF (PRESENT(nlen_discarded)) nlen_discarded = nlen_discarded + stkrc_work%n_chars
ENDDO

! Check if stkrc_any is linked into a tree: if so, unlink it
stkrc_work => stkrc_any

IF (ASSOCIATED(stkrc_any%prev)) NULLIFY(stkrc_any%prev%next)

! Finally deallocate and nullify the root as well
DEALLOCATE(stkrc_work)
NULLIFY(stkrc_any)

NULLIFY(stkrc_work)

RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE STKRC_deallocateStkrc
!-----------------------------------------------------------------------




!-----------------------------------------------------------------------
 SUBROUTINE STKRC_compactStkrc(stkrc_any)
!-----------------------------------------------------------------------

IMPLICIT NONE

! Argument list variables
! -----------------------

TYPE(stack_recchunks), POINTER :: stkrc_any


! Local variables
! ---------------

TYPE(stack_recchunks), POINTER :: stkrc_read
TYPE(stack_recchunks), POINTER :: stkrc_write
TYPE(stack_recchunks), POINTER :: stkrc_discard

INTEGER :: nread_chars, nread_tb
INTEGER :: nwrite_free
INTEGER :: i_e
INTEGER :: n_offset

CHARACTER(LEN=p_maxlen_chunk)  :: str_chunk


IF (.NOT. ASSOCIATED(stkrc_any)) RETURN

NULLIFY(stkrc_discard)


n_offset = 0
i_e = 0
nwrite_free = p_maxlen_chunk

stkrc_read => stkrc_any
stkrc_write => stkrc_read


DO

  nread_chars = stkrc_read%n_chars
  nread_tb = p_maxlen_chunk - nread_chars

                  ! If i_e points to the end of a fully filled chunk
                  ! then, and if we have just read in more characters
                  ! to write back, proceed to the next write-chunk
                  ! and make i_e point to zero; else leave all as it is.
  IF ((i_e == p_maxlen_chunk) .AND. (nread_chars > 0)) THEN
    stkrc_write => stkrc_write%next
    i_e = 0
    nwrite_free = p_maxlen_chunk
  ENDIF


  IF (n_offset == 0) THEN
                  ! No r/w operations necessary, only index updates
    i_e = i_e + nread_chars
    n_offset = n_offset + nread_tb

                  ! If the chunk was not completely filled,
                  ! flag the rest to the free for the next
                  ! passage.
    IF (nread_tb /= 0) nwrite_free = nread_tb

  ELSE
                  ! The reading and writing operations are not
                  ! synchroneous, but there is an offset

    IF (nwrite_free < nread_chars) THEN
                  ! There are not enough empty character places
                  ! left on the current write-chunk

                  ! Copy the read-chunk content to "str_chunk"
                  ! (nread_chars /= 0 in this case), ...
      str_chunk(1:nread_chars) = stkrc_read%str_chunk(1:nread_chars)

                  ! ... and transfer as much as possible to the
                  ! end of the write-chunk
      stkrc_write%str_chunk(i_e+1:p_maxlen_chunk) = str_chunk(1:nwrite_free)

                  ! Update the auxiliary information
      stkrc_write%n_chars = p_maxlen_chunk
      stkrc_write%l_eor = .FALSE.

                  ! Switch to the next chunk for writing
      stkrc_write => stkrc_write%next
                  ! Calculate the number of remaining characters to write
      i_e = nread_chars - nwrite_free

                  ! Transfer the rest of "str_chunk" to the new write-chunk.
      stkrc_write%str_chunk(1:i_e) = str_chunk(nwrite_free+1:nread_chars)

                  ! Update the offsetting and the free-character information
      n_offset = n_offset + nread_tb
      nwrite_free = p_maxlen_chunk - i_e

    ELSE
                  ! The current write-chunk has enough space left
                  ! for the number of characters read.
      IF (nread_chars > 0) THEN
        stkrc_write%str_chunk(i_e+1:i_e+nread_chars) &
          = stkrc_read%str_chunk( 1:    nread_chars)

                  ! Update the indices and counters
        i_e = i_e + nread_chars
        nwrite_free = p_maxlen_chunk - i_e
      ENDIF

      n_offset = n_offset + nread_tb

    ENDIF

  ENDIF

                  ! Now regularise the offsetting: if the read and write
                  ! operations are offset by more than the length of a
                  ! chunk, then discard intermediate unnecessary
                  ! chunks from the stack, one for each chunk-length
                  ! of offsetting.
  DO WHILE (n_offset >= p_maxlen_chunk)

                ! We are discarding the next chunk
    stkrc_discard => stkrc_write%next

    IF (ASSOCIATED(stkrc_discard%next)) THEN
                ! If there is one more afterwards, we need to
                ! update its %prev pointer, and make
                ! the %next pointer of the current writechunk to it
      stkrc_write%next => stkrc_discard%next
      stkrc_discard%next%prev => stkrc_write

    ELSE
                ! No chunk after the one that we want to discard.
                ! The current write-chunk will be the last one
                ! and its %prev pointer needs to be nullified.
      NULLIFY(stkrc_discard%next)

    ENDIF

                ! Now that the write-chunks are correctly linked,
                ! we can deallocate and discard the next chunk
    DEALLOCATE(stkrc_discard)
    NULLIFY(stkrc_discard)

                ! The offset is now reduced by the length of one chunk.
    n_offset = n_offset - p_maxlen_chunk

  ENDDO

                  ! If the most recently read chunk is terminated by EOR,
                  ! transfer this to the most recently written chunk:
                  ! The record is now complete.
  IF (stkrc_read%l_eor) stkrc_write%l_eor = .TRUE.

                  ! Provisionally register the number of characters
                  ! written so far (may be updated later on, if
                  ! more characters get written to this same chunk
  stkrc_write%n_chars = i_e


  IF (ASSOCIATED(stkrc_read%next)) THEN
                  ! There are still more chunks afterwards:

                  ! Go for the next one.
    stkrc_read => stkrc_read%next

    IF (stkrc_write%l_eor) THEN
                  ! If we have just completed a record, then
                  ! we switch to the next chunk, which is also
                  ! the one we are going to read in the next passage.
      stkrc_write => stkrc_read

                  ! If we have just completed a record, then
                  ! re-initialise the counters to start the
                  ! next record. 
      i_e = 0
      n_offset = 0
      nwrite_free = p_maxlen_chunk

    ENDIF

    CYCLE

  ELSE
                ! No more chunks afterwards
    EXIT

  ENDIF

  
ENDDO


RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE STKRC_compactStkrc
!-----------------------------------------------------------------------




!-----------------------------------------------------------------------
 SUBROUTINE STKRC_copyStkrcToStr(stkrc_source, str_dest, nlen_returned,&
                                 l_pruneateor, l_spc4eor)
!-----------------------------------------------------------------------

! Copies <stkrc_source> content into <str_dest>, either completely
! (DEFAULT) or up to the first EOR (if the optional <l_pruneateor> is
! set to .TRUE.). EORs are ignored (DEFAULT) or replaced by SPC (if the
! optional <l_spc4eor> is set to .TRUE.). If necessary, <str_dest> is
! filled with SPC to the right.
! At most LEN(str_dest) characters are transcribed.
! If present, the optional <nlen_returned> is set to the number of
! characters transcribed, or to -1, in case <str_dest> has not enough
! space to hold all the characters to be transcribed.


IMPLICIT NONE


! Argument list variables
! -----------------------

TYPE(stack_recchunks), POINTER :: stkrc_source
CHARACTER(LEN=*), INTENT(OUT)  :: str_dest

INTEGER, INTENT(OUT), OPTIONAL :: nlen_returned
LOGICAL, INTENT(IN),  OPTIONAL :: l_pruneateor
LOGICAL, INTENT(IN),  OPTIONAL :: l_spc4eor


! Local variables
! ---------------

INTEGER :: nlen_dest
INTEGER :: nlen_chunk, nlen_left, nlen_copy, nlen_excess

TYPE(stack_recchunks), POINTER :: stkrc_work
INTEGER :: i_e, i_s

LOGICAL :: lloc_pruneateor, lloc_spc4eor
CHARACTER(LEN=1) :: c_char4eor


IF (.NOT. ASSOCIATED(stkrc_source)) RETURN


IF (PRESENT(l_pruneateor)) THEN
  lloc_pruneateor = l_pruneateor
ELSE
  lloc_pruneateor = .FALSE.
ENDIF


IF (PRESENT(l_spc4eor)) THEN
  lloc_spc4eor = l_spc4eor
ELSE
  lloc_spc4eor = .FALSE.
ENDIF


IF (lloc_spc4eor) THEN
  c_char4eor = ' '      ! SPC
ELSE
  c_char4eor = CHAR(10) ! LF
ENDIF


nlen_dest = LEN(str_dest)
nlen_left = nlen_dest

i_e = 0
i_s = 1

stkrc_work => stkrc_source


DO

  nlen_chunk = stkrc_work%n_chars

  nlen_copy = MIN(nlen_left, nlen_chunk)
  nlen_excess = nlen_chunk - nlen_copy


  IF (nlen_copy > 0) THEN

    i_e = i_e + nlen_copy
    str_dest(i_s:i_e) = stkrc_work%str_chunk(1:nlen_copy)

    i_s = i_e + 1
    nlen_left = nlen_left - nlen_copy

  ENDIF

                  ! If "str_dest" is completely filled now, we are done.
  IF (nlen_left == 0) EXIT

                  ! There are some characters left to fill in "str_dest".
                  ! We have therefore filled a complete chunk into "str_dest".

  IF (stkrc_work%l_eor) THEN
                  ! We have copied a chunk terminating by EOR into "str_dest".
    IF (lloc_pruneateor) THEN

      EXIT        !  - if it is requested to prune at EOR, we are done

    ELSE          !  - if we do not prune at EOR,
                  !    then register the EOR as the required char
                  !    (there remains at least one free char in
                  !    "str_dest" if this code part is reached)
      str_dest(i_s:i_s) = c_char4eor
      i_e = i_e + 1
      i_s = i_s + 1
      nlen_left = nlen_left - 1

    ENDIF

  ENDIF

  IF (.NOT. ASSOCIATED(stkrc_work%next)) THEN
                  ! If there is no chunk following, we are done
    EXIT

  ELSE
                  ! else, proceed to the next chunk
    stkrc_work => stkrc_work%next

  ENDIF

ENDDO

IF (nlen_left > 0) THEN
  str_dest(nlen_dest-nlen_left+1:nlen_dest) = ' '
ENDIF
IF (PRESENT(nlen_returned)) THEN
  nlen_returned = nlen_dest - nlen_left
  IF (nlen_excess > 0) nlen_returned = -1
ENDIF

RETURN


!-----------------------------------------------------------------------
 END SUBROUTINE STKRC_copyStkrcToStr
!-----------------------------------------------------------------------




!-----------------------------------------------------------------------
 SUBROUTINE STKRC_writeStkrc(stkrc_source, i_unit,                     &
                            l_pruneateor, l_spc4eor, l_advance,        &
                            nlen_returned)
!-----------------------------------------------------------------------

IMPLICIT NONE


! Argument list variables
! -----------------------

TYPE(stack_recchunks), POINTER :: stkrc_source
INTEGER, INTENT(IN),  OPTIONAL :: i_unit

LOGICAL, INTENT(IN),  OPTIONAL :: l_pruneateor
LOGICAL, INTENT(IN),  OPTIONAL :: l_spc4eor
LOGICAL, INTENT(IN),  OPTIONAL :: l_advance
INTEGER, INTENT(OUT), OPTIONAL :: nlen_returned


! Local variables
! ---------------

INTEGER :: nlen_dest, nlen_chunk, nlen_printed
INTEGER :: i_e, i_s

TYPE(stack_recchunks), POINTER :: stkrc_work

LOGICAL :: lloc_star, lloc_pruneateor, lloc_spc4eor, lloc_advance
CHARACTER(LEN=1) :: c_char4eor


! End of declarations
! -------------------


IF (.NOT.ASSOCIATED(stkrc_source)) RETURN


IF (PRESENT(i_unit)) THEN
  lloc_star = .FALSE.
ELSE
  lloc_star = .TRUE.
ENDIF


IF (PRESENT(l_pruneateor)) THEN
  lloc_pruneateor = l_pruneateor
ELSE
  lloc_pruneateor = .FALSE.
ENDIF


IF (PRESENT(l_spc4eor)) THEN
  lloc_spc4eor = l_spc4eor
ELSE
  lloc_spc4eor = .FALSE.
ENDIF


IF (lloc_spc4eor) THEN
  c_char4eor = ' '      ! SPC
ELSE
  c_char4eor = CHAR(10) ! LF
ENDIF


IF (PRESENT(l_advance)) THEN
  lloc_advance = l_advance
ELSE
  lloc_advance = .TRUE.
ENDIF


nlen_printed = 0

stkrc_work => stkrc_source


IF (lloc_star) THEN

  DO

    nlen_chunk = stkrc_work%n_chars

    WRITE(*,'(A)', ADVANCE='NO') stkrc_work%str_chunk(1:nlen_chunk)
    nlen_printed = nlen_printed + nlen_chunk

    IF (.NOT. ASSOCIATED(stkrc_work%next)) THEN

      IF (lloc_advance) WRITE(*,'()')
      EXIT

    ELSE

      IF (stkrc_work%l_eor) THEN

        IF (lloc_pruneateor) THEN
          IF (lloc_advance) WRITE(*,'()')
          EXIT
        ELSE
          IF (lloc_spc4eor) THEN
            WRITE(*,'(" ")', ADVANCE='NO')
            nlen_printed = nlen_printed + 1
          ELSE
            WRITE(*,'()')
          ENDIF
        ENDIF

      ENDIF

      stkrc_work => stkrc_work%next

    ENDIF

  ENDDO

  IF (PRESENT(nlen_returned)) nlen_returned = nlen_printed
  RETURN

ELSE

  DO

    nlen_chunk = stkrc_work%n_chars

    WRITE(i_unit,'(A)', ADVANCE='NO') stkrc_work%str_chunk(1:nlen_chunk)
    nlen_printed = nlen_printed + nlen_chunk

    IF (.NOT. ASSOCIATED(stkrc_work%next)) THEN

      IF (lloc_advance) WRITE(i_unit,'()')
      EXIT

    ELSE

      IF (stkrc_work%l_eor) THEN

        IF (lloc_pruneateor) THEN
          IF (lloc_advance) WRITE(i_unit,'()')
          EXIT
        ELSE
          IF (lloc_spc4eor) THEN
            WRITE(i_unit,'(" ")', ADVANCE='NO')
            nlen_printed = nlen_printed + 1
          ELSE
            WRITE(i_unit,'()')
          ENDIF
        ENDIF

      ENDIF

      stkrc_work => stkrc_work%next

    ENDIF

  ENDDO

  IF (PRESENT(nlen_returned)) nlen_returned = nlen_printed
  RETURN

ENDIF


RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE STKRC_writeStkrc
!-----------------------------------------------------------------------




!-----------------------------------------------------------------------
 SUBROUTINE STKRC_dumpStkrc(stkrc_source, i_unit)
!-----------------------------------------------------------------------

IMPLICIT NONE


! Argument list variables
! -----------------------

TYPE(stack_recchunks), POINTER :: stkrc_source
INTEGER, INTENT(IN),  OPTIONAL :: i_unit


! Local variables
! ---------------

INTEGER :: i_chunk, nlen_chunk
INTEGER :: i_e, i_s

TYPE(stack_recchunks), POINTER :: stkrc_work

LOGICAL :: lloc_star


! End of declarations
! -------------------


IF (PRESENT(i_unit)) THEN
  lloc_star = .FALSE.
ELSE
  lloc_star = .TRUE.
ENDIF



stkrc_work => stkrc_source
i_chunk = 1


IF (lloc_star) THEN

  IF (.NOT.ASSOCIATED(stkrc_source)) THEN
    WRITE(*,'("[STKRC_dumpStkrc]: Empty stack")')
    RETURN
  ENDIF



  DO

    nlen_chunk = stkrc_work%n_chars

    WRITE(*,'("[STKRC_dumpStkrc]: chunk ", I0)') i_chunk
    !WRITE(*,'(" - @%str_chunk = """, A, """")') stkrc_work%str_chunk
    WRITE(*,'(" - @%n_chars   = ", I0)') nlen_chunk
    WRITE(*,'(" - @%str_chunk = >", A, "<")') stkrc_work%str_chunk(1:nlen_chunk)
    WRITE(*,'(" - @l_eor             = ", L1)') stkrc_work%l_eor
    WRITE(*,'(" - ASSOCIATED(@%prev) = ", L1)') ASSOCIATED(stkrc_work%prev)
    WRITE(*,'(" - ASSOCIATED(@%next) = ", L1)') ASSOCIATED(stkrc_work%next)
    
    IF (.NOT. ASSOCIATED(stkrc_work%next)) EXIT

    stkrc_work => stkrc_work%next
    i_chunk = i_chunk+1

  ENDDO

ELSE

  IF (.NOT.ASSOCIATED(stkrc_source)) THEN
    WRITE(i_unit,'("[STKRC_dumpStkrc]: Empty stack")')
    RETURN
  ENDIF

  DO

    nlen_chunk = stkrc_work%n_chars

    WRITE(i_unit,'("[STKRC_dumpStkrc]: chunk ", I0)') i_chunk
    !WRITE(i_unit,'(" - @%str_chunk = """, A, """")') stkrc_work%str_chunk
    WRITE(i_unit,'(" - @%n_chars   = ", I0)') nlen_chunk
    WRITE(i_unit,'(" - @%str_chunk = >", A, "<")') stkrc_work%str_chunk(1:nlen_chunk)
    WRITE(i_unit,'(" - @l_eor             = ", L1)') stkrc_work%l_eor
    WRITE(i_unit,'(" - ASSOCIATED(@%prev) = ", L1)') ASSOCIATED(stkrc_work%prev)
    WRITE(i_unit,'(" - ASSOCIATED(@%next) = ", L1)') ASSOCIATED(stkrc_work%next)
    
    IF (.NOT. ASSOCIATED(stkrc_work%next)) EXIT

    stkrc_work => stkrc_work%next
    i_chunk = i_chunk+1

  ENDDO

ENDIF


RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE STKRC_dumpStkrc
!-----------------------------------------------------------------------




!-----------------------------------------------------------------------
 SUBROUTINE STKRC_lentrim(stkrc_source, nlentrim_stkrc, nlen_stkrc)
!-----------------------------------------------------------------------

IMPLICIT NONE


! Argument list variables
! -----------------------

TYPE(stack_recchunks), POINTER :: stkrc_source

INTEGER, INTENT(OUT)           :: nlentrim_stkrc
INTEGER, INTENT(OUT), OPTIONAL :: nlen_stkrc


! Local variables
! ---------------

INTEGER :: nlentrim_chunk
INTEGER :: nlen_chunk
INTEGER :: nlenloc_stkrc

TYPE(stack_recchunks), POINTER :: stkrc_work



IF (.NOT.ASSOCIATED(stkrc_source)) RETURN


stkrc_work => stkrc_source

nlenloc_stkrc = 0
nlentrim_stkrc = 0


DO

  nlen_chunk = stkrc_work%n_chars
  nlentrim_chunk = LEN_TRIM(stkrc_work%str_chunk)

  IF (nlentrim_chunk /= 0) THEN
    nlentrim_stkrc = nlenloc_stkrc + nlentrim_chunk
  ENDIF

  nlenloc_stkrc = nlenloc_stkrc + nlen_chunk

  IF (ASSOCIATED(stkrc_work%next)) THEN
    stkrc_work => stkrc_work%next
  ELSE
    EXIT
  ENDIF

ENDDO


IF (PRESENT(nlen_stkrc)) nlen_stkrc = nlenloc_stkrc

RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE STKRC_lentrim
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
 LOGICAL FUNCTION STKRC_STKRC_EQ_STKRC(stkrc_1, stkrc_2)
!-----------------------------------------------------------------------

IMPLICIT NONE

! Argument list variables
! -----------------------

TYPE(stack_recchunks), POINTER :: stkrc_1, stkrc_2


! Local variables
! ---------------

TYPE(stack_recchunks), POINTER :: stkrc_work1, stkrc_work2

INTEGER :: n_chars1, i_s1, i_e1
INTEGER :: n_chars2, i_s2, i_e2
INTEGER :: n_charstocmp

CHARACTER(LEN=p_maxlen_chunk)  :: str_chunk


IF ((.NOT. ASSOCIATED(stkrc_1)).OR.(.NOT. ASSOCIATED(stkrc_2))) THEN
  STKRC_STKRC_EQ_STKRC = .FALSE.
  RETURN
ENDIF


stkrc_work1 => stkrc_1
stkrc_work2 => stkrc_2

n_chars1 = stkrc_work1%n_chars
DO WHILE (n_chars1 == 0)
  IF (ASSOCIATED(stkrc_work1%next)) THEN
    stkrc_work1 => stkrc_work1%next
    n_chars1 = stkrc_work1%n_chars
  ELSE
    EXIT
  ENDIF
ENDDO

n_chars2 = stkrc_work2%n_chars
DO WHILE (n_chars2 == 0)
  IF (ASSOCIATED(stkrc_work2%next)) THEN
    stkrc_work2 => stkrc_work2%next
    n_chars2 = stkrc_work2%n_chars
  ELSE
    EXIT
  ENDIF
ENDDO


IF ((n_chars1 == 0) .OR. (n_chars2 == 0)) THEN
                  ! If any of the n_charstocmp[1,2] is zero now,
                  ! its stack is made of empty chunks only:
  IF (n_chars1 == n_chars2) THEN
                  ! - both stacks made of empty chunks: accepted equal
    STKRC_STKRC_EQ_STKRC = .TRUE.
  ELSE
                  ! - one empty, the other not: must be different
    STKRC_STKRC_EQ_STKRC = .FALSE.
  ENDIF

  RETURN

ENDIF

                  ! Neither stack is completely empty, so we must
                  ! proceed to the detailed comparision

i_s1 = 1          ! start at the first character in the current (work)
i_s2 = 1          ! chunks in each one of the stacks.

n_charstocmp = MIN(n_chars1, n_chars2)
i_e1 = n_charstocmp
i_e2 = n_charstocmp


DO

  IF (stkrc_work1%str_chunk(i_s1:i_e1) /= &
      stkrc_work2%str_chunk(i_s2:i_e2)    ) THEN
    STKRC_STKRC_EQ_STKRC = .FALSE.
    RETURN
  ENDIF

                  ! Have all the characters from the current stack1
                  ! chunk been processed?
  IF (i_e1 == n_chars1) THEN
                  ! Yes: proceed to the next chunk in stack1
    i_s1 = 1
    i_e1 = 0

    n_chars1 = 0
    DO WHILE (n_chars1 == 0)
      IF (ASSOCIATED(stkrc_work1%next)) THEN
        stkrc_work1 => stkrc_work1%next
        n_chars1 = stkrc_work1%n_chars
      ELSE
        EXIT
      ENDIF
    ENDDO

  ELSE
                  ! No: advance the start index to the next character
    i_s1 = i_e1 + 1

  ENDIF
                  ! If n_chars1 == 0 now, then there are no chars
                  ! left on stack 1


                  ! Have all the characters from the current stack2
                  ! chunk been processed?
  IF (i_e2 == n_chars2) THEN
                  ! Yes: proceed to the next chunk
    i_s2 = 1
    i_e2 = 0

    n_chars2 = 0
    DO WHILE (n_chars2 == 0)
      IF (ASSOCIATED(stkrc_work2%next)) THEN
        stkrc_work2 => stkrc_work2%next
        n_chars2 = stkrc_work2%n_chars
      ELSE
        EXIT
      ENDIF
    ENDDO

  ELSE
                  ! No: advance the start index to the next character
    i_s2 = i_e2 + 1

  ENDIF
                  ! If n_chars2 == 0 now, then there are no chars
                  ! left on stack 2

  IF ((n_chars1 == 0) .OR. (n_chars2 == 0)) THEN
                    ! If one of the n_chars1 or n_chars1 is zero now,
                    ! its stack has no more characters left
    IF (n_chars1 == n_chars2) THEN
                    ! - both stacks have been compared: all done, they are equal
      STKRC_STKRC_EQ_STKRC = .TRUE.
    ELSE
                    ! - one empty, the other not: must be different
      STKRC_STKRC_EQ_STKRC = .FALSE.
    ENDIF

    RETURN
  
  ENDIF

                  ! There are characters left on both
  
  n_charstocmp = MIN(n_chars1 - i_e1, n_chars2 - i_e2)

  i_e1 = i_e1 + n_charstocmp
  i_e2 = i_e2 + n_charstocmp

ENDDO


RETURN

!-----------------------------------------------------------------------
 END FUNCTION STKRC_STKRC_EQ_STKRC
!-----------------------------------------------------------------------




!-----------------------------------------------------------------------
 LOGICAL FUNCTION STKRC_STKRC_EQ_STR(stkrc_1, str_2)
!-----------------------------------------------------------------------

IMPLICIT NONE

! Argument list variables
! -----------------------

TYPE(stack_recchunks), POINTER :: stkrc_1
CHARACTER(LEN=*), INTENT(IN) :: str_2


! Local variables
! ---------------

TYPE(stack_recchunks), POINTER :: stkrc_work1

INTEGER :: n_chars1, i_s1, i_e1
INTEGER :: n_chars2, i_s2, i_e2
INTEGER :: n_charstocmp

CHARACTER(LEN=p_maxlen_chunk)  :: str_chunk


IF (.NOT. ASSOCIATED(stkrc_1)) THEN
  STKRC_STKRC_EQ_STR = .FALSE.
  RETURN
ENDIF


stkrc_work1 => stkrc_1

n_chars1 = stkrc_work1%n_chars
DO WHILE (n_chars1 == 0)
  IF (ASSOCIATED(stkrc_work1%next)) THEN
    stkrc_work1 => stkrc_work1%next
    n_chars1 = stkrc_work1%n_chars
  ELSE
    EXIT
  ENDIF
ENDDO


n_chars2 = LEN_TRIM(str_2)


IF ((n_chars1 == 0) .OR. (n_chars2 == 0)) THEN
                  ! If any of the n_charstocmp[1,2] is zero now,
                  ! its stack is made of empty chunks only:
  IF (n_chars1 == n_chars2) THEN
                  ! - both stacks made of empty chunks: accepted equal
    STKRC_STKRC_EQ_STR = .TRUE.
  ELSE
                  ! - one empty, the other not: must be different
    STKRC_STKRC_EQ_STR = .FALSE.
  ENDIF

  RETURN

ENDIF

                  ! Neither stack is completely empty, so we must
                  ! proceed to the detailed comparision

i_s1 = 1          ! start at the first character in the current (work)
i_s2 = 1          ! chunks in each one of the stacks.

n_charstocmp = MIN(n_chars1, n_chars2)
i_e1 = n_charstocmp
i_e2 = n_charstocmp


DO

  IF (stkrc_work1%str_chunk(i_s1:i_e1) /= str_2(i_s2:i_e2)) THEN
    STKRC_STKRC_EQ_STR = .FALSE.
    RETURN
  ENDIF

                  ! Have all the characters from the current stack1
                  ! chunk been processed?
  IF (i_e1 == n_chars1) THEN
                  ! Yes: proceed to the next chunk in stack1
    i_s1 = 1
    i_e1 = 0

    n_chars1 = 0
    DO WHILE (n_chars1 == 0)
      IF (ASSOCIATED(stkrc_work1%next)) THEN
        stkrc_work1 => stkrc_work1%next
        n_chars1 = stkrc_work1%n_chars
      ELSE
        EXIT
      ENDIF
    ENDDO

  ELSE
                  ! No: advance the start index to the next character
    i_s1 = i_e1 + 1

  ENDIF
                  ! If n_chars1 == 0 now, then there are no chars
                  ! left on stack 1


                  ! Have all the characters from str_2 been processed?
  IF (i_e2 == n_chars2) THEN
                  ! Yes: nothing left
    n_chars2 = 0

  ELSE
                  ! No: advance the start index to the next character
    i_s2 = i_e2 + 1

  ENDIF
                  ! If n_chars2 == 0 now, then there are no chars
                  ! left on stack 2

  IF ((n_chars1 == 0) .OR. (n_chars2 == 0)) THEN
                    ! If n_chars1 is zero now, then there
                    ! are no more characters left on the stack;
                    ! if n_chars2, then there are no characters left 
                    ! in the character string
    IF (n_chars1 == n_chars2) THEN
                    ! - all successfully done and now empty: they are equal
      STKRC_STKRC_EQ_STR = .TRUE.
    ELSE
                    ! - one empty, the other not: must be different
      STKRC_STKRC_EQ_STR = .FALSE.
    ENDIF

    RETURN
  
  ENDIF

                  ! There are characters left on both
  
  n_charstocmp = MIN(n_chars1 - i_e1, n_chars2 - i_e2)

  i_e1 = i_e1 + n_charstocmp
  i_e2 = i_e2 + n_charstocmp

ENDDO


RETURN

!-----------------------------------------------------------------------
 END FUNCTION STKRC_STKRC_EQ_STR
!-----------------------------------------------------------------------





!-----------------------------------------------------------------------
 FUNCTION STKRC_createCopyStkrc(stkrc_in) RESULT(stkrc_out)
!-----------------------------------------------------------------------

IMPLICIT NONE


! Argument list variables
! -----------------------

TYPE(stack_recchunks),         POINTER :: stkrc_in
TYPE(stack_recchunks),         POINTER :: stkrc_out


! Local variables
! ---------------

TYPE(stack_recchunks), POINTER  :: stkrc_inwork, stkrc_outwork


                 ! Pre-set stkrc_out to NULL
NULLIFY(stkrc_out)

IF (.NOT. ASSOCIATED(stkrc_in)) RETURN

stkrc_inwork => stkrc_in

CALL STKRC_createRoot(stkrc_out)
stkrc_outwork => stkrc_out


DO

  stkrc_outwork%n_chars   = stkrc_inwork%n_chars
  stkrc_outwork%str_chunk = stkrc_inwork%str_chunk
  stkrc_outwork%l_eor     = stkrc_inwork%l_eor

  IF (ASSOCIATED(stkrc_inwork%next)) THEN

    CALL STKRC_createNext(stkrc_outwork)
    stkrc_outwork => stkrc_outwork%next
    stkrc_inwork => stkrc_inwork%next

  ELSE

    EXIT

  ENDIF

ENDDO

RETURN

!-----------------------------------------------------------------------
 END FUNCTION STKRC_createCopyStkrc
!-----------------------------------------------------------------------






!-----------------------------------------------------------------------
 FUNCTION STKRC_createSplitCopyStkrc(stkrc_in) RESULT(stkrcp_out)
!-----------------------------------------------------------------------

IMPLICIT NONE


! Argument list variables
! -----------------------

TYPE(stack_recchunks),         POINTER :: stkrc_in
TYPE(stkrc_ptr), DIMENSION(:), POINTER :: stkrcp_out


! Local variables
! ---------------

TYPE(stack_recchunks), POINTER  :: stkrc_inwork, stkrc_outwork
INTEGER :: n_sections, i_section


                 ! Pre-set stkrcp_out to NULL
NULLIFY(stkrcp_out)

IF (.NOT. ASSOCIATED(stkrc_in)) RETURN

! First count number of sections of <stkrc_in> terminated by EOR
n_sections = STKRC_ncountSections(stkrc_in)

                 ! Allocate space for RESULT
ALLOCATE(stkrcp_out(n_sections))

stkrc_inwork => stkrc_in

DO i_section = 1, n_sections

  CALL STKRC_createRoot(stkrcp_out(i_section)%ptr)
  stkrc_outwork => stkrcp_out(i_section)%ptr

  DO

    stkrc_outwork%n_chars   = stkrc_inwork%n_chars
    stkrc_outwork%str_chunk = stkrc_inwork%str_chunk
    stkrc_outwork%l_eor     = stkrc_inwork%l_eor

                 ! If the current chunk is not the last one
                 ! in the stack, proceed with the next chunk
    IF (ASSOCIATED(stkrc_inwork%next)) THEN

      IF (stkrc_inwork%l_eor) THEN
                 ! If the currently pointed stkrc_inwork
                 ! terminates by EOR, then prune
                 ! and proceed with next chunk in new stkrc
        stkrc_inwork => stkrc_inwork%next
        EXIT
      ELSE
        CALL STKRC_createNext(stkrc_outwork)
        stkrc_outwork => stkrc_outwork%next
        stkrc_inwork => stkrc_inwork%next
      ENDIF
      
    ENDIF

  ENDDO

ENDDO

RETURN

!-----------------------------------------------------------------------
 END FUNCTION STKRC_createSplitCopyStkrc
!-----------------------------------------------------------------------





!-----------------------------------------------------------------------
 SUBROUTINE STKRC_createSplitCopyStr(stkrc_in, strarr_out, nlen_returned)
!-----------------------------------------------------------------------

IMPLICIT NONE


! Argument list variables
! -----------------------

TYPE(stack_recchunks),          POINTER :: stkrc_in
CHARACTER(LEN=*), DIMENSION(:), POINTER :: strarr_out
INTEGER, INTENT(OUT), OPTIONAL          :: nlen_returned


! Local variables
! ---------------

TYPE(stack_recchunks), POINTER  :: stkrc_inwork
INTEGER :: n_sections, i_section
INTEGER :: n_chars, nloclen_returned




                 ! Pre-set strarr_out to NULL
NULLIFY(strarr_out)

IF (.NOT. ASSOCIATED(stkrc_in)) RETURN


n_sections = STKRC_ncountSections(stkrc_in)


                 ! Allocate space for RESULT
ALLOCATE(strarr_out(n_sections))

stkrc_inwork => stkrc_in

nloclen_returned = 0

DO i_section = 1, n_sections

  CALL STKRC_copyStkrcToStr(stkrc_inwork, strarr_out(i_section), &
                            n_chars, l_pruneateor=.TRUE.)
    IF (n_chars < 0) THEN
      nloclen_returned = MIN(nloclen_returned, n_chars)
    ELSE
      IF (nloclen_returned >= 0) nloclen_returned = MAX(nloclen_returned, n_chars)
    ENDIF

  IF (i_section < n_sections) THEN
    DO WHILE(.NOT. stkrc_inwork%l_eor)
      stkrc_inwork => stkrc_inwork%next
    ENDDO
  ENDIF
  stkrc_inwork => stkrc_inwork%next

ENDDO
         

IF (PRESENT(nlen_returned)) nlen_returned = nloclen_returned

RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE STKRC_createSplitCopyStr
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
 FUNCTION STKRC_ncountSections(stkrc_in) RESULT(n_sections)
!-----------------------------------------------------------------------

IMPLICIT NONE


! Argument list variables
! -----------------------

TYPE(stack_recchunks), POINTER :: stkrc_in
INTEGER                        :: n_sections


! Local variables
! ---------------

TYPE(stack_recchunks), POINTER  :: stkrc_inwork

n_sections = 0

IF (.NOT. ASSOCIATED(stkrc_in)) RETURN

! Count number of sections of <stkrc_in> terminated by EOR

stkrc_inwork => stkrc_in

DO
                 ! If the current chunk is terminated
                 ! by EOR, count it.
  IF (stkrc_inwork%l_eor) n_sections = n_sections+1

  IF (ASSOCIATED(stkrc_inwork%next)) THEN
                 ! We are not yet done: there are chunks left
    stkrc_inwork => stkrc_inwork%next
  ELSE
                 ! We have just processed the last chunk
                 ! if it was not terminated by EOR, then it has not yet
                 ! been accounted for: count it.
    IF (.NOT. stkrc_inwork%l_eor) n_sections = n_sections+1
    EXIT         ! and we are done
  ENDIF

ENDDO

NULLIFY(stkrc_inwork)

RETURN

!-----------------------------------------------------------------------
 END FUNCTION STKRC_ncountSections
!-----------------------------------------------------------------------


!===============================================================================
END MODULE MODMXM_STKRC
!===============================================================================
