!
!    Copyright 2014, 2015, 2017, 2018, 2020, 2021 Guy Munhoven
!
!    This file is part of µXML.
!
!    µXML is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Affero General Public License as
!    published by the Free Software Foundation, either version 3 of
!    the License, or (at your option) any later version.
!
!    µXML is distributed in the hope that it will be useful, but
!    WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!    See the GNU Affero General Public License for more details.
!
!    You should have received a copy of the Affero GNU General Public
!    License along with µXML.  If not, see <https://www.gnu.org/licenses/>.
!


#include <modmxm.h>
!===============================================================================
 MODULE MODMXM_GENERAL
!===============================================================================

IMPLICIT NONE

#ifdef DEBUG
INTEGER, PARAMETER, PRIVATE :: jp_stddbg = (MXM_STDDBG)
#endif


! Parameters
! ----------

! Configuration of MODMXM

INTEGER, PARAMETER :: p_maxlen_chunk = 10      ! Minimum 10

                                    ! Get the required compiler-specific
                                    ! IOSTAT status values for OK, EOR,
                                    ! EOF and OUT (beyond EOF) flags 
#include "modmxm_general.h"

INTEGER, PARAMETER :: p_unit    = 10
INTEGER, PARAMETER :: p_logunit = 11



CONTAINS


!-------------------------------------------------------------------------------
SUBROUTINE READ_NEXTCHUNK(i_unit, i_iostat, str_chunk, i_size, &
                          nout_rec, iout_chunk, l_resetcounters)
!-------------------------------------------------------------------------------

IMPLICIT NONE

! Argument list variables
! -----------------------

INTEGER,                       INTENT(IN)  :: i_unit
INTEGER,                       INTENT(OUT) :: i_iostat
CHARACTER(LEN=p_maxlen_chunk), INTENT(OUT) :: str_chunk
INTEGER,                       INTENT(OUT) :: i_size
INTEGER,                       INTENT(OUT) :: nout_rec
INTEGER,                       INTENT(OUT) :: iout_chunk
LOGICAL, OPTIONAL,             INTENT(IN)  :: l_resetcounters


! Local Variables
! ---------------

CHARACTER(LEN=*), PARAMETER :: str_fmt_lbr = '("Read Rec.Chunk ", I0,".",I0, ": {")'
CHARACTER(LEN=*), PARAMETER :: str_fmt_lvb = '("Read Rec.Chunk ", I0,".",I0, ": |")'
CHARACTER(LEN=*), PARAMETER :: str_fmt_rbr = '(A, "}", 1X, "size=",I0)'
CHARACTER(LEN=*), PARAMETER :: str_fmt_rvb = '(A, "|", 1X, "size=",I0)'

INTEGER, SAVE :: n_rec = 1
INTEGER, SAVE :: i_chunk = 0
INTEGER, SAVE :: ntot_chunk = 0




IF (PRESENT(l_resetcounters)) THEN
  IF (l_resetcounters) THEN
    ntot_chunk = 0
    n_rec = 1
    i_chunk = 0
  ENDIF
ENDIF
                  ! Read the next chunk of maximum <p_maxlen_chunk>
                  ! characters. If the record is shorter than that,
                  ! the EOR will be diagnosed, and the variable <i_size>
                  ! which holds the number of actually read characters
                  ! will be such that i_size < p_maxlen_fileline

READ(i_unit, '(A)', ADVANCE='NO', IOSTAT=i_iostat, SIZE=i_size) str_chunk


IF (i_iostat == p_iostat_eof) THEN
                  ! If EOF is encountered (normally, the
                  ! previous READ has diagnosed an EOR in this
  str_chunk = ''  ! case), we may return
  i_size = 0
  nout_rec = n_rec
  iout_chunk = i_chunk
  RETURN

ENDIF

i_chunk = i_chunk + 1
ntot_chunk = ntot_chunk + 1

nout_rec = n_rec
iout_chunk = i_chunk


! Diagnostics output
! ------------------
                  ! The whole record is delimited by '{' (start)
                  ! and '}' (end) and the individual record
                  ! chunks delimited by '|'

#ifdef DEBUG_READ_NEXTCHUNK
IF(i_chunk == 1) THEN
                  ! At the first chunk of any record, print out the number
                  ! of the record, and the opening '{' character.

  WRITE(jp_stddbg, str_fmt_lbr, ADVANCE='NO') n_rec, i_chunk

ELSE

  WRITE(jp_stddbg, str_fmt_lvb, ADVANCE='NO') n_rec, i_chunk

ENDIF
#endif


IF(i_iostat == p_iostat_eor) THEN
                  ! We have read past the EOR marker (got the whole line at once,
                  ! or the last chunk). Print out the contents of the chunk,
                  ! the delimiting '}' and its length. 

  n_rec = n_rec + 1
  i_chunk = 0     ! Reset the counter of chunks of the records

# ifdef DEBUG_READ_NEXTCHUNK
  WRITE(jp_stddbg, str_fmt_rbr) str_chunk(1:i_size), i_size

ELSE              ! The EOR marker was not yet read.
                  ! We either only got part of the line (or just exactly the
                  ! correct number of characters -- the next READ will return
                  ! the IOSTAT flag for EOR encountered, and an empty
                  ! <str_chunk>).
                  ! Print the chunk contents and the separating '|'
  WRITE(jp_stddbg, str_fmt_rvb)  str_chunk, i_size
# endif

ENDIF


RETURN

!-------------------------------------------------------------------------------
END SUBROUTINE READ_NEXTCHUNK
!-------------------------------------------------------------------------------


!===============================================================================
END MODULE MODMXM_GENERAL
!===============================================================================
