# septins isodesmic polymerization simulation. started Aug 13, 2019
# proposed units: length -> length of octamer, 32 nm
#				          mass -> mass of octamer, 360 kD
#				          energy -> 1 kT at room temp = 2.479 kJ/mol
#                 time -> 12.2 ns


import hoomd
import hoomd.dybond_plugin as db;
import numpy as np
import hoomd.deprecated
import random
import hoomd.md
import os
import sys
import collections


#command line parameters
N_septins = int(sys.argv[1])           # 5000
areaFrac = float(sys.argv[2])          # 0.25
#kon = float(sys.argv[4])               # 9e-7
koff = float(sys.argv[3])              # 4.5e-4
percentage_ghosts = float(sys.argv[4]) # 0.5
fric = float(sys.argv[5])			         # 1
binding_energy = float(sys.argv[6])    # 16
flex = float(sys.argv[7])              # 311

hoomd.context.initialize("")

# create uniformly distributed, non-overlapping positions
poslist = list()
extent = np.ceil(N_septins ** (1.0/2.0))
if extent % 2 != 0:
  extent += 1
num_added = 0
for x in range(int(-extent/2), int(extent/2)):
  for y in range(int(-extent/2), int(extent/2)):
    if num_added == N_septins:
      break
    else:
      poslist.append([x, y, 0])
      num_added += 1


snapshot = hoomd.data.make_snapshot(N=N_septins, box=hoomd.data.boxdim(Lx=extent, Ly=extent, dimensions=2),
                                    particle_types=["A", "ghost"], bond_types=["dybond"], angle_types=["persistence"])

snapshot.particles.position[:] = poslist

# initialize a certain percentage as ghosts, trying to find fastest was to steady state
# starting all as ghosts takes forever to nucleate filaments, and starting all on the
# membrane locks it into huge filaments that won't break sustainably because of the very
# high area fraction. try initializing 80% of particles as ghosts so that area fraction of
# membrane bound octamers is what it should be at steady state when they are filaments
#percentage_ghosts = 0.8
ghost_list_inds = random.sample(range(0, N_septins), int(np.ceil(N_septins*percentage_ghosts)))
snapshot.particles.typeid[ghost_list_inds] = 1

snapshot.bonds.resize(1)
snapshot.bonds.group[:] = [0, 1]

snapshot.angles.resize(1)
snapshot.angles.group[:] = [0, 1, 2]



# read snapshot
system = hoomd.init.read_snapshot(snapshot)


###########################
### pairwise potentials ###
###########################
nl = hoomd.md.nlist.cell()

# WCA to maintain hard spheres
lj = hoomd.md.pair.lj(r_cut=2**(1.0/6.0), nlist=nl)
lj.pair_coeff.set('A', 'A', epsilon=5, sigma=1)
lj.pair_coeff.set('A', 'ghost', epsilon=0, sigma=0)
lj.pair_coeff.set('ghost', 'ghost', epsilon=0, sigma=0)

# bonds
harm_bond = hoomd.md.bond.harmonic()
harm_bond.bond_coeff.set('dybond', k=40, r0=2**(1.0/6.0))

# angles
cos_angle = hoomd.md.angle.cosinesq()
cos_angle.angle_coeff.set('persistence', k=flex, t0=np.pi)

# set cutoff size for neighborlist
#nl.set_params(r_buff=1.2)


# select integrator
bd_dt = 1e-4
integrator_mode = hoomd.md.integrate.mode_standard(dt=bd_dt)


brownian = hoomd.md.integrate.brownian(group=hoomd.group.all(), kT=1, seed=1)
# will need to calculate what gamma actually is in simulation units. In real units, from PT data, gamma should be 2.85e-9 kg/s
#fric = 4.65e4
#fric = 1
brownian.set_gamma('A', gamma=fric)
brownian.set_gamma('ghost', gamma=0.01)


# output data
filename = 'septins' + str(N_septins) + '_aFrac' + str(areaFrac) + '_koff' + str(koff) + '_pghost' + str(percentage_ghosts) + '_friction' + str(fric) + '_energy' + str(binding_energy) + '_flex' + str(flex)
hoomd.dump.gsd(filename=filename + '.gsd', period=1e5, group=hoomd.group.all(), overwrite=True, dynamic=['attribute','topology'])
#hoomd.analyze.log(filename=filename + '.log', quantities=['potential_energy', 'kinetic_energy', 'temperature', 'pressure', 'pressure_xy', 'pressure_xz', 'pressure_yz'], period=1e2)


#gradually increase/decrease box size to account for memory allocation issues that come up with sudden changes
snapshot = system.take_snapshot(all=True)
dim_cur = snapshot.box.Lx

#rescale box to match specified area fraction
areaX = np.pi * (0.5**2) * N_septins / areaFrac
side = np.sqrt(areaX)
hoomd.update.box_resize(L=hoomd.variant.linear_interp([(0, dim_cur), (1e5, side)]), scale_particles=True)


hoomd.run(1)

#delete fake bond and angle
del system.bonds[0]
del system.angles[0]


hoomd.run(1e5)



type_list = ["A", "ghost"]
partner_list=[("A","A")]

# angle tolerance automatically sets Pb
angle_tol = 10
Pb = (angle_tol*2/360)**2


# kon: should be koff divided by 500, so for the koff below, kon is 9e-7

# koff: if fric coeff set to 1, then this should be 4.5e-4 to match with particle tracking diffusion
# coeff estimate of 1.8 microns^2/sec and koff of 1.42 sec^-1


# update probabilities with 'period' below divided by dt from brownian dynamics
kon = 9e-7
#koff = 4.5e-4

update_period = 1e4
factor = update_period*bd_dt
Pb *= factor
binding_energy = binding_energy - np.log(factor)
kon *= factor
koff *= factor




updater = db.update.dybond(nl, group=hoomd.group.all(), period=update_period)
updater.set_params(bond_type='dybond',
                   particle_type_names=type_list,
                   bond_partners=partner_list,
                   fun_groups=2,
                   rcut=2**(1.0/6.0),
                   Ea=binding_energy,
                   P_b=Pb,
                   add_angles=True,
                   angle_tolerance=angle_tol,
                   ghost=True,
                   konn=kon,
                   koff=koff,
                   alpha=1,
                   percent_bonds_per_step=100,
                   stop_after_percent=100,
                   enable_rxn_enthalpy=False)#False,


for i in range(10):
	hoomd.run(1e9)




