;--------------------------------------------------------------------------------------
;
;  October 2020
;  Script associated to publication:
;  "Sensitivity of precipitation and temperature over Mount Kenya area to
;   physics parameterization options in a high-resolution model simulation
;   performed with WRFV3.8.1"
;  Authors: M. Messmer, S.j. González-Rojí, C.C. Raible and T.F. Stocker
;
;
;  Creative Commons Licence: Attribution-NonCommercial 4.0 International (CC BY-NC 4.0)
;--------------------------------------------------------------------------------------

begin

  dom     = getenv("DOM")    ;get domain number you would like to plot
  country = getenv("COU")    ;give country name you would like to plot, so that filenames are properly named
  setting = getenv("SET")    ;provide parent grid and number of nests, used for naming the input files
  nr_doms = getenv("NDOM")   ;provide number of domains in total, used for naming the input files
  era5_t  = getenv("E5_T")   ;filename of ERA5 2-metre temperature
  alt_t   = getenv("ALT_T")  ;alternative data for 2-metre temperature, i.e., CRU
  wrf_f   = getenv("WRF")
  alt_setting = getenv("ALT")
  nr_stations = 7
  station = new((/nr_stations,12/),float)
  col_p_s = new((/nr_stations,12/),float)
  lat_s   = new(nr_stations, float)
  lon_s   = new(nr_stations, float)
  level_p = (/276,278,282,284,286,288,290,292,294,296,298,300/)+4
  color_p = (/  0, 64, 49, 49, 45, 42, 39, 33, 29, 20, 13,  8, 1/)

  data = asciiread("Station_2008_mon_2metre_temperature.txt",(/nr_stations,14/),"float")
  do ii = 0,dimsizes(lat_s)-1
    do jj = 0,11
      station(ii,jj) = tofloat(data(ii,jj))+273.
      if (station(ii,jj).lt.level_p(0)) then
        col_p_s(ii,jj) = color_p(0)
      end if
      do kk = 0,dimsizes(level_p)-2
        if (station(ii,jj).ge.level_p(kk) .and. station(ii,jj).lt.level_p(kk+1)) then
          col_p_s(ii,jj) = color_p(kk+1)
        end if
      end do
      if (station(ii,jj).ge.level_p(kk)) then
         col_p_s(ii,jj) = color_p(kk+1)
      end if
    end do
    lat_s(ii) = data(ii,12)
    lon_s(ii) = data(ii,13)
  end do

  files_T2 = systemfunc("ls T2-D0"+dom+"-ERA5-"+country+"-"+setting+"-*-"+nr_doms+"-2008-monthlymean.nc")
  dd = dimsizes(files_T2)

  in = addfile(files_T2(0),"r")
  vv = in->tas
  t_d = dimsizes(vv)
  t_dim = dimsizes(t_d)
  delete(vv)
  dSizes = getfiledimsizes(in)

  if (t_dim .eq. 4) then
    ntime = t_d(0)
    nlat  = t_d(2)
    nlon  = t_d(3)
  else if (t_dim .eq.3) then
    ntime = t_d(0)
    nlat  = t_d(1)
    nlon  = t_d(2)
  end if
  end if
 
  experiment = (/"ERA5","AltData","Cumulus3-1Way","Cumulus3","S.America","Europe","NoCumulus"/)
  T2 = new((/7,ntime,nlat,nlon/),float)
  if (dd .eq.5) then
    mi = 3
  else
    mi = 4
  end if
  do i = 0,dimsizes(experiment)-mi 
    if (t_dim .eq. 4) then
      im = addfile(files_T2(i),"r")
      lat = im->lat
      lon = im->lon
      vv = im->tas
      T2(i+2,:,:,:) = vv(:,0,:,:)
    else if (t_dim .eq. 3) then
      im = addfile(files_T2(i),"r")
      T2(i+2,:,:,:) = tofloat(im->tas)
      lat = im->lat
      lon = im->lon
    end if
    end if 
  end do

  im = addfile(era5_t,"r")
  T2(0,:,:,:) = tofloat(im->var167)
  im = addfile(alt_t,"r")
  T2(1,:,:,:) = tofloat(im->tmp)
  ; add 273 to transform CRU data from Celsius into Kelvin
  T2(1,:,:,:) = T2(1,:,:,:) + 273.

  in = addfile(wrf_f,"r")

  wks1 = gsn_open_wks("pdf","T2_"+country+"_2008_D0"+dom+"_"+alt_setting)

  gsn_merge_colormaps(wks1,"matlab_hot","GMT_polar")

  cnres                       = True
  cnres@gsnFrame              = False
  cnres@gsnDraw               = False
  cnres@cnLinesOn             = False
  cnres@cnLineLabelsOn        = False
  cnres@cnInfoLabelOn         = False

  cnres@cnFillOn              = True
  cnres@cnFillMode            = "AreaFill"    ; This is the default.
  cnres@lbLabelFontHeightF    = 0.010
  cnres@lbLabelStride         = 1 
  cnres@lbLabelBarOn          = False

  ; Set contour levels.
; cnres@cnFillPalette         = "matlab_hot"
  cnres@cnLevelSelectionMode  = "ExplicitLevels"
  cnres@gsnMaximize           = True
  cnres@gsnLeftString         = ""
  cnres@gsnCenterString       = ""
  cnres@gsnRightString        = "" 
  res = cnres   ; Copy contour resources for 1st plot.
  res = wrf_map_resources(in,res)
  dx = in@DX
  dx = dx / 1000
  res@mpFillOn                = True
  res@mpPerimOn               = True
  res@mpFillColors            = (/ -1, -1 ,-1, -1/)
; res@mpInlandWaterFillColor  = 3
  res@mpFillDrawOrder         = "PostDraw"
  res@mpOutlineBoundarySets   = "AllBoundaries"
  res@mpNationalLineThicknessF= 1.0
  res@mpGeophysicalLineThicknessF= 1.5
  res@mpGeophysicalLineColor  = "black"
  res@mpNationalLineColor     = "black"
  res@mpDataBaseVersion       = "MediumRes"
  res@mpDataSetName           = "Earth..4"
; res@pmTickMarkDisplayMode   = "Never"
  res@mpGridAndLimbDrawOrder  = "PostDraw"
  res@mpGridAndLimbOn         = True
  res@mpGridLatSpacingF       = 30
  res@mpGridLonSpacingF       = 30
  res@tfDoNDCOverlay          = True
  res@tmXBLabelFontHeightF    = 0.02
  res@tmYLLabelFontHeightF    = 0.02

  gsres               = True
  gsres@gsMarkerIndex = 16
  if (dx.eq.27 .or. dx.eq.25) then
    gsres@gsMarkerSizeF = 0.005
  else if (dx.eq.9) then
    gsres@gsMarkerSizeF = 0.007
  else if (dx.eq.5) then
    gsres@gsMarkerSizeF = 0.008
  else if (dx.eq.3) then
    gsres@gsMarkerSizeF = 0.009
  else if (dx.eq.1) then
    gsres@gsMarkerSizeF = 0.009
  end if
  end if
  end if
  end if
  end if
  gsres@gsMarkerThicknessF = 1.5
  gsres2 = gsres
  gsres2@gsMarkerIndex = 4
  gsres2@gsMarkerThicknessF = 1.0
  gsres2@gsMarkerColor = "black"

  kres = True
  kres@gsMarkerSizeF = gsres@gsMarkerSizeF
  kres@gsMarkerIndex = 12
  kres@gsMarkerThicknessF = 2.0
  kres@gsMarkerColor = (/0.95,0.95,0.95,1/)
  tres = True
  tres@txFontHeightF = 0.02

  t2 = new((/dimsizes(experiment)+1,12/),graphic)
  stat_rr = new((/3,12,dimsizes(lat_s)/),graphic)
  stat_r2 = new((/3,12,dimsizes(lat_s)/),graphic)
  mt_kenya = new((/dimsizes(experiment),12/),graphic)
  rr_text = new((/12,dimsizes(lat_s)/),graphic)

  panres = True
  panres@gsnMaximize = True
  panres@gsnPaperOrientation = "landscape"
  panres@lbLabelFontHeightF  = 0.012
  panres@lbTitlePosition     = "bottom"
  panres@lbTitleFontHeightF  = 0.012
  panres@lbTitleString       = "monthl 2-metre temperature average [K]"
  panres@lbTitleFont         = "helvetica"

  res1                     = True
  res2                     = True
  res1@gsnMaximize         = True
  res2@gsnAttachPlotsYAxis = True

  mon = new(12,string)
  mon(0) = "Jan"
  mon(1) = "Feb"
  mon(2) = "Mar"
  mon(3) = "Apr"
  mon(4) = "May"
  mon(5) = "Jun"
  mon(6) = "Jul"
  mon(7) = "Aug"
  mon(8) = "Sep"
  mon(9) = "Oct"
  mon(10)= "Nov"
  mon(11)= "Dec"

  exp_name = (/"ERA5","CRU","Cumulus3-1Way","Cumulus3","South America","Europe","No Cumulus"/)
  txcolor  = (/(/146,149,152/),(/185,45,91/),(/47,147,165/),(/25,213,222/),(/135,214,51/),(/255,173,9/),(/130,125,200/)/) / 255.
  printVarSummary(txcolor)
  print("txt color")
  
  index = (/1,2,3,4,21,29,30/)
  do ii = 0,dimsizes(experiment)-1
    do jj = 0,11
      
      res@cnLevels    = level_p
      res@cnFillColors  = color_p 
      res@gsnStringFont = "helvetica-bold"
      res@gsnStringFontHeightF = 0.02
      res@gsnStringFontColor = "black"
      res@gsnLeftString = "  "
      res@gsnRightString = ""+mon(jj)+"  "
      
      if (ii.le.2) then
        t2(ii,jj) = gsn_csm_contour_map(wks1, T2(ii,jj,5:nlat-6,5:nlon-6), res)
        res@lbLabelBarOn = True
        plot = gsn_csm_contour_map(wks1, T2(ii,jj,5:nlat-6,5:nlon-6), res)
        res@lbLabelBarOn = False
        ll = 1
        do kk = 0,dimsizes(lat_s)-1
          gsres@gsMarkerColor = col_p_s(kk,jj)
          if (ii.le.2) then
            stat_rr(ii,jj,kk) = gsn_add_polymarker(wks1,t2(ii,jj),lon_s(kk),lat_s(kk),gsres)
            stat_r2(ii,jj,kk) = gsn_add_polymarker(wks1,t2(ii,jj),lon_s(kk),lat_s(kk),gsres2)
          end if
          if (ii .eq. 0) then
            if (dx.eq.1) then
              if (lon_s(kk).gt.min(lon) .and. lon_s(kk).lt.max(lon) .and. lat_s(kk).gt.min(lat) .and. lat_s(kk).lt.max(lat)) then
                rr_text(jj,kk) = gsn_add_text(wks1,t2(ii,jj),tostring(index(ll-1)),lon_s(kk),lat_s(kk)-0.055,tres)
                ll = ll + 1
              end if
            end if
            t2(7,jj) = gsn_csm_contour_map(wks1,T2(0,jj,5:nlat-6,5:nlon-6), res)
          end if
        end do
        delete(res@cnLevels)
        delete(res@cnFillColors)
      else if (ii.gt.2) then
        resa = res
        delete(resa@cnLevels)
        delete(resa@cnFillColors)
        resa@cnLevels     = (/-1.5,-1.0,-0.75,-0.5,-0.25,-0.1,0.1,0.25,0.5,0.75,1.0/)
        resa@cnFillColors = (/   2,   4,    6,   8,    9,  11,-64,  12, 14,  16, 18,  20/)+64
        diff = T2(ii,jj,5:nlat-6,5:nlon-6) - T2(2,jj,5:nlat-6,5:nlon-6)
        t2(ii,jj) = gsn_csm_contour_map(wks1, diff, resa)
        resa@lbLabelBarOn = True
        plot1 = gsn_csm_contour_map(wks1, diff, resa)
        resa@lbLabelBarOn = False
      end if
      end if

      txres                       = True
      txres@txPerimOn             = True
      txres@txFontHeightF         = 0.03
      txres@txBackgroundFillColor = txcolor(ii,:)
      txres@txPerimColor          = "white"
      txres@txFont                = "helvetica-bold"
      txres@txFontColor           = "white"
      text_label = "    "+exp_name(ii)+"    "

      amres = True
      amres@amParallelPosF   = -0.5   ; This is the center of the plot.
      amres@amOrthogonalPosF = -0.507   ; This is the top edge of the plot.
      amres@amJust           = "BottomLeft"
      txid = gsn_create_text(wks1, text_label, txres)
      annoid = gsn_add_annotation(t2(ii,jj), txid, amres)
      do kk = 0,dimsizes(lat_s)-1
        mt_kenya(ii,jj) = gsn_add_polymarker(wks1,t2(ii,jj),37.3085,-0.1518,kres)
      end do
      
    end do
  end do
  
  draw(plot)
  frame(wks1)
  draw(plot1)
  frame(wks1)

  do jj = 0,11
    ann1 = gsn_attach_plots(t2(0,jj),(/t2(2,jj),t2(4,jj),t2(6,jj)/),res1,res2)
    ann2 = gsn_attach_plots(t2(1,jj),(/t2(3,jj),t2(5,jj),t2(7,jj)/),res1,res2)
    gsn_panel(wks1,(/t2(0,jj),t2(1,jj)/),(/2,1/),panres)
  end do

end

