#### packages ####

require(data.table)
require(tidyverse)
require(lavaan)
require(infer)
require(magrittr)

# all reversed variabes are precoded

#### load data ####

Data <- fread("replication_data_study2.csv",na.strings = c("NA",""))

#### balance test (Appendix J) ####

Data %>% 
  select(Gender:`Unemployment periods`,treatment) %>%
  gather(variable,value,-treatment) %>%
  group_by(variable) %>%
  na.omit %>%
  nest() %>%
  mutate(wo_treatment = map(data,~aov(data=.,(treatment == "treatment")~1))) %>%
  mutate(w_treatment = map(data,~aov(data=.,(treatment == "treatment")~value))) %>%
  mutate(anova = map2(.x = wo_treatment,.y = w_treatment,~anova(.x,.y))) %>%
  mutate(F_test = map_dbl(anova,~.$"F"[2])) %>%
  mutate(p_value = map_dbl(anova,~.$"Pr(>F)"[2])) %>%
  transmute(covariate = variable,`P-value`=p_value)

#### measurement model ####

lavaan_data <- Data

names(lavaan_data) <- str_replace_all(names(lavaan_data)," ","_")

cfa_model <- '
stigma=~ NA*Stigma_1 + a * Stigma_1 + b * Stigma_2 + c * Stigma_3
stigma~~ NA*stigma
Stigma_1 ~ ma*1
Stigma_2 ~ mb*1
Stigma_3 ~ mc*1
stigma ~ NA*1
a + b + c == 3
ma + mb + mc == 0
autonomy_loss=~ NA*Autonomy_loss_1 + d * Autonomy_loss_1 + e * Autonomy_loss_2 + f * Autonomy_loss_3 + g * Autonomy_loss_4
autonomy_loss~~ NA*autonomy_loss
Autonomy_loss_1 ~ md*1
Autonomy_loss_2 ~ me*1
Autonomy_loss_3 ~ mf*1
Autonomy_loss_4 ~ mg*1
autonomy_loss ~ NA*1
d + e + f + g == 4
md + me + mf + mg == 0
stress=~ NA*Stress_1 + h * Stress_1 + i * Stress_2 + j * Stress_3
stress~~ NA*stress
Stress_1 ~ mh*1
Stress_2 ~ mi*1
Stress_3 ~ mj*1
stress ~ NA*1
h + i + j == 3
mh + mi + mj == 0
'

result <- lavaan::cfa(model=cfa_model, data = lavaan_data,missing = "ml")

Data[,"stress"] <- lavaan::lavPredict(result)[,"stress"]
Data[,"stigma"] <- lavaan::lavPredict(result)[,"stigma"]
Data[,"autonomy_loss"] <- lavaan::lavPredict(result)[,"autonomy_loss"]

#### Results ####

set.seed(11)

study2_results <- bind_rows(
  infer_ATE(data=Data,y="stress",reps = 5000),
  infer_ATE(data=Data,y="stigma",reps = 5000),
  infer_ATE(data=Data,y="autonomy_loss",reps = 5000)
)

study2_results <- study2_results %>%
  get_cis() %>%
  mutate("DV" = c("Stress","Stigma","Autonomy loss"))

study2_results

#### Table 3 ####

Data %>% 
  select(-c(Stress_1:Stigma_3)) %>%
  gather(variable,value,stress,stigma,autonomy_loss) %>%
  group_by(variable) %>%
  nest() %>%
  mutate(model_wo_controls = map(data,~lm(data=.,value~ treatment ))) %>%
  mutate(model_w_controls = map(data,~lm(data=.,value~ . ))) %>%
  pivot_longer(cols = model_wo_controls:model_w_controls,names_to = "model",values_to = "result") %>%
  mutate(tidy_result = map(result,broom::tidy)) %>%
  unnest(tidy_result) %>%
  select(variable,model,term,estimate,std.error,p.value) 

#### Appendix H ####

parameterestimates(result) %>% filter(op == "=~")
fitmeasures(result,fit.measures = c("chisq","cfi","rmsea","srmr"))
lavInspect(result,"nobs")


#### Appendix Q ####

Data$`Unemployment length` <- ordered(
  factor(
    Data$`Unemployment length`,
    levels = c("0-5 weeks", "6-10 weeks", "11-15 weeks","16-20 weeks","21 or more weeks")
  )
)

study2_median <- median(as.integer(ordered(Data$`Unemployment length`)),na.rm = T)

study2_levels <- levels(Data$`Unemployment length`)[1:study2_median]

Data <- Data %>% mutate(longer = !`Unemployment length` %in% study2_levels)

interaction_models <- Data %>% 
  gather(variable,value,stress,stigma,autonomy_loss) %>%
  group_by(variable) %>%
  nest() %>%
  mutate(model_wo_controls = map(data,~lm(data=.,value~ treatment * longer))) %>%
  mutate(model_w_controls = map(data,~lm(data=.,value ~ Gender + Age + Education + `Unemployment status` + `Number of children` + `Unemployment periods` + 
                                           treatment * longer)))

long_interaction_models <- interaction_models %>%
  gather(model_type,model,model_wo_controls:model_w_controls) %>%
  mutate(tidy_model = map(model,~broom::tidy(.x))) %>%
  unnest(tidy_model) 


